/* (C) Copyright International Business Machines Corporation 23 January */
/* 1990.  All Rights Reserved. */
/*  */
/* See the file USERAGREEMENT distributed with this software for full */
/* terms and conditions of use. */
/* File: o_table.c */
/* Author: David F. Bacon */
#ifndef lint
static char sccsinfo[] = "@(#)o_table.c	1.33 2/17/92";
#endif

#define _BSD 43
/* 
  N.B.: When lazy-copying tables we don't always raise exceptions that
  ought to be raised.  Specifically, UnCopyable is never raised, and
  Depletion is not raised in cases where a deep copy would deplete.
  This has the side-effect that other instructions that modify tables
  can end up raising exceptions they aren't supposed to, because they
  make deep copies as their first step.  For example, a REMOVE
  operation could raise UnCopyable or Depletion!
 */ 

#include <rpc/rpc.h>
#include <varargs.h>
#include <stdio.h>

#include "sysdep.h"

#include "ops.h"
#include "ops_parm.h"
#include "storage.h"
#include "accessors.h"
#include "recursiv.h"

#include "predefined.cd"
#include "interpform.cd"
#include "shape.h"

#define Component 0

#define NO_TBL   -1

#define FIND TRUE
#define GET FALSE

extern datarep dr_table, dr_integer, dr_bottom, dr_boolean, dr_ord_enumeration,
 dr_enumeration;
extern datarep *datarepmap[], *qdatarepmap[];

extern tbldes *tbldescriptors[];

predef_exception doCopyOnWrite();
#define CopyOnWrite(objp) \
  (((objp)->value.table->refcount is 1) ? Normal : doCopyOnWrite(objp))

/* when the user specifies a key or index for lookup, he doesn't count the */
/* ordering.  but for representation numbers, the order comes first, so this */
/* macro normalizes the key number into a representation number. */
#define keynumtorepnum(t,n) ((t)->ordered ? ((n)+1) : (n))

#define foreach_tbl(tbl, tablenum, tblptr) \
  for ((tblptr) = & (tbl)->tbls[tablenum = 0]; tablenum < (tbl)->tblcount; \
       (tblptr) = & (tbl)->tbls[++tablenum])

#define for_lookups(tbl, tablenum, tblptr) \
  for_folltbl(tbl, tablenum, tblptr, firstlookup(tbl))

#define for_folltbl(tbl, tablenum, tblptr, firsttable) \
  for (tblptr = & tbl->tbls[tablenum = firsttable]; \
       tablenum < (tbl)->tblcount; \
       tblptr = & tbl->tbls[++tablenum])

#define forprev_tbl(tbl, tablenum, tblptr, lasttable) \
  for (tblptr = & tbl->tbls[tablenum = 0]; tablenum < lasttable; \
       tblptr = & tbl->tbls[++tablenum])

#define firstlookup(tbl) ((tbl)->ordered ? 1 : 0)

#define ordfunc(table, func) \
  (*(table)->tbls[ORDER_TBL].des->func)

#define doforeach(table) (*(table)->tbls[ORDER_TBL].des->foreach)

/* global variables used in sick kludge to pass extra args to doxdr */
static datarep *doxdr_tsdr;
static shapep doxdr_shape;

NILOP(o_size)
{
    OPCHK(SrcObj,table);
    Dst.integer = Src.table->size;
    set_init(DstObj, dr_integer);
}


NILOP(o_new_table)
{
    predef_exception che_new_table();
    predef_exception retcode;

    if ((retcode = che_new_table(DstObj, args->qualifiers, args->sched))
	isnt Normal)
      raise_builtin(retcode);
}

/* che_new_table is called directly by generated C-code. */
predef_exception
che_new_table(dstobj, qualifier, sched)
objectp dstobj;
valcell qualifier;
schedblock *sched;
{
    predef_exception cp_table();
    void fin_table();
    void re_finalize();

    predef_exception retcode;
    dfd_table *table;
    valcell nonlookinfo;
    valcell idxinfo;
    valcell reps;
    valcell indexset, keyset, lookup_info;
    trepnum tblnum;
    counter repcount;
    tblinfo *tblptr;
    flag nonlooked;
    int nlrepno;
    counter i;
    tbldes *des;
    extern flag cherm_flag;

    keyset.table = nil;
    indexset.table = nil;
    table = nil;

    nonlookinfo = vdot(qualifier, new_table_info__nonlookup);
    nonlooked = case_of(vdot(qualifier, new_table_info__nonlookup)) 
      isnt table_rep_type__none;
	
    if (case_of(vdot(qualifier, new_table_info__opt_reps)) is
	option__absent) {
	reps.table = nil;
	repcount = 1;
    }
    else {
	lookup_info = vdot(vdot(qualifier, new_table_info__opt_reps),
			  Component);
	retcode = cp_table(&keyset, vdot(lookup_info, lookup_info__keys));
	if (retcode is Normal)
	  retcode = cp_table(&indexset,
			     vdot(lookup_info, lookup_info__indices));
	if (retcode isnt Normal)
	  goto cleanup;
	reps = vdot(lookup_info, lookup_info__reps);
	repcount = size_of(reps) + (nonlooked ? 1 : 0);
    }
    
    retcode = Depletion;	/* any further errors are Depletion */

    table = (dfd_table *) 
      getmain(sizeof(dfd_table) + 
	      sizeof(tblinfo) * (repcount - ARBSIZE));
    if (table is nil) goto cleanup;
    
    for (i = 0; i < repcount; i++)
      table->tbls[i].des = nil;

    /* fill in preliminary info */
    table->refcount = 1;
    table->tsdr = & dr_bottom;
    table->size = 0;
    table->tblcount = repcount;
    table->ordered = FALSE;
    table->shareidx = -1;
    table->keyset = keyset;
    table->indexset = indexset;

    /* Fill in representation info for the non-lookup rep. */
    if (nonlooked) {
        valcell info;

	nlrepno = case_of(nonlookinfo);
	tblptr = & table->tbls[ORDER_TBL];
	des = tbldescriptors[nlrepno];
	/* the assembler leaves some components of the nonlookinfo */
	/* component uninit, contrary to the type definition.  The */
	/* allocation functions must notice when this happens so they */
	/* don't try to use information that's not present.  They test */
	/* this by checking for a zero valcell.  Problem is, not all */
	/* bottoms have a zero valcell.  This kluge just ensures that */
	/* a zero valcell is given in this case */
	if (vdotrep(nonlookinfo, Component)->number is dr_bottom.number)
	  info.nominal = nil;
	else
	  info = vdot(nonlookinfo, Component);
	if (!(*des->alloc)(table, ORDER_TBL, info))
	  goto cleanup;
	tblptr->des = des;

	if (nlrepno is table_rep_type__vector or
	    nlrepno is table_rep_type__charstring or
	    nlrepno is table_rep_type__dublink)
	  table->ordered = TRUE;
    }

    /* Then do likewise for all the other reps */
    if (reps.table isnt nil)
      for_lookups(table, tblnum, tblptr) {
	  idxinfo = get_elem(reps, tblnum - firstlookup(table));
	  des = tbldescriptors[case_of(idxinfo)];
	  if (!(*des->alloc)(table, tblnum,
			     vdot(idxinfo, Component)))
	    goto cleanup;
	  tblptr->des = des;
      }
    if (not cherm_flag)
      re_finalize(dstobj, F_DISCARD, sched);
				/* finalize the value of the destination; */

    dstobj->value.table = table;
    set_init(dstobj, dr_table);
    return(Normal);

  cleanup:			/* here when a storage allocation fails */
    if (table isnt nil) {
	foreach_tbl(table, tblnum, tblptr) {
	    if (tblptr->des isnt nil)
	      /* Note: shallow finalization OK for all tbl reps */
	      /* because the table has no elements */
	      (*tblptr->des->finalize)(table, tblnum, SHALLOW, 
				       F_FREE, (schedblock *) nil);
	}
	{ freemain(table, 
		   sizeof(dfd_table) + sizeof(tblinfo)*(repcount - ARBSIZE)); }
    }
    if (keyset.table isnt nil)
      (void) fin_table(keyset, F_DISCARD, nil);
    if (indexset.table isnt nil)
      (void) fin_table(indexset, F_DISCARD, nil);
    return(retcode);
}


NILOP(o_insert)
{
    predef_exception insertfunc();
    predef_exception retcode;

    OPCHK(DstObj,table);
    if ((retcode = insertfunc(DstObj, SrcObj, FALSE, 0)) isnt Normal)
      raise_builtin(retcode);
}


/* insertfunc is called directly by generated C-code. */
predef_exception
insertfunc(dstobj, srcobj, insertat, ix)
objectp dstobj;
objectp srcobj;
flag insertat;
int ix;
{
    predef_exception retcode;
    trepnum tblnum, ptblnum;
    tblinfo *tblptr, *ptblptr;


    retcode = CopyOnWrite(dstobj); /* make private copy if table is shared */
    if (retcode isnt Normal)
      return(retcode);

    foreach_tbl(dstobj->value.table, tblnum, tblptr) {
	if (insertat and tblnum is ORDER_TBL)
	  retcode = (*tblptr->des->insert_at)
	    (dstobj->value.table, tblnum, srcobj->value, ix);
	else
	  retcode = (*tblptr->des->insert)
	    (dstobj->value.table, tblnum, srcobj->value);

	if (retcode isnt Normal) {
	    forprev_tbl(dstobj->value.table, ptblnum, ptblptr, tblnum)
	      if (insertat and tblnum is ORDER_TBL)
		(*ptblptr->des->unmerge_at)
		  (dstobj->value.table, ptblnum, srcobj->value, ix, 1);
	      else
		(*ptblptr->des->uninsert)
		  (dstobj->value.table, ptblnum, srcobj->value);
	    /* there's no harm in not undoing the CopyOnWrite here */
	    return(retcode);
	}
    }
    dstobj->value.table->tsdr = srcobj->tsdr;	/* in case it's not yet set */
    dstobj->value.table->size++;
    set_bottom(srcobj);
    return(Normal);
}



NILOP(o_merge)
{
    predef_exception mergefunc();
    predef_exception retcode;

    OPCHK(SrcObj,table);
    OPCHK(DstObj,table);
    if ((retcode = mergefunc(DstObj, SrcObj, FALSE, 0)) isnt Normal)
      raise_builtin(retcode);
}




/* mergefunc is called directly by generated C-code. */
predef_exception
mergefunc(dstobj, srcobj, mergeat, ix)
objectp dstobj;
objectp srcobj;
flag mergeat;
int ix;
{
    predef_exception domerge();
    int undomerge();
    void finalize_table();

    predef_exception retcode;
    counter curelem, undoelem;
    trepnum curtbl;


    /* make private copies if tables are shared */
    if ((retcode = CopyOnWrite(srcobj)) isnt Normal)
      return(retcode);
    if ((retcode = CopyOnWrite(dstobj)) isnt Normal)
      return(retcode);
    
    /* transfer tsdr in case dst is empty */
    if (size_of(srcobj->value) isnt 0)
      dstobj->value.table->tsdr = srcobj->value.table->tsdr; 

    curtbl = 0;
    /* turn merge of an ordered table into a merge-at with ix set to */
    /* the current size of the destination. */
    if (! mergeat)
      if (dstobj->value.table->ordered) {
	mergeat <- TRUE;
	ix <- dstobj->value.table->size;
      }
      
    if (mergeat) {
	retcode = ordfunc(dstobj->value.table, merge_at)
	  (dstobj->value.table, ORDER_TBL, srcobj->value.table, ix);
	if (retcode isnt Normal)
	  return(retcode);
    }

    retcode = (predef_exception)
      doforeach(srcobj->value.table)(srcobj->value.table, FIRST_TBL, 
				     (int (*)()) domerge, &curelem,
				     (int) Normal, dstobj->value.table,
				     &curtbl, mergeat);
    if (retcode isnt Normal) {
	/* curelem and curtbl were left so as to indicate how far the */
	/* merge proceeded: all elements prior to curelem were merged */
	/* into all table reps; curelem was merged into all reps up to */
	/* but not including curtbl.  Table size was adjusted for */
	/* number of completely merged elements */
 	(void) doforeach(srcobj->value.table)
	  (srcobj->value.table, FIRST_TBL, undomerge, &undoelem, CONT_FOREACH,
	   dstobj->value.table, curelem, curtbl, mergeat);
	if (mergeat)
	  ordfunc(dstobj->value.table, unmerge_at)
	    (dstobj->value.table, ORDER_TBL, ix, srcobj->value.table->size);
	return(retcode);
    }
    
    finalize_table(srcobj->value.table, SHALLOW, F_FREE, (schedblock *) nil);
    set_bottom(srcobj);
    return(Normal);
}


/*ARGSUSED*/
static predef_exception
domerge(thetable, tblnum, val, curelem, argv)
dfd_table *thetable;
trepnum tblnum;
valcell val;
counter curelem;
va_list argv;
{
    predef_exception retcode;
    tblinfo *tblptr;
    flag mergeat;
    trepnum firsttbl;
    dfd_table *dsttable;
    trepnum *curtbl;

    dsttable = va_arg(argv, dfd_table *);
    curtbl = va_arg(argv, trepnum *);
    mergeat = va_arg(argv, flag);

    firsttbl = mergeat ? firstlookup(thetable) : FIRST_TBL;

    for_folltbl(dsttable, *curtbl, tblptr, firsttbl) {
	retcode = (*tblptr->des->insert)(dsttable, *curtbl, val);
	if (retcode isnt Normal)
	  return(retcode);
    }

    dsttable->size++;
    return(Normal);
}


/*ARGSUSED*/
static int
undomerge(thetable, tblnum, val, curelem, argv)
dfd_table *thetable;
trepnum tblnum;
valcell val;
counter curelem;
va_list argv;
{
    flag mergeat;
    counter lastelem;
    trepnum lasttbl;
    trepnum firsttbl;
    dfd_table *dsttable;
    tblinfo *tblptr;

    dsttable = va_arg(argv, dfd_table *);
    lastelem = va_arg(argv, counter);
    lasttbl = va_arg(argv, trepnum);
    mergeat = va_arg(argv, flag);

    firsttbl = mergeat ? firstlookup(thetable) : FIRST_TBL;

    if (curelem < lastelem) {
	/* element was merged into all reps... remove it */
	for_folltbl(dsttable, tblnum, tblptr, firsttbl) {
	    (*tblptr->des->uninsert)(dsttable, tblnum, val); 
	}

	dsttable->size--;
	return(CONT_FOREACH);
    }
    
    if (curelem > lastelem)
      /* past the range of elements that were merged, wholly or */
      /* partially */
      return(STOP_FOREACH);

    /* Element was partially merged... remove from tables up to but */
    /* not including lasttbl */
    for (tblptr = & dsttable->tbls[tblnum = firsttbl];
	 tblnum < lasttbl;
	 tblptr = & dsttable->tbls[++tblnum])
      (*tblptr->des->uninsert)(dsttable, tblnum, val);

    return(CONT_FOREACH);
}



NILOP(o_initget)
{
    void re_finalize();
    predef_exception init_selector();
    predef_exception retcode;

    extern flag cherm_flag;

    OPCHK(SrcObj,table);
    if (not cherm_flag)
      re_finalize(DstObj, F_DISCARD, args->sched);
				/* finalize the value of the iterator; */
    if (args->qualifiers.record)
      retcode = init_selector(DstObj, SrcObj, nil, args->sched, GET,
		    vdot(args->qualifiers, integer_pair__int_one).integer,
		    vdot(args->qualifiers, integer_pair__int_two).integer);
    else /* modifier may be missing (nil) on chermes calls */
      retcode = init_selector(DstObj, SrcObj, nil, args->sched, GET, 0, 0);
    if (retcode isnt Normal)
      raise_builtin(retcode);
}


/* init_selector is called directly by generated C-code. */
predef_exception
init_selector(dstobj, src1obj, src2objs, sched, findorget, repnum, startpos)
objectp dstobj;
objectp src1obj;
object **src2objs;
schedblock *sched;
flag findorget;
int repnum;
int startpos;
{
    handlr_stack *stackelem = nil;
    inspect_frame *frame = nil;
    tblinfo *tblptr;
    position pos;
    predef_exception retcode;
    counter pos_size;

    if ((frame = new(inspect_frame)) is nil) goto cleanup;
    frame->inspectobj = dstobj;
    frame->inspectee = src1obj;
    frame->repnum = repnum;
    frame->rmvcount = 0;

    if ((stackelem = new(handlr_stack)) is nil) goto cleanup;
    stackelem->handler = FALSE;
    stackelem->frame.inspect = frame;

    tblptr = & src1obj->value.table->tbls[repnum];

    if (findorget is GET) {
	if (!(*(tblptr->des->initget))
	    (src1obj->value.table, repnum, & pos, & pos_size, startpos))
	  goto cleanup;
    }
    else {
	if (!(*(tblptr->des->initfind))
	    (src1obj->value.table, repnum, & pos, & pos_size, src2objs))
	  goto cleanup;
    }
    stackelem->frame.inspect->pos = pos;
    stackelem->frame.inspect->pos_size = pos_size;
    stackelem->next = sched->ready->ep.h->info.context->estack;
    sched->ready->ep.h->info.context->estack = stackelem;	
    return(Normal);

  cleanup:
    /* here when something failed... clean up and raise depletion */
    if (stackelem isnt nil)
      { dispose(stackelem, handlr_stack); }
    if (frame isnt nil)
      { dispose(frame, inspect_frame); }
    return(Depletion);
}



NILOP(o_endget)
{
    void end_selector();

    OPCHK(SrcObj,table);
    end_selector(DstObj, SrcObj, args->sched, GET);
}


/* end_selector is called directly by generated C-code. */
void			/* raises no exceptions */
end_selector(dstobj, srcobj, sched, findorget)
objectp dstobj;
objectp srcobj;
schedblock *sched;
flag findorget;
{
    handlr_stack *stackelem;
    inspect_frame *frame;
    tblinfo *tblptr;


    stackelem = sched->ready->ep.h->info.context->estack;
    frame = stackelem->frame.inspect;

#ifdef DEBUG
    if (frame->inspectobj isnt dstobj or frame->inspectee isnt srcobj) {
	nilerror("end_selector", "mismatch in selector begin/end");
	showstatus(sched->ready);
	abort_nili("end_selector");
    }
#endif

    tblptr = & frame->inspectee->value.table->tbls[frame->repnum];

    if (findorget is GET)
      (*tblptr->des->endget)
	(srcobj->value.table, frame->repnum, & frame->pos);
    else
      (*tblptr->des->endfind)
	(srcobj->value.table, frame->repnum, & frame->pos);

    { dispose(frame, inspect_frame); }

    sched->ready->ep.h->info.context->estack = stackelem->next;
    { dispose(stackelem, handlr_stack); }
}


NILOP(o_get_or_err)
{
    predef_exception get_selected_object();

    predef_exception retcode;

    OPCHK(SrcObj,table);
    retcode = get_selected_object(DstObj, SrcObj, args->sched);

    if (retcode isnt Normal)
      raise_builtin(retcode);
}


NILOP(o_get_or_goto)
{
    predef_exception get_selected_object();

    predef_exception retcode;

    OPCHK(SrcObj,table);
    retcode = get_selected_object(DstObj, SrcObj, args->sched);

    if (retcode is Normal)
      return;

    if (retcode is NotFound) {
	args->nextop = args->qualifiers.integer;
				/* on NotFound, branch to target operation */
    }
    else
      raise_builtin(retcode);
}


predef_exception
get_selected_object(dstobj, srcobj, sched)
objectp dstobj;
objectp srcobj;
schedblock *sched;
{
    inspect_frame *get_selector_frame();

    inspect_frame *frame;
    tblinfo *tblptr;
    predef_exception retcode;


    frame = get_selector_frame(dstobj, sched);
    tblptr = & frame->inspectee->value.table->tbls[frame->repnum];
    
    retcode = (*tblptr->des->get)(srcobj->value.table, frame->repnum,
				  & frame->pos,  & dstobj->value);

    if (retcode is Normal)
      dstobj->tsdr = qdatarepmap[srcobj->value.table->tsdr->number];
				/* give the quopy-equivalent to the */
				/* table element tsdr */

    return(retcode);
}


/* get_selector_frame is called directly by generated C-code. */
inspect_frame *
get_selector_frame(obj, sched)
schedblock *sched;
objectp obj;
{
    handlr_stack *stack;

    stack = sched->ready->ep.h->info.context->estack;

    while (stack) {
	if (not stack->handler)	/* is it an inspect frame? */
	  if (stack->frame.inspect->inspectobj is obj)
	    return (stack->frame.inspect);
	stack = cdr(stack);
    }

#ifdef DEBUG
    nilerror("get_selector_frame", "Couldn't find inspect object on stack");
#endif

    return(nil);
}


NILOP(o_remove)
{
    inspect_frame *get_selector_frame();
    predef_exception removefunc();
    predef_exception retcode;
    inspect_frame *frame;

    OPCHK(Src2Obj,table);
    frame = get_selector_frame(Src1Obj, args->sched);
    retcode = removefunc(DstObj, Src1Obj, Src2Obj, args->sched,
			 frame->repnum, frame->repnum, & frame->pos);
    if (retcode is Normal)
      frame->rmvcount++;
    else
      raise_builtin(retcode);
}

/* che_remove is called directly by generated C-code. */
predef_exception
che_remove(dstobj, src1obj, src2obj, sched)
objectp dstobj;
objectp src1obj;
objectp src2obj;
schedblock *sched;
{
    inspect_frame *get_selector_frame();
    predef_exception removefunc();
    predef_exception retcode;
    inspect_frame *frame;

    frame = get_selector_frame(src1obj, sched);
    retcode = removefunc(dstobj, src1obj, src2obj, sched,
			 frame->repnum, frame->repnum, & frame->pos);
    if (retcode is Normal)
      frame->rmvcount++;
    return(retcode);
}


NILOP(o_fremove)
{
    predef_exception removefunc();
    predef_exception retcode;

    OPCHK(Src2Obj,table);
    retcode =
      removefunc(DstObj, Src1Obj, Src2Obj, args->sched,
		 args->qualifiers.integer, NO_TBL, (position *) nil);
    if (retcode isnt Normal)
      raise_builtin(retcode);
}


/* removefunc is called directly by generated C-code. */
predef_exception
removefunc(dstobj, src1obj, src2obj, sched, keyrep, selrep, pos)
objectp dstobj;
objectp src1obj;
objectp src2obj;
schedblock *sched;
trepnum keyrep;
trepnum selrep;
position *pos;
{
    void re_finalize();

    trepnum tblnum;
    tblinfo *tblptr;
    predef_exception retcode;
    extern flag cherm_flag;

    retcode = CopyOnWrite(src2obj); /* make private copy if table is shared */
    if (retcode isnt Normal)
      return(Depletion);

    foreach_tbl(src2obj->value.table, tblnum, tblptr)
      (*tblptr->des->remove)
	(src2obj->value.table, tblnum, src1obj->value, keyrep, selrep, pos);
				/* just remove it from each table.  each */
				/*  implementation of remove must guarantee */
				/*  not to raise depletion. */

    if (not cherm_flag)
      re_finalize(dstobj, F_DISCARD, sched);
				/* finalize the value of the destination; */
    *dstobj = *src1obj;		/* copy object */
    dstobj->tsdr = datarepmap[dstobj->tsdr->number]; /* set non-quopy tsdr */
    set_bottom(src1obj);	/* set source (inspected element) to bottom */

    src2obj->value.table->size--;
    return(Normal);
}

/******************************************************************************
 *                        Generic Table Operations                            *
 *****************************************************************************/

void
fin_table(table, f_op, sched)
valcell table;
finalize_op f_op;
schedblock *sched;
{
    void finalize_table();

    finalize_table(table.table, DEEP, f_op, sched);
}


static void
finalize_table(table, depth, f_op, sched)
dfd_table *table;
flag depth;
finalize_op f_op;
schedblock *sched;
{
    trepnum tblnum;
    tblinfo *tblptr;

    if (--table->refcount isnt 0) /* only free if no more references. */
      return;

    foreach_tbl(table, tblnum, tblptr) {
	if (tblnum is FIRST_TBL)
	  (*tblptr->des->finalize)(table, tblnum, depth, f_op, sched);
	else
	  /* shallow finalize should never revive processes... passing */
	  /* nil for schedblock will cause hermi to abort if that happens */
	  (*tblptr->des->finalize)(table, tblnum, SHALLOW, f_op, nil);
    }

    /* pass nil for schedblock when finalizing auxiliary tables, as */
    /* they should never cause any processes to be revived */
    if (table->keyset.table)		/* get rid of the keys */
      finalize_table(table->keyset.table, DEEP, f_op, nil);
    if (table->indexset.table)	/* get rid of the indices */
      finalize_table(table->indexset.table, DEEP, f_op, nil);

    { freemain(table, sizeof(tblinfo) * (table->tblcount - ARBSIZE) +
	       sizeof(dfd_table)); }
}



predef_exception
cp_table(dst,src)
valcell *dst, src;
{
#ifndef DODEEPCOPY
    if (src.table->refcount is MAXCOUNTER) 
      return(Depletion);

    src.table->refcount++;
    dst->table = src.table;
    return(Normal);
#else
    predef_exception deep_cp_table();

    return(deep_cp_table(dst,src));
#endif
}


NILOP(o_privatize)
{
    predef_exception retcode;

    OPCHK(DstObj,table);
    retcode = CopyOnWrite(DstObj); /* make private copy if table is shared */
    if (retcode isnt Normal)
      raise_builtin(retcode);
}


/* called by the CopyOnWrite() macro, which first tests whether */
/* obj->value.table->refcount = 1; if it is, it does nothing; if it isn't, */
/* it calls doCopyOnWrite(). */
predef_exception
doCopyOnWrite(obj)
object *obj;
{
    predef_exception deep_cp_table();
    predef_exception retcode;
    dfd_table *tbl;
    valcell newtbl;

    tbl = obj->value.table;


				/* make a private copy of the table */
    if ((retcode = deep_cp_table(&newtbl, obj->value)) isnt Normal)
      return(retcode);
    obj->value = newtbl;	/* install the new copy */
    tbl->refcount--;		/* and decrement refcount on shared copy. */
    return(Normal);
}


predef_exception
deep_cp_table(dst,src)
valcell *dst, src;
{
    predef_exception docopy();
    predef_exception retcode;
    counter elemcounter;
    valcell newtab;
    trepnum tblnum;
    tblinfo *tblptr;
    int i;

    newtab.table = (dfd_table *) 
      getmain(sizeof(dfd_table) + 
	      sizeof(tblinfo) * (src.table->tblcount - ARBSIZE));
    if (newtab.table is nil) goto cleanup;

    newtab.table->refcount = 1;
    newtab.table->tsdr = src.table->tsdr;
    newtab.table->ordered = src.table->ordered;
    newtab.table->shareidx = -1;
    newtab.table->tblcount = src.table->tblcount;
    newtab.table->size = 0;

    newtab.table->keyset.table = nil;
    newtab.table->indexset.table = nil;

    for (i = 0; i < newtab.table->tblcount; i++)
      newtab.table->tbls[i].des = nil;

    if (src.table->keyset.table) {
	retcode = cp_table(&newtab.table->keyset,src.table->keyset);
	if (retcode isnt Normal)
	  goto cleanup;
    }
    if (src.table->indexset.table) {
	retcode = cp_table(&newtab.table->indexset,src.table->indexset);
	if (retcode isnt Normal)
	  goto cleanup;
    }

    foreach_tbl(src.table, tblnum, tblptr) {
	if (!(*tblptr->des->precopy)(newtab.table, tblnum, src.table)) {
	    retcode = Depletion;
	    goto cleanup;
	}
	newtab.table->tbls[tblnum].des = tblptr->des;
    }

    retcode = (predef_exception)
      doforeach(src.table)(src.table, FIRST_TBL,
			   (int (*)()) docopy, &elemcounter, (int) Normal,
			   newtab.table);
    if (retcode isnt Normal)
      goto cleanup;

    /* copy operation succeeded... install the new copy */
    dst->table = newtab.table;
    return(Normal);


  cleanup:
    /* The copy failed somewhere... clean up and return exception code */
    if (newtab.table isnt nil) {
	if (newtab.table->size isnt 0) {
	    /* nonempty table is complete and consistent... just */
	    /* finalize it... nil for schedblock is ok since */
	    /* finalizing the new copies of table elements can never */
	    /* revive a process */
	    fin_table(newtab, F_DISCARD, nil);
	    return(retcode);
	}
	if (newtab.table->keyset.table isnt nil)
	  fin_table(newtab.table->keyset, F_DISCARD, nil);
	if (newtab.table->indexset.table isnt nil)
	  fin_table(newtab.table->indexset, F_DISCARD, nil);
	foreach_tbl(src.table, tblnum, tblptr)
	  if (newtab.table->tbls[tblnum].des isnt nil)
	    (*newtab.table->tbls[tblnum].des->finalize)
	      (newtab.table, tblnum, SHALLOW, F_FREE, (schedblock *) nil);
	{ freemain(newtab.table, sizeof(dfd_table) +
		   sizeof(tblinfo)*(src.table->tblcount - ARBSIZE)); }
    }
    return(retcode);
}


/*ARGSUSED*/
static predef_exception
docopy(srctable, srctblnum, val, elemcounter, argv)
dfd_table *srctable;
trepnum srctblnum;
valcell val;
counter elemcounter;
va_list argv;
{
    dfd_table *dsttable;
    trepnum tblnum, ptblnum;
    tblinfo *tblptr, *ptblptr;
    valcell newval;
    predef_exception retcode;


    dsttable = va_arg(argv, dfd_table *);

    retcode = (*srctable->tsdr->copy)(&newval,val);
				/* do the DEEP copy of the value */
    if (retcode isnt Normal)
      return(retcode);
    
    foreach_tbl(srctable, tblnum, tblptr) {
	retcode = (*tblptr->des->insert)(dsttable, tblnum, newval);
	if (retcode isnt Normal)  {
	    forprev_tbl(dsttable, ptblnum, ptblptr, tblnum)
	      (*ptblptr->des->uninsert)(dsttable, ptblnum, newval);
	    (*dsttable->tsdr->finalize)(newval, F_DISCARD, nil);
	    return(retcode);
	}
    }
    dsttable->size++;

    return(Normal);
}


status 
eq_table(t1, t2)
valcell t1, t2;
{
    if (t1.table is t2.table)
      return(SUCCESS);		/* lisp EQ for shared tables */

    if (t1.table->size isnt t2.table->size)
      return(FAILURE);

    if (t1.table->size is 0)
      return(SUCCESS);

    return((*(t1.table->tbls[FIRST_TBL].des->equal))
	   (t1.table, FIRST_TBL, t2.table));
}



comparison
cmp_table(t1, t2)
valcell t1, t2;
{
    if (t1.table is t2.table)
      return(CMP_EQUAL);	/* lisp EQ for shared tables */

    if (t1.table->size < t2.table->size)
      return(CMP_LESS);

    if (t1.table->size > t2.table->size)
      return(CMP_GREATER);

    if (t1.table->size is 0)
      return(CMP_EQUAL);

    return((*(t1.table->tbls[FIRST_TBL].des->comparekeys))
	   (t1.table, FIRST_TBL, t2.table));
}


void
prt_table(f, indent, v)
FILE *f;
int indent;
valcell v;
{
    int doprint();
    void print_keyset();
    void indent_for_print();
    void print_representations();

    datarep *tsdr;
    counter elemcounter;


    tsdr = v.table->tsdr;

    print_representations(f, v.table);

    if (v.table->size is 0) {
	(void) fprintf(f, "; EMPTY");
    }
    else
      (void) fprintf(f, "; %d elements of type %s", v.table->size, tsdr->name);

    if (v.table->keyset.table)
      print_keyset(f, v.table->keyset.table, "Keys");
    if (v.table->indexset.table)
      print_keyset(f, v.table->indexset.table, "Indices");

    if (v.table->size is 0) {
	(void) fprintf(f, "\n");
	return;
    }

    (void) fprintf(f, " --\n");

    if (v.table->ordered and 
	v.table->tbls[ORDER_TBL].des->number is table_rep_type__charstring) {
	indent_for_print(f, indent+1);
	(void) fprintf(f, " \"%s\"\n", 
		       v.table->tbls[ORDER_TBL].rep.chs->elements);
    }
    else {
	(void) doforeach(v.table)(v.table, FIRST_TBL,
				  doprint, &elemcounter, CONT_FOREACH,
				  f, indent+1);

	indent_for_print(f, indent);
	(void) fprintf(f, "End of Table\n");
    }
}



static void
print_representations(f, t)
FILE *f;
dfd_table *t;
{
    int i;

    for (i = 0; i < t->tblcount; i++)
      (void) fprintf(f, "%s%s", 
		     i is 0 ? "" : ",",
		     t->tbls[i].des->name);
}



/*ARGSUSED*/
static int
doprint(table, tblnum, val, elemcounter, argv)
dfd_table *table;
trepnum tblnum;
valcell val;
counter elemcounter;
va_list argv;
{
    void indent_for_print();

    FILE *f;
    int indent;

 
    f = va_arg(argv, FILE *);
    indent = va_arg(argv, int);

    indent_for_print(f, indent);
    if (table->ordered)
      (void) fprintf(f, "[%d] ", elemcounter);
    (*table->tsdr->print)(f, indent, val);

    return(CONT_FOREACH);
}


xdr_status
hxdr_table(xdrs, objp, shape)
XDR *xdrs;
objectp objp;
shapep shape;
{
  int doxdr();
  xdr_status xdr_tblinfo();
  xdr_status xdr_lookup();
  void xdrfree_table();
  dfd_table *get_share_tbl();
  int add_share_tbl(), find_share_tbl();
  
  counter elemcounter;
  trepnum repnum;
  dfd_table *tbl = nil;
  counter size;
  int i;
  trepnum tblnum, ptblnum;
  tblinfo *tblptr, *ptblptr;
  object newobj;
  predef_exception retcode;
  flag chsrep;
  char *chs_elements;
  int test;
  int shareidx;
  dfd_table *sharedtbl;
  valcell v;

  /* XDR_FREE case handled separately because it just doesn't mesh */
  /* well with the rest of it */
  if (xdrs->x_op is XDR_FREE) {
    xdrfree_table(xdrs, & objp->value);
    return(XDR_OK);
  }
  
  /* NOTE: Enormous kluge to avoid requireing a separate entity to */
  /* encode shared table number of lack thereof: */
  /* If the next integer is positive, it's either the number of */
  /* representations (if there's no current shape info) or the size of */
  /* this table (if there's shape info).  If it's negative, this is */
  /* another instance of a table we've already seen, and the absolute */
  /* value is an index into the shared table list. */
  if (xdrs->x_op is XDR_DECODE) {
    if (! xdr_int(xdrs, &test))
      return(XDR_FAIL);
    if (test < 0) {
      sharedtbl = (dfd_table *) get_share_tbl(-test);
      objp->tsdr = &dr_table;
      v.table = sharedtbl;
      return(cp_table(&objp->value, v) is Normal ?
	     XDR_OK : XDR_FAIL);
    }
  }
  else if (xdrs->x_op is XDR_ENCODE) {
    if ((shareidx = find_share_tbl(objp->value.table)) == -1)
      /* first occurrence ... add it to the share list and continue */
      shareidx = add_share_tbl(objp->value.table);
    else {
      /* not first instance... negate and encode the index */
      shareidx = -shareidx;
      return(xdr_int(xdrs, &shareidx));
    }
  }

  /* not the first instance... proceed with ENCODE and DECODE cases... */
  if (xdrs->x_op is XDR_ENCODE) {
    tbl = objp->value.table;
    repnum = tbl->tblcount;
    size = tbl->size;
  }
  /* prevent any deallocation if we fail before anything gets */
  /* allocated on a DECODE */
  if (xdrs->x_op is XDR_DECODE)
    objp->value.table = nil;
  
  /* We assume the following information is independent of instance and */
  /* can be safely gotten from the shape information, if available: */
  /* tbl->tsdr */
  /* tbl->ordered */
  /* tbl->tblcount  (this may not be right) */
  /* tbl->keyset */
  /* tbl->indexset */
  /* tbl->tbls[] (this may not be right, see tblcount) */
  /* WARNING: if size is 0, tbl->tsdr is not the real tsdr (usually set to
     &dr_bottom).  Only set shape->value.table->tbl->tsdr if
     size is > 0. */
  
  if (shape->tsdr is &dr_bottom) {
    /* first encode/decode number of representations */
    if (xdrs->x_op is XDR_DECODE)
      repnum = test;		/* decoded in share check above */
    else
      if (!xdr_int(xdrs, &repnum)) goto cleanup;

    /* allocate table structure for a DECODE operation, and */
    /* initialize fields not present in XDR stream */
    if (xdrs->x_op is XDR_DECODE) {
      tbl = (dfd_table *) getmain((counter) sizeof(dfd_table) +
				  sizeof(tblinfo)*(repnum-ARBSIZE));
      if (tbl is nil) goto cleanup;
      /* make sure this gets in before our keysets or indexsets or */
      /* anything */
      if (add_share_tbl(tbl) == -1)
	return(XDR_FAIL);
      tbl->tblcount = repnum;
      tbl->refcount = 1;
      /* table is initially empty */
      tbl->size = 0;
      tbl->tsdr = &dr_bottom;
      /* clear all table reps in case we don't make it all the way */
      /* through */
      for (i = 0; i < repnum; i++)
	tbl->tbls[i].des = nil;
      /* same for key/index info */
      tbl->keyset.table = tbl->indexset.table = nil;
      /* save the table pointer so we can free it later if */
      /* needed */
      objp->value.table = tbl;
    }
    
    /* encode/decode other table-wide info */
    if (!xdr_u_int(xdrs,  &size)) goto cleanup;
    /* only read/write tbl->tsdr if size > 0 (o.w. its meaningless) */
    if (size) {
      /* harmless to use shape here, since it's bottom.  Not so below */
      if (!xdr_datarep(xdrs, &tbl->tsdr, shape)) goto cleanup;
    }
    if (!myxdr_boolean(xdrs, &tbl->ordered)) goto cleanup;
    
    /* encode/decode representation-specific info for each rep */
    for (i = 0; i < repnum; i++) 
      if (!xdr_tblinfo(xdrs, &tbl->tbls[i], tbl, i, size))
	goto cleanup;
    
    /* encode/decode the key/index information */
    if (!xdr_lookup(xdrs, &tbl->keyset)) goto cleanup;
    if (!xdr_lookup(xdrs, &tbl->indexset)) goto cleanup;
    
    if ((shape->value.table = new(shape_table)) is nil)
      goto cleanup;
    shape->value.table->tbl = tbl;
    /* record in size in shape so that we know if tbl has valid tsdr */
    shape->value.table->size = size;
    shape->value.table->elt_shape.tsdr = &dr_bottom;
    /* make sure shape knows what it is */
    shape->tsdr = objp->tsdr;
  }
  else {
    /* first encode/decode number of representations */
    /* if (!xdr_int(xdrs, &repnum)) goto cleanup; */
    /* we're assuming repnum is instance independant.  This may
       turn out to be wrong. */
    repnum = shape->value.table->tbl->tblcount;
    
    /* allocate table structure for a DECODE operation, and */
    /* initialize fields not present in XDR stream */
    if (xdrs->x_op is XDR_DECODE) {
      tbl = (dfd_table *) getmain((counter) sizeof(dfd_table) +
				  sizeof(tblinfo)*(repnum-ARBSIZE));
      if (tbl is nil) goto cleanup;
      /* make sure this gets in before our keysets or indexsets or */
      /* anything */
      if (add_share_tbl(tbl) == -1)
	return(XDR_FAIL);
      tbl->tblcount = repnum;
      tbl->refcount = 1;
      /* table is initially empty */
      tbl->size = 0;
      /* clear all table reps in case we don't make it all the way */
      /* through */
      for (i = 0; i < repnum; i++)
	tbl->tbls[i].des = nil;
      /* same for key/index info */
      tbl->keyset.table = tbl->indexset.table = nil;
      /* save the table pointer so we can free it later if */
      /* needed */
      objp->value.table = tbl;
      
      /* get most of the other table-wide info from shape */
      
      /* if (!xdr_datarep(xdrs, &tbl->tsdr, shape)) goto cleanup; */
      /* this may be invalid (bottom).  see below */
      tbl->tsdr = shape->value.table->tbl->tsdr;
      /* size is instance-dependant, always read it */
      /* if (!myxdr_boolean(xdrs, &tbl->ordered)) goto cleanup; */
      tbl->ordered = shape->value.table->tbl->ordered;
    }
    
    /* if we're encoding, tbl and shape->value.table->tbl should already
       be equal in the instance-independant places, so we only need to
       write the size of this one (and tbl->tsdr if it didn't get written
       last time). */
    
    /* size is instance dependant */
    if (xdrs->x_op is XDR_DECODE)
      size = test;		/* decoded in share check above */
    else
      if (!xdr_u_int(xdrs,  &size)) goto cleanup;
    
    /* check if shape's previous tbl has invalid (bottom) tsdr, 
       and this one OK.  If so, encode/decode the (valid) tsdr */
    if ((shape->value.table->size is 0) and (size > 0)) {
      /* have to call xdr_datarep with a bottom shape, since shape
	 obviously knows nothing about tbl->tsdr.  This is getting
	 really bogus.  This whole mess needs to be rewritten right */
      shapeobj botshape;
      
      botshape.tsdr = &dr_bottom;
      if (!xdr_datarep(xdrs, &tbl->tsdr, &botshape)) goto cleanup;
    }
    
    if (xdrs->x_op is XDR_DECODE) {
      /* get representation-specific info for each rep */
      /* no need to copy in encode, since they're already equal */
      for (i = 0; i < repnum; i++) 
	/* if (!xdr_tblinfo(xdrs, &tbl->tbls[i], tbl, i, size)) */
	if (!copy_tblinfo(xdrs, &tbl->tbls[i], tbl, i, size, 
			  shape->value.table->tbl->tbls[i].des))
	  goto cleanup;
      
      /* encode/decode the key/index information */
      /* again, no need to copy in encode, since they're already equal */
      if (!copy_lookup(xdrs, &tbl->keyset, 
		       &shape->value.table->tbl->keyset)) 
	goto cleanup;
      if (!copy_lookup(xdrs, &tbl->indexset,
		       &shape->value.table->tbl->indexset)) 
	goto cleanup;
    }
    
    /* check if shape's previous tbl has invalid (bottom) tsdr, 
       and this one OK.  If so, fix up shape. */
    if ((shape->value.table->size is 0) and (size > 0)) {
      shape->value.table->size = size;
      shape->value.table->tbl = tbl;
    }
  }
  
  /* check whether this table has a charstring representation */
  chsrep = tbl->ordered and
    (tbl->tbls[ORDER_TBL].des->number is table_rep_type__charstring);
  if (chsrep)
    chs_elements = &tbl->tbls[ORDER_TBL].rep.chs->elements[0];
  
  if (xdrs->x_op is XDR_ENCODE) {
    /* encode each table element */
    if (chsrep) {
      /* special case to use packed external representation for */
      /* charstrings */
      if (!xdr_bytes(xdrs, &chs_elements, &size, size))
	goto cleanup;
    }
    else {
      datarep *save_doxdr_tsdr;
      shapep save_doxdr_shape;
      /* it's really sick that I have to resort to this to pass
	 the extra arguments to doxdr, but thank whoever devised the
	 foreach signature.  I suppose va_args could be used, somehow. */
      /* what I would really like to do, both here and in the decode
	 case, is to simply loop over the objects doing hxdr_object.
	 With shape, this would be at least as efficient at the current
	 convoluted scheme. */
      save_doxdr_tsdr = doxdr_tsdr;
      save_doxdr_shape = doxdr_shape;
      doxdr_tsdr = tbl->tsdr;
      doxdr_shape = & shape->value.table->elt_shape;
      if (!((xdr_status) 
	    doforeach(tbl)(tbl, FIRST_TBL,
			   (int (*)()) doxdr,
			   &elemcounter, (int) XDR_OK,
			   xdrs))) {
	doxdr_tsdr = save_doxdr_tsdr;
	doxdr_shape = save_doxdr_shape;
	goto cleanup;
      }
      doxdr_tsdr = save_doxdr_tsdr;
      doxdr_shape = save_doxdr_shape;
    }
  }
  else if (xdrs->x_op is XDR_DECODE) {
    /* decode each table element and insert it */
    if (chsrep) {
      /* special case to read packed external representation for */
      /* charstrings */
      counter gotsize;
      
      if (!xdr_bytes(xdrs, &chs_elements, &gotsize, size))
	goto cleanup;
      if (gotsize isnt size)
	goto cleanup;
    }
    for (i = 0; i < size; i++) {
      if (chsrep)
	newobj.value.ord_enum = chs_elements[i];
      else {
	newobj.tsdr = tbl->tsdr;
	if (!(*tbl->tsdr->xdr)(xdrs, &newobj, 
			       & shape->value.table->elt_shape)) {
	  /* doesn't matter what shape is for freeing; its not changed */
	  xdr_free(tbl->tsdr->xdr, &newobj, shape);
	  goto cleanup;
	}
      }
      foreach_tbl(tbl, tblnum, tblptr) {
	if (chsrep and (tblnum is ORDER_TBL))
	  retcode = Normal;
	else
	  retcode = (*tblptr->des->insert)(tbl, tblnum, newobj.value);
	if (retcode isnt Normal) {
	  forprev_tbl(tbl, ptblnum, ptblptr, tblnum)
	    (*ptblptr->des->uninsert)(tbl, ptblnum, newobj.value);
	  (*tbl->tsdr->finalize)(newobj.value, F_DISCARD, nil);
	  goto cleanup;
	}
      }
      tbl->size++;
    }
  }
  return(XDR_OK);
  
 cleanup:
  /* something failed... xdr_free() will be explicitly called to */
  /* free up anything that got allocated */
  return(XDR_FAIL);
  
}


void
free_shape_table(shape)
shapep shape;
{
  if (shape->value.table isnt nil) {
    free_shape(& shape->value.table->elt_shape);
    dispose(shape->value.table, shape_table);
  }
}
  
void
xdrfree_table(xdrs, valp)
XDR *xdrs;
valcell *valp;
{
    if (valp->table isnt nil) {
	/* if table has elements, then it's a complete, consistent */
	/* table and we can use normal finalization stuff */
	if (valp->table->size isnt 0)
	  fin_table(*valp, F_FREE, (schedblock *) nil); 
	else {
	    /* free any pieces that may have been built */
	    if (valp->table->keyset.table isnt nil)
	      (void) xdr_lookup(xdrs, &valp->table->keyset);
	    if (valp->table->indexset.table isnt nil)
	      (void) xdr_lookup(xdrs, &valp->table->indexset);
	    /* free up storage for the table structure itself */
	    { freemain(valp->table, sizeof(dfd_table) + 
		       sizeof(tblinfo)*(valp->table->tblcount - ARBSIZE)); }
	}
    }
}

static xdr_status
xdr_lookup(xdrs, valp)
XDR *xdrs;
valcell *valp;
{
    flag islookup;		/* is there any lookup info here? */
    shapeobj shape;
    object obj;
    xdr_status rc;

    if (xdrs->x_op is XDR_DECODE)
      valp->table = nil;	/* in case we die before allocating anything */

    if (xdrs->x_op is XDR_ENCODE)
      islookup = valp->table isnt nil;

    if (!myxdr_boolean(xdrs, &islookup))
      return(XDR_FAIL);
    
    if (islookup) {
      /* ok, so the recursive call is a little more complex with the
         new interface. */
      shape.tsdr = &dr_bottom;
      obj.tsdr = &dr_table;
      obj.value = *valp;
      rc = hxdr_table(xdrs, &obj, &shape);
      free_shape(&shape);
      *valp = obj.value;
      return(rc);
    }
    else
      return(XDR_OK);
}


xdr_status
copy_lookup(xdrs, valp, srcvalp)
XDR *xdrs;
valcell *valp, *srcvalp;
{
    flag islookup;		/* is there any lookup info here? */

    /* this should only be called in XDR_DECODE */

    if (xdrs->x_op is XDR_DECODE)
      valp->table = nil;	/* in case we die before allocating anything */

    islookup = srcvalp->table isnt nil;

    if (islookup)
      /* would lazy cp_table work okay?? */
      if (deep_cp_table(valp, *srcvalp) isnt Normal)
        return(XDR_FAIL);

    return(XDR_OK);
}


/* following routine only used for XDR_ENCODE operations */

/*ARGSUSED*/
static xdr_status
doxdr(tbl, tblnum, val, elemcount, argv)
dfd_table *tbl;
trepnum tblnum;
valcell val;
counter elemcount;
va_list argv;
{
    XDR *xdrs;
    object obj;

    xdrs = va_arg(argv, XDR *);

    obj.tsdr = doxdr_tsdr;
    obj.value = val;
    return((*tbl->tsdr->xdr)(xdrs, & obj, doxdr_shape));
}
    

xdr_status
xdr_tblinfo(xdrs, infop, tbl, tblnum, size)
XDR *xdrs;
tblinfo *infop;
dfd_table *tbl;
trepnum tblnum;
counter size;
{
    status xdr_tbldes();
    tbldes *des;
    valcell nulval;
    status chs_alloc_to_size(), vec_alloc_to_size();

    if (not xdr_tbldes(xdrs, & infop->des))
      return(XDR_FAIL);
    
    /* allocate/deallocate representation-specific info */
    switch(xdrs->x_op) {
    case XDR_ENCODE:		/* nothing to do on ENCODE */
      break;			
    case XDR_DECODE:		/* allocate for DECODE */
      des = infop->des;
      infop->des = nil;	/* 'not allocated' signal for XDR_FREE */
      nulval.record = nil;
      /* special case for vector or charstring since we know the */
      /* initial allocation size precisely */
      switch (des->number) {
      case table_rep_type__vector:
        if (!vec_alloc_to_size(tbl, tblnum, size))
          return(XDR_FAIL);
        break;
      case table_rep_type__charstring:
        if (!chs_alloc_to_size(tbl, tblnum, size))
	    return(XDR_FAIL);
        break;
      default:
        if (!(*des->alloc)(tbl, tblnum, nulval))
          return(XDR_FAIL);
        break;
      }
      infop->des = des;
      break;
    case XDR_FREE:		/* deallocate for FREE, but only if */
      if (infop->des isnt nil)/* allocated  */
        (*infop->des->finalize)(tbl, tblnum, SHALLOW, F_FREE,
                                (schedblock *) nil);
      break;
    }
    return(XDR_OK);
}

static xdr_status
copy_tblinfo(xdrs, infop, tbl, tblnum, size, des)
XDR *xdrs;
tblinfo *infop;
dfd_table *tbl;
trepnum tblnum;
counter size;
tbldes *des;
{
    status xdr_tbldes();
    valcell nulval;
    status chs_alloc_to_size(), vec_alloc_to_size();

    /* allocate/deallocate representation-specific info */
    switch(xdrs->x_op) {
    case XDR_ENCODE:		/* nothing to do on ENCODE */
      /* info->des should equal des.  If not, something is very wrong */
      if (infop->des isnt des) {
        fprintf(stderr, "big problems in copy_tblinfo\n");
        return(XDR_FAIL);
      }
      break;			
    case XDR_DECODE:		/* allocate for DECODE */
      infop->des = nil;	/* 'not allocated' signal for XDR_FREE */
      nulval.record = nil;
      /* special case for vector or charstring since we know the */
      /* initial allocation size precisely */
      switch (des->number) {
      case table_rep_type__vector:
        if (!vec_alloc_to_size(tbl, tblnum, size))
          return(XDR_FAIL);
        break;
      case table_rep_type__charstring:
        if (!chs_alloc_to_size(tbl, tblnum, size))
	    return(XDR_FAIL);
        break;
      default:
        if (!(*des->alloc)(tbl, tblnum, nulval))
          return(XDR_FAIL);
        break;
      }
      infop->des = des;
      break;
    /* copy_tblinfo shouldn't be called with XDR_FREE */
    }
    return(XDR_OK);
}


static xdr_status
xdr_tbldes(xdrs, despp)
XDR *xdrs;
tbldes **despp;
{
    trepnum desnum;

    if (xdrs->x_op is XDR_ENCODE)
      desnum = (*despp)->number;

    if (!xdr_int(xdrs, &desnum))
      return(XDR_FAIL);

    if (xdrs->x_op is XDR_DECODE)
      *despp = tbldescriptors[desnum];

    return(XDR_OK);
}

/******************************************************************************
 *                     Ordered Table Operations                               *
 *****************************************************************************/

NILOP(o_insert_at)
{
    predef_exception retcode;

    OPCHK(DstObj,table);
    OPCHK(Src2Obj,integer);
    if (Src2.integer > Dst.table->size or Src2.integer < 0)
      retcode = RangeError;
    else
      retcode = insertfunc(DstObj, SrcObj, TRUE, Src2.integer);
    if (retcode isnt Normal)
      raise_builtin(retcode);
}


NILOP(o_merge_at)
{
    predef_exception retcode;

    OPCHK(DstObj,table);
    OPCHK(Src1Obj,table);
    OPCHK(Src2Obj,integer);
    if (Src2.integer > Dst.table->size or Src2.integer < 0)
      raise_builtin(RangeError);
    else
      if ((retcode = mergefunc(DstObj, SrcObj, TRUE, Src2.integer))
	  isnt Normal)
	raise_builtin(retcode);
}

/* Note: remove_at raises NotFound instead of RangeError since it is used 
   for statements like

     remove e from r in t where (position of r = n);

   If N is out of the range of T, this will simply raise a NotFound exception.
 */

NILOP(o_remove_at)
{
    predef_exception retcode;
    predef_exception remove_at_func();

    OPCHK(Src1Obj,table);
    OPCHK(Src2Obj,integer);

    retcode = remove_at_func(DstObj, Src1Obj, Src2.integer);
    if (retcode isnt Normal) {
	raise_builtin(retcode);
	return;
    }
}

/* remove_at_func is called directly by generated C-code. */
predef_exception
remove_at_func(dstobj, srcobj, ix)
objectp dstobj;
objectp srcobj;
int ix;
{
    predef_exception retcode;
    trepnum tblnum;
    tblinfo *tblptr;

    if (ix >= srcobj->value.table->size or ix < 0)
      retcode = NotFound;
    else
      retcode = CopyOnWrite(srcobj); /* make private copy if shared table */
    if (retcode is Normal) {
      ordfunc(srcobj->value.table, remove_at)
	(srcobj->value.table, ORDER_TBL, & dstobj->value, ix);

      for_lookups(srcobj->value.table, tblnum, tblptr)
 	(*tblptr->des->remove)
	  (srcobj->value.table, tblnum, dstobj->value, NO_TBL, NO_TBL,
	   (position *) nil);
				/* just remove it from each table.  each */
				/*  implementation of remove must guarantee */
				/*  not to raise depletion. */

      srcobj->value.table->size--;
      dstobj->tsdr = srcobj->value.table->tsdr;
    }

    return(retcode);
}

NILOP(o_scan_position)
{
    int do_table_position();

    counter elemcounter;

    OPCHK(Src2Obj,table);
    (void) doforeach(Src2.table)(Src2.table, ORDER_TBL,
				 do_table_position, &elemcounter, CONT_FOREACH,
				 Src1);
    Dst.integer = (dfd_integer) elemcounter;
    set_init(DstObj, dr_integer);
}


/* do_table_position is referenced directly by generated C-code. */
int
do_table_position(srctable, srctblnum, val, elemcounter, argv)
dfd_table *srctable;
trepnum srctblnum;
valcell val;
counter elemcounter;
va_list argv;
{
    valcell findval;
			  
    findval = union_va_arg(argv, valcell);

    if (val.nominal is findval.nominal) 
      return(STOP_FOREACH);
    else
      return(CONT_FOREACH);
}


NILOP(o_position)
{
    inspect_frame *frame;
    dfd_table *thetable;

    frame = get_selector_frame(SrcObj, args->sched);
    thetable = frame->inspectee->value.table;

    Dst.integer = frame->rmvcount +
        ordfunc(thetable, position_of)(thetable, ORDER_TBL, frame->pos);

    set_init(DstObj, dr_integer);
}


NILOP(o_lookup_at)
{
    void re_finalize();

    extern flag cherm_flag;

    OPCHK(Src1Obj,table);
    OPCHK(Src2Obj,integer);
    if (Src2.integer < 0 or Src2.integer >= Src1.table->size) {
	raise_builtin(NotFound);	/* NotFound, not RangeError, like remove */
	return;
    }

    if (not cherm_flag)
      re_finalize(DstObj, F_DISCARD, args->sched);
				/* finalize the value of the destination; */
    ordfunc(Src1.table, lookup_at)(Src1.table, ORDER_TBL, & Dst,
				   Src2.integer);
    DstObj->tsdr = qdatarepmap[Src1.table->tsdr->number];
}

NILOP(o_lookup_at_or_goto)
{
    void re_finalize();

    extern flag cherm_flag;

    OPCHK(Src1Obj,table);
    OPCHK(Src2Obj,integer);
    if (Src2.integer < 0 or Src2.integer >= Src1.table->size) {
	args->nextop = args->qualifiers.integer;
	return;
    }

    if (not cherm_flag)
      re_finalize(DstObj, F_DISCARD, args->sched);
				/* finalize the value of the destination; */
    ordfunc(Src1.table, lookup_at)(Src1.table, ORDER_TBL, & Dst,
				   Src2.integer);
    DstObj->tsdr = qdatarepmap[Src1.table->tsdr->number];
}

NILOP(o_concat)
{
    void re_finalize();

    object src1copy;
    object src2copy;
    predef_exception retcode;
    extern flag cherm_flag;
    
    OPCHK(Src1Obj,table);
    OPCHK(Src2Obj,table);

    set_bottom(&src1copy);
    set_bottom(&src2copy);

    /* make copies of source operands to pass on to merge */
    if ((retcode = cp_table(&src1copy.value, Src1)) isnt Normal) goto cleanup;
    set_init(&src1copy, dr_table);
    if ((retcode = cp_table(&src2copy.value, Src2)) isnt Normal) goto cleanup;
    set_init(&src2copy, dr_table);

    /* pretend this is a MERGE operation */
    retcode = mergefunc(&src1copy, &src2copy, FALSE, 0);
    if (retcode isnt Normal) goto cleanup;

    if (not cherm_flag)
      re_finalize(DstObj, F_DISCARD, args->sched);
				/* finalize the value of the destination; */
    *DstObj = src1copy;
    return;

  cleanup:
    Finalize(&src1copy, F_DISCARD, args->sched); /* kill the temp copies */
    Finalize(&src2copy, F_DISCARD, args->sched);
    raise_builtin(retcode);		/* and raise appropriate exception */
}


/* che_concat is called directly by generated C-code. */
predef_exception
che_concat(dstobj, src1obj, src2obj, sched)
objectp dstobj;
objectp src1obj;
objectp src2obj;
schedblock *sched;
{
    void re_finalize();

    object src1copy;
    object src2copy;
    predef_exception retcode;
    
    set_bottom(&src1copy);
    set_bottom(&src2copy);

    /* make copies of source operands to pass on to merge */
    if ((retcode = cp_table(&src1copy.value, src1obj->value)) isnt Normal)
      goto cleanup;
    set_init(&src1copy, dr_table);
    if ((retcode = cp_table(&src2copy.value, src2obj->value)) isnt Normal)
      goto cleanup;
    set_init(&src2copy, dr_table);

    /* pretend this is a MERGE operation */
    retcode = mergefunc(&src1copy, &src2copy, FALSE, 0);
    if (retcode isnt Normal) goto cleanup;

    re_finalize(dstobj, F_DISCARD, sched);
				/* finalize the value of the destination; */
    *dstobj = src1copy;
    return(Normal);

  cleanup:
    Finalize(&src1copy, F_DISCARD, sched); /* kill the temp copies */
    Finalize(&src2copy, F_DISCARD, sched);
    return(retcode);		/* and return appropriate exception */
}

/******************************************************************************
 *                   Keyed/Indexed Table Operations                           *
 *****************************************************************************/

NILOP(o_find)
{
    status findfunc();

    OPCHK(Src1Obj,table);
    if (findfunc(DstObj, Src1Obj, & args->operandstack[Src2Pos], args->sched,
		 args->qualifiers.integer) is FAILURE)
      raise_builtin(NotFound);
}


NILOP(o_find_or_goto)
{
    status findfunc();

    OPCHK(Src1Obj,table);
    if (findfunc(DstObj, Src1Obj, & args->operandstack[Src2Pos], args->sched,
		 vdot(args->qualifiers, integer_pair__int_one).integer)
	is FAILURE)
      args->nextop = 
	vdot(args->qualifiers, integer_pair__int_two).integer;
}


status
findfunc(dstobj, src1obj, src2objs, sched, keynum)
objectp dstobj;
objectp src1obj;
object **src2objs;
schedblock *sched;
int keynum;
{
    void re_finalize();

    int findrep;
    dfd_table *t;
    tbldes *d;
    object foundobj;
    extern flag cherm_flag;

    t = src1obj->value.table;
    findrep = keynumtorepnum(t, keynum);
    d = t->tbls[findrep].des;

    if ((* d->find)(t, findrep, src2objs, & foundobj)
	is SUCCESS) {

        if (not cherm_flag)
	  re_finalize(dstobj, F_DISCARD, sched);
				/* finalize the value of the destination; */

	*(dstobj) = foundobj;	/* "initialize" quopy result */
	dstobj->tsdr = qdatarepmap[t->tsdr->number];
	return(SUCCESS);
    }
    else
      return(FAILURE);
}

/******************************************************************************
 *                         Indexed Table Operations                           *
 *****************************************************************************/

NILOP(o_initidxfind)
{
    void re_finalize();
    predef_exception init_selector();
    predef_exception retcode;

    int repnum;
    extern flag cherm_flag;

    OPCHK(Src1Obj,table);
    if (not cherm_flag)
      re_finalize(DstObj, F_DISCARD, args->sched);
				/* finalize the value of the destination; */
    repnum = keynumtorepnum(Src.table, args->qualifiers.integer);
    retcode = init_selector(DstObj, Src1Obj, & args->operandstack[Src2Pos],
			    args->sched, FIND, repnum, 0);
    if (retcode isnt Normal)
      raise_builtin(retcode);
}


NILOP(o_idxfind_or_err)
{
    predef_exception idxfindnext();

    predef_exception rc;

    OPCHK(SrcObj,table);
    if ((rc = idxfindnext(DstObj, SrcObj, args->sched)) isnt Normal)
      raise_builtin(rc);
}


NILOP(o_idxfind_or_goto)
{
    predef_exception idxfindnext();

    predef_exception rc;

    OPCHK(SrcObj,table);
    switch (rc = idxfindnext(DstObj, SrcObj, args->sched)) {
      case Normal: {
	  return;
      }

      case NotFound: {
	  args->nextop = args->qualifiers.integer;
	  return;
      }

      default: {
	  raise_builtin(rc);
	  return;
      }
    }
    /*NOTREACHED*/
}


NILOP(o_endidxfind)
{
    void end_selector();

    OPCHK(SrcObj,table);
    end_selector(DstObj, SrcObj, args->sched, FIND);
}


/* idxfindnext is called directly by generated C-code. */
predef_exception
idxfindnext(dstobj, srcobj, sched)
objectp dstobj;
objectp srcobj;
schedblock *sched;
{
    inspect_frame *get_selector_frame();

    inspect_frame *frame;
    tblinfo *tblptr;
    predef_exception retcode;

    frame = get_selector_frame(dstobj, sched);
    tblptr = & frame->inspectee->value.table->tbls[frame->repnum];
    
    retcode = (*tblptr->des->findnext)
      (srcobj->value.table, frame->repnum, & frame->pos, & dstobj->value);

    if (retcode is Normal)
      dstobj->tsdr = qdatarepmap[srcobj->value.table->tsdr->number];

    return(retcode);
}



valcell
get_key(t, tblnum)
dfd_table *t;
trepnum tblnum;
{
    return(get_elem(t->keyset, tblnum - firstlookup(t)));
}
