-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: rmanager.d
-- Author: David F. Bacon, Andy Lowry, from Rob Strom
-- SCCS Info: @(#)rmanager.d	1.2 3/13/90

-- Interfaces between the resource manager and its clients

rManager: using (common) definitions
    
  -- ports to access resource manager
  rManager: record (
    post: postResourceFn,	-- post a resource (by copying it)
    insert: insertResourceFn,	-- post a resource (by moving it)
    get: getResourceFn,		-- get a copy of a posted resource
    remove: getResourceFn,	-- remove a posted resource and return it
    delete: deleteResourceFn	-- delete a resource 
  );

  -- Posting a resource causes a copy of the resource to be installed
  -- in the resource manager for later access by other processes
  postResource: callmessage (
    resourceName: charString,	-- name being given to resource
    resource: polymorph,	-- the resource itself (usually)
    accessFunction: accessFn	-- access control function
  ) 
  constant(resourceName, resource, accessFunction)
  exit {full}
  exception duplicateName {full};
    
  postResourceQ: inport of postResource {full};
  postResourceFn: outport of postResourceQ;
    
  -- Inserting a resource causes the resource itself to be installed
  -- (i.e. moved) into the resource manager for later access (via
  -- 'remove') by another process.  This is used instead of 'post'
  -- when the resource is uncopyable.
  insertResource: callmessage (
    resourceName: charString,
    resource: polymorph,
    accessFunction: accessFn
  ) constant(resourceName, accessFunction)
  exit {init(resourceName), init(accessFunction)}
  exception duplicateName {full};
    
  insertResourceQ: inport of insertResource {full};
  insertResourceFn: outport of insertResourceQ;
    
    
  -- 'get' acquires a resource from the resource manager, provided
  -- access is granted by the access function supplied when the
  -- resource was posted.  Use 'remove' instead for uncopyable resources
  -- that were installed via 'insert'
  getResource: callmessage (
    resourceName: charString,	-- name of the resource
    parameter: charString,	-- parameter (e.g. passwd, type of access)
    resource: polymorph		-- the resource granted
  )
  constant(resourceName, parameter)
  exit {full}
  exception notFound {init(resourceName), init(parameter)}
  exception accessDenied {init(resourceName), init(parameter)};
    
  getResourceQ: inport of getResource {init(resourceName), init(parameter)};
  getResourceFn: outport of getResourceQ;
    
    
  -- Use 'delete' to take a resource out of the repository.  Only the
  -- user who posted (inserted) a resource may delete it.
  deleteResource: callmessage (
    resourceName: charString	-- name of resource
  )
  constant(resourceName)
  exit{full}
  exception notOwner{full};
    
  deleteResourceQ: inport of deleteResource {full};
  deleteResourceFn: outport of deleteResourceQ;
    
  -- Interface for access functions:  Given a user name, a resource,
  -- and a parameter string supplied in the 'get' ('remove') request,
  -- the access function must either deny access via an exception, or
  -- wrap up a resource (not necessarily the value that was posted)
  -- and return it.
  access: callmessage (
    userName: charString,	-- user requesting access
    postedResource: polymorph,	-- parameter that was posted (or inserted)
    parameter: charString,	-- arbitrary parameter from requestor
    resource: polymorph		-- resource granted to requestor
  )
  constant (userName, parameter)
  exit {init(userName), init(parameter), init(resource)}
  exception accessDenied 
      {init(userName), init(postedResource), init(parameter)};
    
  accessQ: inport of access
      {init(userName), init(postedResource), init(parameter)}; 
  accessFn: outport of accessQ;
    
    
end definitions
