-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: fixfull.p
-- Author: Rob Strom
-- SCCS Info: @(#)fixfull.p	1.8 1/9/92

-- Procedure to fix a formal typestate which might contain full
--    by replacing all instances of full by the expansion
--    and to insert init(*) if appropriate
-- Algorithm:
-- 1. For each attribute
--    1.1. If it's not FULL, copy it as is
--    1.2. If it's FULL, 
--       1.2.1. get the type of its last component
--       1.2.2. insert INIT(that component) and recursively all components
--              of records/callmessages [if INIT(that component) is already
--              present, just ignore DuplicateKey]
--          1.2.2.1. if we encounter a typename we've seen before (meaning
--                   that we're in a looping structure) generate an error
-- 2. If there is any attribute, insert init(*) if not already present

FixFull : USING(checkdefs_internal, FixFullInt, errors, positions) 
  PROCESS ( FixFullInit : FixFullInport)
  DECLARE
    FP: FixFullCall ;
    NewFormalTypestate: Formal_Typestate; -- result formal typestate
    Type: TypeName; -- of component being examined
    ExpandFull: ExpandFullOutport;
    Nothing: empty;
  BEGIN
    ExpandFull <- PROCEDURE OF PROCESS  (ExpandFullInit: ExpandFullInport)
      DECLARE
        FP: ExpandFullCall;
      BEGIN
        RECEIVE FP FROM ExpandFullInit;
      
        block
          begin
            INSERT (EVALUATE FA: Formal_Attribute FROM
                    NEW FA;
                    UNITE FA.Attribute_Name.Init FROM EVALUATE Empty: Empty FROM END;
                    NEW FA.Parameters;
                    INSERT COPY OF FP.FormalObject INTO FA.Parameters;
                  END) INTO FP.Formal;
          on (DuplicateKey)
            -- Can get here if have a typestate like {full(x), init(x.y)}.
            -- Should probably generate an error message like checkformaltypestate
            -- does with {init(x.y), init(x.y)}, but we'll be extra-special nice
            -- and ignore it.
          end block;
                    
        INSPECT Module IN FP.Definitions WHERE(boolean # (Module.Id = FP.Type.ModuleId))
          BEGIN
            INSPECT Definition IN Module.Type_Definitions WHERE(boolean # (Definition.Id = FP.Type.TypeId))
              BEGIN
                IF CASE OF Definition.Specification = 'recordtype' OR CASE OF Definition.Specification = 'callmessagetype'
                  THEN
                    FOR ComponentDeclaration IN Definition.Component_Declarations WHERE('true')
                      INSPECT
                        CALL FP.Self(FP.Definitions, 
                            FP.Formal, 
                            EVALUATE Subobj: Component_List FROM
                                Subobj := FP.FormalObject;
                                
                                -- Subobj is unkeyed, so no duplicatekey poss.
                                INSERT COPY OF ComponentDeclaration.Id INTO 
                                   Subobj;
                              END,
                            ComponentDeclaration.Type, 
                            evaluate newseen:typeslist from
                                newseen := FP.seen;
                                
                                block begin
                                    insert copy of ComponentDeclaration.Type into
                                       newseen;
                                  on (DuplicateKey)
                                    exit circular;
                                  end block;
                              end,
                            FP.Self);
                      END FOR;
                  END IF;
              END INSPECT;
          END INSPECT;
        RETURN FP;
      on exit (circular)
        return FP exception circular;
      on (ExpandFullCall.circular)
        return FP exception circular;
      on (NotFound)
        -- bogus Module or Definition; checkdefs will catch this.
        return FP;
      END PROCESS;
      
    -- 1.
    RECEIVE FP FROM FixFullInit ;
    
    NEW NewFormalTypestate;
    FOR FormalAttr IN FP.Formal WHERE('true')
      INSPECT
        IF CASE OF FormalAttr.Attribute_Name = 'full'
          THEN
            block begin
                -- 1.2.1.
                Type := FP.RootType;
                INSPECT FormalObject IN FormalAttr.Parameters WHERE(POSITION OF FormalObject = 0)
                  BEGIN
                    FOR Component IN FormalObject WHERE('true')
                      INSPECT
                        INSPECT Module IN FP.Definitions WHERE(boolean # (Module.Id = Type.ModuleId))
                          BEGIN
                            INSPECT Definition IN Module.Type_Definitions WHERE(boolean # (Definition.Id = Type.TypeId))
                              BEGIN
                                INSPECT ComponentDeclaration IN Definition.Component_Declarations WHERE(ComponentDeclaration.Id = Component)
                                  BEGIN
                                    Type := ComponentDeclaration.Type;
                                  END INSPECT;
                              END INSPECT;
                          END INSPECT;
                      END FOR;
                    -- 1.2.2.
                    CALL ExpandFull(FP.Definitions,
                        NewFormalTypestate,
                        FormalObject,
                        Type,
                        evaluate newseen:typeslist from
                            new newseen;
                            insert copy of Type into newseen;
                          end,
                        ExpandFull);
                  END INSPECT;
              on (ExpandFullCall.circular)
                block
                  declare
                    error: error;
                    errObj: errorObject;
                  begin
                    error := FP.errortemplate;
                    unite errObj.charstring from
                    "'full' applied to infinite structure";
                    insert errObj into error.objects;
                    insert error into FP.errors;
                  end block;
              on (NotFound)
                -- bogus FormalObject, Module, or Definition; 
                -- checkdefs will catch this
              end block;
          ELSE
            INSERT COPY OF FormalAttr INTO NewFormalTypestate;
          END IF;
      END FOR;
    -- 2. 
    IF SIZE OF NewFormalTypestate > 0
      THEN
        BLOCK
          BEGIN
	    insert evaluate InitStar: Formal_Attribute FROM
	      NEW InitStar;
	      UNITE InitStar.Attribute_Name.Init FROM EVALUATE Empty: Empty FROM END;
	      NEW InitStar.Parameters;
	      INSERT EVALUATE Star: Component_List FROM
		NEW Star;
		END INTO InitStar.Parameters;
	      END INTO NewFormalTypestate;
	  ON (DuplicateKey)
	  END BLOCK;
      END IF;
    FP.Formal <- NewFormalTypestate;

    
    RETURN FP;
  END PROCESS
