-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: checkdefs.p
-- Author: David F. Bacon, Jim Russell
-- SCCS Info: @(#)checkdefs.p	1.9 1/15/92

      
checkDefs: using (checkDefs, checkdefs_internal, errors, positions
  ) 
  process (q: checkDefsQ) 
    
  declare
    args: checkDefs;
    error: error;
    errorTemplate: error;
    errObj: errorObject;
    modules: module_table;
    
    checkComp_List: checkcomponent_listFn;
    checkFormal_typestate: checkFormal_typestateFn;
    checkConstraints: checkconstraintsFn;
    fixfull: fixfullOutport;
  begin
    receive args from q;
    
    -- we expand abbreviations before checkdefs, and do shortening after
    -- now check the definitions
    
    -- first, create useful subroutines
    checkFormal_typestate <- procedure of args.std.pathload("checkformal_typestate");
    
    checkComp_list <- procedure of args.std.pathload("checkcomponent_list");
    fixfull <- procedure of args.std.pathload("fixfull");
    
    checkconstraints <- create of args.std.pathload("checkconstraints");
    
    -- initialize errors
    new args.errors;

    new errorTemplate;
    errorTemplate.code := 'definition error';
    unite errorTemplate.position.apos from 
      (evaluate pos: aposition from
            new pos; 
            pos.clause := unique;
            pos.statement := unique;
          end);
    new errorTemplate.objects;
    unite errObj.moduleid from copy of args.module.id;
    insert errObj into errorTemplate.objects;
    
    -- build table of definitions modules consisting of the checked modules
    -- plus the one we're checking.  These are all the modules we may need 
    -- to resolve against.
    new modules;
    for defmod in args.defs[]
      inspect
        insert copy of defmod into modules;
      end for;
    insert copy of args.module into modules;
    
    for typedef in args.module.type_definitions[] inspect
        if size of typedef.component_declarations <> 0 then
            if ((case of typedef.specification <> 'recordtype') and
                       (case of typedef.specification <> 'varianttype') and
                       (case of typedef.specification <> 'callmessagetype'))
              then
                error := errorTemplate;
                unite errObj.charstring from 
                "components given for non-record/variant/callmessage type";
                insert errObj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              else
                for comp_decl in typedef.component_declarations[] 
                  inspect
                    block
                      begin
                        inspect defmod in modules[comp_decl.type.moduleid]
                          begin
                            if not (exists of defmod.type_definitions
                                       [comp_decl.type.typeid]
                                  ) then
                                error := errorTemplate;
                                unite errObj.charstring from 
                                "component declaration type not found in module";
                                insert errObj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                unite errobj.componentid from copy of comp_decl.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                              end if;
                          end inspect;
                      on (NotFound)
                        error := errorTemplate;
                        unite errObj.charstring from 
                        "module not found for component declaration";
                        insert errObj into error.objects;
                        unite errobj.typeid from copy of typedef.id;
                        insert errObj into error.objects;
                        unite errobj.componentid from copy of comp_decl.id;
                        insert errObj into error.objects;
                        insert error into args.errors;
                      end block;
                  end for;
              end if;
          else
            -- ? check that typedef is not record, variant or cm ?
            -- No, apparantly not.  Thus, be careful when examining
            -- these types that the component_declarations[] might be
            -- empty.
          end if;
        select case of typedef.specification
               
          where ('booleantype')
            reveal typedef.specification.boolean;
            if typedef.specification.boolean.true_name =
                   typedef.specification.boolean.false_name then
                error := errortemplate;
			    unite errobj.charstring from 
                "boolean with identical names";
			    insert errobj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
			    insert error into args.errors;
              end if;
          where('realtype')
            -- checks on accuracy_info go here
            
          where ('varianttype')
            block
              declare
                enumsize: integer;
              begin
                reveal typedef.specification.variant_info;
                inspect defmod in modules[typedef.specification.variant_info.case_type.moduleid]
                  begin
                    inspect enumdef in defmod.type_definitions[typedef.specification.variant_info.case_type.typeid]
                      begin
                        if case of enumdef.specification <> 'enumerationtype' 
                          then
                            error := errortemplate;
                            unite errobj.charstring from 
                            "variant of non-enumeration type";
                            insert errobj into error.objects;
                            unite errobj.typeid from copy of typedef.id;
                            insert errObj into error.objects;
                            insert error into args.errors;
                          else
                            -- variant_info.case_mapping is 
                            -- keyed by both component_id and case_id, so we
                            -- don't have to check uniqueness.  All we need to 
                            -- check is that the sizes of the components list 
                            -- and case_mapping are the same as the size of
                            -- the enumeration type, and also that each entry
                            -- for case_id in the case_mapping points to a
                            -- valid component of the enumeration.
                            reveal enumdef.specification.enumeration;
                            enumsize <- size of enumdef.specification.enumeration.values;
                            
                            if size of typedef.component_declarations 
                                   <> enumsize 
                              then
                                error := errortemplate;
                                unite errobj.charstring from 
                                "variant components has missing or extra cases";
                                insert errobj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                              end if;
                            if size of typedef.specification.variant_info.case_mapping 
                                   <> enumsize
                              then
                                error := errortemplate;
                                unite errobj.charstring from 
                                "variant case_mapping has missing or extra cases";
                                insert errobj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                              end if;
                            for pinfo in typedef.specification.variant_info.case_mapping[]
                              inspect 
                                if pinfo.case_id < 0 or 
                                       pinfo.case_id > enumsize - 1 then
                                    error := errortemplate;
                                    unite errobj.charstring from 
                                    "variant case_id out of range";
                                    insert errobj into error.objects;
                                    unite errobj.typeid from copy of typedef.id;
                                    insert errObj into error.objects;
                                    insert error into args.errors;
                                  end if;
                                block
                                  begin
                                    inspect cdec in 
                                           typedef.component_declarations 
                                           where (cdec.id = pinfo.component_id)
                                      begin
                                        -- check that pinfo.case_typestate ok
                                        -- cdec.type may not be 'findable',
                                        -- but we've already signaled an 
                                        -- error if this is the case.
                                        -- checkFormal_typestate should still
                                        -- be okay (and not signal an error)
                                        call checkFormal_typestate(pinfo.case_typestate,
                                            cdec.type,
                                            modules,
                                            args.typenames,
                                            evaluate errtem: error from
                                                block declare
                                                    errobj2: errorobject;
                                                  begin
                                                errtem := errortemplate;
                                                unite errobj2.charstring from 
                                                " in variant case_typestate";
                                                insert errobj2 into errtem.objects;
                                                unite errobj2.typeid from copy of typedef.id;
                                                insert errobj2 into errtem.objects;
                                                end block;
                                            end,
                                            args.errors,
                                            checkComp_list, 
                                            fixfull);
                                      end inspect;
                                  on (NotFound)
                                    error := errortemplate;
                                    unite errobj.charstring from 
                                    "variant case_mapping.component_id does not match any component";
                                    insert errobj into error.objects;
                                    unite errobj.typeid from copy of typedef.id;
                                    insert errObj into error.objects;
                                    insert error into args.errors;
                                  end block;
                              end for;
                          end if;
                      end inspect;
                  end inspect;
                
              on (NotFound)  
                -- either moduleid or typeid was out of range in above inspects
                error := errortemplate;
                unite errobj.charstring from 
                "moduleid or typeid in variant_info.case_type not found";
                insert errobj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              end block;
            
          where ('tabletype')
            reveal typedef.specification.table_info;
            -- check that element_type exists
            block
              begin
                inspect defmod in modules[typedef.specification.table_info.element_type.moduleid]
                  begin
                    if not (exists of defmod.type_definitions[typedef.specification.table_info.element_type.typeid])
                      then
                        error := errortemplate;
                        unite errobj.charstring from 
                        "table element type not found in module";
                        insert errobj into error.objects;
                        unite errobj.typeid from copy of typedef.id;
                        insert errObj into error.objects;
                        insert error into args.errors;
                      else
                        -- check that element_typestate is okay
                        call checkFormal_typestate(typedef.specification.table_info.element_typestate,
                            typedef.specification.table_info.element_type,
                            modules,
                            args.typenames,
                            evaluate errtem: error from
                            block declare
                                errobj2: errorobject;
                              begin
                                errtem := errortemplate;
                                unite errobj2.charstring from 
                                " in table element_typestate";
                                insert errobj2 into errtem.objects;
                                unite errobj2.typeid from copy of typedef.id;
                                insert errobj2 into errtem.objects;
                              end block;
                              end,
                            args.errors,
                            checkComp_list,
                            fixfull);
                      end if;
                  end inspect;
              on (NotFound)
                error := errortemplate;
                unite errobj.charstring from 
                "table element definitions module not found";
                insert errobj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              end block;
            -- Check many things about keys:
            -- 1) Each formal_object must be nonempty (i.e. keys () NG)
            -- 2) Component_list makes sense (i.e. first is a component of
            --    the element_type, following are each components of the
            --    preceeding).
            -- 3) No redundant keys.  Redundant keys are when one key 
            --    'implies' another, in the sense that if two elements are
            --    unique in the first key, they must be unique in the second.
            --    This can happen both 'externally' and 'internally'.
            --    External redundancy is between
            --    separate keys - e.g. (A) (A, B), or (A) (A), or (A.X) (A),
            --    or (A) (*), or (A.X) (*).  Internal redundancy is within a 
            --    key - e.g. (A, A), or (A.X, A), or (A, *), or (A.X, *).
            --    Note that (A, B.X) (A.X, B) is not redundant.
            --    Formally, key K is internally redundant if:
            --      there are component lists x, y in K with x 'implies' y.
            --    If keys K and J are not internally redundant, 
            --    K and J are externally redundant if:
            --      K 'implies' J, or J 'implies' K.
            --    Key K 'implies' key J if:
            --      K and J are not internally redundant, AND
            --      For all component lists x in K, 
            --      there exists a component list y in J such that
            --      x 'implies' y.
            --    Component list x 'implies' component list y if:
            --      y is a prefix of x.
            --
            --    Note that the handling of (*) is taken care of automatically.
            --    Any component list (or key) implies *, and since * is
            --    represented as an empty component list, it is a prefix
            --    of any component list.
            --    
            block
              declare
                do_external_check: predefined!boolean;  -- flag for ext chk
                compare: CompareComponent_ListsFn;      -- useful subroutine
                implies: impliesFn;             -- another useful subroutine
                newattr: predefined!formal_attribute; -- dummy attribute
                empty: empty;               -- (unique) element of type empty
              begin
                do_external_check <- 'true';
                
                -- create some useful subroutines
                compare <- procedure of process (init: CompareComponent_ListsQ)
                  declare
                    args: CompareComponent_Lists;
                  begin
                    receive args from init;
                    for cid in args.L[] inspect
                        if cid <> args.R[position of cid] then
                            exit normalreturn; -- 'normal' return
                          end if;
                      end for;
                    if (size of args.L) < (size of args.R) then
                        return args exception LprefixR;
                      else
                        return args exception LequalsR;
                      end if;
                  on exit (normalreturn)
                    return args;
                  on (NotFound)  
                    -- args.R[position of cid] not found, so R is 
                    -- shorter than L
                    return args exception RprefixL;
                  end process;
                
                implies <- procedure of process (init: impliesQ)
                  declare
                    args: implies;
                  begin
                    receive args from init;
                    for lclist in args.L[] inspect
                        block
                          begin
                            for rclist in args.R[] inspect
                                block
                                  begin
                                    -- check if l => r
                                    call args.compare(lclist, rclist);
                                  on (CompareComponent_Lists.LprefixR)
                                    -- do nothing; continue examining args.R[]
                                  end block;
                              end for;
                            -- if we make it here, then l and r are different,
                            -- or LprefixR.
                            exit return_false;
                          on (CompareComponent_Lists.RprefixL, 
                                CompareComponent_Lists.LequalsR)
                            -- do nothing; continue examining args.L[]
                          end block;
                      end for;
                    args.result <- 'true';
                    return args;
                  on exit(return_false)
                    args.result <- 'false';
                    return args;
                  end process;
                
                -- get to work
                for key in typedef.specification.table_info.keys[] inspect
                    if size of key = 0 then
                        error := errortemplate;
                        unite errobj.charstring from 
                        "table has empty key";
                        insert errobj into error.objects;
                        unite errobj.typeid from copy of typedef.id;
                        insert errObj into error.objects;
                        insert error into args.errors;
                        do_external_check <- 'false';
                      else
                        -- check for internal redundancy, check out components,
                        -- and check that key is init in element_typestate
                        new newattr;
                        unite newattr.attribute_name.init from empty;
                        for component_list in key[] inspect
                            block
                              begin
                                for rclist in key where (position of rclist > 
                                           position of component_list)
                                  inspect
                                    call compare(component_list, rclist);
                                  end for;
                              on (CompareComponent_Lists.LprefixR, 
                                    CompareComponent_Lists.RprefixL, 
                                    CompareComponent_Lists.LequalsR)
                                error := errortemplate;
                                unite errobj.charstring from 
                                "internally redundant key in table";
                                insert errobj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                                do_external_check <- 'false';
                              end block;
                            block
                              declare
                                prev: predefined!typename;
                              begin
                                prev := typedef.specification.table_info.element_type;
                                call checkComp_list(component_list,
                                    prev,
                                    modules,
                                    evaluate errtem: error from
                            block declare
                                errobj2: errorobject;
                              begin
                                        errtem := errortemplate;
                                        unite errobj2.charstring from 
                                        " of key";
                                        insert errobj2 into errtem.objects;
                                        unite errobj2.typeid from copy of typedef.id;
                                        insert errobj2 into errtem.objects;
                                      end block;
                                      end,
                                    args.errors);
                              on (checkComponent_list.Error)
                                -- do nothing
                              end block;
                            new newattr.parameters;
                            insert copy of component_list into 
                               newattr.parameters;
                            if not exists of typedef.specification.table_info.element_typestate[newattr]
                              then
                                error := errortemplate;
                                unite errobj.charstring from 
                                "table key not init in element typestate";
                                insert errobj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                                do_external_check <- 'false';
                              end if;
                          end for;
                      end if;
                  end for;
                if (do_external_check) then
                    -- check for external redundancies
                    for lkey in typedef.specification.table_info.keys[] inspect
                        for rkey in typedef.specification.table_info.keys where
                               (position of lkey < position of rkey)
                          inspect
                            if implies(lkey, rkey, compare) then
                                error := errortemplate;
                                unite errobj.charstring from 
                                "externally redundant keys in table";
                                insert errobj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                              else 
                                if implies(rkey, lkey, compare) then
                                    error := errortemplate;
                                    unite errobj.charstring from 
                                    "externally redundant keys in table";
                                    insert errobj into error.objects;
                                    unite errobj.typeid from copy of typedef.id;
                                    insert errObj into error.objects;
                                    insert error into args.errors;
                                  end if;
                              end if;
                          end for;
                      end for;
                  end if;
              end block;
          where ('inporttype')
            reveal typedef.specification.inport_info;
            block
              begin
                inspect defmod in modules[typedef.specification.inport_info.message_type.moduleid]
                  begin
                    inspect tdef in defmod.type_definitions[typedef.specification.inport_info.message_type.typeid]
                      begin
                        -- check that message_typestate is okay
                        call checkFormal_typestate(typedef.specification.inport_info.message_typestate,
                            typedef.specification.inport_info.message_type,
                            modules,
                            args.typenames,
                            evaluate errtem: error from
                            block declare
                                errobj2: errorobject;
                              begin
                                errtem := errortemplate;
                                unite errobj2.charstring from 
                                " in inport message_typestate";
                                insert errobj2 into errtem.objects;
                                unite errobj2.typeid from copy of typedef.id;
                                insert errobj2 into errtem.objects;
                              end block;
                              end,
                            args.errors,
                            checkComp_list,
                            fixfull);
                        
-- Don't do this check any more, so shorten'ed modules will check OK
--                        
--                        if case of tdef.specification = 'callmessagetype' then
--                            -- check that message_typestate is greater than min
--                            reveal tdef.specification.callmessage_info;
--                            block
--                              begin
--                                inspect mexcept in tdef.specification.callmessage_info.exception_specifications[tdef.specification.callmessage_info.minimum]
--                                  begin
--                                    for mattr in mexcept.post_typestate[]
--                                      inspect
--                                        if not exists of typedef.specification.inport_info.message_typestate[mattr]
--                                          then
--                                            error := errortemplate;
--                                            unite errobj.charstring from 
--                                            "inport message typestate not at least corresponding callmessage's minimum";
--                                            insert errobj into error.objects;
--                                            unite errobj.typeid from copy of typedef.id;
--                                            insert errObj into error.objects;
--                                            insert error into args.errors;
--                                            
--                                            exit done;
--                                          end if;
--                                      end for;
--                                  end inspect;
--                              on exit (done)
--                                -- only need one error
--                              on (NotFound)
--                                -- mexcept not found.  Error will be signaled
--                                -- when callmessage checked, not here
--                              end block;                            
--                          end if;
                      end inspect;
                  end inspect;
              on (NotFound)
                error := errortemplate;
                unite errobj.charstring from 
                "inport message type or definitions module not found";
                insert errobj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              end block;
            
          where ('outporttype')
            reveal typedef.specification.outport_info;
            block
              begin
                inspect defmod in modules[typedef.specification.outport_info.moduleid]
                  begin
                    block
                      begin
                        inspect inportdef in defmod.type_definitions[typedef.specification.outport_info.typeid]
                          begin
                            if case of inportdef.specification 
                                   <> 'inporttype' then
                                error := errortemplate;
                                unite errobj.charstring from 
                                "outport of non-inport";
                                insert errobj into error.objects;
                                unite errobj.typeid from copy of typedef.id;
                                insert errObj into error.objects;
                                insert error into args.errors;
                              end if;
                          end inspect;
                      on (NotFound)
                        error := errortemplate;
                        unite errobj.charstring from 
                        "outport message type not found in module";
                        insert errobj into error.objects;
                        unite errobj.typeid from copy of typedef.id;
                        insert errObj into error.objects;
                        insert error into args.errors;
                      end block;
                  end inspect;
              on (NotFound)
                error := errorTemplate;
                unite errObj.charstring from 
                "outport message definitions module not found";
                insert errObj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              end block;
          where ('callmessagetype')
            reveal typedef.specification.callmessage_info;
            for cid in typedef.specification.callmessage_info.constants[]
              inspect
                -- check that cid is a component of the definition
                if not exists of cdecl in typedef.component_declarations
                       where (cid = cdecl.id)
                  then
                    error := errortemplate;
                    unite errobj.charstring from 
                    "constant component of callmessage not declared";
                    insert errobj into error.objects;
                    unite errobj.typeid from copy of typedef.id;
                    insert errObj into error.objects;
                    insert error into args.errors;
                  end if;
              end for;
            
            -- check the normal typestate
            -- make sure init(*) is present (checkformal_typestate will 
            -- do this if typestate is non-empty)
            if size of typedef.specification.callmessage_info.normal = 0
              then
                error := errortemplate;
                unite errobj.charstring from 
                "empty typestate in callmessage normal typestate (must be at least init(*))";
                insert errobj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              else
                call checkFormal_typestate(typedef.specification.callmessage_info.normal,
                    evaluate mytypename: predefined!typename from
                        new mytypename;
                        mytypename.moduleid := args.module.id;
                        mytypename.typeid := typedef.id;
                      end,
                    modules,
                    args.typenames,
                    evaluate errtem: error from
                            block declare
                                errobj2: errorobject;
                              begin
                        errtem := errortemplate;
                        unite errobj2.charstring from 
                        " in callmessage normal typestate";
                        insert errobj2 into errtem.objects;
                        unite errobj2.typeid from copy of typedef.id;
                        insert errobj2 into errtem.objects;
                      end block;
                      end,
                    args.errors,
                    checkComp_list,
                    fixfull);
              end if;
            for except in typedef.specification.callmessage_info.exception_specifications[]
              inspect
                -- check the exception typestates
                -- same as with normal typestate
                if size of except.post_typestate = 0
                  then
                    error := errortemplate;
                    unite errobj.charstring from 
                    "empty typestate in callmessage exception typestate (must be at least init(*))";
                    insert errobj into error.objects;
                    unite errobj.typeid from copy of typedef.id;
                    insert errObj into error.objects;
                    unite errobj.exceptionid from copy of except.exceptionid;
                    insert errObj into error.objects;
                    insert error into args.errors;
                  else
                    call checkFormal_typestate(except.post_typestate,
                        evaluate mytypename2: predefined!typename from
                            new mytypename2;
                            mytypename2.moduleid := args.module.id;
                            mytypename2.typeid := typedef.id;
                          end,
                        modules,
                        args.typenames,
                        evaluate errtem2: error from
                            block declare
                                errobj2: errorobject;
                              begin
                                errtem2 := errortemplate;
                                unite errobj2.charstring from 
                                evaluate message: charstring from
                                    if except.exceptionid = typedef.specification.callmessage_info.minimum
                                      then
                                        message <- " in minimum typestate";
                                      else
                                        message <- " in callmessage exception typestate";
                                      end if;
                                  end;
                                insert errobj2 into errtem2.objects;
                                unite errobj2.typeid from copy of typedef.id;
                                insert errobj2 into errtem2.objects;
                                unite errobj2.exceptionid from copy of except.exceptionid;
                                insert errobj2 into errtem2.objects;
                              end block;
                          end,
                        args.errors,
                        checkComp_list,
                        fixfull);
                  end if;
              end for;
            block
              begin
                inspect mexcept in typedef.specification.callmessage_info.exception_specifications[typedef.specification.callmessage_info.minimum]
                  begin
                    block
                      begin
-- Dont do this check any more                        
--                    -- check that normal and exceptions are at least minimum
--                        for mattr in mexcept.post_typestate[]
--                          inspect
--                            if not exists of typedef.specification.callmessage_info.normal[mattr]
--                              then
--                                error := errortemplate;
--                                unite errobj.charstring from 
--                                "callmessage normal exit typestate not at least minimum";
--                                insert errobj into error.objects;
--                                unite errobj.typeid from copy of typedef.id;
--                                insert errObj into error.objects;
--                                insert error into args.errors;
--                                
--                                exit done;
--                              end if;
--                          end for;
                        -- no error above
                        -- check that normal and minimum are same w.r.t.
                        -- init of constants
                        for nattr in typedef.specification.callmessage_info.normal 
                               where (case of nattr.attribute_name = 'initialized')
                          inspect
                            for clist in nattr.parameters
                                   where (position of clist = 0 and
                                       size of clist > 0)
                              inspect
                                if exists of cid in typedef.specification.callmessage_info.constants 
                                       where (cid = clist[0])
                                  then
                                    if not exists of mexcept.post_typestate[nattr]
                                      then
                                        error := errortemplate;
                                        unite errobj.charstring from 
                                        "callmessage minimum typestate lower than normal exit typestate w.r.t. init of constants";
                                        insert errobj into error.objects;
                                        unite errobj.typeid from copy of typedef.id;
                                        insert errObj into error.objects;
                                        insert error into args.errors;
                                        exit done;
                                      end if;
                                  end if;
                              end for;
                          end for;
                      on exit (done)
                        -- only need one error
                      end block;
                    for except in typedef.specification.callmessage_info.exception_specifications where (except <> mexcept)
                      inspect
                        block
                          begin
-- Dont do this check any more                            
--                            for mattr in mexcept.post_typestate[]
--                              inspect
--                                if not exists of except.post_typestate[mattr]
--                                  then
--                                    error := errortemplate;
--                                    unite errobj.charstring from 
--                                    "callmessage exception exit typestate not at least minimum";
--                                    insert errobj into error.objects;
--                                    unite errobj.typeid from copy of typedef.id;
--                                    insert errObj into error.objects;
--                                    unite errobj.exceptionid from copy of except.exceptionid;
--                                    insert errObj into error.objects;
--                                    insert error into args.errors;
--                                    
--                                    exit done;
--                                  end if;
--                              end for;
                            -- no error above
                            -- check that exception typestate and minimum
                            -- typestate are same w.r.t. init of constants
                            for xattr in except.post_typestate 
                                   where (case of xattr.attribute_name = 'initialized')
                              inspect
                                for clist in xattr.parameters
                                       where (position of clist = 0 and
                                           size of clist > 0)
                                  inspect
                                    if exists of cid in typedef.specification.callmessage_info.constants 
                                           where (cid = clist[0])
                                      then
                                        -- should call tsatc here to check to see if this extra init can
                                        -- be gotten rid of only by hiding variant cases; if so, this
                                        -- shouldn't signal an error.  As it now stands, this may 
                                        -- erroneously signal an error.
                                        if not exists of mexcept.post_typestate[xattr]
                                          then
                                            error := errortemplate;
                                            unite errobj.charstring from 
                                            "callmessage minimum typestate lower than exception exit typestate w.r.t. init of constants";
                                            insert errobj into error.objects;
                                            unite errobj.typeid from copy of typedef.id;
                                            insert errObj into error.objects;
                                            unite errobj.exceptionid from copy of except.exceptionid;
                                            insert errObj into error.objects;
                                            insert error into args.errors;
                                            exit done;
                                          end if;
                                      end if;
                                  end for;
                              end for;
                          on exit (done)
                            -- only need one error per exception
                          end block;
                      end for;
                    
                  end inspect;
              on (NotFound)
                error := errortemplate;
                unite errobj.charstring from 
                "callmessage minimum not found";
                insert errobj into error.objects;
                unite errobj.typeid from copy of typedef.id;
                insert errObj into error.objects;
                insert error into args.errors;
              end block;
            
          otherwise
          end select;
      end for;
  
  call checkconstraints(args.std, args.module.id, modules, args.typenames, 
      checkFormal_typestate, checkComp_list, fixfull, errorTemplate, 
      args.errors); 
  
  return args;
end process    
    

