/* (C) Copyright International Business Machines Corporation 23 January */
/* 1990.  All Rights Reserved. */
/*  */
/* See the file USERAGREEMENT distributed with this software for full */
/* terms and conditions of use. */
/* File: process.ch */
/* Author: David F. Bacon */
#ifndef lint
static char sccsinfo[] = "@(#)process.ch	1.25 2/17/92";
#endif

#include <stdio.h>
#include "cherm.h"
#include "storage.h"

#include "resolve.h"
#include "stack.h"
#include "lex-yacc.h"

#include "predefined.cd"
#include "positions.cd"
#include "posmap.cd"
#include "resproc.cd"
#include "errors.cd"

extern globinfo global;

#define DEFAULT_STMT_COUNT 0
				/* was 16 */

static stackp
  stmt_stack, sel_stack,
  prog_stack, env_stack,
  scope_stack, clause_stack,
  exit_stack, posmap_stack;

static objectp
  CCprog,
  CCenv,
  CCscope,
  CCclause,
  CCposmap,
  deferredConsumes;


objectp copyobj();
void consume();
void defer_consume();
void do_deferred_consumes();

hobject(Builtin_Emap, table);
hobject(Tempids, table);	/* table of rootid: ids of all temporaries */
hobject(Posmaps, table);

void
init_executables()
{
    void init_exec_printmap();

    hobject(Maprec, record);


    init_exec_printmap();

    stack_create(&stmt_stack);
    stack_create(&prog_stack);
    stack_create(&env_stack);
    stack_create(&scope_stack);
    stack_create(&clause_stack);
    stack_create(&sel_stack);
    stack_create(&exit_stack);
    stack_create(&posmap_stack);

    avl_new_table(Posmaps, firstelem_key);

    avl_new_table(Builtin_Emap, firstelem_key);
    avl_new_table(Tempids, whole_key);

    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "caseerror");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__CaseError);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "constrainterror");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__ConstraintError);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "constraintfailure");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__ConstraintFailure);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "depletion");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__Depletion);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "disconnected");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__Disconnected);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "dividebyzero");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__DivideByZero);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "duplicatekey");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__DuplicateKey);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "interfacemismatch");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__InterfaceMismatch);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "notfound");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__NotFound);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "polymorphmismatch");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__PolymorphMismatch);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "definitionerror");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__DefinitionError);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "rangeerror");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__RangeError);
    insert(Builtin_Emap, Maprec);
    new_record(Maprec, builtin_exception_map);
    chs_lit(Maprec@builtin_exception_map__name, "uncopyable");
    ordenum_lit(Maprec@builtin_exception_map__exception,
		 builtin_exception__Uncopyable);
    insert(Builtin_Emap, Maprec);

}

void
cleanup_executables()
{
  discard(Builtin_Emap);
  discard(Tempids);
}



static void
pop_scope()
{
    void pmap_pop_scope();
    void pop_clause();

    stack_pop(&scope_stack);
    CCscope = stack_top(&scope_stack);

    pop_clause();

    pmap_pop_scope();
}


objectp				/* returns constant scopeid */
get_scope_id()
{
    return(CCscope@Id);
}


static void
pop_clause()
{
    stack_pop(&clause_stack);
    CCclause = stack_top(&clause_stack);
}


static void
pop_env()
{
    pop_scope();
    stack_pop(&env_stack);
    CCenv = stack_top(&env_stack);
}

static void
pop_prog()
{
    pop_env();
    stack_pop(&prog_stack);
    CCprog = stack_top(&prog_stack);
}


#define NAMEBUFSIZE 512

void
p_init_process(ip, pragm, isliteral)
decl *ip;			/* in decl: initport declaration */
char *pragm;			/* in string: pragma for whole process */
flag isliteral;			/* is this a process literal? */
{
    void create_proc_printmap();
    void p_init_block();
    objectp p_declare();
    void make_new_env();
    char *get_module_name();

    hobject(SProgram, record);	/* proc */
    hobject(SId, variant);	/* executable_id */
    hobject(SCId, nominal);	/* programid */
    char namebuf[NAMEBUFSIZE];

    CCprog = new_object();
    new_record(CCprog, proc);

    unique(CCprog@Id);	/* create id for this program */

    copy(SCId, CCprog@Id);
    copy(SId, Bottom);
    unite(SId, SCId, executable_thing__process);

    if (not isliteral) 
      create_exec_printmap(SId, get_module_name());
    else {
	strcpy(namebuf, "Subprocess of ");
	strncat(namebuf, get_module_name(), NAMEBUFSIZE-strlen(namebuf)-1);
	create_exec_printmap(SId, namebuf);
    }

    make_new_env();
    cheapcopy(CCprog@proc__executable_part, CCenv);

    copy(CCprog@proc__initport, p_declare(ip));

    cheapcopy(SProgram, CCprog);
    insert(Absprog@program__programs, SProgram);

    stack_push(&prog_stack, CCprog);

    CCposmap = new_object();
    new_record(CCposmap, position_mapping);
    copy(CCposmap@Id, SId);
    avl_new_table(CCposmap@position_mapping__mapping, firstelem_key);
    stack_push(&posmap_stack, CCposmap);

    discard(SId);

    /* kill the deferred consumes list */
    deferredConsumes = nil;

    p_init_block(); p_init_block();
    /* create the outer block which just has the on (others) handler */
    /* plus the block that contains the user's main clause and handlers */
}


objectp
get_proc()
{
    return(CCprog);
}


objectp
get_posmaps()
{
    return(Posmaps);
}


objectp				/* constant processid */
p_end_process()
{
    void end_exec_printmap();
    objectp p_end_compound_scope();
    void p_end_outer_block();

    objectp Procid;
    objectp Stmt;

    Stmt = p_end_compound_scope(); /* end user's main block */
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);
    
    p_end_outer_block();	/* end outer block that has the on(others) */

    insert(Posmaps, stack_pop(&posmap_stack));
    CCposmap = stack_top(&posmap_stack);

    Procid = CCprog@Id;		/* grab pointer to Id */

    pop_prog();
    end_exec_printmap();

    return(Procid);
}



static void
make_new_env()
{
    void make_new_scope();


    CCenv = new_object();

    new_record(CCenv, execution_environment);
    avl_new_table(CCenv@execution_environment__scopes, firstelem_key);
				/* keys (id) */
    avl_new_table(CCenv@execution_environment__clauses, firstelem_key);
				/* keys (id) */

    stack_push(&env_stack, CCenv);

    make_new_scope();
    copy(CCenv@execution_environment__main_scope, CCscope@Id);
}



void
p_init_block()
{
    objectp make_statement();

    hobject(Blockqual, record);
    hobject(Qualcopy, record);
    objectp Stmt;


    Stmt = make_statement(operator__block, 0);
    new_record(Blockqual, block_qualifier);
    avl_new_table(Blockqual@block_qualifier__constants, whole_key);
    avl_new_table(Blockqual@block_qualifier__handlers, firstelem_key);
    cheapcopy(Qualcopy, Blockqual);
    unite(Stmt@statement__qualifier, Blockqual, qualifier_types__block);

    stack_push(&stmt_stack, Stmt);
    make_new_scope();

    copy(Qualcopy@block_qualifier__scope, CCscope@Id);
}


objectp
p_user_exception(Typename, excepname)
objectp Typename;
char *excepname;
{
    objectp resolve_user_exception();
    objectp make_handlername();

    objectp Userex;
    objectp Hname;
    objectp Handler;


    Userex = resolve_user_exception(Typename, excepname);
    Handler = make_handlername(Userex, handler_type__user);

    return(Handler);
}



static objectp
resolve_user_exception(Typename, excepname)
objectp Typename;
char *excepname;
{
    objectp resolve_exception_printname();

    objectp Userex;
    objectp Eid;


    Eid = resolve_exception_printname(Typename, excepname);

    Userex = new_object();
    new_record(Userex, user_exception);
    move(Userex@user_exception__type, Typename);
    move(Userex@user_exception__exceptionid, Eid);

    return(Userex);
}

static objectp
make_handlername(Excepname, exceptype)
objectp Excepname;
dfd_enumeration exceptype;
{
    objectp Hname;

    Hname = new_object();
    unite(Hname, Excepname, exceptype);

    return(Hname);
}


objectp 
p_builtin_exception(name)
char *name;
{
    hobject(Ename, charstring);
    hobject(Maprec, record);
    hobject(Excep, enumeration);


    if (strcmp(name, "others") is 0) 
      return(make_handlername(Bottom, handler_type__others));

    chs_lit(Ename, name);

    if (h_lookup(Maprec, Builtin_Emap, Ename, 0) isnt Normal) {
	fe_error(Inhibit_Codegen, errorcode__general_error,
		 "'%s' is not a valid exception name", name);
	ordenum_lit(Excep, builtin_exception__DefinitionError);
				/* fix later: create an arbitrary (unlikely) */
				/*  exception name. */
    }
    else 
      copy(Excep, Maprec@builtin_exception_map__exception);

    return(make_handlername(Excep, handler_type__builtin));
}


objectp				/* table of handlername */
p_merge_exceptions(Eset, Newexcep)
objectp Eset;			/* table of handlername or nil */
objectp Newexcep;		/* handlername */
{
    if (Eset) {
	if (insert(Eset, Newexcep) isnt Normal) {
	    discard(Newexcep);
	    fe_error(Inhibit_Codegen, errorcode__general_error,
		     "You can't handle an exception twice!");
	}
    }
    else {
	Eset = new_object();
	avl_new_table(Eset, whole_key);
	insert(Eset, Newexcep);
    }

    return(Eset);
}


void
p_init_statement_clause()
{
    void make_new_clause();

    make_new_clause();
}


objectp				/* constant clauseid */
p_end_statement_clause()
{
    objectp Clauseid;


    Clauseid = CCclause@Id;
    pop_clause();
    return(Clauseid);
}
      

void
p_init_expression_clause()
{
    void make_new_clause();

    make_new_clause();
}


objectp
p_end_expression_clause(Exprobj)
objectp Exprobj;
{
    objectp Exinfo;

    Exinfo = new_object();
    new_record(Exinfo, expression_clause_info);
    move(Exinfo@expression_clause_info__result, Exprobj);
    copy(Exinfo@expression_clause_info__clause, CCclause@Id);

    pop_clause();

    return(Exinfo);
}


p_handler(Handlerset, Clauseid)
objectp Handlerset;
objectp Clauseid;
{
    lobject(Handler);
    lobject(Blockhandler);
    objectp Stmt;


    Stmt = stack_top(&stmt_stack);

    initget(Handler, Handlerset, nil);

    while(get_or_err(Handler, Handlerset) is Normal) {
	new_record(Blockhandler, handler);
	copy(Blockhandler@handler__id, Handler);
	copy(Blockhandler@handler__clause, Clauseid);
	insert(Stmt@statement__qualifier@Component@block_qualifier__handlers,
	       Blockhandler);
    }
    endget(Handler, Handlerset);
    /* fix later: return handlerset so we can check that each exception is */
    /* only handled once for the block */
}


objectp
p_end_compound_scope()
{
    objectp Stmt;

    pop_scope();

    Stmt = stack_pop(&stmt_stack);
    return(Stmt);
}


void
p_end_outer_block()
{
    objectp Emptyclause;	/* clauseid */
    objectp Otherset;		/* handlerset */
    objectp Stmt;		/* outermost 'block' statement */

    p_init_statement_clause();	/* create empty clause for on(others) */
    Emptyclause = p_end_statement_clause();

				/* create handlerset which names the handler */
    Otherset = p_merge_exceptions(nil, 
				  p_builtin_exception(copystring("others")));

    p_handler(Otherset, Emptyclause);
				/* stuff the on (others) handler into the */
				/*  outermost block. */
    discard(Otherset);

    Stmt = p_end_compound_scope();	/* end the outermost block */
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);
}


objectp				/* table of handlername */
p_merge_exits(Eset, exitname)
objectp Eset;			/* table of handlername or nil */
char *exitname;			/* exit name */
{
    objectp exit_handler();

    hobject(Ehandler, variant);
    objectp Exitid;


    Exitid = exit_handler(exitname);
    copy(Ehandler, Bottom);
    unite(Ehandler, Exitid, handler_type__exit);

    if (Eset) {
	if (insert(Eset, Ehandler) isnt Normal) {
	    discard(Ehandler);
	    fe_error(Inhibit_Codegen, errorcode__general_error,
		     "You can't handle an exit twice!");
	}
    }
    else {
	Eset = new_object();
	avl_new_table(Eset, whole_key);
	insert(Eset, Ehandler);
    }

    return(Eset);
}



objectp
p_exit_statement(exitname)
char *exitname;
{
    objectp exit_reference();
    objectp p_qniladicop();

    objectp Exitid;
    objectp Equal;		/* exit qualifier, not equal (heh heh) */


    Exitid = exit_reference(exitname);
    Equal = new_object();

    unite(Equal, Exitid, qualifier_types__exit);
    return p_qniladicop(operator__exit, Equal);

}


static void
make_new_scope()
{
    void make_new_clause();
    void pmap_push_scope();

    hobject(SCScope, record);	/* scope */


    CCscope = new_object();
    new_record(CCscope, scope);

    unique(CCscope@Id);
    avl_new_table(CCscope@scope__declarations, firstelem_key);
				/* keys (id) */
    make_new_clause();
    copy(CCscope@scope__clause, CCclause@Id);

    cheapcopy(SCScope, CCscope);
    insert(CCenv@execution_environment__scopes, SCScope);

    stack_push(&scope_stack, CCscope);
    pmap_push_scope(CCscope);
}


static void
make_new_clause()
{
    hobject(SClausetemp, record); /* clause */
    

    CCclause = new_object();
    new_record(CCclause, clause);

    unique(CCclause@Id);
    ord_avl_new_table(CCclause@clause__statements, firstelem_key, nil);

    cheapcopy(SClausetemp, CCclause);
    insert(CCenv@execution_environment__clauses, SClausetemp);

    stack_push(&clause_stack, CCclause);
}


objectp				/* returns constant rootid */
p_declare(rootdecl)
decl *rootdecl;
{
    void set_pragma();
    void check_occlusion();
    void add_root_printname();
    objectp make_declaration();

    objectp Decl;		/* declaration */
    objectp CId;


    Decl = make_declaration(rootdecl->typename, rootdecl->pragm);
    add_root_printname(rootdecl->name, Decl@Id);
    check_occlusion(rootdecl->name, scope_stack);

    CId = Decl@Id;		/* save pointer to ID object */

    insert(CCscope@scope__declarations, Decl);
    return(CId);
}

static objectp
make_declaration(Typename, pragm)
objectp Typename;
char *pragm;
{
    objectp Decl;

    Decl = new_object();

    new_record(Decl, declaration);
    unique(Decl@Id);		/* create root id for declaration */

    unite(Decl@declaration__typename, Typename, typename_option__named);

    set_pragma(Decl@declaration__prag, pragm);

    return(Decl);
}

static objectp
make_statement(opcode,opcount)
dfd_enumeration opcode;
counter opcount;
{
    char *get_srcfile_name();

    lobject(Mapentry);
    objectp Statement;

    Statement = new_object();
    new_record(Statement, statement);
    unique(Statement@Id);
    ordenum_lit(Statement@statement__operator, opcode);

    vec_new_table(Statement@statement__operands, 0);
				/* was opcount instead of 0 */
				/* make operand list with room for one */

    unite(Statement@statement__qualifier, Bottom, qualifier_types__absent);

    new_record(Mapentry, posmap);
    new_record(Mapentry@posmap__aposition, aposition);
    copy(Mapentry@posmap__aposition@aposition__clause, CCclause@Id);
    copy(Mapentry@posmap__aposition@aposition__statement, Statement@Id);
    new_record(Mapentry@posmap__cposition, cposition);
    chs_lit(Mapentry@posmap__cposition@cposition__file, get_srcfile_name());
    ilit(Mapentry@posmap__cposition@cposition__line, global.lineno);
    ilit(Mapentry@posmap__cposition@cposition__column, 0);
				/* should be global.charno */
    insert(CCposmap@position_mapping__mapping, Mapentry);

    return(Statement);
}


static void
set_qual(Stmt, Qual)
objectp Stmt;			/* inout statement */
objectp Qual;			/* in qualifier (or nil) */
{
    if (Qual) {
	move(Stmt@statement__qualifier, Qual);
    }
    else
      unite(Stmt@statement__qualifier, Bottom, qualifier_types__absent);
}


void
p_set_stmt_pragma(pragmaString, Stmt)
char *pragmaString;		/* in string (or nil) */
objectp Stmt;			/* inout statement */
{
    if (pragmaString is nil)
      pragmaString = "";
    chs_lit(Stmt@statement__prag, pragmaString);
}

/* insert the given statement into the current clause, followed by any */
/* discards caused by deferred consumes accumulated while constructing */
/* this statement */
void
p_add_stmt(Stmt)
objectp Stmt;			/* in statment, out bottom */
{
    insert(CCclause@clause__statements, Stmt);
    do_deferred_consumes();
}


static objectp
p_qniladicop(opcode, Qual)
dfd_enumeration opcode;
objectp Qual;			/* in qualifier (or nil) */
{
    objectp Stmt;


    Stmt = make_statement(opcode, 0);

    set_qual(Stmt, Qual);

    return Stmt;
}


static objectp
p_qunaryop(opcode, Dst, Qual)
dfd_enumeration opcode;
objectp Dst;
objectp Qual;			/* in qualifier (or nil) */
{
    objectp Stmt;


    Stmt = make_statement(opcode, 1);

    insert(Stmt@statement__operands, Dst);

    set_qual(Stmt, Qual);

    return(Stmt);
}


objectp
p_unaryop(opcode, Dst)
dfd_enumeration opcode;
objectp Dst;
{
    return p_qunaryop(opcode, Dst, nil);
}


static objectp
p_qbinaryop(opcode, Dst, Src, Qual)
dfd_enumeration opcode;
objectp Dst;
objectp Src;
objectp Qual;
{
    objectp Stmt;


    Stmt = make_statement(opcode, 2);

    insert(Stmt@statement__operands, Dst);
    insert(Stmt@statement__operands, Src);
    set_qual(Stmt, Qual);

    return(Stmt);
}


objectp
p_binaryop(opcode, Dst, Src)
dfd_enumeration opcode;
objectp Dst;
objectp Src;
{
    return(p_qbinaryop(opcode, Dst, Src, nil));
}



static objectp
p_qternaryop(opcode, Dst, Src1, Src2, Qual)
dfd_enumeration opcode;
objectp Dst;
objectp Src1;
objectp Src2;
objectp Qual;
{
    objectp Stmt;


    Stmt = make_statement(opcode, 3);

    insert(Stmt@statement__operands, Dst);
    insert(Stmt@statement__operands, Src1);
    insert(Stmt@statement__operands, Src2);
    set_qual(Stmt, Qual);

    return Stmt;
}


objectp
p_insertormergeat(opcode, Dst, Src1, Src2)
dfd_enumeration opcode;
objectp Dst;
objectp Src1;
objectp Src2;
{
    objectp Stmt;
    
    Stmt = p_qternaryop(opcode, Dst, copyobj(Src1), copyobj(Src2), nil);
    defer_consume(Src2);

    return Stmt;
}


objectp
p_ternaryop(opcode, Dst, Src1, Src2)
dfd_enumeration opcode;
objectp Dst;
objectp Src1;
objectp Src2;
{
    return p_qternaryop(opcode, Dst, Src1, Src2, nil);
}


objectp
p_send(Dst, Src)
objectp Dst;
objectp Src;
{
    objectp Stmt;

    Stmt = p_binaryop(operator__send, copyobj(Dst), copyobj(Src));
    defer_consume(Dst);

    return Stmt;
}


/* do a peephole optimization at the front end: instead of turning
     x := expr
   into				   we simply generate
     temp <- expr		     temp <- expr
     copy x, temp		     move x, temp
     discard temp

  Note that this gets further optimized by p_movestmt below to:
		x <- expr
*/   

objectp
p_copystmt(Dst, Src)
objectp Dst;
objectp Src;
{
    objectp p_movestmt();

    if (is_expr_temp(Src) is SUCCESS)
      return p_movestmt(Dst, Src);
    else
      return p_binaryop(operator__copy, Dst, Src);
}

/* another peephole optimization: don't move an expression temporary; */
/* instead, fix the previous statement to yield its result directly in */
/* the destination operand */
objectp
p_movestmt(Dst, Src)
objectp Dst;
objectp Src;
{
  lobject(OldStmt);
  lobject(OldDst);
  lobject(Position);		/* integer */
  lobject(Consumes);		/* table */
  valcell operand;
  int i;

#if 0
  if (is_expr_temp(Src) is SUCCESS) {
    vec_new_table(Consumes, 0);
    while (TRUE) {
      ilit(Position, obj_size_of(CCclause@clause__statements) - 1);
      remove_at(OldStmt, CCclause@clause__statements, Position);
      switch (enumval(OldStmt@statement__operator)) {
      case operator__discard:	/* consumed temporaries... remember them */
	insert(consumes, OldStmt);
	break;
      case operator__call:
	/* can't patch this one cuz we don't know which operand to replace */
	insert(CCclause@clause__statements, OldStmt);
	merge(CCclause@clause__statements, consumes);
	return p_binaryop(operator__move, Dst, Src);
      default:
	ilit(Position, 0);
	remove_at(OldDst, OldStmt@statement__operands, Position);
	for (i = 0; i < obj_size_of(OldStmt@statement__operands); i++) {
	  operand = OldStmt@statement__operands->value.table->tbls[0].rep.vec
	    ->elements[i];
	  if (operands_overlap(Dst->value, operand, i)) {
	    /* don't do it if it will result in overlapping operands */
	    insert_at(OldStmt@statement__operands, OldDst, Position);
	    insert(CCclause@clause__statements, OldStmt);
	    merge(CCclause@clause__statements, consumes);
	    return p_binaryop(operator__move, Dst, Src);
	  }
	}
	insert_at(OldStmt@statement__operands, Dst, Position);
	if (deferredConsumes is nil)
	  deferredConsumes = consumes;
	else
	  merge(deferredConsumes, consumes);
	return(OldStmt);
      }
    }
  }
  else
#endif
    return p_binaryop(operator__move, Dst, Src);
}

int
operands_overlap(opA,opB)
valcell opA, opB;
{
  int sizeA, sizeB, i;
  valcell *compsA, *compsB;
  hobject(test, boolean);
  lobject(objA);
  lobject(objB);

  objA->value = opA;
  objB->value = opB;
  objA->tsdr = objB->tsdr = &dr_record;
  equal(test, objA@objectname__root, objB@objectname__root);
  if (! booleanval(test))
    return(FALSE);
  sizeA = obj_size_of(objA@objectname__components);
  sizeB = obj_size_of(objB@objectname__components);
  compsA = objA@objectname__components->value.table->tbls[0].rep.vec->elements;
  compsB = objB@objectname__components->value.table->tbls[0].rep.vec->elements;
  objA->tsdr = objB->tsdr = objA@objectname__components->value.table->tsdr;
  for (i = 0; i < sizeA && i < sizeB; i++) {
    objA->value = compsA[i];
    objB->value = compsB[i];
    equal(test, objA, objB);
    if (! booleanval(test))
      return(FALSE);
  }
  return(TRUE);
}

objectp
p_print(Dst)
objectp Dst;
{
    objectp Stmt;

    Stmt = p_unaryop(operator__print, copyobj(Dst));
    defer_consume(Dst);
    return(Stmt);
}


objectp
lookup_root_typename(Rootname)
objectp Rootname;		/* constant rootname */
{
    hobject(SCScope, record);	/* scope */
    hobject(SCDecl, record);	/* decalaration */

    if (h_lookup(SCScope, 
             CCprog@proc__executable_part@execution_environment__scopes,
		     Rootname@rootname__scope, 0) is Normal)
      if (h_lookup(SCDecl, SCScope@scope__declarations,
		       Rootname@rootname__root, 0) is Normal) {
	  if (obj_case_of(SCDecl@declaration__typename)
	      is typename_option__named)
	    return(SCDecl@declaration__typename@Component);
	  else
	    fe_error(Fatal, errorcode__general_error,
		     "lookup_root_typename",
		     "Attempt to resolve type of expression temporary");
      }

    fe_error(Fatal, errorcode__general_error,
	     "lookup_root_typename",
	     "Could not find declaration for resolved rootname");
    /*NOTREACHED*/
}


static objectp
make_expr_temp()
{
    void set_pragma();

    lobject(SDecl);		/* declaration */
    lobject(Tempid);		/* rootid */
    objectp Etemp;		/* objectname */


    new_record(SDecl, declaration);
    unique(SDecl@Id);		/* create root id for declaration */
    unite(SDecl@declaration__typename, Bottom, typename_option__unnamed);
    set_pragma(SDecl@declaration__prag, nil);

    Etemp = new_object();
    new_record(Etemp, objectname);
    new_record(Etemp@objectname__root, rootname);
    copy(Etemp@objectname__root@rootname__scope, CCscope@Id);
    copy(Etemp@objectname__root@rootname__root, SDecl@Id);
    vec_new_table(Etemp@objectname__components, 0);

    copy(Tempid, SDecl@Id);
    insert(Tempids, Tempid);

    insert(CCscope@scope__declarations, SDecl);

    return(Etemp);
}


static status
is_untyped_temp(Objname)
objectp Objname;		/* constant predefined!objectname */
{
    lobject(Tempdecl);		/* predefined!declaration */


    if (obj_size_of(Objname@objectname__components) > 0)
      return(FAILURE);		/* temps are always root names */

    if (h_lookup(Tempdecl, CCscope@scope__declarations, 
		 Objname@objectname__root@rootname__root, 0) isnt Normal)
      return(FAILURE);		/* temps are always in the current scope */

    if (obj_case_of(Tempdecl@declaration__typename) is typename_option__named)
      return(FAILURE);		/* temps are declared with unnamed types */

    return(SUCCESS);		/* pheew!!! got one! */
}


static status
is_expr_temp(Objname)
objectp Objname;		/* constant predefined!objectname */
{
    lobject(Tempid);		/* predefined!declaration */


    if (obj_size_of(Objname@objectname__components) > 0)
      return(FAILURE);		/* temps are always root names */

    if (h_lookup(Tempid, Tempids, Objname@objectname__root@rootname__root, 0)
	isnt Normal)
      return(FAILURE);		/* temps are recorded in the Tempids table */

    return(SUCCESS);		/* got one */
}


static objectp
copyobj(Obj)
objectp Obj;
{
    objectp Objcopy;		/* predefined!objectname */

    Objcopy = new_object();
    copy(Objcopy, Obj);
    return(Objcopy);
}


static void
discardobj(Obj)
objectp Obj;
{
    discard(Obj);
}


static void
consume(Objname)
objectp Objname;		/* in predefined!objectname */
{
    objectp Stmt;
    if (is_expr_temp(Objname) is SUCCESS) {
        Stmt = p_unaryop(operator__discard, Objname);
	chs_lit(Stmt@statement__prag, "");
	insert(CCclause@clause__statements, Stmt);
    }
    else 
      discardobj(Objname);
}

/* like the above, but if a discard is needed it is held until after */
/* the current statement has been inserted into the code. */
static void
defer_consume(Objname)
objectp Objname;
{
    objectp Stmt;

    if (is_expr_temp(Objname) is SUCCESS) {
        Stmt = p_unaryop(operator__discard, Objname);
	chs_lit(Stmt@statement__prag, "");
	if (deferredConsumes is nil) {
	  deferredConsumes = new_object();
	  vec_new_table(deferredConsumes, 0);
	}
	insert(deferredConsumes, Stmt);
    }
    else
      discardobj(Objname);
}

/* insert any discards accumulated by temporaries consumed during the */
/* prior statement build.  The deferred discards table is left nil */
static void
do_deferred_consumes()
{
  if (deferredConsumes isnt nil) {
    merge(CCclause@clause__statements, deferredConsumes);
    deferredConsumes = nil;
  }
}

objectp				/* out objectname */
p_typed_expr(Type, Exprtemp)
objectp Type;			/* the type the expression was labelled with */
objectp Exprtemp;		/* the objectname of the expression temp. */
{
    hobject(Scope, record);	/* scope */
    hobject(Decl, record);	/* declaration */
    hobject(Isok, boolean);	/* boolean */


    if (obj_size_of(Exprtemp@objectname__components) > 0) {
	fe_error(Information, errorcode__general_error,
		 "Can't check type marks on object names yet; sorry");
	return(Exprtemp);
    }

    if (h_lookup(Scope, CCenv@execution_environment__scopes, 
		 Exprtemp@objectname__root@rootname__scope, 0) isnt Normal or
	h_lookup(Decl, Scope@scope__declarations, 
		 Exprtemp@objectname__root@rootname__root, 0) isnt Normal)
      fe_error(Fatal, errorcode__general_error,
	       "p_typed_expr",
	       "Can't find declaration of compiler-generated temporary");

    if (obj_case_of(Decl@declaration__typename) is typename_option__unnamed) {
	unite(Decl@declaration__typename, Type, typename_option__named);
    }
    else {
	equal(Isok, Decl@declaration__typename@Component, Type);
	if (not booleanval(Isok))
	  fe_error(Inhibit_Codegen, errorcode__general_error,
		   "Type specified for expression was '%s', but it is already known to be '%s'",
		   pmap_type(Type),
		   pmap_type(Decl@declaration__typename@Component));
    }

    return(Exprtemp);
}



objectp
p_niladic_expr(opcode)
dfd_enumeration opcode;		/* in opcode */
{
    objectp Etemptemp;
    objectp Etempname;
    objectp Stmt;

    Etempname = make_expr_temp();
    Etemptemp = new_object();  

    copy(Etemptemp, Etempname);
    Stmt = p_unaryop(opcode, Etemptemp);
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);
    
    return(Etempname);
}


objectp
p_unary_expr(opcode,Src)
dfd_enumeration opcode;		/* in opcode */
objectp Src;			/* in objectname */
{
    objectp Etemptemp;
    objectp Etempname;
    objectp Stmt;

    Etempname = make_expr_temp();
    Etemptemp = new_object();  

    copy(Etemptemp, Etempname);
    Stmt = p_binaryop(opcode, Etemptemp, copyobj(Src));
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);
    consume(Src);		/* generate coercion for Src if needed */

    return(Etempname);
}


objectp
p_binary_expr(opcode,Src1, Src2)
dfd_enumeration opcode;		/* in opcode */
objectp Src1;			/* in objectname */
objectp Src2;			/* in objectname */
{
    objectp Etemptemp;
    objectp Etempname;
    objectp Stmt;

    Etempname = make_expr_temp();
    Etemptemp = new_object();  

    copy(Etemptemp, Etempname);
    Stmt = p_ternaryop(opcode, Etemptemp, copyobj(Src1), copyobj(Src2));
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);
    consume(Src1);
    consume(Src2);

    return(Etempname);
}


objectp
p_literal(opcode, value)
dfd_enumeration opcode;		/* in opcode */
char *value;			/* in string: literal value */
{
    objectp Etemptemp;
    objectp Etempname;
    objectp Qual;		/* qualifier */
    hobject(SLit, charstring);
    objectp Stmt;

    chs_lit(SLit, value);

    Qual = new_object();
    unite(Qual, SLit, qualifier_types__literal);

    Etempname = make_expr_temp();
    Etemptemp = new_object();  

    copy(Etemptemp, Etempname);
    Stmt = p_qunaryop(opcode, Etemptemp, Qual);
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);

    return(Etempname);
}



objectp
p_qliteral(opcode, value, qualtype)
dfd_enumeration opcode;		/* in opcode */
char *value;			/* constant object: literal value */
dfd_enumeration qualtype;	/* in qualifier_types */
{
    objectp Etemptemp;
    objectp Etempname;
    objectp Qual;		/* qualifier */
    hobject(Qualv, nominal);
    objectp Stmt;

    Qual = new_object();
    copy(Qualv, value);
    unite(Qual, Qualv, qualtype);

    Etempname = make_expr_temp();
    Etemptemp = new_object();  

    copy(Etemptemp, Etempname);
    Stmt = p_qunaryop(opcode, Etemptemp, Qual);
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);

    return(Etempname);
}


/* selectors */

void
p_init_selector(elementname, Table)
char *elementname;		/* in string: name of representative element */
objectp Table;			/* in objectname: table being queried */
{
    objectp lookup_element_typename();

    selrec *selp;
    decl *declp;
    objectp Selector;
    objectp Typename;
    objectp Eltdecl;


    make_new_scope();

    Selector = new_object();
    new_record(Selector, selector);
    copy(Selector@selector__scope, CCscope@Id);
    Typename = lookup_element_typename(Table);

    if (elementname) {
	declp = tnew(decl);
	declp->name = copystring(elementname);
	declp->typename = Typename;
	declp->pragm = nil;

	copy(Selector@selector__element, p_declare(declp));
    }
    else {
	Eltdecl = make_declaration(Typename, nil);
	copy(Selector@selector__element, Eltdecl@Id);
	insert(CCscope@scope__declarations, Eltdecl);
    }

    selp = tnew(selrec);
    selp->table = Table;
    selp->selector = Selector;
    selp->name = elementname;

    stack_push(&sel_stack, selp);
}


selrec *
p_selector(Exprval)
objectp Exprval;		/* in objectname */
{
    selrec *selp;
    objectp Selector;		/* resproc!selrec */

    selp = (selrec *) stack_pop(&sel_stack);
    Selector = selp->selector;
    move(Selector@selector__result, Exprval);

    pop_scope();

    return(selp);
}

/*
 * mapping shorthands (t[x])
 */

objectp				/* out: expression temp */
mapping_expression(Mappings)
objectp Mappings;		/* ordered table of expression temps */
{
    objectp markboolean();
    objectp lookup_typename();
    objectp lookup_object_typename();

    selrec *selp;		/* selector info record */
    flag done;
    int i;
    int mapsize;

    objectp Selector;		/* selector */
    objectp Table;		/* objectname: the table */
    lobject(Elt);		/* objectname: the representative element */
    objectp Tabletype;		/* typename */
    objectp Tabledef;		/* type_definition */
    objectp Info;		/* table_info */
    lobject(Zero);		/* integer */
    objectp Mapexpr;		/* objectname */
    lobject(Key);		/* formal_objects */
    objectp Expr;		/* objectname */
    objectp Newexpr;		/* objectname */
    lobject(Pos);		/* integer */
    lobject(Keycomp);		/* component_list */
    objectp Compelt;		/* objectname */


    ilit(Zero, 0);

    if (Mappings is nil)	/* expression was t[] */
      return(markboolean(p_literal(operator__named_literal, 
				   copystring("true"))));

    selp = (selrec *) stack_pop(&sel_stack); /* get selector info */
    Table = selp->table;	/* get table objectname */
    Selector = selp->selector;
    stack_push(&sel_stack, selp); /* put selector info back */

    new_record(Elt, objectname);
    new_record(Elt@objectname__root, rootname);
    copy(Elt@objectname__root@rootname__scope, Selector@selector__scope);
    copy(Elt@objectname__root@rootname__root, Selector@selector__element);
    vec_new_table(Elt@objectname__components, 0);

    Tabletype = lookup_object_typename(Table);
    Tabledef = lookup_typename(Tabletype);

    Info = Tabledef@type_definition__specification@Component;

    if (obj_size_of(Mappings) is 1 and /* lookup by position (t[3]) */
	booleanval(Info@table_info__ordered_table)) {
	Mapexpr = new_object();
	remove_at(Mapexpr, Mappings, Zero);
	Expr = markboolean(p_binary_expr(operator__equal, Mapexpr,
	         p_unary_expr(operator__position_of_element, copyobj(Elt))));
    }

    else {			/* lookup by key (t["foo", 'red']) */
	initget(Key, Info@table_info__keys, nil);
	done = FALSE;
	mapsize = obj_size_of(Mappings);

	while (get_or_err(Key, Info@table_info__keys) is Normal) {
	    if (obj_size_of(Key) isnt mapsize)
	      continue;

	    if (done) {
		fe_error(LASTPHASE, errorcode__general_error,
			 "Table type %s has multiple keys with %d components; you must use the long form of the selector", 
			 pmap_type(Tabletype),
			 mapsize);
		break;
	    }

	    /* found a match between number of expressions in Mappings and */
	    /*  one of the keys of Table; generate the expression */

	    for (i = 0, Expr = nil; i < mapsize; i++) {
				/* get the next expression result */
		Mapexpr = new_object();
		remove_at(Mapexpr, Mappings, Zero);

		ilit(Pos, i);	/* get the corresponding key component */
		lookup_at(Keycomp, Key, Pos);

				/* construct objectname from element's */
				/*  rootname and key component's component */
				/*  id's list. */
		Compelt = new_object();
		copy(Compelt, Elt);
		discard(Compelt@objectname__components);
		copy(Compelt@objectname__components, Keycomp);

				/* build the comparison expression */
		Newexpr = markboolean(p_binary_expr(operator__equal,
						    Compelt, Mapexpr));
				/* if more than one, AND them together... */
		if (Expr) 
		  Expr = markboolean(p_binary_expr(operator__and,
						   Expr, Newexpr));
		else
		  Expr = Newexpr;
	    }

	    done = TRUE;
	}

	if (not done) {		/* number of expressions doesn't match size */
				/*  of any of the keys... */
	    fe_error(Stop_Now,  errorcode__general_error,
		     "Table type %s has no key with %d components",
		     pmap_type(Tabletype), mapsize);
	    /*NOTREACHED*/
	}
    }

    discard(Mappings);
    discard(Elt);
    discard(Tabletype);

    return(Expr);
}
    

static objectp
typemark_expr(module, type, Expr)
char *module;
char *type;
objectp Expr;
{
    return(p_typed_expr(p_type_or_attr_name(p_qual_name(module, type), TRUE),
			Expr));
}

static objectp
markboolean(Expr)
objectp Expr;
{
    return(typemark_expr(copystring("predefined"), copystring("boolean"),
			 Expr));
}


objectp
make_mapping(Etemp)
objectp Etemp;
{
    objectp Mapping;

    Mapping = new_object();
    vec_new_table(Mapping, 0);
    insert(Mapping, Etemp);

    return(Mapping);
}


objectp
combine_mappings(M1, M2)
objectp M1, M2;			/* ordered tables of expression temps */
{
    merge(M1, M2);
    return(M1);
}
    

objectp
p_unary_table_expr(opcode, selp)
dfd_enumeration opcode;
selrec *selp;
{
    objectp Etemptemp;
    objectp Etempname;
    objectp Qual;		/* qualifier */
    objectp Stmt;

    Qual = new_object();
    unite(Qual, selp->selector, qualifier_types__selector);

    Etempname = make_expr_temp();
    Etemptemp = new_object();  
    copy(Etemptemp, Etempname);

    Stmt = p_qbinaryop(opcode, Etemptemp, selp->table, Qual);
    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);
				/* this frees all of its operands */
    return(Etempname);
}


objectp
p_remove_extract(opcode, Result, selp)
dfd_enumeration opcode;		/* in opcode: remove or extract */
objectp Result;			/* in objectname: result element or table */
selrec *selp;			/* in selrec: selector and table */
{
    objectp Qual;		/* qualifier */

    Qual = new_object();
    unite(Qual, selp->selector, qualifier_types__selector);

    return p_qbinaryop(opcode, Result, selp->table, Qual);
}



/*** call ***/

void 
p_init_call(Outport)
objectp Outport;		/* objectname */
{
    objectp Stmt;


    Stmt = make_statement(operator__call, 4);
				/* arbitrarily default to 4 parms */

    insert(Stmt@statement__operands, Outport);

    stack_push(&stmt_stack, Stmt);
}


void
p_add_callparm(Parm)
objectp Parm;			/* objectname (from expression) */
{
    objectp Stmt;


    Stmt = stack_top(&stmt_stack);
    insert(Stmt@statement__operands, Parm);
}


objectp
p_end_call()
{
    objectp Stmt;


    Stmt = stack_pop(&stmt_stack);

    return Stmt;
}


objectp
p_function()
{
    hobject(Stsize, integer);
    hobject(One, integer);
    hobject(Lastop, record);
    objectp Stmt;
    objectp Funcval;

    objectp Etemptemp;
    objectp Etempname;


    Etempname = make_expr_temp();
    Etemptemp = new_object();  
    copy(Etemptemp, Etempname);
    p_add_callparm(Etemptemp);

    Stmt = stack_pop(&stmt_stack);

    chs_lit(Stmt@statement__prag,"");
    insert(CCclause@clause__statements, Stmt);

    return(Etempname);
}

objectp
p_return(Callmsg, optexcep)
objectp Callmsg;		/* objectname */
char *optexcep;			/* string (exception name) or nil */
{
    objectp lookup_object_typename(); /* returns typename */
    objectp resolve_exception_printname(); /* returns exceptionid */

    objectp Stmt;
    objectp Cmtype;
    objectp Eid;


    if (optexcep) {
	Stmt = make_statement(operator__return_exception, 1);
	Cmtype = lookup_object_typename(Callmsg);
	Eid = resolve_exception_printname(Cmtype, optexcep);
	unite(Stmt@statement__qualifier, Eid,
	      qualifier_types__return_exception);
    }
    else
      Stmt = make_statement(operator__return, 1);

    insert(Stmt@statement__operands, Callmsg);
    
    return Stmt;
}


objectp
p_while(Testexprclause, Repeatclause)
objectp Testexprclause;		/* expression_clause_info */
objectp Repeatclause;		/* clauseid */
{
    lobject(SQual);
    objectp Stmt;
    objectp Resultid;		/* rootid */


    Stmt = make_statement(operator__while, 0);
    new_record(SQual, while_qualifier);

    Resultid = copyobj(Testexprclause@expression_clause_info__result);

    move(SQual@while_qualifier__test_clause, 
	 Testexprclause@expression_clause_info__clause);
    move(SQual@while_qualifier__result, 
	 Testexprclause@expression_clause_info__result);
    
    copy(SQual@while_qualifier__repeated_clause, Repeatclause);

    unite(Stmt@statement__qualifier, SQual, qualifier_types__while);
   
/* may cause unreachable code
    defer_consume(Resultid);
*/
    discard(Testexprclause);

    return(Stmt);
}


/* for-enumerate */

void
p_init_for_enumerate(enumdecl)
decl *enumdecl;			/* in decl: declaration of enumeration obj */
{
    lobject(SSQual);		/* qualifier */
    hobject(ScopeId, nominal);	/* scopeid */
    objectp Stmt;


    make_new_scope();		/* make scope for the loop */

    Stmt = make_statement(operator__for_enumerate, 0);

    new_record(SSQual, for_enumerate_qualifier);
    copy(SSQual@for_enumerate_qualifier__scope, CCscope@Id);
    copy(SSQual@for_enumerate_qualifier__enumerator, p_declare(enumdecl));
				/* create the declaration for the loop obj */
    unite(Stmt@statement__qualifier, SSQual, qualifier_types__for_enumerate);

    stack_push(&stmt_stack, Stmt);
}




void
p_init_inspect(opcode, selp)
dfd_enumeration opcode;
selrec *selp;
{
    objectp Stmt;
    objectp Qual;		/* qualifier */
    lobject(SQualrec);		/* for_inspect_qualifier */
    decl *declp;


    if (selp->name is nil) {
	fe_error(Stop_Now, errorcode__general_error,
		 "syntax error - an element name is required for a selector used in an Inspect");
	/*NOTREACHED*/
    }

    Stmt = make_statement(opcode, 1);

    make_new_scope();

    declp = tnew(decl);
    declp->name = selp->name;
    declp->typename = lookup_element_typename(selp->table);
    declp->pragm = nil;

    new_record(SQualrec, inspect_table_qualifier);
    copy(SQualrec@inspect_table_qualifier__scope, CCscope@Id);
    move(SQualrec@inspect_table_qualifier__selector, selp->selector);
    copy(SQualrec@inspect_table_qualifier__element, p_declare(declp));

    insert(Stmt@statement__operands, selp->table);
    unite(Stmt@statement__qualifier, SQualrec, qualifier_types__inspect_table);

    stack_push(&stmt_stack, Stmt);
}


void
p_inspect_polymorph(declp, Objname, Formalts)
decl *declp;
objectp Objname;		/* object we are inspecting */
objectp Formalts;		/* its formal typestate */
{
    lobject(Polyqual);
    objectp Stmt;


    Stmt = make_statement(operator__inspect_polymorph, 1);

    make_new_scope();

    new_record(Polyqual, inspect_polymorph_qualifier);
    copy(Polyqual@inspect_polymorph_qualifier__scope, CCscope@Id);
    copy(Polyqual@inspect_polymorph_qualifier__element, p_declare(declp));
    move(Polyqual@inspect_polymorph_qualifier__typestate, Formalts);

    insert(Stmt@statement__operands, Objname);
    unite(Stmt@statement__qualifier, Polyqual, 
	  qualifier_types__inspect_polymorph);

    stack_push(&stmt_stack, Stmt);
}



objectp
p_unwrap(Dst, Src, Formalts)
objectp Dst;			/* objectname of destination */
objectp Src;			/* objectname of polymorph */
objectp Formalts;		/* formal_typestate */
{
    lobject(Unwrapqual);

    copy(Unwrapqual, Bottom);
    unite(Unwrapqual, Formalts, qualifier_types__wrap);
    return p_qbinaryop(operator__unwrap, Dst, Src, Unwrapqual);
}


objectp
p_if(Exprpair, Thenclause, Optelseclause)
objectp Exprpair;		/* expression_clause_info */
objectp Thenclause;		/* const clauseid */
objectp Optelseclause;		/* const clauseid */
{
    objectp Stmt;		/* statement */
    lobject(Qual);		/* if_qualifier */
    lobject(Elseid);		/* clauseid */
    objectp Resultid;		/* rootid: name of result value */

    Stmt = make_statement(operator__if, 0);
    new_record(Qual, if_qualifier);

    Resultid = copyobj(Exprpair@expression_clause_info__result);

    move(Qual@if_qualifier__test_clause, /* test clause */
	 Exprpair@expression_clause_info__clause);
    move(Qual@if_qualifier__test_result, 
	 Exprpair@expression_clause_info__result);

    copy(Qual@if_qualifier__then_clause, Thenclause); /* then clause */

    if (Optelseclause) {	/* else clause */
	copy(Elseid, Optelseclause);
	unite(Qual@if_qualifier__opt_else_clause, Elseid, option__present);
    }
    else
      unite(Qual@if_qualifier__opt_else_clause, Bottom, option__absent);

    unite(Stmt@statement__qualifier, Qual, qualifier_types__if);

/* may cause unreachable code
    defer_consume(Resultid);
*/

    discard(Exprpair);

    return(Stmt);
}


void
p_init_exprblock(resultdecl)
decl *resultdecl;
{
    objectp Stmt;
    hobject(Exprqual, record);

    Stmt = make_statement(operator__expression_block, 0);

    new_record(Exprqual, expression_qualifier);
    new_record(Exprqual@expression_qualifier__result, rootname);
    copy(Exprqual@expression_qualifier__result@rootname__scope, CCscope@Id);
    copy(Exprqual@expression_qualifier__result@rootname__root, 
	 p_declare(resultdecl));

    make_new_scope();
    copy(Exprqual@expression_qualifier__scope, CCscope@Id);
    unite(Stmt@statement__qualifier, Exprqual, 
	  qualifier_types__expression_block);

    stack_push(&stmt_stack, Stmt);
}


objectp
p_end_exprblock()
{
    objectp Stmt;
    objectp Result;


    Stmt = stack_pop(&stmt_stack);
    pop_scope();

    Result = new_object();
    new_record(Result, objectname);
    
    copy(Result@objectname__root, 
	 Stmt@statement__qualifier@Component@expression_qualifier__result);
    vec_new_table(Result@objectname__components, 0);

    chs_lit(Stmt@statement__prag, "");
    insert(CCclause@clause__statements, Stmt);

    return(Result);
}


objectp
p_select(Optexpr, Clauses, Otherwise)
objectp Optexpr;		/* objectname or nil */
objectp Clauses;		/* select_clauses */
objectp Otherwise;		/* clauseid */
{
    objectp Stmt;
    lobject(Selqual);
    objectp Resultid;

    if (Optexpr) {
	Resultid = copyobj(Optexpr);
	Stmt = make_statement(operator__select, 1);
	(void) insert(Stmt@statement__operands, Optexpr);
    }
    else 
      Stmt = make_statement(operator__select, 0);

    new_record(Selqual, select_qualifier);
    move(Selqual@select_qualifier__clauses, Clauses);
    copy(Selqual@select_qualifier__otherwise_clause, Otherwise);

    unite(Stmt@statement__qualifier, Selqual, qualifier_types__select);

/* may cause unreachable code
    if (Optexpr) 
      defer_consume(Resultid);
*/
    return(Stmt);
}


objectp
p_make_select_clauses()
{
    objectp Clauses;

    Clauses = new_object();
    bag_new_table(Clauses);;
    return(Clauses);
}
	

objectp
p_merge_select_clauses(Clauses, Newclause)
objectp Clauses;
objectp Newclause;
{
    if (insert(Clauses, Newclause) isnt Normal)
      fe_error(Stop_Now, errorcode__general_error,
	       "Died during p_select()");

    return(Clauses);
}



static objectp
make_select_guard(guardtype, Clauseinfo, Clauseid)
dfd_enumeration guardtype;	/* guard_type */
objectp Clauseinfo;		/* bothguard, boolguard, or objectname */
objectp Clauseid;		/* const clauseid */
{
    objectp Selclause;

    Selclause = new_object();
    new_record(Selclause, select_clause);
    copy(Selclause@select_clause__clause, Clauseid);
    unite(Selclause@select_clause__info, Clauseinfo, guardtype);

    return(Selclause);
}


objectp
p_select_guard_event(Eventname, Clauseid)
objectp Eventname;
objectp Clauseid;
{
    return(make_select_guard(guard_type__event, Eventname, Clauseid));
}


static objectp
exprclause_to_boolguard(Exprclause)
objectp Exprclause;
{
    objectp Boolg;

    Boolg = new_object();
    new_record(Boolg, boolguard);
    move(Boolg@boolguard__clause, Exprclause@expression_clause_info__clause);
    move(Boolg@boolguard__result, Exprclause@expression_clause_info__result);

    discard(Exprclause);

    return(Boolg);
}

objectp
p_select_guard_boolean(Exprclause, Clauseid)
objectp Exprclause;
objectp Clauseid;
{
    return(make_select_guard(guard_type__boolean, 
			     exprclause_to_boolguard(Exprclause),
			     Clauseid));
}


objectp
p_select_guard_both(Eventname, Exprclause, Clauseid)
objectp Eventname;
objectp Exprclause;
objectp Clauseid;
{
    objectp Boolg;
    objectp Bothg;


    Boolg = exprclause_to_boolguard(Exprclause);

    Bothg = new_object();
    new_record(Bothg, bothguard);
    move(Bothg@bothguard__boolean, Boolg);
    move(Bothg@bothguard__portname, Eventname);


    return(make_select_guard(guard_type__both, Bothg, Clauseid));
}
