-- (C) Copyright International Business Machines Corporation 16 September
-- 1991.  All Rights Reserved.
--
-- See the file USERAGREEMENT distributed with this software for full
-- terms and conditions of use.
-- SCCS Info: @(#)chcode.p	1.4 2/17/92

-- This is the driver process for li-code => C-code transformation.
--
-- Options that can appear on the command line: (environment variables
-- with names beginning with CHC will also be interpreted as setting
-- options, with a lower priority than actual command line options.
-- E.g., setting environment variable CHCLOAD=.:/u/hermes/bin is
-- equivalent to the command line option, -load=.:/u/hermes/bin)
--
--   -verbose - report all activity.
--   -quiet - suppress all reports, except serious diagnostics.
--   -comment - include  /* li-code */ in ccode text.
--   -load=<pathlist> - search list of colon-separated directory paths,
--     used in locating .po files to be transformed.  Default is to
--     just use the pathLoad capability provided in the resource
--     manager to load .po files.
--   -store=<path> - transformed .po files are left in the indicated
--     directory (current directory if no <path> is given or if this
--     option is not present at all).
--   -fullpaths - the object file name stored in the program object's
--     path component is stored as a full absolute path, rather than
--     just a file name that will be interpreted relative to the
--     HCOBJPATH environment variable (the default). 
--   -keepC - the C source text is stored with the program object's
--     text field (discarded by default).
--   -keepObj - the object code is stored with the program object's 
--     object component (discarded by default).
--   -storeC=<path> - the C source file is constructed and left in the
--     indicated directory (current directory if no <path> appears).
--     By default, the C source is written to a temporary file and
--     deleted following compilation.
--   -storeObj=<path> - the object file is constructed and left in the
--     indicated directory (first directory in HCOBJPATH, or current
--     directory failing that, if no <path> is given).  The file name
--     (either full path or just the name, according to -fullpaths
--     option) is stored in the program object's "path" field.  The
--     default is as if "storobj=" is given.
--   -nostoreObj - construct the object file in a temporary file and
--     remove it after it has been processed, or don't compile the C
--     code at all of -keepObj is not also present.
--   -notransform - prevents transforming the original .po file into c
--     code.  In this case, c code must already be stored with the
--     program object, or it must be available in the directory named in
--     (or defaulted by) the -storeC option.  The C files will be
--     re-compiled and stored in the normal manner.  This is useful
--     for recompiling c-coded hermes processes for a new architecture.
--   -interpret=<opcodes> - list of opcodes to interpret
--   -debug=<list> - turns on the debug options named in <list>.
--     Option names should be separated by commas.  Supported options
--     are listed below.  As with all other options, -debug can appear
--     only once; multiple debug options must be listed in that one
--     appearance.
--   -compilecmd=<string> - this is the command that will be invoked
--     to compile a C source module.  Embedded symbols "~;" serve to
--     delimit multiple commands if needed.  The symbol "~C" will be
--     replaced by the full pathname of the C source file, "~O"
--     will be replaced by the full pathname of the object file to be
--     produced, "~N" will be replaced with the name of the
--     generated C function, "~P[VAR]" will be replaced with the
--     value of the VAR parameter from the configuration file, and
--     "~E[VAR]" will be replaced with the value of the environment
--     variable VAR.  The symbol "~~" will replaced by "~".  If this
--     option is missing (it should rarely be necessary to use it),
--     value of the COMPILECMD configruation file parameter is used.
--   -chconfig=<file> - Use the indicated file as the chcode
--     configuration file, from which various parameters are obtained.
--     Each line in the config file should be of the form VAR = TEXT
--     or VAR(ARCH) = TEXT, where the latter allows configuration
--     parameters to be determined on the basis of the architecture on
--     which Hermes is being run (value of the HARCH environment
--     variable, generally the text printed by the arch command).  The
--     default configuration file name is $HROOTDIR/chcode.cfg.
--
--   Non-options in the command line should be process names or ".po" file
--   names; the corresponding ".po" files are input to chcode.
--
-- Currently supported debug options are as follows:
--   compile - echo C compilation commands as they are invoked
--   config - print configuration parameters
--
-- Each input file is transformed into an output file, suffixed ".po"
-- which contains residual li-code.  The residual LI-code consists entirely
-- of 'escape' instructions, unless the "interpret" option has been spec-
-- ified with some significant opcodes.  The c-coded portion of the
-- program is available either in the program object or in files, in
-- various forms as requested by the command line options described above.
--
-- The process chprogram is called to do the transformation and to produce
-- the C-coded output; if it returns exception  already  (the input file
-- was output from a previous transformation), no output file is written.
-- One of the arguments of chprogram, and of many of the nested processes
-- is  ch , a set of control information and capabilities, defined in
-- chinternal.

chcode: using(annotate, chdescriptors, chinternal,
  cload, common, crtoa, cwd, interpform, load, main, objectio,
  main_options, putfile, getfile, rManager, terminalIO, unix, listuff)
linking (chrtoa,chgetoptions,chprogram,listuff,liunstuff)
process(Q: MainQ)
  
declare
  cm: Main;
  rm: rManager;
  ch: chcontrols;
  cload: CLoadFn;
  pathLoad: load_func;
  pathreadobj: readobject_func;
  writeObj: writeobject_func;
  crtoa: ConvertRealToAsciiFn;
  getCwd: getCwdFn;
  stdio: stdio;
  argv: charstringlist;
  cwd: charstring;
  filename: charstring;
  program: program;
  putS: putStringFunc;
  putL: putStringFunc;
  listuff: listuffFn;
  liunstuff: liunstuffFn;
  chprogram: programFn;
  progcount: integer;
begin
  receive cm from Q;
  rm := cm.rm;
  argv := cm.argv;
  putS := cm.terminal.putString;
  putL := cm.terminal.putLine;
  
  unwrap pathLoad from rm.get("pathLoad", "") {init};
  new ch;
  ch.load := pathLoad;
  ch.put := putS;
  unwrap pathreadobj from rm.get("pathReadObj", "") {init};
  unwrap ch.descriptors from pathreadobj("chdescriptors.ho") {init};
  ch.itoa <- procedure of pathLoad("itoa");
  unwrap cload from rm.get("CLoader", "") {init};
  unwrap crtoa from cload("crtoa") {init};
  ch.rtoa <- (initformatrealFn#(create of process chrtoa))(crtoa);
  unwrap stdio from rm.get("stdio", "")
      {init, init(fopen), init(popen), init(access)};
  unwrap writeObj from rm.get("writeObj", "") {init};
  ch.putfile <- (putFileInitFunc#(create of pathLoad("putfile")))(stdio.fopen);
  ch.getfile <- (getFileInitFunc#(create of pathLoad("getfile")))(stdio.fopen);
  ch.access := stdio.access;
  unwrap getCwd from rm.get("getCwd", "") {init};
  ch.runcmd := stdio.popen;
  chprogram <- procedure of process chprogram;
  listuff <- procedure of process listuff;
  liunstuff <- procedure of process liunstuff;
  
  call (getChOptionsFn#(create of process chgetoptions))
      (argv, rm, ch.options, ch.configuration);
  
  if exists of ch.options.debug["config"] then
    call putL("Configuration paramters from file "  
	  | ch.options.configFile | ":");
    for entry in ch.configuration[] inspect
      block declare
	s: charstring;
      begin
	unwrap s from copy of entry.thing {init};
	call putL("  " | entry.name | " = " | s);
      on (others)
      end block;
    end for;
  end if;

  -- force 'interpret' for opcodes listed in -interpret options
  for opcode in ch.options.interpret[] inspect
    block declare
      sd: statement_descriptor;
    begin
      remove sd from d in ch.descriptors
	  where(exists of par in d.parameters
	    where(par.type = 'opcode' and par.string = opcode));
      sd.locus <- 'interpret';
      insert sd into ch.descriptors;
    on (NotFound)
      call cm.terminal.putLine("Unrecognized interpret = " | opcode);
    end block;
  end for;
  
  -- Main service loop.  Read input (.po) files, transform, write
  -- output (.por) files.  Any processes that have been transformed
  -- previously (escape instructions present in the li code) are left
  -- unchanged.
  for name in argv where(position of name > 1) inspect
    block begin
      filename := name;
      if ".po" =  every of c in filename where
	    (position of c >= size of filename - 3) then
	filename <- every of c in filename where 
	    (position of c < size of filename - 3);
      end if;
      -- extract just the module name (after last slash)
      ch.main <- every of c in filename where 
	  (not exists of slash in filename where
	    (slash = '/' and position of c <= position of slash));
      new ch.names;

      block begin
	if not ch.options.quiet then
	  call putS(ch.main | ": load[" | filename | ".po] ");
	end if;
	program <- ch.options.poloader(filename);
      on (load_intf.file_not_found)
	if not ch.options.quiet then
	  call putL("File not found.");
	end if;
	exit noGood;
      on (others)
	if not ch.options.quiet then
	  call putL("Problems loading program");
	end if;
	exit noGood;
      end block;
      
      block declare
	codemap: codemap;
	newcodemap: codemap;
	newentry: codemapentry;
      begin
	ch.modifier <- -1;
	codemap <- liunstuff(program);
	new newcodemap;
	-- do the main program first if it's there, just so the
	-- verbose output looks nicer
	progcount <- 0;
	block begin
	  inspect entry in codemap[program.main_program] begin
	    newentry := entry;
	    newentry.liprog <- chprogram(ch, copy of entry.liprog);
	    progcount <- progcount + 1;
	    insert newentry into newcodemap;
	  end inspect;
	on (notFound)
	  -- main program not currently available in a compiled form
	end block;
	for entry in codemap where
	      (entry.processid <> program.main_program) inspect
	  if progcount <> 0 then
	    call putL("");
	    call putS("  ");
	  end if;
	  newentry := entry;
	  newentry.liprog <- chprogram(ch, copy of entry.liprog);
	  progcount <- progcount + 1;
	  insert newentry into newcodemap;
	end for;
	call listuff(program, newcodemap);
      on (others)
	if not ch.options.quiet then
	  call putL("Problems transforming program");
	end if;
	exit noGood;
      end block;

      block declare
	poly: polymorph;
      begin
	if not ch.options.quiet then
	  if progcount > 1 then
	    call putL("");
	    call putS("... ");
	  end if;
	  call putS("store[" | ch.options.storepath | ch.main | ".po] ");
	end if;
	wrap program as poly;
	call writeObj(ch.options.storepath | ch.main | ".po", poly);
      on (others)
	if not ch.options.quiet then
	  call putL("Problems writing program");
	end if;
	exit noGood;
      end block;

      call putL("");

    on exit(noGood)
      call putL("Transformation of " | ch.main | " failed!");
    end block;
    
  end for;
  
  return cm;
  
end process
