-- (C) Copyright International Business Machines Corporation 16 September
-- 1991.  All Rights Reserved.
--
-- See the file USERAGREEMENT distributed with this software for full
-- terms and conditions of use.
-- SCCS Info: @(#)chbranches.p	1.2 2/4/92

-- This process performs branch optimization
--
-- Input is statements, branches, jumps.
--
-- Modifies statemements (incidentally branches and jumps).
--
-- First unconditional jumps are chained to eliminate jump-to-jumps.
-- 
-- Then each conditional branch statement which branches to an unconditional
-- is retargeted to the ultimate targate of the jump (chain).
--
-- The orphan jumps are 'omit'ed.  An orphan jump is one which is
-- inaccessible, or which jumps to its own successor.  Note that given
-- the two preceding steps, an unconditional jump can only be reached
-- by sequential execution.
--
-- Finally references are cumulated for every statement which is a branch
-- target.
--

chbranches: using(chphase1, chtransform, interpform)

  process(Q: branchesQ)
    
  declare
    
    cm: branches;
    branches: statement_indices;
    reference: referenceFn;
    
    j: integer;
    chst: ch_statement;
    terminal: boolean;
    junk: boolean;

  begin
    
    receive cm from Q;
    
    -- This process follows a chain of execution through unconditional
    -- jump ('branch') and 'omit'ed statements.
    --
    -- Input statements st and an index to st, ix.
    --
    -- Output reply = 'true' if non-null chain; reply = false if null chain.
    --
    -- Modifies ix to index the end of the chain (not 'branch', not 'omit'ed)
    --
    
    reference <- procedure of process (Q: referenceQ)
      declare
	cm: reference;
      begin
	receive cm from Q;
	cm.reply <- 'false';
	while 'true' repeat
	    inspect st in cm.st[cm.ix] begin
		if st.opcode = 'branch' then
		    reveal st.qualifier.integer;
		    cm.ix := st.qualifier.integer;
		    cm.reply <- 'true';
		  else
		    if st.locus <> 'omit' then
			exit done;
		      else
			cm.ix := cm.ix + 1;
			cm.reply <- 'true';
		      end if;
		  end if;
	      end inspect;
	  end while;
	on exit(done)
	  return cm;
      end process;
    
    -- chain jumps - loop executes at most log n times, if n is the
    -- longest jump chain
    while cm.references(cm.statements, cm.jumps, reference) repeat
      -- test's side effects are all we want to accomplish on each
      -- iteration 
    end while;

    -- retarget branches - don't need to iterate since there's no such
    -- thing as a chained conditional branch
    call cm.references(cm.statements, cm.branches, reference, junk);
    
    -- omit orphan jumps
    terminal <- 'false';
    for st in cm.statements[] inspect
	if st.locus <> 'omit' then
	    if st.opcode = 'branch' then
		block begin
		    j := st.position;
		    reveal st.qualifier.integer;
		    if terminal or st.qualifier.integer = j + 1 then
			exit omit;
		      end if;
		    terminal <- 'true';
		    insert j into cm.branches;
		  on exit(omit)
		    remove chst from cm.statements[j];
		    chst.locus <- 'omit';
		    insert chst into cm.statements at j;
		  end block;
	      else
		terminal <- (st.flow = 'emerge');
	      end if;
	  end if;
      end for;
    
    -- This process tallies references for statements which are referenced.
    --
    -- Input statements st and index to st, ix.
    --
    -- Output reply = false.
    --
    
    reference <- procedure of process (Q: referenceQ)
      declare
	cm: reference;
	st: ch_statement;
      begin
	receive cm from Q;
	remove st from cm.st[cm.ix];
	st.References <- st.References + 1;
	insert st into cm.st at cm.ix;
	cm.reply <- 'false';
	return cm;
      end process;
    
    -- compute references for referenced statements.
    
    call cm.references(cm.statements, cm.branches, reference, junk);
    	
    return cm;
    
  end process
