/* (C) Copyright International Business Machines Corporation 23 January */
/* 1990.  All Rights Reserved. */
/*  */
/* See the file USERAGREEMENT distributed with this software for full */
/* terms and conditions of use. */
/* SCCS Info: @(#)stdio.ch	1.30 2/26/92 */

/* C-Hermes processes to provide stdio capabilities to Hermes */
/* processes */

#define _BSD 43

#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#ifdef hpux
#include <sys/ioctl.h>
#endif
#ifdef __convex__
#  include <sys/fcntl.h>
#else
#  include <fcntl.h>
#endif
#include <stdio.h>
#include <errno.h>

#include "ops.h"
#include "storage.h"
#include "cfunc.h"

#include "sysdep.h"

#include "predefined.cd"
#include "unix.cd"
#include "unixint.cd"
#include "streamop.h"

#ifndef FNDELAY
#  ifdef __convex__
#    define FNDELAY O_NONBLOCK
#    undef EWOULDBLOCK
#    define EWOULDBLOCK EAGAIN
#  else
#    define FNDELAY O_NDELAY
#  endif
#endif

#if defined(__convex__)
#  define FASYNC _FASYNC
#endif

extern int errno;
static int lasterr[FD_SETSIZE];

extern fd_close[FD_SETSIZE];

#define IP_INIT		0
#define IP_SERVE	1
#define IP_CONTINUE	2

/* utility routines to translate an open "type" code into the string */
/* required for fopen and friends */
static char *
opentype(code)
int code;
{
  switch (code) {
  case opentype__read: return("r");
  case opentype__write: return("w");
  case opentype__append: return("a");
  case opentype__updateRead: return("r+");
  case opentype__updateWrite: return("w+");
  case opentype__updateAppend: return("a+");
  }
}

static char *
popentype(code)
int code;
{
  switch (code) {
  case popentype__read: return("r");
  case popentype__write: return("w");
  }
}
    
static int
seektype(code)
int code;
{
  switch (code) {
  case seektype__BeginningOfFile: return(0);
  case seektype__CurrentPosition: return(1);
  case seektype__EndOfFile: return(2);
  }
}

static int
accesstype(code)
int code;
{
  switch (code) {
  case fileaccesstype__read: return(R_OK);
  case fileaccesstype__write: return(W_OK);
  case fileaccesstype__execute: return(X_OK);
  case fileaccesstype__exists: return(F_OK);
  }
}

/* set a file descriptor for nonblocking and asynchronous mode I/O */
void
setup_async(fd)
int fd;
{
#ifdef hpux
  int iostat;

  ioctl(fd, FIOSSAIOOWN, getpid());
  fcntl(fd, F_SETFD, 1);
  iostat = 1;
  if (ioctl(fd, FIOSNBIO, &iostat) != -1)
    ioctl(fd, FIOSSAIOSTAT, &iostat);
#else
  fcntl(fd, F_SETOWN, getpid());
  fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) | FASYNC | FNDELAY);
  fd_close[fd] = FALSE;
#endif
}
  


/* routine called during interpreter startup to set stdin and friends */
/* up properly */

#ifdef hpux
static int stdinaio, stdoutaio, stdinnbio, stdoutnbio, stdinown, stdoutown;
#else
static int stdinflags, stdoutflags;
#endif

void
init_stdstreams()
{
  /* save initial state of stdin and stdout, to be restored when */
  /* exiting */
#ifdef hpux
  ioctl(fileno(stdin), FIOGSAIOSTAT, &stdinaio);
  ioctl(fileno(stdin), FIOGNBIO, &stdinnbio);
  ioctl(fileno(stdin), FIOGSAIOOWN, &stdinown);
  ioctl(fileno(stdout), FIOGSAIOSTAT, &stdoutaio);
  ioctl(fileno(stdout), FIOGNBIO, &stdoutnbio);
  ioctl(fileno(stdout), FIOGSAIOOWN, &stdoutown);
#else
  stdinflags = fcntl(fileno(stdin), F_GETFL, 0);
  stdoutflags = fcntl(fileno(stdout), F_GETFL, 0);
#endif

  /* set stdin and stdout non-blocking */
  setup_async(fileno(stdin));
  setup_async(fileno(stdout));

  /* set no buffering for stdout */
  setbuf(stdout, NULL);
}

/* reset stdin and stdout to blocking, synchronous behavior when */
/* Hermes exits */
void
term_stdstreams()
{
#ifdef hpux
  ioctl(fileno(stdin), FIOSSAIOSTAT, &stdinaio);
  ioctl(fileno(stdin), FIOSNBIO, &stdinnbio);
  ioctl(fileno(stdin), FIOSSAIOOWN, &stdinown);
  ioctl(fileno(stdout), FIOSSAIOSTAT, &stdoutaio);
  ioctl(fileno(stdout), FIOSNBIO, &stdoutnbio);
  ioctl(fileno(stdout), FIOSSAIOOWN, &stdoutown);
#else
  fcntl(fileno(stdin), stdinflags);
  fcntl(fileno(stdout), stdoutflags);
#endif
}


/* template for all the one-shot functions that don't maintain a */
/* stream  as part of their process state */
#define NOSTRM_FUNC(name, body) \
CProc(name) \
{ \
  object callm_lobj; objectp callm = &callm_lobj; \
  objectp initp = &current->ep.c->initport; \
  copy(callm, Bottom); \
  if (c_receive(callm, initp) is SUCCESS) { \
    body \
    c_wait(sched, current, initp); \
    c_return(sched, callm); \
  } \
  else if (c_disconn(initp)) { \
    c_discard(sched, initp); \
    c_endprocess(sched, current); \
  } \
  else \
    c_wait(sched, current, initp); \
}

/* clearerrno: clear the errno variable */
NOSTRM_FUNC(C_clearerrno, errno = 0;)

#if defined(NeXT) || defined(ibm032)
#include <ttyent.h>

/* ctermid and cuserid don't appear to exist on the NeXT, so we supply */
/* them here through other means */
static char *
ctermid(s)
char *s;			/* not used */
{
  int slot;
  struct ttyent *ent;
  static char *name = NULL;

  if (name == NULL) {
    slot = ttyslot();		/* find controlling terminal */
    while (slot > 0)
      ent = getttyent();
    endttyent();
    name = (char *) getmain(6+strlen(ent->ty_name));
    strcat(strcpy(name, "/dev/"), ent->ty_name);
  }
  return(name);
}

#include <pwd.h>
static char *
cuserid(s)
char *s;			/* not used */
{
  static char *name = NULL;
  char *x, *getlogin();
  struct passwd *pwd;

  if (name == NULL) {
    if ((x = getlogin()) == NULL) {
      pwd = getpwuid(getuid());
      x = pwd->pw_name;
      endpwent();
    }
    name = (char *) getmain(strlen(x) + 1);
    strcpy(name, x);
  }
  return(name);
}
#endif

/* ctermid: get path to controlling terminal */
NOSTRM_FUNC(C_ctermid, 
  if (!chs_lit(callm@ctermid__id, ctermid(NULL))) { 
    c_discard(sched, callm); 
    c_wait(sched, current, initp); 
    return; 
  })

/* cuserid: return login name */
NOSTRM_FUNC(C_cuserid, 
  if (! chs_lit(callm@cuserid__id, cuserid(NULL))) { 
    c_discard(sched, callm); 
    c_wait(sched, current, initp); 
    return; 
  })

/* tmpnam and tmpfile don't exist on the IBM RT running BSD */
#ifdef ibm032
static char *
tmpnam(s)
char *s;			/* not used */
{
  static char buf[14];
  static unsigned int count = 0;
  unsigned int pid = (unsigned int) getpid();
  char *p;
  int i, x;

  strcat(buf, "/tmp/");
  /* use 4-bit pieces of pid and counter to generate chars starting at */
  /* 'A' for pid, 'a' for counter */
  p = buf+5;
  for (i = 0; i < 4; i++) {
    *p++ = 'A' + (pid & 0xf); 
    pid >>= 4;
  }
  x = count++;
  for (i = 0; i < 4; i++) {
    *p++ = 'a' + (x & 0xf);
    x >>= 4;
  }
  *p = '/0';
  return(buf);
}

static FILE *
tmpfile()
{
  return(fopen(tmpnam(NULL), "r+"));
}

#endif

/* tmpnam: return generated temporary file name */
NOSTRM_FUNC(C_tmpnam, 
  if (! chs_lit(callm@tmpnam__name, tmpnam(NULL))) { 
    c_discard(sched, callm); 
    c_wait(sched, current, initp); 
    return; 
  })

/* fopen: create a stream for I/O on a given file */
NOSTRM_FUNC(C_fopen, 
  { 
    FILE *f = fopen(stringval(callm@fopenx__filename), 
		    opentype(enumval(callm@fopenx__type))); 
    if (f == NULL) { 
      ilit(callm@fopenx__stream, 0); 
      h_boolean(callm@fopenx__opened, FALSE); 
    } 
    else { 
      ilit(callm@fopenx__stream, (int) f); 
      h_boolean(callm@fopenx__opened, TRUE); 
      lasterr[fileno(f)] = 0; 
      setup_async(fileno(f));
    } 
  })

/* fdopen: create a stream for I/O given a Unix file descriptor */
NOSTRM_FUNC(C_fdopen, 
  { 
    FILE *f = fdopen(integerval(callm@fdopenx__filedes), 
		     opentype(enumval(callm@fdopenx__type))); 
    if (f == NULL) { 
      copy(callm@fdopenx__stream, Bottom); 
      h_boolean(callm@fdopenx__opened, FALSE); 
    } 
    else { 
      ilit(callm@fdopenx__stream, (int) f); 
      h_boolean(callm@fdopenx__opened, TRUE); 
      lasterr[fileno(f)] = 0; 
      setup_async(fileno(f));
    } 
  })

/* popen: create a stream for I/O to/from a child process' standard */
/* I/O; this stream must be closed with pclose, not fclose */
NOSTRM_FUNC(C_popen, 
  { 
    FILE *f = popen(stringval(callm@popenx__command), 
		    popentype(enumval(callm@popenx__type))); 
    if (f == NULL) { 
      copy(callm@popenx__stream, Bottom); 
      h_boolean(callm@popenx__opened, FALSE); 
    } 
    else { 
      ilit(callm@popenx__stream, (int) f); 
      h_boolean(callm@popenx__opened, TRUE); 
      lasterr[fileno(f)] = 0; 
      setup_async(fileno(f));
    } 
  })

/* tmpfile: create a temporary file with a name generated by tmpnam, */
/* and return a stream for the file opened in update (w+) mode */
NOSTRM_FUNC(C_tmpfile, 
  { 
    FILE *f = tmpfile(); 
    if (f == NULL) { 
      copy(callm@tmpfilex__stream, Bottom); 
      h_boolean(callm@tmpfilex__opened, FALSE); 
    } 
    else { 
      ilit(callm@tmpfilex__stream, (int) f); 
      h_boolean(callm@tmpfilex__opened, TRUE); 
      lasterr[fileno(f)] = 0; 
      setup_async(fileno(f));
    } 
})

/* access: test accessibility of a file */
NOSTRM_FUNC(C_access, 
  h_boolean(callm@fileaccess__ok, 
	    access(stringval(callm@fileaccess__filename), 
		   accesstype(integerval(callm@fileaccess__mode))) == 0);)

/* read_errno: return current value of the errno variable */
NOSTRM_FUNC(C_read_errno, ilit(callm@readerrno__errno, errno);)

/* stdstreams: fill in stream handles for stdin and friends */
NOSTRM_FUNC(C_stdstreams, 
  ilit(callm@stdstreams__stdin, (int) stdin); 
  ilit(callm@stdstreams__stdout, (int) stdout); 
  ilit(callm@stdstreams__stderr, (int) stderr);)

/* Template for operations on streams that are maintained as process */
/* state from one invocation to the next */

#define STRM_FUNC(name, strm, capa, body) \
CProc(name) \
{  \
  STREAMOPDATA *ep = (STREAMOPDATA *) current->ep.c->data; \
  objectp initp = &current->ep.c->initport; \
  objectp msg = &ep->msg; \
  objectp callp = &ep->callp; \
  switch (current->ip) {  \
  case IP_INIT:  \
    copy(msg, Bottom); \
    copy(callp, Bottom); \
    if (c_receive(msg, initp) is SUCCESS) { \
      if ((c_new_inport(callp) is FAILURE) || \
	   (c_connect(msg@capa, callp) is FAILURE)) { \
	c_discard(sched, msg); \
	c_discard(sched, callp); \
	break; \
      } \
      ep->stream = (FILE *) integerval(msg@strm); \
      current->ip = IP_SERVE; \
      c_wait(sched, current, callp); \
      c_return(sched, msg); \
      c_discard(sched, initp); \
    } \
    else if (c_disconn(initp)) { \
      c_discard(sched, initp); \
      c_endprocess(sched, current); \
    } \
    else \
      c_wait(sched, current, initp); \
    break; \
  case IP_SERVE: \
    copy (msg, Bottom); \
    if (c_receive(msg, callp) is SUCCESS) \
      goto cont; \
    else if (c_disconn(callp)) { \
      c_discard(sched, callp); \
      c_endprocess(sched, current); \
    } \
    else \
      c_wait(sched, current, callp); \
    break; \
  case IP_CONTINUE: \
  cont: { \
    int olderr = errno; \
    errno = lasterr[fileno(ep->stream)]; \
    body \
    lasterr[fileno(ep->stream)] = errno; \
    if (errno == 0) \
      errno = olderr; \
    c_wait(sched, current, callp); \
    c_return(sched, msg); \
    current->ip = IP_SERVE; \
    break; \
  }} \
}

/* clearerr: clear the error status on a stream */
STRM_FUNC(C_clearerr, clearerrinit__stream, clearerrinit__capa, 
	  clearerr(ep->stream); lasterr[fileno(ep->stream)] = 0;)

/* ferror: find whether an error status exists on a stream */
STRM_FUNC(C_ferror, ferrorinit__stream, ferrorinit__capa, 
  h_boolean(msg@ferror__result, ferror(ep->stream));)

/* ferrno: return most recent error code encoutnered on stream */
STRM_FUNC(C_ferrno, ferrnoinit__stream, ferrnoinit__capa, 
  ilit(msg@readerrno__errno, lasterr[fileno(ep->stream)]);)	  

/* feof: query the EOF status of a stream */
STRM_FUNC(C_feof, feofinit__stream, feofinit__capa, 
  h_boolean(msg@feof__result, feof(ep->stream));)

/* fileno: return the file descriptor corresponding to a stream */
STRM_FUNC(C_fileno, filenoinit__stream, filenoinit__capa, 
  ilit(msg@fileno__fileno, fileno(ep->stream));)

/* fflush: flush pending I/O for a stream */
STRM_FUNC(C_fflush, fflushinit__stream, fflushinit__capa, fflush(ep->stream);)

/* fseek: position for I/O at a given byte offset */
STRM_FUNC(C_fseek, fseekinit__stream, fseekinit__capa, 
  ilit(msg@fseekx__result, 
       fseek(ep->stream, (long) integerval(msg@fseekx__offset), 
	     seektype(enumval(msg@fseekx__seektype))));)

/* rewind: seek to the beginning of a stream */
STRM_FUNC(C_rewind, rewindinit__stream, rewindinit__capa, rewind(ep->stream);)

/* ftell: return current I/O position for a stream */
STRM_FUNC(C_ftell, ftellinit__stream, ftellinit__capa, 
  ilit(msg@ftell__offset, (int) ftell(ep->stream));)

/* setlinebuf: set line buffering mode for a stream */
#ifdef hpux
  /* A "real hack" due to Ian Hogg (received from Willard Korfhage) */
  STRM_FUNC(C_setlinebuf, setlinebufinit__stream, setlinebufinit__capa, 
	    ep->stream->__flag |= _IOLBF;)
#else
  STRM_FUNC(C_setlinebuf, setlinebufinit__stream, setlinebufinit__capa, 
	    setlinebuf(ep->stream);)
#endif

/* fstat: get information about a stream */
STRM_FUNC(C_fstat, fstatinit__stream, fstatinit__capa, 
  { 
    objectp stat; 
    struct stat buf; 
    int ok = FALSE; 
 
    if (fstat(fileno(ep->stream), &buf) == 0) 
      if (new_record(msg@fstat__filestatus, filestatus) is Normal) { 
	stat = msg@fstat__filestatus; 
	ilit(stat@filestatus__device, (int) buf.st_dev); 
	ilit(stat@filestatus__inodenumber, (int) buf.st_ino); 
	ilit(stat@filestatus__mode, (int) buf.st_mode); 
	ilit(stat@filestatus__nlink, (int) buf.st_nlink); 
	ilit(stat@filestatus__uid, (int) buf.st_uid); 
	ilit(stat@filestatus__gid, (int) buf.st_gid); 
	ilit(stat@filestatus__deviceid, (int) buf.st_rdev); 
	ilit(stat@filestatus__size, (int) buf.st_size); 
	ilit(stat@filestatus__lastaccess, (int) buf.st_atime); 
	ilit(stat@filestatus__lastmodification, (int) buf.st_mtime); 
	ilit(stat@filestatus__laststatuschange, (int) buf.st_ctime); 
	ilit(stat@filestatus__blocksize, (int) buf.st_blksize); 
	ilit(stat@filestatus__blocks, (int) buf.st_blocks); 
	ok = TRUE; 
      } 
    if (! ok) { 
      c_discard(sched, msg); 
      c_wait(sched, current, callp); 
      break; 
    } 
  })

/* fclose: close a stream and return the close status... this and all */
/* other processes associated with this stream should also be killed */
/* independently. */ 
STRM_FUNC(C_fclose, fcloseinit__stream, fcloseinit__capa, 
  ilit(msg@fclosex__result, fclose(ep->stream));
  fd_close[fileno(ep->stream)] = TRUE;)

/* pclose: like fclose, but returns exit status of child process */
STRM_FUNC(C_pclose, pcloseinit__stream, pcloseinit__capa, pclose(ep->stream);)

/* freopen: reopen an existing stream with a new filename and open */
/* type */
STRM_FUNC(C_freopen, freopeninit__stream, freopeninit__capa, 
  { 
    FILE *f = freopen(stringval(msg@freopenx__filename), 
		      opentype(enumval(msg@freopenx__type)), 
		      ep->stream); 
    h_boolean(msg@freopenx__opened, f==NULL); 
  })

/* readstream: read up to a given number of characters or, optionally, */
/* until a newline character is encountered. */

STRM_FUNC(C_readstream, readstreaminit__stream, readstreaminit__capa, 
  { 
    int n = integerval(msg@readstream__count); 
    int nlstop = booleanval(msg@readstream__stoponnl); 
    int code = readtermcode__ERROR; 
    int myerrno = lasterr[fileno(ep->stream)]; 
    copy(msg@readstream__data, Bottom); 
    if (fd_close[fileno(ep->stream)]) {
      c_discard(sched, msg);
      c_endprocess(sched, current);
      break;
    }
    if (n == 1) { 
      int x = getc(ep->stream); 
      char c; 
      if (x == EOF) 
	myerrno = errno; 
      else { 
	c = x; 
	chs_nlit(msg@readstream__data, &c, 1); 
	code = readtermcode__COUNT; 
      } 
    } 
    else { 
      char *buf = (char *) getmain(nlstop ? n+1 : n); 
      if (buf == NULL) { 
	c_discard(sched, msg); 
	c_wait(sched, current, callp); 
	break; 
      } 
      if (nlstop) { 
	char *x = fgets(buf, n+1, ep->stream); 
	if (x == NULL) 
	  myerrno = errno; 
	else { 
	  int lnth = strlen(x); 
	  chs_nlit(msg@readstream__data, buf, lnth); 
	  code = x[lnth-1]=='\n' ? readtermcode__NEWLINE : readtermcode__COUNT; 
	} 
      } 
      else { 
	int lnth = fread(buf, sizeof(char), n, ep->stream); 
	if (lnth == 0) 
	  myerrno = errno; 
	else { 
	  chs_nlit(msg@readstream__data, buf, lnth); 
	  code = readtermcode__COUNT; 
	} 
      } 
      freemain(buf, nlstop ? n+1 : n);
    } 
    if (code == readtermcode__ERROR) { 
      /* if we got eof on this stream, make sure the asnyc io */
      /* facilities don't continue to wait on it */
      if (feof(ep->stream))
	fd_close[fileno(ep->stream)] = TRUE;
      else if (myerrno == EWOULDBLOCK) {
	/* error ... check if it's because the operation would block */ 
	fd_wait(current, fileno(ep->stream), 1, 0, sched); 
	current->ip = IP_CONTINUE;
	break;
      }
      chs_lit(msg@readstream__data, ""); 
    } 
    enum_lit(msg@readstream__termcode, code); 
    lasterr[fileno(ep->stream)] = myerrno; 
  })

/* readword: read a word from the indicated stream */
STRM_FUNC(C_readword, readwordinit__stream, readwordinit__capa, 
  { 
    int value; 
    int error = FALSE; 
    if (fd_close[fileno(ep->stream)]) {
      c_discard(sched, msg);
      c_endprocess(sched, current);
      break;
    }
    if ((value = getw(ep->stream)) == EOF) 
      /* if we got eof on this stream, make sure the asnyc io */
      /* facilities don't continue to wait on it */
      if (feof(ep->stream))
	fd_close[fileno(ep->stream)] = TRUE;
      else if (errno == EWOULDBLOCK) {
	/* make sure this is the value read, not an error indication */ 
	fd_wait(current, fileno(ep->stream), 1, 0, sched);
	current->ip = IP_CONTINUE;
	break;
      }
      else if (errno != 0)
	error = TRUE;
    ilit(msg@readword__value, value); 
    h_boolean(msg@readword__error, error); 
  })

/* writestream: write the indicated string to the indicated stream */
STRM_FUNC(C_writestream, writestreaminit__stream, writestreaminit__capa, 
  { 
    int n = integerval(msg@writestream__count); 
    int start = integerval(msg@writestream__start);
    int written; 
    if (fd_close[fileno(ep->stream)]) {
      c_discard(sched, msg);
      c_endprocess(sched, current);
      break;
    }
    if (n == 1) 
      written = (putc(stringval(msg@writestream__data)[start], ep->stream) == EOF) 
	? 0 : 1; 
    else 
      written = fwrite(stringval(msg@writestream__data)+start,
			sizeof(char), n, ep->stream); 
    if (written == 0) { 
      /* error ... check if it's because the operation would block */ 
      if (errno == EWOULDBLOCK) {
	fd_wait(current, fileno(ep->stream), 0, 1, sched);
	current->ip = IP_CONTINUE;
	break;
      }
      /* something other than blocking... return normally */ 
    } 
    ilit(msg@writestream__written, written);

  })

/* writeword: write the indicated word to the indicated stream */
STRM_FUNC(C_writeword, writewordinit__stream, writewordinit__capa, 
  { 
    int error = FALSE; 
    if (fd_close[fileno(ep->stream)]) {
      c_discard(sched, msg);
      c_endprocess(sched, current);
      break;
    }
    if (putw(msg@writeword__value, ep->stream) == EOF) 
      /* make sure this is the value written, not an error indication */ 
      if (errno == EWOULDBLOCK) {
	fd_wait(current, fileno(ep->stream), 0, 1, sched);
	current->ip = IP_CONTINUE;
	break;
      }
      else if (errno != 0)
	error = TRUE; 
    h_boolean(msg@writeword__error, error); 
  })
