-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: interfacecheck.bs
-- Author: Dan Yellin
-- SCCS Info: @(#)interfacecheck.bs	1.6 3/13/90

-- This process checks all the operands to a call statement.  The
-- first operand must be an outport.  If this is the case, the
-- callmessage type corresponding to the outport (via its associated
-- inport) is located, and remaining operands are checked according to
-- the specification therein.  Finally, the arguments are checked to
-- ensure that they do not overlap.

interfacecheck: USING(inferredtype, positions, errors,type) 

process(interfaceCheckQ: interfaceCheckQueue)

declare 
  args: interfaceCheckMessage;
  firstArg: objectname;
  type: typename;
  def: type_definition;
  inference: InferredDefinition;
  error: error;
  errorpos: aposition;
  errObj: errorObject;
  ops: objectnames;
  bp: backPatchRecord;
begin
  receive args from interfaceCheckQ;

  -- prepare a position record for use in building error objects
  new errorpos;
  errorpos.clause := args.clause_id;
  errorpos.statement := args.stmt.id;

  -- check that the first statement operand is an outport, and find
  -- its associated callmessage definition
  firstArg := objectname#(op in args.stmt.operands
	where(boolean#(integer#(position of op) = integer#0)));
  block begin
    type:= typename#(args.findTypePort(firstArg, args.scopes,
	args.definitions, args.inferred));
  on (findTypeMessage.unknown_type)
    -- outport type not yet known... make a backpatch entry so this
    -- will be checked later
    new bp;
    bp.triggerObj := firstArg.root;
    unite bp.info.call from aposition#(copy of errorpos);
    insert bp into args.backpatch;
    exit backpatched;
  end block;

  def <- type_definition#(args.finddefPort(type, args.definitions));
  block begin
    reveal def.specification.outport_info;
  on (caseError)
    exit notOutportType;
  end block;

  -- first arg is an outport... get its associated inport specification
  type := def.specification.outport_info;
  def <- type_definition#(args.finddefPort(type, args.definitions));
  block begin
    reveal def.specification.inport_info;
  on (caseError)
    exit notInportType;
  end block;
  
  -- now get the associated callmessage type definition
  type := def.specification.inport_info.message_type;
  def <- type_definition#(args.finddefPort(type, args.definitions));
  block begin
    reveal def.specification.callmessage_info;
  on (caseError)
    exit notCallmessageType;
  end block;

  -- make sure we have the right number of arguments
  if boolean#(integer#(integer#(size of args.stmt.operands)
	    - integer#1)
	  <> integer#(size of def.component_declarations)) then
    new error;
    error.code <- errorcode#'incorrect number of args';
    unite error.position.apos from aposition#(copy of errorpos);
    new error.objects;
    -- error objects are: the output port, the expected number of
    -- args, and the actual nubmer
    unite errObj.objectname from objectname#(copy of firstArg);
    insert errObj into error.objects;
    unite errObj.integer from integer#(size of def.component_declarations);
    insert errObj into error.objects;
    unite errObj.integer 
	from integer#(integer#(size of args.stmt.operands) - integer#1);
    insert errObj into error.objects;
    insert error into args.errors;
  end if;

  -- now check the type of each argument
  for component in def.component_declarations[] inspect
    block begin
      inspect op in args.stmt.operands 
	    where (boolean#(integer#(position of op) =
		integer#(integer#(position of component) + integer#1))) begin
	block begin
	  -- locate the type of this argument
	  type:= typename#(args.findTypePort(op, args.scopes, 
	      args.definitions, args.inferred));
	  if boolean#(type <> component.type) then
	    -- type mismatch... generate an error message
	    new error;
	    error.code <- errorcode#'arg of incorrect type';
	    unite error.position.apos from aposition#(copy of errorpos);
	    new error.objects;
	    -- error objects are: argument position, actual type,
	    -- expected type, and outport
	    unite errObj.integer 
		from integer#(integer#(position of component) + integer#1);
	    insert errObj into error.objects;
	    unite errObj.typename from type; 
	    insert errObj into error.objects;
	    unite errObj.typename from typename#(copy of component.type);
	    insert errObj into error.objects;
	    unite errObj.objectname from objectname#(copy of firstArg);
	    insert errObj into error.objects;
	    insert error into args.errors;
	  end if;
	on (findTypeMessage.unknown_type)
	  -- operand type not known... infer from callmessage definition
	  -- if this is a root, else we need a backpatch entry triggered
	  -- by this object's root
	  if boolean#(integer#(size of op.components) = integer#0) then
	    new inference;
	    inference.root := op.root;
	    inference.type := component.type;
	    insert inference into args.inferred;
	    block begin
	      insert rootname#(copy of op.root) into args.newlyInferred;
	    on (DuplicateKey)
	    end block; 
	  else
	    new bp;
	    bp.triggerObj := op.root;
	    unite bp.info.call from aposition#(copy of errorpos);
	    insert bp into args.backpatch;
	    exit backpatched;
	  end if;
	end block;
      end inspect;
    on (NotFound)
      -- ran out of arguments... message about wrong number of args
      -- already generated, so we need not do anything else
    end block;
  end for;
  
#if FullChecking
  -- now check that parms are non-overlapping
  new ops;
  for operand in args.stmt.operands where (position of operand > 0) inspect
    if exists of op in ops where((op.root = operand.root) and
         args.componentsOverlap(op.components, operand.components)) then
      -- args overlap...generate an error message
      new error;
      error.code <- 'overlapping args';
      unite error.position.apos from aposition#(copy of errorpos);
      new error.objects;
      -- two error object... one of the overlapping operands, and the
      -- outport
      unite errObj.objectname from copy of operand;
      insert errObj into error.objects;
      unite errObj.objectname from copy of firstArg;
      insert errObj into error.objects;
      insert error into args.errors;
    else
      insert copy of operand into ops;
    end if;
  end for;
#endif

  return args;

on exit(backpatched)
  -- can't check this call yet... just return, we'll check it later
  return args;
  
on exit(notOutportType, notInportType, notCallmessageType)
  -- here when the first operand was not an ouport on which
  -- callmessages can be sent
  new error;
  error.code <- errorcode#'outport expected';
  unite error.position.apos from aposition#(copy of errorpos);
  new error.objects;
  -- one error object.. the 1st operand that should have been an outport
  unite errObj.objectname from firstArg;
  insert errObj into error.objects;
  insert error into args.errors;

  return args;

end process
