(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Type.i3                                               *)
(* Last Modified On Mon Feb 10 15:00:33 PST 1992 by kalsow     *)
(*      Modified On Fri Dec 21 00:57:36 1990 by muller         *)

INTERFACE Type;

IMPORT M3, Token, String, MBuf;

TYPE
  T          =  M3.Type;
  FPMap      <: REFANY;
  Assumption <: ADDRESS;

VAR
  recursionDepth: INTEGER := 0;

(*** phase 0 ***)

PROCEDURE Initialize ();
PROCEDURE Reset ();
(* initializes the module and all other type modules. *)

(*** phase 1 ***)

PROCEDURE Parse (READONLY fail: Token.Set): T;
(* parse a type expression *)

PROCEDURE NoteDeclaration (t: T;  decl: M3.Value);
(* record a type's declaration *)

(*** phase 2 ***)

PROCEDURE Check (t: T);
(* type check type t *)

PROCEDURE Strip (t: T): T;
(* return the underlying constructed (ie. not Named) type node *)

PROCEDURE Base (t: T): T;
(* return the base type (strip renaming, packing & subranges) *)

(*** phase 3 ***)

PROCEDURE Size (t: T): INTEGER;
(* return the size in bits of type t;  -1 if variable sized *)

PROCEDURE MinSize (t: T): INTEGER;
(* return the minimum size allowed for values of type t *)

PROCEDURE Alignment (t: T): INTEGER;
(* return the required bit alignment for values of type t *)

PROCEDURE IsEmpty (t: T): BOOLEAN;
(* return TRUE if the type has no values *)

PROCEDURE Number (t: T): INTEGER;
(* return the number of values of the type;  -1 if t is not an ordinal type *)

PROCEDURE GetBounds (t: T;  VAR min, max: INTEGER): BOOLEAN;
(* return the bounds and true for ordinal types, 
   [0,-1] and FALSE for non-ordinal types *)

PROCEDURE IsTraced (t: T): BOOLEAN;
(* TRUE iff t is a traced ref, or contains a traced ref *)

PROCEDURE HasUntraced (t: T): BOOLEAN;
(* TRUE iff t contains an untraced ref, address or procedure *)

PROCEDURE IsEqual (a, b: T;  x: Assumption): BOOLEAN;
(* TRUE iff (a == b) given the assumptions x *)

PROCEDURE DependsOn (a, b: T): BOOLEAN;
(* TRUE iff b's C declaration must precede a's *)

PROCEDURE IsSubtype (a, b: T): BOOLEAN;
(* TRUE iff (a <: b) *)

PROCEDURE IsAssignable (a, b: T): BOOLEAN;
(* TRUE iff (a := b) typechecks *)

PROCEDURE Name (t: T): INTEGER;
(* return the unique id for the type in the generated code *)

(*** phase 4 ***)

PROCEDURE Compile (t: T);
(* generates the C type declartions for a type *)

PROCEDURE InitVariable (t: T;  zeroed: BOOLEAN;  VAR prefix: String.Stack);
(* generates an initial value *)

PROCEDURE InitCost (t: T;  ifZeroed: BOOLEAN): INTEGER;
(* the cost of initializing a t.  (0 IFF no init required) *)

PROCEDURE GenInitialValue (t: T);
(* emits the code for an initial value *)

PROCEDURE GenMap (t: T;  VAR prefix: String.Stack);
(* emits the code for a RC map traversal *)

PROCEDURE ExternalDecl (t: T): M3.Value;
(* = NIL if the type is local *)

PROCEDURE Fingerprint (t: T;  map: FPMap;  wr: MBuf.T);
(* given 'map', write the fingerprint string for 't' on 'wr' *)

(*** phase 5 ***)

PROCEDURE GenLinkerInfo ();
(* generate the "linker" tables that describe the types *)

END Type.

(*
  The following sets of procedures are may be called during the
  various phases of the compilation:

   initialization:
     { Initialize* }
   parsing:
     { Parse, NoteDeclaration }
   type checking:
     { Check, Size, Number, GetBounds, Base, IsEqual,
       IsSubtype, IsAssignable, IsTraced }
   code generation:
     { Name, Size, Number, GetBounds, Base, IsEqual,
       IsSubtype, IsAssignable, IsTraced }

   ( * => may only be called once )
*)
