(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: MultiplyExpr.m3                                       *)
(* Last modified on Mon Jul  8 18:22:18 1991 by kalsow         *)
(*      modified on Wed Nov 21 21:55:34 1990 by muller         *)

MODULE MultiplyExpr;

IMPORT Expr, ExprRep, Type, Error, Int, Reel, EReel;
IMPORT SetExpr, IntegerExpr, ReelExpr, Temp, MBuf, LReel, SetType, Emit;

CONST
  cINT   = 0;
  cREAL  = 1;
  cLONG  = 2;
  cEXTND = 3;
  cSET   = 4;

TYPE
  P = ExprRep.Tabc BRANDED "MultiplyExpr.T" OBJECT
      OVERRIDES
        typeOf       := TypeOf;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := Writer;
        isEqual      := ExprRep.EqCheckAB;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	note_write   := ExprRep.NotWritable;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a, b: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.a := a;
    p.b := b;
    RETURN p;
  END New;

PROCEDURE TypeOf (p: P): Type.T =
  BEGIN
    RETURN Expr.TypeOf (p.a);
  END TypeOf;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  VAR ta, tb, range: Type.T;
  BEGIN
    Expr.TypeCheck (p.a, cs);
    Expr.TypeCheck (p.b, cs);
    ta := Type.Base (Expr.TypeOf (p.a));
    tb := Type.Base (Expr.TypeOf (p.b));
    IF    (tb = Int.T)   AND (ta = Int.T)   THEN  p.class := cINT;
    ELSIF (tb = Reel.T)  AND (ta = Reel.T)  THEN  p.class := cREAL;
    ELSIF (tb = LReel.T) AND (ta = LReel.T) THEN  p.class := cLONG;
    ELSIF (tb = EReel.T) AND (ta = EReel.T) THEN  p.class := cEXTND;
    ELSIF SetType.Split (ta, range)
             AND Type.IsEqual (ta, tb, NIL) THEN  p.class := cSET;
    ELSE Error.Msg ("illegal operands for \'*\'");  p.class := cINT;
    END;
    p.type := ta;
  END Check;

PROCEDURE Compile (p: P): Temp.T =
  VAR t1, t2, t3: Temp.T;
  BEGIN
    t1 := Expr.Compile (p.a);
    t2 := Expr.Compile (p.b);
    IF (p.class # cSET) THEN
      t3 := Temp.AllocMacro (p, FALSE);
      Temp.Depend (t3, t1);
      Temp.Depend (t3, t2);
    ELSE
      t3 := Temp.Alloc (p);
      SetExpr.CompileInter (t1, t2, t3, p.type);
      Temp.Free (t1);
      Temp.Free (t2);
    END;
    RETURN t3;
  END Compile;

PROCEDURE Writer (p: P;  t1, t2: Temp.T) =
  BEGIN
    <* ASSERT p.class # cSET *>
    Emit.OpTT ("(@ * @)", t1, t2);
  END Writer;

PROCEDURE Fold (p: P): Expr.T =
  VAR e1, e2, e3: Expr.T;
  BEGIN
    e1 := Expr.ConstValue (p.a);
    e2 := Expr.ConstValue (p.b);
    e3 := NIL;
    IF    (e1 = NIL) OR (e2 = NIL)          THEN
    ELSIF IntegerExpr.Multiply (e1, e2, e3) THEN
    ELSIF ReelExpr.Multiply    (e1, e2, e3) THEN
    ELSIF SetExpr.Intersection (e1, e2, e3) THEN
    END;
    RETURN e3;
  END Fold;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "* ");
    Expr.Fingerprint (p.a, map, wr);
    Expr.Fingerprint (p.b, map, wr);
  END FPrinter;

BEGIN
END MultiplyExpr.
