/* Mapping tables from GB2312 to BIG5 and vice versa.
   Copyright (C) 1999-2014 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Ulrich Drepper <drepper@cygnus.com>, 1999.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

#include <dlfcn.h>
#include <gconv.h>
#include <stdint.h>

/*
 * This module does conversions between gb2312 and big5 encodings.
 * It is necessary because gb2312 encoding use simplified chinese,
 * while big5 use traditional one. Often times both the simplified
 * version of a chinese character and its traditional counterpart
 * are presented in unicode and have distinctive code values. So
 * an indirect method through UCS4 (GB2312 <=> UCS4 <=> BIG5) will
 * not do the work correctly.
 *
 *   --- Yong Li  <rigel863@yahoo.com>
 */


static const char __from_gb2312_to_big5 [8178][2] =
{
  [0x0000] = "\xa1\x40", [0x0001] = "\xa1\x42", [0x0002] = "\xa1\x43",
  [0x0003] = "\xa1\x50", [0x0004] = "\xa3\xbc", [0x0005] = "\xa3\xbe",
  [0x0006] = "\xa1\x4c", [0x0007] = "\xa1\xb2", [0x0009] = "\xa2\x77",
  [0x000a] = "\xa1\xe3", [0x000b] = "\xa1\xfc", [0x000c] = "\xa1\x4b",
  [0x000d] = "\xa1\xa5", [0x000e] = "\xa1\xa6", [0x000f] = "\xa1\xa7",
  [0x0010] = "\xa1\xa8", [0x0011] = "\xa1\x65", [0x0012] = "\xa1\x66",
  [0x0013] = "\xa1\x71", [0x0014] = "\xa1\x72", [0x0015] = "\xa1\x6d",
  [0x0016] = "\xa1\x6e", [0x0017] = "\xa1\x75", [0x0018] = "\xa1\x76",
  [0x0019] = "\xa1\x79", [0x001a] = "\xa1\x7a", [0x001d] = "\xa1\x69",
  [0x001e] = "\xa1\x6a", [0x001f] = "\xa1\xd3", [0x0020] = "\xa1\xd1",
  [0x0021] = "\xa1\xd2", [0x0022] = "\xa1\x4a", [0x0023] = "\xa1\x73",
  [0x0024] = "\xa1\x74", [0x0025] = "\xa3\x55", [0x0026] = "\xa3\x53",
  [0x0027] = "\xa1\xe5", [0x0028] = "\xa1\xe4", [0x002b] = "\xa1\xd4",
  [0x002c] = "\xa1\xe6", [0x002d] = "\xa1\xfc", [0x002e] = "\xa1\xe7",
  [0x0030] = "\xa1\xf3", [0x0031] = "\xa1\xec", [0x0032] = "\xa1\xed",
  [0x0033] = "\xa1\xdd", [0x0035] = "\xa1\xdc", [0x0038] = "\xa1\xda",
  [0x003b] = "\xa1\xd8", [0x003c] = "\xa1\xd9", [0x003d] = "\xa1\xdb",
  [0x003e] = "\xa1\xee", [0x003f] = "\xa1\xef", [0x0040] = "\xa1\xf1",
  [0x0041] = "\xa1\xf0", [0x0042] = "\xa2\x58", [0x0043] = "\xa1\xac",
  [0x0044] = "\xa1\xaa", [0x0045] = "\xa2\x4a", [0x0046] = "\xa2\x43",
  [0x0048] = "\xa2\x46", [0x0049] = "\xa2\x47", [0x004b] = "\xa1\xb1",
  [0x004d] = "\xa1\xb8", [0x004e] = "\xa1\xb9", [0x004f] = "\xa1\xb3",
  [0x0050] = "\xa1\xb4", [0x0051] = "\xa1\xb7", [0x0052] = "\xa1\xba",
  [0x0053] = "\xa1\xbb", [0x0054] = "\xa1\xbc", [0x0055] = "\xa1\xbd",
  [0x0056] = "\xa1\xb5", [0x0057] = "\xa1\xb6", [0x0058] = "\xa1\xb0",
  [0x0059] = "\xa1\xf7", [0x005a] = "\xa1\xf6", [0x005b] = "\xa1\xf4",
  [0x005c] = "\xa1\xf5", [0x00ae] = "\xa2\xb9", [0x00af] = "\xa2\xba",
  [0x00b0] = "\xa2\xbb", [0x00b1] = "\xa2\xbc", [0x00b2] = "\xa2\xbd",
  [0x00b3] = "\xa2\xbe", [0x00b4] = "\xa2\xbf", [0x00b5] = "\xa2\xc0",
  [0x00b6] = "\xa2\xc1", [0x00b7] = "\xa2\xc2", [0x00bc] = "\xa1\x49",
  [0x00bd] = "\xa1\xaa", [0x00be] = "\xa1\xad", [0x00bf] = "\xa2\x44",
  [0x00c0] = "\xa2\x48", [0x00c1] = "\xa1\xae", [0x00c3] = "\xa1\x5d",
  [0x00c4] = "\xa1\x5e", [0x00c5] = "\xa1\xaf", [0x00c6] = "\xa1\xcf",
  [0x00c7] = "\xa1\x41", [0x00c8] = "\xa1\xd0", [0x00c9] = "\xa1\x44",
  [0x00ca] = "\xa1\xfe", [0x00cb] = "\xa2\xaf", [0x00cc] = "\xa2\xb0",
  [0x00cd] = "\xa2\xb1", [0x00ce] = "\xa2\xb2", [0x00cf] = "\xa2\xb3",
  [0x00d0] = "\xa2\xb4", [0x00d1] = "\xa2\xb5", [0x00d2] = "\xa2\xb6",
  [0x00d3] = "\xa2\xb7", [0x00d4] = "\xa2\xb8", [0x00d5] = "\xa1\x47",
  [0x00d6] = "\xa1\x46", [0x00d7] = "\xa1\xd5", [0x00d8] = "\xa1\xd7",
  [0x00d9] = "\xa1\xd6", [0x00da] = "\xa1\x48", [0x00db] = "\xa2\x49",
  [0x00dc] = "\xa2\xcf", [0x00dd] = "\xa2\xd0", [0x00de] = "\xa2\xd1",
  [0x00df] = "\xa2\xd2", [0x00e0] = "\xa2\xd3", [0x00e1] = "\xa2\xd4",
  [0x00e2] = "\xa2\xd5", [0x00e3] = "\xa2\xd6", [0x00e4] = "\xa2\xd7",
  [0x00e5] = "\xa2\xd8", [0x00e6] = "\xa2\xd9", [0x00e7] = "\xa2\xda",
  [0x00e8] = "\xa2\xdb", [0x00e9] = "\xa2\xdc", [0x00ea] = "\xa2\xdd",
  [0x00eb] = "\xa2\xde", [0x00ec] = "\xa2\xdf", [0x00ed] = "\xa2\xe0",
  [0x00ee] = "\xa2\xe1", [0x00ef] = "\xa2\xe2", [0x00f0] = "\xa2\xe3",
  [0x00f1] = "\xa2\xe4", [0x00f2] = "\xa2\xe5", [0x00f3] = "\xa2\xe6",
  [0x00f4] = "\xa2\xe7", [0x00f5] = "\xa2\xe8", [0x00f6] = "\xa1\xa3",
  [0x00f7] = "\xa2\x40", [0x00f8] = "\xa1\xa4", [0x00fa] = "\xa1\xc4",
  [0x00fb] = "\xa1\xab", [0x00fc] = "\xa2\xe9", [0x00fd] = "\xa2\xea",
  [0x00fe] = "\xa2\xeb", [0x00ff] = "\xa2\xec", [0x0100] = "\xa2\xed",
  [0x0101] = "\xa2\xee", [0x0102] = "\xa2\xef", [0x0103] = "\xa2\xf0",
  [0x0104] = "\xa2\xf1", [0x0105] = "\xa2\xf2", [0x0106] = "\xa2\xf3",
  [0x0107] = "\xa2\xf4", [0x0108] = "\xa2\xf5", [0x0109] = "\xa2\xf6",
  [0x010a] = "\xa2\xf7", [0x010b] = "\xa2\xf8", [0x010c] = "\xa2\xf9",
  [0x010d] = "\xa2\xfa", [0x010e] = "\xa2\xfb", [0x010f] = "\xa2\xfc",
  [0x0110] = "\xa2\xfd", [0x0111] = "\xa2\xfe", [0x0112] = "\xa3\x40",
  [0x0113] = "\xa3\x41", [0x0114] = "\xa3\x42", [0x0115] = "\xa3\x43",
  [0x0116] = "\xa1\x61", [0x0117] = "\xa1\x55", [0x0118] = "\xa1\x62",
  [0x0119] = "\xa1\xc3", [0x01d6] = "\xa3\x44", [0x01d7] = "\xa3\x45",
  [0x01d8] = "\xa3\x46", [0x01d9] = "\xa3\x47", [0x01da] = "\xa3\x48",
  [0x01db] = "\xa3\x49", [0x01dc] = "\xa3\x4a", [0x01dd] = "\xa3\x4b",
  [0x01de] = "\xa3\x4c", [0x01df] = "\xa3\x4d", [0x01e0] = "\xa3\x4e",
  [0x01e1] = "\xa3\x4f", [0x01e2] = "\xa3\x50", [0x01e3] = "\xa3\x51",
  [0x01e4] = "\xa3\x52", [0x01e5] = "\xa3\x53", [0x01e6] = "\xa3\x54",
  [0x01e7] = "\xa3\x55", [0x01e8] = "\xa3\x56", [0x01e9] = "\xa3\x57",
  [0x01ea] = "\xa3\x58", [0x01eb] = "\xa3\x59", [0x01ec] = "\xa3\x5a",
  [0x01ed] = "\xa3\x5b", [0x01f6] = "\xa3\x5c", [0x01f7] = "\xa3\x5d",
  [0x01f8] = "\xa3\x5e", [0x01f9] = "\xa3\x5f", [0x01fa] = "\xa3\x60",
  [0x01fb] = "\xa3\x61", [0x01fc] = "\xa3\x62", [0x01fd] = "\xa3\x63",
  [0x01fe] = "\xa3\x64", [0x01ff] = "\xa3\x65", [0x0200] = "\xa3\x66",
  [0x0201] = "\xa3\x67", [0x0202] = "\xa3\x68", [0x0203] = "\xa3\x69",
  [0x0204] = "\xa3\x6a", [0x0205] = "\xa3\x6b", [0x0206] = "\xa3\x6c",
  [0x0207] = "\xa3\x6d", [0x0208] = "\xa3\x6e", [0x0209] = "\xa3\x6f",
  [0x020a] = "\xa3\x70", [0x020b] = "\xa3\x71", [0x020c] = "\xa3\x72",
  [0x020d] = "\xa3\x73", [0x02b6] = "\xa3\x74", [0x02b7] = "\xa3\x75",
  [0x02b8] = "\xa3\x76", [0x02b9] = "\xa3\x77", [0x02ba] = "\xa3\x78",
  [0x02bb] = "\xa3\x79", [0x02bc] = "\xa3\x7a", [0x02bd] = "\xa3\x7b",
  [0x02be] = "\xa3\x7c", [0x02bf] = "\xa3\x7d", [0x02c0] = "\xa3\x7e",
  [0x02c1] = "\xa3\xa1", [0x02c2] = "\xa3\xa2", [0x02c3] = "\xa3\xa3",
  [0x02c4] = "\xa3\xa4", [0x02c5] = "\xa3\xa5", [0x02c6] = "\xa3\xa6",
  [0x02c7] = "\xa3\xa7", [0x02c8] = "\xa3\xa8", [0x02c9] = "\xa3\xa9",
  [0x02ca] = "\xa3\xaa", [0x02cb] = "\xa3\xab", [0x02cc] = "\xa3\xac",
  [0x02cd] = "\xa3\xad", [0x02ce] = "\xa3\xae", [0x02cf] = "\xa3\xaf",
  [0x02d0] = "\xa3\xb0", [0x02d1] = "\xa3\xb1", [0x02d2] = "\xa3\xb2",
  [0x02d3] = "\xa3\xb3", [0x02d4] = "\xa3\xb4", [0x02d5] = "\xa3\xb5",
  [0x02d6] = "\xa3\xb6", [0x02d7] = "\xa3\xb7", [0x02d8] = "\xa3\xb8",
  [0x02d9] = "\xa3\xb9", [0x02da] = "\xa3\xba", [0x02f3] = "\xa2\x77",
  [0x02f4] = "\xa2\x77", [0x02f5] = "\xa2\x78", [0x02f6] = "\xa2\x78",
  [0x02ff] = "\xa2\x7a", [0x0302] = "\xa2\x7a", [0x0303] = "\xa2\x7b",
  [0x0306] = "\xa2\x7b", [0x0307] = "\xa2\x7c", [0x030a] = "\xa2\x7c",
  [0x030b] = "\xa2\x7d", [0x030e] = "\xa2\x7d", [0x030f] = "\xa2\x75",
  [0x0316] = "\xa2\x75", [0x0317] = "\xa2\x74", [0x031e] = "\xa2\x74",
  [0x031f] = "\xa2\x73", [0x0326] = "\xa2\x73", [0x0327] = "\xa2\x72",
  [0x032e] = "\xa2\x72", [0x032f] = "\xa2\x71", [0x033e] = "\xa2\x71",
  [0x0582] = "\xb0\xda", [0x0583] = "\xaa\xfc", [0x0584] = "\xae\x4a",
  [0x0585] = "\xae\xc1", [0x0586] = "\xab\x75", [0x0587] = "\xad\xfc",
  [0x0588] = "\xab\x73", [0x0589] = "\xbd\x4a", [0x058a] = "\xc0\xf9",
  [0x058b] = "\xc4\xa7", [0x058c] = "\xb8\x47", [0x058d] = "\xa6\xe3",
  [0x058e] = "\xc3\xaa", [0x058f] = "\xb7\x52", [0x0590] = "\xb9\x69",
  [0x0591] = "\xbe\x62", [0x0592] = "\xae\xf2", [0x0593] = "\xa6\x77",
  [0x0594] = "\xad\xcd", [0x0595] = "\xab\xf6", [0x0596] = "\xb7\x74",
  [0x0597] = "\xa9\xa4", [0x0598] = "\xd3\x69", [0x0599] = "\xae\xd7",
  [0x059a] = "\xbb\xea", [0x059b] = "\xa9\xf9", [0x059c] = "\xaf\x73",
  [0x059d] = "\xa5\x57", [0x059e] = "\xb1\xce", [0x059f] = "\xbc\xf5",
  [0x05a0] = "\xbf\xac", [0x05a1] = "\xc3\xcd", [0x05a2] = "\xb6\xc6",
  [0x05a3] = "\xb6\xf8", [0x05a4] = "\xbe\xd2", [0x05a5] = "\xbf\x44",
  [0x05a6] = "\xaa\xdd", [0x05a7] = "\xae\xc3", [0x05a8] = "\xa5\xb6",
  [0x05a9] = "\xa5\x7a", [0x05aa] = "\xa7\x61", [0x05ab] = "\xaf\xb9",
  [0x05ac] = "\xa4\x4b", [0x05ad] = "\xac\xcd", [0x05ae] = "\xa4\xda",
  [0x05af] = "\xa9\xde", [0x05b0] = "\xb6\x5b", [0x05b1] = "\xb9\x76",
  [0x05b2] = "\xa7\xe2", [0x05b3] = "\xaf\xd2", [0x05b4] = "\xc5\xf2",
  [0x05b5] = "\xc5\x51", [0x05b6] = "\xbd\x7d", [0x05b7] = "\xaa\xa8",
  [0x05b8] = "\xa5\xd5", [0x05b9] = "\xac\x66", [0x05ba] = "\xa6\xca",
  [0x05bb] = "\xc2\x5c", [0x05bc] = "\xa8\xd5", [0x05bd] = "\xb1\xd1",
  [0x05be] = "\xab\xf4", [0x05bf] = "\xde\xfb", [0x05c0] = "\xb4\xb3",
  [0x05c1] = "\xaf\x5a", [0x05c2] = "\xb7\x68", [0x05c3] = "\xa7\xe6",
  [0x05c4] = "\xaf\xeb", [0x05c5] = "\xb9\x7b", [0x05c6] = "\xaa\x4f",
  [0x05c7] = "\xaa\xa9", [0x05c8] = "\xa7\xea", [0x05c9] = "\xa9\xd5",
  [0x05ca] = "\xa6\xf1", [0x05cb] = "\xc3\xa4", [0x05cc] = "\xa5\x62",
  [0x05cd] = "\xbf\xec", [0x05ce] = "\xb2\xcc", [0x05cf] = "\xa8\xb9",
  [0x05d0] = "\xc0\xb0", [0x05d1] = "\xb1\xf5", [0x05d2] = "\xba\x5d",
  [0x05d3] = "\xbb\x48", [0x05d4] = "\xb8\x6a", [0x05d5] = "\xb4\xce",
  [0x05d6] = "\xbd\x53", [0x05d7] = "\xb0\x46", [0x05d8] = "\xc2\xe9",
  [0x05d9] = "\xb3\xc4", [0x05da] = "\xc1\xbd", [0x05db] = "\xad\x63",
  [0x05dc] = "\xad\x4d", [0x05dd] = "\xa5\x5d", [0x05de] = "\xbd\xc7",
  [0x05df] = "\xad\xe9", [0x05e0] = "\xc1\xa1", [0x05e1] = "\xb9\x72",
  [0x05e2] = "\xab\x4f", [0x05e3] = "\xb3\xf9", [0x05e4] = "\xb9\xa1",
  [0x05e5] = "\xc4\x5f", [0x05e6] = "\xa9\xea", [0x05e7] = "\xb3\xf8",
  [0x05e8] = "\xbc\xc9", [0x05e9] = "\xb0\x5c", [0x05ea] = "\xc0\x6a",
  [0x05eb] = "\xc3\x7a", [0x05ec] = "\xaa\x4d", [0x05ed] = "\xb8\x4f",
  [0x05ee] = "\xb4\x64", [0x05ef] = "\xa8\xf5", [0x05f0] = "\xa5\x5f",
  [0x05f1] = "\xbd\xfa", [0x05f2] = "\xad\x49", [0x05f3] = "\xa8\xa9",
  [0x05f4] = "\xbe\x58", [0x05f5] = "\xad\xbf", [0x05f6] = "\xaf\x56",
  [0x05f7] = "\xb3\xc6", [0x05f8] = "\xbe\xce", [0x05f9] = "\xb5\x48",
  [0x05fa] = "\xb3\x51", [0x05fb] = "\xa9\x62", [0x05fc] = "\xad\x66",
  [0x05fd] = "\xa5\xbb", [0x05fe] = "\xb2\xc2", [0x05ff] = "\xb1\x59",
  [0x0600] = "\xc1\x5e", [0x0601] = "\xac\xc7", [0x0602] = "\xac\xa6",
  [0x0603] = "\xc2\xdb", [0x0604] = "\xb0\x6e", [0x0605] = "\xb9\x47",
  [0x0606] = "\xbb\xf3", [0x0607] = "\xa4\xf1", [0x0608] = "\xbb\xc0",
  [0x0609] = "\xb5\xa7", [0x060a] = "\xa9\xbc", [0x060b] = "\xba\xd1",
  [0x060c] = "\xe3\xc9", [0x060d] = "\xbd\xaa", [0x060e] = "\xb2\xa6",
  [0x060f] = "\xc0\xc5", [0x0610] = "\xcf\x68", [0x0611] = "\xb9\xf4",
  [0x0612] = "\xa7\xc8", [0x0613] = "\xb7\xf4", [0x0614] = "\xb3\xac",
  [0x0615] = "\xb1\xcd", [0x0616] = "\xb9\xfa", [0x0617] = "\xa5\xb2",
  [0x0618] = "\xc5\x50", [0x0619] = "\xbe\xc0", [0x061a] = "\xc1\x75",
  [0x061b] = "\xc1\xd7", [0x061c] = "\xb0\xa1", [0x061d] = "\xc3\x40",
  [0x061e] = "\xc3\xe4", [0x061f] = "\xbd\x73", [0x0620] = "\xb6\x53",
  [0x0621] = "\xab\xf3", [0x0622] = "\xab\x4b", [0x0623] = "\xc5\xdc",
  [0x0624] = "\xa4\xcb", [0x0625] = "\xbf\xeb", [0x0626] = "\xc5\x47",
  [0x0627] = "\xc4\x7c", [0x0628] = "\xb9\x4d", [0x0629] = "\xbc\xd0",
  [0x062a] = "\xb3\x43", [0x062b] = "\xbd\xa8", [0x062c] = "\xaa\xed",
  [0x062d] = "\xc5\xbe", [0x062e] = "\xe5\x78", [0x062f] = "\xa7\x4f",
  [0x0630] = "\xc3\xa7", [0x0631] = "\xb1\x6c", [0x0632] = "\xd9\x79",
  [0x0633] = "\xc3\x78", [0x0634] = "\xc0\xd8", [0x0635] = "\xbb\xab",
  [0x0636] = "\xec\xe8", [0x0637] = "\xa7\x4c", [0x0638] = "\xa6\x42",
  [0x0639] = "\xac\x60", [0x063a] = "\xa4\xfe", [0x063b] = "\xaa\xc3",
  [0x063c] = "\xbb\xe6", [0x063d] = "\xac\xb1", [0x063e] = "\xaf\x66",
  [0x063f] = "\xa8\xc3", [0x0640] = "\xac\xc1", [0x0641] = "\xb5\xd4",
  [0x0642] = "\xbc\xbd", [0x0643] = "\xbc\xb7", [0x0644] = "\xb2\xda",
  [0x0645] = "\xaa\x69", [0x0646] = "\xb3\xd5", [0x0647] = "\xab\x6b",
  [0x0648] = "\xb7\x69", [0x0649] = "\xb9\x60", [0x064a] = "\xba\xe4",
  [0x064b] = "\xa7\x42", [0x064c] = "\xa9\xad", [0x064d] = "\xb2\xed",
  [0x064e] = "\xb2\xe4", [0x064f] = "\xbb\x4b", [0x0650] = "\xb4\xf1",
  [0x0651] = "\xaa\x79", [0x0652] = "\xbb\xe9", [0x0653] = "\xae\xb7",
  [0x0654] = "\xa4\x52", [0x0655] = "\xad\xf7", [0x0656] = "\xb8\xc9",
  [0x0657] = "\xb0\xf0", [0x0658] = "\xa4\xa3", [0x0659] = "\xa5\xac",
  [0x065a] = "\xa8\x42", [0x065b] = "\xc3\xaf", [0x065c] = "\xb3\xa1",
  [0x065d] = "\xa9\xc6", [0x065e] = "\xc0\xbf", [0x065f] = "\xb2\x71",
  [0x0660] = "\xb5\xf4", [0x0661] = "\xa7\xf7", [0x0662] = "\xa4\x7e",
  [0x0663] = "\xb0\x5d", [0x0664] = "\xb8\x42", [0x0665] = "\xbd\xf2",
  [0x0666] = "\xb1\xc4", [0x0667] = "\xb1\x6d", [0x0668] = "\xb5\xe6",
  [0x0669] = "\xbd\xb2", [0x066a] = "\xc0\x5c", [0x066b] = "\xb0\xd1",
  [0x066c] = "\xc5\xfa", [0x066d] = "\xb4\xdd", [0x066e] = "\xba\x46",
  [0x066f] = "\xba\x47", [0x0670] = "\xc0\xe9", [0x0671] = "\xbb\x61",
  [0x0672] = "\xbf\xb5", [0x0673] = "\xad\xdc", [0x0674] = "\xb7\xc9",
  [0x0675] = "\xc2\xc3", [0x0676] = "\xbe\xde", [0x0677] = "\xc1\x57",
  [0x0678] = "\xbc\xd1", [0x0679] = "\xb1\xe4", [0x067a] = "\xaf\xf3",
  [0x067b] = "\xb4\x5a", [0x067c] = "\xb5\xa6", [0x067d] = "\xb0\xbc",
  [0x067e] = "\xa5\x55", [0x067f] = "\xb4\xfa", [0x0680] = "\xbc\x68",
  [0x0681] = "\xf3\x4f", [0x0682] = "\xb4\xa1", [0x0683] = "\xa4\x65",
  [0x0684] = "\xd3\xb3", [0x0685] = "\xaf\xf9", [0x0686] = "\xac\x64",
  [0x0687] = "\xe2\xed", [0x0688] = "\xb7\x67", [0x0689] = "\xb9\xee",
  [0x068a] = "\xa7\xc3", [0x068b] = "\xae\x74", [0x068c] = "\xb8\xd2",
  [0x068d] = "\xa9\xee", [0x068e] = "\xae\xe3", [0x068f] = "\xb0\x5b",
  [0x0690] = "\xc4\x65", [0x0691] = "\xba\x55", [0x0692] = "\xc2\xcd",
  [0x0693] = "\xc6\x61", [0x0694] = "\xc5\xfe", [0x0695] = "\xc4\xf1",
  [0x0696] = "\xc3\xea", [0x0697] = "\xb2\xa3", [0x0698] = "\xc4\xc4",
  [0x0699] = "\xc5\xb8", [0x069a] = "\xa9\xf7", [0x069b] = "\xb2\x73",
  [0x069c] = "\xb3\xf5", [0x069d] = "\xb9\xc1", [0x069e] = "\xb1\x60",
  [0x069f] = "\xaa\xf8", [0x06a0] = "\xc0\x76", [0x06a1] = "\xb8\x7a",
  [0x06a2] = "\xbc\x74", [0x06a3] = "\xb4\xaf", [0x06a4] = "\xba\x5a",
  [0x06a5] = "\xb0\xdb", [0x06a6] = "\xad\xd2", [0x06a7] = "\xb6\x57",
  [0x06a8] = "\xa7\xdb", [0x06a9] = "\xb6\x72", [0x06aa] = "\xb4\xc2",
  [0x06ab] = "\xbc\x4a", [0x06ac] = "\xbc\xe9", [0x06ad] = "\xb1\x5f",
  [0x06ae] = "\xa7\x6e", [0x06af] = "\xaa\xa3", [0x06b0] = "\xa8\xae",
  [0x06b1] = "\xa7\xe8", [0x06b2] = "\xba\x4d", [0x06b3] = "\xb4\x77",
  [0x06b4] = "\xb9\xfd", [0x06b5] = "\xba\xab", [0x06b6] = "\xd7\xeb",
  [0x06b7] = "\xa6\xda", [0x06b8] = "\xa8\xb0", [0x06b9] = "\xb9\xd0",
  [0x06ba] = "\xb1\xe1", [0x06bb] = "\xa7\xd5", [0x06bc] = "\xa8\x49",
  [0x06bd] = "\xb3\xaf", [0x06be] = "\xb6\x58", [0x06bf] = "\xc5\xa8",
  [0x06c0] = "\xbc\xb5", [0x06c1] = "\xba\xd9", [0x06c2] = "\xab\xb0",
  [0x06c3] = "\xbe\xed", [0x06c4] = "\xa6\xa8", [0x06c5] = "\xa7\x65",
  [0x06c6] = "\xad\xbc", [0x06c7] = "\xb5\x7b", [0x06c8] = "\xc3\x67",
  [0x06c9] = "\xbc\xe1", [0x06ca] = "\xb8\xdb", [0x06cb] = "\xa9\xd3",
  [0x06cc] = "\xb3\x78", [0x06cd] = "\xc1\xfe", [0x06ce] = "\xaf\xaf",
  [0x06cf] = "\xa6\x59", [0x06d0] = "\xc3\xa8", [0x06d1] = "\xab\xf9",
  [0x06d2] = "\xb0\xcd", [0x06d3] = "\xa6\xc0", [0x06d4] = "\xbf\xf0",
  [0x06d5] = "\xa6\xa2", [0x06d6] = "\xb9\xa3", [0x06d7] = "\xae\xa2",
  [0x06d8] = "\xbe\xa6", [0x06d9] = "\xa8\xd7", [0x06da] = "\xa4\xd8",
  [0x06db] = "\xa8\xaa", [0x06dc] = "\xaf\xcd", [0x06dd] = "\xa5\xb8",
  [0x06de] = "\xbf\x4b", [0x06df] = "\xa5\x52", [0x06e0] = "\xbd\xc4",
  [0x06e1] = "\xc2\xce", [0x06e2] = "\xb1\x52", [0x06e3] = "\xc3\x64",
  [0x06e4] = "\xa9\xe2", [0x06e5] = "\xb9\x53", [0x06e6] = "\xc3\xa5",
  [0x06e7] = "\xc5\x43", [0x06e8] = "\xb8\x59", [0x06e9] = "\xb7\x54",
  [0x06ea] = "\xc4\x77", [0x06eb] = "\xa4\xb3", [0x06ec] = "\xba\xf7",
  [0x06ed] = "\xe2\xe4", [0x06ee] = "\xc1\xe0", [0x06ef] = "\xaf\xe4",
  [0x06f0] = "\xaa\xec", [0x06f1] = "\xa5\x58", [0x06f2] = "\xc3\x6f",
  [0x06f3] = "\xbc\x70", [0x06f4] = "\xc3\xdc", [0x06f5] = "\xbe\x53",
  [0x06f6] = "\xc2\xfa", [0x06f7] = "\xde\x51", [0x06f8] = "\xb0\xa3",
  [0x06f9] = "\xb7\xa1", [0x06fa] = "\xc2\xa6", [0x06fb] = "\xc0\x78",
  [0x06fc] = "\xc5\xf7", [0x06fd] = "\xdd\x7a", [0x06fe] = "\xc4\xb2",
  [0x06ff] = "\xb3\x42", [0x0700] = "\xb4\xa2", [0x0701] = "\xa4\x74",
  [0x0702] = "\xac\xef", [0x0703] = "\xdd\xdc", [0x0704] = "\xb6\xc7",
  [0x0705] = "\xb2\xee", [0x0706] = "\xb3\xdd", [0x0707] = "\xa6\xea",
  [0x0708] = "\xbd\x48", [0x0709] = "\xb5\xa1", [0x070a] = "\xbc\x6c",
  [0x070b] = "\xa7\xc9", [0x070c] = "\xc2\xf4", [0x070d] = "\xb3\xd0",
  [0x070e] = "\xa7\x6a", [0x070f] = "\xaa\xa4", [0x0710] = "\xb4\xa8",
  [0x0711] = "\xc2\xf1", [0x0712] = "\xab\xab", [0x0713] = "\xac\x4b",
  [0x0714] = "\xdd\xcf", [0x0715] = "\xbe\x4a", [0x0716] = "\xae\x42",
  [0x0717] = "\xb2\x45", [0x0718] = "\xaf\xc2", [0x0719] = "\xc4\xf8",
  [0x071a] = "\xc2\x57", [0x071b] = "\xba\xef", [0x071c] = "\xb2\xab",
  [0x071d] = "\xaf\xfd", [0x071e] = "\xba\xcf", [0x071f] = "\xbb\xdb",
  [0x0720] = "\xc3\xe3", [0x0721] = "\xb7\x4f", [0x0722] = "\xb2\xa1",
  [0x0723] = "\xb5\xfc", [0x0724] = "\xa6\xb9", [0x0725] = "\xa8\xeb",
  [0x0726] = "\xbd\xe7", [0x0727] = "\xa6\xb8", [0x0728] = "\xc1\x6f",
  [0x0729] = "\xbd\xb5", [0x072a] = "\xa7\x77", [0x072b] = "\xa5\x5e",
  [0x072c] = "\xb1\x71", [0x072d] = "\xc2\x4f", [0x072e] = "\xb4\xea",
  [0x072f] = "\xb2\xca", [0x0730] = "\xbe\x4c", [0x0731] = "\xc1\x4c",
  [0x0732] = "\xab\x50", [0x0733] = "\xf9\x4f", [0x0734] = "\xbf\x79",
  [0x0735] = "\xc2\xab", [0x0736] = "\xba\x52", [0x0737] = "\xb1\x5a",
  [0x0738] = "\xb6\xca", [0x0739] = "\xaf\xdc", [0x073a] = "\xb7\xf1",
  [0x073b] = "\xba\xe9", [0x073c] = "\xb2\x66", [0x073d] = "\xbb\x41",
  [0x073e] = "\xa7\xf8", [0x073f] = "\xa6\x73", [0x0740] = "\xa4\x6f",
  [0x0741] = "\xbd\x52", [0x0742] = "\xbc\xbc", [0x0743] = "\xb7\x62",
  [0x0744] = "\xb1\xb9", [0x0745] = "\xae\xc0", [0x0746] = "\xbf\xf9",
  [0x0747] = "\xb7\x66", [0x0748] = "\xb9\x46", [0x0749] = "\xb5\xaa",
  [0x074a] = "\xbd\x44", [0x074b] = "\xa5\xb4", [0x074c] = "\xa4\x6a",
  [0x074d] = "\xa7\x62", [0x074e] = "\xa4\xef", [0x074f] = "\xd8\x56",
  [0x0750] = "\xc0\xb9", [0x0751] = "\xb1\x61", [0x0752] = "\xac\x70",
  [0x0753] = "\xa5\x4e", [0x0754] = "\xb6\x55", [0x0755] = "\xb3\x55",
  [0x0756] = "\xab\xdd", [0x0757] = "\xb6\x65", [0x0758] = "\xab\xe5",
  [0x0759] = "\xaf\xd4", [0x075a] = "\xbe\xe1", [0x075b] = "\xa4\xa6",
  [0x075c] = "\xb3\xe6", [0x075d] = "\xe8\x58", [0x075e] = "\xbc\xc2",
  [0x075f] = "\xc1\x78", [0x0760] = "\xa5\xb9", [0x0761] = "\xb4\xe1",
  [0x0762] = "\xa6\xfd", [0x0763] = "\xbc\xaa", [0x0764] = "\xb2\x48",
  [0x0765] = "\xbd\xcf", [0x0766] = "\xbc\x75", [0x0767] = "\xb3\x4a",
  [0x0768] = "\xb7\xed", [0x0769] = "\xbe\xd7", [0x076a] = "\xc4\xd2",
  [0x076b] = "\xbf\xba", [0x076c] = "\xc0\xc9", [0x076d] = "\xa4\x4d",
  [0x076e] = "\xb7\x6f", [0x076f] = "\xc1\xd0", [0x0770] = "\xad\xcb",
  [0x0771] = "\xae\x71", [0x0772] = "\xc3\xab", [0x0773] = "\xbe\xc9",
  [0x0774] = "\xa8\xec", [0x0775] = "\xbd\x5f", [0x0776] = "\xb1\xa5",
  [0x0777] = "\xb9\x44", [0x0778] = "\xb5\x73", [0x0779] = "\xbc\x77",
  [0x077a] = "\xb1\x6f", [0x077b] = "\xaa\xba", [0x077c] = "\xc3\xde",
  [0x077d] = "\xbf\x4f", [0x077e] = "\xb5\x6e", [0x077f] = "\xb5\xa5",
  [0x0780] = "\xc0\xfc", [0x0781] = "\xb9\xb9", [0x0782] = "\xbe\x48",
  [0x0783] = "\xb3\xf6", [0x0784] = "\xa7\x43", [0x0785] = "\xba\x77",
  [0x0786] = "\xad\x7d", [0x0787] = "\xbc\xc4", [0x0788] = "\xb2\xc3",
  [0x0789] = "\xa8\x66", [0x078a] = "\xba\xb0", [0x078b] = "\xbb\x43",
  [0x078c] = "\xb9\xde", [0x078d] = "\xa9\xe8", [0x078e] = "\xa9\xb3",
  [0x078f] = "\xa6\x61", [0x0790] = "\xb8\xa6", [0x0791] = "\xb2\xc4",
  [0x0792] = "\xab\xd2", [0x0793] = "\xa7\xcc", [0x0794] = "\xbb\xbc",
  [0x0795] = "\xbd\x6c", [0x0796] = "\xc4\x41", [0x0797] = "\xd5\x69",
  [0x0798] = "\xb7\xbf", [0x0799] = "\xb8\x4b", [0x079a] = "\xc2\x49",
  [0x079b] = "\xa8\xe5", [0x079c] = "\xc0\x51", [0x079d] = "\xb9\xd4",
  [0x079e] = "\xb9\x71", [0x079f] = "\xa6\xfa", [0x07a0] = "\xa8\x6c",
  [0x07a1] = "\xa9\xb1", [0x07a2] = "\xb1\x7d", [0x07a3] = "\xb3\xfe",
  [0x07a4] = "\xbe\xfd", [0x07a5] = "\xb7\xb5", [0x07a6] = "\xb8\x4d",
  [0x07a7] = "\xa5\x70", [0x07a8] = "\xc0\x4a", [0x07a9] = "\xad\xe4",
  [0x07aa] = "\xa4\x4e", [0x07ab] = "\xb1\xbc", [0x07ac] = "\xa6\x51",
  [0x07ad] = "\xb3\xa8", [0x07ae] = "\xbd\xd5", [0x07af] = "\xb6\x5e",
  [0x07b0] = "\xaf\x52", [0x07b1] = "\xba\xd0", [0x07b2] = "\xbd\xba",
  [0x07b3] = "\xad\xa1", [0x07b4] = "\xbf\xd2", [0x07b5] = "\xc5\x7c",
  [0x07b6] = "\xa4\x42", [0x07b7] = "\xa8\x6e", [0x07b8] = "\xa5\x6d",
  [0x07b9] = "\xb0\x76", [0x07ba] = "\xb3\xbb", [0x07bb] = "\xb9\xa9",
  [0x07bc] = "\xbf\xf5", [0x07bd] = "\xa9\x77", [0x07be] = "\xad\x71",
  [0x07bf] = "\xa5\xe1", [0x07c0] = "\xaa\x46", [0x07c1] = "\xa5\x56",
  [0x07c2] = "\xb8\xb3", [0x07c3] = "\xc0\xb4", [0x07c4] = "\xb0\xca",
  [0x07c5] = "\xb4\xc9", [0x07c6] = "\xcb\xbe", [0x07c7] = "\xab\xf0",
  [0x07c8] = "\xad\xe1", [0x07c9] = "\xac\x7d", [0x07ca] = "\xb0\xc2",
  [0x07cb] = "\xa7\xdd", [0x07cc] = "\xb0\xab", [0x07cd] = "\xb0\x7e",
  [0x07ce] = "\xa8\xa7", [0x07cf] = "\xb3\x72", [0x07d0] = "\xb5\x6b",
  [0x07d1] = "\xb3\xa3", [0x07d2] = "\xb7\xfe", [0x07d3] = "\xac\x72",
  [0x07d4] = "\xc3\x7d", [0x07d5] = "\xbf\x57", [0x07d6] = "\xc5\xaa",
  [0x07d7] = "\xb0\xf4", [0x07d8] = "\xb8\x40", [0x07d9] = "\xbd\xe4",
  [0x07da] = "\xa7\xf9", [0x07db] = "\xc1\xe1", [0x07dc] = "\xa8\x7b",
  [0x07dd] = "\xab\xd7", [0x07de] = "\xb4\xe7", [0x07df] = "\xa7\xaa",
  [0x07e0] = "\xba\xdd", [0x07e1] = "\xb5\x75", [0x07e2] = "\xc1\xeb",
  [0x07e3] = "\xac\x71", [0x07e4] = "\xc2\x5f", [0x07e5] = "\xbd\x76",
  [0x07e6] = "\xb0\xef", [0x07e7] = "\xa7\x49", [0x07e8] = "\xb6\xa4",
  [0x07e9] = "\xb9\xef", [0x07ea] = "\xbc\x5b", [0x07eb] = "\xbe\xb7",
  [0x07ec] = "\xc3\xdb", [0x07ed] = "\xb4\xb0", [0x07ee] = "\xb9\x79",
  [0x07ef] = "\xa7\x79", [0x07f0] = "\xb6\x77", [0x07f1] = "\xac\xde",
  [0x07f2] = "\xb9\x50", [0x07f3] = "\xd5\x73", [0x07f4] = "\xce\x47",
  [0x07f5] = "\xa6\x68", [0x07f6] = "\xb9\xdc", [0x07f7] = "\xce\x57",
  [0x07f8] = "\xb8\xfa", [0x07f9] = "\xa6\xb7", [0x07fa] = "\xb8\xf6",
  [0x07fb] = "\xb2\xeb", [0x07fc] = "\xa8\xef", [0x07fd] = "\xb4\x6b",
  [0x07fe] = "\xbc\x5a", [0x07ff] = "\xb8\xbf", [0x0800] = "\xae\x6f",
  [0x0801] = "\xc3\x5a", [0x0802] = "\xab\x58", [0x0803] = "\xc3\x42",
  [0x0804] = "\xb3\x5f", [0x0805] = "\xae\x5a", [0x0806] = "\xb4\x63",
  [0x0807] = "\xa4\xcc", [0x0808] = "\xa7\xe3", [0x0809] = "\xb9\x4b",
  [0x080a] = "\xb6\x6b", [0x080b] = "\xbe\x6a", [0x080c] = "\xae\xa6",
  [0x080d] = "\xa6\xd3", [0x080e] = "\xa8\xe0", [0x080f] = "\xa6\xd5",
  [0x0810] = "\xba\xb8", [0x0811] = "\xbb\xe7", [0x0812] = "\xac\x7c",
  [0x0813] = "\xa4\x47", [0x0814] = "\xb6\x4c", [0x0815] = "\xb5\x6f",
  [0x0816] = "\xbb\x40", [0x0817] = "\xb5\xad", [0x0818] = "\xa5\xef",
  [0x0819] = "\xa5\x46", [0x081a] = "\xbb\xd6", [0x081b] = "\xaa\x6b",
  [0x081c] = "\xb5\x57", [0x081d] = "\xc3\xbf", [0x081e] = "\xa6\x7c",
  [0x081f] = "\xb5\x66", [0x0820] = "\xc2\xbd", [0x0821] = "\xbc\xd4",
  [0x0822] = "\xc4\x73", [0x0823] = "\xb3\xab", [0x0824] = "\xc1\x63",
  [0x0825] = "\xa4\x5a", [0x0826] = "\xb7\xd0", [0x0827] = "\xa4\xcf",
  [0x0828] = "\xaa\xf0", [0x0829] = "\xbd\x64", [0x082a] = "\xb3\x63",
  [0x082b] = "\xa5\xc7", [0x082c] = "\xb6\xba", [0x082d] = "\xaa\x78",
  [0x082e] = "\xa7\x7b", [0x082f] = "\xaa\xda", [0x0830] = "\xa4\xe8",
  [0x0831] = "\xaa\xd5", [0x0832] = "\xa9\xd0", [0x0833] = "\xa8\xbe",
  [0x0834] = "\xa7\xab", [0x0835] = "\xa5\xe9", [0x0836] = "\xb3\x58",
  [0x0837] = "\xaf\xbc", [0x0838] = "\xa9\xf1", [0x0839] = "\xb5\xe1",
  [0x083a] = "\xab\x44", [0x083b] = "\xb0\xd8", [0x083c] = "\xad\xb8",
  [0x083d] = "\xaa\xce", [0x083e] = "\xad\xea", [0x083f] = "\xbd\xda",
  [0x0840] = "\xa7\x70", [0x0841] = "\xaa\xcd", [0x0842] = "\xbc\x6f",
  [0x0843] = "\xaa\x6d", [0x0844] = "\xb6\x4f", [0x0845] = "\xaa\xe2",
  [0x0846] = "\xd7\xf4", [0x0847] = "\xa7\x68", [0x0848] = "\xaa\x5e",
  [0x0849] = "\xa4\xc0", [0x084a] = "\xaf\xc9", [0x084b] = "\xbc\x58",
  [0x084c] = "\xb5\x49", [0x084d] = "\xa8\x57", [0x084e] = "\xaf\xbb",
  [0x084f] = "\xbe\xc4", [0x0850] = "\xa5\xf7", [0x0851] = "\xa9\xc1",
  [0x0852] = "\xbc\xab", [0x0853] = "\xc1\x54", [0x0854] = "\xc2\xd7",
  [0x0855] = "\xab\xca", [0x0856] = "\xb7\xac", [0x0857] = "\xb8\xc1",
  [0x0858] = "\xae\x70", [0x0859] = "\xbe\x57", [0x085a] = "\xad\xb7",
  [0x085b] = "\xba\xc6", [0x085c] = "\xb2\x6c", [0x085d] = "\xb3\x7b",
  [0x085e] = "\xb6\xbe", [0x085f] = "\xc1\x5f", [0x0860] = "\xbf\xd8",
  [0x0861] = "\xa9\x5e", [0x0862] = "\xbb\xf1", [0x0863] = "\xa6\xf2",
  [0x0864] = "\xa7\x5f", [0x0865] = "\xa4\xd2", [0x0866] = "\xbc\xc5",
  [0x0867] = "\xbd\xa7", [0x0868] = "\xb9\xe5", [0x0869] = "\xa7\xdf",
  [0x086a] = "\xa9\xd8", [0x086b] = "\xbf\xe7", [0x086c] = "\xb4\x54",
  [0x086d] = "\xac\x74", [0x086e] = "\xb2\xc5", [0x086f] = "\xa5\xf1",
  [0x0870] = "\xab\x52", [0x0871] = "\xaa\x41", [0x0872] = "\xaf\x42",
  [0x0873] = "\xb2\x65", [0x0874] = "\xba\xd6", [0x0875] = "\xb5\xf6",
  [0x0876] = "\xa5\xb1", [0x0877] = "\xa8\x6a", [0x0878] = "\xbc\xbe",
  [0x0879] = "\xbb\xb2", [0x087a] = "\xad\xc1", [0x087b] = "\xb0\x79",
  [0x087c] = "\xa9\xf2", [0x087d] = "\xb2\xe3", [0x087e] = "\xb5\xc6",
  [0x087f] = "\xa9\xb2", [0x0880] = "\xbb\x47", [0x0881] = "\xad\x75",
  [0x0882] = "\xb0\xc6", [0x0883] = "\xc2\xd0", [0x0884] = "\xbd\xe1",
  [0x0885] = "\xb4\x5f", [0x0886] = "\xb3\xc5", [0x0887] = "\xa5\x49",
  [0x0888] = "\xaa\xfa", [0x0889] = "\xa4\xf7", [0x088a] = "\xb8\xa1",
  [0x088b] = "\xad\x74", [0x088c] = "\xb4\x49", [0x088d] = "\xad\x72",
  [0x088e] = "\xaa\xfe", [0x088f] = "\xb0\xfc", [0x0890] = "\xbf\xa3",
  [0x0891] = "\xa9\x4a", [0x0892] = "\xbe\xbf", [0x0893] = "\xb9\xc7",
  [0x0894] = "\xb8\xd3", [0x0895] = "\xa7\xef", [0x0896] = "\xb7\xa7",
  [0x0897] = "\xb6\x74", [0x0898] = "\xbb\x5c", [0x0899] = "\xb5\x40",
  [0x089a] = "\xb7\x46", [0x089b] = "\xa5\xcc", [0x089c] = "\xb1\xec",
  [0x089d] = "\xac\x61", [0x089e] = "\xac\xf1", [0x089f] = "\xa8\x78",
  [0x08a0] = "\xbb\xb0", [0x08a1] = "\xb7\x50", [0x08a2] = "\xb5\x7a",
  [0x08a3] = "\xb4\xb1", [0x08a4] = "\xc6\x42", [0x08a5] = "\xa9\xa3",
  [0x08a6] = "\xad\xe8", [0x08a7] = "\xbf\xfb", [0x08a8] = "\xac\xfb",
  [0x08a9] = "\xa8\x7a", [0x08aa] = "\xba\xf5", [0x08ab] = "\xb1\x5e",
  [0x08ac] = "\xb4\xe4", [0x08ad] = "\xba\x62", [0x08ae] = "\xbf\x74",
  [0x08af] = "\xaf\x6f", [0x08b0] = "\xb0\xaa", [0x08b1] = "\xbb\x49",
  [0x08b2] = "\xaf\xcc", [0x08b3] = "\xbf\x7c", [0x08b4] = "\xb7\x64",
  [0x08b5] = "\xc2\xee", [0x08b6] = "\xbd\x5a", [0x08b7] = "\xa7\x69",
  [0x08b8] = "\xad\xf4", [0x08b9] = "\xba\x71", [0x08ba] = "\xc0\xc1",
  [0x08bb] = "\xa4\xe0", [0x08bc] = "\xc2\x46", [0x08bd] = "\xaf\xde",
  [0x08be] = "\xaa\xb8", [0x08bf] = "\xb3\xce", [0x08c0] = "\xad\xb2",
  [0x08c1] = "\xb8\xaf", [0x08c2] = "\xae\xe6", [0x08c3] = "\xb5\xf0",
  [0x08c4] = "\xbb\xd5", [0x08c5] = "\xb9\x6a", [0x08c6] = "\xbb\xcc",
  [0x08c7] = "\xad\xd3", [0x08c8] = "\xa6\x55", [0x08c9] = "\xb5\xb9",
  [0x08ca] = "\xae\xda", [0x08cb] = "\xb8\xf2", [0x08cc] = "\xaf\xd1",
  [0x08cd] = "\xa7\xf3", [0x08ce] = "\xa9\xb0", [0x08cf] = "\xc3\xbc",
  [0x08d0] = "\xae\x47", [0x08d1] = "\xaf\xd5", [0x08d2] = "\xb1\xf0",
  [0x08d3] = "\xa4\x75", [0x08d4] = "\xa7\xf0", [0x08d5] = "\xa5\x5c",
  [0x08d6] = "\xae\xa5", [0x08d7] = "\xc5\xc7", [0x08d8] = "\xa8\xd1",
  [0x08d9] = "\xb0\x60", [0x08da] = "\xa4\xbd", [0x08db] = "\xae\x63",
  [0x08dc] = "\xa4\x7d", [0x08dd] = "\xbe\x64", [0x08de] = "\xa8\x45",
  [0x08df] = "\xab\xfd", [0x08e0] = "\xb0\x5e", [0x08e1] = "\xa6\x40",
  [0x08e2] = "\xb9\x5f", [0x08e3] = "\xa4\xc4", [0x08e4] = "\xb7\xbe",
  [0x08e5] = "\xad\x65", [0x08e6] = "\xaa\xaf", [0x08e7] = "\xab\xaf",
  [0x08e8] = "\xba\x63", [0x08e9] = "\xc1\xca", [0x08ea] = "\xb0\xf7",
  [0x08eb] = "\xb6\x64", [0x08ec] = "\xdb\xa3", [0x08ed] = "\xa9\x42",
  [0x08ee] = "\xe3\x54", [0x08ef] = "\xa6\xf4", [0x08f0] = "\xaa\x66",
  [0x08f1] = "\xa9\x74", [0x08f2] = "\xa9\x68", [0x08f3] = "\xb9\xaa",
  [0x08f4] = "\xa5\x6a", [0x08f5] = "\xc5\xdb", [0x08f6] = "\xb0\xa9",
  [0x08f7] = "\xa8\xa6", [0x08f8] = "\xaa\xd1", [0x08f9] = "\xac\x47",
  [0x08fa] = "\xc5\x55", [0x08fb] = "\xa9\x54", [0x08fc] = "\xb6\xb1",
  [0x08fd] = "\xa8\xed", [0x08fe] = "\xa5\xca", [0x08ff] = "\xd4\x6c",
  [0x0900] = "\xb9\xe8", [0x0901] = "\xb1\xbe", [0x0902] = "\xbb\x6f",
  [0x0903] = "\xa8\xc4", [0x0904] = "\xa9\xe4", [0x0905] = "\xa9\xc7",
  [0x0906] = "\xb4\xc3", [0x0907] = "\xc3\xf6", [0x0908] = "\xa9\x78",
  [0x0909] = "\xab\x61", [0x090a] = "\xc6\x5b", [0x090b] = "\xba\xde",
  [0x090c] = "\xc0\x5d", [0x090d] = "\xc5\xf8", [0x090e] = "\xba\x44",
  [0x090f] = "\xc4\xe9", [0x0910] = "\xb3\x65", [0x0911] = "\xa5\xfa",
  [0x0912] = "\xbc\x73", [0x0913] = "\xb3\x7d", [0x0914] = "\xba\xc0",
  [0x0915] = "\xb3\x57", [0x0916] = "\xa6\x63", [0x0917] = "\xd6\xba",
  [0x0918] = "\xc2\x6b", [0x0919] = "\xc0\x74", [0x091a] = "\xbb\xd3",
  [0x091b] = "\xad\x79", [0x091c] = "\xb0\xad", [0x091d] = "\xb8\xde",
  [0x091e] = "\xac\xd1", [0x091f] = "\xae\xdb", [0x0920] = "\xc2\x64",
  [0x0921] = "\xb8\xf7", [0x0922] = "\xb6\x51", [0x0923] = "\xbc\x44",
  [0x0924] = "\xbe\x40", [0x0925] = "\xba\x75", [0x0926] = "\xb4\xd2",
  [0x0927] = "\xc1\xe7", [0x0928] = "\xb3\xa2", [0x0929] = "\xb0\xea",
  [0x092a] = "\xaa\x47", [0x092b] = "\xbb\x71", [0x092c] = "\xb9\x4c",
  [0x092d] = "\xab\xa2", [0x092e] = "\xc0\x65", [0x092f] = "\xab\xc4",
  [0x0930] = "\xae\xfc", [0x0931] = "\xae\xf3", [0x0932] = "\xa5\xe8",
  [0x0933] = "\xae\x60", [0x0934] = "\xc0\x62", [0x0935] = "\xb6\x6f",
  [0x0936] = "\xe9\x77", [0x0937] = "\xcd\xd5", [0x0938] = "\xc1\xfa",
  [0x0939] = "\xa7\x74", [0x093a] = "\xb2\x5b", [0x093b] = "\xb4\x48",
  [0x093c] = "\xa8\xe7", [0x093d] = "\xb3\xdb", [0x093e] = "\xa8\x75",
  [0x093f] = "\xbf\xab", [0x0940] = "\xbe\xd9", [0x0941] = "\xae\xc2",
  [0x0942] = "\xa7\xf2", [0x0943] = "\xbe\xd1", [0x0944] = "\xae\xab",
  [0x0945] = "\xb2\x6b", [0x0946] = "\xa6\xbd", [0x0947] = "\xba\x7e",
  [0x0948] = "\xc9\x71", [0x0949] = "\xaa\x43", [0x094a] = "\xaf\xe8",
  [0x094b] = "\xc0\xa2", [0x094c] = "\xc0\x7a", [0x094d] = "\xbb\xa8",
  [0x094e] = "\xb2\x40", [0x094f] = "\xb0\x71", [0x0950] = "\xa6\x6e",
  [0x0951] = "\xaf\xd3", [0x0952] = "\xb8\xb9", [0x0953] = "\xaf\x45",
  [0x0954] = "\xa8\xfe", [0x0955] = "\xb3\xdc", [0x0956] = "\xb2\xfc",
  [0x0957] = "\xdb\x5d", [0x0958] = "\xae\xd6", [0x0959] = "\xa5\xdd",
  [0x095a] = "\xa9\x4d", [0x095b] = "\xa6\xf3", [0x095c] = "\xa6\x58",
  [0x095d] = "\xb2\xb0", [0x095e] = "\xb8\xe8", [0x095f] = "\xbb\xd2",
  [0x0960] = "\xaa\x65", [0x0961] = "\xb2\x55", [0x0962] = "\xbb\xae",
  [0x0963] = "\xbd\xc5", [0x0964] = "\xc5\x62", [0x0965] = "\xb6\x50",
  [0x0966] = "\xbc\x4b", [0x0967] = "\xb6\xc2", [0x0968] = "\xb2\xaa",
  [0x0969] = "\xab\xdc", [0x096a] = "\xac\xbd", [0x096b] = "\xab\xeb",
  [0x096c] = "\xad\xf3", [0x096d] = "\xa6\xeb", [0x096e] = "\xbe\xee",
  [0x096f] = "\xbf\xc5", [0x0970] = "\xab\xed", [0x0971] = "\xc5\x46",
  [0x0972] = "\xab\xa1", [0x0973] = "\xaf\x4d", [0x0974] = "\xad\x69",
  [0x0975] = "\xc2\x45", [0x0976] = "\xac\x78", [0x0977] = "\xa7\xbb",
  [0x0978] = "\xa5\xb0", [0x0979] = "\xac\xf5", [0x097a] = "\xb3\xef",
  [0x097b] = "\xab\x4a", [0x097c] = "\xb5\x55", [0x097d] = "\xa7\x71",
  [0x097e] = "\xab\x70", [0x097f] = "\xad\xd4", [0x0980] = "\xab\xe1",
  [0x0981] = "\xa9\x49", [0x0982] = "\xa5\x47", [0x0983] = "\xa9\xbf",
  [0x0984] = "\xb7\xe4", [0x0985] = "\xb3\xfd", [0x0986] = "\xb8\xac",
  [0x0987] = "\xad\x4a", [0x0988] = "\xbd\xb9", [0x0989] = "\xaa\xb0",
  [0x098a] = "\xbd\x6b", [0x098b] = "\xb4\xf2", [0x098c] = "\xa9\xb7",
  [0x098d] = "\xaa\xea", [0x098e] = "\xb0\xe4", [0x098f] = "\xc5\x40",
  [0x0990] = "\xa4\xac", [0x0991] = "\xba\xad", [0x0992] = "\xa4\xe1",
  [0x0993] = "\xaa\xe1", [0x0994] = "\xbc\x4d", [0x0995] = "\xb5\xd8",
  [0x0996] = "\xb7\xe2", [0x0997] = "\xb7\xc6", [0x0998] = "\xb5\x65",
  [0x0999] = "\xb9\xba", [0x099a] = "\xa4\xc6", [0x099b] = "\xb8\xdc",
  [0x099c] = "\xba\x69", [0x099d] = "\xab\xde", [0x099e] = "\xc3\x68",
  [0x099f] = "\xb2\x61", [0x09a0] = "\xc3\x61", [0x09a1] = "\xc5\x77",
  [0x09a2] = "\xc0\xf4", [0x09a3] = "\xae\xd9", [0x09a4] = "\xc1\xd9",
  [0x09a5] = "\xbd\x77", [0x09a6] = "\xb4\xab", [0x09a7] = "\xb1\x77",
  [0x09a8] = "\xb3\xea", [0x09a9] = "\xba\xc8", [0x09aa] = "\xb8\xe6",
  [0x09ab] = "\xb7\xd8", [0x09ac] = "\xb5\x41", [0x09ad] = "\xab\xc6",
  [0x09ae] = "\xa4\xdb", [0x09af] = "\xaf\xee", [0x09b0] = "\xb7\x57",
  [0x09b1] = "\xb6\xc0", [0x09b2] = "\xc1\x44", [0x09b3] = "\xbd\xc0",
  [0x09b4] = "\xc2\xae", [0x09b5] = "\xac\xd3", [0x09b6] = "\xb0\xc4",
  [0x09b7] = "\xb4\x71", [0x09b8] = "\xb7\xd7", [0x09b9] = "\xae\xcc",
  [0x09ba] = "\xb7\x45", [0x09bb] = "\xab\xe9", [0x09bc] = "\xc1\xc0",
  [0x09bd] = "\xa6\xc7", [0x09be] = "\xb4\xa7", [0x09bf] = "\xbd\xf7",
  [0x09c0] = "\xc0\xb2", [0x09c1] = "\xab\xec", [0x09c2] = "\xb5\xee",
  [0x09c3] = "\xa6\x5e", [0x09c4] = "\xb7\xb4", [0x09c5] = "\xae\xac",
  [0x09c6] = "\xbc\x7a", [0x09c7] = "\xa5\x63", [0x09c8] = "\xb4\x66",
  [0x09c9] = "\xb1\xe2", [0x09ca] = "\xb8\xec", [0x09cb] = "\xc2\xa9",
  [0x09cc] = "\xb7\x7c", [0x09cd] = "\xc0\xed", [0x09ce] = "\xb6\xd7",
  [0x09cf] = "\xbf\xd0", [0x09d0] = "\xbb\xa3", [0x09d1] = "\xc3\xb8",
  [0x09d2] = "\xb8\xa7", [0x09d3] = "\xa9\xfc", [0x09d4] = "\xb1\x42",
  [0x09d5] = "\xbb\xee", [0x09d6] = "\xb4\xfd", [0x09d7] = "\xb2\x56",
  [0x09d8] = "\xc1\xc5", [0x09d9] = "\xac\xa1", [0x09da] = "\xa5\xeb",
  [0x09db] = "\xa4\xf5", [0x09dc] = "\xc0\xf2", [0x09dd] = "\xa9\xce",
  [0x09de] = "\xb4\x62", [0x09df] = "\xc0\x4e", [0x09e0] = "\xb3\x66",
  [0x09e1] = "\xba\xd7", [0x09e2] = "\xc0\xbb", [0x09e3] = "\xa7\xa3",
  [0x09e4] = "\xb0\xf2", [0x09e5] = "\xbe\xf7", [0x09e6] = "\xb7\xee",
  [0x09e7] = "\xbd\x5d", [0x09e8] = "\xbf\x6e", [0x09e9] = "\xba\xdf",
  [0x09ea] = "\xa6\xd9", [0x09eb] = "\xb0\xa7", [0x09ec] = "\xc2\xdd",
  [0x09ed] = "\xbf\x45", [0x09ee] = "\xc3\xd5", [0x09ef] = "\xc2\xfb",
  [0x09f0] = "\xae\x56", [0x09f1] = "\xc1\x5a", [0x09f2] = "\xbd\x72",
  [0x09f3] = "\xa6\x4e", [0x09f4] = "\xb7\xa5", [0x09f5] = "\xb4\xc6",
  [0x09f6] = "\xbf\xe8", [0x09f7] = "\xc4\x79", [0x09f8] = "\xb6\xb0",
  [0x09f9] = "\xa4\xce", [0x09fa] = "\xab\xe6", [0x09fb] = "\xaf\x65",
  [0x09fc] = "\xa8\x56", [0x09fd] = "\xa7\x59", [0x09fe] = "\xb6\xfa",
  [0x09ff] = "\xaf\xc5", [0x0a00] = "\xc0\xbd", [0x0a01] = "\xb4\x58",
  [0x0a02] = "\xaf\xe1", [0x0a03] = "\xa4\x76", [0x0a04] = "\xc1\xaa",
  [0x0a05] = "\xa7\xde", [0x0a06] = "\xbe\xac", [0x0a07] = "\xa9\x75",
  [0x0a08] = "\xc9\xab", [0x0a09] = "\xb2\xbd", [0x0a0a] = "\xbe\xaf",
  [0x0a0b] = "\xb1\xaa", [0x0a0c] = "\xc0\xd9", [0x0a0d] = "\xb1\x48",
  [0x0a0e] = "\xb1\x49", [0x0a0f] = "\xad\x70", [0x0a10] = "\xb0\x4f",
  [0x0a11] = "\xac\x4a", [0x0a12] = "\xa7\xd2", [0x0a13] = "\xbb\xda",
  [0x0a14] = "\xa7\xb2", [0x0a15] = "\xc4\x7e", [0x0a16] = "\xac\xf6",
  [0x0a17] = "\xb9\xc5", [0x0a18] = "\xcf\x45", [0x0a19] = "\xa7\xa8",
  [0x0a1a] = "\xa8\xce", [0x0a1b] = "\xae\x61", [0x0a1c] = "\xa5\x5b",
  [0x0a1d] = "\xb2\xf3", [0x0a1e] = "\xc0\x55", [0x0a1f] = "\xb8\xeb",
  [0x0a20] = "\xa5\xd2", [0x0a21] = "\xb9\x5b", [0x0a22] = "\xb0\xb2",
  [0x0a23] = "\xbd\x5b", [0x0a24] = "\xbb\xf9", [0x0a25] = "\xac\x5b",
  [0x0a26] = "\xbe\x72", [0x0a27] = "\xb6\xf9", [0x0a28] = "\xc4\xe8",
  [0x0a29] = "\xba\xca", [0x0a2a] = "\xb0\xed", [0x0a2b] = "\xa6\x79",
  [0x0a2c] = "\xba\xe0", [0x0a2d] = "\xb6\xa1", [0x0a2e] = "\xb7\xce",
  [0x0a2f] = "\xad\xdd", [0x0a30] = "\xaa\xd3", [0x0a31] = "\xc1\x7d",
  [0x0a32] = "\xa6\x6c", [0x0a33] = "\xbd\x70", [0x0a34] = "\xc3\xb5",
  [0x0a35] = "\xc0\xcb", [0x0a36] = "\xac\x5a", [0x0a37] = "\xd8\xac",
  [0x0a38] = "\xc6\x50", [0x0a39] = "\xb4\x7a", [0x0a3a] = "\xbe\xdf",
  [0x0a3b] = "\xc2\xb2", [0x0a3c] = "\xbb\xfc", [0x0a3d] = "\xb0\xc5",
  [0x0a3e] = "\xb4\xee", [0x0a3f] = "\xc2\xcb", [0x0a40] = "\xc2\x65",
  [0x0a41] = "\xc5\xb2", [0x0a42] = "\xbd\xee", [0x0a43] = "\xbd\xe2",
  [0x0a44] = "\xa8\xa3", [0x0a45] = "\xc1\xe4", [0x0a46] = "\xbd\x62",
  [0x0a47] = "\xa5\xf3", [0x0a48] = "\xb0\xb7", [0x0a49] = "\xc4\xa5",
  [0x0a4a] = "\xbc\x43", [0x0a4b] = "\xc0\x5e", [0x0a4c] = "\xba\xa5",
  [0x0a4d] = "\xc2\x71", [0x0a4e] = "\xbc\xee", [0x0a4f] = "\xab\xd8",
  [0x0a50] = "\xbb\xf8", [0x0a51] = "\xab\xb8", [0x0a52] = "\xb1\x4e",
  [0x0a53] = "\xbc\xdf", [0x0a54] = "\xa6\xbf", [0x0a55] = "\xc3\xa6",
  [0x0a56] = "\xbd\xb1", [0x0a57] = "\xbc\xd5", [0x0a58] = "\xbc\xfa",
  [0x0a59] = "\xc1\xbf", [0x0a5a] = "\xa6\x4b", [0x0a5b] = "\xc2\xe6",
  [0x0a5c] = "\xad\xb0", [0x0a5d] = "\xbf\xbc", [0x0a5e] = "\xb4\xd4",
  [0x0a5f] = "\xc1\x47", [0x0a60] = "\xb5\x4a", [0x0a61] = "\xbd\xa6",
  [0x0a62] = "\xa5\xe6", [0x0a63] = "\xad\xa5", [0x0a64] = "\xbc\xe5",
  [0x0a65] = "\xc5\xba", [0x0a66] = "\xbc\x62", [0x0a67] = "\xc4\x5a",
  [0x0a68] = "\xc5\xcd", [0x0a69] = "\xbb\xc7", [0x0a6a] = "\xc1\x42",
  [0x0a6b] = "\xb9\xae", [0x0a6c] = "\xb8\x7d", [0x0a6d] = "\xac\xbe",
  [0x0a6e] = "\xa8\xa4", [0x0a6f] = "\xbb\xe5", [0x0a70] = "\xc3\xba",
  [0x0a71] = "\xb5\xb1", [0x0a72] = "\xb6\xd3", [0x0a73] = "\xb1\xd0",
  [0x0a74] = "\xbb\xc3", [0x0a75] = "\xc3\xe2", [0x0a76] = "\xb8\xfb",
  [0x0a77] = "\xa5\x73", [0x0a78] = "\xb5\xa2", [0x0a79] = "\xb4\xa6",
  [0x0a7a] = "\xb1\xb5", [0x0a7b] = "\xac\xd2", [0x0a7c] = "\xd6\xc5",
  [0x0a7d] = "\xb5\xf3", [0x0a7e] = "\xb6\xa5", [0x0a7f] = "\xba\x49",
  [0x0a80] = "\xa7\x54", [0x0a81] = "\xb8\x60", [0x0a82] = "\xae\xdc",
  [0x0a83] = "\xb3\xc7", [0x0a84] = "\xb1\xb6", [0x0a85] = "\xb7\xfb",
  [0x0a86] = "\xba\xdc", [0x0a87] = "\xbc\xe4", [0x0a88] = "\xb5\xb2",
  [0x0a89] = "\xb8\xd1", [0x0a8a] = "\xa9\x6a", [0x0a8b] = "\xa7\xd9",
  [0x0a8c] = "\xc2\xc7", [0x0a8d] = "\xaa\xe3", [0x0a8e] = "\xac\xc9",
  [0x0a8f] = "\xad\xc9", [0x0a90] = "\xa4\xb6", [0x0a91] = "\xac\xce",
  [0x0a92] = "\xbb\x7c", [0x0a93] = "\xa9\xa1", [0x0a94] = "\xa4\x79",
  [0x0a95] = "\xb5\xac", [0x0a96] = "\xa4\xe7", [0x0a97] = "\xaa\xf7",
  [0x0a98] = "\xa4\xb5", [0x0a99] = "\xac\x7a", [0x0a9a] = "\xc3\xcc",
  [0x0a9b] = "\xba\xf2", [0x0a9c] = "\xc0\x41", [0x0a9d] = "\xb6\xc8",
  [0x0a9e] = "\xc2\xd4", [0x0a9f] = "\xb6\x69", [0x0aa0] = "\xe0\xda",
  [0x0aa1] = "\xae\xca", [0x0aa2] = "\xb8\x54", [0x0aa3] = "\xaa\xf1",
  [0x0aa4] = "\xc2\x75", [0x0aa5] = "\xae\xfb", [0x0aa6] = "\xba\xc9",
  [0x0aa7] = "\xab\x6c", [0x0aa8] = "\xaf\xf0", [0x0aa9] = "\xb9\xb8",
  [0x0aaa] = "\xb2\xf4", [0x0aab] = "\xb7\xfa", [0x0aac] = "\xb4\xb9",
  [0x0aad] = "\xc4\x48", [0x0aae] = "\xa8\xca", [0x0aaf] = "\xc5\xe5",
  [0x0ab0] = "\xba\xeb", [0x0ab1] = "\xb8\x65", [0x0ab2] = "\xb8\x67",
  [0x0ab3] = "\xa4\xab", [0x0ab4] = "\xc4\xb5", [0x0ab5] = "\xb4\xba",
  [0x0ab6] = "\xc0\x56", [0x0ab7] = "\xc0\x52", [0x0ab8] = "\xb9\xd2",
  [0x0ab9] = "\xb7\x71", [0x0aba] = "\xc3\xe8", [0x0abb] = "\xae\x7c",
  [0x0abc] = "\xb5\x6a", [0x0abd] = "\xb9\x74", [0x0abe] = "\xb3\xba",
  [0x0abf] = "\xc4\x76", [0x0ac0] = "\xb2\x62", [0x0ac1] = "\xac\xb3",
  [0x0ac2] = "\xb5\x7e", [0x0ac3] = "\xb4\xaa", [0x0ac4] = "\xa8\x73",
  [0x0ac5] = "\xaa\xc8", [0x0ac6] = "\xa8\x68", [0x0ac7] = "\xad\xb4",
  [0x0ac8] = "\xa4\x5b", [0x0ac9] = "\xa8\x62", [0x0aca] = "\xa4\x45",
  [0x0acb] = "\xb0\x73", [0x0acc] = "\xb4\x5c", [0x0acd] = "\xb1\xcf",
  [0x0ace] = "\xc2\xc2", [0x0acf] = "\xa6\xdd", [0x0ad0] = "\xb8\xa4",
  [0x0ad1] = "\xa9\x53", [0x0ad2] = "\xb4\x4e", [0x0ad3] = "\xaa\xb9",
  [0x0ad4] = "\xc1\xf9", [0x0ad5] = "\xa9\xeb", [0x0ad6] = "\xaa\xae",
  [0x0ad7] = "\xaf\x6a", [0x0ad8] = "\xa9\x7e", [0x0ad9] = "\xbe\x73",
  [0x0ada] = "\xb5\xe2", [0x0adb] = "\xa7\xbd", [0x0adc] = "\xa9\x43",
  [0x0add] = "\xaf\x78", [0x0ade] = "\xc1\x7c", [0x0adf] = "\xaa\x71",
  [0x0ae0] = "\xbb\x45", [0x0ae1] = "\xa9\xda", [0x0ae2] = "\xbe\xda",
  [0x0ae3] = "\xa5\xa8", [0x0ae4] = "\xa8\xe3", [0x0ae5] = "\xb6\x5a",
  [0x0ae6] = "\xbd\xf5", [0x0ae7] = "\xbf\xf7", [0x0ae8] = "\xad\xd1",
  [0x0ae9] = "\xa5\x79", [0x0aea] = "\xc4\xdf", [0x0aeb] = "\xac\xb2",
  [0x0aec] = "\xbc\x40", [0x0aed] = "\xae\xbd", [0x0aee] = "\xc3\x59",
  [0x0aef] = "\xae\x53", [0x0af0] = "\xad\xc2", [0x0af1] = "\xb2\xb2",
  [0x0af2] = "\xa8\xf7", [0x0af3] = "\xb8\x68", [0x0af4] = "\xe5\xb1",
  [0x0af5] = "\xc5\xcc", [0x0af6] = "\xa7\xe0", [0x0af7] = "\xb1\xb8",
  [0x0af8] = "\xad\xcf", [0x0af9] = "\xc0\xef", [0x0afa] = "\xc4\xb1",
  [0x0afb] = "\xa8\x4d", [0x0afc] = "\xb3\x5a", [0x0afd] = "\xb5\xb4",
  [0x0afe] = "\xa7\xa1", [0x0aff] = "\xb5\xdf", [0x0b00] = "\xb6\x76",
  [0x0b01] = "\xad\x78", [0x0b02] = "\xa7\x67", [0x0b03] = "\xae\x6d",
  [0x0b04] = "\xab\x54", [0x0b05] = "\xb5\xa4", [0x0b06] = "\xaf\x43",
  [0x0b07] = "\xb0\x70", [0x0b08] = "\xc2\x40", [0x0b09] = "\xb3\xd8",
  [0x0b0a] = "\xa9\x40", [0x0b0b] = "\xa5\x64", [0x0b0c] = "\xab\xa3",
  [0x0b0d] = "\xb6\x7d", [0x0b0e] = "\xb4\x7b", [0x0b0f] = "\xb7\xa2",
  [0x0b10] = "\xb3\xcd", [0x0b11] = "\xb4\x6e", [0x0b12] = "\xa5\x5a",
  [0x0b13] = "\xb3\xf4", [0x0b14] = "\xb0\xc9", [0x0b15] = "\xa7\xa2",
  [0x0b16] = "\xac\xe5", [0x0b17] = "\xac\xdd", [0x0b18] = "\xb1\x64",
  [0x0b19] = "\xba\x42", [0x0b1a] = "\xc1\x52", [0x0b1b] = "\xa6\xaa",
  [0x0b1c] = "\xa7\xdc", [0x0b1d] = "\xa4\xae", [0x0b1e] = "\xaa\xa1",
  [0x0b1f] = "\xa6\xd2", [0x0b20] = "\xab\xfe", [0x0b21] = "\xaf\x4e",
  [0x0b22] = "\xbe\x61", [0x0b23] = "\xa9\x56", [0x0b24] = "\xad\x56",
  [0x0b25] = "\xac\x5f", [0x0b26] = "\xb4\xca", [0x0b27] = "\xbd\x57",
  [0x0b28] = "\xc1\xfb", [0x0b29] = "\xac\xec", [0x0b2a] = "\xb4\xdf",
  [0x0b2b] = "\xab\x79", [0x0b2c] = "\xa5\x69", [0x0b2d] = "\xb4\xf7",
  [0x0b2e] = "\xa7\x4a", [0x0b2f] = "\xa8\xe8", [0x0b30] = "\xab\xc8",
  [0x0b31] = "\xbd\xd2", [0x0b32] = "\xaa\xd6", [0x0b33] = "\xb0\xd9",
  [0x0b34] = "\xbe\xc1", [0x0b35] = "\xc0\xb5", [0x0b36] = "\xa7\x7c",
  [0x0b37] = "\xa7\x5c", [0x0b38] = "\xaa\xc5", [0x0b39] = "\xae\xa3",
  [0x0b3a] = "\xa4\xd5", [0x0b3b] = "\xb1\xb1", [0x0b3c] = "\xe1\xe0",
  [0x0b3d] = "\xa4\x66", [0x0b3e] = "\xa6\xa9", [0x0b3f] = "\xb1\x46",
  [0x0b40] = "\xac\x5c", [0x0b41] = "\xad\xfa", [0x0b42] = "\xb8\x5d",
  [0x0b43] = "\xad\x57", [0x0b44] = "\xbb\xc5", [0x0b45] = "\xae\x77",
  [0x0b46] = "\xbf\xc7", [0x0b47] = "\xb8\xd8", [0x0b48] = "\xab\xb1",
  [0x0b49] = "\xce\xd8", [0x0b4a] = "\xb8\xf3", [0x0b4b] = "\xaf\xe3",
  [0x0b4c] = "\xb6\xf4", [0x0b4d] = "\xb8\x5f", [0x0b4e] = "\xbb\xfb",
  [0x0b4f] = "\xa7\xd6", [0x0b50] = "\xbc\x65", [0x0b51] = "\xb4\xda",
  [0x0b52] = "\xa6\x4a", [0x0b53] = "\xb5\xa8", [0x0b54] = "\xa8\x67",
  [0x0b55] = "\xae\xd8", [0x0b56] = "\xc4\x71", [0x0b57] = "\xb2\xb5",
  [0x0b58] = "\xc3\x6d", [0x0b59] = "\xaa\x70", [0x0b5a] = "\xc1\xab",
  [0x0b5b] = "\xb2\xaf", [0x0b5c] = "\xf5\xc9", [0x0b5d] = "\xbf\x73",
  [0x0b5e] = "\xb8\xaa", [0x0b5f] = "\xab\xb6", [0x0b60] = "\xbb\xed",
  [0x0b61] = "\xb3\xc8", [0x0b62] = "\xf5\x58", [0x0b63] = "\xb7\x5c",
  [0x0b64] = "\xbc\xec", [0x0b65] = "\xa9\x5b", [0x0b66] = "\xa9\xf8",
  [0x0b67] = "\xae\xb9", [0x0b68] = "\xa7\x78", [0x0b69] = "\xac\x41",
  [0x0b6a] = "\xc2\x58", [0x0b6b] = "\xb9\xf8", [0x0b6c] = "\xc1\xef",
  [0x0b6d] = "\xa9\x55", [0x0b6e] = "\xa9\xd4", [0x0b6f] = "\xb3\xe2",
  [0x0b70] = "\xc4\xfa", [0x0b71] = "\xc3\xbe", [0x0b72] = "\xbb\xb6",
  [0x0b73] = "\xb0\xd5", [0x0b74] = "\xb5\xdc", [0x0b75] = "\xa8\xd3",
  [0x0b76] = "\xbf\xe0", [0x0b77] = "\xc2\xc5", [0x0b78] = "\xb0\xfd",
  [0x0b79] = "\xc4\xe6", [0x0b7a] = "\xc4\x64", [0x0b7b] = "\xc4\x78",
  [0x0b7c] = "\xc1\xf1", [0x0b7d] = "\xc4\xf5", [0x0b7e] = "\xc4\x69",
  [0x0b7f] = "\xf8\xc2", [0x0b80] = "\xc5\xf3", [0x0b81] = "\xc4\xfd",
  [0x0b82] = "\xc3\x69", [0x0b83] = "\xc6\x6c", [0x0b84] = "\xc4\xea",
  [0x0b85] = "\xc0\xdd", [0x0b86] = "\xb2\x77", [0x0b87] = "\xb7\x7d",
  [0x0b88] = "\xaf\x54", [0x0b89] = "\xb4\x59", [0x0b8a] = "\xad\xa6",
  [0x0b8b] = "\xae\xd4", [0x0b8c] = "\xae\xf6", [0x0b8d] = "\xbc\xb4",
  [0x0b8e] = "\xb3\xd2", [0x0b8f] = "\xa8\x63", [0x0b90] = "\xa6\xd1",
  [0x0b91] = "\xa8\xd0", [0x0b92] = "\xab\xbe", [0x0b93] = "\xb9\x54",
  [0x0b94] = "\xaf\x4f", [0x0b95] = "\xe5\xfc", [0x0b96] = "\xb0\xc7",
  [0x0b97] = "\xbc\xd6", [0x0b98] = "\xb9\x70", [0x0b99] = "\xc5\x4a",
  [0x0b9a] = "\xc1\xa2", [0x0b9b] = "\xbd\x55", [0x0b9c] = "\xb2\xd6",
  [0x0b9d] = "\xc0\x77", [0x0b9e] = "\xc2\x53", [0x0b9f] = "\xbe\xdd",
  [0x0ba0] = "\xa6\xd8", [0x0ba1] = "\xc3\xfe", [0x0ba2] = "\xb2\x5c",
  [0x0ba3] = "\xb8\x57", [0x0ba4] = "\xb7\xab", [0x0ba5] = "\xa7\x4e",
  [0x0ba6] = "\xc2\xe7", [0x0ba7] = "\xb1\xf9", [0x0ba8] = "\xb2\x70",
  [0x0ba9] = "\xbe\xa4", [0x0baa] = "\xc6\x58", [0x0bab] = "\xaf\x57",
  [0x0bac] = "\xc2\xf7", [0x0bad] = "\xba\x76", [0x0bae] = "\xb2\x7a",
  [0x0baf] = "\xa7\xf5", [0x0bb0] = "\xb8\xcc", [0x0bb1] = "\xc3\x55",
  [0x0bb2] = "\xc2\xa7", [0x0bb3] = "\xb2\xfa", [0x0bb4] = "\xaf\xef",
  [0x0bb5] = "\xa6\x4f", [0x0bb6] = "\xae\xdf", [0x0bb7] = "\xc4\x52",
  [0x0bb8] = "\xbc\x46", [0x0bb9] = "\xc0\x79", [0x0bba] = "\xc4\x74",
  [0x0bbb] = "\xbe\xfa", [0x0bbc] = "\xa7\x51", [0x0bbe] = "\xa8\xd2",
  [0x0bbf] = "\xab\x57", [0x0bc0] = "\xb5\x67", [0x0bc1] = "\xa5\xdf",
  [0x0bc2] = "\xb2\xc9", [0x0bc3] = "\xc3\x77", [0x0bc4] = "\xc1\xf5",
  [0x0bc5] = "\xa4\x4f", [0x0bc6] = "\xbc\xfe", [0x0bc7] = "\xad\xf9",
  [0x0bc8] = "\xad\xc7", [0x0bc9] = "\xc1\x70", [0x0bca] = "\xbd\xac",
  [0x0bcb] = "\xb3\x73", [0x0bcc] = "\xc5\x49", [0x0bcd] = "\xb7\x47",
  [0x0bce] = "\xbc\xa6", [0x0bcf] = "\xba\xa7", [0x0bd0] = "\xc3\xae",
  [0x0bd1] = "\xc0\xc4", [0x0bd2] = "\xc1\x79", [0x0bd3] = "\xc3\xec",
  [0x0bd4] = "\xc5\xca", [0x0bd5] = "\xc1\xe5", [0x0bd6] = "\xbd\x6d",
  [0x0bd7] = "\xc2\xb3", [0x0bd8] = "\xb2\x44", [0x0bd9] = "\xb1\xe7",
  [0x0bda] = "\xb8\x64", [0x0bdb] = "\xa8\x7d", [0x0bdc] = "\xa8\xe2",
  [0x0bdd] = "\xbd\xf8", [0x0bde] = "\xb6\x71", [0x0bdf] = "\xb4\xbd",
  [0x0be0] = "\xab\x47", [0x0be1] = "\xbd\xcc", [0x0be2] = "\xbc\xba",
  [0x0be3] = "\xb2\xe1", [0x0be4] = "\xb9\xb1", [0x0be5] = "\xc0\xf8",
  [0x0be6] = "\xbf\x52", [0x0be7] = "\xb9\xe9", [0x0be8] = "\xbf\xf1",
  [0x0be9] = "\xbc\xe3", [0x0bea] = "\xa4\x46", [0x0beb] = "\xe1\xe6",
  [0x0bec] = "\xf4\xef", [0x0bed] = "\xb9\xf9", [0x0bee] = "\xae\xc6",
  [0x0bef] = "\xa6\x43", [0x0bf0] = "\xb5\xf5", [0x0bf1] = "\xaf\x50",
  [0x0bf2] = "\xa6\x48", [0x0bf3] = "\xc2\x79", [0x0bf4] = "\xb5\x59",
  [0x0bf5] = "\xaa\x4c", [0x0bf6] = "\xc1\x43", [0x0bf7] = "\xc0\x4d",
  [0x0bf8] = "\xc1\x7b", [0x0bf9] = "\xbe\x46", [0x0bfa] = "\xc5\xec",
  [0x0bfb] = "\xb2\x4f", [0x0bfc] = "\xbb\xfe", [0x0bfd] = "\xb8\xee",
  [0x0bfe] = "\xa7\x5b", [0x0bff] = "\xa9\xf0", [0x0c00] = "\xac\xc2",
  [0x0c01] = "\xb5\xd9", [0x0c02] = "\xb9\x73", [0x0c03] = "\xc4\xd6",
  [0x0c04] = "\xb9\x61", [0x0c05] = "\xa7\x44", [0x0c06] = "\xb2\xdc",
  [0x0c07] = "\xad\xe2", [0x0c08] = "\xc6\x46", [0x0c09] = "\xb3\xae",
  [0x0c0a] = "\xc0\xad", [0x0c0b] = "\xbb\xe2", [0x0c0c] = "\xa5\x74",
  [0x0c0d] = "\xa5\x4f", [0x0c0e] = "\xb7\xc8", [0x0c0f] = "\xaf\x5b",
  [0x0c10] = "\xba\x68", [0x0c11] = "\xb2\xb8", [0x0c12] = "\xc3\x48",
  [0x0c13] = "\xaf\x64", [0x0c14] = "\xbc\x42", [0x0c15] = "\xbd\x46",
  [0x0c16] = "\xac\x79", [0x0c17] = "\xac\x68", [0x0c18] = "\xa4\xbb",
  [0x0c19] = "\xc0\x73", [0x0c1a] = "\xc5\xa4", [0x0c1b] = "\xc4\x56",
  [0x0c1c] = "\xc5\xa2", [0x0c1d] = "\xc1\x4b", [0x0c1e] = "\xb6\xa9",
  [0x0c1f] = "\xc3\x62", [0x0c20] = "\xc3\x6c", [0x0c21] = "\xc3\xf7",
  [0x0c22] = "\xbc\xd3", [0x0c23] = "\xb0\xfa", [0x0c24] = "\xba\x4f",
  [0x0c25] = "\xc1\x4d", [0x0c26] = "\xba\x7c", [0x0c27] = "\xad\xae",
  [0x0c28] = "\xc4\xaa", [0x0c29] = "\xbf\x63", [0x0c2a] = "\xc6\x60",
  [0x0c2b] = "\xc3\x66", [0x0c2c] = "\xc4\x6c", [0x0c2d] = "\xbe\xdb",
  [0x0c2e] = "\xba\xb1", [0x0c2f] = "\xb8\xb8", [0x0c30] = "\xbe\x7c",
  [0x0c31] = "\xc4\x53", [0x0c32] = "\xb8\x4c", [0x0c33] = "\xc5\x53",
  [0x0c34] = "\xb8\xf4", [0x0c35] = "\xb8\xef", [0x0c36] = "\xb3\xc0",
  [0x0c37] = "\xe9\xf0", [0x0c38] = "\xb8\x53", [0x0c39] = "\xbf\xfd",
  [0x0c3a] = "\xb3\xb0", [0x0c3b] = "\xbc\xae", [0x0c3c] = "\xc6\x6a",
  [0x0c3d] = "\xa7\x66", [0x0c3e] = "\xbe\x54", [0x0c3f] = "\xab\x51",
  [0x0c40] = "\xae\xc8", [0x0c41] = "\xbc\x69", [0x0c42] = "\xb9\xf0",
  [0x0c43] = "\xc1\x5c", [0x0c44] = "\xbc\x7b", [0x0c45] = "\xb4\xe2",
  [0x0c46] = "\xab\xdf", [0x0c47] = "\xb2\x76", [0x0c48] = "\xc2\x6f",
  [0x0c49] = "\xba\xf1", [0x0c4a] = "\xc5\x72", [0x0c4b] = "\xc5\xcb",
  [0x0c4c] = "\xc5\x70", [0x0c4d] = "\xc6\x65", [0x0c4e] = "\xa7\x5a",
  [0x0c4f] = "\xb6\xc3", [0x0c50] = "\xb1\xb0", [0x0c51] = "\xb2\xa4",
  [0x0c52] = "\xb1\xc1", [0x0c53] = "\xbd\xfc", [0x0c54] = "\xad\xdb",
  [0x0c55] = "\xa8\xda", [0x0c56] = "\xb2\x5f", [0x0c57] = "\xba\xfa",
  [0x0c58] = "\xbd\xd7", [0x0c59] = "\xc5\xda", [0x0c5a] = "\xc1\xb3",
  [0x0c5b] = "\xc3\xb9", [0x0c5c] = "\xc5\xde", [0x0c5d] = "\xc6\x72",
  [0x0c5e] = "\xc6\x59", [0x0c5f] = "\xc5\x5b", [0x0c60] = "\xbb\x72",
  [0x0c61] = "\xb8\xa8", [0x0c62] = "\xac\xa5", [0x0c63] = "\xc0\x64",
  [0x0c64] = "\xb5\xb8", [0x0c65] = "\xb6\xfd", [0x0c66] = "\xb3\xc2",
  [0x0c67] = "\xba\xbf", [0x0c68] = "\xbd\x58", [0x0c69] = "\xbf\xc2",
  [0x0c6a] = "\xb0\xa8", [0x0c6b] = "\xbd\x7c", [0x0c6c] = "\xb9\xc0",
  [0x0c6d] = "\xb6\xdc", [0x0c6e] = "\xae\x49", [0x0c6f] = "\xb6\x52",
  [0x0c70] = "\xb3\xc1", [0x0c71] = "\xbd\xe6", [0x0c72] = "\xc1\xda",
  [0x0c73] = "\xaf\xdf", [0x0c74] = "\xbf\x66", [0x0c75] = "\xc4\x43",
  [0x0c76] = "\xc6\x5a", [0x0c77] = "\xba\xa1", [0x0c78] = "\xbd\xaf",
  [0x0c79] = "\xb0\xd2", [0x0c7a] = "\xba\x43", [0x0c7b] = "\xba\xa9",
  [0x0c7c] = "\xf0\xe3", [0x0c7d] = "\xa8\x7e", [0x0c7e] = "\xaf\xed",
  [0x0c7f] = "\xaa\xbc", [0x0c80] = "\xaa\x5d", [0x0c81] = "\xa6\xa3",
  [0x0c82] = "\xb2\xf5", [0x0c83] = "\xbf\xdf", [0x0c84] = "\xad\x54",
  [0x0c85] = "\xc1\xe3", [0x0c86] = "\xa4\xf2", [0x0c87] = "\xa5\xd9",
  [0x0c88] = "\xb9\x67", [0x0c89] = "\xa5\x66", [0x0c8a] = "\xad\x5a",
  [0x0c8b] = "\xab\x5f", [0x0c8c] = "\xb4\x55", [0x0c8d] = "\xbb\xaa",
  [0x0c8e] = "\xb6\x54", [0x0c8f] = "\xbb\xf2", [0x0c90] = "\xaa\xb4",
  [0x0c91] = "\xaa\x54", [0x0c92] = "\xb1\xf6", [0x0c94] = "\xbe\x60",
  [0x0c95] = "\xb7\xd1", [0x0c96] = "\xa8\x53", [0x0c97] = "\xac\xdc",
  [0x0c98] = "\xb4\x43", [0x0c99] = "\xc1\xe2", [0x0c9a] = "\xa8\x43",
  [0x0c9b] = "\xac\xfc", [0x0c9c] = "\xac\x4e", [0x0c9d] = "\xb4\x4b",
  [0x0c9e] = "\xa9\x66", [0x0c9f] = "\xb4\x41", [0x0ca0] = "\xaa\xf9",
  [0x0ca1] = "\xb4\x65", [0x0ca2] = "\xad\xcc", [0x0ca3] = "\xb5\xde",
  [0x0ca4] = "\xbb\x58", [0x0ca5] = "\xc2\x63", [0x0ca6] = "\xb7\xf9",
  [0x0ca7] = "\xbf\xf8", [0x0ca8] = "\xb2\x72", [0x0ca9] = "\xb9\xda",
  [0x0caa] = "\xa9\x73", [0x0cab] = "\xbd\x4e", [0x0cac] = "\xee\xc5",
  [0x0cad] = "\xc3\xfb", [0x0cae] = "\xc1\x53", [0x0caf] = "\xb0\x67",
  [0x0cb0] = "\xc1\xbc", [0x0cb1] = "\xc0\xb1", [0x0cb2] = "\xa6\xcc",
  [0x0cb3] = "\xaf\xb5", [0x0cb4] = "\xb3\x56", [0x0cb5] = "\xaa\x63",
  [0x0cb6] = "\xbb\x65", [0x0cb7] = "\xb1\x4b", [0x0cb8] = "\xbe\xad",
  [0x0cb9] = "\xb4\xd6", [0x0cba] = "\xaf\x76", [0x0cbb] = "\xba\xf8",
  [0x0cbc] = "\xb0\xc3", [0x0cbd] = "\xa7\x4b", [0x0cbe] = "\xab\x6a",
  [0x0cbf] = "\xae\x59", [0x0cc0] = "\xbd\x71", [0x0cc1] = "\xad\xb1",
  [0x0cc2] = "\xad\x5d", [0x0cc3] = "\xb4\x79", [0x0cc4] = "\xba\xcb",
  [0x0cc5] = "\xc2\xc6", [0x0cc6] = "\xac\xed", [0x0cc7] = "\xb4\xf9",
  [0x0cc8] = "\xbc\x71", [0x0cc9] = "\xa7\xae", [0x0cca] = "\xbd\xb0",
  [0x0ccb] = "\xb7\xc0", [0x0ccc] = "\xa5\xc1", [0x0ccd] = "\xa9\xd7",
  [0x0cce] = "\xa5\xd7", [0x0ccf] = "\xb1\xd3", [0x0cd0] = "\xbc\xa7",
  [0x0cd1] = "\xbb\xd4", [0x0cd2] = "\xa9\xfa", [0x0cd3] = "\xbf\xc1",
  [0x0cd4] = "\xbb\xef", [0x0cd5] = "\xbb\xca", [0x0cd6] = "\xa6\x57",
  [0x0cd7] = "\xa9\x52", [0x0cd8] = "\xc2\xd5", [0x0cd9] = "\xba\x4e",
  [0x0cda] = "\xbc\xb1", [0x0cdb] = "\xc4\xa8", [0x0cdc] = "\xbc\xd2",
  [0x0cdd] = "\xbd\xa4", [0x0cde] = "\xbf\x69", [0x0cdf] = "\xbc\xaf",
  [0x0ce0] = "\xc5\x5d", [0x0ce1] = "\xa9\xd9", [0x0ce2] = "\xa5\xbd",
  [0x0ce3] = "\xb2\xf6", [0x0ce4] = "\xbe\xa5", [0x0ce5] = "\xc0\x71",
  [0x0ce6] = "\xaa\x6a", [0x0ce7] = "\xba\x7a", [0x0ce8] = "\xb9\xe6",
  [0x0ce9] = "\xad\xaf", [0x0cea] = "\xbf\xd1", [0x0ceb] = "\xa6\xc8",
  [0x0cec] = "\xac\x59", [0x0ced] = "\xa9\xe6", [0x0cee] = "\xa8\x64",
  [0x0cef] = "\xaf\x61", [0x0cf0] = "\xa9\x69", [0x0cf1] = "\xa5\xc0",
  [0x0cf2] = "\xb9\xd3", [0x0cf3] = "\xbc\xc7", [0x0cf4] = "\xb9\xf5",
  [0x0cf5] = "\xb6\xd2", [0x0cf6] = "\xbc\x7d", [0x0cf7] = "\xa4\xec",
  [0x0cf8] = "\xa5\xd8", [0x0cf9] = "\xb7\xfc", [0x0cfa] = "\xaa\xaa",
  [0x0cfb] = "\xbf\x70", [0x0cfc] = "\xae\xb3", [0x0cfd] = "\xad\xfe",
  [0x0cfe] = "\xa7\x6f", [0x0cff] = "\xb6\x75", [0x0d00] = "\xa8\xba",
  [0x0d01] = "\xae\x52", [0x0d02] = "\xaf\xc7", [0x0d03] = "\xa6\xbb",
  [0x0d04] = "\xa4\x44", [0x0d05] = "\xa5\xa4", [0x0d06] = "\xad\x40",
  [0x0d07] = "\xa9\x60", [0x0d08] = "\xab\x6e", [0x0d09] = "\xa8\x6b",
  [0x0d0a] = "\xc3\xf8", [0x0d0b] = "\xc5\x6e", [0x0d0c] = "\xbc\xb8",
  [0x0d0d] = "\xb8\xa3", [0x0d0e] = "\xb4\x6f", [0x0d0f] = "\xbe\x78",
  [0x0d10] = "\xd5\xef", [0x0d11] = "\xa9\x4f", [0x0d12] = "\xbe\x6b",
  [0x0d13] = "\xa4\xba", [0x0d14] = "\xb9\xe0", [0x0d15] = "\xaf\xe0",
  [0x0d16] = "\xa9\x67", [0x0d17] = "\xc0\x4f", [0x0d18] = "\xad\xd9",
  [0x0d19] = "\xaa\x64", [0x0d1a] = "\xa5\xa7", [0x0d1b] = "\xc0\xc0",
  [0x0d1c] = "\xa7\x41", [0x0d1d] = "\xb0\xce", [0x0d1e] = "\xbf\xb0",
  [0x0d1f] = "\xb0\x66", [0x0d20] = "\xb7\xc4", [0x0d21] = "\xe7\x59",
  [0x0d22] = "\xa9\xe0", [0x0d23] = "\xa6\x7e", [0x0d24] = "\xbd\x56",
  [0x0d25] = "\xc2\x5b", [0x0d26] = "\xb1\xc9", [0x0d27] = "\xa9\xc0",
  [0x0d28] = "\xae\x51", [0x0d29] = "\xc6\x43", [0x0d2a] = "\xb3\xbe",
  [0x0d2b] = "\xa7\xbf", [0x0d2c] = "\xae\xba", [0x0d2d] = "\xc2\xbf",
  [0x0d2e] = "\xc4\x5e", [0x0d2f] = "\xf8\xa5", [0x0d30] = "\xc6\x68",
  [0x0d31] = "\xc2\xec", [0x0d32] = "\xaf\x49", [0x0d33] = "\xb1\x7a",
  [0x0d34] = "\xc2\x66", [0x0d35] = "\xc0\xf1", [0x0d36] = "\xbe\xae",
  [0x0d37] = "\xb9\xe7", [0x0d38] = "\xc0\xbe", [0x0d39] = "\xc0\xd7",
  [0x0d3a] = "\xa4\xfb", [0x0d3b] = "\xa7\xe1", [0x0d3c] = "\xb6\x73",
  [0x0d3d] = "\xaf\xc3", [0x0d3e] = "\xc1\x77", [0x0d3f] = "\xbf\x40",
  [0x0d40] = "\xb9\x41", [0x0d41] = "\xa7\xcb", [0x0d42] = "\xa5\xa3",
  [0x0d43] = "\xa7\x56", [0x0d44] = "\xab\xe3", [0x0d45] = "\xa4\x6b",
  [0x0d46] = "\xb7\x78", [0x0d47] = "\xad\x68", [0x0d48] = "\xba\xc4",
  [0x0d49] = "\xae\xbf", [0x0d4a] = "\xc0\xb6", [0x0d4b] = "\xc4\x7a",
  [0x0d4c] = "\xbf\xd5", [0x0d4d] = "\xae\x40", [0x0d4e] = "\xbc\xda",
  [0x0d4f] = "\xc5\xc3", [0x0d50] = "\xbc\xde", [0x0d51] = "\xc3\xc2",
  [0x0d52] = "\xb9\xc3", [0x0d53] = "\xb0\xb8", [0x0d54] = "\xe2\x78",
  [0x0d55] = "\xb0\xd4", [0x0d56] = "\xad\x77", [0x0d57] = "\xaa\xa6",
  [0x0d58] = "\xa9\xac", [0x0d59] = "\xa9\xc8", [0x0d5a] = "\xb5\x5d",
  [0x0d5b] = "\xa9\xe7", [0x0d5c] = "\xb1\xc6", [0x0d5d] = "\xb5\x50",
  [0x0d5e] = "\xb1\x72", [0x0d5f] = "\xb4\xfb", [0x0d60] = "\xac\xa3",
  [0x0d61] = "\xc3\x6b", [0x0d62] = "\xbc\xef", [0x0d63] = "\xbd\x4c",
  [0x0d64] = "\xbd\x59", [0x0d65] = "\xac\xdf", [0x0d66] = "\xaf\x60",
  [0x0d67] = "\xa7\x50", [0x0d68] = "\xab\x71", [0x0d69] = "\xa5\xe3",
  [0x0d6a] = "\xc3\x65", [0x0d6b] = "\xae\xc7", [0x0d6c] = "\xea\xe5",
  [0x0d6d] = "\xad\x44", [0x0d6e] = "\xa9\xdf", [0x0d6f] = "\xa9\x48",
  [0x0d70] = "\xa7\x53", [0x0d71] = "\xaf\xa5", [0x0d72] = "\xb3\x54",
  [0x0d73] = "\xb6\x5d", [0x0d74] = "\xaa\x77", [0x0d75] = "\xa9\x41",
  [0x0d76] = "\xad\x46", [0x0d77] = "\xb0\xf6", [0x0d78] = "\xbb\x70",
  [0x0d79] = "\xbd\xdf", [0x0d7a] = "\xb3\xad", [0x0d7b] = "\xb0\x74",
  [0x0d7c] = "\xa8\xd8", [0x0d7d] = "\xa8\x4b", [0x0d7e] = "\xbc\x51",
  [0x0d7f] = "\xac\xd6", [0x0d80] = "\xaf\x79", [0x0d81] = "\xa9\xe1",
  [0x0d82] = "\xb2\x69", [0x0d83] = "\xbc\xea", [0x0d84] = "\xb4\x5e",
  [0x0d85] = "\xbd\xb4", [0x0d86] = "\xb4\xd7", [0x0d87] = "\xb8\x4e",
  [0x0d88] = "\xc1\x4f", [0x0d89] = "\xbf\xb1", [0x0d8a] = "\xaa\x42",
  [0x0d8b] = "\xc4\x50", [0x0d8c] = "\xb1\xb7", [0x0d8d] = "\xb8\x49",
  [0x0d8e] = "\xcb\xf2", [0x0d8f] = "\xcf\xec", [0x0d90] = "\xc5\x52",
  [0x0d91] = "\xa7\xe5", [0x0d92] = "\xa9\xdc", [0x0d93] = "\xbc\x41",
  [0x0d94] = "\xb5\x5c", [0x0d95] = "\xac\x73", [0x0d96] = "\xb0\xe0",
  [0x0d97] = "\xb5\xca", [0x0d98] = "\xaf\x68", [0x0d99] = "\xa5\xd6",
  [0x0d9a] = "\xa4\xc7", [0x0d9b] = "\xb5\x6c", [0x0d9c] = "\xbb\xf7",
  [0x0d9d] = "\xa7\xbe", [0x0d9e] = "\xc4\xb4", [0x0d9f] = "\xbd\x67",
  [0x0da0] = "\xb0\xbe", [0x0da1] = "\xa4\xf9", [0x0da2] = "\xc4\x46",
  [0x0da3] = "\xc4\xc6", [0x0da4] = "\xba\x7d", [0x0da5] = "\xbf\x5d",
  [0x0da6] = "\xb2\xbc", [0x0da7] = "\xba\x4a", [0x0da8] = "\xbf\x68",
  [0x0da9] = "\xab\xf7", [0x0daa] = "\xc0\x57", [0x0dab] = "\xb3\x68",
  [0x0dac] = "\xab\x7e", [0x0dad] = "\xb8\x75", [0x0dae] = "\xa5\xe2",
  [0x0daf] = "\xa9\x57", [0x0db0] = "\xc4\xab", [0x0db1] = "\xb5\xd3",
  [0x0db2] = "\xa5\xad", [0x0db3] = "\xbe\xcc", [0x0db4] = "\xb2\x7e",
  [0x0db5] = "\xb5\xfb", [0x0db6] = "\xab\xcc", [0x0db7] = "\xa9\x59",
  [0x0db8] = "\xbc\xe2", [0x0db9] = "\xbb\xe1", [0x0dba] = "\xb1\x43",
  [0x0dbb] = "\xaf\x7d", [0x0dbc] = "\xbe\x7a", [0x0dbd] = "\xad\xa2",
  [0x0dbe] = "\xb2\xcb", [0x0dbf] = "\xad\xe5", [0x0dc0] = "\xbc\xb3",
  [0x0dc1] = "\xbe\x51", [0x0dc2] = "\xb9\xb2", [0x0dc3] = "\xb3\x40",
  [0x0dc4] = "\xb8\xb2", [0x0dc5] = "\xb5\xd0", [0x0dc6] = "\xbb\x5a",
  [0x0dc7] = "\xae\x48", [0x0dc8] = "\xbe\xeb", [0x0dc9] = "\xae\x45",
  [0x0dca] = "\xb4\xb6", [0x0dcb] = "\xae\xfa", [0x0dcc] = "\xc3\xd0",
  [0x0dcd] = "\xc3\x6e", [0x0dce] = "\xc2\x72", [0x0dcf] = "\xb4\xc1",
  [0x0dd0] = "\xb4\xdb", [0x0dd1] = "\xb4\xcf", [0x0dd2] = "\xb1\xad",
  [0x0dd3] = "\xa9\x64", [0x0dd4] = "\xa4\x43", [0x0dd5] = "\xb2\x59",
  [0x0dd6] = "\xba\xa3", [0x0dd7] = "\xac\x6d", [0x0dd8] = "\xcb\x50",
  [0x0dd9] = "\xa8\xe4", [0x0dda] = "\xb4\xd1", [0x0ddb] = "\xa9\x5f",
  [0x0ddc] = "\xaa\x5b", [0x0ddd] = "\xb2\xa5", [0x0dde] = "\xb1\x54",
  [0x0ddf] = "\xc2\xc0", [0x0de0] = "\xbb\xf4", [0x0de1] = "\xba\x58",
  [0x0de2] = "\xac\xe8", [0x0de3] = "\xaa\xc2", [0x0de4] = "\xc3\x4d",
  [0x0de5] = "\xb0\x5f", [0x0de6] = "\xb0\x5a", [0x0de7] = "\xa4\x5e",
  [0x0de8] = "\xa5\xf8", [0x0de9] = "\xb1\xd2", [0x0dea] = "\xab\xb4",
  [0x0deb] = "\xac\xe4", [0x0dec] = "\xbe\xb9", [0x0ded] = "\xae\xf0",
  [0x0dee] = "\xa8\xb4", [0x0def] = "\xb1\xf3", [0x0df0] = "\xa8\x54",
  [0x0df1] = "\xaa\x5f", [0x0df2] = "\xb0\x57", [0x0df3] = "\xd5\x74",
  [0x0df4] = "\xab\xea", [0x0df5] = "\xac\xa2", [0x0df6] = "\xb2\x6f",
  [0x0df7] = "\xc9\xe7", [0x0df8] = "\xd7\xf7", [0x0df9] = "\xb9\x5d",
  [0x0dfa] = "\xa4\x64", [0x0dfb] = "\xbe\x45", [0x0dfc] = "\xc3\xb1",
  [0x0dfd] = "\xa5\x61", [0x0dfe] = "\xc1\xbe", [0x0dff] = "\xb0\xae",
  [0x0e00] = "\xc0\x72", [0x0e01] = "\xbf\xfa", [0x0e02] = "\xb9\x58",
  [0x0e03] = "\xab\x65", [0x0e04] = "\xbc\xe7", [0x0e05] = "\xbb\xba",
  [0x0e06] = "\xb2\x4c", [0x0e07] = "\xc4\xfe", [0x0e08] = "\xb9\xd5",
  [0x0e09] = "\xb4\x4f", [0x0e0a] = "\xa4\xed", [0x0e0b] = "\xba\x70",
  [0x0e0c] = "\xba\x6a", [0x0e0d] = "\xb6\xe6", [0x0e0e] = "\xb5\xc4",
  [0x0e0f] = "\xaa\xca", [0x0e10] = "\xc0\xf0", [0x0e11] = "\xc1\xa5",
  [0x0e12] = "\xb1\x6a", [0x0e13] = "\xb7\x6d", [0x0e14] = "\xbe\xf5",
  [0x0e15] = "\xc1\xea", [0x0e16] = "\xba\x56", [0x0e17] = "\xae\xa8",
  [0x0e18] = "\xbe\xf4", [0x0e19] = "\xc1\x40", [0x0e1a] = "\xb3\xec",
  [0x0e1b] = "\xb9\xb4", [0x0e1c] = "\xa5\xa9", [0x0e1d] = "\xc0\x54",
  [0x0e1e] = "\xbc\xc0", [0x0e1f] = "\xc2\xbc", [0x0e20] = "\xae\x6b",
  [0x0e21] = "\xab\x4e", [0x0e22] = "\xc2\xac", [0x0e23] = "\xa4\xc1",
  [0x0e24] = "\xad\x58", [0x0e25] = "\xa5\x42", [0x0e26] = "\xa9\xc4",
  [0x0e27] = "\xc5\xd1", [0x0e28] = "\xb4\xdc", [0x0e29] = "\xab\x49",
  [0x0e2a] = "\xbf\xcb", [0x0e2b] = "\xaf\xb3", [0x0e2c] = "\xb5\x5e",
  [0x0e2d] = "\xb6\xd4", [0x0e2e] = "\xaa\xe0", [0x0e2f] = "\xbe\xe0",
  [0x0e30] = "\xb8\x56", [0x0e31] = "\xb9\xec", [0x0e32] = "\xa8\x47",
  [0x0e33] = "\xab\x43", [0x0e34] = "\xbb\xb4", [0x0e35] = "\xb2\x42",
  [0x0e36] = "\xb6\xc9", [0x0e37] = "\xad\xeb", [0x0e38] = "\xb2\x4d",
  [0x0e39] = "\xc0\xba", [0x0e3a] = "\xb4\xb8", [0x0e3b] = "\xd9\xe6",
  [0x0e3c] = "\xb1\xa1", [0x0e3d] = "\xb3\xbc", [0x0e3e] = "\xbd\xd0",
  [0x0e3f] = "\xbc\x79", [0x0e40] = "\xc3\xa3", [0x0e41] = "\xbd\x61",
  [0x0e42] = "\xac\xee", [0x0e43] = "\xa5\x43", [0x0e44] = "\xaa\xf4",
  [0x0e45] = "\xb2\x79", [0x0e46] = "\xa8\x44", [0x0e47] = "\xa5\x7d",
  [0x0e48] = "\xad\xa9", [0x0e49] = "\xaa\x73", [0x0e4a] = "\xc1\xcd",
  [0x0e4b] = "\xb0\xcf", [0x0e4c] = "\xb3\x49", [0x0e4d] = "\xa6\xb1",
  [0x0e4e] = "\xc2\xdf", [0x0e4f] = "\xa9\x7d", [0x0e50] = "\xc5\x58",
  [0x0e51] = "\xb4\xeb", [0x0e52] = "\xa8\xfa", [0x0e53] = "\xb0\xf9",
  [0x0e54] = "\xc6\x54", [0x0e55] = "\xbd\xec", [0x0e56] = "\xa5\x68",
  [0x0e57] = "\xb0\xe9", [0x0e58] = "\xf9\xaf", [0x0e59] = "\xc5\x76",
  [0x0e5a] = "\xee\xc7", [0x0e5b] = "\xac\x75", [0x0e5c] = "\xa5\xfe",
  [0x0e5d] = "\xb2\xac", [0x0e5e] = "\xae\xb1", [0x0e5f] = "\xa4\xfc",
  [0x0e60] = "\xa8\xe9", [0x0e61] = "\xc4\x55", [0x0e62] = "\xaf\xca",
  [0x0e63] = "\xcd\x50", [0x0e64] = "\xbf\x61", [0x0e65] = "\xab\x6f",
  [0x0e66] = "\xc4\x4e", [0x0e67] = "\xba\x65", [0x0e68] = "\xbd\x54",
  [0x0e69] = "\xb3\xb6", [0x0e6a] = "\xb8\xc8", [0x0e6b] = "\xb8\x73",
  [0x0e6c] = "\xb5\x4d", [0x0e6d] = "\xbf\x55", [0x0e6e] = "\xa5\x54",
  [0x0e6f] = "\xac\x56", [0x0e70] = "\xc5\x7b", [0x0e71] = "\xc4\x5b",
  [0x0e72] = "\xc4\x63", [0x0e73] = "\xc4\x57", [0x0e74] = "\xc5\xfd",
  [0x0e75] = "\xc4\xc7", [0x0e76] = "\xc2\x5a", [0x0e77] = "\xc2\xb6",
  [0x0e78] = "\xb7\x53", [0x0e79] = "\xbc\xf6", [0x0e7a] = "\xa4\xd0",
  [0x0e7b] = "\xa4\xaf", [0x0e7c] = "\xa4\x48", [0x0e7d] = "\xa7\xd4",
  [0x0e7e] = "\xb6\xb4", [0x0e7f] = "\xa5\xf4", [0x0e80] = "\xbb\x7b",
  [0x0e81] = "\xa4\x62", [0x0e82] = "\xa7\xb3", [0x0e83] = "\xac\xf7",
  [0x0e84] = "\xa5\xb5", [0x0e85] = "\xa4\xb4", [0x0e86] = "\xa4\xe9",
  [0x0e87] = "\xa6\xa5", [0x0e88] = "\xaf\xf1", [0x0e89] = "\xbb\x54",
  [0x0e8a] = "\xba\x61", [0x0e8b] = "\xbf\xc4", [0x0e8c] = "\xba\xb2",
  [0x0e8d] = "\xb7\xbb", [0x0e8e] = "\xae\x65", [0x0e8f] = "\xb5\xb3",
  [0x0e90] = "\xa4\xbe", [0x0e91] = "\xb4\x7c", [0x0e92] = "\xac\x58",
  [0x0e93] = "\xa6\xd7", [0x0e94] = "\xaf\xf8", [0x0e95] = "\xc4\xaf",
  [0x0e96] = "\xbe\xa7", [0x0e97] = "\xc0\xa9", [0x0e98] = "\xa6\x70",
  [0x0e99] = "\xb0\x64", [0x0e9a] = "\xa8\xc5", [0x0e9b] = "\xa6\xbc",
  [0x0e9c] = "\xa4\x4a", [0x0e9d] = "\xbf\xc8", [0x0e9e] = "\xb3\x6e",
  [0x0e9f] = "\xa8\xbf", [0x0ea0] = "\xbf\xb6", [0x0ea1] = "\xb7\xe7",
  [0x0ea2] = "\xbe\x55", [0x0ea3] = "\xb6\x7c", [0x0ea4] = "\xbc\xed",
  [0x0ea5] = "\xad\x59", [0x0ea6] = "\xae\x7a", [0x0ea7] = "\xbc\xbb",
  [0x0ea8] = "\xc5\x78", [0x0ea9] = "\xc2\xc4", [0x0eaa] = "\xb8\x7c",
  [0x0eab] = "\xc4\xce", [0x0eac] = "\xb6\xeb", [0x0ead] = "\xc1\xc9",
  [0x0eae] = "\xa4\x54", [0x0eaf] = "\xb0\xd1", [0x0eb0] = "\xb3\xca",
  [0x0eb1] = "\xb4\xb2", [0x0eb2] = "\xae\xe1", [0x0eb3] = "\xb6\xda",
  [0x0eb4] = "\xb3\xe0", [0x0eb5] = "\xb7\x6b", [0x0eb6] = "\xc4\xcc",
  [0x0eb7] = "\xb1\xbd", [0x0eb8] = "\xb7\x41", [0x0eb9] = "\xb7\xe6",
  [0x0eba] = "\xa6\xe2", [0x0ebb] = "\xc0\xdf", [0x0ebc] = "\xb4\xcb",
  [0x0ebd] = "\xb9\xac", [0x0ebe] = "\xb2\xef", [0x0ebf] = "\xac\xe2",
  [0x0ec0] = "\xb1\xfe", [0x0ec1] = "\xab\x62", [0x0ec2] = "\xa8\x46",
  [0x0ec3] = "\xaf\xbd", [0x0ec4] = "\xb6\xcc", [0x0ec5] = "\xd4\xa3",
  [0x0ec6] = "\xb7\xd9", [0x0ec7] = "\xbf\x7a", [0x0ec8] = "\xc5\xce",
  [0x0ec9] = "\xac\xc0", [0x0eca] = "\xd0\x73", [0x0ecb] = "\xa7\xfc",
  [0x0ecc] = "\xa4\x73", [0x0ecd] = "\xa7\x52", [0x0ece] = "\xba\xb4",
  [0x0ecf] = "\xad\x6d", [0x0ed0] = "\xb0\x7b", [0x0ed1] = "\xb0\xa2",
  [0x0ed2] = "\xbe\xd5", [0x0ed3] = "\xc4\xba", [0x0ed4] = "\xbf\xaf",
  [0x0ed5] = "\xb5\xbd", [0x0ed6] = "\xa6\xc2", [0x0ed7] = "\xae\xb0",
  [0x0ed8] = "\xc2\xb5", [0x0ed9] = "\xe1\x61", [0x0eda] = "\xb6\xcb",
  [0x0edb] = "\xb0\xd3", [0x0edc] = "\xbd\xe0", [0x0edd] = "\xae\xce",
  [0x0ede] = "\xa4\x57", [0x0edf] = "\xa9\x7c", [0x0ee0] = "\xbb\x6e",
  [0x0ee1] = "\xb1\xe9", [0x0ee2] = "\xae\xb4", [0x0ee3] = "\xb5\x79",
  [0x0ee4] = "\xbf\x4e", [0x0ee5] = "\xa8\xa2", [0x0ee6] = "\xa4\x63",
  [0x0ee7] = "\xbb\xe0", [0x0ee8] = "\xa4\xd6", [0x0ee9] = "\xad\xef",
  [0x0eea] = "\xaa\xf2", [0x0eeb] = "\xb2\xd0", [0x0eec] = "\xb0\xf8",
  [0x0eed] = "\xbb\xad", [0x0eee] = "\xb3\x44", [0x0eef] = "\xa6\xde",
  [0x0ef0] = "\xaa\xd9", [0x0ef1] = "\xb3\x6a", [0x0ef2] = "\xc4\xe1",
  [0x0ef3] = "\xae\x67", [0x0ef4] = "\xc4\xe0", [0x0ef5] = "\xaf\x41",
  [0x0ef6] = "\xaa\xc0", [0x0ef7] = "\xb3\x5d", [0x0ef8] = "\xaf\x7e",
  [0x0ef9] = "\xa5\xd3", [0x0efa] = "\xa9\x44", [0x0efb] = "\xa6\xf9",
  [0x0efc] = "\xa8\xad", [0x0efd] = "\xb2\x60", [0x0efe] = "\xae\x57",
  [0x0eff] = "\xb2\xd4", [0x0f00] = "\xaf\xab", [0x0f01] = "\xa8\x48",
  [0x0f02] = "\xbc\x66", [0x0f03] = "\xc2\x54", [0x0f04] = "\xac\xc6",
  [0x0f05] = "\xb5\xc7", [0x0f06] = "\xb7\x56", [0x0f07] = "\xba\xaf",
  [0x0f08] = "\xc1\x6e", [0x0f09] = "\xa5\xcd", [0x0f0a] = "\xb5\x63",
  [0x0f0b] = "\xac\xb9", [0x0f0c] = "\xa4\xc9", [0x0f0d] = "\xc3\xb7",
  [0x0f0e] = "\xac\xd9", [0x0f0f] = "\xb2\xb1", [0x0f10] = "\xb3\xd1",
  [0x0f11] = "\xb3\xd3", [0x0f12] = "\xb8\x74", [0x0f13] = "\xae\x76",
  [0x0f14] = "\xa5\xa2", [0x0f15] = "\xb7\xe0", [0x0f16] = "\xac\x49",
  [0x0f17] = "\xc0\xe3", [0x0f18] = "\xb8\xd6", [0x0f19] = "\xab\xcd",
  [0x0f1a] = "\xaa\xeb", [0x0f1b] = "\xa4\x51", [0x0f1c] = "\xa5\xdb",
  [0x0f1d] = "\xac\x42", [0x0f1e] = "\xae\xc9", [0x0f1f] = "\xa4\xb0",
  [0x0f20] = "\xad\xb9", [0x0f21] = "\xbb\x6b", [0x0f22] = "\xb9\xea",
  [0x0f23] = "\xc3\xd1", [0x0f24] = "\xa5\x76", [0x0f25] = "\xa5\xda",
  [0x0f26] = "\xa8\xcf", [0x0f27] = "\xab\xcb", [0x0f28] = "\xbe\x70",
  [0x0f29] = "\xa9\x6c", [0x0f2a] = "\xa6\xa1", [0x0f2b] = "\xa5\xdc",
  [0x0f2c] = "\xa4\x68", [0x0f2d] = "\xa5\x40", [0x0f2e] = "\xac\x55",
  [0x0f2f] = "\xa8\xc6", [0x0f30] = "\xab\xf8", [0x0f31] = "\xbb\x7d",
  [0x0f32] = "\xb3\x75", [0x0f33] = "\xb6\xd5", [0x0f34] = "\xac\x4f",
  [0x0f35] = "\xb6\xdd", [0x0f36] = "\xbe\xbd", [0x0f37] = "\xbe\x41",
  [0x0f38] = "\xa5\x4b", [0x0f39] = "\xa8\xcd", [0x0f3a] = "\xc4\xc0",
  [0x0f3b] = "\xb9\xa2", [0x0f3c] = "\xa4\xf3", [0x0f3d] = "\xa5\xab",
  [0x0f3e] = "\xab\xee", [0x0f3f] = "\xab\xc7", [0x0f40] = "\xb5\xf8",
  [0x0f41] = "\xb8\xd5", [0x0f42] = "\xa6\xac", [0x0f43] = "\xa4\xe2",
  [0x0f44] = "\xad\xba", [0x0f45] = "\xa6\x75", [0x0f46] = "\xb9\xd8",
  [0x0f47] = "\xb1\xc2", [0x0f48] = "\xb0\xe2", [0x0f49] = "\xa8\xfc",
  [0x0f4a] = "\xbd\x47", [0x0f4b] = "\xc3\x7e", [0x0f4c] = "\xbd\xad",
  [0x0f4d] = "\xbc\xcf", [0x0f4e] = "\xae\xde", [0x0f4f] = "\xae\xed",
  [0x0f50] = "\xa7\xe7", [0x0f51] = "\xbf\xe9", [0x0f52] = "\xa8\xfb",
  [0x0f53] = "\xb5\xce", [0x0f54] = "\xb2\x51", [0x0f55] = "\xb2\xa8",
  [0x0f56] = "\xae\xd1", [0x0f57] = "\xc5\xab", [0x0f58] = "\xb1\x45",
  [0x0f59] = "\xbc\xf4", [0x0f5a] = "\xc1\xa6", [0x0f5b] = "\xb4\xbb",
  [0x0f5c] = "\xc0\xc6", [0x0f5d] = "\xb8\x70", [0x0f5e] = "\xb8\xbe",
  [0x0f5f] = "\xb6\xc1", [0x0f60] = "\xb9\xab", [0x0f61] = "\xc4\xdd",
  [0x0f62] = "\xb3\x4e", [0x0f63] = "\xad\x7a", [0x0f64] = "\xbe\xf0",
  [0x0f65] = "\xa7\xf4", [0x0f66] = "\xa6\xa7", [0x0f67] = "\xbd\xdd",
  [0x0f68] = "\xb9\xd6", [0x0f69] = "\xb1\x66", [0x0f6a] = "\xbc\xc6",
  [0x0f6b] = "\xba\xa4", [0x0f6c] = "\xae\xa4", [0x0f6d] = "\xa8\xea",
  [0x0f6e] = "\xad\x41", [0x0f6f] = "\xba\x4c", [0x0f70] = "\xb0\x49",
  [0x0f71] = "\xa5\xcf", [0x0f72] = "\xab\xd3", [0x0f73] = "\xae\xea",
  [0x0f74] = "\xac\x43", [0x0f75] = "\xc1\xf7", [0x0f76] = "\xc2\xf9",
  [0x0f77] = "\xb2\x6e", [0x0f78] = "\xbd\xd6", [0x0f79] = "\xa4\xf4",
  [0x0f7a] = "\xba\xce", [0x0f7b] = "\xb5\x7c", [0x0f7c] = "\xa7\x6d",
  [0x0f7d] = "\xc0\xfe", [0x0f7e] = "\xb6\xb6", [0x0f7f] = "\xb5\xcf",
  [0x0f80] = "\xbb\xa1", [0x0f81] = "\xba\xd3", [0x0f82] = "\xae\xd2",
  [0x0f83] = "\xc3\x7b", [0x0f84] = "\xb4\xb5", [0x0f85] = "\xbc\xb9",
  [0x0f86] = "\xbc\x52", [0x0f87] = "\xab\xe4", [0x0f88] = "\xa8\x70",
  [0x0f89] = "\xa5\x71", [0x0f8a] = "\xb5\xb7", [0x0f8b] = "\xa6\xba",
  [0x0f8c] = "\xb8\x76", [0x0f8d] = "\xa6\x78", [0x0f8e] = "\xb6\xe0",
  [0x0f8f] = "\xa5\x7c", [0x0f90] = "\xa6\xf8", [0x0f91] = "\xa6\xfc",
  [0x0f92] = "\xb9\x7d", [0x0f93] = "\xa4\x78", [0x0f94] = "\xc3\x50",
  [0x0f95] = "\xc1\x71", [0x0f96] = "\xbc\xa3", [0x0f97] = "\xb9\x7c",
  [0x0f98] = "\xb0\x65", [0x0f99] = "\xa7\xba", [0x0f9a] = "\xb3\x5e",
  [0x0f9b] = "\xbb\x77", [0x0f9c] = "\xb7\x6a", [0x0f9d] = "\xbf\xb4",
  [0x0f9e] = "\xc2\x5d", [0x0f9f] = "\xb9\xc2", [0x0fa0] = "\xc4\xac",
  [0x0fa1] = "\xb6\x70", [0x0fa2] = "\xab\x55", [0x0fa3] = "\xaf\xc0",
  [0x0fa4] = "\xb3\x74", [0x0fa5] = "\xb5\xaf", [0x0fa6] = "\xe0\xf4",
  [0x0fa7] = "\xb6\xec", [0x0fa8] = "\xb7\xb9", [0x0fa9] = "\xb1\x4a",
  [0x0faa] = "\xb6\x44", [0x0fab] = "\xb5\xc2", [0x0fac] = "\xbb\xc4",
  [0x0fad] = "\xbb\x5b", [0x0fae] = "\xba\xe2", [0x0faf] = "\xc1\xf6",
  [0x0fb0] = "\xb6\xa6", [0x0fb1] = "\xc0\x48", [0x0fb2] = "\xb8\x6b",
  [0x0fb3] = "\xc5\xe8", [0x0fb4] = "\xb8\x48", [0x0fb5] = "\xb7\xb3",
  [0x0fb6] = "\xc1\x4a", [0x0fb7] = "\xb9\x45", [0x0fb8] = "\xc0\x47",
  [0x0fb9] = "\xaf\xa9", [0x0fba] = "\xae\x5d", [0x0fbb] = "\xb7\x6c",
  [0x0fbc] = "\xb5\xab", [0x0fbd] = "\xbf\x75", [0x0fbe] = "\xb1\xf4",
  [0x0fbf] = "\xad\xf6", [0x0fc0] = "\xc1\x59", [0x0fc1] = "\xba\xbe",
  [0x0fc2] = "\xaf\xc1", [0x0fc3] = "\xc2\xea", [0x0fc4] = "\xa9\xd2",
  [0x0fc5] = "\xb6\xf2", [0x0fc6] = "\xa5\x4c", [0x0fc7] = "\xa5\xa6",
  [0x0fc8] = "\xa6\x6f", [0x0fc9] = "\xb6\xf0", [0x0fca] = "\xc3\xa1",
  [0x0fcb] = "\xbe\xd8", [0x0fcc] = "\xc1\xcf", [0x0fcd] = "\xbd\xf1",
  [0x0fce] = "\xad\x4c", [0x0fcf] = "\xad\x61", [0x0fd0] = "\xa9\xef",
  [0x0fd1] = "\xa5\x78", [0x0fd2] = "\xae\xf5", [0x0fd4] = "\xa4\xd3",
  [0x0fd5] = "\xba\x41", [0x0fd6] = "\xa8\x4f", [0x0fd7] = "\xa7\x7e",
  [0x0fd8] = "\xc5\x75", [0x0fd9] = "\xb3\x67", [0x0fda] = "\xc5\xf5",
  [0x0fdb] = "\xc5\x79", [0x0fdc] = "\xbe\xc2", [0x0fdd] = "\xc0\xc8",
  [0x0fde] = "\xb7\xf0", [0x0fdf] = "\xbc\xe6", [0x0fe0] = "\xc3\xd3",
  [0x0fe1] = "\xbd\xcd", [0x0fe2] = "\xa9\x5a", [0x0fe3] = "\xb4\xe0",
  [0x0fe4] = "\xb3\x52", [0x0fe5] = "\xba\xd2", [0x0fe6] = "\xb1\xb4",
  [0x0fe7] = "\xb9\xc4", [0x0fe8] = "\xac\xb4", [0x0fe9] = "\xb4\xf6",
  [0x0fea] = "\xb6\xed", [0x0feb] = "\xb7\x65", [0x0fec] = "\xb0\xf3",
  [0x0fed] = "\xb4\xc5", [0x0fee] = "\xbd\xa3", [0x0fef] = "\xad\xf0",
  [0x0ff0] = "\xbf\x7d", [0x0ff1] = "\xad\xd5", [0x0ff2] = "\xbd\xf6",
  [0x0ff3] = "\xb2\x49", [0x0ff4] = "\xbd\xeb", [0x0ff5] = "\xbf\x53",
  [0x0ff6] = "\xb1\xc7", [0x0ff7] = "\xc0\xdc", [0x0ff8] = "\xb7\xca",
  [0x0ff9] = "\xb8\x6c", [0x0ffa] = "\xb5\xe5", [0x0ffb] = "\xae\xe7",
  [0x0ffc] = "\xb0\x6b", [0x0ffd] = "\xb2\x5e", [0x0ffe] = "\xb3\xb3",
  [0x0fff] = "\xb0\x51", [0x1000] = "\xae\x4d", [0x1001] = "\xaf\x53",
  [0x1002] = "\xc3\xc3", [0x1003] = "\xc4\xcb", [0x1004] = "\xaf\x6b",
  [0x1005] = "\xc1\xc3", [0x1006] = "\xb1\xe8", [0x1007] = "\xad\xe7",
  [0x1008] = "\xbd\xf0", [0x1009] = "\xbe\x4f", [0x100a] = "\xb4\xa3",
  [0x100b] = "\xc3\x44", [0x100c] = "\xbf\xe1", [0x100d] = "\xb3\xda",
  [0x100e] = "\xc5\xe9", [0x100f] = "\xb4\xc0", [0x1010] = "\xc0\xa1",
  [0x1011] = "\xb1\xa7", [0x1012] = "\xae\xf7", [0x1013] = "\xab\x63",
  [0x1014] = "\xb1\x50", [0x1015] = "\xa4\xd1", [0x1016] = "\xb2\x4b",
  [0x1017] = "\xb6\xf1", [0x1018] = "\xa5\xd0", [0x1019] = "\xb2\xa2",
  [0x101a] = "\xab\xef", [0x101b] = "\xbb\x51", [0x101c] = "\xb5\xc9",
  [0x101d] = "\xac\x44", [0x101e] = "\xb1\xf8", [0x101f] = "\xad\x7c",
  [0x1020] = "\xb2\xb7", [0x1021] = "\xb8\xf5", [0x1022] = "\xb6\x4b",
  [0x1023] = "\xc5\x4b", [0x1024] = "\xa9\xab", [0x1025] = "\xc6\x55",
  [0x1026] = "\xc5\xa5", [0x1027] = "\xd6\x4c", [0x1028] = "\xa5\xc5",
  [0x1029] = "\xa7\xca", [0x102a] = "\xb0\xb1", [0x102b] = "\xab\x46",
  [0x102c] = "\xae\x78", [0x102d] = "\xae\xbc", [0x102e] = "\xb8\xa5",
  [0x102f] = "\xb3\x71", [0x1030] = "\xae\xe4", [0x1031] = "\xe0\xac",
  [0x1032] = "\xc0\xfb", [0x1033] = "\xa6\x50", [0x1034] = "\xbb\xc9",
  [0x1035] = "\xa7\xcd", [0x1036] = "\xb5\xa3", [0x1037] = "\xb1\xed",
  [0x1038] = "\xd1\xb6", [0x1039] = "\xb5\xa9", [0x103a] = "\xb2\xce",
  [0x103b] = "\xb5\x68", [0x103c] = "\xb0\xbd", [0x103d] = "\xa7\xeb",
  [0x103e] = "\xc0\x59", [0x103f] = "\xb3\x7a", [0x1040] = "\xa5\x59",
  [0x1041] = "\xa8\x72", [0x1042] = "\xac\xf0", [0x1043] = "\xb9\xcf",
  [0x1044] = "\xae\x7b", [0x1045] = "\xb3\x7e", [0x1046] = "\xb6\xee",
  [0x1047] = "\xb1\x4f", [0x1048] = "\xa4\x67", [0x1049] = "\xa6\x52",
  [0x104a] = "\xa8\xdf", [0x104b] = "\xb4\xf8", [0x104c] = "\xb9\xce",
  [0x104d] = "\xb1\xc0", [0x104e] = "\xc0\x5a", [0x104f] = "\xbb\x4c",
  [0x1050] = "\xb8\xc0", [0x1051] = "\xbf\xc6", [0x1052] = "\xb0\x68",
  [0x1053] = "\xa7\x5d", [0x1054] = "\xa4\xd9", [0x1055] = "\xc1\x76",
  [0x1056] = "\xa9\xec", [0x1057] = "\xb0\x55", [0x1058] = "\xb2\xe6",
  [0x1059] = "\xc0\x6b", [0x105a] = "\xaa\xfb", [0x105b] = "\xb9\xa4",
  [0x105c] = "\xbe\x6d", [0x105d] = "\xbe\xf2", [0x105e] = "\xa7\xb4",
  [0x105f] = "\xa9\xdd", [0x1060] = "\xb3\xe8", [0x1061] = "\xab\xf5",
  [0x1062] = "\xab\x7a", [0x1063] = "\xb5\xec", [0x1064] = "\xba\xda",
  [0x1065] = "\xab\xbd", [0x1066] = "\xa5\xcb", [0x1067] = "\xc4\xfb",
  [0x1068] = "\xac\x6e", [0x1069] = "\xa5\x7e", [0x106a] = "\xbd\xdc",
  [0x106b] = "\xc5\x73", [0x106c] = "\xc6\x57", [0x106d] = "\xaa\xb1",
  [0x106e] = "\xb9\x78", [0x106f] = "\xa4\x59", [0x1070] = "\xd6\x4a",
  [0x1071] = "\xa7\xb9", [0x1072] = "\xb8\x4a", [0x1073] = "\xae\xbe",
  [0x1074] = "\xb1\xdf", [0x1075] = "\xb5\x70", [0x1076] = "\xb1\x7b",
  [0x1077] = "\xa9\x7b", [0x1078] = "\xb0\xfb", [0x1079] = "\xb8\x55",
  [0x107a] = "\xb5\xc3", [0x107b] = "\xa8\x4c", [0x107c] = "\xa4\xfd",
  [0x107d] = "\xa4\x60", [0x107e] = "\xaa\x50", [0x107f] = "\xba\xf4",
  [0x1080] = "\xa9\xb9", [0x1081] = "\xa9\xf4", [0x1082] = "\xb1\xe6",
  [0x1083] = "\xa7\xd1", [0x1084] = "\xa6\x6b", [0x1085] = "\xab\xc2",
  [0x1086] = "\xc4\xde", [0x1087] = "\xb7\x4c", [0x1088] = "\xa6\x4d",
  [0x1089] = "\xad\xb3", [0x108a] = "\xb9\x48", [0x108b] = "\xae\xe9",
  [0x108c] = "\xb3\xf2", [0x108d] = "\xb0\xdf", [0x108e] = "\xb1\xa9",
  [0x108f] = "\xac\xb0", [0x1090] = "\xc0\xe5", [0x1091] = "\xba\xfb",
  [0x1092] = "\xb8\xab", [0x1093] = "\xb5\xe4", [0x1094] = "\xa9\x65",
  [0x1095] = "\xb0\xb6", [0x1096] = "\xb0\xb0", [0x1097] = "\xa7\xc0",
  [0x1098] = "\xbd\x6e", [0x1099] = "\xa5\xbc", [0x109a] = "\xbd\xab",
  [0x109b] = "\xa8\xfd", [0x109c] = "\xac\xc8", [0x109d] = "\xad\x47",
  [0x109e] = "\xb3\xde", [0x109f] = "\xc3\x51", [0x10a0] = "\xa6\xec",
  [0x10a1] = "\xb4\xf4", [0x10a2] = "\xbf\xd7", [0x10a3] = "\xb1\x4c",
  [0x10a4] = "\xbc\xa2", [0x10a5] = "\xbd\xc3", [0x10a6] = "\xbd\x45",
  [0x10a7] = "\xb7\xc5", [0x10a8] = "\xb0\x41", [0x10a9] = "\xa4\xe5",
  [0x10aa] = "\xbb\x44", [0x10ab] = "\xaf\xbe", [0x10ac] = "\xa7\x6b",
  [0x10ad] = "\xc3\xad", [0x10ae] = "\xaf\xbf", [0x10af] = "\xb0\xdd",
  [0x10b0] = "\xb6\xe4", [0x10b1] = "\xaf\xce", [0x10b2] = "\xc2\x7c",
  [0x10b3] = "\xbe\xe2", [0x10b4] = "\xbd\xbd", [0x10b5] = "\xb4\xf5",
  [0x10b6] = "\xba\xdb", [0x10b7] = "\xa7\xda", [0x10b8] = "\xba\x57",
  [0x10b9] = "\xaa\xd7", [0x10ba] = "\xb4\xa4", [0x10bb] = "\xa8\x55",
  [0x10bc] = "\xa7\xc5", [0x10bd] = "\xb6\xe3", [0x10be] = "\xc2\xeb",
  [0x10bf] = "\xaf\x51", [0x10c0] = "\xa6\xc3", [0x10c1] = "\xbb\x7a",
  [0x10c2] = "\xab\xce", [0x10c3] = "\xb5\x4c", [0x10c4] = "\xbf\xbe",
  [0x10c5] = "\xb1\xef", [0x10c6] = "\xa7\x5e", [0x10c7] = "\xa7\x64",
  [0x10c8] = "\xa4\xf0", [0x10c9] = "\xaa\x5a", [0x10ca] = "\xa4\xad",
  [0x10cb] = "\xdd\xb3", [0x10cc] = "\xa4\xc8", [0x10cd] = "\xbb\x52",
  [0x10ce] = "\xa5\xee", [0x10cf] = "\xab\x56", [0x10d0] = "\xb6\xf5",
  [0x10d1] = "\xa5\xb3", [0x10d2] = "\xc3\xfa", [0x10d3] = "\xb1\xe0",
  [0x10d4] = "\xaa\xab", [0x10d5] = "\xa4\xc5", [0x10d6] = "\xb0\xc8",
  [0x10d7] = "\xae\xa9", [0x10d8] = "\xbb\x7e", [0x10d9] = "\xa9\xf5",
  [0x10da] = "\xba\xb3", [0x10db] = "\xaa\x52", [0x10dc] = "\xa6\xe8",
  [0x10dd] = "\xd6\xb4", [0x10de] = "\xaa\xbf", [0x10df] = "\xb4\xb7",
  [0x10e0] = "\xbc\x48", [0x10e1] = "\xa7\x6c", [0x10e2] = "\xbf\xfc",
  [0x10e3] = "\xc4\xeb", [0x10e4] = "\xb5\x7d", [0x10e5] = "\xae\xa7",
  [0x10e6] = "\xa7\xc6", [0x10e7] = "\xb1\x78", [0x10e8] = "\xbd\xa5",
  [0x10e9] = "\xa4\x69", [0x10ea] = "\xb1\xa4", [0x10eb] = "\xba\xb6",
  [0x10ec] = "\xb2\x6d", [0x10ed] = "\xb7\xcb", [0x10ee] = "\xa6\xc1",
  [0x10ef] = "\xb5\x52", [0x10f0] = "\xc0\xca", [0x10f1] = "\xc5\xa7",
  [0x10f2] = "\xae\x75", [0x10f3] = "\xb2\xdf", [0x10f4] = "\xb7\x40",
  [0x10f5] = "\xb3\xdf", [0x10f6] = "\xbb\xd1", [0x10f7] = "\xac\x7e",
  [0x10f8] = "\xa8\x74", [0x10f9] = "\xbb\xd8", [0x10fa] = "\xc0\xb8",
  [0x10fb] = "\xb2\xd3", [0x10fc] = "\xbd\x4d", [0x10fd] = "\xbd\xbc",
  [0x10fe] = "\xa7\x58", [0x10ff] = "\xc1\xf8", [0x1100] = "\xc1\xd2",
  [0x1101] = "\xb7\x76", [0x1102] = "\xae\x6c", [0x1103] = "\xab\x4c",
  [0x1104] = "\xaf\x55", [0x1105] = "\xa4\x55", [0x1106] = "\xb7\x48",
  [0x1107] = "\xae\x4c", [0x1108] = "\xc0\x7e", [0x1109] = "\xb1\xc8",
  [0x110b] = "\xa5\xfd", [0x110c] = "\xa5\x50", [0x110d] = "\xc2\x41",
  [0x110e] = "\xc5\xd6", [0x110f] = "\xc4\xd0", [0x1110] = "\xbd\xe5",
  [0x1111] = "\xbb\xce", [0x1112] = "\xb2\xec", [0x1113] = "\xb6\xa2",
  [0x1114] = "\xb2\x43", [0x1115] = "\xa9\xb6", [0x1116] = "\xb6\xfb",
  [0x1117] = "\xc5\xe3", [0x1118] = "\xc0\x49", [0x1119] = "\xb2\x7b",
  [0x111a] = "\xc4\x6d", [0x111b] = "\xbf\xa4", [0x111c] = "\xb8\xa2",
  [0x111d] = "\xc0\x60", [0x111e] = "\xb8\x72", [0x111f] = "\xbe\xcb",
  [0x1120] = "\xb3\xb4", [0x1121] = "\xad\xad", [0x1122] = "\xbd\x75",
  [0x1123] = "\xac\xdb", [0x1124] = "\xb4\x5b", [0x1125] = "\xc6\x5e",
  [0x1126] = "\xad\xbb", [0x1127] = "\xbd\x63", [0x1128] = "\xc1\xb8",
  [0x1129] = "\xb4\xf0", [0x112a] = "\xb6\x6d", [0x112b] = "\xb5\xbe",
  [0x112c] = "\xb2\xbb", [0x112d] = "\xb8\xd4", [0x112e] = "\xb7\x51",
  [0x112f] = "\xc5\x54", [0x1130] = "\xa8\xc9", [0x1131] = "\xb6\xb5",
  [0x1132] = "\xab\xd1", [0x1133] = "\xbe\xf3", [0x1134] = "\xb9\xb3",
  [0x1135] = "\xa6\x56", [0x1136] = "\xb6\x48", [0x1137] = "\xbf\xbd",
  [0x1138] = "\xb5\x76", [0x1139] = "\xbe\x5d", [0x113a] = "\xab\x64",
  [0x113b] = "\xad\xfd", [0x113c] = "\xc4\xdb", [0x113d] = "\xbe\x50",
  [0x113e] = "\xae\xf8", [0x113f] = "\xae\x64", [0x1140] = "\xb2\x63",
  [0x1141] = "\xbe\xe5", [0x1142] = "\xa4\x70", [0x1143] = "\xa7\xb5",
  [0x1144] = "\xae\xd5", [0x1145] = "\xa8\x76", [0x1146] = "\xbc\x53",
  [0x1147] = "\xaf\xba", [0x1148] = "\xae\xc4", [0x1149] = "\xb7\xa4",
  [0x114a] = "\xa8\xc7", [0x114b] = "\xb7\xb2", [0x114c] = "\xc3\xc8",
  [0x114d] = "\xbe\x63", [0x114e] = "\xa8\xf3", [0x114f] = "\xae\xb5",
  [0x1150] = "\xc4\xe2", [0x1151] = "\xa8\xb8", [0x1152] = "\xb1\xd7",
  [0x1153] = "\xaf\xd9", [0x1154] = "\xbf\xd3", [0x1155] = "\xbc\x67",
  [0x1156] = "\xb1\xf1", [0x1157] = "\xa8\xf8", [0x1158] = "\xc3\xc9",
  [0x1159] = "\xbe\xd3", [0x115a] = "\xac\xaa", [0x115b] = "\xc2\x6d",
  [0x115c] = "\xc1\xc2", [0x115d] = "\xae\x68", [0x115e] = "\xc1\x7e",
  [0x115f] = "\xaa\xe4", [0x1160] = "\xbe\x4e", [0x1161] = "\xaa\x59",
  [0x1162] = "\xa8\xaf", [0x1163] = "\xb7\x73", [0x1164] = "\xca\xe3",
  [0x1165] = "\xa4\xdf", [0x1166] = "\xab\x48", [0x1167] = "\xc6\x5d",
  [0x1168] = "\xac\x50", [0x1169] = "\xb8\x7b", [0x116a] = "\xb5\x56",
  [0x116b] = "\xb4\x69", [0x116c] = "\xbf\xb3", [0x116d] = "\xa6\x44",
  [0x116e] = "\xab\xac", [0x116f] = "\xa7\xce", [0x1170] = "\xa8\xb7",
  [0x1171] = "\xa6\xe6", [0x1172] = "\xbf\xf4", [0x1173] = "\xa9\xaf",
  [0x1174] = "\xa7\xf6", [0x1175] = "\xa9\xca", [0x1176] = "\xa9\x6d",
  [0x1177] = "\xa5\x53", [0x1178] = "\xa5\xfb", [0x1179] = "\xaf\xdd",
  [0x117a] = "\xa6\x49", [0x117b] = "\xac\xa4", [0x117c] = "\xb6\xaf",
  [0x117d] = "\xba\xb5", [0x117e] = "\xa5\xf0", [0x117f] = "\xad\xd7",
  [0x1180] = "\xb2\xdb", [0x1181] = "\xa6\xb4", [0x1182] = "\xb6\xe5",
  [0x1183] = "\xc4\xc3", [0x1184] = "\xa8\x71", [0x1185] = "\xb3\x53",
  [0x1186] = "\xc2\xb8", [0x1187] = "\xbc\x56", [0x1188] = "\xa6\xa6",
  [0x1189] = "\xbb\xdd", [0x118a] = "\xb5\xea", [0x118b] = "\xbc\x4e",
  [0x118c] = "\xb6\xb7", [0x118d] = "\xae\x7d", [0x118e] = "\xb3\x5c",
  [0x118f] = "\xbb\x57", [0x1190] = "\xb3\xa4", [0x1191] = "\xb1\xd4",
  [0x1192] = "\xa6\xb0", [0x1193] = "\xa7\xc7", [0x1194] = "\xaf\x62",
  [0x1195] = "\xab\xf2", [0x1196] = "\xb5\xb6", [0x1197] = "\xb4\x42",
  [0x1198] = "\xba\xfc", [0x1199] = "\xc4\xf2", [0x119a] = "\xb0\x61",
  [0x119b] = "\xb3\xd9", [0x119c] = "\xab\xc5", [0x119d] = "\xc4\x61",
  [0x119e] = "\xb1\xdb", [0x119f] = "\xa5\xc8", [0x11a0] = "\xbf\xef",
  [0x11a1] = "\xc5\x7e", [0x11a2] = "\xaf\x74", [0x11a3] = "\xb5\xba",
  [0x11a4] = "\xb9\x75", [0x11a5] = "\xc1\xa7", [0x11a6] = "\xbe\xc7",
  [0x11a7] = "\xa5\xde", [0x11a8] = "\xb3\xb7", [0x11a9] = "\xa6\xe5",
  [0x11aa] = "\xbe\xb1", [0x11ab] = "\xc2\x74", [0x11ac] = "\xb4\x60",
  [0x11ad] = "\xa6\xaf", [0x11ae] = "\xb8\xdf", [0x11af] = "\xb4\x4d",
  [0x11b0] = "\xb9\xa5", [0x11b1] = "\xa8\xb5", [0x11b2] = "\xae\xee",
  [0x11b3] = "\xa6\xc4", [0x11b4] = "\xb0\x56", [0x11b5] = "\xb0\x54",
  [0x11b6] = "\xbb\xb9", [0x11b7] = "\xa8\xb3", [0x11b8] = "\xc0\xa3",
  [0x11b9] = "\xa9\xe3", [0x11ba] = "\xbe\x7e", [0x11bb] = "\xc0\x6e",
  [0x11bc] = "\xa7\x72", [0x11bd] = "\xa4\x58", [0x11be] = "\xaa\xde",
  [0x11bf] = "\xa4\xfa", [0x11c0] = "\xb0\x48", [0x11c1] = "\xb1\x56",
  [0x11c2] = "\xb8\xc5", [0x11c3] = "\xb2\x50", [0x11c4] = "\xb6\xae",
  [0x11c5] = "\xb0\xd7", [0x11c6] = "\xa8\xc8", [0x11c7] = "\xb3\x59",
  [0x11c8] = "\xb2\x6a", [0x11c9] = "\xab\x7c", [0x11ca] = "\xec\x49",
  [0x11cb] = "\xb7\xcf", [0x11cc] = "\xb2\x54", [0x11cd] = "\xc6\x51",
  [0x11ce] = "\xc4\x59", [0x11cf] = "\xac\xe3", [0x11d0] = "\xe3\xe9",
  [0x11d1] = "\xa9\xa5", [0x11d2] = "\xa9\xb5", [0x11d3] = "\xa8\xa5",
  [0x11d4] = "\xc3\x43", [0x11d5] = "\xc0\x46", [0x11d6] = "\xaa\xa2",
  [0x11d7] = "\xaa\x75", [0x11d8] = "\xa9\x61", [0x11d9] = "\xb1\xbb",
  [0x11da] = "\xb2\xb4", [0x11db] = "\xad\x6c", [0x11dc] = "\xba\x74",
  [0x11dd] = "\xc6\x76", [0x11de] = "\xb3\xf7", [0x11df] = "\xbf\x50",
  [0x11e0] = "\xb9\xbd", [0x11e1] = "\xb5\x78", [0x11e2] = "\xb6\xad",
  [0x11e3] = "\xad\xf1", [0x11e4] = "\xab\xdb", [0x11e5] = "\xbf\x56",
  [0x11e6] = "\xae\x62", [0x11e7] = "\xbf\xce", [0x11e8] = "\xc5\xe7",
  [0x11e9] = "\xac\x6f", [0x11ea] = "\xa5\xa1", [0x11eb] = "\xc0\x6d",
  [0x11ec] = "\xaf\xb1", [0x11ed] = "\xb7\xa8", [0x11ee] = "\xb4\xad",
  [0x11ef] = "\xa8\xcb", [0x11f0] = "\xba\xc5", [0x11f1] = "\xa6\xcf",
  [0x11f2] = "\xac\x76", [0x11f3] = "\xb6\xa7", [0x11f4] = "\xae\xf1",
  [0x11f5] = "\xa5\xf5", [0x11f6] = "\xc4\x6f", [0x11f7] = "\xbe\x69",
  [0x11f8] = "\xbc\xcb", [0x11f9] = "\xba\x79", [0x11fa] = "\xc1\xdc",
  [0x11fb] = "\xb8\x79", [0x11fc] = "\xa7\xaf", [0x11fd] = "\xba\xbd",
  [0x11fe] = "\xb7\x6e", [0x11ff] = "\xb3\xf3", [0x1200] = "\xbb\xbb",
  [0x1201] = "\xbd\x60", [0x1202] = "\xc1\xc1", [0x1203] = "\xab\xc0",
  [0x1204] = "\xab\x72", [0x1205] = "\xaf\xe6", [0x1206] = "\xc3\xc4",
  [0x1207] = "\xad\x6e", [0x1208] = "\xc4\xa3", [0x1209] = "\xb7\xa6",
  [0x120a] = "\xbc\x4f", [0x120b] = "\xad\x43", [0x120c] = "\xb7\xdd",
  [0x120d] = "\xb3\xa5", [0x120e] = "\xa7\x4d", [0x120f] = "\xa4\x5d",
  [0x1210] = "\xad\xb6", [0x1211] = "\xb1\xb3", [0x1212] = "\xb7\x7e",
  [0x1213] = "\xb8\xad", [0x1214] = "\xa6\xb2", [0x1215] = "\xb5\xc5",
  [0x1216] = "\xa9\x5d", [0x1217] = "\xb2\x47", [0x1218] = "\xa4\x40",
  [0x1219] = "\xb3\xfc", [0x121a] = "\xc2\xe5", [0x121b] = "\xb4\xa5",
  [0x121c] = "\xe4\x76", [0x121d] = "\xa8\xcc", [0x121e] = "\xa5\xec",
  [0x121f] = "\xa6\xe7", [0x1220] = "\xc0\x5b", [0x1221] = "\xa6\x69",
  [0x1222] = "\xbf\xf2", [0x1223] = "\xb2\xbe", [0x1224] = "\xbb\xf6",
  [0x1225] = "\xaf\xd8", [0x1226] = "\xba\xc3", [0x1227] = "\xa8\x5e",
  [0x1228] = "\xa9\x79", [0x1229] = "\xab\xbc", [0x122a] = "\xc2\x55",
  [0x122b] = "\xb4\xc8", [0x122c] = "\xc3\xc6", [0x122d] = "\xad\xca",
  [0x122e] = "\xa4\x77", [0x122f] = "\xa4\x41", [0x1230] = "\xa8\x6f",
  [0x1231] = "\xa5\x48", [0x1232] = "\xc3\xc0", [0x1233] = "\xa7\xed",
  [0x1234] = "\xa9\xf6", [0x1235] = "\xa8\xb6", [0x1236] = "\xa6\x7a",
  [0x1237] = "\xbb\xf5", [0x1238] = "\xa7\xd0", [0x1239] = "\xc1\x72",
  [0x123a] = "\xb6\x68", [0x123b] = "\xb8\x77", [0x123c] = "\xac\xcc",
  [0x123d] = "\xa5\xe7", [0x123e] = "\xb8\xc7", [0x123f] = "\xb7\x4e",
  [0x1240] = "\xbc\xdd", [0x1241] = "\xbe\xd0", [0x1242] = "\xb8\x71",
  [0x1243] = "\xaf\x71", [0x1244] = "\xb7\xb8", [0x1245] = "\xb8\xda",
  [0x1246] = "\xc4\xb3", [0x1247] = "\xbd\xcb", [0x1248] = "\xc4\xb6",
  [0x1249] = "\xb2\xa7", [0x124a] = "\xc1\x6c", [0x124b] = "\xb2\xdd",
  [0x124c] = "\xc3\xb6", [0x124d] = "\xaf\xf4", [0x124e] = "\xbd\xae",
  [0x124f] = "\xa6\x5d", [0x1250] = "\xae\xef", [0x1251] = "\xad\xb5",
  [0x1252] = "\xb3\xb1", [0x1253] = "\xab\xc3", [0x1254] = "\xa7\x75",
  [0x1255] = "\xbb\xc8", [0x1256] = "\xb2\x5d", [0x1257] = "\xb1\x47",
  [0x1258] = "\xb6\xbc", [0x1259] = "\xa4\xa8", [0x125a] = "\xa4\xde",
  [0x125b] = "\xc1\xf4", [0x125c] = "\xa6\x4c", [0x125d] = "\xad\x5e",
  [0x125e] = "\xc4\xe5", [0x125f] = "\xc0\xa6", [0x1260] = "\xc6\x4e",
  [0x1261] = "\xc0\xb3", [0x1262] = "\xc5\xd5", [0x1263] = "\xbc\xfc",
  [0x1264] = "\xbf\xc3", [0x1265] = "\xc0\xe7", [0x1266] = "\xba\xb7",
  [0x1267] = "\xc3\xc7", [0x1268] = "\xaa\xef", [0x1269] = "\xc4\xb9",
  [0x126a] = "\xac\xd5", [0x126b] = "\xbc\x76", [0x126c] = "\xbf\x6f",
  [0x126d] = "\xb5\x77", [0x126e] = "\xac\x4d", [0x126f] = "\xb3\xe9",
  [0x1270] = "\xbe\xd6", [0x1271] = "\xb6\xc4", [0x1272] = "\xc1\x73",
  [0x1273] = "\xf7\xea", [0x1274] = "\xb1\x65", [0x1275] = "\xb9\x6c",
  [0x1276] = "\xbf\xe3", [0x1277] = "\xb8\xba", [0x1278] = "\xb5\xfa",
  [0x1279] = "\xaa\x61", [0x127a] = "\xb4\xe9", [0x127b] = "\xa5\xc3",
  [0x127c] = "\xb1\x76", [0x127d] = "\xab\x69", [0x127e] = "\xa5\xce",
  [0x127f] = "\xab\xd5", [0x1280] = "\xc0\x75", [0x1281] = "\xb1\x79",
  [0x1282] = "\xbc\x7e", [0x1283] = "\xa4\xd7", [0x1284] = "\xa5\xd1",
  [0x1285] = "\xb6\x6c", [0x1286] = "\xb9\x5c", [0x1287] = "\xb5\x53",
  [0x1288] = "\xaa\x6f", [0x1289] = "\xb9\x43", [0x128a] = "\xa8\xbb",
  [0x128b] = "\xa6\xb3", [0x128c] = "\xa4\xcd", [0x128d] = "\xa5\x6b",
  [0x128e] = "\xa6\xf6", [0x128f] = "\xb9\x56", [0x1290] = "\xbb\xa4",
  [0x1291] = "\xa4\x53", [0x1292] = "\xa5\xae", [0x1293] = "\xa8\xb1",
  [0x1294] = "\xb2\x4a", [0x1295] = "\xa4\x5f", [0x1296] = "\xaa\xbb",
  [0x1297] = "\xb7\xae", [0x1298] = "\xb8\xb7", [0x1299] = "\xb7\x4d",
  [0x129a] = "\xc1\xd6", [0x129b] = "\xbe\x6c", [0x129c] = "\xab\x5c",
  [0x129d] = "\xb9\x4f", [0x129e] = "\xb3\xbd", [0x129f] = "\xb4\x72",
  [0x12a0] = "\xb4\xfc", [0x12a1] = "\xba\xae", [0x12a2] = "\xb6\xa8",
  [0x12a3] = "\xa4\xa9", [0x12a4] = "\xae\x54", [0x12a5] = "\xab\x42",
  [0x12a6] = "\xbb\x50", [0x12a7] = "\xc0\xac", [0x12a8] = "\xac\xea",
  [0x12a9] = "\xa6\x74", [0x12aa] = "\xbb\x79", [0x12ab] = "\xa6\xd0",
  [0x12ac] = "\xa5\xc9", [0x12ad] = "\xb0\xec", [0x12ae] = "\xa8\xa1",
  [0x12af] = "\xc6\x7b", [0x12b0] = "\xc6\x7e", [0x12b1] = "\xb9\x4a",
  [0x12b2] = "\xb3\xeb", [0x12b3] = "\xae\x6e", [0x12b4] = "\xbf\x6d",
  [0x12b5] = "\xb7\x55", [0x12b6] = "\xb1\xfd", [0x12b7] = "\xba\xbb",
  [0x12b8] = "\xa8\x7c", [0x12b9] = "\xc5\x41", [0x12ba] = "\xaf\x44",
  [0x12bb] = "\xb4\x4a", [0x12bc] = "\xb8\xce", [0x12bd] = "\xb9\x77",
  [0x12be] = "\xbf\xdd", [0x12bf] = "\xb6\xbf", [0x12c0] = "\xc0\x70",
  [0x12c1] = "\xb2\x57", [0x12c2] = "\xad\xde", [0x12c3] = "\xa4\xb8",
  [0x12c4] = "\xab\xae", [0x12c5] = "\xb0\x4b", [0x12c6] = "\xad\xec",
  [0x12c7] = "\xb4\xa9", [0x12c8] = "\xc1\xd5", [0x12c9] = "\xb6\xe9",
  [0x12ca] = "\xad\xfb", [0x12cb] = "\xb6\xea", [0x12cc] = "\xb7\xe1",
  [0x12cd] = "\xb7\xbd", [0x12ce] = "\xbd\x74", [0x12cf] = "\xbb\xb7",
  [0x12d0] = "\xad\x62", [0x12d1] = "\xc4\x40", [0x12d2] = "\xab\xe8",
  [0x12d3] = "\xb0\x7c", [0x12d4] = "\xa4\xea", [0x12d5] = "\xac\xf9",
  [0x12d6] = "\xb6\x56", [0x12d7] = "\xc5\x44", [0x12d8] = "\xc6\x5f",
  [0x12d9] = "\xa9\xa8", [0x12da] = "\xb8\x66", [0x12db] = "\xa4\xeb",
  [0x12dc] = "\xae\xae", [0x12dd] = "\xbe\x5c", [0x12de] = "\xaf\xd0",
  [0x12df] = "\xb6\xb3", [0x12e0] = "\xe0\xa8", [0x12e1] = "\xa4\xc3",
  [0x12e2] = "\xb9\x6b", [0x12e3] = "\xa4\xb9", [0x12e4] = "\xb9\x42",
  [0x12e5] = "\xc4\xad", [0x12e6] = "\xc1\xdf", [0x12e7] = "\xb7\x77",
  [0x12e8] = "\xc3\xfd", [0x12e9] = "\xa5\xa5", [0x12ea] = "\xa5\x60",
  [0x12eb] = "\xaf\x7b", [0x12ec] = "\xc2\xf8", [0x12ed] = "\xae\xe2",
  [0x12ee] = "\xab\x76", [0x12ef] = "\xa8\x61", [0x12f0] = "\xae\x5f",
  [0x12f1] = "\xb8\xfc", [0x12f2] = "\xa6\x41", [0x12f3] = "\xa6\x62",
  [0x12f4] = "\xab\xa5", [0x12f5] = "\xf6\xe3", [0x12f6] = "\xbc\xc8",
  [0x12f7] = "\xc3\xd9", [0x12f8] = "\xc5\x42", [0x12f9] = "\xc5\xbc",
  [0x12fa] = "\xb8\xae", [0x12fb] = "\xbe\x44", [0x12fc] = "\xc1\x56",
  [0x12fd] = "\xc6\x77", [0x12fe] = "\xc4\xa6", [0x12ff] = "\xb4\xc7",
  [0x1300] = "\xa6\xad", [0x1301] = "\xbe\xfe", [0x1302] = "\xb0\x44",
  [0x1303] = "\xc4\xbc", [0x1304] = "\xbe\xb8", [0x1305] = "\xb3\x79",
  [0x1306] = "\xa8\x6d", [0x1307] = "\xa8\x5f", [0x1308] = "\xc0\xea",
  [0x1309] = "\xb3\x64", [0x130a] = "\xbe\xdc", [0x130b] = "\xab\x68",
  [0x130c] = "\xbf\x41", [0x130d] = "\xb8\xe9", [0x130e] = "\xab\xe7",
  [0x130f] = "\xbc\x57", [0x1310] = "\xbc\xa8", [0x1311] = "\xb4\xbf",
  [0x1312] = "\xc3\xd8", [0x1313] = "\xa4\xe3", [0x1314] = "\xb3\xe5",
  [0x1315] = "\xb4\xed", [0x1316] = "\xa5\xbe", [0x1317] = "\xaa\xee",
  [0x1318] = "\xee\xd4", [0x1319] = "\xb9\x68", [0x131a] = "\xaf\x77",
  [0x131b] = "\xac\x5d", [0x131c] = "\xba\x5e", [0x131d] = "\xa9\x51",
  [0x131e] = "\xa5\x45", [0x131f] = "\xac\xb5", [0x1320] = "\xb6\x42",
  [0x1321] = "\xba\x4b", [0x1322] = "\xc2\x4e", [0x1323] = "\xa6\x76",
  [0x1324] = "\xaf\xb6", [0x1325] = "\xb6\xc5", [0x1326] = "\xb9\xeb",
  [0x1327] = "\xc2\xa4", [0x1328] = "\xc0\xd6", [0x1329] = "\xb8\xe2",
  [0x132a] = "\xd6\xdf", [0x132b] = "\xaa\x67", [0x132c] = "\xb7\xf8",
  [0x132d] = "\xb1\xd9", [0x132e] = "\xc1\xd3", [0x132f] = "\xb9\xf1",
  [0x1330] = "\xae\x69", [0x1331] = "\xc5\xd9", [0x1332] = "\xb4\xcc",
  [0x1333] = "\xa6\xfb", [0x1334] = "\xbe\xd4", [0x1335] = "\xaf\xb8",
  [0x1336] = "\xb4\xef", [0x1337] = "\xba\xec", [0x1338] = "\xbc\xcc",
  [0x1339] = "\xb3\xb9", [0x133a] = "\xb9\xfc", [0x133b] = "\xba\x73",
  [0x133c] = "\xb1\x69", [0x133d] = "\xb4\x78", [0x133e] = "\xba\xa6",
  [0x133f] = "\xa7\xfa", [0x1340] = "\xa4\x56", [0x1341] = "\xb1\x62",
  [0x1342] = "\xbd\xe3", [0x1343] = "\xa5\x4d", [0x1344] = "\xb5\xc8",
  [0x1345] = "\xbf\x60", [0x1346] = "\xbb\xd9", [0x1347] = "\xa9\xdb",
  [0x1348] = "\xac\x4c", [0x1349] = "\xa7\xe4", [0x134a] = "\xaa\x68",
  [0x134b] = "\xbb\xaf", [0x134c] = "\xb7\xd3", [0x134d] = "\xb8\x6e",
  [0x134e] = "\xa5\xfc", [0x134f] = "\xbb\x46", [0x1350] = "\xa5\x6c",
  [0x1351] = "\xbe\x42", [0x1352] = "\xa7\xe9", [0x1353] = "\xad\xf5",
  [0x1354] = "\xee\x68", [0x1355] = "\xc2\xe1", [0x1356] = "\xaa\xcc",
  [0x1357] = "\xeb\xe3", [0x1358] = "\xbd\xa9", [0x1359] = "\xb3\x6f",
  [0x135a] = "\xae\xfd", [0x135b] = "\xac\xc3", [0x135c] = "\xb7\x72",
  [0x135d] = "\xaf\x75", [0x135e] = "\xba\xc2", [0x135f] = "\xaf\x7a",
  [0x1360] = "\xbf\xb2", [0x1361] = "\xad\x73", [0x1362] = "\xb0\x77",
  [0x1363] = "\xb0\xbb", [0x1364] = "\xaa\x45", [0x1365] = "\xaf\x6c",
  [0x1366] = "\xb6\x45", [0x1367] = "\xbe\x5f", [0x1368] = "\xae\xb6",
  [0x1369] = "\xc2\xed", [0x136a] = "\xb0\x7d", [0x136b] = "\xbb\x5d",
  [0x136c] = "\xb1\xc3", [0x136d] = "\xb8\x43", [0x136e] = "\xa9\xba",
  [0x136f] = "\xb2\x75", [0x1370] = "\xaa\xa7", [0x1371] = "\xa9\xc3",
  [0x1372] = "\xbe\xe3", [0x1373] = "\xac\x40", [0x1374] = "\xa5\xbf",
  [0x1375] = "\xac\x46", [0x1376] = "\xb4\x56", [0x1377] = "\xaf\x67",
  [0x1378] = "\xbe\x47", [0x1379] = "\xc3\xd2", [0x137a] = "\xaa\xdb",
  [0x137b] = "\xaa\x4b", [0x137c] = "\xa4\xe4", [0x137d] = "\xa7\x73",
  [0x137e] = "\xbb\x6a", [0x137f] = "\xaa\xbe", [0x1380] = "\xaa\xcf",
  [0x1381] = "\xaf\xd7", [0x1382] = "\xa5\xc4", [0x1383] = "\xa4\xa7",
  [0x1384] = "\xc2\xb4", [0x1385] = "\xc2\xbe", [0x1386] = "\xaa\xbd",
  [0x1387] = "\xb4\xd3", [0x1388] = "\xb4\xde", [0x1389] = "\xb0\xf5",
  [0x138a] = "\xad\xc8", [0x138b] = "\xab\xbf", [0x138c] = "\xa7\x7d",
  [0x138d] = "\xab\xfc", [0x138e] = "\xa4\xee", [0x138f] = "\xb3\x6b",
  [0x1390] = "\xa5\x75", [0x1391] = "\xa6\xae", [0x1392] = "\xaf\xc8",
  [0x1393] = "\xa7\xd3", [0x1394] = "\xbc\xb0", [0x1395] = "\xc2\x59",
  [0x1396] = "\xa6\xdc", [0x1397] = "\xad\x50", [0x1398] = "\xb8\x6d",
  [0x1399] = "\xbc\x6d", [0x139a] = "\xab\xcf", [0x139b] = "\xa8\xee",
  [0x139c] = "\xb4\xbc", [0x139d] = "\xaf\xb4", [0x139e] = "\xb8\x58",
  [0x139f] = "\xbd\xe8", [0x13a0] = "\xaa\xa5", [0x13a1] = "\xb2\xa9",
  [0x13a2] = "\xba\xa2", [0x13a3] = "\xaa\x76", [0x13a4] = "\xb2\xbf",
  [0x13a5] = "\xa4\xa4", [0x13a6] = "\xac\xd8", [0x13a7] = "\xa9\xbe",
  [0x13a8] = "\xc4\xc1", [0x13a9] = "\xb0\x4a", [0x13aa] = "\xb2\xd7",
  [0x13ab] = "\xba\xd8", [0x13ac] = "\xb8\x7e", [0x13ad] = "\xad\xab",
  [0x13ae] = "\xa5\xf2", [0x13af] = "\xb2\xb3", [0x13b0] = "\xa6\xe0",
  [0x13b1] = "\xb6\x67", [0x13b2] = "\xa6\x7b", [0x13b3] = "\xac\x77",
  [0x13b4] = "\xee\xa4", [0x13b5] = "\xb5\xb0", [0x13b6] = "\xb6\x62",
  [0x13b7] = "\xa8\x79", [0x13b8] = "\xa9\xaa", [0x13b9] = "\xa9\x47",
  [0x13ba] = "\xbd\x4b", [0x13bb] = "\xa9\x7a", [0x13bc] = "\xb1\xde",
  [0x13bd] = "\xc6\x4a", [0x13be] = "\xaf\x5d", [0x13bf] = "\xae\xe8",
  [0x13c0] = "\xb5\xef", [0x13c1] = "\xa6\xb6", [0x13c2] = "\xbd\xde",
  [0x13c3] = "\xbd\xd1", [0x13c4] = "\xb8\xdd", [0x13c5] = "\xb3\x76",
  [0x13c6] = "\xa6\xcb", [0x13c7] = "\xc0\xeb", [0x13c8] = "\xb5\x4e",
  [0x13c9] = "\xa9\xd6", [0x13ca] = "\xc6\x66", [0x13cb] = "\xc5\xf1",
  [0x13cc] = "\xa5\x44", [0x13cd] = "\xb5\xdb", [0x13ce] = "\xac\x57",
  [0x13cf] = "\xa7\x55", [0x13d0] = "\xb3\x45", [0x13d1] = "\xb6\x4a",
  [0x13d2] = "\xc5\xb1", [0x13d3] = "\xbf\x76", [0x13d4] = "\xa6\xed",
  [0x13d5] = "\xaa\x60", [0x13d6] = "\xaf\xac", [0x13d7] = "\xbe\x6e",
  [0x13d8] = "\xa7\xec", [0x13d9] = "\xa4\xf6", [0x13da] = "\xab\xfb",
  [0x13db] = "\xb1\x4d", [0x13dc] = "\xbf\x6a", [0x13dd] = "\xc2\xe0",
  [0x13de] = "\xbc\xb6", [0x13df] = "\xc1\xc8", [0x13e0] = "\xbd\x66",
  [0x13e1] = "\xbc\xce", [0x13e2] = "\xb2\xf8", [0x13e3] = "\xb8\xcb",
  [0x13e4] = "\xa7\xa9", [0x13e5] = "\xbc\xb2", [0x13e6] = "\xa7\xa7",
  [0x13e7] = "\xaa\xac", [0x13e8] = "\xb4\xd5", [0x13e9] = "\xc0\x40",
  [0x13ea] = "\xb0\x6c", [0x13eb] = "\xc2\xd8", [0x13ec] = "\xbc\x59",
  [0x13ed] = "\xba\xf3", [0x13ee] = "\xbd\xce", [0x13ef] = "\xb7\xc7",
  [0x13f0] = "\xae\xbb", [0x13f1] = "\xa9\xe5", [0x13f2] = "\xa8\xf4",
  [0x13f3] = "\xae\xe0", [0x13f4] = "\xb5\x5a", [0x13f5] = "\xad\x5f",
  [0x13f6] = "\xb0\x75", [0x13f7] = "\xb0\xd6", [0x13f8] = "\xb5\xdb",
  [0x13f9] = "\xa8\x60", [0x13fa] = "\xbf\x42", [0x13fb] = "\xaf\xf7",
  [0x13fc] = "\xab\x74", [0x13fd] = "\xb8\xea", [0x13fe] = "\xab\xba",
  [0x13ff] = "\xb4\xfe", [0x1400] = "\xb2\x64", [0x1401] = "\xa7\xb6",
  [0x1402] = "\xb5\xb5", [0x1403] = "\xa5\x4a", [0x1404] = "\xac\xf3",
  [0x1405] = "\xb7\xba", [0x1406] = "\xa4\x6c", [0x1407] = "\xa6\xdb",
  [0x1408] = "\xba\x7b", [0x1409] = "\xa6\x72", [0x140a] = "\xc3\x4f",
  [0x140b] = "\xb4\xc4", [0x140c] = "\xc2\xdc", [0x140d] = "\xa9\x76",
  [0x140e] = "\xba\xee", [0x140f] = "\xc1\x60", [0x1410] = "\xc1\x61",
  [0x1411] = "\xb9\x51", [0x1412] = "\xa8\xab", [0x1413] = "\xab\xb5",
  [0x1414] = "\xb4\x7e", [0x1415] = "\xaf\xb2", [0x1416] = "\xa8\xac",
  [0x1417] = "\xa8\xf2", [0x1418] = "\xb1\xda", [0x1419] = "\xaf\xaa",
  [0x141a] = "\xb6\x41", [0x141b] = "\xaa\xfd", [0x141c] = "\xb2\xd5",
  [0x141d] = "\xc6\x70", [0x141e] = "\xc4\xa1", [0x141f] = "\xbc\x4c",
  [0x1420] = "\xbe\x4b", [0x1421] = "\xb3\xcc", [0x1422] = "\xb8\x6f",
  [0x1423] = "\xb4\x4c", [0x1424] = "\xbf\xed", [0x1425] = "\xac\x51",
  [0x1426] = "\xa5\xaa", [0x1427] = "\xa6\xf5", [0x1428] = "\xac\x67",
  [0x1429] = "\xb0\xb5", [0x142a] = "\xa7\x40", [0x142b] = "\xa7\xa4",
  [0x142c] = "\xae\x79", [0x1432] = "\xc9\x48", [0x1433] = "\xc9\x46",
  [0x1434] = "\xa4\x61", [0x1435] = "\xa4\xa2", [0x1436] = "\xa4\xdc",
  [0x1437] = "\xa4\xca", [0x1438] = "\xa5\x41", [0x1439] = "\xa5\xe5",
  [0x143a] = "\xa5\xe0", [0x143b] = "\xb0\xac", [0x143c] = "\xd1\x65",
  [0x143d] = "\xbe\xb5", [0x143f] = "\xac\xeb", [0x1441] = "\xa4\x50",
  [0x1442] = "\xc9\x47", [0x1443] = "\xa4\xd4", [0x1444] = "\xa4\xf8",
  [0x1445] = "\xa5\x67", [0x1446] = "\xa5\xc2", [0x1447] = "\xc9\xc7",
  [0x1448] = "\xad\x4e", [0x1449] = "\xd8\x50", [0x144a] = "\xb7\xb6",
  [0x144b] = "\xb8\x41", [0x144e] = "\xab\x45", [0x144f] = "\xe8\xf3",
  [0x1450] = "\xc9\x41", [0x1451] = "\xa5\xe4", [0x1452] = "\xc9\x51",
  [0x1453] = "\xaa\xcb", [0x1454] = "\xa7\xb8", [0x1455] = "\xb6\xde",
  [0x1456] = "\xe1\x4c", [0x1457] = "\xa4\xb7", [0x1458] = "\xcd\xf8",
  [0x1459] = "\xad\xed", [0x145a] = "\xf1\xef", [0x145b] = "\xb3\xd6",
  [0x145c] = "\xbc\x72", [0x145d] = "\xc5\xe2", [0x145e] = "\xc5\xac",
  [0x145f] = "\xc9\x43", [0x1460] = "\xa5\x72", [0x1461] = "\xd4\x6f",
  [0x1462] = "\xb9\xbc", [0x1463] = "\xb0\xd0", [0x1464] = "\xf0\xf3",
  [0x1465] = "\xa8\xf6", [0x1466] = "\xca\x72", [0x1467] = "\xa4\x4d",
  [0x1468] = "\xa4\xc2", [0x1469] = "\xa6\x46", [0x146a] = "\xcd\xf0",
  [0x146b] = "\xcb\xd2", [0x146c] = "\xe4\xf3", [0x146d] = "\xb3\xcf",
  [0x146e] = "\xab\x66", [0x146f] = "\xd0\xe9", [0x1470] = "\xd0\xe6",
  [0x1471] = "\xad\xe6", [0x1472] = "\xe3\xc7", [0x1473] = "\xb6\xd1",
  [0x1474] = "\xb9\xbb", [0x1475] = "\xe1\x42", [0x1477] = "\xbe\xb0",
  [0x1479] = "\xaa\xc9", [0x147a] = "\xa4\x48", [0x147b] = "\xa4\xb1",
  [0x147c] = "\xc9\x53", [0x147d] = "\xc9\x52", [0x147e] = "\xc9\x65",
  [0x147f] = "\xc9\x68", [0x1481] = "\xa5\x51", [0x1482] = "\xdc\xba",
  [0x1483] = "\xa5\xf6", [0x1484] = "\xc9\xb7", [0x1485] = "\xca\x5f",
  [0x1486] = "\xc9\xae", [0x1487] = "\xad\xce", [0x1488] = "\xb3\xc9",
  [0x1489] = "\xa5\xea", [0x148a] = "\xa6\xee", [0x148b] = "\xa6\xf0",
  [0x148c] = "\xca\x61", [0x148d] = "\xa7\xf1", [0x148e] = "\xa7\x48",
  [0x148f] = "\xa7\x46", [0x1490] = "\xca\x63", [0x1491] = "\xa6\xef",
  [0x1492] = "\xa7\x41", [0x1493] = "\xa6\xf7", [0x1494] = "\xcb\xb8",
  [0x1495] = "\xcb\xb9", [0x1496] = "\xa8\xdd", [0x1497] = "\xcb\xba",
  [0x1498] = "\xa8\xd4", [0x1499] = "\xa8\xdc", [0x149a] = "\xa8\xdb",
  [0x149b] = "\xa8\xd9", [0x149c] = "\xbe\xab", [0x149d] = "\xcb\xb3",
  [0x149e] = "\xbb\xfa", [0x149f] = "\xcb\xc6", [0x14a0] = "\xbe\xa9",
  [0x14a1] = "\xc5\x6b", [0x14a2] = "\xc4\xd7", [0x14a3] = "\xcd\xde",
  [0x14a4] = "\xab\x5a", [0x14a5] = "\xcd\xe3", [0x14a6] = "\xcd\xe5",
  [0x14a7] = "\xab\x4d", [0x14a8] = "\xab\x53", [0x14a9] = "\xad\xc4",
  [0x14aa] = "\xad\xc5", [0x14ab] = "\xb0\xb4", [0x14ac] = "\xad\xd6",
  [0x14ad] = "\xd0\xd3", [0x14ae] = "\xb0\xbf", [0x14af] = "\xbb\x72",
  [0x14b0] = "\xad\xd8", [0x14b1] = "\xad\xda", [0x14b2] = "\xd0\xd7",
  [0x14b3] = "\xad\xbe", [0x14b4] = "\xad\xc3", [0x14b5] = "\xad\xd0",
  [0x14b6] = "\xe0\xf3", [0x14b7] = "\xb0\xb3", [0x14b8] = "\xb0\xba",
  [0x14b9] = "\xd4\x55", [0x14ba] = "\xb0\xb9", [0x14bb] = "\xb6\xcd",
  [0x14bc] = "\xdc\xbc", [0x14bd] = "\xc5\x6c", [0x14be] = "\xbe\xaa",
  [0x14bf] = "\xf5\xc0", [0x14c0] = "\xdc\xbf", [0x14c1] = "\xb9\xaf",
  [0x14c2] = "\xe4\xea", [0x14c3] = "\xb9\xb0", [0x14c4] = "\xe0\xfa",
  [0x14c5] = "\xe0\xf0", [0x14c6] = "\xb9\xad", [0x14c7] = "\xe4\xeb",
  [0x14c8] = "\xe4\xee", [0x14c9] = "\xc9\x69", [0x14ca] = "\xc9\xf8",
  [0x14cb] = "\xca\x65", [0x14cc] = "\xdc\xc2", [0x14cd] = "\xab\x5b",
  [0x14ce] = "\xef\xb6", [0x14cf] = "\xc9\xf8", [0x14d0] = "\xf7\x40",
  [0x14d1] = "\xa4\xbc", [0x14d2] = "\xb4\x53", [0x14d3] = "\xc6\x64",
  [0x14d4] = "\xef\x55", [0x14d5] = "\xf6\xd8", [0x14d6] = "\xc4\xdc",
  [0x14d8] = "\xab\x6d", [0x14d9] = "\xd0\xb1", [0x14da] = "\xb0\xcb",
  [0x14db] = "\xe0\xed", [0x14dc] = "\xa6\x67", [0x14dd] = "\xa8\xe1",
  [0x14df] = "\xab\x5e", [0x14e0] = "\xad\xbd", [0x14e1] = "\xb3\x4f",
  [0x14e2] = "\xd7\xc0", [0x14e3] = "\xc1\xb6", [0x14e4] = "\xf9\x43",
  [0x14e5] = "\xb8\xcf", [0x14e6] = "\xb8\x5b", [0x14e7] = "\xbe\xc6",
  [0x14e8] = "\xf2\xdd", [0x14e9] = "\xc3\xbd", [0x14eb] = "\xc9\xbc",
  [0x14ec] = "\xa8\xe6", [0x14ed] = "\xcb\xce", [0x14ee] = "\xb2\x53",
  [0x14f0] = "\xb6\xef", [0x14f1] = "\xad\xdf", [0x14f2] = "\xa8\xa5",
  [0x14f3] = "\xb0\x50", [0x14f4] = "\xb0\x52", [0x14f5] = "\xb0\x53",
  [0x14f6] = "\xf0\xdd", [0x14f7] = "\xdb\xd6", [0x14f8] = "\xb3\x5b",
  [0x14f9] = "\xb5\xfe", [0x14fa] = "\xb6\x46", [0x14fb] = "\xb6\x43",
  [0x14fc] = "\xb6\x40", [0x14fd] = "\xdb\xdc", [0x14fe] = "\xdb\xdf",
  [0x14ff] = "\xe0\x46", [0x1500] = "\xe0\x4c", [0x1501] = "\xe0\x47",
  [0x1502] = "\xb8\xd7", [0x1503] = "\xb8\xd9", [0x1504] = "\xe0\x4d",
  [0x1505] = "\xb8\xe1", [0x1506] = "\xb8\xe0", [0x1507] = "\xbd\xd8",
  [0x1508] = "\xeb\xa1", [0x1509] = "\xe0\x48", [0x150a] = "\xbb\xa6",
  [0x150b] = "\xbb\xa2", [0x150c] = "\xbb\xa5", [0x150d] = "\xe4\x4d",
  [0x150e] = "\xe7\xd4", [0x150f] = "\xe7\xd8", [0x1510] = "\xbd\xd3",
  [0x1511] = "\xbd\xdb", [0x1512] = "\xe7\xdc", [0x1513] = "\xbd\xd4",
  [0x1514] = "\xbd\xd9", [0x1515] = "\xbf\xdb", [0x1516] = "\xbf\xcf",
  [0x1517] = "\xeb\xa6", [0x1518] = "\xbf\xd6", [0x1519] = "\xeb\xa7",
  [0x151a] = "\xbf\xd9", [0x151b] = "\xbf\xdc", [0x151c] = "\xbf\xda",
  [0x151d] = "\xbf\xcd", [0x151e] = "\xbf\xd4", [0x151f] = "\xeb\xab",
  [0x1520] = "\xc2\xd3", [0x1521] = "\xc6\x6d", [0x1522] = "\xee\xa2",
  [0x1523] = "\xee\xb0", [0x1524] = "\xc1\xc4", [0x1525] = "\xc2\xd6",
  [0x1526] = "\xf7\x56", [0x1527] = "\xf2\xfb", [0x1528] = "\xc3\xd7",
  [0x1529] = "\xc3\xd4", [0x152a] = "\xf9\xa9", [0x152b] = "\xc4\xb8",
  [0x152c] = "\xc6\x40", [0x152d] = "\xb3\xa1", [0x152e] = "\xce\xaf",
  [0x152f] = "\xaa\xfa", [0x1530] = "\xca\x56", [0x1531] = "\xa6\xe9",
  [0x1532] = "\xa8\xc0", [0x1533] = "\xa8\xc1", [0x1534] = "\xcd\xd9",
  [0x1535] = "\xcd\xda", [0x1536] = "\xab\x40", [0x1537] = "\xb0\xa4",
  [0x1538] = "\xd0\xc2", [0x1539] = "\xd4\x46", [0x153a] = "\xdc\xa1",
  [0x153b] = "\xb3\xb5", [0x153c] = "\xb6\xab", [0x153d] = "\xb3\xb2",
  [0x153e] = "\xdc\x7e", [0x153f] = "\xb6\xaa", [0x1540] = "\xe0\xd4",
  [0x1541] = "\xee\xf5", [0x1542] = "\xca\x52", [0x1543] = "\xca\x54",
  [0x1544] = "\xf1\x4b", [0x1545] = "\xca\x51", [0x1546] = "\xe0\xa9",
  [0x1547] = "\xcb\xa8", [0x1548] = "\xcd\xd4", [0x1549] = "\xcd\xd6",
  [0x154a] = "\xaa\xf5", [0x154b] = "\xbf\xf3", [0x154c] = "\xaa\xf3",
  [0x154d] = "\xcd\xd7", [0x154e] = "\xd3\xf7", [0x154f] = "\xd0\xbb",
  [0x1550] = "\xd0\xbc", [0x1551] = "\xeb\xd7", [0x1552] = "\xbb\xd8",
  [0x1553] = "\xd0\xbd", [0x1554] = "\xdc\x50", [0x1555] = "\xc5\xb0",
  [0x1556] = "\xb0\x72", [0x1557] = "\xd3\xfc", [0x1558] = "\xd3\xfb",
  [0x1559] = "\xd3\xfa", [0x155a] = "\xd7\xef", [0x155b] = "\xd7\xe8",
  [0x155c] = "\xb6\x6e", [0x155d] = "\xdc\x52", [0x155e] = "\xe4\x6b",
  [0x155f] = "\xbb\xc2", [0x1560] = "\xe4\x68", [0x1561] = "\xbe\x49",
  [0x1562] = "\xe8\x54", [0x1563] = "\xc1\xdd", [0x1564] = "\xf4\xe1",
  [0x1565] = "\xf6\x5a", [0x1566] = "\xaf\xec", [0x1567] = "\xab\xb7",
  [0x1568] = "\xe4\xf4", [0x1569] = "\xa7\x57", [0x156a] = "\xca\x6f",
  [0x156b] = "\xa8\xf0", [0x156c] = "\xd0\xf8", [0x156e] = "\xb1\xe5",
  [0x156f] = "\xbc\x45", [0x1570] = "\xad\xee", [0x1571] = "\xc0\xe8",
  [0x1572] = "\xf4\x77", [0x1574] = "\xc9\x42", [0x1575] = "\xab\xd5",
  [0x1576] = "\xd4\x4a", [0x1577] = "\xa8\x70", [0x1578] = "\xa5\xaf",
  [0x1579] = "\xaf\x63", [0x157a] = "\xdd\x57", [0x157b] = "\xca\xac",
  [0x157c] = "\xb0\xee", [0x157e] = "\xb9\xd1", [0x157f] = "\xe9\x4f",
  [0x1580] = "\xbe\xc3", [0x1581] = "\xc0\xa4", [0x1582] = "\xa6\x66",
  [0x1583] = "\xa6\x64", [0x1584] = "\xc9\xc9", [0x1585] = "\xa6\x60",
  [0x1586] = "\xc2\x52", [0x1587] = "\xc9\xc8", [0x1588] = "\xa6\x65",
  [0x1589] = "\xc3\x63", [0x158a] = "\xa7\xa6", [0x158b] = "\xa8\xc1",
  [0x158c] = "\xa9\x58", [0x158d] = "\xc3\x62", [0x158e] = "\xcb\xf5",
  [0x158f] = "\xf1\xf6", [0x1590] = "\xa9\x5c", [0x1591] = "\xcb\xfb",
  [0x1592] = "\xcc\x40", [0x1593] = "\xcb\xf4", [0x1594] = "\xcb\xf8",
  [0x1595] = "\xcb\xfc", [0x1596] = "\xd4\xc1", [0x1597] = "\xce\x53",
  [0x1598] = "\xce\x54", [0x1599] = "\xdc\xe5", [0x159a] = "\xd4\xba",
  [0x159d] = "\xab\xb2", [0x159e] = "\xab\xad", [0x159f] = "\xd1\x4c",
  [0x15a0] = "\xb6\xf6", [0x15a1] = "\xb3\xfa", [0x15a2] = "\xc0\xa5",
  [0x15a3] = "\xd1\x4d", [0x15a4] = "\xd1\x52", [0x15a5] = "\xd4\xb2",
  [0x15a7] = "\xd4\xb7", [0x15a8] = "\xb0\xf1", [0x15aa] = "\xd4\xb8",
  [0x15ab] = "\xd4\xcb", [0x15ac] = "\xd4\xb5", [0x15ad] = "\xd4\xb4",
  [0x15ae] = "\xd4\xb3", [0x15af] = "\xd8\xa2", [0x15b0] = "\xd8\xa1",
  [0x15b2] = "\xb3\xfb", [0x15b3] = "\xdc\xed", [0x15b5] = "\xe1\x5b",
  [0x15b6] = "\xe1\x56", [0x15b8] = "\xbc\x55", [0x15b9] = "\xc4\xc9",
  [0x15ba] = "\xc5\x67", [0x15bb] = "\xc5\x74", [0x15bc] = "\xca\x4a",
  [0x15bd] = "\xca\x4d", [0x15be] = "\xca\x4e", [0x15bf] = "\xcb\x7a",
  [0x15c0] = "\xcb\xa1", [0x15c1] = "\xcd\xbf", [0x15c2] = "\xcb\xa3",
  [0x15c3] = "\xcb\x7c", [0x15c4] = "\xcb\x7d", [0x15c5] = "\xeb\x52",
  [0x15c6] = "\xaa\xdc", [0x15c7] = "\xcd\xb4", [0x15c8] = "\xaa\xe5",
  [0x15c9] = "\xaa\xe8", [0x15ca] = "\xaa\xe7", [0x15cb] = "\xf4\xbe",
  [0x15cd] = "\xad\x55", [0x15ce] = "\xcd\xb6", [0x15cf] = "\xaa\xe9",
  [0x15d0] = "\xcd\xba", [0x15d1] = "\xb3\x41", [0x15d2] = "\xb5\xe7",
  [0x15d3] = "\xe7\x76", [0x15d4] = "\xcd\xc1", [0x15d5] = "\xcd\xbe",
  [0x15d6] = "\xcd\xc0", [0x15d8] = "\xaa\xdf", [0x15da] = "\xad\x52",
  [0x15db] = "\xcd\xc3", [0x15dc] = "\xd0\x77", [0x15dd] = "\xad\x5b",
  [0x15df] = "\xd0\xa1", [0x15e0] = "\xf4\xb8", [0x15e1] = "\xd0\x6e",
  [0x15e2] = "\xad\x60", [0x15e3] = "\xd0\x75", [0x15e4] = "\xad\x5c",
  [0x15e6] = "\xd0\x7a", [0x15e7] = "\xd0\x7b", [0x15e8] = "\xad\x64",
  [0x15e9] = "\xe7\x72", [0x15eb] = "\xad\x67", [0x15ec] = "\xb6\xf7",
  [0x15ed] = "\xde\x72", [0x15ee] = "\xd0\xa2", [0x15ef] = "\xd0\x71",
  [0x15f0] = "\xd3\x7d", [0x15f1] = "\xd3\x77", [0x15f2] = "\xea\xfe",
  [0x15f3] = "\xe7\x6e", [0x15f4] = "\xd3\xa5", [0x15f5] = "\xb2\xf7",
  [0x15f6] = "\xd3\xa6", [0x15f7] = "\xaf\xf5", [0x15f8] = "\xaf\xfc",
  [0x15f9] = "\xd7\x6c", [0x15fa] = "\xeb\x4c", [0x15fb] = "\xd3\xac",
  [0x15fc] = "\xaf\xf6", [0x15fd] = "\xd3\xae", [0x15fe] = "\xaf\xfe",
  [0x15ff] = "\xee\x50", [0x1600] = "\xaf\xfb", [0x1601] = "\xaf\xfa",
  [0x1602] = "\xc2\xc9", [0x1603] = "\xd3\x74", [0x1604] = "\xd0\x72",
  [0x1605] = "\xd3\x73", [0x1606] = "\xba\xba", [0x1607] = "\xe2\x6c",
  [0x1608] = "\xeb\x43", [0x1609] = "\xd3\xa3", [0x160a] = "\xf0\xb0",
  [0x160b] = "\xf2\xcc", [0x160c] = "\xbb\x5e", [0x160d] = "\xdf\xd2",
  [0x160f] = "\xd7\x74", [0x1610] = "\xb2\xf2", [0x1611] = "\xe3\xc2",
  [0x1612] = "\xb8\xb1", [0x1613] = "\xb2\xfb", [0x1614] = "\xd7\x6d",
  [0x1615] = "\xb2\xf9", [0x1617] = "\xbb\x59", [0x1618] = "\xb2\xfe",
  [0x1619] = "\xee\x56", [0x161a] = "\xd7\x67", [0x161b] = "\xd7\x68",
  [0x161c] = "\xeb\x4f", [0x161d] = "\xb2\xfd", [0x161e] = "\xb2\xf1",
  [0x161f] = "\xb2\xf0", [0x1620] = "\xd7\x5b", [0x1621] = "\xc5\x61",
  [0x1622] = "\xe7\x5d", [0x1623] = "\xb5\xd7", [0x1624] = "\xdb\x6d",
  [0x1625] = "\xdb\x6f", [0x1626] = "\xdb\x70", [0x1627] = "\xd4\xbb",
  [0x1629] = "\xb5\xd6", [0x162a] = "\xdb\x6e", [0x162b] = "\xb5\xe0",
  [0x162c] = "\xdb\x75", [0x162e] = "\xb5\xe3", [0x162f] = "\xdb\x7b",
  [0x1630] = "\xdb\x7c", [0x1631] = "\xb5\xe8", [0x1632] = "\xb5\xe9",
  [0x1633] = "\xdb\x79", [0x1634] = "\xb5\xd1", [0x1635] = "\xb5\xd2",
  [0x1636] = "\xdb\x5e", [0x1637] = "\xdb\xa5", [0x1638] = "\xb5\xd5",
  [0x1639] = "\xdb\x60", [0x163a] = "\xbf\xa2", [0x163b] = "\xb5\xdd",
  [0x163c] = "\xdb\x72", [0x163e] = "\xdf\xb1", [0x163f] = "\xdf\xb8",
  [0x1640] = "\xdf\xb9", [0x1641] = "\xdf\xbb", [0x1642] = "\xeb\x41",
  [0x1644] = "\xdf\xc2", [0x1645] = "\xeb\x44", [0x1646] = "\xdf\xc4",
  [0x1647] = "\xb8\xb0", [0x1648] = "\xb8\xb6", [0x1649] = "\xb8\xb4",
  [0x164a] = "\xdf\xac", [0x164b] = "\xe7\x6f", [0x164c] = "\xdf\xd5",
  [0x164d] = "\xb8\xa9", [0x164e] = "\xb8\xb5", [0x164f] = "\xe3\xb8",
  [0x1650] = "\xe3\xb9", [0x1651] = "\xe3\xbd", [0x1652] = "\xc5\x5a",
  [0x1654] = "\xbb\x5f", [0x1655] = "\xbb\x63", [0x1656] = "\xbb\x55",
  [0x1657] = "\xe3\xb0", [0x1658] = "\xf7\xf0", [0x1659] = "\xe3\xae",
  [0x165a] = "\xe3\xb6", [0x165b] = "\xe3\xb7", [0x165c] = "\xe3\xac",
  [0x165d] = "\xf1\x6a", [0x165e] = "\xee\x40", [0x165f] = "\xe7\x5c",
  [0x1660] = "\xbf\x5f", [0x1662] = "\xe7\x75", [0x1663] = "\xf5\xfd",
  [0x1664] = "\xe7\x54", [0x1665] = "\xc4\xa9", [0x1666] = "\xea\xf8",
  [0x1667] = "\xe7\x4d", [0x1668] = "\xbd\xb6", [0x1669] = "\xe7\x64",
  [0x166a] = "\xbf\xb7", [0x166b] = "\xbf\xb8", [0x166c] = "\xbf\xb9",
  [0x166d] = "\xeb\x42", [0x166e] = "\xbf\xbf", [0x166f] = "\xee\x44",
  [0x1670] = "\xea\x7b", [0x1671] = "\xbf\xbb", [0x1672] = "\xf4\xbf",
  [0x1673] = "\xed\xfb", [0x1674] = "\xed\xfc", [0x1675] = "\xc1\xa9",
  [0x1676] = "\xc1\xa8", [0x1677] = "\xed\xf5", [0x1678] = "\xee\x52",
  [0x1679] = "\xc3\xc1", [0x167a] = "\xc1\xa3", [0x167b] = "\xee\x51",
  [0x167c] = "\xc2\xca", [0x167d] = "\xf0\xb8", [0x167e] = "\xc2\xc8",
  [0x167f] = "\xc4\xf6", [0x1681] = "\xf2\xd4", [0x1682] = "\xf4\xb9",
  [0x1683] = "\xf5\xf8", [0x1684] = "\xf4\xc1", [0x1685] = "\xf5\xfc",
  [0x1686] = "\xc4\xf4", [0x1687] = "\xf7\xf3", [0x1688] = "\xa4\x7b",
  [0x1689] = "\xab\xd9", [0x168a] = "\xc9\xcb", [0x168b] = "\xb9\xdd",
  [0x168c] = "\xd0\x55", [0x168d] = "\xab\xb3", [0x168e] = "\xae\x4f",
  [0x168f] = "\xae\x4e", [0x1690] = "\xb0\xcc", [0x1691] = "\xa4\x71",
  [0x1692] = "\xc9\xd5", [0x1693] = "\xa7\xbc", [0x1694] = "\xc0\xaa",
  [0x1695] = "\xa4\xe2", [0x1696] = "\xb1\xbf", [0x1697] = "\xe1\xde",
  [0x1698] = "\xcc\xbd", [0x1699] = "\xcc\xb9", [0x169a] = "\xa9\xe9",
  [0x169b] = "\xa9\xed", [0x169c] = "\xab\xfa", [0x169d] = "\xe5\xb9",
  [0x169e] = "\xce\xdf", [0x169f] = "\xd1\xba", [0x16a0] = "\xd1\xbb",
  [0x16a1] = "\xd1\xb8", [0x16a2] = "\xd5\x6d", [0x16a3] = "\xd9\x5e",
  [0x16a4] = "\xb1\xba", [0x16a5] = "\xb1\xcc", [0x16a6] = "\xd5\x71",
  [0x16a7] = "\xba\x51", [0x16a8] = "\xd5\x79", [0x16a9] = "\xb1\xc5",
  [0x16aa] = "\xd5\x68", [0x16ab] = "\xb1\xca", [0x16ac] = "\xd5\x7a",
  [0x16ad] = "\xe1\xe8", [0x16ae] = "\xd9\x60", [0x16b0] = "\xd9\x5d",
  [0x16b1] = "\xbc\xc3", [0x16b2] = "\xd9\x67", [0x16b4] = "\xd9\x55",
  [0x16b5] = "\xb4\xac", [0x16b6] = "\xb4\x7d", [0x16b7] = "\xd9\x64",
  [0x16b8] = "\xef\xd2", [0x16b9] = "\xdd\xae", [0x16ba] = "\xdd\xb6",
  [0x16bb] = "\xdd\xb8", [0x16bc] = "\xdd\x7d", [0x16bd] = "\xdd\xaa",
  [0x16be] = "\xdd\xab", [0x16bf] = "\xdd\xba", [0x16c0] = "\xe1\xe7",
  [0x16c1] = "\xf4\x4c", [0x16c2] = "\xba\x54", [0x16c3] = "\xe5\xaf",
  [0x16c4] = "\xba\x50", [0x16c5] = "\xc2\x5e", [0x16c6] = "\xef\xd7",
  [0x16c7] = "\xbc\xc1", [0x16c8] = "\xf5\xcd", [0x16c9] = "\xdd\xa3",
  [0x16ca] = "\xe9\xa6", [0x16cb] = "\xe9\xa4", [0x16cc] = "\xec\xec",
  [0x16cd] = "\xc0\xc2", [0x16ce] = "\xf2\x45", [0x16cf] = "\xf7\xdd",
  [0x16d0] = "\xf8\xef", [0x16d1] = "\xa4\x7c", [0x16d2] = "\xca\xd6",
  [0x16d4] = "\xb7\x49", [0x16d6] = "\xa5\x77", [0x16d7] = "\xbc\x54",
  [0x16d8] = "\xa5\x6e", [0x16d9] = "\xa5\x6f", [0x16da] = "\xa5\x7b",
  [0x16db] = "\xa6\x5c", [0x16dd] = "\xa6\x5b", [0x16df] = "\xe5\x46",
  [0x16e0] = "\xc5\x6d", [0x16e1] = "\xca\x79", [0x16e2] = "\xf1\xf2",
  [0x16e3] = "\xa7\x63", [0x16e5] = "\xd0\xf9", [0x16e6] = "\xce\x4a",
  [0x16e9] = "\xcb\xdf", [0x16eb] = "\xa9\x45", [0x16ec] = "\xa9\x4b",
  [0x16ed] = "\xcb\xef", [0x16ee] = "\xa9\x4e", [0x16ef] = "\xc0\x7b",
  [0x16f0] = "\xa9\x46", [0x16f1] = "\xa9\x4c", [0x16f2] = "\xcb\xe7",
  [0x16f5] = "\xcd\xfc", [0x16f6] = "\xab\x7b", [0x16f8] = "\xe8\xfd",
  [0x16f9] = "\xab\xa8", [0x16fa] = "\xab\x78", [0x16fb] = "\xe4\xf9",
  [0x16fc] = "\xb9\xcd", [0x16fd] = "\xce\x4d", [0x16ff] = "\xe9\x41",
  [0x1700] = "\xab\xa6", [0x1701] = "\xab\xa9", [0x1702] = "\xa9\x4b",
  [0x1703] = "\xe9\x44", [0x1705] = "\xec\xc1", [0x1706] = "\xab\xa7",
  [0x1707] = "\xab\x7d", [0x1709] = "\xbe\xba", [0x170a] = "\xcd\xfe",
  [0x170b] = "\xce\x4e", [0x170c] = "\xe1\x4d", [0x170d] = "\xd0\xf4",
  [0x170e] = "\xbc\x47", [0x170f] = "\xae\x43", [0x1710] = "\xad\xf8",
  [0x1711] = "\xd0\xf5", [0x1712] = "\xdc\xd4", [0x1714] = "\xae\x44",
  [0x1715] = "\xd0\xfc", [0x1716] = "\xae\x41", [0x1717] = "\xd4\x77",
  [0x1718] = "\xb9\xc9", [0x1719] = "\xd8\x6f", [0x171a] = "\xd8\x70",
  [0x171c] = "\xc4\xda", [0x171d] = "\xb0\xe7", [0x171e] = "\xb0\xde",
  [0x1720] = "\xd4\x75", [0x1721] = "\xd4\x73", [0x1722] = "\xad\xf2",
  [0x1723] = "\xb0\xdc", [0x1724] = "\xd4\x71", [0x1725] = "\xd4\x72",
  [0x1726] = "\xd8\x6a", [0x1727] = "\xb0\xe6", [0x1728] = "\xd4\x7c",
  [0x1729] = "\xb0\xe3", [0x172a] = "\xb3\xe3", [0x172b] = "\xdc\xd6",
  [0x172c] = "\xb3\xe4", [0x172d] = "\xb3\xed", [0x172f] = "\xd8\x6e",
  [0x1730] = "\xd8\x71", [0x1731] = "\xb3\xe7", [0x1732] = "\xb3\xee",
  [0x1733] = "\xdc\xdb", [0x1734] = "\xd8\x66", [0x1735] = "\xb3\xd7",
  [0x1736] = "\xb6\xd8", [0x1737] = "\xb9\xc6", [0x1738] = "\xf4\x41",
  [0x1739] = "\xb3\xe1", [0x173a] = "\xb3\xf1", [0x173c] = "\xb9\xc8",
  [0x173d] = "\xb6\xe8", [0x173e] = "\xb9\xca", [0x173f] = "\xb6\xdf",
  [0x1740] = "\xc4\xd9", [0x1741] = "\xa8\xfe", [0x1742] = "\xdc\xd2",
  [0x1743] = "\xb6\xdb", [0x1744] = "\xdc\xd0", [0x1745] = "\xdc\xd3",
  [0x1746] = "\xb6\xe2", [0x1747] = "\xb6\xe7", [0x1748] = "\xdc\xdd",
  [0x1749] = "\xbe\xbc", [0x174a] = "\xdc\xcc", [0x174b] = "\xdc\xde",
  [0x174c] = "\xb6\xd9", [0x174e] = "\xb6\xe1", [0x174f] = "\xc5\xaf",
  [0x1751] = "\xb9\xcb", [0x1752] = "\xe1\x49", [0x1753] = "\xe1\x4e",
  [0x1754] = "\xc4\x58", [0x1756] = "\xb9\xbe", [0x1757] = "\xb9\xbf",
  [0x1758] = "\xe1\x47", [0x175a] = "\xe4\xfe", [0x175b] = "\xbc\x49",
  [0x175c] = "\xbc\x50", [0x175d] = "\xe5\x44", [0x175f] = "\xbe\xbe",
  [0x1760] = "\xbe\xb2", [0x1761] = "\xc2\x50", [0x1762] = "\xe4\xf8",
  [0x1764] = "\xec\xc3", [0x1765] = "\xbe\xb6", [0x1766] = "\xbe\xbb",
  [0x1767] = "\xbe\xb3", [0x176a] = "\xc0\x7d", [0x176b] = "\xec\xc0",
  [0x176d] = "\xf8\xee", [0x176e] = "\xc9\x49", [0x176f] = "\xa6\x5f",
  [0x1770] = "\xc9\xc6", [0x1771] = "\xd4\xac", [0x1772] = "\xa7\x7a",
  [0x1773] = "\xcb\xf1", [0x1774] = "\xab\xaa", [0x1775] = "\xae\x46",
  [0x1776] = "\xd4\xab", [0x1777] = "\xb0\xeb", [0x1778] = "\xe9\x47",
  [0x1779] = "\xb4\x57", [0x177a] = "\xcc\x6f", [0x177b] = "\xcc\x6e",
  [0x177c] = "\xa9\xae", [0x177d] = "\xec\xdb", [0x177e] = "\xe1\xb2",
  [0x177f] = "\xb9\xf6", [0x1780] = "\xb1\x63", [0x1781] = "\xd8\xf2",
  [0x1782] = "\xb9\xf7", [0x1783] = "\xb9\xf3", [0x1785] = "\xbc\x6e",
  [0x1786] = "\xa7\xc4", [0x1787] = "\xc9\xd7", [0x1788] = "\xca\xc0",
  [0x1789] = "\xa7\xc1", [0x178a] = "\xb9\xf2", [0x178b] = "\xca\xc2",
  [0x178c] = "\xae\x73", [0x178e] = "\xa7\xc2", [0x178f] = "\xb4\x50",
  [0x1791] = "\xcc\x5e", [0x1792] = "\xcc\x65", [0x1793] = "\xd4\xf8",
  [0x1794] = "\xcc\x61", [0x1795] = "\xa9\xa6", [0x1796] = "\xa9\xa7",
  [0x1797] = "\xcc\x63", [0x1799] = "\xa9\xa2", [0x179a] = "\xe9\x61",
  [0x179b] = "\xab\xd0", [0x179c] = "\xe5\x69", [0x179d] = "\xce\x7d",
  [0x179e] = "\xb1\x57", [0x179f] = "\xe5\x63", [0x17a0] = "\xd4\xfb",
  [0x17a1] = "\xb1\x5d", [0x17a2] = "\xd4\xfc", [0x17a3] = "\xd5\x44",
  [0x17a4] = "\xb1\x5c", [0x17a5] = "\xd4\xf4", [0x17a6] = "\xb1\x53",
  [0x17a7] = "\xb1\x55", [0x17a8] = "\xc0\xaf", [0x17aa] = "\xb4\x51",
  [0x17ab] = "\xd8\xe6", [0x17ac] = "\xdd\x52", [0x17ad] = "\xd8\xe3",
  [0x17ae] = "\xb7\x44", [0x17af] = "\xe1\xa8", [0x17b0] = "\xd8\xdb",
  [0x17b1] = "\xd8\xdd", [0x17b2] = "\xdd\x4f", [0x17b3] = "\xb7\x43",
  [0x17b5] = "\xe1\xa3", [0x17b6] = "\xe5\x62", [0x17b7] = "\xbc\x6a",
  [0x17b8] = "\xc1\xc7", [0x17b9] = "\xec\xda", [0x17ba] = "\xc5\x71",
  [0x17bb] = "\xc9\x4c", [0x17bc] = "\xa7\xcf", [0x17bd] = "\xcc\x75",
  [0x17be] = "\xab\xe0", [0x17bf] = "\xab\xe2", [0x17c0] = "\xab\xe1",
  [0x17c1] = "\xb1\x74", [0x17c2] = "\xb1\x70", [0x17c3] = "\xb1\x75",
  [0x17c4] = "\xb4\x61", [0x17c5] = "\xdd\x64", [0x17c6] = "\xbc\x78",
  [0x17c7] = "\xe9\x75", [0x17c8] = "\xc5\xfc", [0x17ca] = "\xa4\xfc",
  [0x17cb] = "\xc9\xa6", [0x17cc] = "\xca\x43", [0x17cd] = "\xc2\x78",
  [0x17cf] = "\xcb\x63", [0x17d0] = "\xcb\x65", [0x17d1] = "\xaa\xad",
  [0x17d3] = "\xcd\x61", [0x17d4] = "\xcf\xc0", [0x17d5] = "\xea\x61",
  [0x17d6] = "\xac\xbc", [0x17d7] = "\xde\xac", [0x17d8] = "\xd2\xa5",
  [0x17d9] = "\xaf\x58", [0x17da] = "\xd2\xaa", [0x17db] = "\xd2\xa8",
  [0x17dc] = "\xea\x60", [0x17dd] = "\xd2\xa4", [0x17de] = "\xd2\xa9",
  [0x17df] = "\xd6\x61", [0x17e0] = "\xb2\x74", [0x17e1] = "\xc5\x7a",
  [0x17e2] = "\xd6\x65", [0x17e3] = "\xd6\x69", [0x17e4] = "\xd6\x60",
  [0x17e5] = "\xf4\x6e", [0x17e6] = "\xda\x70", [0x17e8] = "\xb5\x54",
  [0x17e9] = "\xe7\xb3", [0x17eb] = "\xda\x71", [0x17ec] = "\xba\xbc",
  [0x17ed] = "\xe2\xcb", [0x17ee] = "\xbc\xfb", [0x17ef] = "\xe6\x79",
  [0x17f0] = "\xea\x5e", [0x17f1] = "\xed\x65", [0x17f2] = "\xf5\xdd",
  [0x17f3] = "\xa6\xdf", [0x17f4] = "\xb9\xd9", [0x17f5] = "\xb6\xb8",
  [0x17f6] = "\xb9\xdb", [0x17f8] = "\xad\xb9", [0x17f9] = "\xef\x4f",
  [0x17fa] = "\xb6\xbb", [0x17fb] = "\xf1\xad", [0x17fc] = "\xb6\xb9",
  [0x17fd] = "\xdc\xae", [0x17fe] = "\xb6\xbd", [0x17ff] = "\xb9\x7e",
  [0x1800] = "\xbb\xe8", [0x1801] = "\xe8\xc1", [0x1802] = "\xbe\x6c",
  [0x1803] = "\xc0\x5f", [0x1805] = "\xc3\x49", [0x1806] = "\xc1\x55",
  [0x1807] = "\xf3\xaf", [0x1808] = "\xc4\x44", [0x180a] = "\xf5\x57",
  [0x180c] = "\xc9\x79", [0x180d] = "\xe5\x75", [0x180e] = "\xca\xcd",
  [0x180f] = "\xa9\xb4", [0x1810] = "\xce\xba", [0x1811] = "\xab\xd6",
  [0x1812] = "\xd5\x49", [0x1813] = "\xb1\x67", [0x1814] = "\xb1\x68",
  [0x1815] = "\xd5\x4b", [0x1816] = "\xbd\xe9", [0x1817] = "\xe1\xbc",
  [0x1818] = "\xe1\xb6", [0x1819] = "\xe5\x73", [0x181a] = "\xe9\x72",
  [0x181b] = "\xe9\x6f", [0x181c] = "\xc1\x74", [0x181d] = "\xa4\xdf",
  [0x181e] = "\xc9\x7b", [0x181f] = "\xa6\xa4", [0x1820] = "\xc4\x62",
  [0x1821] = "\xbc\xad", [0x1822] = "\xca\xdb", [0x1823] = "\xe1\xd3",
  [0x1824] = "\xca\xdd", [0x1825] = "\xca\xde", [0x1826] = "\xb7\x5a",
  [0x1827] = "\xb1\xa3", [0x1828] = "\xb7\x5b", [0x1829] = "\xa7\xd8",
  [0x182a] = "\xca\xd9", [0x182b] = "\xa7\xd7", [0x182c] = "\xcc\x7c",
  [0x182d] = "\xa9\xc5", [0x182e] = "\xcc\x7b", [0x182f] = "\xa9\xcd",
  [0x1830] = "\xa9\xc2", [0x1831] = "\xcc\xa9", [0x1832] = "\xa9\xcb",
  [0x1833] = "\xa9\xcc", [0x1834] = "\xcc\xa2", [0x1835] = "\xe9\x7d",
  [0x1836] = "\xa9\xc9", [0x1837] = "\xba\x45", [0x1838] = "\xec\xe6",
  [0x1839] = "\xb4\x6c", [0x183a] = "\xb7\x5f", [0x183b] = "\xce\xcb",
  [0x183c] = "\xab\xf1", [0x183d] = "\xd9\x40", [0x183e] = "\xae\xaf",
  [0x183f] = "\xae\xaa", [0x1840] = "\xe1\xc7", [0x1841] = "\xd1\xa7",
  [0x1842] = "\xd1\xa8", [0x1843] = "\xd1\xa5", [0x1844] = "\xae\xad",
  [0x1845] = "\xd1\xaa", [0x1846] = "\xb4\x67", [0x1847] = "\xb1\xa2",
  [0x1848] = "\xd5\x5e", [0x1849] = "\xd5\x5c", [0x184a] = "\xb1\xa6",
  [0x184b] = "\xb1\xa8", [0x184c] = "\xb1\xab", [0x184d] = "\xb1\x7c",
  [0x184e] = "\xb7\x59", [0x184f] = "\xe5\xa6", [0x1850] = "\xb4\x6a",
  [0x1851] = "\xb4\x68", [0x1852] = "\xb4\x6d", [0x1853] = "\xb4\x73",
  [0x1854] = "\xb4\x70", [0x1855] = "\xdd\x68", [0x1856] = "\xdd\x67",
  [0x1857] = "\xba\x48", [0x1858] = "\xbc\xa9", [0x1859] = "\xbc\xac",
  [0x185a] = "\xbc\xa5", [0x185c] = "\xbe\xcf", [0x185d] = "\xc3\x6a",
  [0x185e] = "\xa9\xbd", [0x185f] = "\xf1\x6f", [0x1860] = "\xad\xac",
  [0x1861] = "\xd8\x45", [0x1862] = "\xc1\xf2", [0x1863] = "\xb6\xa3",
  [0x1864] = "\xb6\x7b", [0x1865] = "\xdc\x7a", [0x1866] = "\xf6\x6b",
  [0x1867] = "\xbe\x5b", [0x1868] = "\xe8\xa7", [0x1869] = "\xf9\xc4",
  [0x186a] = "\xe8\xa6", [0x186b] = "\xec\x48", [0x186c] = "\xec\x4b",
  [0x186d] = "\xf1\xbe", [0x186e] = "\xec\x4c", [0x186f] = "\xec\x4e",
  [0x1870] = "\xec\x46", [0x1871] = "\xee\xf1", [0x1872] = "\xc1\xf0",
  [0x1873] = "\xc2\xf3", [0x1874] = "\xc2\xf5", [0x1875] = "\xc2\xf6",
  [0x1876] = "\xf5\x45", [0x1877] = "\xc9\x62", [0x1878] = "\xc9\x62",
  [0x1879] = "\xa9\xcf", [0x187a] = "\xa4\xf4", [0x187b] = "\xc9\xfd",
  [0x187c] = "\xc9\xfa", [0x187d] = "\xc9\xfc", [0x187e] = "\xf5\xd3",
  [0x187f] = "\xa8\x4a", [0x1880] = "\xa8\x4e", [0x1881] = "\xa8\x5c",
  [0x1882] = "\xa8\x50", [0x1883] = "\xa8\x51", [0x1884] = "\xcb\x53",
  [0x1885] = "\xa8\x58", [0x1886] = "\xa8\x5a", [0x1887] = "\xa8\x59",
  [0x1888] = "\xd9\xec", [0x1889] = "\xcc\xf8", [0x188a] = "\xcc\xf4",
  [0x188b] = "\xcc\xf5", [0x188c] = "\xf2\x5d", [0x188d] = "\xc3\x79",
  [0x188e] = "\xaa\x74", [0x188f] = "\xaa\x72", [0x1890] = "\xaa\x62",
  [0x1891] = "\xaa\x7e", [0x1892] = "\xaa\x7d", [0x1893] = "\xef\xf6",
  [0x1894] = "\xcc\xf0", [0x1895] = "\xcc\xf1", [0x1896] = "\xaa\x62",
  [0x1897] = "\xaa\x6c", [0x1898] = "\xaa\x7b", [0x1899] = "\xae\xf9",
  [0x189a] = "\xac\xa7", [0x189b] = "\xac\xa8", [0x189c] = "\xac\x7b",
  [0x189d] = "\xaf\x48", [0x189e] = "\xd9\xfe", [0x189f] = "\xcf\xa7",
  [0x18a0] = "\xcf\x77", [0x18a1] = "\xcf\x78", [0x18a2] = "\xac\xad",
  [0x18a3] = "\xac\xae", [0x18a4] = "\xe9\xf8", [0x18a5] = "\xac\xab",
  [0x18a6] = "\xac\xac", [0x18a7] = "\xcf\x7a", [0x18a8] = "\xc2\x73",
  [0x18a9] = "\xe2\x71", [0x18aa] = "\xbc\xf1", [0x18ab] = "\xcf\x76",
  [0x18ac] = "\xd2\x50", [0x18ad] = "\xd2\x4f", [0x18ae] = "\xd5\xe9",
  [0x18af] = "\xe6\x5c", [0x18b0] = "\xd2\x55", [0x18b1] = "\xae\xfe",
  [0x18b2] = "\xaf\x4b", [0x18b3] = "\xc0\xd8", [0x18b4] = "\xd2\x57",
  [0x18b5] = "\xd2\x5a", [0x18b6] = "\xd2\x46", [0x18b7] = "\xb2\x5a",
  [0x18b8] = "\xb2\x4e", [0x18b9] = "\xb2\x58", [0x18ba] = "\xb2\x53",
  [0x18bb] = "\xc2\x70", [0x18bc] = "\xb2\x67", [0x18bd] = "\xd5\xed",
  [0x18be] = "\xbf\x49", [0x18bf] = "\xb2\x68", [0x18c0] = "\xd5\xf3",
  [0x18c1] = "\xb2\x46", [0x18c2] = "\xc2\x6e", [0x18c3] = "\xd5\xe1",
  [0x18c4] = "\xd5\xf1", [0x18c5] = "\xb2\x52", [0x18c6] = "\xd9\xf1",
  [0x18c7] = "\xb4\xf3", [0x18c8] = "\xb5\x42", [0x18c9] = "\xda\x47",
  [0x18ca] = "\xde\x59", [0x18cb] = "\xb5\x47", [0x18cc] = "\xe2\xa9",
  [0x18cd] = "\xda\x4b", [0x18ce] = "\xb4\xe6", [0x18cf] = "\xb4\xe8",
  [0x18d0] = "\xb4\xec", [0x18d1] = "\xb5\x44", [0x18d2] = "\xf9\xd0",
  [0x18d3] = "\xde\x4c", [0x18d4] = "\xb7\xc2", [0x18d5] = "\xf5\xd4",
  [0x18d6] = "\xe2\xa5", [0x18d7] = "\xef\xef", [0x18d8] = "\xb7\xc1",
  [0x18d9] = "\xb7\xcc", [0x18da] = "\xde\x50", [0x18dc] = "\xde\x54",
  [0x18dd] = "\xe6\x5f", [0x18de] = "\xb7\xcd", [0x18df] = "\xde\x58",
  [0x18e0] = "\xde\x46", [0x18e1] = "\xb7\xbc", [0x18e2] = "\xde\x48",
  [0x18e3] = "\xe6\x43", [0x18e4] = "\xf2\x5e", [0x18e5] = "\xc3\x74",
  [0x18e7] = "\xba\xa8", [0x18e8] = "\xe2\xa3", [0x18e9] = "\xba\xaa",
  [0x18ea] = "\xe2\xa1", [0x18eb] = "\xc4\x6b", [0x18ec] = "\xef\xf5",
  [0x18ed] = "\xba\xac", [0x18ee] = "\xe2\x75", [0x18ef] = "\xba\x78",
  [0x18f0] = "\xe6\x41", [0x18f1] = "\xe6\x40", [0x18f2] = "\xe6\x42",
  [0x18f3] = "\xbc\xe8", [0x18f4] = "\xe6\x4c", [0x18f5] = "\xbc\xe0",
  [0x18f6] = "\xbc\xeb", [0x18f7] = "\xc3\x75", [0x18f8] = "\xea\x41",
  [0x18f9] = "\xbf\x43", [0x18fa] = "\xbf\x46", [0x18fb] = "\xbf\x47",
  [0x18fc] = "\xbe\xfc", [0x18fd] = "\xc0\xe1", [0x18fe] = "\xc0\xe4",
  [0x18ff] = "\xed\x57", [0x1900] = "\xc0\xda", [0x1901] = "\xc0\xde",
  [0x1902] = "\xc3\x76", [0x1903] = "\xf2\x63", [0x1904] = "\xc3\x73",
  [0x1905] = "\xf4\x65", [0x1906] = "\xf4\x5c", [0x1907] = "\xf8\xaf",
  [0x1908] = "\xc5\xf4", [0x190a] = "\xc9\x73", [0x190b] = "\xcc\x58",
  [0x190c] = "\xcc\x57", [0x190d] = "\xab\xc9", [0x190e] = "\xae\x66",
  [0x190f] = "\xda\xac", [0x1910] = "\xc4\xca", [0x1911] = "\xba\x53",
  [0x1912] = "\xb9\xed", [0x1913] = "\xbc\x64", [0x1914] = "\xeb\x72",
  [0x1915] = "\xbe\xc8", [0x1916] = "\xee\xbf", [0x1917] = "\xee\xaa",
  [0x1919] = "\xcd\xce", [0x191a] = "\xcd\xd1", [0x191b] = "\xad\x7e",
  [0x191c] = "\xd0\xb5", [0x191d] = "\xad\xa3", [0x191e] = "\xc2\xe2",
  [0x191f] = "\xad\x7b", [0x1920] = "\xb3\x77", [0x1921] = "\xad\xa4",
  [0x1922] = "\xb0\x6d", [0x1923] = "\xd3\xef", [0x1924] = "\xd7\xe3",
  [0x1925] = "\xc5\xdd", [0x1926] = "\xd7\xe4", [0x1927] = "\xb3\x70",
  [0x1928] = "\xb3\x7c", [0x1929] = "\xd7\xe7", [0x192a] = "\xb6\x66",
  [0x192b] = "\xb6\x6a", [0x192c] = "\xdc\x4d", [0x192d] = "\xdc\x4f",
  [0x192e] = "\xe0\xa1", [0x192f] = "\xb9\x4e", [0x1930] = "\xe0\x7d",
  [0x1931] = "\xb9\x49", [0x1932] = "\xbe\x43", [0x1933] = "\xbb\xb8",
  [0x1934] = "\xbb\xbd", [0x1935] = "\xbb\xbf", [0x1936] = "\xbe\xe6",
  [0x1937] = "\xbf\xee", [0x1938] = "\xc1\xd8", [0x1939] = "\xc1\xdb",
  [0x193a] = "\xc2\xe4", [0x193b] = "\xc2\xe3", [0x193c] = "\xc3\xe5",
  [0x193e] = "\xb1\x6b", [0x193f] = "\xce\xbd", [0x1940] = "\xd8\xf4",
  [0x1941] = "\xc9\x75", [0x1942] = "\xab\xa4", [0x1943] = "\xae\x6a",
  [0x1944] = "\xd4\xf3", [0x1945] = "\xb4\x47", [0x1946] = "\xe1\xa2",
  [0x1947] = "\xc0\xab", [0x1948] = "\xc4\xf3", [0x1949] = "\xd1\x79",
  [0x194a] = "\xa9\xb8", [0x194b] = "\xab\xda", [0x194c] = "\xd7\x58",
  [0x194d] = "\xb4\x5d", [0x194e] = "\xf7\xa7", [0x194f] = "\xc9\x4b",
  [0x1950] = "\xa6\x71", [0x1951] = "\xa6\x6d", [0x1952] = "\xa7\xb0",
  [0x1953] = "\xbc\x61", [0x1954] = "\xb9\xe1", [0x1955] = "\xa7\xad",
  [0x1956] = "\xca\xb5", [0x1957] = "\xa9\x6e", [0x1958] = "\xd8\xb5",
  [0x1959] = "\xa7\xac", [0x195a] = "\xa7\xb1", [0x195b] = "\xa9\x71",
  [0x195c] = "\xcc\x48", [0x195d] = "\xa9\x6f", [0x195e] = "\xa9\x6b",
  [0x195f] = "\xa9\x63", [0x1960] = "\xd4\xd5", [0x1961] = "\xbc\x63",
  [0x1962] = "\xce\x67", [0x1963] = "\xf6\xdc", [0x1964] = "\xab\xbb",
  [0x1965] = "\xab\xb9", [0x1967] = "\xae\x5b", [0x1968] = "\xae\x5c",
  [0x1969] = "\xb4\x45", [0x196a] = "\xbc\x5f", [0x196b] = "\xae\x50",
  [0x196c] = "\xae\x58", [0x196d] = "\xae\x55", [0x196e] = "\xb0\xfe",
  [0x196f] = "\xd4\xd1", [0x1970] = "\xb1\x44", [0x1971] = "\xd4\xd0",
  [0x1972] = "\xb1\x40", [0x1973] = "\xb1\x41", [0x1974] = "\xbc\x60",
  [0x1976] = "\xb6\xfe", [0x1977] = "\xb4\x44", [0x1978] = "\xb4\x40",
  [0x1979] = "\xd8\xb8", [0x197a] = "\xb6\xfc", [0x197b] = "\xe1\x72",
  [0x197c] = "\xb7\x42", [0x197d] = "\xe9\x54", [0x197e] = "\xc0\xa7",
  [0x197f] = "\xdc\xf7", [0x1980] = "\xe1\x75", [0x1981] = "\xb9\xe4",
  [0x1982] = "\xe9\x52", [0x1983] = "\xb9\xe2", [0x1984] = "\xb9\xdf",
  [0x1985] = "\xb9\xe3", [0x1986] = "\xe1\x6b", [0x1987] = "\xbc\x5e",
  [0x1988] = "\xe9\x51", [0x1989] = "\xe9\x58", [0x198a] = "\xec\xd0",
  [0x198b] = "\xc0\xa8", [0x198c] = "\xc4\x5c", [0x198d] = "\xc9\xd4",
  [0x198e] = "\xb9\xc7", [0x198f] = "\xa7\xb7", [0x1990] = "\xcc\x56",
  [0x1991] = "\xb4\x46", [0x1992] = "\xa4\x6d", [0x1993] = "\xa4\x6e",
  [0x1994] = "\xcc\x55", [0x1995] = "\xe8\xca", [0x1996] = "\xbe\x6f",
  [0x1997] = "\xbe\x74", [0x1998] = "\xf5\x5f", [0x1999] = "\xc5\xe6",
  [0x199a] = "\xbe\x71", [0x199b] = "\xe8\xce", [0x199c] = "\xc5\xbb",
  [0x199d] = "\xf7\x7e", [0x199e] = "\xc0\x63", [0x199f] = "\xc6\x7a",
  [0x19a0] = "\xf1\xaf", [0x19a1] = "\xf1\xb2", [0x19a2] = "\xf1\xb6",
  [0x19a3] = "\xf6\x7c", [0x19a4] = "\xf5\x63", [0x19a5] = "\xc4\x45",
  [0x19a6] = "\xf6\x7d", [0x19a7] = "\xf5\x61", [0x19a8] = "\xf5\x62",
  [0x19a9] = "\xc5\x59", [0x19aa] = "\xf6\x7b", [0x19ab] = "\xf7\x7c",
  [0x19ac] = "\xc6\x6b", [0x19ad] = "\xf9\xb3", [0x19ae] = "\xb5\xb7",
  [0x19af] = "\xac\xfa", [0x19b0] = "\xac\xf4", [0x19b1] = "\xac\xf8",
  [0x19b2] = "\xd0\x4b", [0x19b3] = "\xf5\xe7", [0x19b4] = "\xaf\xc6",
  [0x19b5] = "\xaf\xc4", [0x19b6] = "\xd3\x56", [0x19b7] = "\xd6\xe6",
  [0x19b8] = "\xb2\xd8", [0x19b9] = "\xb2\xd9", [0x19ba] = "\xbf\xa8",
  [0x19bb] = "\xb2\xd1", [0x19bc] = "\xb2\xd2", [0x19bd] = "\xd6\xed",
  [0x19be] = "\xbf\xc7", [0x19bf] = "\xda\xfb", [0x19c0] = "\xb5\xbc",
  [0x19c1] = "\xdf\x58", [0x19c2] = "\xdf\x60", [0x19c3] = "\xdf\x57",
  [0x19c4] = "\xba\xf0", [0x19c5] = "\xba\xf6", [0x19c6] = "\xe3\x6e",
  [0x19c8] = "\xe3\x6c", [0x19c9] = "\xbd\x78", [0x19ca] = "\xba\xfe",
  [0x19cb] = "\xe3\x71", [0x19cc] = "\xe3\x65", [0x19cd] = "\xba\xed",
  [0x19ce] = "\xba\xfd", [0x19cf] = "\xbd\x79", [0x19d0] = "\xe6\xe7",
  [0x19d1] = "\xbd\x7b", [0x19d2] = "\xbd\x7a", [0x19d3] = "\xf0\x72",
  [0x19d4] = "\xe6\xeb", [0x19d5] = "\xe6\xec", [0x19d6] = "\xe6\xed",
  [0x19d7] = "\xea\xd8", [0x19d8] = "\xe6\xe8", [0x19d9] = "\xbf\xa7",
  [0x19da] = "\xbf\xa6", [0x19db] = "\xea\xd0", [0x19dc] = "\xbf\xa5",
  [0x19dd] = "\xed\xd0", [0x19de] = "\xbf\x7e", [0x19df] = "\xbf\xa1",
  [0x19e0] = "\xc4\x7d", [0x19e1] = "\xc1\x65", [0x19e2] = "\xc1\x67",
  [0x19e3] = "\xc1\x5d", [0x19e4] = "\xc1\x5b", [0x19e5] = "\xc1\x62",
  [0x19e6] = "\xf5\xe9", [0x19e7] = "\xc2\xb7", [0x19e8] = "\xc2\xb9",
  [0x19e9] = "\xc5\xb7", [0x19ea] = "\xf4\xad", [0x19eb] = "\xf2\xb2",
  [0x19ec] = "\xf2\xb4", [0x19ed] = "\xf9\x40", [0x19ee] = "\xa4\x5c",
  [0x19ef] = "\xbd\x42", [0x19f1] = "\xcd\x72", [0x19f2] = "\xb0\x6f",
  [0x19f3] = "\xca\x45", [0x19f4] = "\xbf\x59", [0x19f5] = "\xde\xb3",
  [0x19f6] = "\xcd\x6a", [0x19f7] = "\xaa\xb3", [0x19f8] = "\xaa\xb2",
  [0x19f9] = "\xcf\xc8", [0x19fa] = "\xc4\x6e", [0x19fb] = "\xac\xbf",
  [0x19fc] = "\xac\xc5", [0x19fd] = "\xac\xc4", [0x19fe] = "\xaa\xb3",
  [0x19ff] = "\xcf\xc9", [0x1a00] = "\xd2\xad", [0x1a01] = "\xd2\xac",
  [0x1a02] = "\xb9\x7a", [0x1a03] = "\xb2\x78", [0x1a04] = "\xd2\xb2",
  [0x1a05] = "\xd2\xb0", [0x1a06] = "\xaf\x5f", [0x1a07] = "\xc3\xa2",
  [0x1a08] = "\xb7\xe9", [0x1a09] = "\xe6\xa3", [0x1a0a] = "\xb5\x58",
  [0x1a0b] = "\xb7\xeb", [0x1a0c] = "\xb5\x61", [0x1a0d] = "\xb5\x5b",
  [0x1a0e] = "\xb5\x62", [0x1a0f] = "\xda\x7c", [0x1a10] = "\xda\x7a",
  [0x1a11] = "\xda\x7b", [0x1a12] = "\xb5\x60", [0x1a13] = "\xda\xa1",
  [0x1a14] = "\xb7\xe8", [0x1a15] = "\xb7\xec", [0x1a16] = "\xde\xb6",
  [0x1a17] = "\xb7\xe5", [0x1a18] = "\xb7\xea", [0x1a19] = "\xc0\xf5",
  [0x1a1a] = "\xba\xc1", [0x1a1b] = "\xbd\x40", [0x1a1c] = "\xbf\x58",
  [0x1a1d] = "\xc4\xed", [0x1a1e] = "\xbd\x41", [0x1a1f] = "\xe6\xa6",
  [0x1a20] = "\xe6\xa2", [0x1a21] = "\xbc\xfd", [0x1a22] = "\xbf\x5c",
  [0x1a23] = "\xc0\xf6", [0x1a24] = "\xc0\xf3", [0x1a25] = "\xed\x69",
  [0x1a26] = "\xc2\x7a", [0x1a27] = "\xc5\xd0", [0x1a28] = "\xf4\x6f",
  [0x1a29] = "\xf1\xa4", [0x1a2a] = "\xf3\xa5", [0x1a2b] = "\xc3\xfc",
  [0x1a2c] = "\xcb\x42", [0x1a2d] = "\xa8\x40", [0x1a2e] = "\xa7\xfb",
  [0x1a2f] = "\xcb\x43", [0x1a30] = "\xe2\x59", [0x1a31] = "\xf4\x55",
  [0x1a32] = "\xaa\x4a", [0x1a33] = "\xaa\x57", [0x1a34] = "\xaa\x48",
  [0x1a35] = "\xcc\xd9", [0x1a37] = "\xaa\x53", [0x1a38] = "\xd9\xb3",
  [0x1a39] = "\xbc\xd7", [0x1a3a] = "\xb1\xfa", [0x1a3b] = "\xaa\x44",
  [0x1a3c] = "\xaa\x49", [0x1a3d] = "\xaa\x56", [0x1a3e] = "\xcf\x55",
  [0x1a3f] = "\xc0\xcd", [0x1a40] = "\xcf\x43", [0x1a41] = "\xf4\x53",
  [0x1a42] = "\xac\x5e", [0x1a43] = "\xac\x69", [0x1a44] = "\xf4\x56",
  [0x1a45] = "\xac\x6a", [0x1a46] = "\xcf\x4a", [0x1a47] = "\xac\x63",
  [0x1a48] = "\xcf\x4c", [0x1a49] = "\xac\x6c", [0x1a4a] = "\xb1\xf7",
  [0x1a4b] = "\xcf\x61", [0x1a4c] = "\xac\x65", [0x1a4d] = "\xac\x6b",
  [0x1a4e] = "\xf2\x52", [0x1a4f] = "\xce\xfb", [0x1a50] = "\xec\xf7",
  [0x1a51] = "\xd1\xdc", [0x1a52] = "\xd1\xdd", [0x1a53] = "\xd9\xb2",
  [0x1a54] = "\xbe\xf8", [0x1a55] = "\xd1\xe8", [0x1a56] = "\xb7\xa9",
  [0x1a57] = "\xd1\xe9", [0x1a58] = "\xe2\x4e", [0x1a59] = "\xb1\xf2",
  [0x1a5a] = "\xd1\xeb", [0x1a5c] = "\xbe\xec", [0x1a5d] = "\xae\xec",
  [0x1a5e] = "\xc0\xcc", [0x1a5f] = "\xae\xe5", [0x1a60] = "\xf7\xe1",
  [0x1a62] = "\xd1\xdb", [0x1a63] = "\xae\xdd", [0x1a64] = "\xb1\xeb",
  [0x1a65] = "\xd5\xc0", [0x1a66] = "\xd5\xbe", [0x1a67] = "\xd5\xc1",
  [0x1a68] = "\xb1\xea", [0x1a69] = "\xd5\xc4", [0x1a6a] = "\xc4\xe7",
  [0x1a6b] = "\xb4\xd8", [0x1a6c] = "\xd9\xc8", [0x1a6d] = "\xc3\x70",
  [0x1a6e] = "\xe5\xde", [0x1a6f] = "\xb4\xcd", [0x1a70] = "\xf7\xe2",
  [0x1a71] = "\xbd\x69", [0x1a72] = "\xd9\xca", [0x1a73] = "\xbc\xcd",
  [0x1a74] = "\xdd\xd4", [0x1a75] = "\xb4\xd0", [0x1a76] = "\xd9\xb9",
  [0x1a77] = "\xdd\xce", [0x1a78] = "\xdd\xd2", [0x1a79] = "\xb7\xa3",
  [0x1a7a] = "\xdd\xd3", [0x1a7b] = "\xb7\xaf", [0x1a7c] = "\xc6\x56",
  [0x1a7d] = "\xb7\xaa", [0x1a7f] = "\xaf\x78", [0x1a80] = "\xdd\xdf",
  [0x1a81] = "\xdd\xe0", [0x1a82] = "\xba\x6c", [0x1a83] = "\xc4\x68",
  [0x1a84] = "\xc3\x71", [0x1a85] = "\xe2\x41", [0x1a86] = "\xf5\xd1",
  [0x1a87] = "\xdd\xc9", [0x1a88] = "\xb7\xb0", [0x1a89] = "\xb7\xad",
  [0x1a8a] = "\xba\x64", [0x1a8b] = "\xe2\x49", [0x1a8c] = "\xba\x66",
  [0x1a8d] = "\xba\x67", [0x1a8e] = "\xba\x6b", [0x1a8f] = "\xe2\x52",
  [0x1a90] = "\xe1\xfc", [0x1a91] = "\xba\x60", [0x1a92] = "\xe2\x54",
  [0x1a93] = "\xc2\x62", [0x1a94] = "\xba\x5f", [0x1a95] = "\xf2\x4c",
  [0x1a96] = "\xe2\x4a", [0x1a97] = "\xe5\xd3", [0x1a98] = "\xc0\xce",
  [0x1a99] = "\xbc\xd8", [0x1a9a] = "\xe5\xcc", [0x1a9b] = "\xe5\xe8",
  [0x1a9c] = "\xf2\x55", [0x1a9d] = "\xe5\xda", [0x1a9e] = "\xbe\xf1",
  [0x1a9f] = "\xe9\xc4", [0x1aa0] = "\xc0\xd2", [0x1aa1] = "\xe9\xd1",
  [0x1aa2] = "\xe9\xc8", [0x1aa3] = "\xbe\xf6", [0x1aa4] = "\xec\xfe",
  [0x1aa5] = "\xc3\x72", [0x1aa6] = "\xbe\xea", [0x1aa7] = "\xe9\xca",
  [0x1aa8] = "\xbe\xef", [0x1aa9] = "\xf2\x58", [0x1aaa] = "\xed\x44",
  [0x1aab] = "\xc3\xb2", [0x1aac] = "\xec\xf5", [0x1aad] = "\xc0\xd0",
  [0x1aaf] = "\xb7\xdf", [0x1ab0] = "\xe6\x76", [0x1ab1] = "\xaa\x5c",
  [0x1ab2] = "\xcf\x65", [0x1ab3] = "\xbc\xdc", [0x1ab4] = "\xcf\x66",
  [0x1ab5] = "\xe2\x66", [0x1ab6] = "\xc0\xd4", [0x1ab7] = "\xd5\xdc",
  [0x1ab8] = "\xe9\xe3", [0x1ab9] = "\xdd\xfb", [0x1aba] = "\xc2\x6c",
  [0x1abb] = "\xe9\xe2", [0x1abc] = "\xb0\x62", [0x1abd] = "\xb3\x6d",
  [0x1abf] = "\xb6\x61", [0x1ac0] = "\xf8\x47", [0x1ac1] = "\xdc\x40",
  [0x1ac2] = "\xb6\x63", [0x1ac3] = "\xdc\x48", [0x1ac5] = "\xf7\x65",
  [0x1ac6] = "\xdb\xfe", [0x1ac7] = "\xb8\xfd", [0x1ac8] = "\xb8\xfe",
  [0x1ac9] = "\xe0\x79", [0x1aca] = "\xe0\x78", [0x1acb] = "\xbb\xb3",
  [0x1acc] = "\xbd\xfb", [0x1acd] = "\xbd\xfe", [0x1ace] = "\xbd\xf9",
  [0x1acf] = "\xbd\xfd", [0x1ad0] = "\xbf\xea", [0x1ad1] = "\xf1\x46",
  [0x1ad2] = "\xc3\xe1", [0x1ad4] = "\xcc\xb1", [0x1ad5] = "\xe1\xd7",
  [0x1ad6] = "\xb1\xae", [0x1ad7] = "\xb4\x75", [0x1ad8] = "\xb7\x61",
  [0x1ad9] = "\xb7\x60", [0x1ada] = "\xdd\x75", [0x1adb] = "\xdd\x76",
  [0x1adc] = "\xe1\xd6", [0x1add] = "\xbb\x4e", [0x1ade] = "\xbf\x5e",
  [0x1adf] = "\xd2\xbe", [0x1ae0] = "\xde\xc0", [0x1ae1] = "\xed\x72",
  [0x1ae2] = "\xed\x70", [0x1ae3] = "\xf0\x49", [0x1ae4] = "\xbc\xb3",
  [0x1ae5] = "\xc9\xee", [0x1ae6] = "\xc9\xed", [0x1ae7] = "\xca\xf8",
  [0x1ae8] = "\xa9\xfe", [0x1ae9] = "\xbe\xe8", [0x1aea] = "\xaa\x58",
  [0x1aeb] = "\xcc\xca", [0x1aec] = "\xa9\xfd", [0x1aed] = "\xa9\xfb",
  [0x1aee] = "\xcd\x52", [0x1aef] = "\xac\x54", [0x1af0] = "\xce\xf5",
  [0x1af1] = "\xce\xf6", [0x1af2] = "\xac\x52", [0x1af3] = "\xce\xeb",
  [0x1af4] = "\xbc\xca", [0x1af5] = "\xaf\xcf", [0x1af6] = "\xd1\xd4",
  [0x1af7] = "\xbe\xe7", [0x1af8] = "\xae\xd0", [0x1af9] = "\xae\xcb",
  [0x1afa] = "\xb7\x75", [0x1afb] = "\xd5\xaa", [0x1afd] = "\xb4\xbe",
  [0x1afe] = "\xb7\x79", [0x1aff] = "\xdd\xc0", [0x1b00] = "\xc0\xc7",
  [0x1b01] = "\xba\x5c", [0x1b02] = "\xe9\xb4", [0x1b03] = "\xef\xdb",
  [0x1b04] = "\xc2\x60", [0x1b05] = "\xc4\x66", [0x1b06] = "\xc4\xe4",
  [0x1b07] = "\xb6\x4e", [0x1b08] = "\xdb\xea", [0x1b09] = "\xdb\xe8",
  [0x1b0a] = "\xb6\x4d", [0x1b0b] = "\xf0\xf4", [0x1b0c] = "\xb8\xed",
  [0x1b0d] = "\xb8\xf0", [0x1b0e] = "\xf6\x52", [0x1b0f] = "\xbb\xac",
  [0x1b10] = "\xe7\xec", [0x1b11] = "\xe4\x53", [0x1b12] = "\xf6\xd2",
  [0x1b13] = "\xe7\xee", [0x1b14] = "\xc1\xcc", [0x1b15] = "\xdb\xd1",
  [0x1b16] = "\xc1\xbb", [0x1b17] = "\xe4\x44", [0x1b18] = "\xf7\x53",
  [0x1b19] = "\xbf\xcc", [0x1b1a] = "\xee\x7a", [0x1b1b] = "\xc2\xd1",
  [0x1b1c] = "\xf2\xf3", [0x1b1d] = "\xcf\xbd", [0x1b1f] = "\xa6\xc9",
  [0x1b20] = "\xbc\xf9", [0x1b21] = "\xac\xba", [0x1b22] = "\xd6\x5c",
  [0x1b23] = "\xd6\x5f", [0x1b24] = "\xb5\x51", [0x1b25] = "\xda\x6c",
  [0x1b26] = "\xde\xa5", [0x1b28] = "\xba\xb9", [0x1b29] = "\xae\xb2",
  [0x1b2a] = "\xd5\x66", [0x1b2b] = "\xd9\x54", [0x1b2c] = "\xe1\xee",
  [0x1b2d] = "\xc0\xbc", [0x1b2e] = "\xac\xfe", [0x1b30] = "\xd9\xe5",
  [0x1b31] = "\xb7\xb7", [0x1b32] = "\xe5\xf7", [0x1b33] = "\xde\x40",
  [0x1b34] = "\xbe\xfb", [0x1b35] = "\xf2\x5b", [0x1b36] = "\xe9\xe8",
  [0x1b37] = "\xf6\xe7", [0x1b38] = "\xc9\x7e", [0x1b39] = "\xc9\xf7",
  [0x1b3a] = "\xcb\x49", [0x1b3b] = "\xcb\x4a", [0x1b3c] = "\xcf\x6c",
  [0x1b3d] = "\xb4\xe3", [0x1b3e] = "\xae\xf4", [0x1b3f] = "\xd5\xdf",
  [0x1b40] = "\xba\x72", [0x1b42] = "\xb1\xd5", [0x1b44] = "\xc3\x7c",
  [0x1b45] = "\xb7\xde", [0x1b46] = "\xbc\xf8", [0x1b47] = "\xac\xb8",
  [0x1b48] = "\xe7\x7e", [0x1b49] = "\xa6\x47", [0x1b4b] = "\xcb\x78",
  [0x1b4c] = "\xa8\x77", [0x1b4e] = "\xcc\xd2", [0x1b50] = "\xaa\xd0",
  [0x1b51] = "\xaa\xd2", [0x1b52] = "\xcd\xb0", [0x1b53] = "\xc0\x61",
  [0x1b55] = "\xc4\x67", [0x1b58] = "\xc4\xa4", [0x1b59] = "\xad\x4b",
  [0x1b5a] = "\xd0\x5b", [0x1b5b] = "\xab\x60", [0x1b5c] = "\xd0\x5f",
  [0x1b5d] = "\xd0\x67", [0x1b5e] = "\xd0\x65", [0x1b5f] = "\xd0\x61",
  [0x1b60] = "\xad\x4f", [0x1b61] = "\xd7\x48", [0x1b62] = "\xaf\xd6",
  [0x1b63] = "\xaf\xdb", [0x1b64] = "\xaf\xda", [0x1b65] = "\xc1\x7a",
  [0x1b67] = "\xd3\x6a", [0x1b68] = "\xaf\xe2", [0x1b69] = "\xae\xd3",
  [0x1b6b] = "\xb3\x62", [0x1b6c] = "\xdf\xa2", [0x1b6d] = "\xd7\x4c",
  [0x1b6e] = "\xd7\x4b", [0x1b6f] = "\xd7\x45", [0x1b70] = "\xa7\xbf",
  [0x1b72] = "\xbe\x4d", [0x1b73] = "\xb5\xcc", [0x1b74] = "\xb5\xcd",
  [0x1b77] = "\xb8\x78", [0x1b78] = "\xdf\x73", [0x1b79] = "\xdf\x76",
  [0x1b7a] = "\xc0\x53", [0x1b7b] = "\xe3\xa6", [0x1b7c] = "\xf8\xed",
  [0x1b7d] = "\xdf\x7d", [0x1b7e] = "\xdc\xe6", [0x1b7f] = "\xdc\xf9",
  [0x1b80] = "\xbb\x4a", [0x1b81] = "\xbb\x4d", [0x1b82] = "\xc2\xc1",
  [0x1b83] = "\xbc\xf0", [0x1b84] = "\xe6\xfd", [0x1b86] = "\xed\xea",
  [0x1b87] = "\xc2\x61", [0x1b88] = "\xed\xeb", [0x1b89] = "\xed\xe8",
  [0x1b8b] = "\xea\xe9", [0x1b8c] = "\xc2\x6a", [0x1b8d] = "\xd5\xd8",
  [0x1b8e] = "\xd9\xda", [0x1b8f] = "\xdd\xf7", [0x1b90] = "\xdd\xf5",
  [0x1b91] = "\xbe\xf9", [0x1b92] = "\xe4\xcd", [0x1b93] = "\xbb\xe3",
  [0x1b94] = "\xc1\xfc", [0x1b95] = "\xc4\x42", [0x1b96] = "\xf6\x74",
  [0x1b98] = "\xc9\x5f", [0x1b99] = "\xdd\x62", [0x1b9a] = "\xc1\xd4",
  [0x1b9b] = "\xee\x7d", [0x1b9c] = "\xb4\xb4", [0x1b9d] = "\xf8\x7e",
  [0x1b9e] = "\xc4\xe3", [0x1b9f] = "\xa9\xf3", [0x1ba0] = "\xd1\xcf",
  [0x1ba1] = "\xd1\xd1", [0x1ba2] = "\xd1\xd0", [0x1ba3] = "\xb1\xdc",
  [0x1ba4] = "\xb1\xdd", [0x1ba5] = "\xd9\x7e", [0x1ba6] = "\xba\x59",
  [0x1ba7] = "\xb7\xd5", [0x1ba8] = "\xde\x6d", [0x1ba9] = "\xbf\x4c",
  [0x1baa] = "\xe2\xc4", [0x1bad] = "\xcf\xae", [0x1bae] = "\xac\xaf",
  [0x1baf] = "\xcf\xb1", [0x1bb0] = "\xea\x4d", [0x1bb1] = "\xaf\x4c",
  [0x1bb2] = "\xd6\x51", [0x1bb3] = "\xd6\x55", [0x1bb4] = "\xbf\x54",
  [0x1bb5] = "\xda\x5a", [0x1bb6] = "\xda\x5d", [0x1bb8] = "\xb7\xd4",
  [0x1bb9] = "\xb7\xdb", [0x1bba] = "\xc1\xeb", [0x1bbb] = "\xde\x73",
  [0x1bbd] = "\xde\x74", [0x1bc0] = "\xe6\x72", [0x1bc1] = "\xe6\x69",
  [0x1bc2] = "\xbc\xf7", [0x1bc3] = "\xe6\x66", [0x1bc4] = "\xc0\xee",
  [0x1bc5] = "\xea\x4f", [0x1bc6] = "\xc0\xe6", [0x1bc7] = "\xef\xfb",
  [0x1bc8] = "\xf5\xda", [0x1bc9] = "\xc6\x79", [0x1bca] = "\xa4\xf5",
  [0x1bcb] = "\xc2\x76", [0x1bcc] = "\xb7\xd6", [0x1bcd] = "\xbf\x51",
  [0x1bce] = "\xa9\xd1", [0x1bcf] = "\xcc\xb2", [0x1bd0] = "\xce\xd3",
  [0x1bd1] = "\xb1\xaf", [0x1bd2] = "\xb4\x76", [0x1bd3] = "\xa5\xdc",
  [0x1bd4] = "\xaa\xc1", [0x1bd5] = "\xac\xe6", [0x1bd6] = "\xac\xe7",
  [0x1bd7] = "\xd7\xb6", [0x1bd8] = "\xd2\xe9", [0x1bd9] = "\xd2\xea",
  [0x1bda] = "\xaf\xae", [0x1bdb] = "\xf2\xa5", [0x1bdc] = "\xaf\xad",
  [0x1bdd] = "\xaf\xa8", [0x1bde] = "\xba\xd5", [0x1bdf] = "\xd6\xbd",
  [0x1be0] = "\xb8\x52", [0x1be1] = "\xc1\x49", [0x1be2] = "\xe2\xf9",
  [0x1be3] = "\xe6\xcb", [0x1be4] = "\xc1\x48", [0x1be5] = "\xc5\xa1",
  [0x1be6] = "\xca\xd7", [0x1be7] = "\xca\xd8", [0x1be8] = "\xef\xc8",
  [0x1be9] = "\xd1\x7c", [0x1bea] = "\xd1\x7d", [0x1beb] = "\xd1\x7e",
  [0x1bec] = "\xd1\xa1", [0x1bed] = "\xae\x7e", [0x1bee] = "\xae\xa1",
  [0x1bef] = "\xe1\xc3", [0x1bf0] = "\xb7\x5e", [0x1bf1] = "\xb7\x5d",
  [0x1bf2] = "\xbc\x7c", [0x1bf3] = "\xbe\xcd", [0x1bf4] = "\xe9\x76",
  [0x1bf5] = "\xc0\xb7", [0x1bf6] = "\xc2\x56", [0x1bf7] = "\xf9\xba",
  [0x1bf8] = "\xa6\xd6", [0x1bf9] = "\xa6\xd6", [0x1bfa] = "\xcc\xee",
  [0x1bfb] = "\xed\x4f", [0x1bfc] = "\xd9\xe7", [0x1bfd] = "\xc1\x46",
  [0x1bfe] = "\xcd\x7a", [0x1bff] = "\xe2\xee", [0x1c00] = "\xcf\xf1",
  [0x1c01] = "\xda\xcf", [0x1c03] = "\xcf\xeb", [0x1c04] = "\xac\x48",
  [0x1c05] = "\xaf\xa2", [0x1c07] = "\xaf\x7c", [0x1c09] = "\xc4\x72",
  [0x1c0a] = "\xf5\xe2", [0x1c0b] = "\xaf\xa4", [0x1c0d] = "\xaf\xa1",
  [0x1c0e] = "\xd2\xda", [0x1c0f] = "\xd2\xd9", [0x1c10] = "\xd2\xe2",
  [0x1c11] = "\xb2\xba", [0x1c12] = "\xda\xc9", [0x1c13] = "\xda\xc6",
  [0x1c14] = "\xed\xa6", [0x1c15] = "\xd6\xb9", [0x1c16] = "\xd6\xbb",
  [0x1c18] = "\xd6\xb8", [0x1c19] = "\xda\xcb", [0x1c1a] = "\xbf\x6c",
  [0x1c1b] = "\xb8\x50", [0x1c1c] = "\xde\xe5", [0x1c1d] = "\xde\xe4",
  [0x1c1e] = "\xea\xa7", [0x1c1f] = "\xde\xec", [0x1c20] = "\xba\xd4",
  [0x1c21] = "\xe2\xeb", [0x1c23] = "\xe2\xf3", [0x1c24] = "\xe6\xc6",
  [0x1c26] = "\xe6\xca", [0x1c27] = "\xbf\x6b", [0x1c28] = "\xed\xaa",
  [0x1c29] = "\xed\xab", [0x1c2a] = "\xc1\x45", [0x1c2b] = "\xf0\x58",
  [0x1c2c] = "\xf4\x7d", [0x1c2d] = "\xf2\xa4", [0x1c2e] = "\xf2\xa1",
  [0x1c2f] = "\xf7\xce", [0x1c30] = "\xdc\xb0", [0x1c31] = "\xef\xb0",
  [0x1c32] = "\xf3\xf5", [0x1c33] = "\xcd\x77", [0x1c34] = "\xcf\xe2",
  [0x1c36] = "\xac\xda", [0x1c37] = "\xac\xe0", [0x1c38] = "\xcf\xe0",
  [0x1c39] = "\xd2\xd6", [0x1c3a] = "\xd2\xd7", [0x1c3b] = "\xd2\xd5",
  [0x1c3c] = "\xd6\xaa", [0x1c3d] = "\xd6\xb0", [0x1c3e] = "\xd6\xb2",
  [0x1c3f] = "\xb2\xb6", [0x1c40] = "\xb7\xfd", [0x1c41] = "\xc2\xa5",
  [0x1c42] = "\xda\xbb", [0x1c44] = "\xde\xdd", [0x1c45] = "\xb8\x45",
  [0x1c46] = "\xb8\x46", [0x1c47] = "\xb8\x44", [0x1c48] = "\xe8\xfb",
  [0x1c49] = "\xe6\xbc", [0x1c4a] = "\xba\xcc", [0x1c4b] = "\xe2\xe7",
  [0x1c4c] = "\xbd\x4f", [0x1c4d] = "\xbd\x50", [0x1c4e] = "\xbf\x67",
  [0x1c4f] = "\xbf\x65", [0x1c50] = "\xc0\xfd", [0x1c51] = "\xed\x7a",
  [0x1c52] = "\xc2\xa2", [0x1c53] = "\xcb\x6d", [0x1c54] = "\xcd\x71",
  [0x1c55] = "\xac\xca", [0x1c56] = "\xac\xcb", [0x1c57] = "\xcf\xdb",
  [0x1c58] = "\xd2\xc1", [0x1c59] = "\xda\xae", [0x1c5a] = "\xde\xc4",
  [0x1c5c] = "\xd0\x4d", [0x1c5d] = "\xd3\x5c", [0x1c5e] = "\xaf\xcb",
  [0x1c5f] = "\xdb\xe0", [0x1c60] = "\xdf\x6a", [0x1c61] = "\xf2\xbc",
  [0x1c63] = "\xbf\xa9", [0x1c64] = "\xc5\xf9", [0x1c65] = "\xed\xe3",
  [0x1c66] = "\xaf\x72", [0x1c67] = "\xbf\x64", [0x1c68] = "\xf7\xf8",
  [0x1c69] = "\xaa\xf7", [0x1c6a] = "\xd0\xc1", [0x1c6b] = "\xd0\xc0",
  [0x1c6c] = "\xb0\x7a", [0x1c6d] = "\xb0\x78", [0x1c6e] = "\xd4\x42",
  [0x1c6f] = "\xd8\x41", [0x1c70] = "\xb3\xa9", [0x1c71] = "\xd7\xfc",
  [0x1c72] = "\xeb\xea", [0x1c73] = "\xb3\xa6", [0x1c74] = "\xd7\xfd",
  [0x1c76] = "\xdc\x67", [0x1c77] = "\xb9\x64", [0x1c78] = "\xb6\x7a",
  [0x1c79] = "\xb6\x78", [0x1c7a] = "\xdc\x62", [0x1c7b] = "\xdc\x70",
  [0x1c7c] = "\xdc\x6f", [0x1c7d] = "\xdc\x64", [0x1c7e] = "\xdc\x6a",
  [0x1c7f] = "\xe0\xb1", [0x1c80] = "\xe0\xb2", [0x1c81] = "\xb9\x57",
  [0x1c82] = "\xe0\xb3", [0x1c84] = "\xb9\x5a", [0x1c85] = "\xb9\x59",
  [0x1c86] = "\xe0\xb5", [0x1c87] = "\xe0\xbb", [0x1c88] = "\xe0\xba",
  [0x1c89] = "\xb9\x66", [0x1c8a] = "\xc5\xe0", [0x1c8b] = "\xe0\xb0",
  [0x1c8c] = "\xb9\x62", [0x1c8d] = "\xe0\xae", [0x1c8e] = "\xb9\x63",
  [0x1c8f] = "\xe0\xb7", [0x1c90] = "\xb9\x65", [0x1c91] = "\xc5\x4d",
  [0x1c92] = "\xbe\x52", [0x1c93] = "\xe4\x7b", [0x1c94] = "\xe4\x7a",
  [0x1c95] = "\xe4\x7d", [0x1c96] = "\xe4\xa8", [0x1c97] = "\xe8\x68",
  [0x1c98] = "\xc4\xc2", [0x1c9a] = "\xc5\x4c", [0x1c9c] = "\xe4\xa1",
  [0x1c9d] = "\xf1\x5a", [0x1c9e] = "\xbb\xcb", [0x1c9f] = "\xe8\x6f",
  [0x1ca0] = "\xe4\xb7", [0x1ca1] = "\xf4\xf3", [0x1ca2] = "\xbb\xcd",
  [0x1ca3] = "\xe4\xa6", [0x1ca4] = "\xf3\x6a", [0x1ca5] = "\xe4\xa2",
  [0x1ca6] = "\xbf\xfe", [0x1ca7] = "\xe4\xa3", [0x1ca8] = "\xe0\xb6",
  [0x1ca9] = "\xf4\xe7", [0x1caa] = "\xbb\xcf", [0x1cab] = "\xe4\xa7",
  [0x1cac] = "\xf4\xf9", [0x1cad] = "\xeb\xe4", [0x1cae] = "\xe8\x69",
  [0x1caf] = "\xc3\xf4", [0x1cb1] = "\xbe\x59", [0x1cb2] = "\xe8\x70",
  [0x1cb3] = "\xe8\x72", [0x1cb4] = "\xbe\x56", [0x1cb5] = "\xe8\x6e",
  [0x1cb9] = "\xe8\x63", [0x1cba] = "\xe8\x6a", [0x1cbb] = "\xe8\x76",
  [0x1cbc] = "\xeb\xe8", [0x1cbd] = "\xeb\xe1", [0x1cbf] = "\xeb\xe6",
  [0x1cc1] = "\xeb\xe9", [0x1cc2] = "\xc0\x43", [0x1cc3] = "\xc0\x44",
  [0x1cc6] = "\xeb\xdf", [0x1cc7] = "\xeb\xe0", [0x1cc8] = "\xc0\x45",
  [0x1cc9] = "\xc1\xe6", [0x1cca] = "\xee\xd2", [0x1ccb] = "\xee\xd6",
  [0x1ccc] = "\xc1\xee", [0x1ccd] = "\xee\xd0", [0x1cce] = "\xf1\x5c",
  [0x1ccf] = "\xc1\xec", [0x1cd1] = "\xc3\xf3", [0x1cd2] = "\xc3\xf2",
  [0x1cd3] = "\xf4\xf0", [0x1cd5] = "\xf3\x68", [0x1cd6] = "\xc2\xf0",
  [0x1cd7] = "\xc5\x4f", [0x1cd9] = "\xf1\x5e", [0x1cda] = "\xc2\xef",
  [0x1cdb] = "\xf1\x53", [0x1cdc] = "\xf7\x67", [0x1cdd] = "\xc3\xf0",
  [0x1cde] = "\xc3\xed", [0x1cdf] = "\xc3\xee", [0x1ce0] = "\xc3\xf1",
  [0x1ce1] = "\xf3\x60", [0x1ce2] = "\xc3\xeb", [0x1ce3] = "\xf3\x61",
  [0x1ce4] = "\xc3\xe9", [0x1ce5] = "\xf4\xec", [0x1ce7] = "\xf4\xf5",
  [0x1ce9] = "\xf4\xe8", [0x1cea] = "\xf9\x53", [0x1ceb] = "\xf4\xea",
  [0x1ced] = "\xf3\x66", [0x1cee] = "\xf4\xf1", [0x1cef] = "\xf7\x69",
  [0x1cf0] = "\xc5\x4e", [0x1cf1] = "\xf6\x5e", [0x1cf3] = "\xc5\xdf",
  [0x1cf4] = "\xc1\xe9", [0x1cf5] = "\xcf\xe8", [0x1cf6] = "\xda\xc4",
  [0x1cf7] = "\xb9\x6e", [0x1cf8] = "\xcf\xf9", [0x1cf9] = "\xd2\xf1",
  [0x1cfa] = "\xaf\xb0", [0x1cfb] = "\xd2\xed", [0x1cfd] = "\xb4\x52",
  [0x1cfe] = "\xda\xd8", [0x1cff] = "\xda\xd6", [0x1d00] = "\xb8\x5c",
  [0x1d01] = "\xb8\x5a", [0x1d02] = "\xe6\xd2", [0x1d03] = "\xbd\x5e",
  [0x1d04] = "\xc2\xa8", [0x1d05] = "\xc2\x48", [0x1d06] = "\xc3\x4c",
  [0x1d07] = "\xf6\xf8", [0x1d08] = "\xac\xd4", [0x1d09] = "\xb2\xae",
  [0x1d0a] = "\xb5\x71", [0x1d0b] = "\xde\xd5", [0x1d0c] = "\xed\x78",
  [0x1d0d] = "\xd2\xbc", [0x1d0e] = "\xb2\x7d", [0x1d0f] = "\xa8\x69",
  [0x1d10] = "\xb9\xa7", [0x1d11] = "\xbb\xf0", [0x1d12] = "\xe8\xe9",
  [0x1d13] = "\xbe\x7d", [0x1d14] = "\xc0\x6c", [0x1d15] = "\xf3\xf0",
  [0x1d16] = "\xf9\xb7", [0x1d17] = "\xec\xa9", [0x1d18] = "\xec\xae",
  [0x1d19] = "\xc5\xee", [0x1d1a] = "\xef\x7b", [0x1d1b] = "\xf7\xc1",
  [0x1d1c] = "\xef\x7e", [0x1d1d] = "\xef\xa2", [0x1d1e] = "\xc6\x7d",
  [0x1d1f] = "\xf1\xd0", [0x1d20] = "\xf9\xcf", [0x1d21] = "\xc3\x5b",
  [0x1d22] = "\xf1\xd9", [0x1d23] = "\xf8\x70", [0x1d24] = "\xf1\xcf",
  [0x1d25] = "\xc4\x4d", [0x1d27] = "\xc4\x4f", [0x1d28] = "\xf3\xed",
  [0x1d29] = "\xc4\x4c", [0x1d2a] = "\xf5\x7c", [0x1d2b] = "\xf5\xa7",
  [0x1d2c] = "\xf5\xac", [0x1d2e] = "\xf5\xad", [0x1d2f] = "\xc5\x63",
  [0x1d30] = "\xf6\xb4", [0x1d31] = "\xc6\x78", [0x1d32] = "\xc5\xc2",
  [0x1d33] = "\xf7\xba", [0x1d34] = "\xf8\x63", [0x1d35] = "\xf8\x6b",
  [0x1d36] = "\xf8\x6c", [0x1d37] = "\xf8\x66", [0x1d39] = "\xc6\x4f",
  [0x1d3a] = "\xc6\x7c", [0x1d3c] = "\xcb\x6f", [0x1d3d] = "\xf0\x4b",
  [0x1d3e] = "\xc2\x7e", [0x1d3f] = "\xaa\xb7", [0x1d41] = "\xac\xd0",
  [0x1d42] = "\xaf\x69", [0x1d43] = "\xde\xcb", [0x1d44] = "\xaf\x6e",
  [0x1d45] = "\xd2\xc6", [0x1d46] = "\xaf\x70", [0x1d47] = "\xd2\xc3",
  [0x1d49] = "\xaf\x6d", [0x1d4b] = "\xb2\xad", [0x1d4c] = "\xb5\x69",
  [0x1d4d] = "\xc0\xf7", [0x1d4e] = "\xda\xb2", [0x1d4f] = "\xda\xb5",
  [0x1d50] = "\xed\x77", [0x1d51] = "\xda\xaf", [0x1d52] = "\xde\xc9",
  [0x1d53] = "\xb7\xf3", [0x1d54] = "\xde\xcc", [0x1d55] = "\xb7\xf5",
  [0x1d56] = "\xde\xcf", [0x1d57] = "\xb7\xef", [0x1d58] = "\xed\x76",
  [0x1d59] = "\xe2\xdc", [0x1d5a] = "\xe6\xb0", [0x1d5b] = "\xe2\xdf",
  [0x1d5c] = "\xe6\xaf", [0x1d5d] = "\xbf\x62", [0x1d5e] = "\xe2\xdd",
  [0x1d5f] = "\xe6\xb1", [0x1d60] = "\xe6\xb7", [0x1d61] = "\xea\x70",
  [0x1d62] = "\xbd\x49", [0x1d63] = "\xbd\x43", [0x1d65] = "\xea\x6c",
  [0x1d66] = "\xea\x73", [0x1d67] = "\xf6\xf4", [0x1d68] = "\xea\x71",
  [0x1d69] = "\xed\x74", [0x1d6a] = "\xc5\x7d", [0x1d6b] = "\xea\x6f",
  [0x1d6d] = "\xc4\xee", [0x1d6f] = "\xf0\x4a", [0x1d70] = "\xc2\x7d",
  [0x1d71] = "\xc5\xf6", [0x1d73] = "\xd6\xf6", [0x1d74] = "\xda\xdc",
  [0x1d75] = "\xcd\xa5", [0x1d76] = "\xaa\xc6", [0x1d77] = "\xcf\xfe",
  [0x1d78] = "\xd2\xf6", [0x1d79] = "\xaf\xb7", [0x1d7a] = "\xb2\xc0",
  [0x1d7b] = "\xc4\x75", [0x1d7c] = "\xb8\x5e", [0x1d7d] = "\xe3\x4c",
  [0x1d7e] = "\xe3\x4a", [0x1d7f] = "\xea\xb2", [0x1d80] = "\xe6\xd5",
  [0x1d81] = "\xa6\xe7", [0x1d82] = "\xd0\xae", [0x1d83] = "\xd3\xd4",
  [0x1d84] = "\xb0\x4d", [0x1d85] = "\xd3\xd7", [0x1d86] = "\xb0\x4c",
  [0x1d87] = "\xd7\xb5", [0x1d88] = "\xc3\xcb", [0x1d89] = "\xdb\xcb",
  [0x1d8a] = "\xdb\xca", [0x1d8b] = "\xdb\xcf", [0x1d8c] = "\xee\x72",
  [0x1d8d] = "\xdf\xf4", [0x1d8e] = "\xf2\xf0", [0x1d8f] = "\xf0\xd9",
  [0x1d90] = "\xe3\xf7", [0x1d91] = "\xbb\x75", [0x1d92] = "\xe3\xfc",
  [0x1d93] = "\xbb\x74", [0x1d94] = "\xe3\xfa", [0x1d95] = "\xe4\x40",
  [0x1d96] = "\xbf\xca", [0x1d97] = "\xe7\xc8", [0x1d98] = "\xbd\xc8",
  [0x1d99] = "\xc1\xb9", [0x1d9a] = "\xbd\xca", [0x1d9b] = "\xc4\xb0",
  [0x1d9c] = "\xbf\xc9", [0x1d9d] = "\xc1\xb7", [0x1d9e] = "\xee\x75",
  [0x1d9f] = "\xf4\xcb", [0x1da0] = "\xf9\x4b", [0x1da1] = "\xa5\xd4",
  [0x1da2] = "\xad\x45", [0x1da3] = "\xe2\xe1", [0x1da4] = "\xb5\x72",
  [0x1da5] = "\xac\xe1", [0x1da6] = "\xa6\xd4", [0x1da7] = "\xd0\x54",
  [0x1da8] = "\xd3\x66", [0x1da9] = "\xb2\xe0", [0x1dad] = "\xbd\xa2",
  [0x1dae] = "\xed\xe7", [0x1daf] = "\xea\xe6", [0x1db0] = "\xbf\xae",
  [0x1db2] = "\xb5\xc0", [0x1db3] = "\xcd\xaa", [0x1db4] = "\xd7\x43",
  [0x1db5] = "\xb2\xe2", [0x1db6] = "\xf4\xb2", [0x1db7] = "\xb5\xc1",
  [0x1db8] = "\xf0\x7c", [0x1db9] = "\xc1\x6d", [0x1dba] = "\xb5\xf7",
  [0x1dbb] = "\xdc\xac", [0x1dbc] = "\xe0\xe0", [0x1dbd] = "\xe0\xde",
  [0x1dbe] = "\xbe\x65", [0x1dbf] = "\xbe\x67", [0x1dc0] = "\xe5\xf9",
  [0x1dc1] = "\xe8\xb8", [0x1dc2] = "\xc0\x58", [0x1dc3] = "\xc3\x45",
  [0x1dc4] = "\xc3\x46", [0x1dc5] = "\xf9\xae", [0x1dc6] = "\xf5\x4f",
  [0x1dc7] = "\xf3\xa8", [0x1dc8] = "\xc5\x56", [0x1dca] = "\xc6\x49",
  [0x1dcb] = "\xca\x4f", [0x1dcc] = "\xb0\x40", [0x1dcd] = "\xcd\xc8",
  [0x1dce] = "\xf0\xca", [0x1dcf] = "\xf2\xe3", [0x1dd0] = "\xad\x6b",
  [0x1dd1] = "\xd0\xa8", [0x1dd2] = "\xad\x6a", [0x1dd3] = "\xd3\xba",
  [0x1dd4] = "\xd3\xbc", [0x1dd5] = "\xd3\xc2", [0x1dd6] = "\xb8\xc3",
  [0x1dd7] = "\xc4\xae", [0x1dd8] = "\xd3\xc8", [0x1dd9] = "\xb0\x47",
  [0x1dda] = "\xb0\x42", [0x1ddb] = "\xb0\x43", [0x1ddc] = "\xb0\x45",
  [0x1ddd] = "\xb3\x46", [0x1dde] = "\xb3\x47", [0x1ddf] = "\xb3\x48",
  [0x1de0] = "\xc4\xf7", [0x1de1] = "\xd7\xa5", [0x1de2] = "\xd7\xa4",
  [0x1de3] = "\xb3\x4b", [0x1de4] = "\xb3\x4c", [0x1de5] = "\xb3\x4d",
  [0x1de6] = "\xf2\xde", [0x1de7] = "\xd7\xab", [0x1de8] = "\xdb\xbc",
  [0x1de9] = "\xdf\xe6", [0x1dea] = "\xc2\xcc", [0x1deb] = "\xb5\xed",
  [0x1dec] = "\xeb\x6b", [0x1ded] = "\xb5\xf1", [0x1dee] = "\xb8\xbb",
  [0x1def] = "\xb5\xf2", [0x1df0] = "\xf4\xc5", [0x1df1] = "\xb5\xeb",
  [0x1df2] = "\xdb\xb1", [0x1df3] = "\xdb\xbf", [0x1df4] = "\xb8\xc2",
  [0x1df5] = "\xb8\xbd", [0x1df6] = "\xdf\xea", [0x1df7] = "\xb8\xbc",
  [0x1df8] = "\xb8\xc4", [0x1df9] = "\xdf\xef", [0x1dfa] = "\xdf\xec",
  [0x1dfb] = "\xe3\xd6", [0x1dfc] = "\xbb\x66", [0x1dfd] = "\xe3\xdb",
  [0x1dfe] = "\xbb\x68", [0x1dff] = "\xe3\xda", [0x1e00] = "\xe3\xf2",
  [0x1e01] = "\xe3\xe2", [0x1e02] = "\xc1\xb4", [0x1e03] = "\xbb\x69",
  [0x1e04] = "\xe3\xeb", [0x1e05] = "\xbb\x6d", [0x1e06] = "\xbb\x6c",
  [0x1e07] = "\xbb\x64", [0x1e08] = "\xbd\xb8", [0x1e09] = "\xbb\x67",
  [0x1e0b] = "\xf4\xc6", [0x1e0c] = "\xe7\xbc", [0x1e0d] = "\xbd\xbb",
  [0x1e0e] = "\xad\x6b", [0x1e0f] = "\xbd\xc1", [0x1e10] = "\xe7\xb8",
  [0x1e12] = "\xbd\xc2", [0x1e13] = "\xe7\xa2", [0x1e14] = "\xc1\xb2",
  [0x1e15] = "\xe7\xa3", [0x1e16] = "\xbd\xbf", [0x1e17] = "\xe7\xba",
  [0x1e18] = "\xeb\x5e", [0x1e19] = "\xee\x67", [0x1e1b] = "\xc1\xaf",
  [0x1e1c] = "\xc1\xb0", [0x1e1d] = "\xeb\x60", [0x1e1e] = "\xeb\x67",
  [0x1e1f] = "\xee\x5a", [0x1e20] = "\xeb\x5d", [0x1e21] = "\xbf\xc0",
  [0x1e22] = "\xc1\xb1", [0x1e23] = "\xf0\xc1", [0x1e24] = "\xee\x5d",
  [0x1e25] = "\xee\x5f", [0x1e26] = "\xc1\xae", [0x1e27] = "\xc1\xb5",
  [0x1e28] = "\xf0\xc7", [0x1e29] = "\xee\x6e", [0x1e2a] = "\xc1\xad",
  [0x1e2b] = "\xc1\xac", [0x1e2c] = "\xee\x69", [0x1e2d] = "\xf0\xbe",
  [0x1e2e] = "\xf0\xc0", [0x1e2f] = "\xc2\xcf", [0x1e30] = "\xf2\xdc",
  [0x1e31] = "\xf4\xc9", [0x1e32] = "\xf4\xc8", [0x1e33] = "\xc3\xca",
  [0x1e34] = "\xf2\xe7", [0x1e35] = "\xf6\x42", [0x1e36] = "\xc4\xf9",
  [0x1e37] = "\xc5\xfb", [0x1e38] = "\xf9\x6e", [0x1e39] = "\xa6\xce",
  [0x1e3a] = "\xc4\xa2", [0x1e3b] = "\xc1\x6a", [0x1e3c] = "\xed\xe1",
  [0x1e3d] = "\xaf\xe7", [0x1e3e] = "\xaa\xc7", [0x1e3f] = "\xac\xf2",
  [0x1e40] = "\xd3\x44", [0x1e41] = "\xbf\x77", [0x1e42] = "\xd3\x40",
  [0x1e43] = "\xb8\x63", [0x1e44] = "\xd3\x45", [0x1e45] = "\xd6\xd7",
  [0x1e46] = "\xd3\x43", [0x1e47] = "\xda\xdf", [0x1e48] = "\xd6\xdb",
  [0x1e49] = "\xd6\xd4", [0x1e4a] = "\xb2\xc6", [0x1e4b] = "\xb2\xc8",
  [0x1e4c] = "\xd6\xd6", [0x1e4d] = "\xb2\xc1", [0x1e4e] = "\xd6\xce",
  [0x1e4f] = "\xd6\xd1", [0x1e50] = "\xd6\xd2", [0x1e51] = "\xf8\xfb",
  [0x1e52] = "\xb2\xc7", [0x1e53] = "\xda\xe5", [0x1e54] = "\xed\xc3",
  [0x1e55] = "\xda\xe6", [0x1e56] = "\xba\xe1", [0x1e57] = "\xda\xe2",
  [0x1e58] = "\xba\xe5", [0x1e59] = "\xb8\x61", [0x1e5a] = "\xb8\x62",
  [0x1e5d] = "\xdf\x4b", [0x1e5e] = "\xc1\x51", [0x1e5f] = "\xe3\x52",
  [0x1e60] = "\xed\xbd", [0x1e61] = "\xe6\xd7", [0x1e62] = "\xba\xe6",
  [0x1e63] = "\xe6\xd9", [0x1e64] = "\xba\xe3", [0x1e65] = "\xf6\xfb",
  [0x1e66] = "\xe3\x58", [0x1e67] = "\xc2\xb0", [0x1e68] = "\xe3\x4f",
  [0x1e6a] = "\xc2\xad", [0x1e6b] = "\xbd\x65", [0x1e6c] = "\xc2\xb1",
  [0x1e6d] = "\xbd\x68", [0x1e6e] = "\xbd\x6a", [0x1e6f] = "\xea\xba",
  [0x1e70] = "\xea\xbd", [0x1e71] = "\xea\xbc", [0x1e72] = "\xbf\x7b",
  [0x1e73] = "\xea\xc1", [0x1e74] = "\xc1\x50", [0x1e75] = "\xc1\x4e",
  [0x1e77] = "\xed\xbb", [0x1e78] = "\xf8\xb8", [0x1e79] = "\xed\xc2",
  [0x1e7a] = "\xf0\x65", [0x1e7b] = "\xc2\xaf", [0x1e7c] = "\xf0\x68",
  [0x1e7d] = "\xc3\xb0", [0x1e7e] = "\xc5\xa3", [0x1e7f] = "\xc3\xb3",
  [0x1e80] = "\xaa\xd8", [0x1e81] = "\xd3\x70", [0x1e82] = "\xb2\xea",
  [0x1e83] = "\xdb\x57", [0x1e84] = "\xaf\xe5", [0x1e85] = "\xd3\xd0",
  [0x1e86] = "\xd0\x69", [0x1e87] = "\xad\x51", [0x1e88] = "\xf2\xc3",
  [0x1e8a] = "\xd3\x71", [0x1e8b] = "\xaf\xea", [0x1e8c] = "\xaf\xe9",
  [0x1e8d] = "\xd7\x53", [0x1e8e] = "\xf7\x46", [0x1e8f] = "\xd7\x54",
  [0x1e90] = "\xd7\x56", [0x1e92] = "\xdf\xa5", [0x1e93] = "\xdf\xa4",
  [0x1e94] = "\xbb\x53", [0x1e95] = "\xe7\x46", [0x1e96] = "\xed\xf1",
  [0x1e97] = "\xf0\xa2", [0x1e98] = "\xf4\xb6", [0x1e99] = "\xd3\xda",
  [0x1e9a] = "\xb8\xcd", [0x1e9b] = "\xb3\x50", [0x1e9c] = "\xb8\xca",
  [0x1e9d] = "\xb8\xc6", [0x1e9e] = "\xc3\xce", [0x1e9f] = "\xd6\xf4",
  [0x1ea0] = "\xdf\x6d", [0x1ea1] = "\xdf\x6e", [0x1ea2] = "\xbd\x7e",
  [0x1ea3] = "\xe6\xf3", [0x1ea4] = "\xbf\xaa", [0x1ea5] = "\xaa\xc4",
  [0x1ea6] = "\xae\xc5", [0x1ea7] = "\xd3\x4a", [0x1ea8] = "\xf5\xe6",
  [0x1ea9] = "\xf8\xfd", [0x1eaa] = "\xda\xe8", [0x1eab] = "\xda\xe7",
  [0x1eac] = "\xdf\x54", [0x1ead] = "\xe3\x60", [0x1eae] = "\xba\xea",
  [0x1eaf] = "\xc1\x58", [0x1eb0] = "\xef\x53", [0x1eb1] = "\xe6\xe1",
  [0x1eb2] = "\xe8\xbf", [0x1eb3] = "\xe6\xe0", [0x1eb4] = "\xe6\xdf",
  [0x1eb5] = "\xea\xc9", [0x1eb6] = "\xed\xcf", [0x1eb7] = "\xa6\xe1",
  [0x1eb8] = "\xba\x5b", [0x1eb9] = "\xac\xfd", [0x1eba] = "\xb2\xde",
  [0x1ebb] = "\xb5\xbf", [0x1ebc] = "\xe3\x7d", [0x1ebd] = "\xbb\x42",
  [0x1ebe] = "\xe6\xf9", [0x1ebf] = "\xbd\xa1", [0x1ec0] = "\xbf\xad",
  [0x1ec1] = "\xc1\x6b", [0x1ec2] = "\xa6\xcd", [0x1ec3] = "\xed\xde",
  [0x1ec4] = "\xe3\x75", [0x1ec5] = "\xe3\x76", [0x1ec6] = "\xed\xd9",
  [0x1ec7] = "\xf9\x41", [0x1ec8] = "\xbe\xa2", [0x1ec9] = "\xc4\x54",
  [0x1eca] = "\xad\x76", [0x1ecb] = "\xdb\xed", [0x1ecc] = "\xe0\x64",
  [0x1ecd] = "\xe0\x5e", [0x1ece] = "\xf9\x6f", [0x1ecf] = "\xb3\x69",
  [0x1ed0] = "\xbd\xea", [0x1ed1] = "\xd3\xde", [0x1ed2] = "\xb3\x61",
  [0x1ed3] = "\xad\xaa", [0x1ed4] = "\xd3\xfe", [0x1ed5] = "\xd4\x40",
  [0x1ed6] = "\xd4\x41", [0x1ed7] = "\xdc\x5e", [0x1ed8] = "\xdc\x60",
  [0x1ed9] = "\xdc\x5d", [0x1edb] = "\xb9\x55", [0x1edc] = "\xe0\xad",
  [0x1edd] = "\xc6\x6f", [0x1ede] = "\xf9\x71", [0x1edf] = "\xe4\x73",
  [0x1ee0] = "\xbb\xc6", [0x1ee1] = "\xe4\x74", [0x1ee3] = "\xe8\x5b",
  [0x1ee4] = "\xeb\xd9", [0x1ee5] = "\xeb\xdb", [0x1ee6] = "\xeb\xda",
  [0x1ee7] = "\xee\xc6", [0x1ee8] = "\xc1\xde", [0x1ee9] = "\xf1\x52",
  [0x1eea] = "\xf3\x5f", [0x1eeb] = "\xc3\xe7", [0x1eec] = "\xf1\x51",
  [0x1eed] = "\xf4\xe4", [0x1eee] = "\xc4\xbf", [0x1eef] = "\xc5\x48",
  [0x1ef0] = "\xa8\xa8", [0x1ef1] = "\xf6\xc9", [0x1ef2] = "\xc4\xbb",
  [0x1ef3] = "\xe0\x72", [0x1ef4] = "\xe4\x60", [0x1ef5] = "\xc2\xd9",
  [0x1ef6] = "\xf1\x44", [0x1ef7] = "\xd3\xe5", [0x1ef8] = "\xd7\xd9",
  [0x1ef9] = "\xd7\xd6", [0x1efa] = "\xb3\x6c", [0x1efb] = "\xee\xbe",
  [0x1efc] = "\xdb\xf4", [0x1efd] = "\xdb\xfa", [0x1efe] = "\xb6\x5c",
  [0x1eff] = "\xf7\x60", [0x1f00] = "\xb6\x59", [0x1f01] = "\xdb\xf6",
  [0x1f02] = "\xb6\x5f", [0x1f03] = "\xb6\x60", [0x1f04] = "\xe0\x69",
  [0x1f05] = "\xc3\xdf", [0x1f06] = "\xc2\xde", [0x1f07] = "\xe0\x6e",
  [0x1f08] = "\xf7\x5f", [0x1f09] = "\xc5\x45", [0x1f0a] = "\xb8\xf8",
  [0x1f0b] = "\xe4\x57", [0x1f0c] = "\xe4\x5b", [0x1f0d] = "\xe8\x40",
  [0x1f0e] = "\xbd\xef", [0x1f0f] = "\xbd\xf3", [0x1f10] = "\xc5\xae",
  [0x1f11] = "\xe7\xf7", [0x1f12] = "\xe7\xf4", [0x1f13] = "\xc5\xad",
  [0x1f15] = "\xeb\xc0", [0x1f16] = "\xbf\xe5", [0x1f17] = "\xbf\xe6",
  [0x1f18] = "\xeb\xc4", [0x1f19] = "\xbf\xe2", [0x1f1a] = "\xc1\xce",
  [0x1f1b] = "\xeb\xc5", [0x1f1c] = "\xbf\xe4", [0x1f1d] = "\xc6\x5c",
  [0x1f1e] = "\xc2\xda", [0x1f1f] = "\xc1\xd1", [0x1f20] = "\xf7\x5d",
  [0x1f21] = "\xc3\xdd", [0x1f22] = "\xc3\xda", [0x1f23] = "\xf3\x53",
  [0x1f24] = "\xc3\xe0", [0x1f25] = "\xc4\xbd", [0x1f26] = "\xc6\x6e",
  [0x1f27] = "\xf7\x5e", [0x1f28] = "\xf7\x61", [0x1f29] = "\xf9\x70",
  [0x1f2a] = "\xf8\xc3", [0x1f2b] = "\xcb\xa4", [0x1f2c] = "\xb6\x49",
  [0x1f2d] = "\xb8\xe7", [0x1f2e] = "\xe0\x5a", [0x1f2f] = "\xf0\xf1",
  [0x1f30] = "\xee\xb7", [0x1f31] = "\xb1\xd8", [0x1f32] = "\xd7\xc6",
  [0x1f33] = "\xc2\xd2", [0x1f34] = "\xdb\xd4", [0x1f35] = "\xe0\x43",
  [0x1f36] = "\xdf\xfd", [0x1f37] = "\xe4\x48", [0x1f38] = "\xf2\xf5",
  [0x1f39] = "\xb8\xe4", [0x1f3a] = "\xf0\xe7", [0x1f3b] = "\xe8\xb0",
  [0x1f3c] = "\xb3\xb8", [0x1f3d] = "\xc6\x45", [0x1f3e] = "\xb6\xb2",
  [0x1f3f] = "\xbe\x5e", [0x1f40] = "\xc5\xb4", [0x1f41] = "\xe8\xae",
  [0x1f42] = "\xc0\x50", [0x1f43] = "\xc0\x4b", [0x1f44] = "\xc3\xf9",
  [0x1f45] = "\xc6\x47", [0x1f46] = "\xc4\xc5", [0x1f47] = "\xc5\xb5",
  [0x1f48] = "\xef\xb5", [0x1f49] = "\xc4\xd4", [0x1f4a] = "\xf5\xbe",
  [0x1f4b] = "\xf5\xbb", [0x1f4c] = "\xc5\x68", [0x1f4d] = "\xc5\x69",
  [0x1f4e] = "\xc5\xc5", [0x1f4f] = "\xc5\xc6", [0x1f50] = "\xc6\x53",
  [0x1f51] = "\xe0\xef", [0x1f52] = "\xef\xb1", [0x1f53] = "\xf9\x65",
  [0x1f54] = "\xab\x41", [0x1f55] = "\xd4\x47", [0x1f56] = "\xb9\x6d",
  [0x1f57] = "\xe0\xd5", [0x1f58] = "\xbb\xdc", [0x1f59] = "\xc2\xa3",
  [0x1f5a] = "\xf8\x41", [0x1f5b] = "\xe4\xa9", [0x1f5c] = "\xc6\x71",
  [0x1f5d] = "\xe8\x74", [0x1f5e] = "\xc3\xf5", [0x1f5f] = "\xee\xde",
  [0x1f60] = "\xf3\x6c", [0x1f61] = "\xee\xdc", [0x1f63] = "\xf8\xca",
  [0x1f64] = "\xbe\x7b", [0x1f65] = "\xe8\xdb", [0x1f68] = "\xf3\xce",
  [0x1f69] = "\xc6\x74", [0x1f6a] = "\xbf\x71", [0x1f6b] = "\xec\x7b",
  [0x1f6c] = "\xc6\x4d", [0x1f6d] = "\xec\x7c", [0x1f6e] = "\xc2\x44",
  [0x1f6f] = "\xef\x6a", [0x1f70] = "\xc2\x43", [0x1f71] = "\xef\x6c",
  [0x1f72] = "\xf9\x5c", [0x1f73] = "\xc2\x42", [0x1f74] = "\xf3\xd0",
  [0x1f75] = "\xf8\x58", [0x1f76] = "\xf1\xc7", [0x1f77] = "\xf9\xce",
  [0x1f78] = "\xc5\xbf", [0x1f79] = "\xf7\xae", [0x1f7a] = "\xf6\xad",
  [0x1f7b] = "\xf7\xb2", [0x1f7c] = "\xc3\x58", [0x1f7d] = "\xc3\x54",
  [0x1f7e] = "\xf1\xc3", [0x1f7f] = "\xc3\x56", [0x1f80] = "\xc4\x4a",
  [0x1f81] = "\xf3\xc7", [0x1f82] = "\xf3\xc8", [0x1f83] = "\xf3\xd6",
  [0x1f84] = "\xf3\xcb", [0x1f85] = "\xc4\x49", [0x1f87] = "\xf3\xcd",
  [0x1f88] = "\xf3\xce", [0x1f89] = "\xc4\x4b", [0x1f8b] = "\xf3\xcf",
  [0x1f8d] = "\xf5\x6b", [0x1f8e] = "\xc6\x73", [0x1f8f] = "\xc4\xcf",
  [0x1f90] = "\xf5\x6e", [0x1f91] = "\xf5\x76", [0x1f94] = "\xf7\xb4",
  [0x1f95] = "\xc5\x5f", [0x1f96] = "\xf6\xae", [0x1f97] = "\xc5\x60",
  [0x1f98] = "\xf6\xaf", [0x1f99] = "\xf7\xaf", [0x1f9a] = "\xc5\xc0",
  [0x1f9b] = "\xf7\xab", [0x1f9c] = "\xc5\xc1", [0x1f9f] = "\xc5\xed",
  [0x1fa0] = "\xc5\xeb", [0x1fa1] = "\xf8\x57", [0x1fa2] = "\xf8\xd7",
  [0x1fa3] = "\xbb\xde", [0x1fa4] = "\xbb\xdf", [0x1fa5] = "\xc5\xb6",
  [0x1fa6] = "\xbe\xf5", [0x1fa7] = "\xec\x56", [0x1fa8] = "\xc6\x69",
  [0x1fa9] = "\xf1\x7c", [0x1faa] = "\xc2\xfd", [0x1fad] = "\xe4\xda",
  [0x1fae] = "\xbb\xeb", [0x1faf] = "\xbe\x75", [0x1fb0] = "\xf6\xbc",
  [0x1fb3] = "\xc0\x66", [0x1fb4] = "\xc3\x4e", [0x1fb5] = "\xf1\xb9",
  [0x1fb6] = "\xc5\x5c", [0x1fb7] = "\xf3\xc1", [0x1fb8] = "\xc6\x62",
  [0x1fb9] = "\xf8\xd4", [0x1fba] = "\xc5\xea", [0x1fbb] = "\xbe\x79",
  [0x1fbc] = "\xe8\xd9", [0x1fbd] = "\xc6\x4c", [0x1fbe] = "\xc3\x52",
  [0x1fbf] = "\xef\x69", [0x1fc0] = "\xc3\x53", [0x1fc1] = "\xc5\x5e",
  [0x1fc2] = "\xc5\x57", [0x1fc3] = "\xc5\xe4", [0x1fc4] = "\xc3\x4b",
  [0x1fc5] = "\xc5\xb9", [0x1fc6] = "\xf7\x76", [0x1fc7] = "\xd4\x49",
  [0x1fc8] = "\xb9\xa6", [0x1fc9] = "\xbb\xec", [0x1fca] = "\xbe\x77",
  [0x1fcb] = "\xe8\xd4", [0x1fcc] = "\xc0\x67", [0x1fcd] = "\xc0\x68",
  [0x1fce] = "\xec\x70", [0x1fcf] = "\xf1\xbb", [0x1fd1] = "\xc6\x4b",
  [0x1fd2] = "\xf8\x55", [0x1fd3] = "\xc6\x63", [0x1fd4] = "\xbb\xf2",
  [0x1fd5] = "\xbe\xa3", [0x1fd6] = "\xed\xdd", [0x1fd7] = "\xb9\xa8",
  [0x1fd8] = "\xec\xb1", [0x1fd9] = "\xec\xaf", [0x1fda] = "\xc2\x47",
  [0x1fdb] = "\xc4\x51", [0x1fdc] = "\xc3\xef", [0x1fdd] = "\xc5\x65",
  [0x1fde] = "\xc5\xef", [0x1fdf] = "\xc2\x4c", [0x1fe0] = "\xc2\x4a",
  [0x1fe1] = "\xc2\x4b", [0x1fe2] = "\xc3\x5c", [0x1fe3] = "\xf1\xe0",
  [0x1fe5] = "\xc6\x75", [0x1fe6] = "\xf5\xb6", [0x1fe7] = "\xf5\xb4",
  [0x1fe8] = "\xf8\x76", [0x1fe9] = "\xc5\x66", [0x1fea] = "\xef\xb4",
  [0x1feb] = "\xc3\x5e", [0x1fec] = "\xc4\xd3", [0x1fed] = "\xc5\xc4",
  [0x1fee] = "\xf8\x7b", [0x1fef] = "\xec\xb8", [0x1ff0] = "\xc2\x4d",
};

const char __from_big5_to_gb2312 [13973][2] =
{
  [0x0000] = "\xa1\xa1", [0x0001] = "\xa3\xac", [0x0002] = "\xa1\xa2",
  [0x0003] = "\xa1\xa3", [0x0004] = "\xa3\xae", [0x0005] = "\xa1\xa4",
  [0x0006] = "\xa3\xbb", [0x0007] = "\xa3\xba", [0x0008] = "\xa3\xbf",
  [0x0009] = "\xa3\xa1", [0x000a] = "\xa1\xc3", [0x000b] = "\xa1\xad",
  [0x000c] = "\xa1\xa7", [0x000d] = "\xa3\xac", [0x000e] = "\xa1\xa2",
  [0x000f] = "\xa3\xae", [0x0010] = "\xa1\xa4", [0x0011] = "\xa3\xbb",
  [0x0012] = "\xa3\xba", [0x0013] = "\xa3\xbf", [0x0014] = "\xa3\xa1",
  [0x0015] = "\xa3\xfc", [0x0017] = "\xa3\xfc", [0x0019] = "\xa3\xfc",
  [0x001a] = "\xa3\xdf", [0x001d] = "\xa3\xa8", [0x001e] = "\xa3\xa9",
  [0x0021] = "\xa3\xfb", [0x0022] = "\xa3\xfd", [0x0025] = "\xa1\xb2",
  [0x0026] = "\xa1\xb3", [0x0029] = "\xa1\xbe", [0x002a] = "\xa1\xbf",
  [0x002d] = "\xa1\xb6", [0x002e] = "\xa1\xb7", [0x0031] = "\xa1\xb4",
  [0x0032] = "\xa1\xb5", [0x0033] = "\xa1\xc4", [0x0034] = "\xa1\xc5",
  [0x0035] = "\xa1\xb8", [0x0036] = "\xa1\xb9", [0x0039] = "\xa1\xba",
  [0x003a] = "\xa1\xbb", [0x003d] = "\xa3\xa8", [0x003e] = "\xa3\xa9",
  [0x003f] = "\xa3\xfb", [0x0040] = "\xa3\xfd", [0x0041] = "\xa3\xdb",
  [0x0042] = "\xa3\xdd", [0x0043] = "\xa1\xae", [0x0044] = "\xa1\xaf",
  [0x0045] = "\xa1\xb0", [0x0046] = "\xa1\xb1", [0x0047] = "\xa3\xa2",
  [0x0048] = "\xa1\xe5", [0x0049] = "\xa3\xe0", [0x004a] = "\xa1\xe4",
  [0x004b] = "\xa3\xa3", [0x004c] = "\xa3\xa6", [0x004d] = "\xa3\xaa",
  [0x004e] = "\xa1\xf9", [0x004f] = "\xa1\xec", [0x0050] = "\xa1\xa8",
  [0x0051] = "\xa1\xf0", [0x0052] = "\xa1\xf1", [0x0053] = "\xa1\xf7",
  [0x0054] = "\xa1\xf8", [0x0055] = "\xa1\xf2", [0x0056] = "\xa1\xee",
  [0x0057] = "\xa1\xef", [0x0058] = "\xa1\xf3", [0x0059] = "\xa1\xf4",
  [0x005a] = "\xa1\xf5", [0x005b] = "\xa1\xf6", [0x0060] = "\xa1\xa5",
  [0x0061] = "\xa3\xfe", [0x0062] = "\xa3\xdf", [0x006a] = "\xa3\xa3",
  [0x006b] = "\xa3\xa6", [0x006c] = "\xa3\xaa", [0x006d] = "\xa3\xab",
  [0x006e] = "\xa3\xad", [0x006f] = "\xa1\xc1", [0x0070] = "\xa1\xc2",
  [0x0071] = "\xa1\xc0", [0x0072] = "\xa1\xcc", [0x0073] = "\xa3\xbc",
  [0x0074] = "\xa3\xbe", [0x0075] = "\xa3\xbd", [0x0076] = "\xa1\xdc",
  [0x0077] = "\xa1\xdd", [0x0078] = "\xa1\xd9", [0x0079] = "\xa1\xde",
  [0x007a] = "\xa1\xd6", [0x007b] = "\xa1\xd4", [0x007c] = "\xa3\xab",
  [0x007d] = "\xa3\xad", [0x007e] = "\xa3\xbc", [0x007f] = "\xa3\xbe",
  [0x0080] = "\xa3\xbd", [0x0081] = "\xa1\xab", [0x0082] = "\xa1\xc9",
  [0x0083] = "\xa1\xc8", [0x0084] = "\xa1\xcd", [0x0085] = "\xa1\xcf",
  [0x008a] = "\xa1\xd2", [0x008b] = "\xa1\xd3", [0x008c] = "\xa1\xdf",
  [0x008d] = "\xa1\xe0", [0x008e] = "\xa1\xe2", [0x008f] = "\xa1\xe1",
  [0x0091] = "\xa1\xd1", [0x0092] = "\xa1\xfc", [0x0093] = "\xa1\xfd",
  [0x0094] = "\xa1\xfb", [0x0095] = "\xa1\xfa", [0x009a] = "\xa1\xac",
  [0x009c] = "\xa3\xaf", [0x009d] = "\xa3\xdc", [0x00a0] = "\xa1\xe7",
  [0x00a1] = "\xa3\xa4", [0x00a3] = "\xa1\xe9", [0x00a4] = "\xa1\xea",
  [0x00a5] = "\xa3\xa5", [0x00a6] = "\xa3\xc0", [0x00a7] = "\xa1\xe6",
  [0x00a9] = "\xa1\xe7", [0x00aa] = "\xa3\xa5", [0x00ab] = "\xa3\xc0",
  [0x00b5] = "\xa1\xe3", [0x00ce] = "\xa9\xe0", [0x00cf] = "\xa9\xd8",
  [0x00d0] = "\xa9\xd0", [0x00d1] = "\xa9\xc8", [0x00d2] = "\xa9\xc0",
  [0x00d4] = "\xa1\xaa", [0x00d5] = "\xa9\xa6", [0x00d7] = "\xa9\xb0",
  [0x00d8] = "\xa9\xb4", [0x00d9] = "\xa9\xb8", [0x00da] = "\xa9\xbc",
  [0x00e7] = "\xa3\xaf", [0x00e8] = "\xa3\xdc", [0x00ea] = "\xa3\xb0",
  [0x00eb] = "\xa3\xb1", [0x00ec] = "\xa3\xb2", [0x00ed] = "\xa3\xb3",
  [0x00ee] = "\xa3\xb4", [0x00ef] = "\xa3\xb5", [0x00f0] = "\xa3\xb6",
  [0x00f1] = "\xa3\xb7", [0x00f2] = "\xa3\xb8", [0x00f3] = "\xa3\xb9",
  [0x00f4] = "\xa2\xf1", [0x00f5] = "\xa2\xf2", [0x00f6] = "\xa2\xf3",
  [0x00f7] = "\xa2\xf4", [0x00f8] = "\xa2\xf5", [0x00f9] = "\xa2\xf6",
  [0x00fa] = "\xa2\xf7", [0x00fb] = "\xa2\xf8", [0x00fc] = "\xa2\xf9",
  [0x00fd] = "\xa2\xfa", [0x0108] = "\xd8\xa5", [0x010a] = "\xa3\xc1",
  [0x010b] = "\xa3\xc2", [0x010c] = "\xa3\xc3", [0x010d] = "\xa3\xc4",
  [0x010e] = "\xa3\xc5", [0x010f] = "\xa3\xc6", [0x0110] = "\xa3\xc7",
  [0x0111] = "\xa3\xc8", [0x0112] = "\xa3\xc9", [0x0113] = "\xa3\xca",
  [0x0114] = "\xa3\xcb", [0x0115] = "\xa3\xcc", [0x0116] = "\xa3\xcd",
  [0x0117] = "\xa3\xce", [0x0118] = "\xa3\xcf", [0x0119] = "\xa3\xd0",
  [0x011a] = "\xa3\xd1", [0x011b] = "\xa3\xd2", [0x011c] = "\xa3\xd3",
  [0x011d] = "\xa3\xd4", [0x011e] = "\xa3\xd5", [0x011f] = "\xa3\xd6",
  [0x0120] = "\xa3\xd7", [0x0121] = "\xa3\xd8", [0x0122] = "\xa3\xd9",
  [0x0123] = "\xa3\xda", [0x0124] = "\xa3\xe1", [0x0125] = "\xa3\xe2",
  [0x0126] = "\xa3\xe3", [0x0127] = "\xa3\xe4", [0x0128] = "\xa3\xe5",
  [0x0129] = "\xa3\xe6", [0x012a] = "\xa3\xe7", [0x012b] = "\xa3\xe8",
  [0x012c] = "\xa3\xe9", [0x012d] = "\xa3\xea", [0x012e] = "\xa3\xeb",
  [0x012f] = "\xa3\xec", [0x0130] = "\xa3\xed", [0x0131] = "\xa3\xee",
  [0x0132] = "\xa3\xef", [0x0133] = "\xa3\xf0", [0x0134] = "\xa3\xf1",
  [0x0135] = "\xa3\xf2", [0x0136] = "\xa3\xf3", [0x0137] = "\xa3\xf4",
  [0x0138] = "\xa3\xf5", [0x0139] = "\xa3\xf6", [0x013a] = "\xa3\xf7",
  [0x013b] = "\xa3\xf8", [0x013c] = "\xa3\xf9", [0x013d] = "\xa3\xfa",
  [0x013e] = "\xa6\xa1", [0x013f] = "\xa6\xa2", [0x0140] = "\xa6\xa3",
  [0x0141] = "\xa6\xa4", [0x0142] = "\xa6\xa5", [0x0143] = "\xa6\xa6",
  [0x0144] = "\xa6\xa7", [0x0145] = "\xa6\xa8", [0x0146] = "\xa6\xa9",
  [0x0147] = "\xa6\xaa", [0x0148] = "\xa6\xab", [0x0149] = "\xa6\xac",
  [0x014a] = "\xa6\xad", [0x014b] = "\xa6\xae", [0x014c] = "\xa6\xaf",
  [0x014d] = "\xa1\xc7", [0x014e] = "\xa6\xb1", [0x014f] = "\xa1\xc6",
  [0x0150] = "\xa6\xb3", [0x0151] = "\xa6\xb4", [0x0152] = "\xa6\xb5",
  [0x0153] = "\xa6\xb6", [0x0154] = "\xa6\xb7", [0x0155] = "\xa6\xb8",
  [0x0156] = "\xa6\xc1", [0x0157] = "\xa6\xc2", [0x0158] = "\xa6\xc3",
  [0x0159] = "\xa6\xc4", [0x015a] = "\xa6\xc5", [0x015b] = "\xa6\xc6",
  [0x015c] = "\xa6\xc7", [0x015d] = "\xa6\xc8", [0x015e] = "\xa6\xc9",
  [0x015f] = "\xa6\xca", [0x0160] = "\xa6\xcb", [0x0161] = "\xa6\xcc",
  [0x0162] = "\xa6\xcd", [0x0163] = "\xa6\xce", [0x0164] = "\xa6\xcf",
  [0x0165] = "\xa6\xd0", [0x0166] = "\xa6\xd1", [0x0167] = "\xa6\xd2",
  [0x0168] = "\xa6\xd3", [0x0169] = "\xa6\xd4", [0x016a] = "\xa6\xd5",
  [0x016b] = "\xa6\xd6", [0x016c] = "\xa6\xd7", [0x016d] = "\xa6\xd8",
  [0x016e] = "\xa8\xc5", [0x016f] = "\xa8\xc6", [0x0170] = "\xa8\xc7",
  [0x0171] = "\xa8\xc8", [0x0172] = "\xa8\xc9", [0x0173] = "\xa8\xca",
  [0x0174] = "\xa8\xcb", [0x0175] = "\xa8\xcc", [0x0176] = "\xa8\xcd",
  [0x0177] = "\xa8\xce", [0x0178] = "\xa8\xcf", [0x0179] = "\xa8\xd0",
  [0x017a] = "\xa8\xd1", [0x017b] = "\xa8\xd2", [0x017c] = "\xa8\xd3",
  [0x017d] = "\xa8\xd4", [0x017e] = "\xa8\xd5", [0x017f] = "\xa8\xd6",
  [0x0180] = "\xa8\xd7", [0x0181] = "\xa8\xd8", [0x0182] = "\xa8\xd9",
  [0x0183] = "\xa8\xda", [0x0184] = "\xa8\xdb", [0x0185] = "\xa8\xdc",
  [0x0186] = "\xa8\xdd", [0x0187] = "\xa8\xde", [0x0188] = "\xa8\xdf",
  [0x0189] = "\xa8\xe0", [0x018a] = "\xa8\xe1", [0x018b] = "\xa8\xe2",
  [0x018c] = "\xa8\xe3", [0x018d] = "\xa8\xe4", [0x018e] = "\xa8\xe5",
  [0x018f] = "\xa8\xe6", [0x0190] = "\xa8\xe7", [0x0191] = "\xa8\xe8",
  [0x0192] = "\xa8\xe9", [0x0194] = "\xa1\xa5", [0x0196] = "\xa1\xa6",
  [0x01d7] = "\xd2\xbb", [0x01d8] = "\xd2\xd2", [0x01d9] = "\xb6\xa1",
  [0x01da] = "\xc6\xdf", [0x01db] = "\xc4\xcb", [0x01dc] = "\xbe\xc5",
  [0x01dd] = "\xc1\xcb", [0x01de] = "\xb6\xfe", [0x01df] = "\xc8\xcb",
  [0x01e0] = "\xb6\xf9", [0x01e1] = "\xc8\xeb", [0x01e2] = "\xb0\xcb",
  [0x01e3] = "\xbc\xb8", [0x01e4] = "\xb5\xb6", [0x01e5] = "\xb5\xf3",
  [0x01e6] = "\xc1\xa6", [0x01e7] = "\xd8\xb0", [0x01e8] = "\xca\xae",
  [0x01e9] = "\xb2\xb7", [0x01ea] = "\xd3\xd6", [0x01eb] = "\xc8\xfd",
  [0x01ec] = "\xcf\xc2", [0x01ed] = "\xd5\xc9", [0x01ee] = "\xc9\xcf",
  [0x01ef] = "\xd1\xbe", [0x01f0] = "\xcd\xe8", [0x01f1] = "\xb7\xb2",
  [0x01f2] = "\xbe\xc3", [0x01f3] = "\xc3\xb4", [0x01f4] = "\xd2\xb2",
  [0x01f5] = "\xc6\xf2", [0x01f6] = "\xd3\xda", [0x01f7] = "\xcd\xf6",
  [0x01f8] = "\xd8\xa3", [0x01f9] = "\xc8\xd0", [0x01fa] = "\xc9\xd7",
  [0x01fb] = "\xc7\xa7", [0x01fc] = "\xb2\xe6", [0x01fd] = "\xbf\xda",
  [0x01fe] = "\xcd\xc1", [0x01ff] = "\xca\xbf", [0x0200] = "\xcf\xa6",
  [0x0201] = "\xb4\xf3", [0x0202] = "\xc5\xae", [0x0203] = "\xd7\xd3",
  [0x0204] = "\xe6\xdd", [0x0205] = "\xe6\xde", [0x0206] = "\xb4\xe7",
  [0x0207] = "\xd0\xa1", [0x0208] = "\xde\xcc", [0x0209] = "\xca\xac",
  [0x020a] = "\xc9\xbd", [0x020b] = "\xb4\xa8", [0x020c] = "\xb9\xa4",
  [0x020d] = "\xbc\xba", [0x020e] = "\xd2\xd1", [0x020f] = "\xcb\xc8",
  [0x0210] = "\xbd\xed", [0x0211] = "\xb8\xc9", [0x0212] = "\xde\xc3",
  [0x0213] = "\xdf\xae", [0x0214] = "\xb9\xad", [0x0215] = "\xb2\xc5",
  [0x0216] = "\xb3\xf3", [0x0217] = "\xd8\xa4", [0x0218] = "\xb2\xbb",
  [0x0219] = "\xd6\xd0", [0x021a] = "\xb7\xe1", [0x021b] = "\xb5\xa4",
  [0x021c] = "\xd6\xae", [0x021d] = "\xd2\xfc", [0x021e] = "\xd3\xe8",
  [0x021f] = "\xd4\xc6", [0x0220] = "\xbe\xae", [0x0221] = "\xbb\xa5",
  [0x0222] = "\xce\xe5", [0x0223] = "\xbf\xba", [0x0224] = "\xc8\xca",
  [0x0225] = "\xca\xb2", [0x0226] = "\xd8\xea", [0x0227] = "\xc6\xcd",
  [0x0228] = "\xb3\xf0", [0x0229] = "\xc8\xd4", [0x022a] = "\xbd\xf1",
  [0x022b] = "\xbd\xe9", [0x022c] = "\xd8\xc6", [0x022d] = "\xd4\xaa",
  [0x022e] = "\xd4\xca", [0x022f] = "\xc4\xda", [0x0230] = "\xc1\xf9",
  [0x0231] = "\xd9\xe2", [0x0232] = "\xb9\xab", [0x0233] = "\xc8\xdf",
  [0x0234] = "\xd0\xd7", [0x0235] = "\xb7\xd6", [0x0236] = "\xc7\xd0",
  [0x0237] = "\xd8\xd7", [0x0238] = "\xd4\xc8", [0x0239] = "\xb9\xb4",
  [0x023a] = "\xce\xf0", [0x023b] = "\xbb\xaf", [0x023c] = "\xc6\xa5",
  [0x023d] = "\xce\xe7", [0x023e] = "\xc9\xfd", [0x023f] = "\xd8\xa6",
  [0x0240] = "\xb1\xe5", [0x0241] = "\xb6\xf2", [0x0242] = "\xd3\xd1",
  [0x0243] = "\xbc\xb0", [0x0244] = "\xb7\xb4", [0x0245] = "\xc8\xc9",
  [0x0246] = "\xcc\xec", [0x0247] = "\xb7\xf2", [0x0248] = "\xcc\xab",
  [0x0249] = "\xd8\xb2", [0x024a] = "\xbf\xd7", [0x024b] = "\xc9\xd9",
  [0x024c] = "\xd3\xc8", [0x024d] = "\xb3\xdf", [0x024e] = "\xcd\xcd",
  [0x024f] = "\xb0\xcd", [0x0250] = "\xbb\xc3", [0x0251] = "\xd8\xa5",
  [0x0252] = "\xb5\xf5", [0x0253] = "\xd2\xfd", [0x0254] = "\xd0\xc4",
  [0x0255] = "\xb8\xea", [0x0256] = "\xbb\xa7", [0x0257] = "\xca\xd6",
  [0x0258] = "\xd4\xfa", [0x0259] = "\xd6\xa7", [0x025a] = "\xce\xc4",
  [0x025b] = "\xb6\xb7", [0x025c] = "\xbd\xef", [0x025d] = "\xb7\xbd",
  [0x025e] = "\xc8\xd5", [0x025f] = "\xd4\xbb", [0x0260] = "\xd4\xc2",
  [0x0261] = "\xc4\xbe", [0x0262] = "\xc7\xb7", [0x0263] = "\xd6\xb9",
  [0x0264] = "\xb4\xf5", [0x0265] = "\xce\xe3", [0x0266] = "\xb1\xc8",
  [0x0267] = "\xc3\xab", [0x0268] = "\xca\xcf", [0x0269] = "\xcb\xae",
  [0x026a] = "\xbb\xf0", [0x026b] = "\xd7\xa6", [0x026c] = "\xb8\xb8",
  [0x026d] = "\xd8\xb3", [0x026e] = "\xc6\xac", [0x026f] = "\xd1\xc0",
  [0x0270] = "\xc5\xa3", [0x0271] = "\xc8\xae", [0x0272] = "\xcd\xf5",
  [0x0273] = "\xb1\xfb", [0x0274] = "\xca\xc0", [0x0275] = "\xd8\xa7",
  [0x0276] = "\xc7\xd2", [0x0277] = "\xc7\xf0", [0x0278] = "\xd6\xf7",
  [0x0279] = "\xd5\xa7", [0x027a] = "\xb7\xa6", [0x027b] = "\xba\xf5",
  [0x027c] = "\xd2\xd4", [0x027d] = "\xb8\xb6", [0x027e] = "\xd7\xd0",
  [0x027f] = "\xca\xcb", [0x0280] = "\xcb\xfb", [0x0281] = "\xd5\xcc",
  [0x0282] = "\xb4\xfa", [0x0283] = "\xc1\xee", [0x0284] = "\xcf\xc9",
  [0x0285] = "\xd8\xf0", [0x0286] = "\xb3\xe4", [0x0287] = "\xd0\xd6",
  [0x0288] = "\xc8\xbd", [0x0289] = "\xb2\xe1", [0x028a] = "\xb6\xac",
  [0x028b] = "\xb0\xbc", [0x028c] = "\xb3\xf6", [0x028d] = "\xcd\xb9",
  [0x028e] = "\xbf\xaf", [0x028f] = "\xbc\xd3", [0x0290] = "\xb9\xa6",
  [0x0291] = "\xb0\xfc", [0x0292] = "\xb4\xd2", [0x0293] = "\xb1\xb1",
  [0x0294] = "\xd4\xd1", [0x0295] = "\xc7\xaa", [0x0296] = "\xb0\xeb",
  [0x0297] = "\xbb\xdc", [0x0298] = "\xbf\xa8", [0x0299] = "\xd5\xbc",
  [0x029a] = "\xc3\xae", [0x029b] = "\xd8\xb4", [0x029c] = "\xc8\xa5",
  [0x029d] = "\xbf\xc9", [0x029e] = "\xb9\xc5", [0x029f] = "\xd3\xd2",
  [0x02a0] = "\xd5\xd9", [0x02a1] = "\xb6\xa3", [0x02a2] = "\xdf\xb5",
  [0x02a3] = "\xdf\xb6", [0x02a4] = "\xb5\xf0", [0x02a5] = "\xcb\xbe",
  [0x02a6] = "\xd8\xcf", [0x02a7] = "\xbd\xd0", [0x02a8] = "\xc1\xed",
  [0x02a9] = "\xd6\xbb", [0x02aa] = "\xca\xb7", [0x02ab] = "\xdf\xb3",
  [0x02ac] = "\xcc\xa8", [0x02ad] = "\xbe\xe4", [0x02ae] = "\xb0\xc8",
  [0x02af] = "\xdf\xb7", [0x02b0] = "\xcb\xc4", [0x02b1] = "\xc7\xf4",
  [0x02b2] = "\xcd\xe2", [0x02b3] = "\xd1\xeb", [0x02b4] = "\xca\xa7",
  [0x02b5] = "\xc5\xab", [0x02b6] = "\xc4\xcc", [0x02b7] = "\xd4\xd0",
  [0x02b8] = "\xcb\xfc", [0x02b9] = "\xc4\xe1", [0x02ba] = "\xbe\xde",
  [0x02bb] = "\xc7\xc9", [0x02bc] = "\xd7\xf3", [0x02bd] = "\xca\xd0",
  [0x02be] = "\xb2\xbc", [0x02bf] = "\xc6\xbd", [0x02c0] = "\xd3\xd7",
  [0x02c1] = "\xdb\xcd", [0x02c2] = "\xba\xeb", [0x02c3] = "\xb8\xa5",
  [0x02c4] = "\xb1\xd8", [0x02c5] = "\xce\xec", [0x02c6] = "\xb4\xf2",
  [0x02c7] = "\xc8\xd3", [0x02c8] = "\xb0\xc7", [0x02c9] = "\xc6\xcb",
  [0x02ca] = "\xb3\xe2", [0x02cb] = "\xb5\xa9", [0x02cc] = "\xca\xf5",
  [0x02cd] = "\xb1\xbe", [0x02ce] = "\xce\xb4", [0x02cf] = "\xc4\xa9",
  [0x02d0] = "\xd4\xfd", [0x02d1] = "\xd5\xfd", [0x02d2] = "\xc4\xb8",
  [0x02d3] = "\xc3\xf1", [0x02d4] = "\xd8\xb5", [0x02d5] = "\xd3\xc0",
  [0x02d6] = "\xd6\xad", [0x02d7] = "\xcd\xa1", [0x02d8] = "\xb7\xba",
  [0x02d9] = "\xb7\xb8", [0x02da] = "\xd0\xfe", [0x02db] = "\xd3\xf1",
  [0x02dc] = "\xb9\xcf", [0x02dd] = "\xcd\xdf", [0x02de] = "\xb8\xca",
  [0x02df] = "\xc9\xfa", [0x02e0] = "\xd3\xc3", [0x02e1] = "\xcb\xa6",
  [0x02e2] = "\xcc\xef", [0x02e3] = "\xd3\xc9", [0x02e4] = "\xbc\xd7",
  [0x02e5] = "\xc9\xea", [0x02e6] = "\xf1\xe2", [0x02e7] = "\xb0\xd7",
  [0x02e8] = "\xc6\xa4", [0x02e9] = "\xc3\xf3", [0x02ea] = "\xc4\xbf",
  [0x02eb] = "\xc3\xac", [0x02ec] = "\xca\xb8", [0x02ed] = "\xca\xaf",
  [0x02ee] = "\xca\xbe", [0x02ef] = "\xba\xcc", [0x02f0] = "\xd1\xa8",
  [0x02f1] = "\xc1\xa2", [0x02f2] = "\xd8\xa9", [0x02f3] = "\xb6\xaa",
  [0x02f4] = "\xc6\xb9", [0x02f5] = "\xc5\xd2", [0x02f6] = "\xd8\xc0",
  [0x02f7] = "\xd8\xa8", [0x02f8] = "\xbd\xbb", [0x02f9] = "\xd2\xe0",
  [0x02fa] = "\xba\xa5", [0x02fb] = "\xb7\xc2", [0x02fc] = "\xd8\xf8",
  [0x02fd] = "\xbb\xef", [0x02fe] = "\xd2\xc1", [0x02ff] = "\xb7\xf2",
  [0x0300] = "\xce\xe9", [0x0301] = "\xb7\xa5", [0x0302] = "\xd0\xdd",
  [0x0303] = "\xb7\xfc", [0x0304] = "\xd6\xd9", [0x0305] = "\xbc\xfe",
  [0x0306] = "\xc8\xce", [0x0307] = "\xd1\xf6", [0x0308] = "\xd8\xf2",
  [0x0309] = "\xb7\xdd", [0x030a] = "\xc6\xf3", [0x030c] = "\xb9\xe2",
  [0x030d] = "\xd0\xd7", [0x030e] = "\xd5\xd7", [0x030f] = "\xcf\xc8",
  [0x0310] = "\xc8\xab", [0x0311] = "\xb9\xb2", [0x0312] = "\xd4\xd9",
  [0x0313] = "\xb1\xf9", [0x0314] = "\xc1\xd0", [0x0315] = "\xd0\xcc",
  [0x0316] = "\xbb\xae", [0x0317] = "\xd8\xd8", [0x0318] = "\xeb\xbe",
  [0x0319] = "\xc1\xd3", [0x031a] = "\xd0\xd9", [0x031b] = "\xbf\xef",
  [0x031c] = "\xbd\xb3", [0x031d] = "\xd3\xa1", [0x031e] = "\xce\xa3",
  [0x031f] = "\xbc\xaa", [0x0320] = "\xc0\xf4", [0x0321] = "\xcd\xac",
  [0x0322] = "\xb5\xf5", [0x0323] = "\xcd\xc2", [0x0324] = "\xd3\xf5",
  [0x0325] = "\xb4\xe7", [0x0326] = "\xb8\xf7", [0x0327] = "\xcf\xf2",
  [0x0328] = "\xc3\xfb", [0x0329] = "\xba\xcf", [0x032a] = "\xb3\xd4",
  [0x032b] = "\xba\xf3", [0x032c] = "\xdf\xba", [0x032d] = "\xdf\xb8",
  [0x032e] = "\xd2\xf2", [0x032f] = "\xbb\xd8", [0x0330] = "\xe0\xee",
  [0x0331] = "\xdb\xda", [0x0332] = "\xb5\xd8", [0x0333] = "\xd4\xda",
  [0x0334] = "\xb9\xe7", [0x0335] = "\xdb\xd8", [0x0336] = "\xdb\xdd",
  [0x0337] = "\xdb\xd7", [0x0338] = "\xd9\xed", [0x0339] = "\xb6\xe0",
  [0x033a] = "\xd2\xc4", [0x033b] = "\xbf\xe4", [0x033c] = "\xcd\xfd",
  [0x033d] = "\xbc\xe9", [0x033e] = "\xe5\xfa", [0x033f] = "\xba\xc3",
  [0x0340] = "\xcb\xfd", [0x0341] = "\xc8\xe7", [0x0342] = "\xe5\xf9",
  [0x0343] = "\xd7\xd6", [0x0344] = "\xb4\xe6", [0x0345] = "\xd3\xee",
  [0x0346] = "\xca\xd8", [0x0347] = "\xd5\xac", [0x0348] = "\xb0\xb2",
  [0x0349] = "\xcb\xc2", [0x034a] = "\xbc\xe2", [0x034b] = "\xd2\xd9",
  [0x034c] = "\xd6\xdd", [0x034d] = "\xb7\xab", [0x034e] = "\xb2\xa2",
  [0x034f] = "\xc4\xea", [0x0350] = "\xca\xbd", [0x0351] = "\xb3\xda",
  [0x0352] = "\xc3\xa6", [0x0353] = "\xe2\xe2", [0x0354] = "\xc8\xd6",
  [0x0355] = "\xd0\xe7", [0x0356] = "\xca\xf9", [0x0357] = "\xb3\xc9",
  [0x0358] = "\xbf\xdb", [0x0359] = "\xbf\xb8", [0x035a] = "\xcd\xd0",
  [0x035b] = "\xca\xd5", [0x035c] = "\xd4\xe7", [0x035d] = "\xd6\xbc",
  [0x035e] = "\xd1\xae", [0x035f] = "\xd0\xf1", [0x0360] = "\xc7\xfa",
  [0x0361] = "\xd2\xb7", [0x0362] = "\xd3\xd0", [0x0363] = "\xd0\xe0",
  [0x0364] = "\xc6\xd3", [0x0365] = "\xd6\xec", [0x0366] = "\xb6\xe4",
  [0x0367] = "\xb4\xce", [0x0368] = "\xb4\xcb", [0x0369] = "\xcb\xc0",
  [0x036a] = "\xc4\xca", [0x036b] = "\xc8\xea", [0x036c] = "\xba\xb9",
  [0x036d] = "\xce\xdb", [0x036e] = "\xbd\xad", [0x036f] = "\xb3\xd8",
  [0x0370] = "\xcf\xab", [0x0371] = "\xc9\xc7", [0x0372] = "\xce\xdb",
  [0x0373] = "\xd1\xb4", [0x0374] = "\xcd\xe8", [0x0375] = "\xb7\xba",
  [0x0376] = "\xbb\xd2", [0x0377] = "\xc4\xb2", [0x0378] = "\xea\xf2",
  [0x0379] = "\xb0\xd9", [0x037a] = "\xd6\xf1", [0x037b] = "\xc3\xd7",
  [0x037c] = "\xf4\xe9", [0x037d] = "\xf3\xbe", [0x037e] = "\xd1\xf2",
  [0x037f] = "\xd3\xf0", [0x0380] = "\xc0\xcf", [0x0381] = "\xbf\xbc",
  [0x0382] = "\xb6\xf8", [0x0383] = "\xf1\xe7", [0x0384] = "\xb6\xfa",
  [0x0385] = "\xed\xb2", [0x0386] = "\xc8\xe2", [0x0387] = "\xc0\xdf",
  [0x0388] = "\xbc\xa1", [0x0389] = "\xb3\xbc", [0x038a] = "\xd7\xd4",
  [0x038b] = "\xd6\xc1", [0x038c] = "\xbe\xca", [0x038d] = "\xc9\xe0",
  [0x038e] = "\xe2\xb6", [0x038f] = "\xd6\xdb", [0x0390] = "\xf4\xde",
  [0x0391] = "\xc9\xab", [0x0392] = "\xb0\xac", [0x0393] = "\xb3\xe6",
  [0x0394] = "\xd1\xaa", [0x0395] = "\xd0\xd0", [0x0396] = "\xd2\xc2",
  [0x0397] = "\xce\xf7", [0x0398] = "\xda\xe4", [0x0399] = "\xb4\xae",
  [0x039a] = "\xba\xe0", [0x039b] = "\xce\xbb", [0x039c] = "\xd7\xa1",
  [0x039d] = "\xd8\xf9", [0x039e] = "\xd9\xa2", [0x039f] = "\xd8\xfa",
  [0x03a0] = "\xb0\xe9", [0x03a1] = "\xb7\xf0", [0x03a2] = "\xba\xce",
  [0x03a3] = "\xb9\xc0", [0x03a4] = "\xd7\xf4", [0x03a5] = "\xd3\xd3",
  [0x03a6] = "\xd9\xa4", [0x03a7] = "\xcb\xc5", [0x03a8] = "\xc9\xec",
  [0x03a9] = "\xb5\xe8", [0x03aa] = "\xd5\xbc", [0x03ab] = "\xcb\xc6",
  [0x03ac] = "\xb5\xab", [0x03ad] = "\xd3\xb6", [0x03ae] = "\xd7\xf7",
  [0x03af] = "\xc4\xe3", [0x03b0] = "\xb2\xae", [0x03b1] = "\xb5\xcd",
  [0x03b2] = "\xc1\xe6", [0x03b3] = "\xd3\xe0", [0x03b4] = "\xd8\xfe",
  [0x03b5] = "\xb2\xbc", [0x03b6] = "\xd8\xfd", [0x03b7] = "\xb6\xd2",
  [0x03b8] = "\xbf\xcb", [0x03b9] = "\xc3\xe2", [0x03ba] = "\xb1\xf8",
  [0x03bb] = "\xd2\xb1", [0x03bc] = "\xc0\xe4", [0x03bd] = "\xb1\xf0",
  [0x03be] = "\xc5\xd0", [0x03bf] = "\xc0\xfb", [0x03c0] = "\xc9\xbe",
  [0x03c1] = "\xc5\xd9", [0x03c2] = "\xbd\xd9", [0x03c3] = "\xd6\xfa",
  [0x03c4] = "\xc5\xac", [0x03c5] = "\xdb\xbe", [0x03c6] = "\xcf\xbb",
  [0x03c7] = "\xbc\xb4", [0x03c8] = "\xc2\xd1", [0x03c9] = "\xc1\xdf",
  [0x03ca] = "\xbf\xd4", [0x03cb] = "\xcd\xcc", [0x03cc] = "\xce\xe1",
  [0x03cd] = "\xb7\xf1", [0x03ce] = "\xb3\xdf", [0x03cf] = "\xb0\xc9",
  [0x03d0] = "\xb4\xf4", [0x03d1] = "\xdf\xc0", [0x03d2] = "\xce\xe2",
  [0x03d3] = "\xb3\xca", [0x03d4] = "\xc2\xc0", [0x03d5] = "\xbe\xfd",
  [0x03d6] = "\xb7\xd4", [0x03d7] = "\xb8\xe6", [0x03d8] = "\xb4\xb5",
  [0x03d9] = "\xce\xc7", [0x03da] = "\xce\xfc", [0x03db] = "\xcb\xb1",
  [0x03dc] = "\xb3\xb3", [0x03dd] = "\xc4\xc5", [0x03de] = "\xb7\xcd",
  [0x03df] = "\xba\xf0", [0x03e0] = "\xd1\xbd", [0x03e1] = "\xd6\xa8",
  [0x03e2] = "\xba\xac", [0x03e3] = "\xd2\xf7", [0x03e4] = "\xcc\xfd",
  [0x03e5] = "\xb4\xd1", [0x03e6] = "\xc0\xa7", [0x03e7] = "\xb6\xda",
  [0x03e8] = "\xe0\xf1", [0x03e9] = "\xb7\xbb", [0x03ea] = "\xbf\xd3",
  [0x03eb] = "\xd6\xb7", [0x03ec] = "\xcc\xae", [0x03ed] = "\xbe\xf9",
  [0x03ee] = "\xbf\xb2", [0x03ef] = "\xbb\xf8", [0x03f0] = "\xd7\xf8",
  [0x03f1] = "\xbb\xb5", [0x03f2] = "\xdb\xdf", [0x03f3] = "\xd7\xb3",
  [0x03f4] = "\xbc\xd0", [0x03f5] = "\xd7\xb1", [0x03f6] = "\xb6\xca",
  [0x03f7] = "\xb7\xc1", [0x03f8] = "\xe6\xa4", [0x03f9] = "\xe5\xfe",
  [0x03fa] = "\xc3\xee", [0x03fb] = "\xd1\xfd", [0x03fc] = "\xe5\xfb",
  [0x03fd] = "\xe6\xa5", [0x03fe] = "\xbc\xcb", [0x03ff] = "\xc8\xd1",
  [0x0400] = "\xcd\xd7", [0x0401] = "\xd0\xa2", [0x0402] = "\xd7\xce",
  [0x0403] = "\xe6\xda", [0x0404] = "\xd8\xc3", [0x0405] = "\xcd\xea",
  [0x0406] = "\xcb\xce", [0x0407] = "\xba\xea", [0x0408] = "\xde\xce",
  [0x0409] = "\xbe\xd6", [0x040a] = "\xc6\xa8", [0x040b] = "\xc4\xf2",
  [0x040c] = "\xce\xb2", [0x040d] = "\xe1\xaa", [0x040e] = "\xe1\xaf",
  [0x040f] = "\xb2\xed", [0x0410] = "\xe1\xa7", [0x0411] = "\xce\xd7",
  [0x0412] = "\xcf\xa3", [0x0413] = "\xd0\xf2", [0x0414] = "\xb1\xd3",
  [0x0415] = "\xb4\xb2", [0x0416] = "\xcd\xa2", [0x0417] = "\xc5\xaa",
  [0x0418] = "\xb5\xdc", [0x0419] = "\xcd\xae", [0x041a] = "\xd0\xce",
  [0x041b] = "\xe1\xdd", [0x041c] = "\xd2\xdb", [0x041d] = "\xcd\xfc",
  [0x041e] = "\xbc\xc9", [0x041f] = "\xd6\xbe", [0x0420] = "\xc8\xcc",
  [0x0421] = "\xb3\xc0", [0x0422] = "\xbf\xec", [0x0423] = "\xe2\xee",
  [0x0424] = "\xe2\xec", [0x0425] = "\xbd\xe4", [0x0426] = "\xce\xd2",
  [0x0427] = "\xb3\xad", [0x0428] = "\xbf\xb9", [0x0429] = "\xb6\xb6",
  [0x042a] = "\xbc\xbc", [0x042b] = "\xb7\xf6", [0x042c] = "\xbe\xf1",
  [0x042d] = "\xc5\xa4", [0x042e] = "\xb0\xd1", [0x042f] = "\xb6\xf3",
  [0x0430] = "\xd5\xd2", [0x0431] = "\xc5\xfa", [0x0432] = "\xb0\xe2",
  [0x0433] = "\xca\xe3", [0x0434] = "\xb3\xb6", [0x0435] = "\xd5\xdb",
  [0x0436] = "\xb0\xe7", [0x0437] = "\xcd\xb6", [0x0438] = "\xd7\xa5",
  [0x0439] = "\xd2\xd6", [0x043b] = "\xb8\xc4", [0x043c] = "\xb9\xa5",
  [0x043d] = "\xd8\xfc", [0x043e] = "\xba\xb5", [0x043f] = "\xb8\xfc",
  [0x0440] = "\xca\xf8", [0x0441] = "\xc0\xee", [0x0442] = "\xd0\xd3",
  [0x0443] = "\xb2\xc4", [0x0444] = "\xb4\xe5", [0x0445] = "\xb6\xc5",
  [0x0446] = "\xd5\xc8", [0x0447] = "\xe8\xbd", [0x0448] = "\xc9\xbc",
  [0x0449] = "\xb8\xcb", [0x044a] = "\xb8\xdc", [0x044b] = "\xe8\xbc",
  [0x044d] = "\xb2\xbd", [0x044e] = "\xc3\xbf", [0x044f] = "\xc7\xf3",
  [0x0450] = "\xb9\xaf", [0x0451] = "\xc9\xb3", [0x0452] = "\xc7\xdf",
  [0x0453] = "\xc9\xf2", [0x0454] = "\xb3\xc1", [0x0455] = "\xe3\xe4",
  [0x0456] = "\xc5\xe6", [0x0457] = "\xcd\xf4", [0x0458] = "\xbe\xf6",
  [0x0459] = "\xe3\xe5", [0x045a] = "\xcc\xad", [0x045b] = "\xe3\xe7",
  [0x045c] = "\xe3\xe8", [0x045d] = "\xb3\xe5", [0x045e] = "\xc3\xbb",
  [0x045f] = "\xc6\xfb", [0x0460] = "\xce\xd6", [0x0461] = "\xbc\xb3",
  [0x0462] = "\xb7\xda", [0x0463] = "\xe3\xea", [0x0464] = "\xe3\xec",
  [0x0465] = "\xe3\xeb", [0x0466] = "\xd9\xfc", [0x0467] = "\xe3\xe6",
  [0x0469] = "\xd2\xca", [0x046a] = "\xd4\xee", [0x046b] = "\xd7\xc6",
  [0x046c] = "\xd4\xd6", [0x046d] = "\xbe\xc4", [0x046e] = "\xc0\xce",
  [0x046f] = "\xc4\xb5", [0x0470] = "\xcb\xfc", [0x0471] = "\xb5\xd2",
  [0x0472] = "\xbf\xf1", [0x0473] = "\xbe\xc1", [0x0474] = "\xf0\xae",
  [0x0475] = "\xb8\xa6", [0x0476] = "\xc4\xd0", [0x0477] = "\xb5\xe9",
  [0x0478] = "\xd4\xed", [0x0479] = "\xb6\xa2", [0x047a] = "\xd2\xd3",
  [0x047b] = "\xcb\xbd", [0x047c] = "\xd0\xe3", [0x047d] = "\xcd\xba",
  [0x047e] = "\xbe\xbf", [0x047f] = "\xcf\xb5", [0x0480] = "\xba\xb1",
  [0x0481] = "\xd0\xa4", [0x0482] = "\xeb\xc1", [0x0483] = "\xb8\xce",
  [0x0484] = "\xd6\xe2", [0x0485] = "\xb8\xd8", [0x0486] = "\xb6\xc7",
  [0x0487] = "\xd3\xfd", [0x0488] = "\xc1\xbc", [0x0489] = "\xc3\xa2",
  [0x048a] = "\xd3\xf3", [0x048b] = "\xc9\xd6", [0x048c] = "\xbc\xfb",
  [0x048d] = "\xbd\xc7", [0x048e] = "\xd1\xd4", [0x048f] = "\xb9\xc8",
  [0x0490] = "\xb6\xb9", [0x0491] = "\xf5\xb9", [0x0492] = "\xb1\xb4",
  [0x0493] = "\xb3\xe0", [0x0494] = "\xd7\xdf", [0x0495] = "\xd7\xe3",
  [0x0496] = "\xc9\xed", [0x0497] = "\xb3\xb5", [0x0498] = "\xd0\xc1",
  [0x0499] = "\xb3\xbd", [0x049a] = "\xd3\xd8", [0x049b] = "\xe5\xc6",
  [0x049c] = "\xd1\xb8", [0x049d] = "\xc6\xf9", [0x049e] = "\xd1\xb2",
  [0x049f] = "\xd2\xd8", [0x04a0] = "\xd0\xcf", [0x04a1] = "\xd0\xb0",
  [0x04a2] = "\xb0\xee", [0x04a3] = "\xc4\xc7", [0x04a4] = "\xd3\xcf",
  [0x04a5] = "\xb2\xc9", [0x04a6] = "\xc0\xef", [0x04a7] = "\xb7\xc0",
  [0x04a8] = "\xc8\xee", [0x04a9] = "\xda\xe5", [0x04aa] = "\xda\xe6",
  [0x04ab] = "\xbf\xd3", [0x04ac] = "\xb2\xa2", [0x04ad] = "\xb9\xd4",
  [0x04ae] = "\xc8\xe9", [0x04af] = "\xca\xc2", [0x04b0] = "\xd0\xa9",
  [0x04b1] = "\xd1\xc7", [0x04b2] = "\xcf\xed", [0x04b3] = "\xbe\xa9",
  [0x04b4] = "\xd1\xf0", [0x04b5] = "\xd2\xc0", [0x04b6] = "\xca\xcc",
  [0x04b7] = "\xbc\xd1", [0x04b8] = "\xca\xb9", [0x04b9] = "\xc0\xd0",
  [0x04ba] = "\xb9\xa9", [0x04bb] = "\xc0\xfd", [0x04bc] = "\xc0\xb4",
  [0x04bd] = "\xd9\xa9", [0x04be] = "\xb0\xdb", [0x04bf] = "\xb2\xa2",
  [0x04c0] = "\xb3\xde", [0x04c1] = "\xc5\xe5", [0x04c2] = "\xd9\xac",
  [0x04c3] = "\xc2\xd8", [0x04c4] = "\xd9\xab", [0x04c5] = "\xd9\xaa",
  [0x04c6] = "\xd9\xa7", [0x04c8] = "\xcd\xc3", [0x04c9] = "\xb6\xf9",
  [0x04ca] = "\xd9\xee", [0x04cb] = "\xc1\xbd", [0x04cc] = "\xbe\xdf",
  [0x04cd] = "\xc6\xe4", [0x04ce] = "\xb5\xe4", [0x04cf] = "\xd9\xfd",
  [0x04d0] = "\xba\xaf", [0x04d1] = "\xbf\xcc", [0x04d2] = "\xc8\xaf",
  [0x04d3] = "\xcb\xa2", [0x04d4] = "\xb4\xcc", [0x04d5] = "\xb5\xbd",
  [0x04d6] = "\xb9\xce", [0x04d7] = "\xd6\xc6", [0x04d8] = "\xb6\xe7",
  [0x04d9] = "\xdb\xc0", [0x04da] = "\xbf\xef", [0x04db] = "\xd7\xe4",
  [0x04dc] = "\xd0\xad", [0x04dd] = "\xd7\xbf", [0x04de] = "\xb1\xb0",
  [0x04df] = "\xd8\xd4", [0x04e0] = "\xbe\xed", [0x04e1] = "\xd0\xb6",
  [0x04e2] = "\xd0\xf4", [0x04e3] = "\xc8\xa1", [0x04e4] = "\xca\xe5",
  [0x04e5] = "\xca\xdc", [0x04e6] = "\xce\xb6", [0x04e7] = "\xba\xc7",
  [0x04e8] = "\xbf\xa7", [0x04e9] = "\xc5\xde", [0x04ea] = "\xb9\xbe",
  [0x04eb] = "\xbe\xd7", [0x04ec] = "\xc9\xeb", [0x04ed] = "\xdf\xc8",
  [0x04ee] = "\xdf\xcd", [0x04ef] = "\xd6\xe4", [0x04f0] = "\xc5\xd8",
  [0x04f1] = "\xba\xf4", [0x04f2] = "\xb8\xc0", [0x04f3] = "\xdf\xc9",
  [0x04f4] = "\xdf\xce", [0x04f5] = "\xba\xcd", [0x04f6] = "\xdf\xcb",
  [0x04f7] = "\xc4\xd8", [0x04f8] = "\xd6\xdc", [0x04f9] = "\xd5\xa6",
  [0x04fa] = "\xc3\xfc", [0x04fb] = "\xbe\xcc", [0x04fc] = "\xb9\xcc",
  [0x04fd] = "\xc0\xac", [0x04fe] = "\xbf\xc0", [0x04ff] = "\xc6\xba",
  [0x0500] = "\xdb\xe1", [0x0501] = "\xc6\xc2", [0x0502] = "\xcc\xb9",
  [0x0503] = "\xc0\xa4", [0x0504] = "\xdb\xe5", [0x0505] = "\xd2\xb9",
  [0x0506] = "\xb7\xee", [0x0507] = "\xc6\xe6", [0x0508] = "\xc4\xce",
  [0x0509] = "\xd1\xd9", [0x050a] = "\xb1\xbc", [0x050b] = "\xe6\xaa",
  [0x050c] = "\xc6\xde", [0x050d] = "\xce\xaf", [0x050e] = "\xc3\xc3",
  [0x050f] = "\xc4\xdd", [0x0510] = "\xb9\xc3", [0x0511] = "\xc4\xb7",
  [0x0512] = "\xbd\xe3", [0x0513] = "\xe6\xa9", [0x0514] = "\xca\xbc",
  [0x0515] = "\xd0\xd5", [0x0516] = "\xe6\xa2", [0x0517] = "\xe6\xa8",
  [0x0518] = "\xc4\xe3", [0x0519] = "\xe6\xa6", [0x051b] = "\xc3\xcf",
  [0x051c] = "\xb9\xc2", [0x051d] = "\xbc\xbe", [0x051e] = "\xd7\xda",
  [0x051f] = "\xb6\xa8", [0x0520] = "\xb9\xd9", [0x0521] = "\xd2\xcb",
  [0x0522] = "\xd6\xe6", [0x0523] = "\xcd\xf0", [0x0524] = "\xc9\xd0",
  [0x0525] = "\xc7\xfc", [0x0526] = "\xbe\xd3", [0x0527] = "\xbd\xec",
  [0x0528] = "\xe1\xba", [0x0529] = "\xb8\xd4", [0x052a] = "\xb0\xb6",
  [0x052b] = "\xd1\xd2", [0x052c] = "\xe1\xb6", [0x052d] = "\xe1\xb7",
  [0x052e] = "\xd4\xc0", [0x052f] = "\xc1\xb1", [0x0530] = "\xd6\xe3",
  [0x0531] = "\xcc\xfb", [0x0532] = "\xc5\xc1", [0x0533] = "\xb2\xaf",
  [0x0534] = "\xe0\xfb", [0x0535] = "\xd0\xd2", [0x0536] = "\xb8\xfd",
  [0x0537] = "\xb5\xea", [0x0538] = "\xb8\xae", [0x0539] = "\xb5\xd7",
  [0x053a] = "\xe2\xd2", [0x053b] = "\xd1\xd3", [0x053c] = "\xcf\xd2",
  [0x053d] = "\xbb\xa1", [0x053e] = "\xe5\xf3", [0x053f] = "\xcd\xf9",
  [0x0540] = "\xd5\xf7", [0x0541] = "\xb7\xf0", [0x0542] = "\xb1\xcb",
  [0x0543] = "\xe3\xc3", [0x0544] = "\xd6\xd2", [0x0545] = "\xba\xf6",
  [0x0546] = "\xc4\xee", [0x0547] = "\xb7\xde", [0x0548] = "\xe2\xf3",
  [0x0549] = "\xd5\xfa", [0x054a] = "\xc7\xd3", [0x054b] = "\xe2\xf0",
  [0x054c] = "\xb2\xc0", [0x054d] = "\xb9\xd6", [0x054e] = "\xc5\xc2",
  [0x054f] = "\xe2\xf9", [0x0550] = "\xd0\xd4", [0x0551] = "\xe2\xf5",
  [0x0552] = "\xe2\xf6", [0x0553] = "\xe2\xf2", [0x0554] = "\xbb\xf2",
  [0x0555] = "\xe3\xde", [0x0556] = "\xb7\xbf", [0x0557] = "\xec\xe5",
  [0x0558] = "\xcb\xf9", [0x0559] = "\xb3\xd0", [0x055a] = "\xc0\xad",
  [0x055b] = "\xb0\xe8", [0x055c] = "\xd6\xf4", [0x055d] = "\xc3\xf2",
  [0x055e] = "\xb7\xf7", [0x055f] = "\xc4\xa8", [0x0560] = "\xbe\xdc",
  [0x0561] = "\xd5\xd0", [0x0562] = "\xc5\xfb", [0x0563] = "\xcd\xd8",
  [0x0564] = "\xb0\xce", [0x0565] = "\xc5\xd7", [0x0566] = "\xc4\xe9",
  [0x0567] = "\xc5\xea", [0x0568] = "\xb3\xe9", [0x0569] = "\xd1\xba",
  [0x056a] = "\xb9\xd5", [0x056b] = "\xd7\xbe", [0x056c] = "\xc4\xb4",
  [0x056d] = "\xc5\xc4", [0x056e] = "\xb5\xd6", [0x056f] = "\xde\xd5",
  [0x0570] = "\xb1\xa7", [0x0571] = "\xbe\xd0", [0x0572] = "\xcd\xcf",
  [0x0573] = "\xde\xd6", [0x0574] = "\xb2\xf0", [0x0575] = "\xcc\xa7",
  [0x0576] = "\xc1\xe0", [0x0577] = "\xb7\xc5", [0x0578] = "\xb8\xab",
  [0x0579] = "\xec\xb6", [0x057a] = "\xcd\xfa", [0x057b] = "\xce\xf4",
  [0x057c] = "\xd2\xd7", [0x057d] = "\xb2\xfd", [0x057e] = "\xc0\xa5",
  [0x057f] = "\xb0\xba", [0x0580] = "\xc3\xf7", [0x0581] = "\xea\xc0",
  [0x0582] = "\xbb\xe8", [0x0583] = "\xea\xbf", [0x0584] = "\xea\xbb",
  [0x0585] = "\xc9\xfd", [0x0586] = "\xb7\xfe", [0x0587] = "\xc5\xf3",
  [0x0588] = "\xba\xbc", [0x0589] = "\xe8\xca", [0x058a] = "\xd5\xed",
  [0x058b] = "\xb6\xab", [0x058c] = "\xb9\xfb", [0x058d] = "\xe8\xc3",
  [0x058e] = "\xe8\xcb", [0x058f] = "\xe8\xc1", [0x0590] = "\xd6\xa6",
  [0x0591] = "\xc1\xd6", [0x0592] = "\xb1\xad", [0x0593] = "\xbd\xdc",
  [0x0594] = "\xb0\xe5", [0x0595] = "\xcd\xf7", [0x0596] = "\xcb\xc9",
  [0x0597] = "\xce\xf6", [0x0598] = "\xe8\xc6", [0x0599] = "\xc3\xb6",
  [0x059b] = "\xe8\xcc", [0x059c] = "\xe8\xc2", [0x059d] = "\xea\xbd",
  [0x059e] = "\xd0\xc0", [0x059f] = "\xce\xe4", [0x05a0] = "\xc6\xe7",
  [0x05a1] = "\xe9\xe2", [0x05a2] = "\xc3\xa5", [0x05a3] = "\xb7\xd5",
  [0x05a4] = "\xc6\xfc", [0x05a5] = "\xd7\xa2", [0x05a6] = "\xd3\xbe",
  [0x05a7] = "\xe3\xfb", [0x05a8] = "\xc3\xda", [0x05a9] = "\xc4\xe0",
  [0x05aa] = "\xba\xd3", [0x05ab] = "\xb9\xc1", [0x05ac] = "\xd5\xb4",
  [0x05ad] = "\xd5\xd3", [0x05ae] = "\xb2\xa8", [0x05af] = "\xc4\xad",
  [0x05b0] = "\xb7\xa8", [0x05b1] = "\xe3\xfc", [0x05b2] = "\xb7\xd0",
  [0x05b3] = "\xd0\xb9", [0x05b4] = "\xd3\xcd", [0x05b5] = "\xbf\xf6",
  [0x05b6] = "\xbe\xda", [0x05b7] = "\xe3\xf4", [0x05b8] = "\xc7\xf6",
  [0x05b9] = "\xe3\xf3", [0x05ba] = "\xd1\xd8", [0x05bb] = "\xd6\xce",
  [0x05bc] = "\xc5\xdd", [0x05bd] = "\xb7\xba", [0x05be] = "\xb2\xb4",
  [0x05c0] = "\xe3\xfd", [0x05c2] = "\xe3\xf7", [0x05c3] = "\xe3\xf6",
  [0x05c4] = "\xbf\xbb", [0x05c5] = "\xd1\xd7", [0x05c6] = "\xb3\xb4",
  [0x05c7] = "\xb4\xb6", [0x05c8] = "\xd6\xcb", [0x05c9] = "\xc5\xc0",
  [0x05ca] = "\xd5\xf9", [0x05cb] = "\xb0\xd6", [0x05cc] = "\xb0\xe6",
  [0x05cd] = "\xc4\xc1", [0x05ce] = "\xce\xef", [0x05cf] = "\xd7\xb4",
  [0x05d0] = "\xe1\xf2", [0x05d1] = "\xbe\xd1", [0x05d2] = "\xb9\xb7",
  [0x05d3] = "\xba\xfc", [0x05d4] = "\xcd\xe6", [0x05d5] = "\xe7\xe5",
  [0x05d6] = "\xe7\xe4", [0x05d7] = "\xc3\xb5", [0x05da] = "\xf0\xde",
  [0x05db] = "\xb8\xed", [0x05dc] = "\xbe\xce", [0x05dd] = "\xb5\xc4",
  [0x05de] = "\xd3\xdb", [0x05df] = "\xc3\xa4", [0x05e0] = "\xd6\xb1",
  [0x05e1] = "\xd6\xaa", [0x05e2] = "\xce\xf9", [0x05e3] = "\xc9\xe7",
  [0x05e4] = "\xec\xeb", [0x05e5] = "\xc6\xee", [0x05e6] = "\xb1\xfc",
  [0x05e7] = "\xf4\xcc", [0x05e8] = "\xbf\xd5", [0x05e9] = "\xf1\xb7",
  [0x05ea] = "\xf3\xc3", [0x05eb] = "\xbe\xc0", [0x05ec] = "\xd8\xe8",
  [0x05ed] = "\xc7\xbc", [0x05ee] = "\xd8\xc2", [0x05ef] = "\xd5\xdf",
  [0x05f0] = "\xb7\xce", [0x05f1] = "\xb7\xca", [0x05f2] = "\xd6\xab",
  [0x05f3] = "\xeb\xc5", [0x05f4] = "\xb9\xc9", [0x05f5] = "\xeb\xc6",
  [0x05f6] = "\xbc\xe7", [0x05f7] = "\xeb\xc8", [0x05f8] = "\xb7\xbe",
  [0x05f9] = "\xbf\xcf", [0x05fa] = "\xce\xd4", [0x05fb] = "\xf4\xa7",
  [0x05fc] = "\xc9\xe1", [0x05fd] = "\xb7\xbc", [0x05fe] = "\xd6\xa5",
  [0x05ff] = "\xdc\xbd", [0x0600] = "\xb0\xc5", [0x0601] = "\xd1\xbf",
  [0x0602] = "\xdc\xcf", [0x0603] = "\xc7\xdb", [0x0604] = "\xbb\xa8",
  [0x0605] = "\xb7\xd2", [0x0606] = "\xbd\xe6", [0x0607] = "\xd0\xbe",
  [0x0608] = "\xdc\xbf", [0x060a] = "\xdc\xc1", [0x060b] = "\xdc\xc0",
  [0x060c] = "\xdc\xc6", [0x060d] = "\xbb\xa2", [0x060e] = "\xca\xad",
  [0x060f] = "\xb3\xf5", [0x0610] = "\xb1\xed", [0x0611] = "\xd4\xfe",
  [0x0612] = "\xd3\xad", [0x0613] = "\xb7\xb5", [0x0614] = "\xbd\xfc",
  [0x0615] = "\xc9\xdb", [0x0616] = "\xdb\xa1", [0x0617] = "\xc7\xf1",
  [0x0618] = "\xda\xfd", [0x0619] = "\xb2\xc9", [0x061a] = "\xbd\xf0",
  [0x061b] = "\xb3\xa4", [0x061c] = "\xc3\xc5", [0x061d] = "\xb8\xb7",
  [0x061e] = "\xcd\xd3", [0x061f] = "\xb0\xa2", [0x0620] = "\xd7\xe8",
  [0x0621] = "\xb8\xbd", [0x0622] = "\xda\xe9", [0x0623] = "\xf6\xbf",
  [0x0624] = "\xd3\xea", [0x0625] = "\xc7\xe0", [0x0626] = "\xb7\xc7",
  [0x0627] = "\xd8\xbd", [0x0628] = "\xcd\xa4", [0x0629] = "\xc1\xc1",
  [0x062a] = "\xd0\xc5", [0x062b] = "\xc7\xd6", [0x062c] = "\xba\xee",
  [0x062d] = "\xb1\xe3", [0x062e] = "\xcf\xc0", [0x062f] = "\xd9\xb8",
  [0x0630] = "\xc7\xce", [0x0631] = "\xb1\xa3", [0x0632] = "\xb4\xd9",
  [0x0633] = "\xc2\xc2", [0x0634] = "\xb7\xfd", [0x0635] = "\xd9\xb9",
  [0x0636] = "\xbf\xa1", [0x0637] = "\xcb\xd7", [0x0638] = "\xce\xea",
  [0x0639] = "\xc0\xfe", [0x063a] = "\xb6\xed", [0x063b] = "\xcf\xb5",
  [0x063c] = "\xd9\xb5", [0x063d] = "\xd9\xde", [0x063e] = "\xd3\xe1",
  [0x063f] = "\xbe\xd6", [0x0640] = "\xd9\xf0", [0x0641] = "\xc3\xb0",
  [0x0642] = "\xeb\xd0", [0x0643] = "\xb9\xda", [0x0644] = "\xc9\xb2",
  [0x0645] = "\xcc\xea", [0x0646] = "\xcf\xf7", [0x0647] = "\xc7\xb0",
  [0x0648] = "\xd8\xdd", [0x0649] = "\xbf\xcb", [0x064a] = "\xd4\xf2",
  [0x064b] = "\xd3\xc2", [0x064c] = "\xc3\xe3", [0x064d] = "\xb2\xaa",
  [0x064e] = "\xbe\xa2", [0x064f] = "\xd9\xe9", [0x0650] = "\xc4\xcf",
  [0x0651] = "\xc8\xb4", [0x0652] = "\xba\xf1", [0x0653] = "\xc5\xd1",
  [0x0654] = "\xd2\xa7", [0x0655] = "\xb0\xa7", [0x0656] = "\xd7\xc9",
  [0x0657] = "\xb0\xa5", [0x0658] = "\xd4\xd5", [0x0659] = "\xcf\xcc",
  [0x065a] = "\xdf\xd7", [0x065b] = "\xbf\xc8", [0x065c] = "\xcd\xdb",
  [0x065d] = "\xdf\xd3", [0x065e] = "\xd1\xca", [0x065f] = "\xdf\xe4",
  [0x0660] = "\xc6\xb7", [0x0661] = "\xba\xe5", [0x0662] = "\xb9\xfe",
  [0x0663] = "\xbf\xa9", [0x0664] = "\xe5\xeb", [0x0665] = "\xd4\xdb",
  [0x0666] = "\xdf\xdd", [0x0667] = "\xdf\xe3", [0x0668] = "\xdf\xd6",
  [0x0669] = "\xdf\xde", [0x066a] = "\xe0\xf3", [0x066b] = "\xb4\xb9",
  [0x066c] = "\xd0\xcd", [0x066d] = "\xdb\xf3", [0x066e] = "\xd4\xab",
  [0x066f] = "\xb9\xb8", [0x0670] = "\xb3\xc7", [0x0671] = "\xbf\xe5",
  [0x0672] = "\xdb\xf2", [0x0673] = "\xde\xc8", [0x0674] = "\xc6\xf5",
  [0x0675] = "\xd7\xe0", [0x0676] = "\xbf\xfc", [0x0677] = "\xdb\xbc",
  [0x0678] = "\xbd\xaa", [0x0679] = "\xe6\xb0", [0x067a] = "\xd7\xcb",
  [0x067b] = "\xe6\xaf", [0x067c] = "\xd2\xcc", [0x067d] = "\xcd\xde",
  [0x067e] = "\xc0\xd1", [0x067f] = "\xd6\xb6", [0x0680] = "\xd2\xa6",
  [0x0681] = "\xbc\xe9", [0x0682] = "\xcd\xfe", [0x0683] = "\xd2\xf6",
  [0x0684] = "\xba\xa2", [0x0685] = "\xd0\xfb", [0x0686] = "\xbb\xc2",
  [0x0687] = "\xca\xd2", [0x0688] = "\xbf\xcd", [0x0689] = "\xe5\xb6",
  [0x068a] = "\xb7\xe2", [0x068b] = "\xca\xba", [0x068c] = "\xc6\xc1",
  [0x068d] = "\xca\xac", [0x068e] = "\xce\xdd", [0x068f] = "\xd6\xc5",
  [0x0690] = "\xe1\xbc", [0x0691] = "\xcf\xef", [0x0692] = "\xb5\xdb",
  [0x0693] = "\xcb\xa7", [0x0695] = "\xd3\xc4", [0x0696] = "\xe2\xd4",
  [0x0697] = "\xb6\xc8", [0x0698] = "\xbd\xa8", [0x0699] = "\xde\xc4",
  [0x069a] = "\xe5\xf4", [0x069b] = "\xd1\xe5", [0x069c] = "\xba\xdc",
  [0x069d] = "\xb4\xfd", [0x069e] = "\xbb\xb2", [0x069f] = "\xc2\xc9",
  [0x06a0] = "\xe1\xdf", [0x06a1] = "\xba\xf3", [0x06a2] = "\xe1\xe0",
  [0x06a3] = "\xc5\xad", [0x06a4] = "\xcb\xbc", [0x06a5] = "\xb5\xa1",
  [0x06a6] = "\xbc\xb1", [0x06a7] = "\xd4\xf5", [0x06a8] = "\xd4\xb9",
  [0x06a9] = "\xbb\xd0", [0x06aa] = "\xc7\xa1", [0x06ab] = "\xba\xde",
  [0x06ac] = "\xbb\xd6", [0x06ad] = "\xba\xe3", [0x06ae] = "\xca\xd1",
  [0x06af] = "\xcc\xf1", [0x06b0] = "\xb6\xb2", [0x06b1] = "\xe3\xa1",
  [0x06b2] = "\xd0\xf4", [0x06b3] = "\xb1\xe2", [0x06b4] = "\xb0\xdd",
  [0x06b5] = "\xcd\xda", [0x06b6] = "\xb0\xb4", [0x06b7] = "\xc6\xb4",
  [0x06b8] = "\xca\xc3", [0x06b9] = "\xb3\xd6", [0x06ba] = "\xde\xd7",
  [0x06bb] = "\xd7\xa7", [0x06bc] = "\xd6\xb8", [0x06bd] = "\xb9\xb0",
  [0x06be] = "\xbf\xbd", [0x06bf] = "\xd5\xfc", [0x06c0] = "\xc0\xa8",
  [0x06c1] = "\xca\xb0", [0x06c2] = "\xcb\xa9", [0x06c3] = "\xcc\xf4",
  [0x06c4] = "\xb9\xd2", [0x06c5] = "\xd5\xfe", [0x06c6] = "\xb9\xca",
  [0x06c7] = "\xed\xbd", [0x06c8] = "\xca\xa9", [0x06c9] = "\xbc\xc8",
  [0x06ca] = "\xb4\xba", [0x06cb] = "\xd5\xd1", [0x06cc] = "\xd3\xb3",
  [0x06cd] = "\xc3\xc1", [0x06ce] = "\xca\xc7", [0x06cf] = "\xd0\xc7",
  [0x06d0] = "\xd7\xf2", [0x06d1] = "\xea\xc5", [0x06d3] = "\xea\xc2",
  [0x06d4] = "\xca\xc1", [0x06d5] = "\xc8\xbe", [0x06d6] = "\xd6\xf9",
  [0x06d7] = "\xc8\xe1", [0x06d8] = "\xc4\xb3", [0x06d9] = "\xbc\xed",
  [0x06da] = "\xbc\xdc", [0x06db] = "\xbf\xdd", [0x06dc] = "\xd5\xa4",
  [0x06dd] = "\xe8\xd1", [0x06de] = "\xbf\xc2", [0x06df] = "\xb1\xfa",
  [0x06e0] = "\xb8\xcc", [0x06e1] = "\xb9\xd5", [0x06e2] = "\xe8\xd6",
  [0x06e3] = "\xb2\xe9", [0x06e4] = "\xe8\xdb", [0x06e5] = "\xb0\xd8",
  [0x06e6] = "\xd7\xf5", [0x06e7] = "\xc1\xf8", [0x06e8] = "\xe8\xd2",
  [0x06e9] = "\xe8\xd4", [0x06ea] = "\xe8\xdc", [0x06eb] = "\xe8\xd8",
  [0x06ec] = "\xc6\xe2", [0x06ed] = "\xcd\xe1", [0x06ee] = "\xd1\xea",
  [0x06ef] = "\xb4\xf9", [0x06f0] = "\xb6\xce", [0x06f1] = "\xb6\xbe",
  [0x06f2] = "\xc5\xfe", [0x06f3] = "\xb7\xfa", [0x06f4] = "\xc8\xaa",
  [0x06f5] = "\xd1\xf3", [0x06f6] = "\xd6\xde", [0x06f7] = "\xba\xe9",
  [0x06f8] = "\xc1\xf7", [0x06f9] = "\xbd\xf2", [0x06fa] = "\xe4\xa3",
  [0x06fb] = "\xb6\xfd", [0x06fc] = "\xb6\xb4", [0x06fd] = "\xcf\xb4",
  [0x06fe] = "\xbb\xee", [0x06ff] = "\xc7\xa2", [0x0700] = "\xc5\xc9",
  [0x0701] = "\xd0\xda", [0x0702] = "\xc2\xe5", [0x0703] = "\xb1\xc3",
  [0x0704] = "\xe4\xa1", [0x0705] = "\xe4\xa2", [0x0707] = "\xd0\xb9",
  [0x0708] = "\xe4\xac", [0x0709] = "\xe4\xad", [0x070a] = "\xe4\xa9",
  [0x070b] = "\xe4\xaa", [0x070c] = "\xec\xc5", [0x070d] = "\xce\xaa",
  [0x070e] = "\xb1\xfe", [0x070f] = "\xbe\xe6", [0x0710] = "\xbe\xbc",
  [0x0711] = "\xcc\xbf", [0x0712] = "\xd5\xa8", [0x0713] = "\xc5\xda",
  [0x0714] = "\xd5\xd5", [0x0715] = "\xeb\xbc", [0x0716] = "\xc9\xfc",
  [0x0717] = "\xea\xf4", [0x0718] = "\xb5\xd6", [0x0719] = "\xe1\xf7",
  [0x071a] = "\xba\xdd", [0x071b] = "\xbd\xc6", [0x071c] = "\xe7\xe8",
  [0x071d] = "\xc9\xba", [0x071e] = "\xb2\xa3", [0x071f] = "\xc1\xe1",
  [0x0720] = "\xd5\xe4", [0x0721] = "\xe7\xea", [0x0722] = "\xe7\xe9",
  [0x0723] = "\xc9\xf5", [0x0724] = "\xb1\xc2", [0x0725] = "\xce\xb7",
  [0x0726] = "\xbd\xe7", [0x0727] = "\xee\xb0", [0x0728] = "\xee\xb1",
  [0x0729] = "\xd2\xdf", [0x072a] = "\xb0\xcc", [0x072b] = "\xbd\xea",
  [0x072d] = "\xf0\xe0", [0x072e] = "\xb9\xef", [0x072f] = "\xbd\xd4",
  [0x0730] = "\xbb\xca", [0x0731] = "\xf0\xa7", [0x0732] = "\xd3\xaf",
  [0x0733] = "\xc5\xe8", [0x0734] = "\xb1\xad", [0x0735] = "\xd6\xd1",
  [0x0736] = "\xca\xa1", [0x0737] = "\xed\xef", [0x0738] = "\xcf\xe0",
  [0x0739] = "\xc3\xbc", [0x073a] = "\xbf\xb4", [0x073b] = "\xb6\xdc",
  [0x073c] = "\xc5\xce", [0x073d] = "\xed\xf0", [0x073e] = "\xf1\xe6",
  [0x073f] = "\xc9\xb0", [0x0740] = "\xd1\xd0", [0x0741] = "\xc6\xf6",
  [0x0742] = "\xbf\xb3", [0x0743] = "\xec\xec", [0x0744] = "\xec\xed",
  [0x0745] = "\xc6\xed", [0x0746] = "\xd6\xbb", [0x0747] = "\xd3\xed",
  [0x0748] = "\xd8\xae", [0x0749] = "\xbf\xc6", [0x074a] = "\xc3\xeb",
  [0x074b] = "\xc7\xef", [0x074c] = "\xb4\xa9", [0x074d] = "\xcd\xbb",
  [0x074e] = "\xb8\xcd", [0x074f] = "\xf3\xc4", [0x0750] = "\xd7\xd1",
  [0x0751] = "\xe6\xfb", [0x0752] = "\xba\xec", [0x0753] = "\xbc\xcd",
  [0x0754] = "\xc8\xd2", [0x0755] = "\xe6\xfc", [0x0756] = "\xd4\xbc",
  [0x0757] = "\xe6\xfa", [0x0758] = "\xb8\xd7", [0x0759] = "\xc3\xc0",
  [0x075a] = "\xf4\xe0", [0x075b] = "\xeb\xa3", [0x075c] = "\xc4\xcd",
  [0x075d] = "\xcb\xa3", [0x075e] = "\xb6\xcb", [0x075f] = "\xd2\xae",
  [0x0760] = "\xc5\xd6", [0x0761] = "\xf1\xe3", [0x0762] = "\xc5\xdf",
  [0x0763] = "\xce\xb8", [0x0764] = "\xeb\xd0", [0x0765] = "\xb1\xb3",
  [0x0766] = "\xba\xfa", [0x0767] = "\xeb\xce", [0x0768] = "\xcc\xa5",
  [0x0769] = "\xb0\xfb", [0x076a] = "\xd8\xb7", [0x076b] = "\xeb\xd5",
  [0x076c] = "\xd6\xc2", [0x076d] = "\xf4\xae", [0x076e] = "\xdc\xd1",
  [0x076f] = "\xb7\xb6", [0x0770] = "\xc3\xa9", [0x0771] = "\xdc\xc4",
  [0x0772] = "\xbf\xc1", [0x0773] = "\xbf\xe0", [0x0774] = "\xc7\xd1",
  [0x0775] = "\xc8\xf4", [0x0776] = "\xc3\xaf", [0x0777] = "\xdc\xd4",
  [0x0778] = "\xdc\xdb", [0x0779] = "\xc3\xe7", [0x077a] = "\xd3\xa2",
  [0x077b] = "\xd7\xc2", [0x077c] = "\xdc\xd9", [0x077d] = "\xcc\xa6",
  [0x077e] = "\xd4\xb7", [0x077f] = "\xb0\xfa", [0x0780] = "\xdc\xdf",
  [0x0781] = "\xb9\xb6", [0x0782] = "\xb1\xbd", [0x0783] = "\xdc\xe2",
  [0x0784] = "\xc5\xb0", [0x0785] = "\xba\xe7", [0x0786] = "\xf2\xb5",
  [0x0787] = "\xf2\xb3", [0x0788] = "\xd1\xdc", [0x0789] = "\xc9\xc0",
  [0x078a] = "\xd2\xaa", [0x078b] = "\xbd\xee", [0x078c] = "\xbc\xc6",
  [0x078d] = "\xb6\xa9", [0x078e] = "\xb8\xbc", [0x078f] = "\xd5\xea",
  [0x0790] = "\xb8\xba", [0x0791] = "\xb8\xb0", [0x0792] = "\xf4\xf1",
  [0x0793] = "\xc5\xbf", [0x0794] = "\xbe\xfc", [0x0795] = "\xb9\xec",
  [0x0796] = "\xca\xf6", [0x0797] = "\xe5\xc8", [0x0798] = "\xcc\xf6",
  [0x0799] = "\xb5\xcf", [0x079a] = "\xe5\xc4", [0x079b] = "\xb5\xfc",
  [0x079c] = "\xc6\xc8", [0x079d] = "\xe5\xc6", [0x079e] = "\xe5\xca",
  [0x079f] = "\xbd\xbc", [0x07a0] = "\xc0\xc9", [0x07a1] = "\xd3\xf4",
  [0x07a3] = "\xc7\xf5", [0x07a4] = "\xf4\xfa", [0x07a5] = "\xd6\xd8",
  [0x07a6] = "\xe3\xc5", [0x07a7] = "\xcf\xde", [0x07a8] = "\xc2\xaa",
  [0x07a9] = "\xc4\xb0", [0x07aa] = "\xbd\xb5", [0x07ab] = "\xc3\xe6",
  [0x07ac] = "\xb8\xef", [0x07ad] = "\xce\xa4", [0x07ae] = "\xbe\xc2",
  [0x07af] = "\xd2\xf4", [0x07b0] = "\xd2\xb3", [0x07b1] = "\xb7\xe7",
  [0x07b2] = "\xb7\xc9", [0x07b3] = "\xca\xb3", [0x07b4] = "\xca\xd7",
  [0x07b5] = "\xcf\xe3", [0x07b6] = "\xb3\xcb", [0x07b7] = "\xd9\xf1",
  [0x07b8] = "\xd9\xc4", [0x07b9] = "\xb1\xb6", [0x07ba] = "\xb7\xc2",
  [0x07bb] = "\xb8\xa9", [0x07bc] = "\xbe\xeb", [0x07bd] = "\xd9\xc5",
  [0x07be] = "\xd9\xba", [0x07bf] = "\xd9\xbb", [0x07c0] = "\xd0\xd2",
  [0x07c1] = "\xc1\xa9", [0x07c2] = "\xd6\xb5", [0x07c3] = "\xbd\xe8",
  [0x07c4] = "\xd2\xd0", [0x07c5] = "\xb5\xb9", [0x07c6] = "\xc3\xc7",
  [0x07c7] = "\xb0\xb3", [0x07c8] = "\xd8\xf6", [0x07c9] = "\xbe\xf3",
  [0x07ca] = "\xd9\xc6", [0x07cb] = "\xbe\xe3", [0x07cc] = "\xb3\xab",
  [0x07cd] = "\xb8\xf6", [0x07ce] = "\xba\xf2", [0x07cf] = "\xcc\xc8",
  [0x07d0] = "\xd9\xbd", [0x07d1] = "\xd0\xde", [0x07d2] = "\xd9\xc1",
  [0x07d3] = "\xc4\xdf", [0x07d4] = "\xd9\xc2", [0x07d5] = "\xc2\xd7",
  [0x07d6] = "\xb2\xd6", [0x07d7] = "\xbc\xe6", [0x07d8] = "\xd4\xa9",
  [0x07d9] = "\xda\xa4", [0x07da] = "\xda\xa3", [0x07db] = "\xb6\xb3",
  [0x07dc] = "\xc1\xe8", [0x07dd] = "\xd7\xbc", [0x07de] = "\xb5\xf2",
  [0x07df] = "\xc6\xca", [0x07e0] = "\xd8\xe0", [0x07e1] = "\xcc\xde",
  [0x07e2] = "\xb8\xd5", [0x07e3] = "\xb0\xfe", [0x07e4] = "\xb7\xcb",
  [0x07e5] = "\xc7\xe4", [0x07e6] = "\xd4\xad", [0x07e7] = "\xd8\xc8",
  [0x07e8] = "\xdb\xc5", [0x07e9] = "\xc9\xda", [0x07ea] = "\xcc\xc6",
  [0x07eb] = "\xd1\xe4", [0x07ec] = "\xe0\xa1", [0x07ed] = "\xba\xdf",
  [0x07ee] = "\xb8\xe7", [0x07ef] = "\xd5\xdc", [0x07f0] = "\xcb\xf4",
  [0x07f1] = "\xb2\xb8", [0x07f2] = "\xdf\xed", [0x07f3] = "\xc1\xa8",
  [0x07f4] = "\xbf\xde", [0x07f5] = "\xd4\xb1", [0x07f6] = "\xb0\xa6",
  [0x07f7] = "\xcf\xf8", [0x07f8] = "\xc4\xc4", [0x07f9] = "\xc5\xb6",
  [0x07fa] = "\xdf\xf3", [0x07fb] = "\xb4\xbd", [0x07fc] = "\xdf\xec",
  [0x07fd] = "\xdf\xf1", [0x07fe] = "\xc6\xd4", [0x07ff] = "\xe0\xf4",
  [0x0800] = "\xb9\xa1", [0x0801] = "\xc6\xd2", [0x0802] = "\xc2\xf1",
  [0x0803] = "\xb0\xa3", [0x0805] = "\xcf\xc4", [0x0806] = "\xcc\xd7",
  [0x0807] = "\xde\xca", [0x0808] = "\xde\xc9", [0x0809] = "\xe6\xb6",
  [0x080a] = "\xc4\xef", [0x080b] = "\xc4\xc8", [0x080c] = "\xbe\xea",
  [0x080d] = "\xd3\xe9", [0x080e] = "\xe6\xb8", [0x080f] = "\xbc\xa7",
  [0x0810] = "\xc9\xef", [0x0811] = "\xe6\xb7", [0x0812] = "\xc3\xe4",
  [0x0813] = "\xb6\xf0", [0x0814] = "\xe6\xb2", [0x0815] = "\xe6\xb3",
  [0x0816] = "\xcb\xef", [0x0818] = "\xd4\xd7", [0x0819] = "\xba\xa6",
  [0x081a] = "\xbc\xd2", [0x081b] = "\xd1\xe7", [0x081c] = "\xb9\xac",
  [0x081d] = "\xcf\xfc", [0x081e] = "\xc8\xdd", [0x081f] = "\xe5\xb7",
  [0x0820] = "\xc9\xe4", [0x0821] = "\xd0\xbc", [0x0822] = "\xd5\xb9",
  [0x0823] = "\xe5\xec", [0x0824] = "\xc7\xcd", [0x0825] = "\xcf\xbf",
  [0x0826] = "\xbe\xfe", [0x0827] = "\xd3\xf8", [0x0828] = "\xb6\xeb",
  [0x0829] = "\xb7\xe5", [0x082a] = "\xb5\xba", [0x082c] = "\xe1\xad",
  [0x082d] = "\xb2\xee", [0x082e] = "\xcf\xaf", [0x082f] = "\xca\xa6",
  [0x0830] = "\xbf\xe2", [0x0831] = "\xcd\xa5", [0x0832] = "\xd7\xf9",
  [0x0833] = "\xc8\xf5", [0x0834] = "\xcd\xbd", [0x0835] = "\xbe\xb6",
  [0x0836] = "\xd0\xec", [0x0837] = "\xed\xa6", [0x0838] = "\xed\xa7",
  [0x0839] = "\xb3\xdc", [0x083a] = "\xbf\xd6", [0x083b] = "\xcb\xa1",
  [0x083c] = "\xb9\xa7", [0x083d] = "\xb6\xf7", [0x083e] = "\xcf\xa2",
  [0x083f] = "\xc7\xc4", [0x0840] = "\xce\xf2", [0x0841] = "\xe3\xa4",
  [0x0842] = "\xba\xb7", [0x0843] = "\xbb\xda", [0x0844] = "\xe3\xa9",
  [0x0845] = "\xd4\xc3", [0x0846] = "\xe3\xa3", [0x0847] = "\xc9\xc8",
  [0x0848] = "\xc8\xad", [0x0849] = "\xea\xfc", [0x084a] = "\xc4\xc3",
  [0x084b] = "\xc9\xd3", [0x084c] = "\xd0\xae", [0x084d] = "\xd5\xf1",
  [0x084e] = "\xb2\xb6", [0x084f] = "\xce\xe6", [0x0850] = "\xc0\xa6",
  [0x0851] = "\xc4\xf3", [0x0852] = "\xd7\xbd", [0x0853] = "\xcd\xa6",
  [0x0854] = "\xbe\xe8", [0x0855] = "\xcd\xec", [0x0856] = "\xc5\xb2",
  [0x0857] = "\xb4\xec", [0x0858] = "\xb0\xa4", [0x0859] = "\xba\xb4",
  [0x085a] = "\xb0\xc6", [0x085b] = "\xd0\xa7", [0x085c] = "\xf4\xcd",
  [0x085d] = "\xc1\xcf", [0x085e] = "\xc5\xd4", [0x085f] = "\xc2\xc3",
  [0x0860] = "\xca\xb1", [0x0861] = "\xbd\xfa", [0x0862] = "\xea\xcc",
  [0x0863] = "\xbb\xce", [0x0864] = "\xc9\xb9", [0x0865] = "\xc9\xce",
  [0x0867] = "\xea\xcb", [0x0868] = "\xca\xe9", [0x0869] = "\xcb\xb7",
  [0x086a] = "\xeb\xde", [0x086b] = "\xc0\xca", [0x086c] = "\xd0\xa3",
  [0x086d] = "\xba\xcb", [0x086e] = "\xb0\xb8", [0x086f] = "\xbf\xf2",
  [0x0870] = "\xbb\xb8", [0x0871] = "\xb8\xf9", [0x0872] = "\xb9\xf0",
  [0x0873] = "\xbd\xdb", [0x0874] = "\xe8\xf2", [0x0875] = "\xca\xe1",
  [0x0876] = "\xc0\xf5", [0x0877] = "\xd7\xc0", [0x0878] = "\xc9\xa3",
  [0x0879] = "\xd4\xd4", [0x087a] = "\xb2\xf1", [0x087b] = "\xcd\xa9",
  [0x087c] = "\xe8\xee", [0x087d] = "\xb8\xf1", [0x087e] = "\xcc\xd2",
  [0x087f] = "\xd6\xea", [0x0880] = "\xce\xa6", [0x0881] = "\xcb\xa8",
  [0x0883] = "\xe8\xec", [0x0884] = "\xca\xe2", [0x0885] = "\xd1\xb3",
  [0x0886] = "\xd2\xf3", [0x0887] = "\xc6\xf8", [0x0888] = "\xd1\xf5",
  [0x0889] = "\xb0\xb1", [0x088a] = "\xba\xa4", [0x088b] = "\xeb\xb3",
  [0x088c] = "\xcc\xa9", [0x088d] = "\xc0\xcb", [0x088e] = "\xcc\xe9",
  [0x088f] = "\xcf\xfb", [0x0890] = "\xe3\xfe", [0x0891] = "\xc6\xd6",
  [0x0892] = "\xbd\xfe", [0x0893] = "\xba\xa3", [0x0894] = "\xd5\xe3",
  [0x0895] = "\xe4\xb8", [0x0896] = "\xc0\xef", [0x0897] = "\xc9\xe6",
  [0x0898] = "\xb8\xa1", [0x0899] = "\xbf\xa3", [0x089a] = "\xd4\xa1",
  [0x089b] = "\xba\xc6", [0x089c] = "\xd3\xbf", [0x089e] = "\xe4\xa4",
  [0x089f] = "\xc4\xf9", [0x08a1] = "\xe4\xb9", [0x08a2] = "\xec\xc8",
  [0x08a3] = "\xba\xe6", [0x08a4] = "\xbf\xbe", [0x08a5] = "\xc0\xd3",
  [0x08a6] = "\xc1\xd2", [0x08a7] = "\xce\xda", [0x08a8] = "\xb5\xf9",
  [0x08a9] = "\xcc\xd8", [0x08aa] = "\xc0\xc7", [0x08ab] = "\xcf\xc1",
  [0x08ac] = "\xb1\xb7", [0x08ad] = "\xc0\xea", [0x08ae] = "\xe1\xfa",
  [0x08af] = "\xd7\xc8", [0x08b0] = "\xb0\xe0", [0x08b1] = "\xc1\xf0",
  [0x08b2] = "\xc5\xe5", [0x08b3] = "\xd6\xe9", [0x08b4] = "\xb9\xe7",
  [0x08b5] = "\xe7\xf3", [0x08b6] = "\xc5\xcf", [0x08b7] = "\xc4\xb6",
  [0x08b8] = "\xd0\xf3", [0x08b9] = "\xdb\xce", [0x08ba] = "\xc1\xf4",
  [0x08bb] = "\xbc\xb2", [0x08bc] = "\xb2\xa1", [0x08bd] = "\xd6\xa2",
  [0x08be] = "\xc6\xa3", [0x08bf] = "\xf0\xe1", [0x08c0] = "\xbe\xd2",
  [0x08c1] = "\xcc\xdb", [0x08c2] = "\xd5\xee", [0x08c3] = "\xf0\xe8",
  [0x08c4] = "\xf0\xe3", [0x08c5] = "\xb8\xde", [0x08c6] = "\xf0\xe5",
  [0x08c7] = "\xd2\xe6", [0x08c8] = "\xee\xc1", [0x08c9] = "\xb0\xbb",
  [0x08ca] = "\xd1\xa3", [0x08cb] = "\xd5\xe6", [0x08cc] = "\xc3\xdf",
  [0x08cd] = "\xd5\xa3", [0x08ce] = "\xbe\xd8", [0x08cf] = "\xc5\xe9",
  [0x08d0] = "\xd5\xe8", [0x08d1] = "\xd4\xd2", [0x08d2] = "\xed\xc0",
  [0x08d3] = "\xc6\xc6", [0x08d4] = "\xc9\xe9", [0x08d5] = "\xed\xc6",
  [0x08d6] = "\xed\xbe", [0x08d8] = "\xed\xc4", [0x08d9] = "\xc5\xda",
  [0x08da] = "\xc3\xd8", [0x08db] = "\xd3\xd3", [0x08dc] = "\xec\xf4",
  [0x08dd] = "\xcb\xee", [0x08de] = "\xd7\xe6", [0x08df] = "\xc9\xf1",
  [0x08e0] = "\xd7\xa3", [0x08e1] = "\xec\xf3", [0x08e2] = "\xec\xf1",
  [0x08e3] = "\xb3\xd3", [0x08e4] = "\xef\xf7", [0x08e5] = "\xd1\xed",
  [0x08e6] = "\xd7\xe2", [0x08e7] = "\xc7\xd8", [0x08e8] = "\xd6\xc8",
  [0x08e9] = "\xc3\xd8", [0x08ea] = "\xd5\xad", [0x08eb] = "\xf1\xba",
  [0x08ec] = "\xd5\xbe", [0x08ed] = "\xb0\xca", [0x08ee] = "\xd0\xa6",
  [0x08ef] = "\xb7\xdb", [0x08f0] = "\xb7\xc4", [0x08f1] = "\xc9\xb4",
  [0x08f2] = "\xce\xc6", [0x08f3] = "\xce\xc9", [0x08f4] = "\xcb\xd8",
  [0x08f5] = "\xcb\xf7", [0x08f6] = "\xb4\xbf", [0x08f7] = "\xc5\xa6",
  [0x08f8] = "\xe7\xa2", [0x08f9] = "\xbc\xb6", [0x08fa] = "\xe7\xa1",
  [0x08fb] = "\xc4\xc9", [0x08fc] = "\xd6\xbd", [0x08fd] = "\xb7\xd7",
  [0x08fe] = "\xc8\xb1", [0x08ff] = "\xee\xb9", [0x0900] = "\xb8\xe1",
  [0x0901] = "\xb3\xe1", [0x0902] = "\xce\xcc", [0x0903] = "\xea\xc8",
  [0x0904] = "\xd4\xc5", [0x0905] = "\xb8\xfb", [0x0906] = "\xb0\xd2",
  [0x0907] = "\xba\xc4", [0x0908] = "\xb5\xa2", [0x0909] = "\xb9\xa2",
  [0x090a] = "\xeb\xd7", [0x090b] = "\xd6\xac", [0x090c] = "\xd2\xc8",
  [0x090d] = "\xd0\xb2", [0x090e] = "\xeb\xd9", [0x090f] = "\xeb\xd8",
  [0x0910] = "\xb4\xe0", [0x0911] = "\xd0\xd8", [0x0912] = "\xb8\xec",
  [0x0913] = "\xc2\xf6", [0x0914] = "\xc4\xdc", [0x0915] = "\xbc\xb9",
  [0x0916] = "\xeb\xdd", [0x0917] = "\xbf\xe8", [0x0918] = "\xb3\xf4",
  [0x0919] = "\xf4\xab", [0x091a] = "\xd2\xa8", [0x091b] = "\xf3\xc2",
  [0x091c] = "\xba\xbd", [0x091d] = "\xf4\xb3", [0x091e] = "\xf4\xb2",
  [0x091f] = "\xb0\xe3", [0x0920] = "\xdb\xbb", [0x0921] = "\xc3\xa3",
  [0x0922] = "\xbb\xc4", [0x0923] = "\xc0\xf3", [0x0924] = "\xbe\xa3",
  [0x0925] = "\xc8\xd7", [0x0926] = "\xbc\xf6", [0x0927] = "\xb2\xdd",
  [0x0928] = "\xd2\xf0", [0x0929] = "\xdc\xee", [0x092a] = "\xdc\xf3",
  [0x092b] = "\xd7\xc8", [0x092c] = "\xc8\xe3", [0x092d] = "\xb2\xe8",
  [0x092e] = "\xdc\xf8", [0x092f] = "\xdc\xf7", [0x0930] = "\xdc\xef",
  [0x0931] = "\xb4\xc4", [0x0932] = "\xdc\xf5", [0x0933] = "\xf2\xaf",
  [0x0934] = "\xce\xc3", [0x0935] = "\xf2\xbd", [0x0936] = "\xf2\xbe",
  [0x0937] = "\xd4\xe9", [0x0938] = "\xf2\xbf", [0x0939] = "\xb0\xf6",
  [0x093a] = "\xf2\xbc", [0x093b] = "\xd1\xc1", [0x093c] = "\xcb\xa5",
  [0x093d] = "\xd6\xd4", [0x093e] = "\xd4\xac", [0x093f] = "\xf1\xc7",
  [0x0940] = "\xf1\xc5", [0x0941] = "\xd6\xbb", [0x0942] = "\xbc\xc7",
  [0x0943] = "\xda\xa6", [0x0944] = "\xcc\xd6", [0x0945] = "\xda\xa7",
  [0x0946] = "\xda\xa8", [0x0947] = "\xd1\xb6", [0x0948] = "\xcd\xd0",
  [0x0949] = "\xd1\xb5", [0x094a] = "\xc6\xfd", [0x094d] = "\xc6\xf1",
  [0x094e] = "\xb2\xf2", [0x094f] = "\xb1\xaa", [0x0950] = "\xb2\xc6",
  [0x0951] = "\xb9\xb1", [0x0952] = "\xc6\xf0", [0x0953] = "\xb9\xaa",
  [0x0954] = "\xd0\xf9", [0x0955] = "\xe9\xed", [0x0957] = "\xc8\xe8",
  [0x0958] = "\xcb\xcd", [0x0959] = "\xc4\xe6", [0x095a] = "\xc3\xd4",
  [0x095b] = "\xcd\xcb", [0x095c] = "\xc4\xcb", [0x095d] = "\xbb\xd8",
  [0x095e] = "\xcc\xd3", [0x095f] = "\xd7\xb7", [0x0960] = "\xe5\xcb",
  [0x0961] = "\xb1\xc5", [0x0962] = "\xe7\xdf", [0x0963] = "\xbf\xa4",
  [0x0964] = "\xba\xc2", [0x0965] = "\xdb\xab", [0x0966] = "\xbe\xc6",
  [0x0967] = "\xc5\xe4", [0x0968] = "\xd7\xc3", [0x0969] = "\xb6\xa4",
  [0x096a] = "\xd5\xeb", [0x096b] = "\xee\xc8", [0x096c] = "\xb8\xaa",
  [0x096d] = "\xee\xc7", [0x096e] = "\xc9\xc1", [0x096f] = "\xd4\xba",
  [0x0970] = "\xd5\xf3", [0x0971] = "\xb6\xb8", [0x0972] = "\xb1\xdd",
  [0x0973] = "\xc9\xc2", [0x0974] = "\xb3\xfd", [0x0975] = "\xda\xea",
  [0x0976] = "\xc9\xfd", [0x0977] = "\xd6\xbb", [0x0978] = "\xbc\xa2",
  [0x0979] = "\xc2\xed", [0x097a] = "\xb9\xc7", [0x097b] = "\xb8\xdf",
  [0x097c] = "\xb6\xb7", [0x097d] = "\xd8\xaa", [0x097e] = "\xb9\xed",
  [0x097f] = "\xb8\xc9", [0x0980] = "\xd4\xdb", [0x0981] = "\xce\xb1",
  [0x0982] = "\xcd\xa3", [0x0983] = "\xbc\xd9", [0x0984] = "\xd9\xc8",
  [0x0985] = "\xd9\xbc", [0x0986] = "\xd7\xf6", [0x0987] = "\xce\xb0",
  [0x0988] = "\xbd\xa1", [0x0989] = "\xc5\xbc", [0x098a] = "\xd9\xcb",
  [0x098b] = "\xd9\xc9", [0x098c] = "\xd5\xec", [0x098d] = "\xb2\xe0",
  [0x098e] = "\xcd\xb5", [0x098f] = "\xc6\xab", [0x0990] = "\xd9\xbf",
  [0x0993] = "\xb6\xb5", [0x0994] = "\xc3\xe1", [0x0995] = "\xbb\xcb",
  [0x0996] = "\xbc\xf4", [0x0997] = "\xb8\xb1", [0x0998] = "\xc0\xd5",
  [0x0999] = "\xce\xf1", [0x099a] = "\xbf\xb1", [0x099b] = "\xb6\xaf",
  [0x099c] = "\xd9\xeb", [0x099d] = "\xde\xcb", [0x099e] = "\xb3\xd7",
  [0x099f] = "\xc4\xe4", [0x09a0] = "\xc7\xf8", [0x09a1] = "\xd8\xd2",
  [0x09a2] = "\xb2\xce", [0x09a3] = "\xc2\xfc", [0x09a4] = "\xc9\xcc",
  [0x09a5] = "\xc5\xbe", [0x09a6] = "\xc0\xb2", [0x09a7] = "\xd7\xc4",
  [0x09a8] = "\xd1\xc6", [0x09a9] = "\xb7\xc8", [0x09aa] = "\xbf\xd0",
  [0x09ab] = "\xb0\xa1", [0x09ac] = "\xb3\xaa", [0x09ad] = "\xe0\xa2",
  [0x09ae] = "\xce\xca", [0x09af] = "\xdf\xfb", [0x09b0] = "\xce\xa8",
  [0x09b1] = "\xc6\xa1", [0x09b2] = "\xc4\xee", [0x09b3] = "\xca\xdb",
  [0x09b4] = "\xe0\xa8", [0x09b5] = "\xbb\xa3", [0x09b6] = "\xcf\xce",
  [0x09b7] = "\xe0\xa6", [0x09b8] = "\xdf\xfa", [0x09b9] = "\xe0\xa2",
  [0x09ba] = "\xc8\xa6", [0x09bb] = "\xb9\xfa", [0x09bc] = "\xe0\xf6",
  [0x09bd] = "\xd3\xf2", [0x09be] = "\xbc\xe1", [0x09bf] = "\xdb\xd1",
  [0x09c0] = "\xb6\xd1", [0x09c1] = "\xb2\xba", [0x09c2] = "\xdb\xfd",
  [0x09c3] = "\xbb\xf9", [0x09c4] = "\xcc\xc3", [0x09c5] = "\xb6\xc2",
  [0x09c6] = "\xd6\xb4", [0x09c7] = "\xc5\xe0", [0x09c8] = "\xb9\xbb",
  [0x09c9] = "\xc9\xdd", [0x09ca] = "\xc8\xa2", [0x09cb] = "\xc2\xa6",
  [0x09cc] = "\xcd\xf1", [0x09cd] = "\xb8\xbe", [0x09ce] = "\xc0\xb7",
  [0x09cf] = "\xe6\xb9", [0x09d0] = "\xe6\xbd", [0x09d1] = "\xe6\xbe",
  [0x09d2] = "\xbb\xe9", [0x09d3] = "\xc6\xc5", [0x09d4] = "\xe6\xbb",
  [0x09d5] = "\xca\xeb", [0x09d6] = "\xbf\xdc", [0x09d7] = "\xd2\xfa",
  [0x09d8] = "\xbc\xc4", [0x09d9] = "\xbc\xc5", [0x09da] = "\xcb\xde",
  [0x09db] = "\xc3\xdc", [0x09dc] = "\xce\xbe", [0x09dd] = "\xd7\xa8",
  [0x09de] = "\xbd\xab", [0x09df] = "\xcd\xc0", [0x09e0] = "\xcc\xeb",
  [0x09e1] = "\xec\xe9", [0x09e2] = "\xb3\xe7", [0x09e3] = "\xe1\xc7",
  [0x09e4] = "\xc6\xe9", [0x09e5] = "\xe1\xc8", [0x09e6] = "\xd1\xc2",
  [0x09e7] = "\xe1\xbf", [0x09e8] = "\xc0\xa5", [0x09e9] = "\xb1\xc0",
  [0x09ea] = "\xb4\xde", [0x09eb] = "\xc2\xd8", [0x09ec] = "\xe1\xc5",
  [0x09ed] = "\xe1\xc2", [0x09ee] = "\xb8\xda", [0x09ef] = "\xb3\xb2",
  [0x09f0] = "\xb3\xa3", [0x09f1] = "\xb4\xf8", [0x09f2] = "\xd5\xca",
  [0x09f3] = "\xe1\xa1", [0x09f4] = "\xbf\xb5", [0x09f5] = "\xd3\xb9",
  [0x09f6] = "\xca\xfc", [0x09f7] = "\xe2\xd6", [0x09f8] = "\xe2\xd7",
  [0x09f9] = "\xd5\xc5", [0x09fa] = "\xc7\xbf", [0x09fb] = "\xe5\xe7",
  [0x09fc] = "\xb1\xf2", [0x09fd] = "\xb2\xca", [0x09fe] = "\xb5\xf1",
  [0x09ff] = "\xb5\xc3", [0x0a00] = "\xe1\xe3", [0x0a01] = "\xb4\xd3",
  [0x0a02] = "\xc5\xc7", [0x0a03] = "\xd3\xf9", [0x0a04] = "\xe1\xe2",
  [0x0a05] = "\xe1\xe4", [0x0a06] = "\xd3\xc1", [0x0a07] = "\xbb\xbc",
  [0x0a08] = "\xcf\xa4", [0x0a09] = "\xd3\xc6", [0x0a0a] = "\xc4\xfa",
  [0x0a0b] = "\xcd\xef", [0x0a0c] = "\xe3\xb2", [0x0a0d] = "\xb5\xeb",
  [0x0a0e] = "\xc6\xe0", [0x0a0f] = "\xc7\xe9", [0x0a10] = "\xe3\xac",
  [0x0a11] = "\xe2\xea", [0x0a12] = "\xcf\xa7", [0x0a13] = "\xb5\xbf",
  [0x0a14] = "\xe3\xaf", [0x0a15] = "\xcc\xe8", [0x0a16] = "\xe3\xb0",
  [0x0a17] = "\xce\xa9", [0x0a18] = "\xbc\xc2", [0x0a19] = "\xe3\xb1",
  [0x0a1a] = "\xb6\xd8", [0x0a1b] = "\xc6\xdd", [0x0a1c] = "\xea\xa9",
  [0x0a1d] = "\xec\xe8", [0x0a1e] = "\xc2\xd3", [0x0a1f] = "\xbf\xd8",
  [0x0a20] = "\xbe\xed", [0x0a21] = "\xd2\xb4", [0x0a22] = "\xcc\xbd",
  [0x0a23] = "\xbd\xd3", [0x0a24] = "\xbd\xdd", [0x0a25] = "\xc5\xf5",
  [0x0a26] = "\xbe\xf2", [0x0a27] = "\xb4\xeb", [0x0a28] = "\xde\xdf",
  [0x0a29] = "\xd1\xda", [0x0a2a] = "\xb5\xf4", [0x0a2b] = "\xc9\xa8",
  [0x0a2c] = "\xb9\xd2", [0x0a2d] = "\xde\xd1", [0x0a2e] = "\xcd\xc6",
  [0x0a2f] = "\xc2\xd5", [0x0a30] = "\xca\xda", [0x0a31] = "\xd5\xf5",
  [0x0a32] = "\xb2\xc9", [0x0a33] = "\xde\xe4", [0x0a34] = "\xc5\xc5",
  [0x0a35] = "\xcc\xcd", [0x0a36] = "\xcf\xc6", [0x0a37] = "\xc4\xed",
  [0x0a38] = "\xde\xe6", [0x0a39] = "\xc9\xe1", [0x0a3a] = "\xde\xe0",
  [0x0a3b] = "\xb1\xd6", [0x0a3c] = "\xb0\xbd", [0x0a3d] = "\xbe\xc8",
  [0x0a3e] = "\xbd\xcc", [0x0a3f] = "\xb0\xdc", [0x0a40] = "\xc6\xf4",
  [0x0a41] = "\xc3\xf4", [0x0a42] = "\xd0\xf0", [0x0a43] = "\xeb\xb7",
  [0x0a45] = "\xd0\xb1", [0x0a46] = "\xf5\xfa", [0x0a47] = "\xd5\xb6",
  [0x0a48] = "\xd7\xe5", [0x0a49] = "\xd0\xfd", [0x0a4a] = "\xec\xba",
  [0x0a4b] = "\xec\xbb", [0x0a4c] = "\xd6\xe7", [0x0a4d] = "\xcd\xed",
  [0x0a4e] = "\xce\xee", [0x0a4f] = "\xb3\xbf", [0x0a50] = "\xbb\xde",
  [0x0a51] = "\xea\xd8", [0x0a52] = "\xb2\xdc", [0x0a53] = "\xdb\xc3",
  [0x0a54] = "\xcd\xfb", [0x0a55] = "\xc1\xba", [0x0a56] = "\xcc\xdd",
  [0x0a57] = "\xc9\xd2", [0x0a58] = "\xe8\xf7", [0x0a59] = "\xe8\xf3",
  [0x0a5a] = "\xb8\xcb", [0x0a5b] = "\xcd\xb0", [0x0a5c] = "\xc0\xa6",
  [0x0a5d] = "\xce\xe0", [0x0a5e] = "\xb9\xa3", [0x0a5f] = "\xd0\xb5",
  [0x0a60] = "\xe8\xe8", [0x0a61] = "\xc6\xfa", [0x0a62] = "\xcb\xf3",
  [0x0a63] = "\xb0\xf0", [0x0a64] = "\xc3\xb7", [0x0a65] = "\xe8\xd9",
  [0x0a66] = "\xcc\xf5", [0x0a67] = "\xc0\xe6", [0x0a68] = "\xe8\xc9",
  [0x0a6b] = "\xd3\xfb", [0x0a6c] = "\xc9\xb1", [0x0a6d] = "\xba\xc1",
  [0x0a6e] = "\xc7\xf2", [0x0a6f] = "\xc7\xe2", [0x0a70] = "\xcf\xd1",
  [0x0a71] = "\xc1\xb9", [0x0a72] = "\xb4\xbe", [0x0a73] = "\xe4\xc8",
  [0x0a74] = "\xd2\xba", [0x0a75] = "\xb5\xad", [0x0a76] = "\xcc\xca",
  [0x0a77] = "\xd3\xd9", [0x0a78] = "\xcc\xed", [0x0a79] = "\xc7\xb3",
  [0x0a7a] = "\xc7\xe5", [0x0a7b] = "\xe4\xbf", [0x0a7c] = "\xc1\xdc",
  [0x0a7d] = "\xd1\xc4", [0x0a7e] = "\xca\xe7", [0x0a7f] = "\xe4\xcc",
  [0x0a80] = "\xe4\xc1", [0x0a81] = "\xd1\xcd", [0x0a82] = "\xba\xd4",
  [0x0a83] = "\xbb\xec", [0x0a84] = "\xd4\xa8", [0x0a85] = "\xe4\xc0",
  [0x0a86] = "\xc6\xe0", [0x0a87] = "\xe4\xbe", [0x0a88] = "\xba\xad",
  [0x0a89] = "\xc0\xe1", [0x0a8a] = "\xd2\xf9", [0x0a8b] = "\xcc\xd4",
  [0x0a8c] = "\xc2\xd9", [0x0a8d] = "\xc9\xee", [0x0a8e] = "\xbb\xb4",
  [0x0a8f] = "\xbe\xbb", [0x0a90] = "\xcf\xfd", [0x0a91] = "\xd7\xcd",
  [0x0a92] = "\xb8\xa2", [0x0a93] = "\xb4\xe3", [0x0a94] = "\xe4\xc3",
  [0x0a95] = "\xe4\xc6", [0x0a96] = "\xc5\xeb", [0x0a97] = "\xd1\xc9",
  [0x0a98] = "\xba\xb8", [0x0a99] = "\xb7\xe9", [0x0a9a] = "\xcf\xa9",
  [0x0a9b] = "\xcb\xac", [0x0a9c] = "\xc7\xa3", [0x0a9d] = "\xc0\xe7",
  [0x0a9e] = "\xb2\xc2", [0x0a9f] = "\xc3\xcd", [0x0aa0] = "\xb2\xfe",
  [0x0aa1] = "\xe2\xa3", [0x0aa2] = "\xd5\xf8", [0x0aa3] = "\xc2\xca",
  [0x0aa4] = "\xc0\xc5", [0x0aa5] = "\xe7\xf0", [0x0aa6] = "\xc7\xf2",
  [0x0aa7] = "\xc0\xed", [0x0aa8] = "\xcf\xd6", [0x0aaa] = "\xf0\xad",
  [0x0aab] = "\xc6\xbf", [0x0aac] = "\xb4\xc9", [0x0aad] = "\xcc\xf0",
  [0x0aae] = "\xb2\xfa", [0x0aaf] = "\xc2\xd4", [0x0ab0] = "\xc6\xe8",
  [0x0ab1] = "\xb1\xcf", [0x0ab2] = "\xd2\xec", [0x0ab3] = "\xca\xe8",
  [0x0ab4] = "\xd6\xcc", [0x0ab5] = "\xba\xdb", [0x0ab6] = "\xb4\xc3",
  [0x0ab7] = "\xc8\xac", [0x0ab8] = "\xf0\xea", [0x0ab9] = "\xf0\xa8",
  [0x0aba] = "\xbf\xf8", [0x0abb] = "\xba\xd0", [0x0abc] = "\xca\xa2",
  [0x0abd] = "\xbe\xec", [0x0abe] = "\xd6\xda", [0x0abf] = "\xd1\xdb",
  [0x0ac0] = "\xbf\xf4", [0x0ac1] = "\xed\xf8", [0x0ac2] = "\xcc\xf7",
  [0x0ac3] = "\xc1\xf2", [0x0ac4] = "\xd6\xec", [0x0ac5] = "\xed\xca",
  [0x0ac6] = "\xcf\xe9", [0x0ac7] = "\xc6\xb1", [0x0ac8] = "\xbc\xc0",
  [0x0ac9] = "\xd2\xc6", [0x0aca] = "\xd6\xcf", [0x0acb] = "\xf1\xbb",
  [0x0acc] = "\xf3\xd2", [0x0acd] = "\xb1\xbf", [0x0ace] = "\xb5\xd1",
  [0x0acf] = "\xb5\xda", [0x0ad0] = "\xb7\xfb", [0x0ad1] = "\xf3\xcf",
  [0x0ad2] = "\xf3\xd7", [0x0ad3] = "\xf3\xd0", [0x0ad4] = "\xc1\xa3",
  [0x0ad5] = "\xb4\xd6", [0x0ad6] = "\xc6\xc9", [0x0ad7] = "\xb0\xed",
  [0x0ad8] = "\xcf\xd2", [0x0ad9] = "\xcd\xb3", [0x0ada] = "\xd4\xfa",
  [0x0adb] = "\xc9\xdc", [0x0adc] = "\xe7\xa8", [0x0add] = "\xe7\xa9",
  [0x0ade] = "\xcf\xb8", [0x0adf] = "\xc9\xf0", [0x0ae0] = "\xd7\xe9",
  [0x0ae1] = "\xc0\xdb", [0x0ae2] = "\xd6\xd5", [0x0ae3] = "\xe7\xa5",
  [0x0ae4] = "\xe7\xa6", [0x0ae5] = "\xb2\xa7", [0x0ae6] = "\xd0\xdf",
  [0x0ae7] = "\xc1\xe7", [0x0ae8] = "\xd2\xee", [0x0ae9] = "\xf4\xe1",
  [0x0aea] = "\xcf\xb0", [0x0aeb] = "\xf1\xea", [0x0aec] = "\xc1\xc4",
  [0x0aed] = "\xf1\xf6", [0x0aee] = "\xb8\xac", [0x0aef] = "\xb2\xb1",
  [0x0af0] = "\xb4\xbd", [0x0af1] = "\xcd\xd1", [0x0af2] = "\xd0\xde",
  [0x0af5] = "\xf4\xa9", [0x0af6] = "\xb6\xe6", [0x0af7] = "\xcf\xcf",
  [0x0af8] = "\xb2\xb0", [0x0af9] = "\xb4\xac", [0x0afa] = "\xc9\xaf",
  [0x0afb] = "\xdd\xb8", [0x0afc] = "\xdd\xb7", [0x0afd] = "\xdd\xa9",
  [0x0afe] = "\xbc\xd4", [0x0aff] = "\xbe\xa5", [0x0b00] = "\xc3\xa7",
  [0x0b01] = "\xc4\xaa", [0x0b02] = "\xdc\xec", [0x0b03] = "\xd7\xaf",
  [0x0b04] = "\xdd\xae", [0x0b05] = "\xc0\xf2", [0x0b06] = "\xdd\xac",
  [0x0b07] = "\xba\xc9", [0x0b08] = "\xdd\xb6", [0x0b09] = "\xdd\xb1",
  [0x0b0a] = "\xc6\xce", [0x0b0b] = "\xdc\xc8", [0x0b0c] = "\xb4\xa6",
  [0x0b0d] = "\xb1\xeb", [0x0b0e] = "\xc9\xdf", [0x0b0f] = "\xd6\xfb",
  [0x0b10] = "\xf2\xc0", [0x0b11] = "\xf2\xc1", [0x0b12] = "\xf2\xc2",
  [0x0b13] = "\xc7\xf9", [0x0b14] = "\xb5\xb0", [0x0b15] = "\xf2\xc6",
  [0x0b16] = "\xf2\xc7", [0x0b17] = "\xf2\xc8", [0x0b18] = "\xca\xf5",
  [0x0b19] = "\xd9\xf2", [0x0b1a] = "\xf4\xc2", [0x0b1b] = "\xb1\xbb",
  [0x0b1c] = "\xcc\xbb", [0x0b1d] = "\xd0\xe4", [0x0b1e] = "\xc5\xdb",
  [0x0b1f] = "\xb4\xfc", [0x0b20] = "\xc3\xd9", [0x0b21] = "\xb9\xe6",
  [0x0b22] = "\xb7\xc3", [0x0b23] = "\xd1\xc8", [0x0b24] = "\xbe\xf7",
  [0x0b25] = "\xda\xab", [0x0b26] = "\xd0\xed", [0x0b27] = "\xc9\xe8",
  [0x0b28] = "\xcb\xcf", [0x0b29] = "\xb6\xef", [0x0b2a] = "\xd0\xc0",
  [0x0b2b] = "\xf4\xf9", [0x0b2c] = "\xeb\xe0", [0x0b2d] = "\xb7\xb7",
  [0x0b2e] = "\xd4\xf0", [0x0b2f] = "\xb9\xe1", [0x0b30] = "\xbb\xf5",
  [0x0b31] = "\xcc\xb0", [0x0b32] = "\xc6\xb6", [0x0b33] = "\xf4\xf6",
  [0x0b34] = "\xc9\xe2", [0x0b35] = "\xd6\xba", [0x0b36] = "\xf5\xc3",
  [0x0b37] = "\xe9\xee", [0x0b38] = "\xc8\xed", [0x0b39] = "\xd5\xe2",
  [0x0b3a] = "\xe5\xd0", [0x0b3b] = "\xcd\xa8", [0x0b3c] = "\xb6\xba",
  [0x0b3d] = "\xc1\xac", [0x0b3e] = "\xcb\xd9", [0x0b3f] = "\xca\xc5",
  [0x0b40] = "\xd6\xf0", [0x0b41] = "\xe5\xc9", [0x0b42] = "\xb3\xd1",
  [0x0b43] = "\xd4\xec", [0x0b44] = "\xcd\xb8", [0x0b45] = "\xb7\xea",
  [0x0b46] = "\xe5\xd1", [0x0b47] = "\xb9\xe4", [0x0b48] = "\xcd\xbe",
  [0x0b49] = "\xb2\xbf", [0x0b4a] = "\xb9\xf9", [0x0b4b] = "\xb6\xbc",
  [0x0b4c] = "\xd0\xef", [0x0b4d] = "\xd2\xb0", [0x0b4e] = "\xee\xce",
  [0x0b4f] = "\xbf\xdb", [0x0b50] = "\xb5\xf6", [0x0b51] = "\xee\xcb",
  [0x0b53] = "\xb7\xb0", [0x0b54] = "\xb1\xd5", [0x0b55] = "\xc5\xe3",
  [0x0b56] = "\xc1\xea", [0x0b57] = "\xb3\xc2", [0x0b58] = "\xc2\xbd",
  [0x0b59] = "\xd2\xf5", [0x0b5a] = "\xda\xf0", [0x0b5b] = "\xcc\xd5",
  [0x0b5c] = "\xcf\xdd", [0x0b5d] = "\xda\xee", [0x0b5e] = "\xc8\xb8",
  [0x0b5f] = "\xd1\xa9", [0x0b60] = "\xf6\xa7", [0x0b61] = "\xd5\xc2",
  [0x0b62] = "\xbe\xb9", [0x0b63] = "\xb6\xa5", [0x0b64] = "\xc7\xea",
  [0x0b65] = "\xd3\xe3", [0x0b66] = "\xc4\xf1", [0x0b67] = "\xc2\xb1",
  [0x0b68] = "\xc2\xb9", [0x0b69] = "\xc2\xf3", [0x0b6a] = "\xc2\xe9",
  [0x0b6b] = "\xbc\xd2", [0x0b6c] = "\xb0\xf8", [0x0b6d] = "\xb8\xb5",
  [0x0b6e] = "\xb1\xb8", [0x0b6f] = "\xbd\xdc", [0x0b70] = "\xbf\xfe",
  [0x0b71] = "\xd8\xf7", [0x0b72] = "\xc9\xa1", [0x0b73] = "\xd0\xa7",
  [0x0b74] = "\xd7\xee", [0x0b75] = "\xbf\xad", [0x0b76] = "\xb8\xee",
  [0x0b77] = "\xd8\xdc", [0x0b78] = "\xb4\xb4", [0x0b79] = "\xca\xa3",
  [0x0b7a] = "\xc0\xcd", [0x0b7b] = "\xca\xa4", [0x0b7c] = "\xd1\xab",
  [0x0b7d] = "\xb2\xa9", [0x0b7e] = "\xd8\xca", [0x0b7f] = "\xe0\xb4",
  [0x0b80] = "\xbf\xa6", [0x0b81] = "\xd0\xfa", [0x0b82] = "\xcc\xe4",
  [0x0b83] = "\xba\xb0", [0x0b84] = "\xba\xc8", [0x0b85] = "\xb4\xad",
  [0x0b86] = "\xce\xb9", [0x0b87] = "\xcf\xb2", [0x0b88] = "\xc9\xa5",
  [0x0b89] = "\xe0\xb8", [0x0b8a] = "\xc0\xae", [0x0b8b] = "\xe0\xa9",
  [0x0b8c] = "\xe0\xab", [0x0b8d] = "\xd4\xfb", [0x0b8e] = "\xb5\xa5",
  [0x0b8f] = "\xe0\xb0", [0x0b90] = "\xcd\xd9", [0x0b91] = "\xd3\xb4",
  [0x0b92] = "\xbb\xbd", [0x0b93] = "\xd3\xf7", [0x0b94] = "\xc7\xc7",
  [0x0b95] = "\xe0\xac", [0x0b96] = "\xe0\xb1", [0x0b97] = "\xba\xed",
  [0x0b98] = "\xb3\xd4", [0x0b99] = "\xe0\xb9", [0x0b9a] = "\xce\xa7",
  [0x0b9b] = "\xd2\xa2", [0x0b9c] = "\xbf\xb0", [0x0b9d] = "\xb3\xa1",
  [0x0b9e] = "\xb5\xcc", [0x0b9f] = "\xd1\xdf", [0x0ba0] = "\xb1\xa8",
  [0x0ba1] = "\xb1\xa4", [0x0ba2] = "\xdb\xf6", [0x0ba3] = "\xdc\xa9",
  [0x0ba4] = "\xd2\xbc", [0x0ba5] = "\xba\xf8", [0x0ba6] = "\xb5\xec",
  [0x0ba7] = "\xe6\xc3", [0x0ba8] = "\xc3\xc4", [0x0ba9] = "\xd0\xf6",
  [0x0baa] = "\xc3\xbd", [0x0bab] = "\xe6\xc2", [0x0bac] = "\xe6\xb4",
  [0x0bad] = "\xe6\xdc", [0x0bae] = "\xe5\xee", [0x0baf] = "\xba\xae",
  [0x0bb0] = "\xb8\xbb", [0x0bb1] = "\xd4\xa2", [0x0bb2] = "\xc3\xc2",
  [0x0bb3] = "\xd7\xf0", [0x0bb4] = "\xd1\xb0", [0x0bb5] = "\xbe\xcd",
  [0x0bb6] = "\xc7\xb6", [0x0bb7] = "\xe1\xb0", [0x0bb8] = "\xe1\xcb",
  [0x0bb9] = "\xef\xfa", [0x0bba] = "\xd9\xe3", [0x0bbb] = "\xb7\xf9",
  [0x0bbc] = "\xc3\xb1", [0x0bbd] = "\xd6\xa1", [0x0bbe] = "\xe0\xf8",
  [0x0bbf] = "\xbc\xb8", [0x0bc0] = "\xc0\xc8", [0x0bc1] = "\xb2\xde",
  [0x0bc2] = "\xcf\xe1", [0x0bc3] = "\xbe\xc7", [0x0bc4] = "\xe5\xf6",
  [0x0bc5] = "\xc5\xed", [0x0bc6] = "\xb8\xb4", [0x0bc7] = "\xd1\xad",
  [0x0bc8] = "\xe1\xe5", [0x0bc9] = "\xbb\xf3", [0x0bca] = "\xb6\xf1",
  [0x0bcb] = "\xb1\xaf", [0x0bcc] = "\xc3\xc6", [0x0bcd] = "\xbb\xdd",
  [0x0bce] = "\xe3\xab", [0x0bcf] = "\xe3\xb6", [0x0bd0] = "\xd0\xca",
  [0x0bd1] = "\xe3\xb5", [0x0bd2] = "\xb6\xe8", [0x0bd3] = "\xe2\xfc",
  [0x0bd4] = "\xe3\xb7", [0x0bd5] = "\xbf\xae", [0x0bd6] = "\xc4\xd5",
  [0x0bd7] = "\xe3\xb9", [0x0bd8] = "\xbb\xcc", [0x0bd9] = "\xd3\xe4",
  [0x0bda] = "\xe3\xb8", [0x0bdc] = "\xea\xaa", [0x0bdd] = "\xec\xe9",
  [0x0bde] = "\xb3\xb8", [0x0bdf] = "\xd5\xc6", [0x0be0] = "\xc3\xe8",
  [0x0be1] = "\xbc\xf0", [0x0be2] = "\xbf\xab", [0x0be3] = "\xc8\xe0",
  [0x0be4] = "\xde\xf1", [0x0be5] = "\xd7\xe1", [0x0be6] = "\xb2\xe5",
  [0x0be7] = "\xb4\xa7", [0x0be8] = "\xcc\xe1", [0x0be9] = "\xce\xd5",
  [0x0bea] = "\xd2\xbe", [0x0beb] = "\xbd\xd2", [0x0bec] = "\xbb\xd3",
  [0x0bed] = "\xb4\xb7", [0x0bee] = "\xd4\xae", [0x0bef] = "\xbe\xbe",
  [0x0bf0] = "\xbb\xbb", [0x0bf1] = "\xde\xf0", [0x0bf2] = "\xd1\xef",
  [0x0bf3] = "\xb1\xb3", [0x0bf4] = "\xb3\xa8", [0x0bf5] = "\xb6\xd8",
  [0x0bf6] = "\xb8\xd2", [0x0bf7] = "\xc9\xa2", [0x0bf8] = "\xb0\xdf",
  [0x0bf9] = "\xec\xb3", [0x0bfa] = "\xcb\xb9", [0x0bfb] = "\xc6\xd5",
  [0x0bfc] = "\xce\xfa", [0x0bfd] = "\xc7\xe7", [0x0bfe] = "\xbe\xa7",
  [0x0bff] = "\xbe\xb0", [0x0c00] = "\xca\xee", [0x0c01] = "\xd6\xc7",
  [0x0c02] = "\xc1\xc0", [0x0c03] = "\xea\xd0", [0x0c04] = "\xd4\xf8",
  [0x0c05] = "\xcc\xe6", [0x0c06] = "\xc6\xda", [0x0c07] = "\xb3\xaf",
  [0x0c08] = "\xb9\xd7", [0x0c09] = "\xd7\xd8", [0x0c0a] = "\xcc\xc4",
  [0x0c0b] = "\xbc\xac", [0x0c0c] = "\xd4\xe6", [0x0c0d] = "\xd2\xce",
  [0x0c0e] = "\xb6\xb0", [0x0c0f] = "\xbf\xc3", [0x0c10] = "\xc9\xad",
  [0x0c11] = "\xd5\xbb", [0x0c12] = "\xe8\xfe", [0x0c13] = "\xb0\xf4",
  [0x0c14] = "\xc6\xdc", [0x0c15] = "\xe9\xa6", [0x0c16] = "\xc6\xe5",
  [0x0c17] = "\xb9\xf7", [0x0c18] = "\xd6\xb2", [0x0c19] = "\xbd\xb7",
  [0x0c1a] = "\xd7\xb5", [0x0c1b] = "\xc3\xde", [0x0c1c] = "\xc5\xef",
  [0x0c1d] = "\xe8\xfa", [0x0c1f] = "\xbf\xee", [0x0c20] = "\xc6\xdb",
  [0x0c21] = "\xc7\xd5", [0x0c22] = "\xb2\xd0", [0x0c23] = "\xd6\xb3",
  [0x0c24] = "\xbf\xc7", [0x0c25] = "\xcc\xba", [0x0c26] = "\xb5\xaa",
  [0x0c27] = "\xc2\xc8", [0x0c28] = "\xeb\xb2", [0x0c29] = "\xb8\xdb",
  [0x0c2a] = "\xd3\xce", [0x0c2b] = "\xe4\xd5", [0x0c2c] = "\xb6\xc9",
  [0x0c2d] = "\xe4\xd6", [0x0c2e] = "\xd3\xbf", [0x0c2f] = "\xb4\xd5",
  [0x0c30] = "\xc7\xfe", [0x0c31] = "\xe4\xd7", [0x0c32] = "\xd4\xfc",
  [0x0c33] = "\xbc\xf5", [0x0c34] = "\xd5\xbf", [0x0c35] = "\xcf\xe6",
  [0x0c36] = "\xb2\xb3", [0x0c37] = "\xba\xfe", [0x0c38] = "\xe4\xce",
  [0x0c39] = "\xce\xbc", [0x0c3a] = "\xce\xd0", [0x0c3b] = "\xcc\xc0",
  [0x0c3c] = "\xbf\xca", [0x0c3d] = "\xcd\xc4", [0x0c3e] = "\xc3\xec",
  [0x0c3f] = "\xb2\xe2", [0x0c40] = "\xc5\xc8", [0x0c41] = "\xd3\xe5",
  [0x0c42] = "\xbb\xeb", [0x0c43] = "\xd7\xcc", [0x0c44] = "\xb8\xc8",
  [0x0c45] = "\xbb\xc1", [0x0c46] = "\xe4\xcf", [0x0c47] = "\xe3\xc9",
  [0x0c48] = "\xe4\xd8", [0x0c4b] = "\xe4\xd2", [0x0c4c] = "\xb1\xba",
  [0x0c4d] = "\xb7\xd9", [0x0c4e] = "\xbd\xb9", [0x0c4f] = "\xd1\xe6",
  [0x0c50] = "\xce\xde", [0x0c51] = "\xc8\xbb", [0x0c52] = "\xd6\xf3",
  [0x0c54] = "\xc5\xc6", [0x0c55] = "\xea\xf7", [0x0c56] = "\xcf\xac",
  [0x0c57] = "\xd3\xcc", [0x0c58] = "\xe2\xab", [0x0c59] = "\xba\xef",
  [0x0c5a] = "\xd0\xc9", [0x0c5b] = "\xb7\xa9", [0x0c5c] = "\xe7\xf7",
  [0x0c5d] = "\xc1\xd5", [0x0c5e] = "\xd7\xc1", [0x0c5f] = "\xe7\xfa",
  [0x0c60] = "\xc5\xfd", [0x0c61] = "\xc5\xc3", [0x0c62] = "\xc7\xd9",
  [0x0c64] = "\xe8\xa1", [0x0c65] = "\xe7\xf9", [0x0c66] = "\xe7\xfb",
  [0x0c67] = "\xc9\xfb", [0x0c68] = "\xcb\xd5", [0x0c69] = "\xbb\xad",
  [0x0c6a] = "\xb7\xac", [0x0c6b] = "\xc1\xa1", [0x0c6c] = "\xcd\xb4",
  [0x0c6d] = "\xf0\xeb", [0x0c6e] = "\xbe\xb7", [0x0c6f] = "\xb6\xbb",
  [0x0c70] = "\xc6\xa6", [0x0c71] = "\xcb\xe1", [0x0c72] = "\xb5\xc7",
  [0x0c73] = "\xb7\xa2", [0x0c74] = "\xcd\xee", [0x0c75] = "\xf0\xa9",
  [0x0c76] = "\xf1\xe5", [0x0c77] = "\xb5\xc1", [0x0c78] = "\xc0\xa7",
  [0x0c79] = "\xb6\xcc", [0x0c7a] = "\xcf\xf5", [0x0c7b] = "\xd3\xb2",
  [0x0c7c] = "\xd1\xe2", [0x0c7d] = "\xc9\xd4", [0x0c7e] = "\xb8\xd1",
  [0x0c7f] = "\xb3\xcc", [0x0c80] = "\xcb\xb0", [0x0c81] = "\xcf\xa1",
  [0x0c82] = "\xbe\xbd", [0x0c83] = "\xb4\xb0", [0x0c84] = "\xbd\xd1",
  [0x0c85] = "\xcd\xaf", [0x0c86] = "\xbf\xa2", [0x0c87] = "\xb5\xc8",
  [0x0c88] = "\xb2\xdf", [0x0c89] = "\xb1\xca", [0x0c8a] = "\xbf\xf0",
  [0x0c8b] = "\xcd\xb2", [0x0c8c] = "\xb4\xf0", [0x0c8d] = "\xcb\xf1",
  [0x0c8e] = "\xbd\xee", [0x0c8f] = "\xb7\xa4", [0x0c90] = "\xd6\xfe",
  [0x0c91] = "\xcb\xda", [0x0c92] = "\xd6\xe0", [0x0c93] = "\xbd\xca",
  [0x0c94] = "\xbd\xe1", [0x0c95] = "\xc8\xde", [0x0c96] = "\xbe\xf8",
  [0x0c97] = "\xd7\xcf", [0x0c98] = "\xd0\xf5", [0x0c99] = "\xcb\xbf",
  [0x0c9a] = "\xc2\xe7", [0x0c9b] = "\xb8\xf8", [0x0c9c] = "\xd1\xa4",
  [0x0c9e] = "\xe7\xad", [0x0c9f] = "\xc9\xc6", [0x0ca0] = "\xcf\xe8",
  [0x0ca1] = "\xf4\xe2", [0x0ca2] = "\xf1\xf3", [0x0ca3] = "\xf1\xf8",
  [0x0ca4] = "\xcb\xe0", [0x0ca5] = "\xcd\xf3", [0x0ca6] = "\xc7\xbb",
  [0x0ca7] = "\xd2\xb8", [0x0ca8] = "\xb8\xad", [0x0ca9] = "\xc9\xf6",
  [0x0caa] = "\xd5\xcd", [0x0cab] = "\xcc\xf3", [0x0cac] = "\xc6\xa2",
  [0x0cad] = "\xeb\xe7", [0x0cae] = "\xeb\xe8", [0x0caf] = "\xeb\xe9",
  [0x0cb0] = "\xca\xe6", [0x0cb1] = "\xcb\xb4", [0x0cb2] = "\xc6\xd0",
  [0x0cb3] = "\xdd\xcd", [0x0cb4] = "\xdd\xce", [0x0cb5] = "\xc6\xbc",
  [0x0cb6] = "\xb2\xa4", [0x0cb7] = "\xdd\xd1", [0x0cb8] = "\xdd\xc2",
  [0x0cb9] = "\xdd\xbc", [0x0cba] = "\xbb\xaa", [0x0cbb] = "\xc1\xe2",
  [0x0cbc] = "\xe2\xd6", [0x0cbd] = "\xd6\xf8", [0x0cbe] = "\xc0\xb3",
  [0x0cbf] = "\xdd\xd4", [0x0cc0] = "\xc3\xc8", [0x0cc1] = "\xbe\xfa",
  [0x0cc2] = "\xdd\xc4", [0x0cc3] = "\xb7\xc6", [0x0cc4] = "\xbe\xd5",
  [0x0cc5] = "\xdd\xc7", [0x0cc6] = "\xce\xae", [0x0cc7] = "\xcc\xd1",
  [0x0cc8] = "\xb2\xcb", [0x0cc9] = "\xdc\xc9", [0x0cca] = "\xdd\xca",
  [0x0ccb] = "\xdd\xcb", [0x0ccc] = "\xd0\xe9", [0x0ccd] = "\xf2\xd4",
  [0x0cce] = "\xcd\xdc", [0x0ccf] = "\xf2\xce", [0x0cd0] = "\xbb\xd7",
  [0x0cd1] = "\xd6\xeb", [0x0cd2] = "\xb8\xf2", [0x0cd3] = "\xf2\xd0",
  [0x0cd4] = "\xf2\xd2", [0x0cd5] = "\xbd\xd6", [0x0cd6] = "\xb2\xc3",
  [0x0cd7] = "\xc1\xd1", [0x0cd8] = "\xb8\xa4", [0x0cd9] = "\xf1\xfb",
  [0x0cda] = "\xca\xd3", [0x0cdb] = "\xd7\xa2", [0x0cdc] = "\xd3\xbd",
  [0x0cdd] = "\xc6\xc0", [0x0cde] = "\xb4\xca", [0x0cdf] = "\xd6\xa4",
  [0x0ce0] = "\xda\xac", [0x0ce1] = "\xda\xaf", [0x0ce2] = "\xd7\xe7",
  [0x0ce3] = "\xd5\xa9", [0x0ce4] = "\xda\xae", [0x0ce5] = "\xcb\xdf",
  [0x0ce6] = "\xd5\xef", [0x0ce7] = "\xda\xad", [0x0ce9] = "\xcf\xf3",
  [0x0cea] = "\xf5\xf5", [0x0ceb] = "\xd6\xfc", [0x0cec] = "\xcc\xf9",
  [0x0ced] = "\xb7\xa1", [0x0cee] = "\xea\xdd", [0x0cef] = "\xea\xda",
  [0x0cf0] = "\xb7\xd1", [0x0cf1] = "\xba\xd8", [0x0cf2] = "\xb9\xf3",
  [0x0cf3] = "\xc2\xf2", [0x0cf4] = "\xb1\xe1", [0x0cf5] = "\xc3\xb3",
  [0x0cf6] = "\xb4\xfb", [0x0cf7] = "\xd4\xbd", [0x0cf8] = "\xb3\xac",
  [0x0cf9] = "\xb3\xc3", [0x0cfa] = "\xf5\xc9", [0x0cfb] = "\xbe\xe0",
  [0x0cfc] = "\xb0\xcf", [0x0cfd] = "\xf5\xc7", [0x0cfe] = "\xc5\xdc",
  [0x0cff] = "\xb5\xf8", [0x0d00] = "\xf5\xcb", [0x0d01] = "\xf5\xcc",
  [0x0d02] = "\xe9\xf0", [0x0d03] = "\xd6\xe1", [0x0d04] = "\xe9\xf3",
  [0x0d05] = "\xb9\xbc", [0x0d06] = "\xb4\xfe", [0x0d07] = "\xe5\xd3",
  [0x0d08] = "\xd6\xdc", [0x0d09] = "\xd2\xdd", [0x0d0a] = "\xbd\xf8",
  [0x0d0b] = "\xe5\xd4", [0x0d0c] = "\xb6\xf5", [0x0d0d] = "\xd3\xca",
  [0x0d0e] = "\xcf\xe7", [0x0d0f] = "\xdb\xb1", [0x0d10] = "\xba\xa8",
  [0x0d11] = "\xcb\xd6", [0x0d12] = "\xc1\xbf", [0x0d13] = "\xb3\xae",
  [0x0d14] = "\xc5\xa5", [0x0d15] = "\xb8\xc6", [0x0d16] = "\xc4\xc6",
  [0x0d17] = "\xbe\xfb", [0x0d18] = "\xb6\xdb", [0x0d19] = "\xee\xd4",
  [0x0d1b] = "\xee\xd3", [0x0d1c] = "\xe3\xc9", [0x0d1d] = "\xc8\xf2",
  [0x0d1e] = "\xbf\xaa", [0x0d1f] = "\xcf\xd0", [0x0d20] = "\xbc\xe4",
  [0x0d21] = "\xcf\xd0", [0x0d22] = "\xe3\xc8", [0x0d23] = "\xb6\xd3",
  [0x0d24] = "\xbd\xd7", [0x0d25] = "\xcb\xe5", [0x0d26] = "\xd1\xf4",
  [0x0d27] = "\xd3\xe7", [0x0d28] = "\xc2\xa1", [0x0d29] = "\xda\xf2",
  [0x0d2a] = "\xda\xef", [0x0d2b] = "\xb5\xcc", [0x0d2c] = "\xd1\xe3",
  [0x0d2d] = "\xd1\xc5", [0x0d2e] = "\xd0\xdb", [0x0d2f] = "\xbc\xaf",
  [0x0d30] = "\xb9\xcd", [0x0d31] = "\xf6\xa9", [0x0d32] = "\xd4\xc6",
  [0x0d33] = "\xc8\xcd", [0x0d34] = "\xcf\xee", [0x0d35] = "\xcb\xb3",
  [0x0d36] = "\xd0\xeb", [0x0d37] = "\xe2\xb8", [0x0d38] = "\xe2\xbf",
  [0x0d39] = "\xb7\xb9", [0x0d3a] = "\xe2\xbd", [0x0d3b] = "\xd2\xfb",
  [0x0d3c] = "\xe2\xc1", [0x0d3d] = "\xb7\xeb", [0x0d3e] = "\xd4\xa6",
  [0x0d3f] = "\xbb\xc6", [0x0d40] = "\xca\xf2", [0x0d41] = "\xba\xda",
  [0x0d42] = "\xc2\xd2", [0x0d43] = "\xd3\xb6", [0x0d44] = "\xd5\xae",
  [0x0d45] = "\xb0\xc1", [0x0d46] = "\xb4\xab", [0x0d47] = "\xbd\xf6",
  [0x0d48] = "\xc7\xe3", [0x0d49] = "\xb4\xdf", [0x0d4a] = "\xc9\xcb",
  [0x0d4b] = "\xc9\xb5", [0x0d4c] = "\xd9\xcc", [0x0d4e] = "\xbd\xcb",
  [0x0d4f] = "\xb2\xf9", [0x0d50] = "\xd8\xe2", [0x0d51] = "\xc4\xbc",
  [0x0d52] = "\xbd\xcb", [0x0d53] = "\xc7\xda", [0x0d54] = "\xca\xc6",
  [0x0d55] = "\xbc\xa8", [0x0d56] = "\xbb\xe3", [0x0d57] = "\xe0\xb5",
  [0x0d58] = "\xe0\xcb", [0x0d59] = "\xc9\xa4", [0x0d5a] = "\xe0\xc2",
  [0x0d5b] = "\xc2\xf0", [0x0d5c] = "\xca\xc8", [0x0d5d] = "\xd8\xc4",
  [0x0d5e] = "\xe0\xbe", [0x0d5f] = "\xcb\xc3", [0x0d60] = "\xe0\xcd",
  [0x0d61] = "\xe0\xc5", [0x0d62] = "\xce\xd8", [0x0d63] = "\xce\xcb",
  [0x0d64] = "\xd0\xe1", [0x0d65] = "\xc7\xba", [0x0d66] = "\xe0\xc6",
  [0x0d67] = "\xe0\xbc", [0x0d68] = "\xd4\xb0", [0x0d69] = "\xd4\xb2",
  [0x0d6a] = "\xc8\xfb", [0x0d6b] = "\xcb\xdc", [0x0d6c] = "\xcc\xc1",
  [0x0d6d] = "\xcd\xbf", [0x0d6e] = "\xda\xa3", [0x0d6f] = "\xcb\xfe",
  [0x0d70] = "\xcc\xee", [0x0d71] = "\xcb\xfa", [0x0d73] = "\xbf\xe9",
  [0x0d74] = "\xce\xeb", [0x0d75] = "\xdb\xf5", [0x0d76] = "\xdc\xe3",
  [0x0d77] = "\xb0\xc2", [0x0d78] = "\xbc\xde", [0x0d79] = "\xbc\xb5",
  [0x0d7a] = "\xcf\xd3", [0x0d7b] = "\xe6\xc5", [0x0d7c] = "\xc2\xe8",
  [0x0d7d] = "\xe6\xc1", [0x0d7e] = "\xcf\xb1", [0x0d7f] = "\xc9\xa9",
  [0x0d80] = "\xe6\xc7", [0x0d81] = "\xe1\xd4", [0x0d82] = "\xe1\xcf",
  [0x0d83] = "\xbb\xcf", [0x0d84] = "\xb8\xc9", [0x0d85] = "\xc1\xae",
  [0x0d86] = "\xcf\xc3", [0x0d87] = "\xdf\xb1", [0x0d88] = "\xbb\xe3",
  [0x0d89] = "\xc5\xd4", [0x0d8a] = "\xce\xa2", [0x0d8b] = "\xd3\xde",
  [0x0d8c] = "\xd2\xe2", [0x0d8d] = "\xb4\xc8", [0x0d8e] = "\xb8\xd0",
  [0x0d8f] = "\xcf\xeb", [0x0d90] = "\xb0\xae", [0x0d91] = "\xc8\xc7",
  [0x0d92] = "\xb3\xee", [0x0d93] = "\xd3\xfa", [0x0d94] = "\xc9\xf7",
  [0x0d95] = "\xbb\xc5", [0x0d96] = "\xc0\xf5", [0x0d97] = "\xe3\xb3",
  [0x0d98] = "\xe2\xe9", [0x0d99] = "\xe2\xeb", [0x0d9a] = "\xc0\xa2",
  [0x0d9b] = "\xed\xaa", [0x0d9c] = "\xed\xa9", [0x0d9d] = "\xe2\xfd",
  [0x0d9e] = "\xea\xac", [0x0d9f] = "\xea\xab", [0x0da0] = "\xb4\xea",
  [0x0da1] = "\xd5\xa5", [0x0da2] = "\xb8\xe3", [0x0da3] = "\xcc\xc2",
  [0x0da4] = "\xb4\xee", [0x0da5] = "\xb2\xeb", [0x0da6] = "\xb0\xe1",
  [0x0da7] = "\xb2\xab", [0x0da8] = "\xcb\xd1", [0x0da9] = "\xc9\xa6",
  [0x0daa] = "\xcb\xf0", [0x0dab] = "\xc7\xc0", [0x0dac] = "\xd2\xa1",
  [0x0dad] = "\xb5\xb7", [0x0dae] = "\xb9\xb9", [0x0daf] = "\xbe\xb4",
  [0x0db0] = "\xd5\xe5", [0x0db1] = "\xd0\xc2", [0x0db2] = "\xb0\xb5",
  [0x0db3] = "\xea\xcd", [0x0db4] = "\xcf\xbe", [0x0db5] = "\xd4\xce",
  [0x0db6] = "\xc5\xaf", [0x0db7] = "\xea\xd1", [0x0db8] = "\xd1\xf4",
  [0x0dba] = "\xbb\xe1", [0x0dbb] = "\xc0\xc6", [0x0dbc] = "\xd2\xb5",
  [0x0dbd] = "\xb3\xfe", [0x0dbe] = "\xbf\xac", [0x0dbf] = "\xe9\xaa",
  [0x0dc0] = "\xd0\xa8", [0x0dc1] = "\xbc\xab", [0x0dc2] = "\xd2\xac",
  [0x0dc3] = "\xb8\xc5", [0x0dc4] = "\xd1\xee", [0x0dc5] = "\xe8\xe5",
  [0x0dc6] = "\xe9\xae", [0x0dc7] = "\xc0\xe3", [0x0dc8] = "\xb7\xe3",
  [0x0dc9] = "\xe9\xba", [0x0dca] = "\xd3\xdc", [0x0dcb] = "\xe9\xac",
  [0x0dcc] = "\xe9\xb9", [0x0dce] = "\xd0\xaa", [0x0dcf] = "\xcb\xea",
  [0x0dd0] = "\xbb\xd9", [0x0dd1] = "\xb5\xee", [0x0dd2] = "\xd8\xb9",
  [0x0dd3] = "\xeb\xa6", [0x0dd4] = "\xd2\xe7", [0x0dd5] = "\xcb\xdd",
  [0x0dd6] = "\xd7\xd2", [0x0dd7] = "\xc8\xdc", [0x0dd8] = "\xe4\xe8",
  [0x0dd9] = "\xd4\xb4", [0x0dda] = "\xb9\xb5", [0x0ddb] = "\xb5\xe1",
  [0x0ddc] = "\xc3\xf0", [0x0ddd] = "\xe4\xdf", [0x0dde] = "\xe4\xdb",
  [0x0ddf] = "\xca\xaa", [0x0de0] = "\xc4\xe7", [0x0de1] = "\xce\xc2",
  [0x0de2] = "\xbb\xac", [0x0de3] = "\xd7\xbc", [0x0de4] = "\xc1\xef",
  [0x0de5] = "\xb2\xd7", [0x0de6] = "\xcc\xcf", [0x0de7] = "\xcf\xaa",
  [0x0de8] = "\xe4\xe0", [0x0de9] = "\xe4\xe5", [0x0dea] = "\xbc\xe5",
  [0x0deb] = "\xd1\xcc", [0x0dec] = "\xb7\xb3", [0x0ded] = "\xc3\xba",
  [0x0dee] = "\xc1\xb6", [0x0def] = "\xd5\xd5", [0x0df0] = "\xec\xcf",
  [0x0df1] = "\xec\xbe", [0x0df2] = "\xec\xe3", [0x0df3] = "\xbb\xcd",
  [0x0df4] = "\xbb\xc0", [0x0df5] = "\xc9\xb7", [0x0df7] = "\xec\xd0",
  [0x0df8] = "\xc5\xaf", [0x0df9] = "\xd2\xaf", [0x0dfa] = "\xeb\xba",
  [0x0dfb] = "\xe9\xe0", [0x0dfc] = "\xca\xa8", [0x0dfd] = "\xd4\xb3",
  [0x0dfe] = "\xbb\xab", [0x0dff] = "\xc0\xc5", [0x0e00] = "\xba\xf7",
  [0x0e01] = "\xe8\xa6", [0x0e02] = "\xc9\xaa", [0x0e03] = "\xc8\xf0",
  [0x0e04] = "\xe8\xa3", [0x0e05] = "\xe7\xf5", [0x0e06] = "\xe8\xa7",
  [0x0e07] = "\xe7\xf8", [0x0e08] = "\xe8\xa4", [0x0e09] = "\xb5\xb1",
  [0x0e0a] = "\xbb\xfb", [0x0e0b] = "\xf0\xf6", [0x0e0c] = "\xcc\xb5",
  [0x0e0d] = "\xb4\xe1", [0x0e0e] = "\xc2\xe9", [0x0e0f] = "\xf0\xf2",
  [0x0e10] = "\xb1\xd4", [0x0e11] = "\xf0\xf4", [0x0e12] = "\xb3\xd5",
  [0x0e13] = "\xc2\xe9", [0x0e14] = "\xd5\xb5", [0x0e15] = "\xc3\xcb",
  [0x0e16] = "\xbe\xa6", [0x0e17] = "\xbd\xde", [0x0e18] = "\xc4\xc0",
  [0x0e19] = "\xed\xf9", [0x0e1a] = "\xb6\xbd", [0x0e1b] = "\xb6\xc3",
  [0x0e1c] = "\xd8\xba", [0x0e1d] = "\xb2\xc7", [0x0e1e] = "\xd5\xf6",
  [0x0e1f] = "\xee\xa2", [0x0e20] = "\xed\xfe", [0x0e21] = "\xee\xa1",
  [0x0e22] = "\xb0\xab", [0x0e23] = "\xcb\xe9", [0x0e24] = "\xc5\xf6",
  [0x0e25] = "\xcd\xeb", [0x0e26] = "\xb5\xe2", [0x0e27] = "\xc2\xb5",
  [0x0e28] = "\xb5\xef", [0x0e29] = "\xc5\xf0", [0x0e2a] = "\xb1\xae",
  [0x0e2b] = "\xed\xd4", [0x0e2d] = "\xec\xf7", [0x0e2e] = "\xc2\xbb",
  [0x0e2f] = "\xbd\xfb", [0x0e30] = "\xcd\xf2", [0x0e31] = "\xc7\xdd",
  [0x0e32] = "\xc0\xe2", [0x0e33] = "\xd6\xc9", [0x0e34] = "\xb3\xed",
  [0x0e35] = "\xef\xfe", [0x0e36] = "\xd9\xf7", [0x0e37] = "\xef\xfd",
  [0x0e38] = "\xbf\xdf", [0x0e39] = "\xf1\xbd", [0x0e3a] = "\xbf\xea",
  [0x0e3b] = "\xbd\xda", [0x0e3c] = "\xf3\xde", [0x0e3d] = "\xf3\xdf",
  [0x0e3e] = "\xf3\xc8", [0x0e3f] = "\xc1\xbb", [0x0e40] = "\xbe\xac",
  [0x0e41] = "\xd4\xc1", [0x0e42] = "\xbe\xad", [0x0e43] = "\xbe\xee",
  [0x0e44] = "\xc0\xa6", [0x0e45] = "\xb0\xf3", [0x0e46] = "\xcb\xe7",
  [0x0e47] = "\xcc\xd0", [0x0e48] = "\xd6\xc3", [0x0e49] = "\xd5\xd6",
  [0x0e4a] = "\xd7\xef", [0x0e4b] = "\xca\xf0", [0x0e4c] = "\xd2\xe5",
  [0x0e4d] = "\xcf\xdb", [0x0e4e] = "\xc8\xba", [0x0e4f] = "\xca\xa5",
  [0x0e50] = "\xc6\xb8", [0x0e51] = "\xcb\xc1", [0x0e52] = "\xd2\xde",
  [0x0e53] = "\xeb\xec", [0x0e54] = "\xd1\xfc", [0x0e55] = "\xb3\xa6",
  [0x0e56] = "\xd0\xc8", [0x0e57] = "\xc8\xf9", [0x0e58] = "\xbd\xc5",
  [0x0e59] = "\xd6\xd7", [0x0e5a] = "\xb8\xb9", [0x0e5b] = "\xcf\xd9",
  [0x0e5c] = "\xc4\xd4", [0x0e5d] = "\xbe\xcb", [0x0e5e] = "\xcd\xa7",
  [0x0e5f] = "\xb5\xd9", [0x0e60] = "\xbb\xe7", [0x0e61] = "\xc2\xe4",
  [0x0e62] = "\xdd\xe6", [0x0e63] = "\xbf\xfb", [0x0e64] = "\xce\xad",
  [0x0e65] = "\xba\xf9", [0x0e66] = "\xd2\xb6", [0x0e67] = "\xd4\xe1",
  [0x0e68] = "\xb8\xf0", [0x0e69] = "\xdd\xe0", [0x0e6a] = "\xdd\xab",
  [0x0e6b] = "\xc6\xcf", [0x0e6c] = "\xb6\xad", [0x0e6d] = "\xdd\xe2",
  [0x0e6e] = "\xdd\xe7", [0x0e6f] = "\xdd\xe1", [0x0e70] = "\xd3\xdd",
  [0x0e71] = "\xc2\xb2", [0x0e72] = "\xba\xc5", [0x0e73] = "\xd3\xbc",
  [0x0e74] = "\xf2\xd1", [0x0e75] = "\xf2\xda", [0x0e76] = "\xf2\xd8",
  [0x0e77] = "\xca\xf1", [0x0e78] = "\xb6\xea", [0x0e79] = "\xcd\xc9",
  [0x0e7a] = "\xb7\xe4", [0x0e7b] = "\xf2\xd7", [0x0e7c] = "\xf2\xb9",
  [0x0e7d] = "\xf2\xdb", [0x0e7e] = "\xd1\xc3", [0x0e7f] = "\xf4\xc4",
  [0x0e80] = "\xd2\xe1", [0x0e81] = "\xc8\xb9", [0x0e82] = "\xb2\xb9",
  [0x0e83] = "\xf4\xc3", [0x0e84] = "\xd7\xb0", [0x0e85] = "\xc0\xef",
  [0x0e86] = "\xf4\xc1", [0x0e87] = "\xd4\xa3", [0x0e88] = "\xd9\xf6",
  [0x0e89] = "\xcc\xf7", [0x0e8a] = "\xbd\xe2", [0x0e8b] = "\xb2\xef",
  [0x0e8c] = "\xb8\xc3", [0x0e8d] = "\xcf\xea", [0x0e8e] = "\xca\xd4",
  [0x0e8f] = "\xca\xab", [0x0e90] = "\xda\xb5", [0x0e91] = "\xbf\xe4",
  [0x0e92] = "\xda\xb6", [0x0e93] = "\xd2\xe8", [0x0e94] = "\xb3\xcf",
  [0x0e95] = "\xbb\xb0", [0x0e96] = "\xd6\xef", [0x0e97] = "\xb9\xee",
  [0x0e98] = "\xd1\xaf", [0x0e99] = "\xda\xb9", [0x0e9a] = "\xda\xb8",
  [0x0e9b] = "\xd5\xb2", [0x0e9c] = "\xbf\xa9", [0x0e9d] = "\xf6\xa4",
  [0x0e9f] = "\xbb\xbf", [0x0ea0] = "\xf5\xf6", [0x0ea1] = "\xba\xd1",
  [0x0ea2] = "\xd4\xf4", [0x0ea3] = "\xd7\xca", [0x0ea4] = "\xbc\xd6",
  [0x0ea5] = "\xbb\xdf", [0x0ea6] = "\xea\xdf", [0x0ea7] = "\xc1\xde",
  [0x0ea8] = "\xc2\xb8", [0x0ea9] = "\xea\xe0", [0x0eaa] = "\xbc\xa3",
  [0x0eab] = "\xb8\xfa", [0x0eac] = "\xbf\xe7", [0x0ead] = "\xc2\xb7",
  [0x0eae] = "\xcc\xf8", [0x0eaf] = "\xb6\xe5", [0x0eb0] = "\xb9\xf2",
  [0x0eb1] = "\xf5\xd3", [0x0eb3] = "\xb6\xe3", [0x0eb4] = "\xbd\xcf",
  [0x0eb5] = "\xd4\xd8", [0x0eb6] = "\xe9\xf8", [0x0eb7] = "\xe9\xf9",
  [0x0eb8] = "\xb1\xd9", [0x0eb9] = "\xc5\xa9", [0x0eba] = "\xd4\xcb",
  [0x0ebb] = "\xd3\xce", [0x0ebc] = "\xb5\xc0", [0x0ebd] = "\xcb\xec",
  [0x0ebe] = "\xb4\xef", [0x0ebf] = "\xb1\xc6", [0x0ec0] = "\xce\xa5",
  [0x0ec1] = "\xe5\xda", [0x0ec2] = "\xd3\xf6", [0x0ec3] = "\xb6\xf4",
  [0x0ec4] = "\xb9\xfd", [0x0ec5] = "\xb1\xe9", [0x0ec6] = "\xe5\xd8",
  [0x0ec7] = "\xd3\xe2", [0x0ec8] = "\xb6\xdd", [0x0ec9] = "\xd7\xde",
  [0x0ecb] = "\xb3\xea", [0x0ecc] = "\xc0\xd2", [0x0ecd] = "\xf5\xa4",
  [0x0ece] = "\xd3\xd4", [0x0ecf] = "\xee\xdc", [0x0ed0] = "\xc7\xaf",
  [0x0ed1] = "\xee\xe0", [0x0ed2] = "\xee\xdf", [0x0ed3] = "\xbc\xd8",
  [0x0ed4] = "\xd3\xcb", [0x0ed5] = "\xc7\xa6", [0x0ed6] = "\xc5\xd9",
  [0x0ed7] = "\xb9\xb3", [0x0ed8] = "\xb2\xac", [0x0ed9] = "\xc1\xe5",
  [0x0eda] = "\xee\xe7", [0x0edb] = "\xee\xe9", [0x0edc] = "\xbe\xde",
  [0x0edd] = "\xee\xeb", [0x0ede] = "\xee\xe4", [0x0edf] = "\xc3\xad",
  [0x0ee0] = "\xd5\xa2", [0x0ee1] = "\xb0\xaf", [0x0ee2] = "\xb8\xf4",
  [0x0ee3] = "\xd4\xc9", [0x0ee4] = "\xd3\xba", [0x0ee5] = "\xf6\xc1",
  [0x0ee6] = "\xef\xf4", [0x0ee8] = "\xc0\xd7", [0x0ee9] = "\xb5\xe7",
  [0x0eea] = "\xb1\xa2", [0x0eeb] = "\xc1\xe3", [0x0eec] = "\xbe\xb8",
  [0x0eed] = "\xd1\xa5", [0x0eee] = "\xb0\xd0", [0x0eef] = "\xd4\xa4",
  [0x0ef0] = "\xcd\xe7", [0x0ef1] = "\xb6\xd9", [0x0ef2] = "\xe7\xef",
  [0x0ef3] = "\xb0\xe4", [0x0ef4] = "\xcb\xcc", [0x0ef5] = "\xcb\xc7",
  [0x0ef6] = "\xe2\xc2", [0x0ef7] = "\xb1\xa5", [0x0ef8] = "\xca\xce",
  [0x0ef9] = "\xb3\xdb", [0x0efa] = "\xcd\xd4", [0x0efb] = "\xd1\xb1",
  [0x0efc] = "\xf7\xd5", [0x0efd] = "\xf0\xaf", [0x0efe] = "\xf7\xe4",
  [0x0eff] = "\xb6\xa6", [0x0f00] = "\xb9\xc4", [0x0f01] = "\xca\xf3",
  [0x0f02] = "\xc9\xae", [0x0f03] = "\xd9\xd7", [0x0f04] = "\xbd\xc4",
  [0x0f05] = "\xd9\xd2", [0x0f06] = "\xd9\xd4", [0x0f07] = "\xc1\xc5",
  [0x0f08] = "\xc6\xcd", [0x0f09] = "\xcf\xf1", [0x0f0a] = "\xc7\xc8",
  [0x0f0b] = "\xb9\xcd", [0x0f0e] = "\xbe\xa4", [0x0f0f] = "\xb5\xca",
  [0x0f10] = "\xbb\xae", [0x0f11] = "\xd8\xe3", [0x0f12] = "\xd8\xd1",
  [0x0f13] = "\xd1\xe1", [0x0f14] = "\xe0\xd5", [0x0f15] = "\xe0\xd6",
  [0x0f16] = "\xc2\xef", [0x0f17] = "\xb3\xa2", [0x0f18] = "\xcb\xd4",
  [0x0f19] = "\xc5\xbb", [0x0f1a] = "\xcc\xbe", [0x0f1b] = "\xbc\xce",
  [0x0f1c] = "\xe0\xb6", [0x0f1d] = "\xb8\xc2", [0x0f1e] = "\xe0\xbb",
  [0x0f1f] = "\xdf\xf5", [0x0f20] = "\xe0\xbd", [0x0f21] = "\xe0\xd0",
  [0x0f23] = "\xdf\xd9", [0x0f24] = "\xcd\xc5", [0x0f25] = "\xcd\xbc",
  [0x0f26] = "\xb3\xbe", [0x0f27] = "\xdb\xd3", [0x0f28] = "\xbe\xb3",
  [0x0f29] = "\xc4\xb9", [0x0f2a] = "\xb5\xe6", [0x0f2b] = "\xc7\xb5",
  [0x0f2c] = "\xca\xfb", [0x0f2e] = "\xca\xd9", [0x0f2f] = "\xbb\xef",
  [0x0f30] = "\xc3\xce", [0x0f31] = "\xe2\xb9", [0x0f32] = "\xb6\xe1",
  [0x0f33] = "\xde\xc6", [0x0f34] = "\xb5\xd5", [0x0f35] = "\xe6\xcf",
  [0x0f36] = "\xc4\xdb", [0x0f37] = "\xe5\xfd", [0x0f38] = "\xe6\xce",
  [0x0f39] = "\xe6\xd0", [0x0f3a] = "\xe6\xcc", [0x0f3b] = "\xb7\xf5",
  [0x0f3c] = "\xc4\xaf", [0x0f3d] = "\xc4\xfe", [0x0f3e] = "\xb9\xd1",
  [0x0f3f] = "\xc1\xc8", [0x0f40] = "\xca\xb5", [0x0f41] = "\xd5\xaf",
  [0x0f42] = "\xc7\xde", [0x0f43] = "\xe5\xbb", [0x0f44] = "\xb2\xec",
  [0x0f45] = "\xb6\xd4", [0x0f46] = "\xc2\xc5", [0x0f47] = "\xd5\xb8",
  [0x0f48] = "\xe1\xab", [0x0f49] = "\xe1\xa4", [0x0f4a] = "\xb1\xd2",
  [0x0f4b] = "\xc4\xbb", [0x0f4c] = "\xe0\xfe", [0x0f4d] = "\xe1\xa3",
  [0x0f4e] = "\xc0\xaa", [0x0f4f] = "\xc1\xce", [0x0f50] = "\xb1\xd7",
  [0x0f51] = "\xb1\xf0", [0x0f52] = "\xd5\xc3", [0x0f53] = "\xb3\xb9",
  [0x0f54] = "\xd2\xf3", [0x0f55] = "\xd4\xb8", [0x0f56] = "\xcc\xac",
  [0x0f57] = "\xbf\xb6", [0x0f58] = "\xc2\xfd", [0x0f59] = "\xb9\xdf",
  [0x0f5a] = "\xe2\xfa", [0x0f5b] = "\xb2\xd1", [0x0f5c] = "\xb2\xd2",
  [0x0f5d] = "\xe3\xbc", [0x0f5e] = "\xbd\xd8", [0x0f5f] = "\xc6\xb2",
  [0x0f60] = "\xd5\xaa", [0x0f61] = "\xcb\xa4", [0x0f62] = "\xb3\xb7",
  [0x0f63] = "\xc3\xfe", [0x0f64] = "\xc2\xa7", [0x0f65] = "\xdf\xa1",
  [0x0f66] = "\xde\xe2", [0x0f67] = "\xb4\xdd", [0x0f68] = "\xe5\xba",
  [0x0f69] = "\xde\xfd", [0x0f6a] = "\xb2\xf4", [0x0f6b] = "\xc7\xc3",
  [0x0f6c] = "\xce\xd3", [0x0f6d] = "\xc6\xec", [0x0f6e] = "\xec\xbd",
  [0x0f6f] = "\xb3\xa9", [0x0f70] = "\xf4\xdf", [0x0f71] = "\xea\xd4",
  [0x0f72] = "\xb0\xf1", [0x0f73] = "\xd5\xa5", [0x0f74] = "\xe9\xc5",
  [0x0f75] = "\xe9\xc2", [0x0f76] = "\xc8\xd9", [0x0f77] = "\xb8\xdc",
  [0x0f78] = "\xb9\xb9", [0x0f79] = "\xe9\xbb", [0x0f7a] = "\xc8\xb6",
  [0x0f7b] = "\xe9\xbd", [0x0f7c] = "\xe9\xbe", [0x0f7d] = "\xc1\xf1",
  [0x0f7e] = "\xbb\xb1", [0x0f7f] = "\xc7\xb9", [0x0f80] = "\xe9\xbf",
  [0x0f81] = "\xe9\xb3", [0x0f82] = "\xb8\xc9", [0x0f83] = "\xc5\xcc",
  [0x0f85] = "\xc7\xb8", [0x0f86] = "\xb8\xe8", [0x0f87] = "\xeb\xb5",
  [0x0f88] = "\xd5\xc4", [0x0f89] = "\xd1\xdd", [0x0f8a] = "\xb9\xf6",
  [0x0f8b] = "\xc0\xec", [0x0f8c] = "\xb5\xce", [0x0f8d] = "\xe4\xf6",
  [0x0f8e] = "\xd1\xfa", [0x0f8f] = "\xc4\xae", [0x0f90] = "\xd7\xd5",
  [0x0f91] = "\xc2\xa9", [0x0f92] = "\xc6\xaf", [0x0f93] = "\xba\xba",
  [0x0f94] = "\xc2\xfa", [0x0f95] = "\xd6\xcd", [0x0f96] = "\xc6\xe1",
  [0x0f97] = "\xca\xfe", [0x0f98] = "\xbd\xa5", [0x0f99] = "\xd5\xc7",
  [0x0f9a] = "\xc1\xb0", [0x0f9b] = "\xe4\xee", [0x0f9c] = "\xc2\xfe",
  [0x0f9d] = "\xe4\xf0", [0x0f9e] = "\xb3\xba", [0x0f9f] = "\xe4\xf4",
  [0x0fa0] = "\xbb\xa6", [0x0fa1] = "\xd3\xe6", [0x0fa2] = "\xc9\xf8",
  [0x0fa3] = "\xb5\xd3", [0x0fa4] = "\xc2\xb1", [0x0fa5] = "\xc8\xdb",
  [0x0fa6] = "\xce\xf5", [0x0fa7] = "\xc9\xbf", [0x0fa8] = "\xd0\xdc",
  [0x0fa9] = "\xcf\xa8", [0x0faa] = "\xd3\xab", [0x0fab] = "\xb6\xfb",
  [0x0fac] = "\xea\xfb", [0x0fad] = "\xdc\xfd", [0x0fae] = "\xd3\xfc",
  [0x0faf] = "\xe2\xaf", [0x0fb0] = "\xd1\xfe", [0x0fb1] = "\xcb\xf6",
  [0x0fb2] = "\xc2\xea", [0x0fb3] = "\xb9\xe5", [0x0fb4] = "\xe8\xa9",
  [0x0fb5] = "\xd5\xe7", [0x0fb6] = "\xd2\xc9", [0x0fb7] = "\xc5\xb1",
  [0x0fb8] = "\xd1\xf1", [0x0fb9] = "\xb7\xe8", [0x0fba] = "\xd3\xfa",
  [0x0fbb] = "\xbb\xbe", [0x0fbc] = "\xbe\xa1", [0x0fbd] = "\xbc\xe0",
  [0x0fbe] = "\xc3\xe9", [0x0fbf] = "\xee\xa5", [0x0fc0] = "\xee\xa3",
  [0x0fc1] = "\xcb\xaf", [0x0fc2] = "\xb4\xc5", [0x0fc3] = "\xb5\xfa",
  [0x0fc4] = "\xb1\xcc", [0x0fc5] = "\xcc\xbc", [0x0fc6] = "\xcb\xb6",
  [0x0fc7] = "\xed\xd9", [0x0fc8] = "\xec\xf5", [0x0fc9] = "\xb8\xa3",
  [0x0fca] = "\xbb\xf6", [0x0fcb] = "\xd6\xd6", [0x0fcc] = "\xb3\xc6",
  [0x0fcd] = "\xcd\xdd", [0x0fce] = "\xce\xd1", [0x0fcf] = "\xbd\xdf",
  [0x0fd0] = "\xb6\xcb", [0x0fd1] = "\xb9\xdc", [0x0fd2] = "\xbb\xfe",
  [0x0fd3] = "\xbc\xe3", [0x0fd4] = "\xf3\xdb", [0x0fd5] = "\xcb\xe3",
  [0x0fd6] = "\xf3\xe9", [0x0fd7] = "\xb2\xad", [0x0fd8] = "\xf3\xdd",
  [0x0fd9] = "\xf3\xe7", [0x0fda] = "\xb8\xf6", [0x0fdb] = "\xf3\xeb",
  [0x0fdc] = "\xb4\xe2", [0x0fdd] = "\xf4\xd5", [0x0fde] = "\xbe\xab",
  [0x0fdf] = "\xd5\xc0", [0x0fe0] = "\xe7\xba", [0x0fe1] = "\xd7\xdb",
  [0x0fe2] = "\xb4\xc2", [0x0fe3] = "\xe7\xb1", [0x0fe4] = "\xc2\xcc",
  [0x0fe5] = "\xbd\xf4", [0x0fe6] = "\xd7\xba", [0x0fe7] = "\xcd\xf8",
  [0x0fe8] = "\xb8\xd9", [0x0fe9] = "\xe7\xb2", [0x0fea] = "\xb3\xf1",
  [0x0feb] = "\xc3\xe0", [0x0fec] = "\xb2\xca", [0x0fed] = "\xc2\xda",
  [0x0fee] = "\xce\xac", [0x0fef] = "\xd0\xf7", [0x0ff0] = "\xe7\xbb",
  [0x0ff1] = "\xe7\xb7", [0x0ff2] = "\xb7\xa3", [0x0ff3] = "\xb4\xe4",
  [0x0ff4] = "\xf4\xe4", [0x0ff5] = "\xb5\xd4", [0x0ff6] = "\xce\xc5",
  [0x0ff7] = "\xbe\xdb", [0x0ff8] = "\xd5\xd8", [0x0ff9] = "\xb8\xaf",
  [0x0ffa] = "\xb0\xf2", [0x0ffb] = "\xb8\xe0", [0x0ffc] = "\xeb\xf5",
  [0x0ffd] = "\xb2\xb2", [0x0ffe] = "\xcd\xc8", [0x0fff] = "\xeb\xf6",
  [0x1000] = "\xea\xb0", [0x1001] = "\xcc\xa8", [0x1002] = "\xd3\xeb",
  [0x1003] = "\xcc\xf2", [0x1004] = "\xce\xe8", [0x1005] = "\xf4\xbb",
  [0x1006] = "\xc8\xd8", [0x1007] = "\xdd\xef", [0x1008] = "\xcf\xaf",
  [0x1009] = "\xd0\xee", [0x100a] = "\xc3\xc9", [0x100b] = "\xdd\xb0",
  [0x100c] = "\xc6\xd1", [0x100d] = "\xcb\xe2", [0x100e] = "\xb8\xc7",
  [0x100f] = "\xd5\xf4", [0x1010] = "\xdd\xa5", [0x1011] = "\xdd\xed",
  [0x1012] = "\xcb\xd1", [0x1013] = "\xb2\xd4", [0x1014] = "\xcb\xf2",
  [0x1015] = "\xdd\xee", [0x1016] = "\xf2\xea", [0x1017] = "\xc3\xdb",
  [0x1018] = "\xf2\xdf", [0x1019] = "\xf2\xec", [0x101a] = "\xf2\xe1",
  [0x101b] = "\xf2\xe6", [0x101c] = "\xd6\xa9", [0x101d] = "\xca\xb4",
  [0x101e] = "\xf2\xe9", [0x101f] = "\xf2\xe8", [0x1020] = "\xc9\xd1",
  [0x1021] = "\xb9\xd3", [0x1022] = "\xc5\xe1", [0x1023] = "\xb9\xfc",
  [0x1024] = "\xc2\xe3", [0x1025] = "\xd6\xc6", [0x1026] = "\xf1\xd4",
  [0x1027] = "\xf1\xd2", [0x1029] = "\xcb\xd0", [0x102a] = "\xd6\xbe",
  [0x102b] = "\xd3\xef", [0x102c] = "\xce\xdc", [0x102d] = "\xc8\xcf",
  [0x102e] = "\xbd\xeb", [0x102f] = "\xca\xc4", [0x1030] = "\xce\xf3",
  [0x1031] = "\xcb\xb5", [0x1032] = "\xda\xbe", [0x1033] = "\xbb\xe5",
  [0x1034] = "\xd3\xd5", [0x1035] = "\xda\xbf", [0x1036] = "\xda\xbd",
  [0x1038] = "\xba\xc0", [0x1039] = "\xc0\xea", [0x103a] = "\xc3\xb2",
  [0x103b] = "\xb1\xf6", [0x103c] = "\xea\xe2", [0x103d] = "\xc9\xde",
  [0x103e] = "\xba\xd5", [0x103f] = "\xd5\xd4", [0x1040] = "\xb8\xcf",
  [0x1041] = "\xbe\xd6", [0x1042] = "\xb8\xa8", [0x1043] = "\xe9\xfc",
  [0x1044] = "\xc7\xe1", [0x1045] = "\xcd\xec", [0x1046] = "\xc0\xb1",
  [0x1047] = "\xd4\xb6", [0x1048] = "\xe5\xdc", [0x1049] = "\xd1\xb7",
  [0x104a] = "\xc7\xb2", [0x104b] = "\xd2\xa3", [0x104c] = "\xb5\xdd",
  [0x104d] = "\xe5\xdd", [0x104f] = "\xe5\xde", [0x1050] = "\xb1\xc9",
  [0x1052] = "\xdb\xb4", [0x1053] = "\xbd\xcd", [0x1054] = "\xcb\xe1",
  [0x1055] = "\xbf\xe1", [0x1056] = "\xf5\xa9", [0x1057] = "\xbd\xc2",
  [0x1058] = "\xd2\xf8", [0x1059] = "\xcd\xad", [0x105a] = "\xc3\xfa",
  [0x105b] = "\xee\xf9", [0x105c] = "\xb8\xf5", [0x105d] = "\xee\xfd",
  [0x105e] = "\xcf\xce", [0x105f] = "\xef\xa7", [0x1061] = "\xcf\xb3",
  [0x1062] = "\xba\xd2", [0x1063] = "\xb9\xeb", [0x1064] = "\xc3\xf6",
  [0x1065] = "\xb8\xf3", [0x1066] = "\xb7\xa7", [0x1067] = "\xba\xcf",
  [0x1068] = "\xcf\xb6", [0x1069] = "\xd5\xcf", [0x106a] = "\xbc\xca",
  [0x106b] = "\xb4\xc6", [0x106c] = "\xf6\xc3", [0x106d] = "\xd0\xe8",
  [0x106e] = "\xf7\xb0", [0x106f] = "\xf7\xb1", [0x1070] = "\xc9\xd8",
  [0x1071] = "\xc6\xc4", [0x1072] = "\xc1\xec", [0x1073] = "\xec\xaa",
  [0x1074] = "\xcc\xa8", [0x1075] = "\xbd\xc8", [0x1076] = "\xb1\xfd",
  [0x1077] = "\xb6\xfc", [0x1078] = "\xe2\xc3", [0x1079] = "\xb2\xb5",
  [0x107a] = "\xb0\xb9", [0x107b] = "\xf7\xbb", [0x107c] = "\xf7\xd6",
  [0x107d] = "\xbf\xfd", [0x107e] = "\xbb\xea", [0x107f] = "\xc3\xf9",
  [0x1080] = "\xf0\xb0", [0x1081] = "\xb7\xef", [0x1082] = "\xc3\xb4",
  [0x1083] = "\xb1\xc7", [0x1084] = "\xc6\xeb", [0x1085] = "\xd2\xda",
  [0x1086] = "\xd2\xc7", [0x1087] = "\xc6\xa7", [0x1088] = "\xbd\xa9",
  [0x1089] = "\xbc\xdb", [0x108a] = "\xd9\xaf", [0x108b] = "\xbf\xeb",
  [0x108c] = "\xbc\xf3", [0x108d] = "\xb5\xb1", [0x108e] = "\xc1\xdd",
  [0x108f] = "\xbe\xe7", [0x1090] = "\xc5\xfc", [0x1091] = "\xc1\xf5",
  [0x1092] = "\xbd\xa3", [0x1093] = "\xb9\xf4", [0x1094] = "\xdb\xc4",
  [0x1095] = "\xc0\xf7", [0x1096] = "\xdf\xeb", [0x1097] = "\xce\xfb",
  [0x1098] = "\xe0\xda", [0x1099] = "\xb3\xb0", [0x109a] = "\xba\xd9",
  [0x109b] = "\xd7\xec", [0x109c] = "\xbb\xa9", [0x109d] = "\xd0\xea",
  [0x109e] = "\xd2\xad", [0x109f] = "\xe0\xdb", [0x10a0] = "\xc5\xe7",
  [0x10a1] = "\xcb\xbb", [0x10a2] = "\xd0\xa5", [0x10a3] = "\xdf\xb4",
  [0x10a4] = "\xdc\xaf", [0x10a5] = "\xd0\xe6", [0x10a6] = "\xd4\xf6",
  [0x10a7] = "\xb7\xd8", [0x10a8] = "\xd7\xb9", [0x10a9] = "\xb6\xe9",
  [0x10aa] = "\xb6\xd5", [0x10ad] = "\xe6\xd2", [0x10ae] = "\xe6\xb5",
  [0x10af] = "\xe6\xbf", [0x10b0] = "\xe5\xfc", [0x10b1] = "\xbd\xbf",
  [0x10b2] = "\xe6\xac", [0x10b3] = "\xe5\xbc", [0x10b4] = "\xbf\xed",
  [0x10b5] = "\xc9\xf3", [0x10b6] = "\xd0\xb4", [0x10b7] = "\xb2\xe3",
  [0x10b8] = "\xc2\xc4", [0x10b9] = "\xe1\xd8", [0x10bb] = "\xb4\xb1",
  [0x10bc] = "\xd6\xc4", [0x10bd] = "\xe1\xa6", [0x10be] = "\xb7\xcf",
  [0x10bf] = "\xb3\xf8", [0x10c0] = "\xc3\xed", [0x10c1] = "\xd8\xcb",
  [0x10c2] = "\xb9\xe3", [0x10c3] = "\xb3\xa7", [0x10c4] = "\xb5\xaf",
  [0x10c5] = "\xd3\xb0", [0x10c6] = "\xb5\xc2", [0x10c7] = "\xe1\xe7",
  [0x10c8] = "\xc7\xec", [0x10c9] = "\xbb\xdb", [0x10ca] = "\xc2\xc7",
  [0x10cb] = "\xed\xab", [0x10cc] = "\xc4\xbd", [0x10cd] = "\xd3\xc7",
  [0x10ce] = "\xc6\xdd", [0x10cf] = "\xce\xbf", [0x10d0] = "\xcb\xcb",
  [0x10d1] = "\xd3\xfb", [0x10d2] = "\xe3\xbf", [0x10d3] = "\xc1\xaf",
  [0x10d4] = "\xc3\xf5", [0x10d5] = "\xd4\xf7", [0x10d6] = "\xe3\xbd",
  [0x10d7] = "\xb5\xac", [0x10d8] = "\xb7\xdf", [0x10d9] = "\xe3\xbe",
  [0x10da] = "\xe2\xe4", [0x10db] = "\xc2\xbe", [0x10dc] = "\xc4\xa6",
  [0x10dd] = "\xd6\xbf", [0x10de] = "\xc4\xa1", [0x10df] = "\xd7\xb2",
  [0x10e0] = "\xc6\xcb", [0x10e1] = "\xc0\xcc", [0x10e2] = "\xb3\xc5",
  [0x10e3] = "\xd7\xab", [0x10e4] = "\xb2\xa6", [0x10e5] = "\xc4\xd3",
  [0x10e6] = "\xcb\xba", [0x10e7] = "\xc1\xc3", [0x10e8] = "\xc8\xf6",
  [0x10e9] = "\xb4\xe9", [0x10ea] = "\xb2\xa5", [0x10eb] = "\xb8\xa7",
  [0x10ec] = "\xc4\xed", [0x10ed] = "\xc7\xcb", [0x10ee] = "\xdf\xa4",
  [0x10ef] = "\xb5\xa7", [0x10f0] = "\xde\xec", [0x10f1] = "\xb5\xd0",
  [0x10f2] = "\xb7\xf3", [0x10f3] = "\xca\xfd", [0x10f4] = "\xc4\xba",
  [0x10f5] = "\xd4\xdd", [0x10f6] = "\xb1\xa9", [0x10f7] = "\xea\xc7",
  [0x10f8] = "\xd1\xf9", [0x10f9] = "\xd5\xc1", [0x10fa] = "\xe9\xa4",
  [0x10fb] = "\xd7\xae", [0x10fc] = "\xca\xe0", [0x10fd] = "\xb1\xea",
  [0x10fe] = "\xb2\xdb", [0x10ff] = "\xc4\xa3", [0x1100] = "\xc2\xa5",
  [0x1101] = "\xb7\xae", [0x1102] = "\xbd\xb0", [0x1103] = "\xc0\xd6",
  [0x1104] = "\xe8\xc8", [0x1105] = "\xe9\xca", [0x1106] = "\xc1\xba",
  [0x1107] = "\xc5\xb7", [0x1108] = "\xcc\xbe", [0x1109] = "\xe9\xe4",
  [0x110a] = "\xd2\xe3", [0x110b] = "\xc5\xb9", [0x110c] = "\xbd\xac",
  [0x110d] = "\xe4\xfc", [0x110e] = "\xb3\xce", [0x110f] = "\xc6\xc3",
  [0x1110] = "\xc1\xca", [0x1111] = "\xbd\xe0", [0x1112] = "\xbd\xbd",
  [0x1113] = "\xcc\xb6", [0x1114] = "\xc7\xb1", [0x1115] = "\xe4\xfa",
  [0x1116] = "\xb3\xb1", [0x1117] = "\xc5\xec", [0x1118] = "\xe4\xfd",
  [0x1119] = "\xc0\xa3", [0x111a] = "\xc8\xf3", [0x111b] = "\xbd\xa7",
  [0x111c] = "\xc5\xcb", [0x111d] = "\xeb\xf8", [0x111e] = "\xe4\xb1",
  [0x1120] = "\xf4\xaa", [0x1121] = "\xca\xec", [0x1122] = "\xb0\xbe",
  [0x1123] = "\xc8\xc8", [0x1124] = "\xec\xd9", [0x1125] = "\xeb\xbb",
  [0x1126] = "\xea\xf3", [0x1127] = "\xbd\xb1", [0x1128] = "\xe2\xb1",
  [0x1129] = "\xd3\xa8", [0x112a] = "\xe8\xb0", [0x112b] = "\xc1\xa7",
  [0x112c] = "\xe8\xaa", [0x112d] = "\xe8\xad", [0x112e] = "\xe7\xdc",
  [0x112f] = "\xf1\xa4", [0x1130] = "\xb4\xf1", [0x1131] = "\xce\xc1",
  [0x1132] = "\xc1\xf6", [0x1133] = "\xca\xdd", [0x1134] = "\xb4\xaf",
  [0x1135] = "\xf1\xa3", [0x1136] = "\xb0\xa8", [0x1137] = "\xd6\xe5",
  [0x1138] = "\xc5\xcc", [0x1139] = "\xcf\xb9", [0x113a] = "\xc3\xd0",
  [0x113b] = "\xee\xa7", [0x113c] = "\xee\xa8", [0x113e] = "\xb4\xe8",
  [0x113f] = "\xb0\xf5", [0x1140] = "\xc8\xb7", [0x1141] = "\xc0\xda",
  [0x1142] = "\xc4\xeb", [0x1143] = "\xbf\xc4", [0x1144] = "\xc2\xeb",
  [0x1145] = "\xc5\xcd", [0x1146] = "\xb8\xe5", [0x1147] = "\xbc\xda",
  [0x1148] = "\xb9\xc8", [0x1149] = "\xbb\xfc", [0x114a] = "\xf0\xa2",
  [0x114b] = "\xb5\xbe", [0x114c] = "\xd2\xa4", [0x114d] = "\xc7\xee",
  [0x114e] = "\xbc\xfd", [0x114f] = "\xcf\xe4", [0x1150] = "\xb7\xb6",
  [0x1151] = "\xf3\xf0", [0x1152] = "\xd7\xad", [0x1153] = "\xc6\xaa",
  [0x1154] = "\xf3\xf2", [0x1155] = "\xe9\xa2", [0x1156] = "\xf3\xf3",
  [0x1157] = "\xba\xfd", [0x1158] = "\xb5\xde", [0x1159] = "\xc1\xb7",
  [0x115a] = "\xce\xb3", [0x115b] = "\xd6\xc2", [0x115c] = "\xbc\xea",
  [0x115d] = "\xc3\xe5", [0x115e] = "\xbc\xa9", [0x115f] = "\xb1\xe0",
  [0x1160] = "\xd4\xb5", [0x1161] = "\xcf\xdf", [0x1162] = "\xb6\xd0",
  [0x1163] = "\xbb\xba", [0x1164] = "\xe7\xb6", [0x1165] = "\xe7\xbc",
  [0x1166] = "\xe7\xbf", [0x1167] = "\xe7\xbe", [0x1168] = "\xc2\xee",
  [0x1169] = "\xb0\xd5", [0x116a] = "\xf4\xc9", [0x116b] = "\xf4\xe6",
  [0x116c] = "\xf1\xee", [0x116d] = "\xcc\xc5", [0x116e] = "\xc4\xa4",
  [0x116f] = "\xcf\xa5", [0x1170] = "\xbd\xba", [0x1171] = "\xb7\xf4",
  [0x1172] = "\xb1\xec", [0x1173] = "\xd5\xe1", [0x1174] = "\xb1\xce",
  [0x1175] = "\xce\xb5", [0x1176] = "\xc1\xab", [0x1177] = "\xca\xdf",
  [0x1178] = "\xd2\xf1", [0x1179] = "\xc2\xfb", [0x117a] = "\xc3\xef",
  [0x117b] = "\xbd\xaf", [0x117c] = "\xb2\xcc", [0x117d] = "\xb2\xb7",
  [0x117e] = "\xc5\xee", [0x117f] = "\xb4\xd0", [0x1180] = "\xde\xa3",
  [0x1181] = "\xc1\xe2", [0x1182] = "\xf2\xeb", [0x1183] = "\xba\xfb",
  [0x1184] = "\xb5\xfb", [0x1185] = "\xf2\xf0", [0x1186] = "\xcf\xba",
  [0x1187] = "\xce\xcf", [0x1188] = "\xca\xad", [0x1189] = "\xf2\xf9",
  [0x118a] = "\xbb\xc8", [0x118b] = "\xf2\xf2", [0x118c] = "\xf2\xf5",
  [0x118d] = "\xce\xc0", [0x118e] = "\xb3\xe5", [0x118f] = "\xba\xd6",
  [0x1190] = "\xb8\xb4", [0x1191] = "\xb0\xfd", [0x1192] = "\xf1\xd9",
  [0x1194] = "\xf1\xdb", [0x1195] = "\xd2\xea", [0x1196] = "\xc1\xc2",
  [0x1197] = "\xcc\xb8", [0x1198] = "\xd7\xbb", [0x1199] = "\xb5\xae",
  [0x119a] = "\xc7\xeb", [0x119b] = "\xd6\xee", [0x119c] = "\xbf\xce",
  [0x119d] = "\xda\xc3", [0x119e] = "\xda\xc6", [0x119f] = "\xb5\xf7",
  [0x11a0] = "\xcb\xad", [0x11a1] = "\xc2\xdb", [0x11a2] = "\xda\xba",
  [0x11a3] = "\xda\xc7", [0x11a4] = "\xb7\xcc", [0x11a5] = "\xda\xc4",
  [0x11a6] = "\xcd\xe3", [0x11a7] = "\xca\xfa", [0x11a8] = "\xd6\xed",
  [0x11a9] = "\xc5\xe2", [0x11aa] = "\xc9\xcd", [0x11ab] = "\xb8\xb3",
  [0x11ac] = "\xbc\xfa", [0x11ad] = "\xd5\xcb", [0x11ae] = "\xb6\xc4",
  [0x11af] = "\xcf\xcd", [0x11b0] = "\xc2\xf4", [0x11b1] = "\xb4\xcd",
  [0x11b2] = "\xd6\xca", [0x11b3] = "\xe2\xd9", [0x11b4] = "\xf4\xf7",
  [0x11b5] = "\xcc\xcb", [0x11b6] = "\xc8\xa4", [0x11b7] = "\xc5\xf6",
  [0x11b8] = "\xbc\xf9", [0x11b9] = "\xf5\xd7", [0x11ba] = "\xcc\xdf",
  [0x11bb] = "\xcc\xa4", [0x11bc] = "\xb2\xc8", [0x11bd] = "\xf5\xd8",
  [0x11be] = "\xf2\xe9", [0x11bf] = "\xbe\xe1", [0x11c0] = "\xcc\xc9",
  [0x11c1] = "\xbb\xd4", [0x11c2] = "\xc1\xbe", [0x11c3] = "\xea\xa1",
  [0x11c4] = "\xb1\xb2", [0x11c5] = "\xe9\xfd", [0x11c6] = "\xc2\xd6",
  [0x11c7] = "\xea\xa2", [0x11c8] = "\xe9\xfe", [0x11c9] = "\xb9\xf5",
  [0x11ca] = "\xca\xca", [0x11cb] = "\xd5\xda", [0x11cc] = "\xe5\xdb",
  [0x11cd] = "\xd4\xe2", [0x11ce] = "\xc7\xa8", [0x11cf] = "\xc1\xda",
  [0x11d0] = "\xd6\xa3", [0x11d1] = "\xb5\xcb", [0x11d2] = "\xdb\xb6",
  [0x11d3] = "\xb4\xbc", [0x11d4] = "\xd7\xed", [0x11d5] = "\xb4\xd7",
  [0x11d6] = "\xeb\xe7", [0x11d7] = "\xd0\xbf", [0x11d8] = "\xcc\xe0",
  [0x11d9] = "\xcf\xfa", [0x11da] = "\xc6\xcc", [0x11db] = "\xee\xed",
  [0x11dc] = "\xb3\xfa", [0x11dd] = "\xc2\xc1", [0x11de] = "\xc8\xf1",
  [0x11df] = "\xef\xb1", [0x11e0] = "\xb7\xe6", [0x11e1] = "\xb1\xb5",
  [0x11e2] = "\xef\xae", [0x11e3] = "\xba\xb8", [0x11e4] = "\xe3\xcc",
  [0x11e5] = "\xd4\xc4", [0x11e6] = "\xcf\xf6", [0x11e7] = "\xf6\xaa",
  [0x11e8] = "\xd5\xf0", [0x11e9] = "\xc3\xb9", [0x11ea] = "\xbf\xbf",
  [0x11eb] = "\xb0\xb0", [0x11ec] = "\xd0\xac", [0x11ed] = "\xb9\xae",
  [0x11ee] = "\xf2\xa1", [0x11ef] = "\xb8\xa9", [0x11f0] = "\xf2\xa2",
  [0x11f1] = "\xb9\xce", [0x11f2] = "\xd1\xf8", [0x11f3] = "\xb6\xf6",
  [0x11f4] = "\xc4\xd9", [0x11f5] = "\xd3\xe0", [0x11f6] = "\xcd\xd5",
  [0x11f7] = "\xd7\xa4", [0x11f8] = "\xe6\xe1", [0x11f9] = "\xca\xbb",
  [0x11fa] = "\xe6\xe5", [0x11fb] = "\xbc\xdd", [0x11fc] = "\xbe\xd4",
  [0x11fd] = "\xe6\xe2", [0x11fe] = "\xf7\xbc", [0x11ff] = "\xb7\xa2",
  [0x1200] = "\xf7\xd7", [0x1201] = "\xc4\xd6", [0x1202] = "\xf7\xc8",
  [0x1203] = "\xc6\xc7", [0x1204] = "\xf6\xcf", [0x1205] = "\xc2\xb3",
  [0x1206] = "\xf0\xb2", [0x1207] = "\xd1\xbb", [0x1209] = "\xf4\xef",
  [0x120a] = "\xf7\xe2", [0x120b] = "\xc0\xe8", [0x120c] = "\xc4\xab",
  [0x120d] = "\xb3\xdd", [0x120e] = "\xc8\xe5", [0x120f] = "\xbe\xa1",
  [0x1210] = "\xd9\xb1", [0x1211] = "\xd9\xcf", [0x1212] = "\xd9\xad",
  [0x1213] = "\xbc\xbd", [0x1214] = "\xc3\xdd", [0x1215] = "\xc4\xfd",
  [0x1216] = "\xbc\xc1", [0x1217] = "\xd8\xe6", [0x1218] = "\xd1\xab",
  [0x1219] = "\xe0\xdf", [0x121a] = "\xe0\xe6", [0x121b] = "\xb5\xb1",
  [0x121c] = "\xd8\xac", [0x121d] = "\xe0\xe4", [0x121e] = "\xb6\xd6",
  [0x121f] = "\xd4\xeb", [0x1220] = "\xc6\xf7", [0x1221] = "\xdf\xe6",
  [0x1222] = "\xe0\xe5", [0x1223] = "\xe0\xc8", [0x1224] = "\xca\xc9",
  [0x1225] = "\xe0\xde", [0x1226] = "\xb8\xc1", [0x1227] = "\xb1\xda",
  [0x1228] = "\xbf\xd1", [0x1229] = "\xcc\xb3", [0x122a] = "\xdb\xd5",
  [0x122b] = "\xb7\xdc", [0x122c] = "\xf4\xc1", [0x122d] = "\xd9\xf8",
  [0x122e] = "\xd1\xa7", [0x122f] = "\xe5\xbe", [0x1230] = "\xb5\xbc",
  [0x1231] = "\xc7\xbf", [0x1232] = "\xcf\xdc", [0x1233] = "\xc6\xbe",
  [0x1234] = "\xed\xac", [0x1235] = "\xb1\xb9", [0x1236] = "\xe3\xc1",
  [0x1237] = "\xd2\xe4", [0x1238] = "\xba\xb6", [0x1239] = "\xb0\xc3",
  [0x123a] = "\xd0\xb8", [0x123b] = "\xd5\xbd", [0x123c] = "\xc9\xc3",
  [0x123d] = "\xd3\xb5", [0x123e] = "\xb5\xb2", [0x123f] = "\xcc\xa2",
  [0x1240] = "\xba\xb3", [0x1241] = "\xbe\xdd", [0x1242] = "\xc2\xb0",
  [0x1243] = "\xd4\xf1", [0x1244] = "\xc0\xde", [0x1245] = "\xb2\xd9",
  [0x1246] = "\xbc\xf1", [0x1247] = "\xc7\xdc", [0x1248] = "\xb5\xa3",
  [0x1249] = "\xce\xce", [0x124a] = "\xd5\xfb", [0x124b] = "\xc0\xfa",
  [0x124c] = "\xcf\xfe", [0x124d] = "\xe5\xdf", [0x124e] = "\xea\xca",
  [0x124f] = "\xea\xbc", [0x1250] = "\xc1\xcb", [0x1251] = "\xe9\xd7",
  [0x1252] = "\xc6\xd3", [0x1253] = "\xe8\xeb", [0x1254] = "\xb3\xc8",
  [0x1255] = "\xba\xe1", [0x1256] = "\xe9\xd9", [0x1257] = "\xca\xf7",
  [0x1258] = "\xe9\xcf", [0x1259] = "\xcd\xd6", [0x125a] = "\xcf\xf0",
  [0x125b] = "\xc7\xc5", [0x125c] = "\xc7\xc1", [0x125d] = "\xe9\xd4",
  [0x125e] = "\xbb\xfa", [0x125f] = "\xe8\xe3", [0x1260] = "\xec\xa8",
  [0x1261] = "\xc0\xfa", [0x1262] = "\xeb\xa9", [0x1263] = "\xe5\xa5",
  [0x1264] = "\xb5\xed", [0x1265] = "\xd4\xe8", [0x1266] = "\xc5\xa8",
  [0x1267] = "\xd4\xf3", [0x1268] = "\xd7\xc7", [0x1269] = "\xe5\xa2",
  [0x126a] = "\xb0\xc4", [0x126b] = "\xbc\xa4", [0x126c] = "\xe5\xa3",
  [0x126d] = "\xe5\xa4", [0x126f] = "\xe4\xc5", [0x1271] = "\xb3\xe3",
  [0x1272] = "\xec\xc0", [0x1273] = "\xc1\xd7", [0x1274] = "\xc9\xd5",
  [0x1275] = "\xb5\xc6", [0x1276] = "\xd1\xe0", [0x1277] = "\xec\xe4",
  [0x1278] = "\xc1\xc7", [0x1279] = "\xcc\xcc", [0x127a] = "\xec\xcb",
  [0x127b] = "\xc8\xbc", [0x127c] = "\xd1\xe6", [0x127d] = "\xb6\xc0",
  [0x127e] = "\xe8\xab", [0x127f] = "\xe7\xe1", [0x1282] = "\xe8\xb1",
  [0x1283] = "\xc6\xb0", [0x1284] = "\xea\xb1", [0x1285] = "\xdd\xf9",
  [0x1286] = "\xd5\xce", [0x1287] = "\xc8\xb3", [0x1288] = "\xf0\xfc",
  [0x1289] = "\xc2\xac", [0x128a] = "\xee\xc2", [0x128b] = "\xee\xaa",
  [0x128c] = "\xc2\xf7", [0x128d] = "\xee\xa9", [0x128e] = "\xc6\xb3",
  [0x128f] = "\xc4\xa5", [0x1290] = "\xd7\xa9", [0x1291] = "\xed\xe0",
  [0x1292] = "\xed\xd3", [0x1293] = "\xd3\xf9", [0x1294] = "\xbb\xfd",
  [0x1295] = "\xd3\xb1", [0x1296] = "\xc4\xc2", [0x1297] = "\xf6\xd5",
  [0x1299] = "\xbf\xfa", [0x129a] = "\xb8\xdd", [0x129b] = "\xcb\xf2",
  [0x129c] = "\xd6\xfe", [0x129d] = "\xf3\xc6", [0x129e] = "\xf3\xe8",
  [0x129f] = "\xb4\xdb", [0x12a0] = "\xc9\xb8", [0x12a1] = "\xf3\xf7",
  [0x12a2] = "\xb8\xe2", [0x12a3] = "\xcc\xc7", [0x12a4] = "\xe7\xcb",
  [0x12a5] = "\xe7\xcc", [0x12a6] = "\xdd\xd3", [0x12a7] = "\xb8\xbf",
  [0x12a8] = "\xcf\xd8", [0x12a9] = "\xe7\xc9", [0x12aa] = "\xe7\xc7",
  [0x12ab] = "\xe7\xc6", [0x12ac] = "\xe7\xa7", [0x12ad] = "\xee\xbe",
  [0x12ae] = "\xf4\xcb", [0x12af] = "\xba\xb2", [0x12b0] = "\xb0\xbf",
  [0x12b1] = "\xf4\xe7", [0x12b2] = "\xf1\xf1", [0x12b3] = "\xc9\xc5",
  [0x12b4] = "\xc4\xe5", [0x12b5] = "\xc5\xf2", [0x12b6] = "\xd5\xe9",
  [0x12b7] = "\xd0\xcb", [0x12b8] = "\xcb\xd2", [0x12b9] = "\xb2\xd5",
  [0x12ba] = "\xc8\xef", [0x12bb] = "\xde\xa5", [0x12bc] = "\xde\xa6",
  [0x12bd] = "\xde\xa7", [0x12be] = "\xb5\xb4", [0x12bf] = "\xde\xac",
  [0x12c0] = "\xbd\xb6", [0x12c1] = "\xcf\xf4", [0x12c2] = "\xce\xdf",
  [0x12c3] = "\xde\xa9", [0x12c4] = "\xf3\xa6", [0x12c5] = "\xc3\xf8",
  [0x12c6] = "\xc2\xec", [0x12c7] = "\xd3\xa9", [0x12c8] = "\xc8\xda",
  [0x12c9] = "\xba\xe2", [0x12ca] = "\xcd\xca", [0x12cb] = "\xbf\xe3",
  [0x12cc] = "\xc8\xec", [0x12cd] = "\xf1\xdd", [0x12ce] = "\xf1\xd7",
  [0x12cf] = "\xc7\xd7", [0x12d0] = "\xea\xec", [0x12d1] = "\xda\xd0",
  [0x12d2] = "\xd1\xe8", [0x12d3] = "\xda\xc9", [0x12d4] = "\xbb\xe4",
  [0x12d5] = "\xc4\xb1", [0x12d6] = "\xb5\xfd", [0x12d7] = "\xd0\xb3",
  [0x12d8] = "\xda\xd1", [0x12d9] = "\xc5\xb5", [0x12da] = "\xda\xcb",
  [0x12db] = "\xce\xbd", [0x12dc] = "\xb7\xed", [0x12dd] = "\xda\xcd",
  [0x12de] = "\xda\xcf", [0x12df] = "\xda\xc8", [0x12e0] = "\xda\xce",
  [0x12e1] = "\xd4\xa5", [0x12e3] = "\xc3\xa8", [0x12e4] = "\xc0\xb5",
  [0x12e5] = "\xcc\xe3", [0x12e6] = "\xf5\xe2", [0x12e7] = "\xd3\xbb",
  [0x12e8] = "\xf5\xe5", [0x12e9] = "\xf5\xdf", [0x12ea] = "\xf5\xe0",
  [0x12eb] = "\xb7\xf8", [0x12ec] = "\xbc\xad", [0x12ed] = "\xca\xe4",
  [0x12ee] = "\xea\xa3", [0x12ef] = "\xb1\xe6", [0x12f0] = "\xb0\xec",
  [0x12f1] = "\xd7\xf1", [0x12f2] = "\xe5\xe0", [0x12f3] = "\xd1\xa1",
  [0x12f4] = "\xb3\xd9", [0x12f5] = "\xc1\xc9", [0x12f6] = "\xd2\xc5",
  [0x12f7] = "\xda\xfe", [0x12f8] = "\xd0\xd1", [0x12f9] = "\xb6\xa7",
  [0x12fa] = "\xb1\xed", [0x12fb] = "\xbe\xe2", [0x12fc] = "\xc3\xcc",
  [0x12fd] = "\xb4\xed", [0x12fe] = "\xc7\xae", [0x12ff] = "\xb8\xd6",
  [0x1300] = "\xce\xfd", [0x1301] = "\xc2\xbc", [0x1302] = "\xef\xa3",
  [0x1303] = "\xd7\xb6", [0x1304] = "\xbd\xf5", [0x1306] = "\xef\xbf",
  [0x1307] = "\xef\xc0", [0x1308] = "\xef\xc5", [0x1309] = "\xd1\xd6",
  [0x130a] = "\xcb\xed", [0x130b] = "\xcb\xe6", [0x130c] = "\xcf\xd5",
  [0x130d] = "\xb5\xf1", [0x130e] = "\xf6\xae", [0x130f] = "\xd5\xb4",
  [0x1310] = "\xc1\xd8", [0x1311] = "\xbb\xf4", [0x1312] = "\xc4\xde",
  [0x1313] = "\xf6\xad", [0x1314] = "\xb5\xe5", [0x1315] = "\xbe\xb2",
  [0x1316] = "\xeb\xef", [0x1317] = "\xc7\xca", [0x1318] = "\xbc\xd5",
  [0x1319] = "\xbe\xb1", [0x131a] = "\xc6\xb5", [0x131b] = "\xf2\xa5",
  [0x131c] = "\xcd\xb7", [0x131d] = "\xcd\xc7", [0x131e] = "\xd2\xc3",
  [0x131f] = "\xb2\xcd", [0x1320] = "\xb9\xdd", [0x1321] = "\xbd\xa4",
  [0x1322] = "\xe2\xc6", [0x1323] = "\xcf\xda", [0x1324] = "\xeb\xc8",
  [0x1325] = "\xba\xa7", [0x1326] = "\xe6\xe9", [0x1327] = "\xc2\xe6",
  [0x1328] = "\xba\xa1", [0x1329] = "\xf7\xc0", [0x132a] = "\xf7\xd9",
  [0x132b] = "\xf7\xda", [0x132c] = "\xba\xe5", [0x132d] = "\xb1\xab",
  [0x132e] = "\xcd\xd2", [0x132f] = "\xf0\xb3", [0x1330] = "\xd1\xec",
  [0x1331] = "\xd1\xbc", [0x1333] = "\xd4\xa7", [0x1334] = "\xc4\xac",
  [0x1335] = "\xc7\xad", [0x1336] = "\xc1\xfa", [0x1337] = "\xb9\xea",
  [0x1338] = "\xd3\xc5", [0x1339] = "\xb3\xa5", [0x133a] = "\xc0\xdc",
  [0x133b] = "\xb4\xa2", [0x133c] = "\xc0\xf8", [0x133d] = "\xba\xbf",
  [0x133e] = "\xdf\xcc", [0x133f] = "\xb3\xa2", [0x1340] = "\xe0\xe9",
  [0x1341] = "\xcf\xc5", [0x1342] = "\xcc\xe7", [0x1343] = "\xba\xbe",
  [0x1344] = "\xd1\xb9", [0x1345] = "\xdb\xd6", [0x1346] = "\xdb\xf7",
  [0x1347] = "\xd3\xa4", [0x1348] = "\xe6\xc9", [0x1349] = "\xe6\xd6",
  [0x134a] = "\xc8\xe6", [0x134b] = "\xde\xcf", [0x134c] = "\xe5\xf0",
  [0x134d] = "\xd3\xec", [0x134e] = "\xc1\xeb", [0x134f] = "\xd4\xc0",
  [0x1350] = "\xe1\xc9", [0x1351] = "\xb0\xef", [0x1352] = "\xc3\xd6",
  [0x1353] = "\xbb\xd5", [0x1354] = "\xd3\xa6", [0x1355] = "\xb6\xae",
  [0x1356] = "\xbf\xd2", [0x1357] = "\xc5\xb3", [0x1358] = "\xed\xae",
  [0x1359] = "\xcf\xb7", [0x135a] = "\xb4\xf7", [0x135b] = "\xc7\xe6",
  [0x135c] = "\xbb\xf7", [0x135d] = "\xeb\xa2", [0x135e] = "\xbc\xb7",
  [0x135f] = "\xc5\xa1", [0x1360] = "\xb2\xc1", [0x1361] = "\xc4\xe2",
  [0x1362] = "\xb8\xe9", [0x1363] = "\xdf\xaa", [0x1365] = "\xc1\xb2",
  [0x1366] = "\xb1\xd0", [0x1367] = "\xca\xef", [0x1368] = "\xea\xd3",
  [0x1369] = "\xcc\xb4", [0x136a] = "\xb5\xb5", [0x136b] = "\xcf\xad",
  [0x136c] = "\xbc\xec", [0x136d] = "\xe8\xed", [0x136e] = "\xe8\xce",
  [0x136f] = "\xe9\xc9", [0x1371] = "\xe9\xde", [0x1372] = "\xe9\xdc",
  [0x1373] = "\xe9\xd1", [0x1375] = "\xe9\xe7", [0x1377] = "\xd5\xb1",
  [0x1378] = "\xc5\xa2", [0x1379] = "\xb1\xf5", [0x137a] = "\xbc\xc3",
  [0x137b] = "\xe5\xa9", [0x137c] = "\xc3\xc9", [0x137d] = "\xcc\xce",
  [0x137e] = "\xc0\xc4", [0x137f] = "\xe5\xaa", [0x1380] = "\xc9\xac",
  [0x1381] = "\xbf\xa3", [0x1382] = "\xe5\xa6", [0x1384] = "\xca\xaa",
  [0x1385] = "\xe5\xa7", [0x1386] = "\xce\xab", [0x1387] = "\xec\xdd",
  [0x1388] = "\xd3\xaa", [0x1389] = "\xdb\xc6", [0x138a] = "\xb2\xd3",
  [0x138b] = "\xd4\xef", [0x138c] = "\xd6\xf2", [0x138d] = "\xbb\xd9",
  [0x138e] = "\xbb\xe2", [0x138f] = "\xec\xdb", [0x1390] = "\xbe\xf4",
  [0x1391] = "\xc7\xbd", [0x1392] = "\xc4\xfc", [0x1393] = "\xbb\xf1",
  [0x1394] = "\xe8\xb3", [0x1395] = "\xbb\xb7", [0x1396] = "\xe8\xa8",
  [0x1397] = "\xe8\xb2", [0x1398] = "\xf0\xec", [0x1399] = "\xc1\xc6",
  [0x139a] = "\xb0\xa9", [0x139b] = "\xb5\xb4", [0x139c] = "\xcd\xab",
  [0x139d] = "\xb5\xc9", [0x139e] = "\xee\xab", [0x139f] = "\xcb\xb2",
  [0x13a0] = "\xc7\xc6", [0x13a1] = "\xc1\xcb", [0x13a2] = "\xbd\xc3",
  [0x13a3] = "\xc1\xd7", [0x13a4] = "\xbb\xc7", [0x13a5] = "\xed\xe3",
  [0x13a6] = "\xed\xb6", [0x13a7] = "\xbd\xb8", [0x13a8] = "\xec\xfb",
  [0x13a9] = "\xec\xf8", [0x13aa] = "\xcb\xeb", [0x13ab] = "\xc1\xfe",
  [0x13ac] = "\xb4\xd8", [0x13ad] = "\xc2\xa8", [0x13ae] = "\xf3\xfa",
  [0x13af] = "\xc5\xf1", [0x13b0] = "\xf3\xf9", [0x13b1] = "\xf3\xe3",
  [0x13b2] = "\xbf\xb7", [0x13b3] = "\xc3\xd3", [0x13b4] = "\xb7\xe0",
  [0x13b5] = "\xe2\xc9", [0x13b6] = "\xd4\xe3", [0x13b7] = "\xb2\xda",
  [0x13b8] = "\xf4\xd6", [0x13b9] = "\xcb\xf5", [0x13ba] = "\xbc\xa8",
  [0x13bb] = "\xe7\xd1", [0x13bc] = "\xc2\xc6", [0x13bd] = "\xe7\xd0",
  [0x13be] = "\xb1\xc1", [0x13bf] = "\xb7\xec", [0x13c0] = "\xd7\xdc",
  [0x13c1] = "\xd7\xdd", [0x13c2] = "\xe7\xd2", [0x13c3] = "\xb7\xb1",
  [0x13c4] = "\xcf\xcb", [0x13c5] = "\xe7\xce", [0x13c6] = "\xf1\xdf",
  [0x13c7] = "\xe7\xcf", [0x13c9] = "\xd1\xdd", [0x13ca] = "\xf3\xc0",
  [0x13cb] = "\xf4\xe8", [0x13cc] = "\xd2\xed", [0x13cd] = "\xf1\xfa",
  [0x13ce] = "\xc9\xf9", [0x13cf] = "\xb4\xcf", [0x13d0] = "\xc1\xaa",
  [0x13d1] = "\xcb\xca", [0x13d2] = "\xd2\xdc", [0x13d3] = "\xd3\xb7",
  [0x13d4] = "\xe2\xdf", [0x13d5] = "\xb1\xdb", [0x13d6] = "\xcd\xce",
  [0x13d7] = "\xc5\xa7", [0x13d8] = "\xb5\xa8", [0x13d9] = "\xc1\xb3",
  [0x13da] = "\xeb\xda", [0x13db] = "\xc1\xd9", [0x13dc] = "\xbe\xd9",
  [0x13dd] = "\xbc\xe8", [0x13de] = "\xd0\xbd", [0x13df] = "\xb1\xa1",
  [0x13e0] = "\xc0\xd9", [0x13e1] = "\xde\xb5", [0x13e2] = "\xbd\xaa",
  [0x13e3] = "\xc7\xbe", [0x13e4] = "\xca\xed", [0x13e5] = "\xd1\xa6",
  [0x13e6] = "\xde\xb1", [0x13e7] = "\xde\xb0", [0x13e8] = "\xbc\xbb",
  [0x13e9] = "\xbf\xf7", [0x13ea] = "\xf3\xb0", [0x13eb] = "\xf3\xaf",
  [0x13ec] = "\xf3\xab", [0x13ed] = "\xf2\xfe", [0x13ee] = "\xf3\xa1",
  [0x13ef] = "\xf3\xa7", [0x13f0] = "\xf2\xf7", [0x13f1] = "\xc2\xdd",
  [0x13f2] = "\xf2\xe5", [0x13f3] = "\xf3\xac", [0x13f4] = "\xd9\xf4",
  [0x13f5] = "\xf1\xde", [0x13f6] = "\xcf\xe5", [0x13f7] = "\xf1\xda",
  [0x13f9] = "\xea\xe9", [0x13fa] = "\xc3\xd5", [0x13fb] = "\xb0\xf9",
  [0x13fc] = "\xc7\xab", [0x13fd] = "\xbd\xb2", [0x13fe] = "\xbb\xd1",
  [0x13ff] = "\xd2\xa5", [0x1400] = "\xd0\xbb", [0x1401] = "\xcc\xdc",
  [0x1402] = "\xda\xd7", [0x1403] = "\xbb\xed", [0x1404] = "\xcf\xaa",
  [0x1405] = "\xe1\xd9", [0x1406] = "\xd7\xac", [0x1407] = "\xc8\xfc",
  [0x1408] = "\xb9\xba", [0x1409] = "\xca\xa3", [0x140a] = "\xea\xe7",
  [0x140b] = "\xc7\xf7", [0x140c] = "\xf5\xe3", [0x140d] = "\xcc\xa3",
  [0x140e] = "\xb5\xb8", [0x140f] = "\xf5\xe8", [0x1410] = "\xcf\xbd",
  [0x1411] = "\xd5\xb7", [0x1412] = "\xec\xb1", [0x1413] = "\xd4\xaf",
  [0x1414] = "\xd3\xdf", [0x1415] = "\xb1\xdc", [0x1416] = "\xe5\xe1",
  [0x1417] = "\xbb\xb9", [0x1418] = "\xc2\xf5", [0x1419] = "\xe5\xe2",
  [0x141a] = "\xd1\xfb", [0x141b] = "\xdb\xb8", [0x141c] = "\xf5\xb1",
  [0x141d] = "\xd4\xcd", [0x141e] = "\xb3\xf3", [0x141f] = "\xb6\xc6",
  [0x1420] = "\xc3\xbe", [0x1421] = "\xc3\xaa", [0x1422] = "\xbc\xfc",
  [0x1423] = "\xc1\xb6", [0x1424] = "\xef\xc6", [0x1425] = "\xb9\xf8",
  [0x1426] = "\xb4\xb8", [0x1427] = "\xd6\xd3", [0x1428] = "\xc7\xc2",
  [0x1429] = "\xb6\xcd", [0x142a] = "\xef\xcc", [0x142c] = "\xef\xc9",
  [0x142d] = "\xc0\xab", [0x142e] = "\xe3\xd7", [0x142f] = "\xc0\xbb",
  [0x1430] = "\xe3\xc7", [0x1431] = "\xb0\xe5", [0x1432] = "\xd2\xfe",
  [0x1433] = "\xc1\xa5", [0x1434] = "\xcb\xe4", [0x1435] = "\xcb\xaa",
  [0x1436] = "\xcf\xbc", [0x1437] = "\xbe\xcf", [0x1438] = "\xba\xab",
  [0x1439] = "\xbf\xc5", [0x143a] = "\xec\xab", [0x143b] = "\xce\xb9",
  [0x143c] = "\xb3\xd2", [0x143d] = "\xbf\xa5", [0x143e] = "\xcf\xca",
  [0x143f] = "\xf6\xde", [0x1440] = "\xf6\xdb", [0x1441] = "\xf6\xd9",
  [0x1442] = "\xba\xe8", [0x1443] = "\xb8\xeb", [0x1444] = "\xf7\xe7",
  [0x1445] = "\xf0\xa4", [0x1446] = "\xb5\xe3", [0x1447] = "\xf7\xed",
  [0x1448] = "\xf7\xee", [0x1449] = "\xf7\xec", [0x144a] = "\xf7\xfd",
  [0x144b] = "\xd5\xab", [0x144c] = "\xb4\xd4", [0x144d] = "\xe0\xe0",
  [0x144e] = "\xcf\xf2", [0x144f] = "\xdb\xdb", [0x1450] = "\xc0\xdd",
  [0x1451] = "\xc9\xf4", [0x1452] = "\xd2\xcd", [0x1453] = "\xed\xaf",
  [0x1454] = "\xb4\xc1", [0x1455] = "\xc0\xa9", [0x1456] = "\xd6\xc0",
  [0x1457] = "\xc8\xc5", [0x1458] = "\xc4\xec", [0x1459] = "\xb0\xda",
  [0x145a] = "\xcb\xd3", [0x145b] = "\xdf\xa2", [0x145c] = "\xb6\xcf",
  [0x145d] = "\xea\xd7", [0x145e] = "\xeb\xfc", [0x145f] = "\xe9\xc4",
  [0x1460] = "\xc3\xca", [0x1461] = "\xb9\xf1", [0x1462] = "\xbc\xf7",
  [0x1463] = "\xc4\xfb", [0x1464] = "\xe8\xfe", [0x1466] = "\xcc\xa8",
  [0x1467] = "\xec\xa3", [0x1468] = "\xb9\xe9", [0x1469] = "\xe9\xeb",
  [0x146a] = "\xd0\xba", [0x146b] = "\xc9\xf2", [0x146c] = "\xc2\xcb",
  [0x146d] = "\xe4\xc2", [0x146e] = "\xbd\xa6", [0x146f] = "\xc6\xd9",
  [0x1470] = "\xe4\xaf", [0x1471] = "\xd1\xac", [0x1472] = "\xbd\xfd",
  [0x1473] = "\xec\xe2", [0x1475] = "\xe1\xee", [0x1476] = "\xc1\xd4",
  [0x1477] = "\xe8\xb5", [0x1478] = "\xe8\xaf", [0x1479] = "\xce\xcd",
  [0x147a] = "\xf1\xb1", [0x147b] = "\xf0\xdd", [0x147c] = "\xd3\xfa",
  [0x147d] = "\xee\xad", [0x147e] = "\xf6\xc4", [0x147f] = "\xd5\xb0",
  [0x1480] = "\xed\xfa", [0x1481] = "\xb4\xa1", [0x1482] = "\xc0\xf1",
  [0x1483] = "\xf0\xa3", [0x1484] = "\xbb\xe0", [0x1486] = "\xb4\xdc",
  [0x1487] = "\xc7\xcf", [0x1488] = "\xf3\xef", [0x1489] = "\xbb\xc9",
  [0x148a] = "\xf4\xa2", [0x148b] = "\xf3\xec", [0x148c] = "\xf3\xf1",
  [0x148d] = "\xbc\xf2", [0x148e] = "\xc1\xb8", [0x148f] = "\xd6\xaf",
  [0x1490] = "\xc9\xc9", [0x1491] = "\xc8\xc6", [0x1492] = "\xe7\xd4",
  [0x1493] = "\xd0\xe5", [0x1494] = "\xe7\xd5", [0x1495] = "\xb7\xad",
  [0x1496] = "\xcc\xb3", [0x1497] = "\xc7\xcc", [0x1498] = "\xb7\xad",
  [0x1499] = "\xd6\xb0", [0x149a] = "\xc4\xf4", [0x149b] = "\xc6\xea",
  [0x149c] = "\xeb\xf7", [0x149d] = "\xbe\xc9", [0x149e] = "\xb2\xd8",
  [0x149f] = "\xc8\xf8", [0x14a0] = "\xc0\xb6", [0x14a1] = "\xc3\xea",
  [0x14a2] = "\xbd\xe5", [0x14a3] = "\xde\xb9", [0x14a4] = "\xdc\xf9",
  [0x14a5] = "\xde\xb7", [0x14a6] = "\xbc\xf6", [0x14a7] = "\xf2\xcd",
  [0x14a8] = "\xb2\xf5", [0x14a9] = "\xb3\xe6", [0x14aa] = "\xf3\xb4",
  [0x14ab] = "\xb8\xb2", [0x14ac] = "\xea\xee", [0x14ad] = "\xf5\xfc",
  [0x14ae] = "\xda\xd3", [0x14af] = "\xbd\xf7", [0x14b0] = "\xc3\xfd",
  [0x14b1] = "\xda\xd8", [0x14b2] = "\xb7\xe1", [0x14b3] = "\xd7\xb8",
  [0x14b4] = "\xf5\xbe", [0x14b5] = "\xf5\xe7", [0x14b6] = "\xb1\xc4",
  [0x14b7] = "\xd7\xd9", [0x14b8] = "\xbc\xa3", [0x14b9] = "\xf5\xcf",
  [0x14ba] = "\xc7\xfb", [0x14bb] = "\xd7\xaa", [0x14bc] = "\xd5\xde",
  [0x14bd] = "\xe5\xc7", [0x14be] = "\xe5\xe4", [0x14bf] = "\xe5\xe3",
  [0x14c0] = "\xd2\xbd", [0x14c1] = "\xbd\xb4", [0x14c2] = "\xc0\xe5",
  [0x14c3] = "\xc8\xdb", [0x14c4] = "\xb0\xf7", [0x14c5] = "\xcb\xf8",
  [0x14c6] = "\xce\xd9", [0x14c7] = "\xc4\xf8", [0x14c8] = "\xd5\xf2",
  [0x14c9] = "\xb8\xe4", [0x14ca] = "\xef\xd7", [0x14cb] = "\xef\xd3",
  [0x14cc] = "\xb4\xb8", [0x14cd] = "\xc7\xb9", [0x14ce] = "\xe3\xd8",
  [0x14cf] = "\xb4\xb3", [0x14d0] = "\xe3\xd9", [0x14d1] = "\xe3\xda",
  [0x14d2] = "\xc0\xeb", [0x14d3] = "\xd4\xd3", [0x14d4] = "\xcb\xab",
  [0x14d5] = "\xb3\xfb", [0x14d6] = "\xbc\xa6", [0x14d7] = "\xc1\xef",
  [0x14d8] = "\xf7\xb7", [0x14d9] = "\xc7\xef", [0x14da] = "\xb1\xde",
  [0x14dc] = "\xb6\xee", [0x14dd] = "\xd1\xd5", [0x14de] = "\xcc\xe2",
  [0x14df] = "\xf2\xa6", [0x14e0] = "\xf2\xa7", [0x14e1] = "\xd1\xef",
  [0x14e2] = "\xc1\xf3", [0x14e3] = "\xe2\xc8", [0x14e4] = "\xc0\xa1",
  [0x14e5] = "\xf7\xd1", [0x14e6] = "\xf0\xa5", [0x14e7] = "\xc6\xef",
  [0x14e8] = "\xf7\xc1", [0x14e9] = "\xd7\xd7", [0x14ea] = "\xcb\xc9",
  [0x14eb] = "\xce\xba", [0x14ec] = "\xf7\xcb", [0x14ed] = "\xf7\xcd",
  [0x14ee] = "\xf6\xe8", [0x14ef] = "\xc0\xf0", [0x14f0] = "\xf6\xea",
  [0x14f2] = "\xf6\xe7", [0x14f3] = "\xbe\xe9", [0x14f4] = "\xb6\xec",
  [0x14f5] = "\xf0\xc0", [0x14f6] = "\xf7\xef", [0x14f7] = "\xb6\xac",
  [0x14f8] = "\xf7\xf8", [0x14fa] = "\xd1\xca", [0x14fb] = "\xbb\xb5",
  [0x14fc] = "\xc2\xa2", [0x14fd] = "\xdb\xde", [0x14fe] = "\xb3\xe8",
  [0x14ff] = "\xc5\xd3", [0x1500] = "\xc2\xae", [0x1501] = "\xb3\xcd",
  [0x1502] = "\xbb\xb3", [0x1503] = "\xc0\xc1", [0x1504] = "\xe3\xc2",
  [0x1505] = "\xc5\xca", [0x1506] = "\xc2\xa3", [0x1507] = "\xbf\xf5",
  [0x1508] = "\xc6\xd8", [0x1509] = "\xb3\xf7", [0x150a] = "\xe8\xfc",
  [0x150b] = "\xe9\xb5", [0x150c] = "\xe9\xd6", [0x150d] = "\xe5\xad",
  [0x150e] = "\xe4\xec", [0x150f] = "\xe4\xfe", [0x1510] = "\xe5\xab",
  [0x1511] = "\xc1\xa4", [0x1512] = "\xb1\xf4", [0x1513] = "\xe3\xf2",
  [0x1514] = "\xb1\xac", [0x1515] = "\xcb\xb8", [0x1516] = "\xeb\xb9",
  [0x1517] = "\xb6\xbf", [0x1518] = "\xca\xde", [0x1519] = "\xcc\xa1",
  [0x151a] = "\xe7\xf4", [0x151b] = "\xc7\xed", [0x151c] = "\xb0\xea",
  [0x151d] = "\xb3\xeb", [0x151e] = "\xbd\xae", [0x151f] = "\xb1\xf1",
  [0x1520] = "\xb3\xd5", [0x1521] = "\xc3\xc9", [0x1522] = "\xb0\xad",
  [0x1523] = "\xb5\xbb", [0x1524] = "\xbb\xf1", [0x1525] = "\xce\xc8",
  [0x1526] = "\xc1\xb1", [0x1527] = "\xb2\xbe", [0x1528] = "\xf4\xa4",
  [0x1529] = "\xc7\xa9", [0x152a] = "\xe9\xdc", [0x152b] = "\xf4\xa6",
  [0x152c] = "\xcf\xb5", [0x152d] = "\xbc\xeb", [0x152e] = "\xd2\xef",
  [0x152f] = "\xc9\xfe", [0x1530] = "\xbb\xe6", [0x1531] = "\xc2\xde",
  [0x1532] = "\xbd\xc9", [0x1533] = "\xeb\xfe", [0x1534] = "\xb8\xfe",
  [0x1535] = "\xd9\xfa", [0x1536] = "\xc0\xb0", [0x1537] = "\xb7\xaa",
  [0x1538] = "\xd2\xd5", [0x1539] = "\xde\xb4", [0x153a] = "\xc5\xba",
  [0x153b] = "\xcc\xd9", [0x153c] = "\xd2\xa9", [0x153d] = "\xca\xed",
  [0x153e] = "\xd2\xcf", [0x153f] = "\xd3\xac", [0x1540] = "\xd0\xab",
  [0x1541] = "\xd0\xb7", [0x1542] = "\xf3\xb8", [0x1543] = "\xf1\xc9",
  [0x1544] = "\xbd\xf3", [0x1545] = "\xb0\xc0", [0x1546] = "\xf4\xc5",
  [0x1547] = "\xbb\xa9", [0x1548] = "\xc6\xd7", [0x1549] = "\xca\xb6",
  [0x154a] = "\xd6\xa4", [0x154b] = "\xcc\xb7", [0x154c] = "\xda\xdc",
  [0x154d] = "\xbc\xa5", [0x154e] = "\xce\xfb", [0x154f] = "\xda\xdb",
  [0x1550] = "\xd4\xf9", [0x1551] = "\xd4\xde", [0x1552] = "\xf5\xeb",
  [0x1553] = "\xb6\xd7", [0x1554] = "\xb3\xf9", [0x1555] = "\xf5\xea",
  [0x1556] = "\xb5\xc5", [0x1557] = "\xf5\xce", [0x1558] = "\xf5\xed",
  [0x1559] = "\xea\xa5", [0x155a] = "\xbd\xce", [0x155b] = "\xb4\xc7",
  [0x155c] = "\xb1\xdf", [0x155d] = "\xe5\xe5", [0x155e] = "\xb7\xa2",
  [0x155f] = "\xf5\xb4", [0x1560] = "\xbe\xb5", [0x1561] = "\xef\xe1",
  [0x1562] = "\xb2\xf9", [0x1563] = "\xef\xdf", [0x1564] = "\xc1\xb4",
  [0x1565] = "\xef\xdb", [0x1566] = "\xef\xdc", [0x1567] = "\xf7\xe9",
  [0x1568] = "\xef\xda", [0x1569] = "\xef\xdd", [0x156a] = "\xef\xcf",
  [0x156b] = "\xef\xce", [0x156c] = "\xef\xac", [0x156d] = "\xf6\xc9",
  [0x156e] = "\xb9\xd8", [0x156f] = "\xc2\xa4", [0x1570] = "\xc4\xd1",
  [0x1571] = "\xf6\xaf", [0x1572] = "\xce\xed", [0x1573] = "\xc3\xd2",
  [0x1574] = "\xe8\xba", [0x1575] = "\xd4\xcf", [0x1576] = "\xc0\xe0",
  [0x1577] = "\xd4\xb8", [0x1578] = "\xb5\xdf", [0x1579] = "\xec\xac",
  [0x157a] = "\xc2\xf8", [0x157b] = "\xe2\xcb", [0x157c] = "\xe6\xf0",
  [0x157d] = "\xc6\xad", [0x157e] = "\xba\xfa", [0x157f] = "\xbe\xa8",
  [0x1580] = "\xf6\xf0", [0x1581] = "\xf6\xeb", [0x1582] = "\xf6\xf4",
  [0x1583] = "\xf0\xc8", [0x1584] = "\xf0\xc4", [0x1585] = "\xc8\xb5",
  [0x1586] = "\xf0\xc6", [0x1587] = "\xc5\xf4", [0x1588] = "\xf7\xe8",
  [0x1589] = "\xc0\xf6", [0x158a] = "\xc2\xb4", [0x158b] = "\xf4\xf0",
  [0x158c] = "\xc8\xb0", [0x158d] = "\xc1\xfc", [0x158e] = "\xc8\xc2",
  [0x158f] = "\xe0\xd3", [0x1590] = "\xd1\xcf", [0x1591] = "\xbd\xc0",
  [0x1592] = "\xc8\xc0", [0x1593] = "\xe6\xd7", [0x1594] = "\xc4\xef",
  [0x1595] = "\xc4\xf5", [0x1596] = "\xb1\xa6", [0x1598] = "\xd0\xfc",
  [0x1599] = "\xe2\xe3", [0x159a] = "\xc8\xc1", [0x159b] = "\xc0\xb9",
  [0x159c] = "\xb2\xf3", [0x159d] = "\xea\xd8", [0x159e] = "\xeb\xca",
  [0x159f] = "\xe9\xb4", [0x15a0] = "\xc0\xbd", [0x15a1] = "\xc3\xd6",
  [0x15a2] = "\xe4\xf2", [0x15a3] = "\xc2\xaf", [0x15a4] = "\xcf\xd7",
  [0x15a5] = "\xe7\xe7", [0x15a6] = "\xd1\xf7", [0x15a7] = "\xd6\xa2",
  [0x15a8] = "\xbf\xf3", [0x15a9] = "\xed\xc2", [0x15aa] = "\xb7\xaf",
  [0x15ab] = "\xc0\xf9", [0x15ac] = "\xf1\xbc", [0x15ad] = "\xbe\xba",
  [0x15ae] = "\xb3\xef", [0x15af] = "\xc0\xba", [0x15b0] = "\xbc\xae",
  [0x15b1] = "\xc5\xb4", [0x15b2] = "\xcd\xc5", [0x15b3] = "\xb1\xe8",
  [0x15b4] = "\xe7\xcd", [0x15b5] = "\xbc\xcc", [0x15b6] = "\xd7\xeb",
  [0x15b7] = "\xf3\xbf", [0x15b8] = "\xd2\xab", [0x15b9] = "\xeb\xcd",
  [0x15ba] = "\xbd\xa2", [0x15bb] = "\xd4\xe5", [0x15bc] = "\xb0\xaa",
  [0x15bd] = "\xc4\xa2", [0x15be] = "\xdd\xfe", [0x15bf] = "\xc2\xab",
  [0x15c0] = "\xc6\xbb", [0x15c1] = "\xcb\xd5", [0x15c2] = "\xd4\xcc",
  [0x15c3] = "\xf2\xba", [0x15c4] = "\xc8\xe4", [0x15c5] = "\xf1\xdc",
  [0x15c6] = "\xbe\xf5", [0x15c7] = "\xb4\xa5", [0x15c8] = "\xd2\xe9",
  [0x15c9] = "\xc6\xa9", [0x15ca] = "\xbe\xaf", [0x15cb] = "\xd2\xeb",
  [0x15cc] = "\xd4\xeb", [0x15cd] = "\xda\xde", [0x15ce] = "\xd3\xae",
  [0x15cf] = "\xc9\xc4", [0x15d0] = "\xf5\xbb", [0x15d1] = "\xd4\xea",
  [0x15d2] = "\xf5\xee", [0x15d4] = "\xf5\xb7", [0x15d5] = "\xca\xcd",
  [0x15d6] = "\xd6\xd3", [0x15d7] = "\xee\xf3", [0x15d8] = "\xd0\xe2",
  [0x15d9] = "\xb2\xfb", [0x15da] = "\xf6\xb1", [0x15db] = "\xc6\xae",
  [0x15dc] = "\xc8\xc4", [0x15dd] = "\xbc\xa2", [0x15de] = "\xdc\xb0",
  [0x15df] = "\xe5\xb9", [0x15e0] = "\xcc\xda", [0x15e1] = "\xc9\xa7",
  [0x15e3] = "\xc8\xfa", [0x15e4] = "\xf6\xfa", [0x15e5] = "\xcf\xcc",
  [0x15e6] = "\xc3\xe6", [0x15e7] = "\xb5\xb3", [0x15e8] = "\xf7\xf9",
  [0x15e9] = "\xf6\xb4", [0x15ea] = "\xb3\xf6", [0x15eb] = "\xc1\xe4",
  [0x15ec] = "\xd9\xb3", [0x15ed] = "\xc2\xde", [0x15ee] = "\xe0\xbf",
  [0x15ef] = "\xdf\xf9", [0x15f0] = "\xcf\xf9", [0x15f1] = "\xd9\xe7",
  [0x15f2] = "\xca\xf4", [0x15f3] = "\xce\xa1", [0x15f4] = "\xbe\xe5",
  [0x15f5] = "\xc9\xe5", [0x15f6] = "\xc9\xe3", [0x15f7] = "\xd0\xaf",
  [0x15f8] = "\xec\xb5", [0x15f9] = "\xea\xd9", [0x15fa] = "\xd3\xa3",
  [0x15fb] = "\xc0\xb8", [0x15fc] = "\xe8\xf9", [0x15fd] = "\xbc\xdf",
  [0x15fe] = "\xb9\xe0", [0x15ff] = "\xc0\xc3", [0x1600] = "\xce\xfe",
  [0x1601] = "\xcf\xe2", [0x1602] = "\xe8\xac", [0x1603] = "\xf1\xae",
  [0x1604] = "\xeb\xca", [0x1605] = "\xcc\xd9", [0x1606] = "\xb2\xf8",
  [0x1607] = "\xd0\xf8", [0x1608] = "\xe5\xf1", [0x1609] = "\xde\xc1",
  [0x160a] = "\xc0\xbc", [0x160b] = "\xde\xba", [0x160c] = "\xf2\xc3",
  [0x160d] = "\xb4\xc0", [0x160e] = "\xf3\xbb", [0x160f] = "\xc0\xaf",
  [0x1610] = "\xcd\xe0", [0x1611] = "\xb0\xda", [0x1612] = "\xc0\xc0",
  [0x1613] = "\xc7\xb4", [0x1614] = "\xbb\xa4", [0x1615] = "\xd3\xfe",
  [0x1616] = "\xd4\xdf", [0x1617] = "\xb3\xec", [0x1618] = "\xd4\xbe",
  [0x1619] = "\xf5\xd2", [0x161a] = "\xba\xe4", [0x161b] = "\xb1\xe7",
  [0x161c] = "\xf5\xb8", [0x161d] = "\xc1\xad", [0x161e] = "\xc0\xd8",
  [0x161f] = "\xcc\xfa", [0x1620] = "\xee\xf5", [0x1621] = "\xee\xec",
  [0x1622] = "\xef\xed", [0x1623] = "\xef\xd4", [0x1624] = "\xb1\xd9",
  [0x1625] = "\xb0\xd4", [0x1626] = "\xc5\xf9", [0x1627] = "\xc2\xb6",
  [0x1628] = "\xcf\xec", [0x1629] = "\xb9\xcb", [0x162a] = "\xf2\xab",
  [0x162b] = "\xf7\xcf", [0x162c] = "\xc7\xfd", [0x162d] = "\xe6\xf4",
  [0x162e] = "\xdd\xeb", [0x162f] = "\xc2\xe2", [0x1630] = "\xf7\xc3",
  [0x1631] = "\xc4\xa7", [0x1632] = "\xf7\xce", [0x1633] = "\xf7\xa2",
  [0x1634] = "\xf7\xa4", [0x1635] = "\xdd\xba", [0x1636] = "\xba\xd7",
  [0x1637] = "\xf0\xce", [0x1639] = "\xf7\xea", [0x163a] = "\xf7\xf6",
  [0x163b] = "\xdc\xb1", [0x163c] = "\xf6\xb7", [0x163d] = "\xf6\xb8",
  [0x163e] = "\xc4\xf6", [0x163f] = "\xd9\xb2", [0x1640] = "\xd9\xce",
  [0x1641] = "\xdf\xbd", [0x1642] = "\xc4\xd2", [0x1643] = "\xc2\xde",
  [0x1644] = "\xc2\xcf", [0x1645] = "\xe1\xdb", [0x1646] = "\xc2\xcd",
  [0x1647] = "\xcd\xe4", [0x1648] = "\xdc\xb2", [0x1649] = "\xcc\xaf",
  [0x164a] = "\xc8\xa8", [0x164b] = "\xbb\xb6", [0x164c] = "\xc8\xf7",
  [0x164d] = "\xcc\xb2", [0x164e] = "\xe2\xa4", [0x164f] = "\xc8\xbf",
  [0x1650] = "\xb5\xfe", [0x1651] = "\xf1\xab", [0x1652] = "\xd1\xa2",
  [0x1653] = "\xec\xfc", [0x1654] = "\xc1\xfd", [0x1655] = "\xf4\xa5",
  [0x1656] = "\xc1\xfb", [0x1657] = "\xcc\xfd", [0x1658] = "\xd4\xe0",
  [0x1659] = "\xcf\xae", [0x165a] = "\xb3\xc4", [0x165c] = "\xb6\xc1",
  [0x165d] = "\xca\xea", [0x165e] = "\xd8\xcd", [0x165f] = "\xf5\xdc",
  [0x1660] = "\xf5\xd9", [0x1661] = "\xe0\xce", [0x1662] = "\xdb\xaa",
  [0x1663] = "\xd6\xfd", [0x1664] = "\xbc\xf8", [0x1665] = "\xbc\xf8",
  [0x1666] = "\xf6\xab", [0x1667] = "\xf6\xb2", [0x1668] = "\xf7\xb2",
  [0x1669] = "\xe7\xd6", [0x166a] = "\xb2\xfc", [0x166b] = "\xf7\xd2",
  [0x166c] = "\xbd\xbe", [0x166d] = "\xe6\xe7", [0x166e] = "\xd4\xe0",
  [0x166f] = "\xd0\xeb", [0x1670] = "\xb1\xee", [0x1671] = "\xf6\xe3",
  [0x1672] = "\xf7\xa7", [0x1673] = "\xf7\xa9", [0x1674] = "\xf0\xd1",
  [0x1675] = "\xc5\xb8", [0x1676] = "\xf7\xfa", [0x1677] = "\xf6\xb9",
  [0x1678] = "\xf6\xba", [0x1679] = "\xb9\xa8", [0x167a] = "\xcb\xd5",
  [0x167b] = "\xd1\xd2", [0x167c] = "\xc1\xb5", [0x167d] = "\xc2\xce",
  [0x167e] = "\xbe\xf0", [0x167f] = "\xbd\xc1", [0x1680] = "\xc9\xb9",
  [0x1682] = "\xe8\xb6", [0x1683] = "\xc7\xd4", [0x1684] = "\xc7\xa9",
  [0x1686] = "\xd9\xdf", [0x1687] = "\xd3\xa7", [0x1688] = "\xcf\xcb",
  [0x1689] = "\xb2\xc5", [0x168b] = "\xd5\xba", [0x168c] = "\xc2\xdc",
  [0x168d] = "\xb9\xc6", [0x168e] = "\xb1\xe4", [0x168f] = "\xe5\xce",
  [0x1690] = "\xc2\xdf", [0x1691] = "\xef\xf0", [0x1692] = "\xee\xe5",
  [0x1693] = "\xc5\xd9", [0x1694] = "\xd8\xcc", [0x1695] = "\xcf\xd4",
  [0x1696] = "\xf7\xd0", [0x1697] = "\xbe\xaa", [0x1698] = "\xe6\xe4",
  [0x1699] = "\xd1\xe9", [0x169a] = "\xcb\xe8", [0x169b] = "\xcc\xe5",
  [0x169c] = "\xf7\xc7", [0x169d] = "\xf7\xad", [0x169e] = "\xc1\xdb",
  [0x169f] = "\xf7\xac", [0x16a0] = "\xf0\xb8", [0x16a1] = "\xf7\xeb",
  [0x16a2] = "\xc3\xb9", [0x16a3] = "\xd6\xf6", [0x16a4] = "\xb0\xd3",
  [0x16a5] = "\xc0\xbf", [0x16a6] = "\xe5\xb1", [0x16a7] = "\xcc\xb1",
  [0x16a8] = "\xf1\xb2", [0x16a9] = "\xb4\xa3", [0x16aa] = "\xb9\xde",
  [0x16ab] = "\xee\xbf", [0x16ac] = "\xb2\xcf", [0x16ad] = "\xf3\xbc",
  [0x16ae] = "\xe1\xe9", [0x16af] = "\xc8\xc3", [0x16b0] = "\xb2\xf7",
  [0x16b1] = "\xda\xdf", [0x16b2] = "\xd1\xde", [0x16b3] = "\xb8\xd3",
  [0x16b4] = "\xc4\xf0", [0x16b5] = "\xc2\xaf", [0x16b6] = "\xf6\xa8",
  [0x16b7] = "\xc1\xe9", [0x16b8] = "\xf6\xb0", [0x16b9] = "\xc7\xa7",
  [0x16ba] = "\xf2\xad", [0x16bb] = "\xd6\xe8", [0x16bc] = "\xf7\xde",
  [0x16bd] = "\xf7\xca", [0x16be] = "\xf6\xd7", [0x16bf] = "\xd3\xa5",
  [0x16c0] = "\xf0\xd8", [0x16c1] = "\xbc\xef", [0x16c2] = "\xd1\xce",
  [0x16c3] = "\xf7\xa1", [0x16c4] = "\xf6\xbb", [0x16c5] = "\xc8\xa3",
  [0x16c6] = "\xcc\xfc", [0x16c7] = "\xe9\xad", [0x16c8] = "\xcd\xe5",
  [0x16c9] = "\xc0\xe9", [0x16ca] = "\xc2\xe1", [0x16cb] = "\xc2\xf9",
  [0x16cc] = "\xb9\xdb", [0x16cd] = "\xf5\xe6", [0x16ce] = "\xd0\xc6",
  [0x16cf] = "\xcf\xe2", [0x16d0] = "\xd4\xbf", [0x16d1] = "\xc2\xad",
  [0x16d2] = "\xb2\xf6", [0x16d3] = "\xf7\xc5", [0x16d4] = "\xf7\xe0",
  [0x16d5] = "\xd9\xe4", [0x16d6] = "\xc2\xd0", [0x16d7] = "\xd6\xf5",
  [0x16d8] = "\xd4\xde", [0x16d9] = "\xc4\xf7", [0x16da] = "\xf7\xb5",
  [0x16db] = "\xc2\xbf", [0x16dc] = "\xe6\xf7", [0x16dd] = "\xc0\xc2",
  [0x16de] = "\xda\xd4", [0x16df] = "\xf5\xef", [0x16e0] = "\xf5\xa6",
  [0x16e1] = "\xd7\xea", [0x16e2] = "\xf6\xc7", [0x16e3] = "\xc2\xe0",
  [0x16e4] = "\xf6\xf9", [0x16e5] = "\xf6\xd4", [0x16e6] = "\xf7\xf2",
  [0x16e7] = "\xd1\xde", [0x16e8] = "\xd4\xe4", [0x16e9] = "\xf0\xd0",
  [0x16ea] = "\xec\xe0", [0x16eb] = "\xe6\xea", [0x16ec] = "\xd3\xf4",
  [0x16ed] = "\xf0\xd9", [0x16ee] = "\xf0\xbd", [0x16ef] = "\xd3\xf5",
  [0x1889] = "\xd8\xbf", [0x188a] = "\xdb\xc9", [0x188b] = "\xd8\xce",
  [0x188c] = "\xb3\xa7", [0x188d] = "\xcd\xf2", [0x188e] = "\xd8\xa2",
  [0x188f] = "\xd8\xb1", [0x1890] = "\xd8\xa1", [0x1891] = "\xe0\xed",
  [0x1892] = "\xd8\xa3", [0x1893] = "\xe5\xf8", [0x1894] = "\xe1\xdc",
  [0x1897] = "\xd3\xeb", [0x1899] = "\xd8\xc1", [0x189a] = "\xd8\xec",
  [0x189b] = "\xd8\xeb", [0x189f] = "\xb0\xba", [0x18a7] = "\xec\xaf",
  [0x18a8] = "\xce\xe3", [0x18a9] = "\xc6\xf8", [0x18aa] = "\xe3\xdd",
  [0x18ac] = "\xbe\xae", [0x18ad] = "\xd8\xed", [0x18b0] = "\xd8\xee",
  [0x18b1] = "\xd9\xda", [0x18b7] = "\xca\xa5", [0x18b9] = "\xba\xbb",
  [0x18ba] = "\xc4\xfe", [0x18bb] = "\xe5\xb3", [0x18bd] = "\xe5\xea",
  [0x18c1] = "\xe2\xcf", [0x18c2] = "\xd8\xc6", [0x18c3] = "\xe2\xe1",
  [0x18c4] = "\xee\xe1", [0x18c5] = "\xd8\xec", [0x18c6] = "\xeb\xad",
  [0x18cc] = "\xe1\xec", [0x18d1] = "\xbc\xbf", [0x18d2] = "\xd3\xc5",
  [0x18d4] = "\xd8\xf5", [0x18d8] = "\xbc\xdb", [0x18dd] = "\xd8\xf3",
  [0x18e2] = "\xd9\xfc", [0x18ec] = "\xe0\xef", [0x18ed] = "\xd8\xb6",
  [0x18ee] = "\xdb\xdc", [0x18ef] = "\xdb\xd9", [0x18f1] = "\xde\xc5",
  [0x18fa] = "\xe6\xd8", [0x18fb] = "\xde\xcd", [0x18fd] = "\xe1\xa8",
  [0x1901] = "\xbf\xaa", [0x1902] = "\xd7\xaf", [0x1903] = "\xd2\xec",
  [0x1908] = "\xe2\xe3", [0x190a] = "\xba\xb4", [0x190d] = "\xc7\xa4",
  [0x1913] = "\xea\xb9", [0x1914] = "\xea\xb8", [0x1919] = "\xbb\xfa",
  [0x191d] = "\xeb\xae", [0x191e] = "\xd9\xdb", [0x1920] = "\xe3\xe1",
  [0x1922] = "\xe3\xe2", [0x1923] = "\xe3\xe0", [0x1928] = "\xe1\xed",
  [0x192a] = "\xe7\xe0", [0x192b] = "\xbd\xc7", [0x192e] = "\xcd\xf8",
  [0x192f] = "\xdc\xb3", [0x1932] = "\xdc\xb4", [0x1933] = "\xdc\xb5",
  [0x1934] = "\xf2\xae", [0x1935] = "\xce\xf7", [0x1936] = "\xda\xf8",
  [0x1937] = "\xda\xf5", [0x1939] = "\xda\xf6", [0x193b] = "\xda\xe3",
  [0x1943] = "\xcc\xe5", [0x1944] = "\xd8\xf4", [0x1946] = "\xd8\xfb",
  [0x1948] = "\xd9\xa1", [0x194a] = "\xd9\xdc", [0x1954] = "\xdb\xbf",
  [0x1957] = "\xd8\xd5", [0x195e] = "\xdf\xbe", [0x1967] = "\xb6\xd6",
  [0x196f] = "\xdb\xd0", [0x1978] = "\xe6\xa1", [0x1983] = "\xe1\xa9",
  [0x1985] = "\xe1\xac", [0x1990] = "\xe2\xd1", [0x1999] = "\xdf\xaf",
  [0x199a] = "\xec\xfd", [0x199b] = "\xec\xfe", [0x199c] = "\xe2\xed",
  [0x199e] = "\xe2\xe5", [0x19a0] = "\xe2\xe7", [0x19a1] = "\xe2\xe8",
  [0x19a6] = "\xd0\xc3", [0x19a7] = "\xbb\xb3", [0x19aa] = "\xde\xd5",
  [0x19ae] = "\xb1\xa7", [0x19b0] = "\xb2\xe5", [0x19b2] = "\xb5\xd6",
  [0x19b3] = "\xc8\xc5", [0x19ba] = "\xbf\xbc", [0x19bb] = "\xea\xba",
  [0x19c1] = "\xdb\xd8", [0x19c4] = "\xe8\xbb", [0x19c5] = "\xe8\xbe",
  [0x19cb] = "\xeb\xaf", [0x19cc] = "\xeb\xb0", [0x19d2] = "\xc6\xe3",
  [0x19d5] = "\xe3\xe9", [0x19e5] = "\xe1\xf0", [0x19e7] = "\xe1\xf1",
  [0x19ef] = "\xee\xae", [0x19f1] = "\xf0\xdb", [0x19f3] = "\xd4\xed",
  [0x19f8] = "\xb8\xec", [0x19fa] = "\xeb\xc0", [0x19fc] = "\xdc\xb6",
  [0x19fe] = "\xdc\xba", [0x19ff] = "\xdc\xbb", [0x1a01] = "\xdc\xb7",
  [0x1a03] = "\xdc\xb9", [0x1a04] = "\xf5\xf4", [0x1a08] = "\xda\xfa",
  [0x1a0e] = "\xb6\xf2", [0x1a0f] = "\xd6\xb7", [0x1a13] = "\xd9\xae",
  [0x1a18] = "\xd9\xa5", [0x1a19] = "\xd9\xa6", [0x1a1a] = "\xd9\xa8",
  [0x1a1b] = "\xd6\xb6", [0x1a1e] = "\xb6\xb1", [0x1a1f] = "\xbb\xb2",
  [0x1a20] = "\xe1\xdf", [0x1a26] = "\xd9\xb0", [0x1a2e] = "\xd9\xfe",
  [0x1a32] = "\xd8\xda", [0x1a37] = "\xcc\xd5", [0x1a3f] = "\xdf\xc6",
  [0x1a47] = "\xdf\xcf", [0x1a4f] = "\xdf\xca", [0x1a51] = "\xe0\xf2",
  [0x1a52] = "\xc5\xf7", [0x1a54] = "\xdb\xe8", [0x1a55] = "\xdb\xe3",
  [0x1a58] = "\xdb\xe9", [0x1a5a] = "\xc7\xf0", [0x1a5b] = "\xdb\xe6",
  [0x1a5c] = "\xdb\xea", [0x1a5f] = "\xdb\xe7", [0x1a67] = "\xe6\xa7",
  [0x1a74] = "\xe6\xdf", [0x1a75] = "\xe6\xdb", [0x1a76] = "\xe5\xb5",
  [0x1a77] = "\xe5\xb4", [0x1a7d] = "\xe1\xb2", [0x1a80] = "\xe1\xb5",
  [0x1a82] = "\xe1\xb8", [0x1a83] = "\xc1\xeb", [0x1a84] = "\xe1\xb3",
  [0x1a8d] = "\xe0\xfa", [0x1a8e] = "\xe0\xf9", [0x1a94] = "\xe1\xde",
  [0x1a9a] = "\xe2\xf1", [0x1a9b] = "\xe2\xef", [0x1a9f] = "\xe2\xf7",
  [0x1aa6] = "\xe2\xf4", [0x1aad] = "\xc1\xaf", [0x1aae] = "\xea\xa7",
  [0x1aaf] = "\xec\xe6", [0x1ab1] = "\xd2\xb7", [0x1ab2] = "\xc7\xaf",
  [0x1ab6] = "\xde\xd4", [0x1aba] = "\xde\xd3", [0x1ac7] = "\xea\xbe",
  [0x1acf] = "\xeb\xc3", [0x1ad6] = "\xe8\xc4", [0x1ad8] = "\xb9\xb9",
  [0x1ae2] = "\xbc\xab", [0x1ae7] = "\xd8\xb2", [0x1aeb] = "\xed\xb3",
  [0x1aed] = "\xe3\xf9", [0x1aee] = "\xe3\xfa", [0x1af1] = "\xe3\xef",
  [0x1af2] = "\xe3\xf0", [0x1af5] = "\xe3\xee", [0x1afd] = "\xcb\xdd",
  [0x1b02] = "\xc5\xa2", [0x1b0c] = "\xc8\xb2", [0x1b0e] = "\xea\xc1",
  [0x1b13] = "\xec\xc0", [0x1b1d] = "\xe1\xf4", [0x1b1f] = "\xb5\xa9",
  [0x1b26] = "\xe7\xe3", [0x1b2d] = "\xee\xaf", [0x1b2e] = "\xe7\xde",
  [0x1b33] = "\xed\xec", [0x1b36] = "\xed\xb7", [0x1b3c] = "\xcb\xfb",
  [0x1b3f] = "\xf1\xb6", [0x1b44] = "\xf1\xf4", [0x1b45] = "\xb2\xd9",
  [0x1b46] = "\xb0\xb9", [0x1b4a] = "\xeb\xc7", [0x1b4e] = "\xdc\xbe",
  [0x1b50] = "\xdc\xc5", [0x1b54] = "\xdc\xc7", [0x1b58] = "\xdc\xcc",
  [0x1b59] = "\xdc\xb8", [0x1b5a] = "\xdc\xcd", [0x1b5b] = "\xdc\xcb",
  [0x1b5d] = "\xdc\xd2", [0x1b62] = "\xf2\xb0", [0x1b64] = "\xf2\xb1",
  [0x1b65] = "\xd8\xa1", [0x1b68] = "\xe5\xc2", [0x1b6b] = "\xe5\xc3",
  [0x1b6e] = "\xda\xfb", [0x1b6f] = "\xba\xaa", [0x1b70] = "\xda\xfc",
  [0x1b71] = "\xdb\xa2", [0x1b73] = "\xda\xe7", [0x1b74] = "\xda\xe8",
  [0x1b78] = "\xd9\xb4", [0x1b7d] = "\xd9\xb6", [0x1b7f] = "\xd9\xb7",
  [0x1b8a] = "\xd8\xd9", [0x1b8b] = "\xef\xb1", [0x1b92] = "\xd8\xc7",
  [0x1b93] = "\xc0\xe5", [0x1b96] = "\xdf\xd2", [0x1b98] = "\xdf\xe7",
  [0x1b9a] = "\xb9\xb6", [0x1b9b] = "\xdf\xfb", [0x1ba0] = "\xb6\xdf",
  [0x1ba3] = "\xdf\xc3", [0x1ba6] = "\xdf\xda", [0x1ba7] = "\xdf\xe8",
  [0x1bac] = "\xdb\xec", [0x1bad] = "\xdb\xed", [0x1bb0] = "\xb6\xe2",
  [0x1bb9] = "\xb8\xb4", [0x1bc0] = "\xe6\xad", [0x1bd6] = "\xe1\xbe",
  [0x1be6] = "\xda\xe1", [0x1bf1] = "\xe2\xd3", [0x1bf4] = "\xe5\xe8",
  [0x1c02] = "\xe2\xfe", [0x1c0a] = "\xec\xe7", [0x1c0f] = "\xbf\xe6",
  [0x1c14] = "\xb8\xf1", [0x1c16] = "\xde\xd9", [0x1c22] = "\xea\xc6",
  [0x1c25] = "\xea\xc7", [0x1c2c] = "\xea\xc3", [0x1c2d] = "\xea\xc4",
  [0x1c31] = "\xeb\xd4", [0x1c32] = "\xe8\xde", [0x1c36] = "\xb9\xf1",
  [0x1c39] = "\xe8\xcf", [0x1c3b] = "\xbc\xcf", [0x1c40] = "\xe8\xd5",
  [0x1c42] = "\xe8\xd7", [0x1c4b] = "\xe8\xcd", [0x1c54] = "\xb9\xd5",
  [0x1c57] = "\xe8\xda", [0x1c5b] = "\xe9\xe3", [0x1c5c] = "\xe9\xe5",
  [0x1c5e] = "\xb1\xd1", [0x1c5f] = "\xc5\xfe", [0x1c62] = "\xeb\xb1",
  [0x1c66] = "\xcc\xe9", [0x1c67] = "\xcd\xdd", [0x1c69] = "\xc8\xf7",
  [0x1c6c] = "\xe4\xb2", [0x1c6d] = "\xe4\xa7", [0x1c6e] = "\xe4\xa8",
  [0x1c70] = "\xe4\xae", [0x1c75] = "\xbd\xe0", [0x1c7b] = "\xe4\xa6",
  [0x1c82] = "\xec\xc4", [0x1c85] = "\xec\xc6", [0x1c86] = "\xc5\xda",
  [0x1c91] = "\xea\xf0", [0x1c94] = "\xe1\xf5", [0x1c9a] = "\xc3\xee",
  [0x1c9c] = "\xe7\xe6", [0x1c9d] = "\xe7\xec", [0x1cac] = "\xce\xcd",
  [0x1caf] = "\xee\xb2", [0x1cb4] = "\xed\xf1", [0x1cb6] = "\xed\xed",
  [0x1cbc] = "\xef\xf2", [0x1cbf] = "\xed\xbc", [0x1cc0] = "\xc5\xf8",
  [0x1cc5] = "\xed\xb9", [0x1ccb] = "\xec\xec", [0x1ccd] = "\xef\xf5",
  [0x1cce] = "\xd6\xd6", [0x1ccf] = "\xba\xc4", [0x1cd2] = "\xf1\xb8",
  [0x1cde] = "\xe6\xfd", [0x1ce0] = "\xee\xb7", [0x1ce7] = "\xf1\xe8",
  [0x1ce8] = "\xde\xc7", [0x1cea] = "\xb7\xce", [0x1cee] = "\xeb\xcf",
  [0x1cf2] = "\xeb\xd1", [0x1cf3] = "\xca\xa4", [0x1cf4] = "\xeb\xd4",
  [0x1cf8] = "\xeb\xd3", [0x1cfa] = "\xeb\xd2", [0x1cfc] = "\xf4\xad",
  [0x1d00] = "\xc6\xbb", [0x1d01] = "\xdc\xd8", [0x1d04] = "\xdc\xe6",
  [0x1d05] = "\xdc\xfb", [0x1d06] = "\xc9\xbb", [0x1d08] = "\xdc\xda",
  [0x1d0a] = "\xdc\xd3", [0x1d0d] = "\xdc\xdd", [0x1d0e] = "\xdc\xde",
  [0x1d12] = "\xdc\xd6", [0x1d13] = "\xdc\xe5", [0x1d14] = "\xdd\xae",
  [0x1d19] = "\xf2\xb4", [0x1d1f] = "\xf1\xc3", [0x1d22] = "\xd9\xea",
  [0x1d26] = "\xe5\xc5", [0x1d2c] = "\xdb\xa4", [0x1d2d] = "\xdb\xa5",
  [0x1d2e] = "\xdb\xa8", [0x1d31] = "\xee\xc6", [0x1d32] = "\xee\xc5",
  [0x1d33] = "\xda\xeb", [0x1d34] = "\xcb\xe5", [0x1d44] = "\xd9\xbe",
  [0x1d48] = "\xd9\xc3", [0x1d4f] = "\xb5\xb3", [0x1d53] = "\xc6\xe0",
  [0x1d54] = "\xba\xd4", [0x1d55] = "\xbe\xbb", [0x1d57] = "\xd8\xdf",
  [0x1d5a] = "\xd8\xde", [0x1d65] = "\xdf\xea", [0x1d66] = "\xdf\xee",
  [0x1d69] = "\xdb\xc1", [0x1d6a] = "\xdf\xc2", [0x1d6d] = "\xdf\xf2",
  [0x1d7c] = "\xdb\xf4", [0x1d7d] = "\xdb\xf8", [0x1d82] = "\xdb\xf9",
  [0x1d95] = "\xd8\xab", [0x1da9] = "\xe5\xf2", [0x1dac] = "\xed\xa2",
  [0x1dad] = "\xed\xa3", [0x1dae] = "\xed\xa4", [0x1daf] = "\xed\xa5",
  [0x1db3] = "\xe3\xa8", [0x1db5] = "\xe3\xa6", [0x1db6] = "\xe3\xa7",
  [0x1db8] = "\xe3\xaa", [0x1dc4] = "\xcd\xb1", [0x1dc5] = "\xbe\xd6",
  [0x1dc6] = "\xde\xdc", [0x1dc8] = "\xde\xda", [0x1dc9] = "\xde\xdb",
  [0x1ddd] = "\xec\xb7", [0x1dde] = "\xec\xb9", [0x1ddf] = "\xec\xb8",
  [0x1de0] = "\xc6\xec", [0x1de2] = "\xea\xc9", [0x1de9] = "\xe8\xf1",
  [0x1dea] = "\xe8\xe0", [0x1deb] = "\xe8\xe1", [0x1def] = "\xc6\xdc",
  [0x1df6] = "\xe8\xe4", [0x1df7] = "\xe8\xe6", [0x1df9] = "\xe8\xe9",
  [0x1dfa] = "\xd1\xae", [0x1e13] = "\xe4\xbd", [0x1e1c] = "\xe4\xb4",
  [0x1e1d] = "\xe4\xb3", [0x1e22] = "\xe4\xb7", [0x1e24] = "\xe4\xbb",
  [0x1e27] = "\xe4\xbc", [0x1e29] = "\xcd\xbf", [0x1e4f] = "\xe1\xfe",
  [0x1e50] = "\xe1\xf9", [0x1e53] = "\xe1\xfc", [0x1e54] = "\xe2\xa1",
  [0x1e55] = "\xe1\xfb", [0x1e57] = "\xe7\xee", [0x1e58] = "\xe7\xed",
  [0x1e5b] = "\xe7\xf2", [0x1e5d] = "\xe7\xf1", [0x1e67] = "\xf0\xac",
  [0x1e69] = "\xea\xb2", [0x1e6c] = "\xee\xb3", [0x1e6e] = "\xf0\xe6",
  [0x1e71] = "\xf0\xe4", [0x1e73] = "\xf0\xf2", [0x1e80] = "\xed\xf4",
  [0x1e81] = "\xed\xf2", [0x1e82] = "\xed\xf3", [0x1e84] = "\xed\xc8",
  [0x1e85] = "\xed\xc7", [0x1e8d] = "\xed\xc9", [0x1e92] = "\xec\xee",
  [0x1e94] = "\xec\xef", [0x1e95] = "\xec\xf0", [0x1e98] = "\xef\xf8",
  [0x1e9c] = "\xef\xf6", [0x1ea1] = "\xf1\xb9", [0x1eaa] = "\xf3\xc7",
  [0x1ead] = "\xf3\xcb", [0x1eae] = "\xf3\xc5", [0x1eaf] = "\xf3\xc9",
  [0x1eb4] = "\xf4\xce", [0x1ec0] = "\xe7\xa3", [0x1ec6] = "\xee\xb8",
  [0x1ed0] = "\xf1\xe9", [0x1ed3] = "\xb0\xb7", [0x1ed4] = "\xeb\xdc",
  [0x1eda] = "\xf4\xa8", [0x1edb] = "\xf4\xb1", [0x1edd] = "\xdc\xfc",
  [0x1ede] = "\xdc\xfa", [0x1ee1] = "\xdc\xe8", [0x1ee7] = "\xdc\xe7",
  [0x1eeb] = "\xdd\xa2", [0x1eed] = "\xdc\xeb", [0x1eee] = "\xdc\xed",
  [0x1ef4] = "\xdc\xf2", [0x1ef6] = "\xdc\xf4", [0x1ef7] = "\xb4\xf0",
  [0x1efb] = "\xb2\xe7", [0x1efd] = "\xbc\xeb", [0x1f02] = "\xf2\xb6",
  [0x1f04] = "\xf2\xb7", [0x1f0a] = "\xf2\xb8", [0x1f10] = "\xf2\xbb",
  [0x1f11] = "\xb2\xcf", [0x1f14] = "\xf2\xba", [0x1f18] = "\xf4\xac",
  [0x1f1c] = "\xf1\xc4", [0x1f1f] = "\xf1\xc6", [0x1f22] = "\xf4\xc0",
  [0x1f26] = "\xf4\xf8", [0x1f2b] = "\xb8\xcf", [0x1f2d] = "\xf5\xc0",
  [0x1f34] = "\xca\xca", [0x1f37] = "\xe5\xcc", [0x1f3f] = "\xdb\xa3",
  [0x1f42] = "\xdb\xae", [0x1f43] = "\xdb\xad", [0x1f44] = "\xdb\xac",
  [0x1f46] = "\xf4\xfb", [0x1f47] = "\xf4\xfc", [0x1f48] = "\xf4\xfd",
  [0x1f49] = "\xee\xc9", [0x1f4d] = "\xda\xec", [0x1f4e] = "\xf6\xc0",
  [0x1f50] = "\xf7\xd4", [0x1f51] = "\xdb\xcb", [0x1f54] = "\xb1\xc6",
  [0x1f5c] = "\xd9\xca", [0x1f73] = "\xb9\xd0", [0x1f74] = "\xdb\xc3",
  [0x1f76] = "\xd8\xd0", [0x1f78] = "\xe0\xa3", [0x1f79] = "\xe0\xa4",
  [0x1f7a] = "\xdf\xfe", [0x1f7c] = "\xdf\xfd", [0x1f7e] = "\xdf\xf4",
  [0x1f83] = "\xe0\xa7", [0x1f88] = "\xc9\xb6", [0x1f90] = "\xe0\xf5",
  [0x1f91] = "\xe0\xf0", [0x1f96] = "\xd2\xb0", [0x1f97] = "\xdb\xfa",
  [0x1f98] = "\xdc\xa5", [0x1f99] = "\xdc\xa4", [0x1f9a] = "\xdc\xa3",
  [0x1f9c] = "\xdb\xfc", [0x1f9d] = "\xdc\xa1", [0x1f9f] = "\xdb\xef",
  [0x1fa0] = "\xdd\xc0", [0x1fa6] = "\xdb\xeb", [0x1fa8] = "\xc6\xe9",
  [0x1fb0] = "\xdc\xa2", [0x1fb5] = "\xe6\xbc", [0x1fb6] = "\xe6\xba",
  [0x1fba] = "\xe6\xab", [0x1fbe] = "\xd2\xf9", [0x1fd8] = "\xe5\xed",
  [0x1fd9] = "\xe1\xc6", [0x1fdd] = "\xe1\xb4", [0x1fe0] = "\xe1\xc1",
  [0x1fe1] = "\xe1\xc3", [0x1fe8] = "\xe1\xc4", [0x1fed] = "\xe2\xd5",
  [0x1fef] = "\xe2\xd8", [0x2000] = "\xe3\xae", [0x2002] = "\xe3\xad",
  [0x2005] = "\xbe\xaa", [0x200a] = "\xea\xfd", [0x200c] = "\xde\xe5",
  [0x200d] = "\xb5\xe0", [0x2011] = "\xde\xdd", [0x2015] = "\xde\xe1",
  [0x2017] = "\xb6\xde", [0x2018] = "\xc6\xfe", [0x2019] = "\xbe\xdd",
  [0x201d] = "\xde\xe3", [0x201e] = "\xde\xe7", [0x202c] = "\xea\xce",
  [0x2030] = "\xce\xfa", [0x2037] = "\xb1\xad", [0x2040] = "\xe8\xf5",
  [0x2042] = "\xe8\xf4", [0x2043] = "\xe8\xf6", [0x2046] = "\xe8\xf8",
  [0x205a] = "\xec\xa4", [0x205b] = "\xb0\xa6", [0x205e] = "\xe9\xe8",
  [0x2061] = "\xeb\xb4", [0x2062] = "\xb5\xed", [0x2063] = "\xe4\xca",
  [0x206b] = "\xe4\xb5", [0x206f] = "\xe4\xc4", [0x2071] = "\xc4\xd7",
  [0x2073] = "\xe4\xcb", [0x2075] = "\xe4\xc7", [0x208b] = "\xcd\xe9",
  [0x208d] = "\xcc\xfe", [0x2092] = "\xec\xc9", [0x2096] = "\xec\xca",
  [0x209d] = "\xea\xf5", [0x20a0] = "\xea\xf6", [0x20a1] = "\xe2\xa7",
  [0x20a2] = "\xe2\xa2", [0x20a6] = "\xe2\xa5", [0x20aa] = "\xe2\xa6",
  [0x20be] = "\xd1\xf7", [0x20c8] = "\xc3\xd0", [0x20c9] = "\xed\xf5",
  [0x20cf] = "\xed\xf6", [0x20d1] = "\xed\xf7", [0x20d3] = "\xce\xf8",
  [0x20d7] = "\xed\xd1", [0x20d8] = "\xed\xce", [0x20d9] = "\xb9\xe8",
  [0x20da] = "\xed\xcf", [0x20dc] = "\xec\xf6", [0x20e2] = "\xc0\xeb",
  [0x20e4] = "\xbd\xd5", [0x20eb] = "\xf3\xcc", [0x20ed] = "\xf3\xd3",
  [0x20f0] = "\xf3\xd4", [0x20f1] = "\xf3\xd5", [0x20f3] = "\xf3\xce",
  [0x20f5] = "\xf3\xd1", [0x20f6] = "\xf3\xca", [0x20fa] = "\xf3\xcd",
  [0x20fe] = "\xd5\xb3", [0x2105] = "\xe7\xa4", [0x210c] = "\xe7\xaa",
  [0x2113] = "\xf4\xc6", [0x2115] = "\xf1\xb4", [0x2121] = "\xf1\xf5",
  [0x2123] = "\xeb\xe4", [0x2126] = "\xeb\xd6", [0x2129] = "\xeb\xe3",
  [0x212a] = "\xeb\xe2", [0x2131] = "\xf4\xb4", [0x2132] = "\xf4\xb6",
  [0x2134] = "\xf4\xb7", [0x2136] = "\xe5\xf5", [0x2139] = "\xdd\xb9",
  [0x213c] = "\xb6\xb9", [0x2145] = "\xdd\xb3", [0x2146] = "\xdd\xb4",
  [0x214a] = "\xdc\xf0", [0x214b] = "\xdd\xad", [0x2152] = "\xdd\xa8",
  [0x215a] = "\xba\xf4", [0x2160] = "\xf2\xc5", [0x2161] = "\xf2\xc4",
  [0x2167] = "\xf2\xca", [0x216d] = "\xec\xc5", [0x2171] = "\xf1\xc8",
  [0x2172] = "\xec\xee", [0x217c] = "\xd9\xf3", [0x2182] = "\xf5\xfb",
  [0x2192] = "\xf5\xc2", [0x2195] = "\xf5\xc1", [0x219f] = "\xe5\xcd",
  [0x21a0] = "\xe5\xcf", [0x21a3] = "\xe5\xd2", [0x21a4] = "\xdb\xb0",
  [0x21a7] = "\xb3\xbb", [0x21ab] = "\xdb\xaf", [0x21ae] = "\xf0\xb2",
  [0x21b0] = "\xb7\xd3", [0x21b3] = "\xc7\xa5", [0x21b8] = "\xee\xcc",
  [0x21b9] = "\xee\xcf", [0x21bc] = "\xee\xca", [0x21c0] = "\xe3\xc6",
  [0x21cb] = "\xd8\xb8", [0x21d1] = "\xb4\xf6", [0x21e1] = "\xe0\xb3",
  [0x21e5] = "\xe0\xa5", [0x21e9] = "\xe0\xae", [0x21ea] = "\xdf\xf6",
  [0x21eb] = "\xdf\xf7", [0x21ec] = "\xe0\xaf", [0x21fa] = "\xdc\xa7",
  [0x21fb] = "\xdc\xa6", [0x2205] = "\xbc\xee", [0x220e] = "\xe6\xa3",
  [0x2211] = "\xe6\xc4", [0x221a] = "\xcd\xb5", [0x2234] = "\xe1\xd1",
  [0x2236] = "\xe1\xd2", [0x223c] = "\xe1\xce", [0x223e] = "\xd1\xd2",
  [0x223f] = "\xe1\xcc", [0x224b] = "\xe1\xa2", [0x224d] = "\xe5\xe9",
  [0x2258] = "\xe3\xa2", [0x226c] = "\xea\xfe", [0x226d] = "\xde\xef",
  [0x2275] = "\xde\xeb", [0x2276] = "\xde\xde", [0x2278] = "\xde\xe9",
  [0x227c] = "\xde\xf2", [0x227f] = "\xde\xed", [0x2291] = "\xb1\xf3",
  [0x2296] = "\xec\xbc", [0x2299] = "\xb0\xb5", [0x22a7] = "\xc0\xe2",
  [0x22a8] = "\xe8\xe2", [0x22a9] = "\xe8\xc7", [0x22af] = "\xe9\xa7",
  [0x22be] = "\xe8\xfb", [0x22c0] = "\xe9\xa3", [0x22d0] = "\xec\xa5",
  [0x22d8] = "\xeb\xc8", [0x22db] = "\xeb\xa5", [0x22dc] = "\xc7\xe8",
  [0x22dd] = "\xed\xb5", [0x22e2] = "\xe3\xed", [0x22e7] = "\xe4\xcd",
  [0x22f4] = "\xe4\xa5", [0x22fc] = "\xe4\xd0", [0x2300] = "\xe4\xd4",
  [0x230f] = "\xec\xcc", [0x2312] = "\xec\xcd", [0x2321] = "\xea\xf8",
  [0x2325] = "\xe2\xa9", [0x2326] = "\xe2\xae", [0x232f] = "\xe7\xfd",
  [0x2330] = "\xe7\xfe", [0x2331] = "\xe7\xfc", [0x2334] = "\xe8\xa2",
  [0x2337] = "\xb5\xf1", [0x233f] = "\xc4\xfe", [0x2341] = "\xee\xb4",
  [0x2342] = "\xf0\xf0", [0x2345] = "\xf0\xed", [0x2348] = "\xf0\xee",
  [0x234e] = "\xed\xfb", [0x2357] = "\xef\xf3", [0x2359] = "\xed\xcc",
  [0x235c] = "\xed\xcb", [0x235e] = "\xed\xd2", [0x235f] = "\xc8\xb7",
  [0x2362] = "\xed\xba", [0x2369] = "\xef\xfc", [0x236b] = "\xef\xfb",
  [0x236f] = "\xf1\xb5", [0x2372] = "\xf3\xcc", [0x2375] = "\xf3\xdc",
  [0x2378] = "\xf3\xd8", [0x2379] = "\xf3\xda", [0x237a] = "\xf4\xd2",
  [0x237b] = "\xf4\xd1", [0x238e] = "\xe7\xac", [0x238f] = "\xc6\xbf",
  [0x2392] = "\xee\xb7", [0x2395] = "\xcf\xdb", [0x239d] = "\xc0\xb0",
  [0x23a9] = "\xf4\xaa", [0x23af] = "\xba\xca", [0x23b0] = "\xdd\xcf",
  [0x23b2] = "\xdd\xd2", [0x23bc] = "\xdd\xc0", [0x23bf] = "\xdd\xbd",
  [0x23c0] = "\xdd\xc3", [0x23c1] = "\xdd\xbe", [0x23c2] = "\xdd\xbf",
  [0x23c4] = "\xdd\xd5", [0x23c7] = "\xdd\xc5", [0x23cb] = "\xdd\xcc",
  [0x23cd] = "\xdd\xc8", [0x23ce] = "\xdd\xc9", [0x23d3] = "\xb9\xbd",
  [0x23d5] = "\xdd\xd0", [0x23e1] = "\xf2\xd5", [0x23ec] = "\xf2\xcb",
  [0x23ef] = "\xf2\xd6", [0x23fa] = "\xf1\xcb", [0x23fb] = "\xf1\xca",
  [0x23ff] = "\xf1\xcc", [0x2401] = "\xea\xe8", [0x2403] = "\xb5\xd6",
  [0x2404] = "\xf5\xfd", [0x2406] = "\xda\xaa", [0x240c] = "\xda\xb0",
  [0x240f] = "\xda\xb1", [0x2410] = "\xee\xba", [0x2418] = "\xea\xdc",
  [0x241a] = "\xea\xdb", [0x241d] = "\xf4\xf2", [0x2424] = "\xf5\xc5",
  [0x2426] = "\xf5\xca", [0x242a] = "\xf5\xc6", [0x242e] = "\xe9\xf7",
  [0x242f] = "\xe9\xf2", [0x2437] = "\xe9\xf4", [0x243c] = "\xe5\xd5",
  [0x243e] = "\xe5\xd6", [0x243f] = "\xdb\xa9", [0x2441] = "\xdb\xb2",
  [0x244c] = "\xf5\xa2", [0x244d] = "\xf4\xfe", [0x244f] = "\xf5\xa1",
  [0x2451] = "\xee\xd5", [0x2453] = "\xee\xd8", [0x2456] = "\xee\xd1",
  [0x2459] = "\xee\xd9", [0x245e] = "\xee\xd7", [0x245f] = "\xee\xd6",
  [0x2469] = "\xe3\xca", [0x246d] = "\xda\xf1", [0x246e] = "\xda\xed",
  [0x2479] = "\xf1\xfc", [0x247b] = "\xe2\xc0", [0x247d] = "\xed\xe9",
  [0x2486] = "\xcf\xc9", [0x2487] = "\xd8\xf1", [0x2489] = "\xd9\xcd",
  [0x248c] = "\xd9\xd1", [0x248f] = "\xd9\xdd", [0x2499] = "\xe0\xc9",
  [0x249d] = "\xe0\xc3", [0x249f] = "\xe0\xc1", [0x24a0] = "\xe0\xc4",
  [0x24a1] = "\xdf\xef", [0x24a3] = "\xe0\xaa", [0x24a8] = "\xe0\xb2",
  [0x24aa] = "\xe0\xc7", [0x24ab] = "\xe0\xca", [0x24b1] = "\xdb\xf7",
  [0x24b2] = "\xdb\xee", [0x24b3] = "\xeb\xf3", [0x24ba] = "\xdc\xaa",
  [0x24c2] = "\xf4\xc1", [0x24c4] = "\xe6\xca", [0x24c6] = "\xeb\xf4",
  [0x24c8] = "\xc0\xa2", [0x24d5] = "\xd6\xc3", [0x24db] = "\xe1\xd3",
  [0x24de] = "\xe1\xcd", [0x24e3] = "\xdb\xcf", [0x24ee] = "\xec\xb0",
  [0x24f0] = "\xe1\xe6", [0x24f1] = "\xb4\xc0", [0x24f3] = "\xe3\xbb",
  [0x24f4] = "\xe3\xba", [0x2501] = "\xea\xad", [0x2502] = "\xea\xae",
  [0x2506] = "\xb4\xa4", [0x2509] = "\xde\xf7", [0x250c] = "\xbf\xb8",
  [0x250d] = "\xdf\xa6", [0x2514] = "\xde\xf8", [0x2515] = "\xde\xf9",
  [0x2517] = "\xb4\xf2", [0x2518] = "\xde\xf4", [0x251a] = "\xcc\xcd",
  [0x251d] = "\xce\xe6", [0x251e] = "\xb4\xb7", [0x2520] = "\xde\xf5",
  [0x2522] = "\xde\xf6", [0x2524] = "\xde\xfa", [0x252a] = "\xea\xd2",
  [0x2533] = "\xe9\xb8", [0x2538] = "\xe9\xa8", [0x2539] = "\xb4\xbb",
  [0x253c] = "\xe9\xa9", [0x253d] = "\xe9\xab", [0x253e] = "\xe9\xa5",
  [0x253f] = "\xc3\xaf", [0x2546] = "\xb4\xaa", [0x2548] = "\xe9\xa2",
  [0x2549] = "\xe9\xb1", [0x254a] = "\xe9\xb2", [0x254e] = "\xd8\xd2",
  [0x2557] = "\xba\xfa", [0x255f] = "\xec\xa7", [0x2561] = "\xec\xa6",
  [0x2565] = "\xe9\xea", [0x2569] = "\xeb\xa8", [0x256f] = "\xe4\xe7",
  [0x2571] = "\xe4\xe9", [0x2575] = "\xe4\xda", [0x2579] = "\xe4\xe1",
  [0x257a] = "\xb3\xfc", [0x257d] = "\xe4\xe3", [0x2581] = "\xe4\xe6",
  [0x2582] = "\xe4\xd1", [0x2596] = "\xec\xbf", [0x259b] = "\xdc\xe4",
  [0x259c] = "\xec\xd2", [0x259d] = "\xec\xd4", [0x25ac] = "\xea\xf9",
  [0x25b3] = "\xe1\xf8", [0x25ba] = "\xe7\xe2", [0x25bd] = "\xe8\xa5",
  [0x25c7] = "\xea\xb3", [0x25cb] = "\xee\xb5", [0x25d0] = "\xf0\xf1",
  [0x25d2] = "\xf0\xe2", [0x25d3] = "\xf0\xf3", [0x25d4] = "\xb1\xd4",
  [0x25d6] = "\xf0\xf5", [0x25dc] = "\xf0\xaa", [0x25e4] = "\xed\xfd",
  [0x25eb] = "\xed\xd6", [0x25ec] = "\xed\xd5", [0x25f0] = "\xc6\xe9",
  [0x25f3] = "\xed\xd8", [0x25fa] = "\xc2\xe3", [0x2602] = "\xb0\xde",
  [0x260f] = "\xb2\xdf", [0x2611] = "\xf3\xe2", [0x2614] = "\xf3\xe3",
  [0x261a] = "\xf4\xd3", [0x261d] = "\xe7\xb0", [0x261e] = "\xe7\xae",
  [0x2626] = "\xe7\xaf", [0x2630] = "\xee\xbb", [0x2633] = "\xf4\xc7",
  [0x2634] = "\xf4\xc8", [0x2637] = "\xd6\xfa", [0x2639] = "\xeb\xed",
  [0x263c] = "\xeb\xee", [0x2643] = "\xeb\xf2", [0x2646] = "\xeb\xe1",
  [0x2648] = "\xf4\xba", [0x2649] = "\xf4\xb9", [0x2650] = "\xdd\xe3",
  [0x2655] = "\xdd\xd7", [0x265c] = "\xdd\xd8", [0x265d] = "\xdd\xd9",
  [0x265f] = "\xdd\xda", [0x2666] = "\xdd\xdd", [0x2668] = "\xdd\xdf",
  [0x2676] = "\xdd\xa6", [0x2677] = "\xd2\xa9", [0x2679] = "\xdd\xe5",
  [0x268a] = "\xf2\xcc", [0x268e] = "\xf2\xd9", [0x2690] = "\xf2\xdd",
  [0x2693] = "\xf2\xdc", [0x2698] = "\xf1\xce", [0x26a1] = "\xf6\xa1",
  [0x26a6] = "\xf5\xfe", [0x26a7] = "\xb4\xa5", [0x26a8] = "\xb3\xea",
  [0x26a9] = "\xda\xb2", [0x26aa] = "\xda\xb4", [0x26ab] = "\xda\xbc",
  [0x26af] = "\xda\xb3", [0x26b0] = "\xda\xb7", [0x26bd] = "\xf5\xf7",
  [0x26c1] = "\xf4\xf4", [0x26c7] = "\xf4\xf3", [0x26cc] = "\xf5\xcd",
  [0x26d1] = "\xf5\xd0", [0x26d5] = "\xf5\xbc", [0x26db] = "\xe9\xfb",
  [0x26dc] = "\xe9\xfa", [0x26e0] = "\xe5\xd9", [0x26e2] = "\xe5\xd7",
  [0x26e9] = "\xd4\xc7", [0x26ea] = "\xda\xf9", [0x26ed] = "\xcd\xaa",
  [0x26ee] = "\xf5\xa5", [0x26ef] = "\xee\xe8", [0x26f1] = "\xee\xe6",
  [0x26f2] = "\xee\xda", [0x26f3] = "\xee\xdb", [0x26f4] = "\xee\xdd",
  [0x26f6] = "\xee\xe1", [0x26f7] = "\xef\xa5", [0x26f8] = "\xee\xea",
  [0x26fa] = "\xd7\xea", [0x26fb] = "\xee\xe3", [0x26fc] = "\xee\xe2",
  [0x2715] = "\xda\xf3", [0x2716] = "\xf6\xc2", [0x271b] = "\xbd\xf9",
  [0x271f] = "\xf1\xfe", [0x2721] = "\xf1\xfd", [0x272e] = "\xd9\xec",
  [0x2730] = "\xf6\xbc", [0x2731] = "\xd9\xd6", [0x2734] = "\xd9\xc7",
  [0x2735] = "\xcb\xdb", [0x273b] = "\xd9\xd5", [0x273e] = "\xd9\xd4",
  [0x2742] = "\xd8\xe4", [0x2747] = "\xe0\xd7", [0x2749] = "\xe0\xd1",
  [0x274c] = "\xd8\xc5", [0x274d] = "\xdf\xe9", [0x274e] = "\xe0\xd2",
  [0x2756] = "\xdc\xad", [0x275b] = "\xdc\xac", [0x2761] = "\xc9\xca",
  [0x276b] = "\xe6\xd1", [0x2772] = "\xe6\xc6", [0x2775] = "\xe6\xcb",
  [0x2780] = "\xe5\xef", [0x2781] = "\xe1\xd6", [0x2786] = "\xe1\xd0",
  [0x2790] = "\xe0\xfd", [0x2794] = "\xe2\xdb", [0x279a] = "\xe2\xda",
  [0x27a1] = "\xed\xa8", [0x27a5] = "\xe3\xa5", [0x27b1] = "\xe2\xe6",
  [0x27b4] = "\xea\xaf", [0x27b5] = "\xea\xa8", [0x27bc] = "\xde\xd2",
  [0x27be] = "\xbf\xd9", [0x27c2] = "\xb3\xb6", [0x27c4] = "\xc1\xcc",
  [0x27c5] = "\xde\xfb", [0x27c6] = "\xde\xe8", [0x27cc] = "\xeb\xa1",
  [0x27d9] = "\xcd\xfb", [0x27da] = "\xe9\xc1", [0x27de] = "\xe9\xb6",
  [0x27e6] = "\xe9\xbc", [0x27e7] = "\xe9\xc7", [0x27eb] = "\xe8\xe7",
  [0x27ef] = "\xe9\xc0", [0x27f1] = "\xe9\xc3", [0x27f6] = "\xe8\xbf",
  [0x2803] = "\xe9\xe6", [0x2809] = "\xdc\xfe", [0x280e] = "\xe4\xb0",
  [0x2812] = "\xe4\xf5", [0x2815] = "\xc5\xbd", [0x281c] = "\xe4\xf1",
  [0x281e] = "\xe4\xef", [0x2820] = "\xe4\xdd", [0x2824] = "\xe4\xd3",
  [0x283b] = "\xd1\xac", [0x283f] = "\xec\xc1", [0x2845] = "\xb4\xf4",
  [0x2846] = "\xe2\xb0", [0x2857] = "\xf0\xf8", [0x2858] = "\xf0\xfd",
  [0x285a] = "\xf0\xfa", [0x285c] = "\xf1\xe4", [0x285f] = "\xb3\xf2",
  [0x2862] = "\xee\xa6", [0x2864] = "\xd8\xba", [0x2866] = "\xed\xda",
  [0x2867] = "\xd5\xe8", [0x2868] = "\xb2\xea", [0x2869] = "\xed\xb8",
  [0x286e] = "\xed\xdc", [0x2874] = "\xec\xf9", [0x2882] = "\xb1\xe2",
  [0x2884] = "\xf1\xbf", [0x2886] = "\xf1\xbe", [0x2889] = "\xf3\xed",
  [0x288c] = "\xf3\xe4", [0x288e] = "\xb9\xbf", [0x2892] = "\xf3\xeb",
  [0x2894] = "\xd4\xfd", [0x289a] = "\xf4\xd4", [0x289f] = "\xe7\xb9",
  [0x28a6] = "\xe7\xb5", [0x28a8] = "\xe7\xb3", [0x28ab] = "\xe7\xb8",
  [0x28af] = "\xf4\xeb", [0x28b0] = "\xf4\xec", [0x28b7] = "\xf4\xe3",
  [0x28b9] = "\xbd\xe5", [0x28bd] = "\xe0\xbc", [0x28be] = "\xeb\xf0",
  [0x28c4] = "\xdd\xf5", [0x28c6] = "\xdd\xf2", [0x28c8] = "\xdd\xf0",
  [0x28ce] = "\xdd\xf3", [0x28cf] = "\xdd\xf4", [0x28d0] = "\xdd\xe8",
  [0x28d1] = "\xdd\xe9", [0x28d5] = "\xdd\xea", [0x28da] = "\xdd\xaa",
  [0x28df] = "\xd8\xe1", [0x28e1] = "\xb1\xcd", [0x28ee] = "\xf2\xde",
  [0x28f2] = "\xf2\xe2", [0x28f3] = "\xf2\xe0", [0x28f4] = "\xc0\xaf",
  [0x28fa] = "\xf2\xe4", [0x2901] = "\xd1\xd1", [0x2902] = "\xc4\xde",
  [0x2903] = "\xf2\xe7", [0x290a] = "\xf2\xe3", [0x290f] = "\xf1\xd1",
  [0x2912] = "\xf1\xd5", [0x2914] = "\xf1\xd3", [0x2917] = "\xf1\xd6",
  [0x291b] = "\xea\xea", [0x291f] = "\xf6\xa2", [0x2924] = "\xda\xc0",
  [0x292a] = "\xea\xe4", [0x292e] = "\xf5\xd4", [0x2932] = "\xf5\xd5",
  [0x2933] = "\xd3\xbb", [0x2937] = "\xf5\xbd", [0x293f] = "\xdb\xb5",
  [0x2942] = "\xdb\xb3", [0x294a] = "\xf5\xa8", [0x294b] = "\xf5\xaa",
  [0x294d] = "\xd2\xbf", [0x2951] = "\xee\xef", [0x2952] = "\xee\xee",
  [0x2954] = "\xee\xf0", [0x2956] = "\xee\xf7", [0x2957] = "\xef\xa2",
  [0x2958] = "\xef\xa4", [0x295b] = "\xee\xfe", [0x295c] = "\xef\xa8",
  [0x295d] = "\xee\xf1", [0x295e] = "\xf6\xc6", [0x296c] = "\xee\xfb",
  [0x2982] = "\xec\xa9", [0x298f] = "\xf7\xba", [0x2990] = "\xb7\xc2",
  [0x299f] = "\xd9\xd3", [0x29a0] = "\xd9\xd8", [0x29a3] = "\xd9\xd9",
  [0x29a8] = "\xd8\xdb", [0x29a9] = "\xdb\xbd", [0x29ad] = "\xe0\xe1",
  [0x29ae] = "\xdf\xd8", [0x29af] = "\xb6\xf1", [0x29b3] = "\xe0\xd9",
  [0x29b8] = "\xe0\xdc", [0x29ba] = "\xdf\xbc", [0x29be] = "\xe9\xd7",
  [0x29d6] = "\xe1\xd7", [0x29d7] = "\xe1\xc0", [0x29dd] = "\xe1\xbd",
  [0x29e7] = "\xe2\xdc", [0x29e9] = "\xe2\xd0", [0x29ec] = "\xb1\xef",
  [0x29f8] = "\xe3\xb4", [0x2a01] = "\xde\xfe", [0x2a03] = "\xbe\xef",
  [0x2a0a] = "\xb5\xa7", [0x2a0b] = "\xde\xd8", [0x2a1e] = "\xe9\xcb",
  [0x2a25] = "\xe9\xc8", [0x2a2c] = "\xe9\xce", [0x2a30] = "\xe8\xfd",
  [0x2a3a] = "\xe9\xcc", [0x2a49] = "\xeb\xa7", [0x2a4b] = "\xf2\xa3",
  [0x2a4e] = "\xc0\xd4", [0x2a51] = "\xe4\xf8", [0x2a52] = "\xe4\xf7",
  [0x2a53] = "\xe4\xf9", [0x2a54] = "\xe4\xea", [0x2a5b] = "\xb3\xce",
  [0x2a5d] = "\xe4\xfb", [0x2a61] = "\xba\xc6", [0x2a6d] = "\xe4\xb6",
  [0x2a70] = "\xe4\xe4", [0x2a77] = "\xec\xda", [0x2a7a] = "\xec\xd8",
  [0x2a83] = "\xec\xd7", [0x2a87] = "\xe9\xe1", [0x2a8a] = "\xe2\xb2",
  [0x2a91] = "\xe8\xaf", [0x2a92] = "\xe7\xf6", [0x2a95] = "\xe8\xae",
  [0x2a9e] = "\xf0\xfb", [0x2a9f] = "\xf0\xf9", [0x2aa0] = "\xf0\xfe",
  [0x2aa6] = "\xf1\xa1", [0x2aab] = "\xee\xa4", [0x2ab5] = "\xed\xdd",
  [0x2ab9] = "\xed\xdf", [0x2aba] = "\xec\xfa", [0x2ac1] = "\xf0\xa1",
  [0x2ac4] = "\xf1\xc1", [0x2ac6] = "\xf3\xe6", [0x2ac8] = "\xf3\xe8",
  [0x2ace] = "\xf4\xdb", [0x2acf] = "\xf4\xda", [0x2ad0] = "\xf4\xd8",
  [0x2ad6] = "\xe7\xbd", [0x2ad7] = "\xe7\xc5", [0x2ada] = "\xe7\xc1",
  [0x2adb] = "\xe7\xc2", [0x2adc] = "\xe7\xc3", [0x2ae2] = "\xf4\xca",
  [0x2ae8] = "\xf4\xe5", [0x2aec] = "\xeb\xf9", [0x2af3] = "\xc6\xcc",
  [0x2af4] = "\xf4\xbc", [0x2afb] = "\xde\xa2", [0x2b02] = "\xdd\xfd",
  [0x2b07] = "\xc4\xe8", [0x2b0a] = "\xdd\xf8", [0x2b0b] = "\xdd\xbb",
  [0x2b12] = "\xde\xa4", [0x2b1c] = "\xdc\xea", [0x2b1d] = "\xdd\xe4",
  [0x2b20] = "\xdc\xe0", [0x2b23] = "\xdd\xfb", [0x2b24] = "\xdc\xca",
  [0x2b2c] = "\xeb\xbd", [0x2b2e] = "\xf2\xf6", [0x2b2f] = "\xf2\xf8",
  [0x2b3e] = "\xd0\xab", [0x2b3f] = "\xe2\xac", [0x2b44] = "\xf2\xf3",
  [0x2b46] = "\xf2\xfa", [0x2b48] = "\xf2\xef", [0x2b4d] = "\xba\xfa",
  [0x2b54] = "\xf1\xd8", [0x2b58] = "\xd0\xe4", [0x2b60] = "\xda\xc1",
  [0x2b64] = "\xda\xc2", [0x2b68] = "\xda\xc5", [0x2b76] = "\xd6\xdc",
  [0x2b78] = "\xea\xe3", [0x2b7a] = "\xea\xe6", [0x2b80] = "\xf5\xdb",
  [0x2b83] = "\xf5\xda", [0x2b8b] = "\xf5\xd6", [0x2b9f] = "\xdb\xb7",
  [0x2ba3] = "\xb5\xa6", [0x2ba6] = "\xf5\xac", [0x2ba7] = "\xd5\xb5",
  [0x2bae] = "\xef\xb6", [0x2bb3] = "\xee\xf2", [0x2bb4] = "\xef\xab",
  [0x2bb5] = "\xef\xb7", [0x2bb9] = "\xef\xb2", [0x2bba] = "\xee\xfa",
  [0x2bbb] = "\xef\xaf", [0x2bbd] = "\xef\xb0", [0x2bbf] = "\xf6\xc8",
  [0x2bc1] = "\xef\xb8", [0x2bcf] = "\xe3\xcf", [0x2bd0] = "\xe3\xcd",
  [0x2bd3] = "\xcd\xc7", [0x2bd7] = "\xf6\xac", [0x2bd9] = "\xf6\xa6",
  [0x2be1] = "\xf2\xa4", [0x2be8] = "\xf4\xd9", [0x2bea] = "\xe2\xc4",
  [0x2beb] = "\xb2\xb8", [0x2bf3] = "\xe6\xe0", [0x2bf7] = "\xe6\xe6",
  [0x2bfd] = "\xf7\xd8", [0x2c02] = "\xf7\xc9", [0x2c04] = "\xf6\xd0",
  [0x2c0a] = "\xeb\xe0", [0x2c12] = "\xf0\xb1", [0x2c16] = "\xd1\xe3",
  [0x2c1c] = "\xd8\xbe", [0x2c24] = "\xee\xa3", [0x2c26] = "\xdf\xd5",
  [0x2c29] = "\xdf\xdc", [0x2c2c] = "\xdf\xe0", [0x2c2f] = "\xe0\xf7",
  [0x2c37] = "\xdb\xd4", [0x2c39] = "\xe6\xd3", [0x2c3a] = "\xe6\xcd",
  [0x2c3c] = "\xe6\xc8", [0x2c40] = "\xe6\xd4", [0x2c49] = "\xe1\xbb",
  [0x2c57] = "\xe2\xde", [0x2c5a] = "\xe2\xdd", [0x2c5d] = "\xe1\xe8",
  [0x2c5e] = "\xed\xad", [0x2c5f] = "\xba\xa9", [0x2c65] = "\xe2\xf8",
  [0x2c6a] = "\xdf\xa8", [0x2c6c] = "\xdf\xa7", [0x2c7a] = "\xea\xd5",
  [0x2c7e] = "\xcf\xf2", [0x2c85] = "\xb4\xb1", [0x2c8a] = "\xe9\xd0",
  [0x2c8e] = "\xe9\xd3", [0x2c90] = "\xe9\xd8", [0x2c97] = "\xe9\xd2",
  [0x2ca8] = "\xe9\xec", [0x2ca9] = "\xe9\xe9", [0x2cae] = "\xeb\xab",
  [0x2cb6] = "\xc2\xba", [0x2cbe] = "\xe4\xab", [0x2cc6] = "\xe5\xa1",
  [0x2cd2] = "\xec\xc7", [0x2cd4] = "\xec\xdc", [0x2ce3] = "\xe2\xb3",
  [0x2ce5] = "\xe1\xfd", [0x2ce6] = "\xe1\xf6", [0x2cf1] = "\xf1\xa6",
  [0x2cf4] = "\xf1\xac", [0x2cf5] = "\xf1\xa2", [0x2cf6] = "\xf1\xa9",
  [0x2cf8] = "\xf1\xa7", [0x2d00] = "\xde\xab", [0x2d0a] = "\xed\xd7",
  [0x2d15] = "\xf1\xc0", [0x2d1d] = "\xf3\xf4", [0x2d1f] = "\xf3\xf6",
  [0x2d20] = "\xf3\xf5", [0x2d24] = "\xf3\xf8", [0x2d2c] = "\xf4\xdc",
  [0x2d33] = "\xe7\xc8", [0x2d39] = "\xcc\xd0", [0x2d3b] = "\xe7\xc4",
  [0x2d48] = "\xc5\xd5", [0x2d49] = "\xf1\xf0", [0x2d4c] = "\xec\xa2",
  [0x2d5b] = "\xde\xa1", [0x2d5f] = "\xdc\xbf", [0x2d61] = "\xdc\xe9",
  [0x2d63] = "\xdd\xdb", [0x2d64] = "\xde\xa8", [0x2d65] = "\xdd\xa1",
  [0x2d66] = "\xdd\xde", [0x2d6e] = "\xdc\xf1", [0x2d71] = "\xdd\xb5",
  [0x2d74] = "\xdc\xbc", [0x2d7f] = "\xf3\xa5", [0x2d80] = "\xf2\xfb",
  [0x2d82] = "\xf3\xa2", [0x2d89] = "\xf3\xa3", [0x2d8d] = "\xf2\xcf",
  [0x2d94] = "\xe5\xbd", [0x2da0] = "\xd0\xfa", [0x2da1] = "\xda\xbb",
  [0x2da6] = "\xda\xca", [0x2da7] = "\xda\xcc", [0x2dab] = "\xda\xd2",
  [0x2dc0] = "\xf5\xde", [0x2dc4] = "\xf5\xe1", [0x2dc5] = "\xf5\xe4",
  [0x2dc6] = "\xd3\xe2", [0x2dd7] = "\xdb\xa6", [0x2dd9] = "\xf5\xad",
  [0x2dda] = "\xf5\xaf", [0x2ddb] = "\xf5\xae", [0x2ddf] = "\xef\xc3",
  [0x2de0] = "\xef\xc4", [0x2de1] = "\xef\xba", [0x2de3] = "\xd5\xe0",
  [0x2de4] = "\xef\xaa", [0x2de6] = "\xef\xbc", [0x2de8] = "\xef\xb9",
  [0x2de9] = "\xef\xbe", [0x2dea] = "\xee\xcd", [0x2e03] = "\xbc\xf8",
  [0x2e05] = "\xe3\xd5", [0x2e07] = "\xe3\xd0", [0x2e08] = "\xd1\xcb",
  [0x2e0a] = "\xe3\xd1", [0x2e0b] = "\xe3\xd3", [0x2e0d] = "\xe3\xd4",
  [0x2e15] = "\xf7\xb4", [0x2e20] = "\xb2\xb5", [0x2e2f] = "\xf7\xdb",
  [0x2e3a] = "\xf6\xd6", [0x2e3b] = "\xf6\xd8", [0x2e46] = "\xf0\xb6",
  [0x2e4b] = "\xf0\xb7", [0x2e4c] = "\xf7\xe6", [0x2e4e] = "\xf7\xe5",
  [0x2e55] = "\xf7\xfc", [0x2e5d] = "\xe0\xea", [0x2e5e] = "\xdf\xe2",
  [0x2e60] = "\xe0\xe3", [0x2e6b] = "\xc4\xcc", [0x2e6d] = "\xe6\xd5",
  [0x2e77] = "\xe1\xda", [0x2e78] = "\xe0\xfc", [0x2e7c] = "\xc7\xda",
  [0x2e83] = "\xe2\xfb", [0x2e84] = "\xc3\xc9", [0x2e85] = "\xb1\xf7",
  [0x2e87] = "\xb5\xb7", [0x2e89] = "\xdf\xa9", [0x2e92] = "\xe9\xdd",
  [0x2e94] = "\xe8\xdf", [0x2e9b] = "\xe9\xd5", [0x2ea0] = "\xe9\xdb",
  [0x2ea8] = "\xbd\xa9", [0x2eab] = "\xed\xb4", [0x2eb3] = "\xe5\xa8",
  [0x2ec1] = "\xe2\xb4", [0x2ec5] = "\xe8\xb4", [0x2ecc] = "\xea\xb5",
  [0x2ece] = "\xea\xb4", [0x2ed0] = "\xf1\xaa", [0x2ed2] = "\xf0\xf7",
  [0x2ed3] = "\xf0\xef", [0x2ed4] = "\xf0\xab", [0x2ed6] = "\xee\xac",
  [0x2ee0] = "\xed\xcd", [0x2ee4] = "\xed\xe1", [0x2ee5] = "\xed\xe2",
  [0x2ef5] = "\xf3\xfc", [0x2ef6] = "\xe5\xe7", [0x2ef7] = "\xf3\xe5",
  [0x2efc] = "\xf3\xfe", [0x2efd] = "\xf3\xd9", [0x2f06] = "\xc0\xe9",
  [0x2f09] = "\xf4\xdd", [0x2f0a] = "\xe7\xca", [0x2f0d] = "\xb8\xbf",
  [0x2f13] = "\xf4\xed", [0x2f17] = "\xf7\xe3", [0x2f18] = "\xf4\xea",
  [0x2f1b] = "\xf3\xc1", [0x2f1d] = "\xee\xc0", [0x2f21] = "\xf1\xef",
  [0x2f22] = "\xeb\xfe", [0x2f24] = "\xeb\xfb", [0x2f25] = "\xeb\xfd",
  [0x2f2b] = "\xf4\xbd", [0x2f2f] = "\xde\xb2", [0x2f35] = "\xde\xae",
  [0x2f36] = "\xde\xaf", [0x2f39] = "\xdd\xf7", [0x2f3d] = "\xde\xaa",
  [0x2f44] = "\xcc\xea", [0x2f49] = "\xdc\xf6", [0x2f4a] = "\xde\xb6",
  [0x2f4b] = "\xde\xb3", [0x2f4f] = "\xdd\xb2", [0x2f53] = "\xf3\xa4",
  [0x2f56] = "\xf3\xa9", [0x2f58] = "\xf3\xaa", [0x2f60] = "\xf2\xfc",
  [0x2f61] = "\xd5\xdd", [0x2f62] = "\xf3\xb1", [0x2f67] = "\xf3\xae",
  [0x2f6b] = "\xf1\xcd", [0x2f6e] = "\xf1\xdf", [0x2f73] = "\xea\xed",
  [0x2f76] = "\xec\xb2", [0x2f79] = "\xda\xd5", [0x2f7b] = "\xd6\xdf",
  [0x2f81] = "\xe5\xc0", [0x2f87] = "\xda\xd6", [0x2f8e] = "\xf5\xf9",
  [0x2f95] = "\xf5\xc4", [0x2f96] = "\xe5\xbf", [0x2f9c] = "\xc3\xd1",
  [0x2f9d] = "\xf5\xb0", [0x2f9e] = "\xc8\xa9", [0x2fa7] = "\xef\xca",
  [0x2fa9] = "\xef\xc7", [0x2faa] = "\xd5\xeb", [0x2fab] = "\xd5\xa1",
  [0x2fad] = "\xef\xc8", [0x2fb3] = "\xf6\xcc", [0x2fb5] = "\xf6\xca",
  [0x2fc5] = "\xb0\xb5", [0x2fc8] = "\xe3\xd6", [0x2fcc] = "\xda\xf4",
  [0x2fe3] = "\xba\xfd", [0x2fe5] = "\xe2\xbc", [0x2fe9] = "\xf4\xd7",
  [0x2feb] = "\xd9\xe5", [0x2ff9] = "\xb4\xf4", [0x2fff] = "\xf7\xcc",
  [0x3000] = "\xf6\xda", [0x3002] = "\xf6\xdc", [0x3011] = "\xf0\xb9",
  [0x3014] = "\xf0\xbb", [0x3016] = "\xf0\xbc", [0x3024] = "\xed\xea",
  [0x3025] = "\xf6\xbd", [0x3028] = "\xf7\xf7", [0x3029] = "\xf6\xb3",
  [0x302a] = "\xd9\xdf", [0x3033] = "\xc4\xf6", [0x303c] = "\xed\xa1",
  [0x3046] = "\xde\xf3", [0x304b] = "\xdf\xa3", [0x304d] = "\xe1\xa6",
  [0x304f] = "\xea\xd6", [0x3062] = "\xd1\xfa", [0x3063] = "\xe4\xde",
  [0x3069] = "\xe4\xf3", [0x306a] = "\xe3\xf8", [0x306e] = "\xd2\xab",
  [0x306f] = "\xec\xde", [0x307c] = "\xea\xb6", [0x307d] = "\xf1\xb0",
  [0x307e] = "\xf0\xdc", [0x308b] = "\xed\xe4", [0x3098] = "\xf4\xa1",
  [0x309b] = "\xf4\xa3", [0x30a5] = "\xe7\xc0", [0x30af] = "\xf1\xf9",
  [0x30b3] = "\xf4\xbe", [0x30c1] = "\xdd\xa3", [0x30c9] = "\xde\xb8",
  [0x30cf] = "\xf3\xb2", [0x30d1] = "\xf3\xb3", [0x30d2] = "\xf3\xa8",
  [0x30d8] = "\xf3\xad", [0x30db] = "\xf2\xb1", [0x30ea] = "\xf1\xd0",
  [0x30ee] = "\xda\xa9", [0x30f3] = "\xba\xf4", [0x30f4] = "\xc3\xa1",
  [0x30f8] = "\xf6\xa5", [0x3102] = "\xf5\xf8", [0x3104] = "\xd8\xd3",
  [0x3105] = "\xea\xde", [0x310a] = "\xf5\xc5", [0x310f] = "\xd4\xe3",
  [0x3114] = "\xf5\xbf", [0x3116] = "\xea\xa4", [0x311b] = "\xda\xf7",
  [0x3121] = "\xf5\xb5", [0x3122] = "\xf5\xb2", [0x3123] = "\xef\xd8",
  [0x3124] = "\xc1\xad", [0x312a] = "\xee\xf8", [0x312c] = "\xef\xcb",
  [0x312e] = "\xef\xd6", [0x313a] = "\xdd\xf6", [0x313f] = "\xe3\xc4",
  [0x314c] = "\xf7\xb6", [0x3152] = "\xe8\xb8", [0x315b] = "\xe2\xbe",
  [0x315d] = "\xe6\xeb", [0x3160] = "\xe6\xec", [0x3164] = "\xe6\xed",
  [0x3167] = "\xf7\xc2", [0x3169] = "\xf7\xdc", [0x316c] = "\xe3\xd2",
  [0x3171] = "\xf6\xe9", [0x3175] = "\xf6\xe1", [0x317d] = "\xf0\xc3",
  [0x317e] = "\xf0\xbe", [0x3187] = "\xf0\xc1", [0x318e] = "\xf7\xf0",
  [0x319d] = "\xd8\xc9", [0x31a0] = "\xdf\xbf", [0x31a4] = "\xdb\xe4",
  [0x31b2] = "\xdf\xab", [0x31b9] = "\xe9\xc6", [0x31bf] = "\xe8\xdd",
  [0x31c1] = "\xc0\xdb", [0x31c2] = "\xe9\xcd", [0x31c5] = "\xe9\xda",
  [0x31c8] = "\xeb\xaa", [0x31ca] = "\xe3\xf1", [0x31cb] = "\xe4\xeb",
  [0x31d0] = "\xe5\xac", [0x31ec] = "\xed\xe7", [0x31ef] = "\xed\xe6",
  [0x31f0] = "\xec\xf2", [0x31f2] = "\xcd\xc7", [0x31fd] = "\xe7\xd8",
  [0x31ff] = "\xe7\xd9", [0x3207] = "\xee\xbc", [0x320d] = "\xb1\xec",
  [0x320e] = "\xf4\xaf", [0x3210] = "\xe9\xd6", [0x3217] = "\xdd\xa4",
  [0x321f] = "\xde\xbc", [0x3227] = "\xf3\xb5", [0x3228] = "\xd9\xf9",
  [0x3229] = "\xf2\xc9", [0x322e] = "\xf2\xb2", [0x3232] = "\xf3\xb9",
  [0x323b] = "\xf1\xcf", [0x323d] = "\xba\xcb", [0x323e] = "\xea\xef",
  [0x3240] = "\xf6\xa3", [0x3246] = "\xda\xda", [0x3247] = "\xd7\xab",
  [0x3259] = "\xb2\xe4", [0x325d] = "\xf5\xec", [0x3269] = "\xf5\xb3",
  [0x326a] = "\xef\xde", [0x326b] = "\xef\xe0", [0x3270] = "\xef\xea",
  [0x3272] = "\xef\xd2", [0x3274] = "\xef\xa1", [0x3276] = "\xf6\xcb",
  [0x328d] = "\xe8\xb9", [0x3290] = "\xf2\xaa", [0x3297] = "\xe2\xca",
  [0x3299] = "\xe2\xc9", [0x32a5] = "\xd7\xd7", [0x32a9] = "\xf7\xc4",
  [0x32af] = "\xf6\xec", [0x32b0] = "\xf6\xed", [0x32b3] = "\xf6\xef",
  [0x32b5] = "\xf6\xf2", [0x32b6] = "\xf6\xd3", [0x32b7] = "\xf6\xf6",
  [0x32b8] = "\xf6\xdf", [0x32be] = "\xf6\xee", [0x32cf] = "\xb5\xf1",
  [0x32d5] = "\xf0\xc7", [0x32d8] = "\xf0\xb4", [0x32dd] = "\xed\xeb",
  [0x32e8] = "\xe0\xb7", [0x32f3] = "\xde\xfc", [0x32fa] = "\xe8\xd0",
  [0x32fc] = "\xe8\xc0", [0x32fd] = "\xe8\xd3", [0x3303] = "\xe5\xaf",
  [0x330c] = "\xe5\xae", [0x3315] = "\xe2\xa8", [0x3316] = "\xe8\xb7",
  [0x331e] = "\xdb\xc7", [0x3324] = "\xed\xe5", [0x3332] = "\xe7\xd7",
  [0x3337] = "\xf1\xf7", [0x3339] = "\xeb\xd9", [0x333a] = "\xd0\xc6",
  [0x333b] = "\xf4\xbf", [0x333d] = "\xdc\xd7", [0x333e] = "\xde\xbd",
  [0x3343] = "\xdc\xc2", [0x3344] = "\xde\xad", [0x3346] = "\xde\xbf",
  [0x334a] = "\xf2\xd3", [0x334b] = "\xf2\xee", [0x334d] = "\xf3\xb7",
  [0x334e] = "\xf3\xb6", [0x3350] = "\xf1\xe0", [0x335a] = "\xbb\xd9",
  [0x3366] = "\xdb\xb9", [0x3369] = "\xf5\xb6", [0x336c] = "\xef\xa6",
  [0x336d] = "\xef\xe6", [0x336f] = "\xef\xe8", [0x3371] = "\xef\xe2",
  [0x3374] = "\xc1\xcd", [0x3375] = "\xef\xd0", [0x3376] = "\xef\xeb",
  [0x3378] = "\xee\xfc", [0x337a] = "\xef\xe4", [0x337e] = "\xef\xa9",
  [0x3380] = "\xcf\xb3", [0x3389] = "\xe3\xdb", [0x3393] = "\xf2\xa9",
  [0x339b] = "\xe2\xcd", [0x339c] = "\xc0\xa1", [0x33a3] = "\xe6\xe3",
  [0x33a5] = "\xe6\xf2", [0x33a6] = "\xe6\xf3", [0x33a7] = "\xe6\xef",
  [0x33af] = "\xf6\xf8", [0x33b2] = "\xf6\xfb", [0x33ba] = "\xf6\xfc",
  [0x33c0] = "\xf0\xc9", [0x33c9] = "\xf0\xca", [0x33ce] = "\xf0\xcb",
  [0x33cf] = "\xf0\xcd", [0x33d6] = "\xf7\xf4", [0x33d8] = "\xf7\xf3",
  [0x33dd] = "\xf6\xb6", [0x33e0] = "\xf6\xb5", [0x33e2] = "\xd9\xd0",
  [0x33eb] = "\xbf\xf9", [0x33ed] = "\xd3\xb8", [0x33ef] = "\xdf\xa5",
  [0x33f3] = "\xe9\xb7", [0x33f5] = "\xe3\xe3", [0x33f6] = "\xe4\xdc",
  [0x33fc] = "\xec\xdf", [0x33ff] = "\xe2\xb5", [0x3404] = "\xed\xc3",
  [0x3408] = "\xf4\xcf", [0x3409] = "\xe6\xfe", [0x340b] = "\xe7\xd3",
  [0x340e] = "\xc0\xdb", [0x341a] = "\xde\xbe", [0x341e] = "\xde\xc0",
  [0x341f] = "\xdd\xfc", [0x3423] = "\xf3\xba", [0x3428] = "\xc3\xef",
  [0x3433] = "\xea\xe1", [0x343b] = "\xdb\xba", [0x343f] = "\xef\xee",
  [0x3441] = "\xbb\xb7", [0x344c] = "\xe3\xcb", [0x3455] = "\xec\xad",
  [0x345c] = "\xe6\xf5", [0x345d] = "\xe6\xee", [0x345e] = "\xe6\xf1",
  [0x346c] = "\xf6\xe5", [0x346d] = "\xf7\xa3", [0x346e] = "\xf7\xa5",
  [0x3473] = "\xf0\xcf", [0x347b] = "\xf7\xbd", [0x3482] = "\xe5\xb9",
  [0x3488] = "\xf5\xba", [0x3491] = "\xea\xe5", [0x3497] = "\xd9\xe6",
  [0x349b] = "\xe6\xae", [0x34a2] = "\xd4\xdc", [0x34a6] = "\xeb\xac",
  [0x34a7] = "\xc0\xec", [0x34b3] = "\xf1\xa8", [0x34b7] = "\xf0\xa6",
  [0x34ba] = "\xf3\xea", [0x34be] = "\xd9\xe1", [0x34c4] = "\xf4\xb5",
  [0x34d1] = "\xea\xeb", [0x34d4] = "\xda\xd9", [0x34db] = "\xf5\xe9",
  [0x34dc] = "\xf5\xf0", [0x34dd] = "\xf5\xd1", [0x34de] = "\xf5\xc8",
  [0x34df] = "\xf5\xf1", [0x34e3] = "\xe9\xf6", [0x34e5] = "\xef\xd9",
  [0x34e7] = "\xef\xec", [0x34f4] = "\xf7\xd3", [0x34fa] = "\xe6\xf6",
  [0x34fc] = "\xe6\xe8", [0x3503] = "\xe5\xf7", [0x3507] = "\xf7\xa8",
  [0x350a] = "\xf6\xe4", [0x350b] = "\xf7\xa6", [0x350e] = "\xf6\xe6",
  [0x3510] = "\xf7\xa1", [0x3516] = "\xf0\xd2", [0x351d] = "\xf0\xba",
  [0x352a] = "\xed\xe8", [0x352b] = "\xba\xcd", [0x3539] = "\xdf\xac",
  [0x353d] = "\xe8\xef", [0x353e] = "\xe9\xa1", [0x3546] = "\xd3\xb8",
  [0x354c] = "\xdd\xf1", [0x354f] = "\xde\xc2", [0x3554] = "\xee\xc3",
  [0x355b] = "\xd1\xe7", [0x355c] = "\xf6\xc5", [0x3562] = "\xe9\xf1",
  [0x3564] = "\xd1\xe7", [0x3570] = "\xf7\xdf", [0x3572] = "\xf7\xae",
  [0x3573] = "\xf6\xe0", [0x357e] = "\xf0\xd3", [0x3581] = "\xf0\xd6",
  [0x3586] = "\xf0\xd4", [0x3587] = "\xf0\xd5", [0x3588] = "\xd1\xe0",
  [0x358b] = "\xf0\xc2", [0x358c] = "\xf0\xc2", [0x3591] = "\xf7\xf5",
  [0x3596] = "\xf7\xfb", [0x3599] = "\xec\xb4", [0x359e] = "\xc4\xf6",
  [0x35a8] = "\xe5\xb0", [0x35b1] = "\xf3\xfd", [0x35b7] = "\xf3\xbd",
  [0x35bb] = "\xc0\xbe", [0x35bc] = "\xf5\xf3", [0x35c3] = "\xf6\xce",
  [0x35cd] = "\xf7\xc6", [0x35d0] = "\xf7\xaf", [0x35e6] = "\xeb\xf1",
  [0x35e7] = "\xe0\xec", [0x35e8] = "\xdf\xad", [0x35f4] = "\xf3\xd6",
  [0x35f6] = "\xf4\xd0", [0x35f8] = "\xe7\xda", [0x35f9] = "\xf4\xee",
  [0x35fb] = "\xd9\xf5", [0x3603] = "\xf1\xe1", [0x3607] = "\xb4\xda",
  [0x360b] = "\xef\xe7", [0x3611] = "\xe2\xc3", [0x3614] = "\xf6\xdd",
  [0x361d] = "\xf6\xbe", [0x3626] = "\xf3\xbd", [0x3627] = "\xf4\xf5",
  [0x3628] = "\xf5\xf2", [0x3629] = "\xf5\xa7", [0x362d] = "\xb9\xde",
  [0x363f] = "\xda\xdd", [0x3644] = "\xf2\xa8", [0x3645] = "\xc8\xa7",
  [0x3649] = "\xe6\xf8", [0x364d] = "\xf0\xb5", [0x3650] = "\xed\xb0",
  [0x3651] = "\xe8\xf9", [0x365a] = "\xe3\xce", [0x3664] = "\xf6\xe2",
  [0x3665] = "\xf0\xbf", [0x3666] = "\xe4\xd9", [0x366c] = "\xc6\xe5",
  [0x366d] = "\xd0\xe2", [0x366e] = "\xc0\xef", [0x366f] = "\xc7\xbd",
  [0x3670] = "\xba\xe3", [0x3671] = "\xd7\xb1",
};


/* Definitions used in the body of the `gconv' function.  */
#define CHARSET_NAME		"EUC-CN//"
#define FROM_LOOP		from_gb2312_to_big5
#define TO_LOOP			from_big5_to_gb2312
#define DEFINE_INIT		1
#define DEFINE_FINI		1
#define MIN_NEEDED_FROM		1
#define MAX_NEEDED_FROM		2
#define MIN_NEEDED_TO		1
#define MAX_NEEDED_TO		2
#define ONE_DIRECTION		0


/* First define the conversion function from GB2312 to BIG5.  */
#define MIN_NEEDED_INPUT	MIN_NEEDED_FROM
#define MAX_NEEDED_INPUT	MAX_NEEDED_FROM
#define MIN_NEEDED_OUTPUT	MIN_NEEDED_TO
#define MAX_NEEDED_OUTPUT	MAX_NEEDED_TO
#define LOOPFCT			FROM_LOOP
#define BODY \
  {									      \
    uint32_t ch = *inptr;						      \
									      \
    if (ch <= 0x7f)							      \
      *outptr++ = *inptr++;						      \
    else if (ch >= 0xa1 && ch <= 0xf7)					      \
      {									      \
	/* Two-byte character.  First test whether the		      	      \
	   next byte is also available.  */				      \
	const char *cp;							      \
	int idx;							      \
									      \
	if (__glibc_unlikely (inptr + 1 >= inend))			      \
	  {								      \
	    /* The second character is not available.  Store		      \
	       the intermediate result.  */				      \
	    result = __GCONV_INCOMPLETE_INPUT;				      \
	    break;							      \
	  }								      \
									      \
	idx = (ch - 0xa1) * 94;						      \
	ch = inptr[1];						     	      \
									      \
	/* All second bytes of a multibyte character must be >= 0xa1. */      \
	if (__glibc_unlikely (ch < 0xa1))				      \
	  {								      \
	    /* This is an illegal character.  */			      \
	    STANDARD_FROM_LOOP_ERR_HANDLER (1);				      \
	  }								      \
									      \
	idx += ch - 0xa1;						      \
									      \
	/* Get the value from the table.  */				      \
	cp = __from_gb2312_to_big5[idx];				      \
	if (__glibc_unlikely (cp[0] == '\0'))				      \
	  {								      \
	    /* We do not have a mapping for this character.		      \
	       If ignore errors, map it to 0xa1bc - big5 box character */     \
	    result = __GCONV_ILLEGAL_INPUT;				      \
	    if (! ignore_errors_p ())					      \
	      break;							      \
									      \
	    /* See if there is enough room to write the second byte. */	      \
	    if (__glibc_unlikely (outptr + 1 >= outend))		      \
	      {								      \
		result = __GCONV_FULL_OUTPUT;				      \
	        break;							      \
	      }								      \
									      \
	    *outptr++ = '\xa1';						      \
	    *outptr++ = '\xbc';						      \
	    inptr += 2;							      \
	    ++*irreversible;						      \
	    continue;							      \
	  }								      \
	else								      \
	  {								      \
	    /* See whether there is enough room to write the second byte. */  \
	    if (__builtin_expect (cp[1], '\1') != '\0'			      \
	        && __builtin_expect (outptr + 1 >= outend, 0))		      \
	      {								      \
	        /* We do not have enough room.  */			      \
	        result = __GCONV_FULL_OUTPUT;				      \
	        break;							      \
	      }								      \
									      \
	    *outptr++ = cp[0];						      \
	    if (cp[1] != '\0')						      \
	      *outptr++ = cp[1];					      \
	  }								      \
      									      \
	inptr += 2;							      \
      }									      \
    else								      \
      {									      \
	/* This is illegal.  */						      \
	STANDARD_FROM_LOOP_ERR_HANDLER (1);				      \
      }									      \
  }
#define LOOP_NEED_FLAGS
#include <iconv/loop.c>

/* Next, define the other direction.  */
#define MIN_NEEDED_INPUT	MIN_NEEDED_TO
#define MAX_NEEDED_INPUT	MAX_NEEDED_TO
#define MIN_NEEDED_OUTPUT	MIN_NEEDED_FROM
#define MAX_NEEDED_OUTPUT	MAX_NEEDED_FROM
#define LOOPFCT			TO_LOOP
#define BODY \
  {									      \
    uint32_t ch = *inptr;						      \
									      \
    if (ch <= 0x7f)							      \
      *outptr++ = *inptr++;						      \
    else if (ch >= 0xa1 && ch <= 0xf9)					      \
      {									      \
	/* Two byte character.  First test whether the			      \
	   next byte is also available.  */				      \
	const char *cp;							      \
	int idx;							      \
									      \
	if (__glibc_unlikely (inptr + 1 >= inend))			      \
	  {								      \
	    /* The second character is not available.  Store		      \
	       the intermediate result.  */				      \
	    result = __GCONV_INCOMPLETE_INPUT;				      \
	    break;							      \
	  }								      \
									      \
	idx = (ch - 0xa1) * 157;					      \
	ch = inptr[1];						     	      \
	/* See if the second byte is in the correct range. */		      \
	if (ch >= 0x40 && ch <= 0x7e)					      \
	  idx += ch - 0x40;						      \
	else if (__builtin_expect (ch >= 0xa1, 1)		  	      \
		 && __builtin_expect (ch <= 0xfe, 1))			      \
	  idx += 0x3f + (ch - 0xa1);					      \
	else								      \
	  {								      \
	    /* This is an illegal character.  */			      \
	    STANDARD_FROM_LOOP_ERR_HANDLER (1);				      \
	  }								      \
									      \
	/* Get the value from the table.  */				      \
	cp = __from_big5_to_gb2312 [idx];				      \
	if (__glibc_unlikely (cp[0] == '\0'))				      \
	  {								      \
	    /* We do not have a mapping for this character.		      \
	       If ignore errors, map it to 0xa1f5 - gb box character */       \
	    result = __GCONV_ILLEGAL_INPUT;				      \
	    if (! ignore_errors_p ())					      \
	      break;							      \
									      \
	    /* See if there is enough room to write the second byte. */	      \
	    if (__glibc_unlikely (outptr + 1 >= outend))		      \
	      {								      \
		result = __GCONV_FULL_OUTPUT;				      \
	        break;							      \
	      }								      \
									      \
	    *outptr++ = '\xa1';						      \
	    *outptr++ = '\xf5';						      \
	    inptr += 2;							      \
	    ++*irreversible;						      \
	    continue;							      \
	  }								      \
	else								      \
	  {								      \
	    /* see if there is enough room to write the second byte. */	      \
	    if (__builtin_expect (cp[1], '\1') != '\0'			      \
	        && __builtin_expect (outptr + 1 >= outend, 0))		      \
	      {								      \
	        /* We do not have enough room.  */			      \
	        result = __GCONV_FULL_OUTPUT;				      \
	        break;							      \
	      }								      \
									      \
	    *outptr++ = cp[0];						      \
	    if (cp[1] != '\0')						      \
	      *outptr++ = cp[1];					      \
	  }								      \
      									      \
	inptr += 2;							      \
      }									      \
    else								      \
      {									      \
	/* This is illegal.  */						      \
	STANDARD_FROM_LOOP_ERR_HANDLER (1);				      \
      }									      \
  }
#define LOOP_NEED_FLAGS
#include <iconv/loop.c>

/* Now define the toplevel functions.  */
#include <iconv/skeleton.c>
