/* m3derode.c  --  3d morphology erosion operators */

/* contains: BinBinErode(), BinGrayErode(), GrayBinErode(),  */
/*           GrayGraySetErode(), GrayGrayFctErode()          */

/* dependencies: none */
 
/* morph3d version 4.0   1 June 1993               */
/* 3D image morphology program                     */    
/*                                                 */ 
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */

#include "morph3d_sub.h"


/* Erosion via hit-or-miss transform of a binary image with a binary SE.    */
/* Binary is interpreted as zero and positive (not zero). Images are        */
/* unsigned. A negative SE element is treated as a DON'T CARE. Positive,    */
/* non-zero SE elements are hit-points and zero SE elements are miss-points */

/* note: at hit occurs when the SE element is >0 and the corresponding pixel */
/* is >0. a miss occurs when the SE element is 0 and the corresponding pixel */
/* is 0. Most SE's will contain only 1's and -1's (don't cares)              */

/* New in v1.2: when the "not" flag is true, the pixel under the SE origin   */
/* in input image is passed to the output on a FALSE condition and BLACK is  */
/* passed on a TRUE condition. If I is the original (binary) image and E is  */
/* the eroded image, then the "not" flag causes I AND (NOT E) pixelwise to   */
/* be output.  This option when used with a hit-or-miss transform can delete */ 
/* features with a specific shape from an image. */


void BinBinErode( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, not ) 
   byte **In, *Out;       /* input and output images */
   int X;                 /* image horizontal dimension */
   int tX,tY;             /* transform scan horiz., vert. dimensions */
   int *stelem;           /* SE in row-major order */
   int sX,sY,sZ;          /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int not;               /* T => pass pixel on dilation true, black on false */
   {
   byte *I,*O;    /* image pointers */
   int i,j,k;     /* transform indices */
   int x,y;       /* image indices */
   int p,op;      /* pixel value */
   int r;         /* transform result */
   int *s;        /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         r = 1;
         s = stelem;     
         op = *(In[sorgz] + sorgy*X + sorgx); /* pixel at SE origin */
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s >= 0) 
                     { 
                     p = *(I + j*X + i); /* get pixel   */
                     r = *s ? p : !p ;   /* hit-or-miss */
                     if (!r)  /* FALSE--for erosion, fail once and we're done */
                        {
                        *(O++) = not ? op : BLACK;
                        break; /* out of SE column scan */
                        }
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               if (!r) break; /* out of SE row scan */
               }                    /* end of SE column scan */ 
            if (!r) break; /* out of SE z-dim scan */
            }                       /* end of SE z-dim scan */
         if (r) /* TRUE -- every hit hit and every miss missed */
             *(O++) = not ? BLACK : WHITE; 
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of BinBinErode */





/* Morphological erosion of a binary image with a gray-level SE.            */
/* Binary is interpreted as zero and positive (not zero). Images are        */
/* unsigned. A negative SE element is treated as a DON'T CARE. The erosion  */
/* is performed over the support of nonnegative SE elements.                */

/* New in v1.2: when the "not" flag is true, the pixel under the SE origin  */
/* in input image is passed to the output on a FALSE condition and BLACK is */
/* passed on a TRUE condition. If I is the original (binary) image and E is */
/* the eroded image, then the "not" flag causes I AND (NOT E) pixelwise to  */
/* be output.  When used with a convex, symmetric, non-hit-or-miss SE,      */ 
/* this causes an interior delete of white regions */

void BinGrayErode( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, not ) 
   byte **In, *Out;       /* input and output images */
   int X;                 /* image horizontal dimension */
   int tX,tY;             /* transform scan horiz., vert. dimensions */
   int *stelem;           /* SE in row-major order */
   int sX,sY,sZ;          /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int not;               /* T => pass pixel on dilation true, black on false */
   {
   byte *I,*O;    /* image pointers */
   int i,j,k;     /* transform indices */
   int x,y;       /* image indices */
   int r;         /* transform result */
   int op;        /* pixel at SE origin */
   int *s;        /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         r = 1;
         s = stelem;     
         op = *(In[sorgz] + sorgy*X + sorgx); /* pixel at SE origin */
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s >= 0)  
                     {
                     r = *(I + j*X + i);  
                     if (!r)  /* FALSE--for erosion, fail once and we're done */
                        {
                        *(O++) = not ? op : BLACK;
                        break; /* out of SE column scan */
                        }
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               if (!r) break; /* out of SE row scan */
               }                       /* end of SE row scan */
            if (!r) break; /* out of SE z-dim scan */
            }                       /* end of SE z-dim scan */
         if (r) /* TRUE -- every hit hit and every miss missed */
             *(O++) = not ? BLACK : WHITE; 
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of GrayBinErode */




/* Morphological (function-set) erosion of a gray-level image with a binary */
/* SE. Binary is interpreted as zero and positive (not zero). Images are    */
/* unsigned. A negative SE element is treated as a DON'T CARE. The erosion  */
/* is perfomed over the positive (>0) support of of the SE.                 */

void GrayBinErode( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, unused ) 
   byte **In, *Out;       /* input and output images */
   int X;                 /* image horizontal dimension */
   int tX,tY;             /* transform scan horiz., vert. dimensions */
   int *stelem;           /* SE in row-major order */
   int sX,sY,sZ;          /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int unused;            /* to conform with other routines' param list */
   {
   byte *I,*O;    /* image pointers */
   int i,j,k;      /* transform indices */
   int x,y;       /* image indices */
   int p;         /* pixel value */
   int r;         /* transform result */
   int *s;        /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         s = stelem;     
         r = WHITE;
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s > 0)  
                     {
                     p = *(I + j*X + i);  /* get pixel   */
                     r = MIN(r,p);
                     if ( r == BLACK ) break; /* out of col. scan */
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               if ( r == BLACK ) break;    /* out of SE row scan */
               }                       /* end of SE row scan */
            if ( r == BLACK ) break;       /* out of SE z-dim scan */
            }                          /* end of SE z-dim scan */
         *(O++) = r; /* output result */
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of GrayBinErode */




/* Morphological function-set erosion of a gray-level image with a      */
/* gray-level SE. Images are unsigned. A negative SE element is treated */
/* as a DON'T CARE. The erosion is perfomed as a function-set operation */
/* over the nonnegative (>= 0) support of of the SE. */

void GrayGraySetErode( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, unused ) 
   byte **In, *Out;       /* input and output images */
   int X;                 /* image horizontal dimension */
   int tX,tY;             /* transform scan horiz., vert. dimensions */
   int *stelem;           /* SE in row-major order */
   int sX,sY,sZ;          /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int unused;            /* to conform with other routines' param list */
   {
   byte *I,*O;    /* image pointers */
   int i,j,k;     /* transform indices */
   int x,y;       /* image indices */
   int p;         /* pixel value */
   int r;         /* transform result */
   int *s;        /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         s = stelem;     
         r = WHITE;
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s >= 0)  
                     {
                     p = *(I + j*X + i);  /* get pixel   */
                     r = MIN(r,p);
                     if ( r == BLACK ) break; /* out of col. scan */
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               if ( r == BLACK ) break;    /* out of SE row scan */
               }                       /* end of SE row scan */
            if ( r == BLACK ) break;       /* out of SE z-dim scan */
            }                          /* end of SE z-dim scan */
        *(O++) = r; /* output result */
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of GrayGraySetErode */




/* Morphological function erosion of a gray-level image with a          */
/* gray-level SE. Images are unsigned. A negative SE element is treated */
/* as a DON'T CARE. The erosion is perfomed as a function operation     */
/* over the nonnegative (>= 0) support of of the SE. */

void GrayGrayFctErode( In, Out, X, tX, tY, stelem, sX, sY, sZ, sorgx, sorgy, sorgz, unused ) 
   word **In, *Out;       /* input and output images */
   int X;                 /* image horizontal dimension */
   int tX,tY;             /* transform scan horiz., vert. dimensions */
   int *stelem;           /* SE in row-major order */
   int sX,sY,sZ;          /* SE horizontal, vertical, depth dimensions */
   int sorgx,sorgy,sorgz; /* SE origin offset from upper LH corner. */
   int unused;            /* to conform with other routines' param list */
   {
   word *I,*O;    /* image pointers */
   int i,j,k;     /* transform indices */
   int x,y;       /* image indices */
   int t,r;       /* transform result */
   int *s;        /* pointer to SE  */
   
   for ( y=0; y<tY; ++y )           /* image row scan */
      {
      O = Out + (y+sorgy)*X + sorgx;
      for ( x=0; x<tX; ++x )        /* image column scan */
         {
         s = stelem;     
         r = WHITE;
         for ( k=0; k<sZ; ++k )        /* SE z-dim scan */
            {
            I = In[k] + y*X + x;
            for ( j=0; j<sY; ++j )     /* SE row scan */
               {
               for ( i=0; i<sX; ++i )  /* SE column scan */
                  {
                  if (*s >= 0)  
                     {
                     t = *(I + j*X + i) - *s; /* pixel minus SE value */
                     r = MIN(t,r);
                     }
                  ++s;
                  }                    /* end of SE column scan */ 
               }                       /* end of SE row scan */
            }                          /* end of SE z-dim scan */
         *(O++) = r; /* output result */
         }                          /* end of image column scan */
      }                             /* end of image row scan */
  return;
  }                                 /* end of GrayGrayFctErode */
