/*
 *  @(#)IErrors.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1;


/**
 * A container for registering errors or warnings caused by invalid
 * validation of a state or transition.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:24 $
 * @since      June 12, 2002
 */
public interface IErrors
{
    
    /**
     * Immediately stop the path processing, and do not continue other paths
     * for processing.  This will throw a <tt>RuntimeException</tt>.
     * <P>
     * Halts should be a last-recourse to indicate that the system cannot be
     * used for further testing.
     *
     * @param msg a human-readable error message.
     * @exception TestHaltRuntimeException will always be generated.
     */
    public void halt( String msg )
            throws TestHaltRuntimeException;
    
    
    /**
     * Add a failure to the list of current errors.  Validation methods that
     * register failures will halt the current path's testing.  This method
     * will not throw an exception, so validation processing must leave the
     * method on its own.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     */
    public void addFailure( String msg );
    
    
    /**
     * Add a failure associated with a Throwable to the list of current errors.
     * Validation methods that register failures will halt the current path's
     * testing.  This method will not throw an exception, so validation
     * processing must leave the method on its own.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     * @param t the exception associated with the error.
     */
    public void addFailure( String msg, Throwable t );
    
    
    /**
     * Add a failure to the list of current errors.  Validation methods that
     * register failures will halt the current path's testing.  This method
     * will not throw a <tt>TestFailRuntimeException</tt>, so validation
     * processing must leave the method on its own.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     * @exception TestFailRuntimeException allows for easy exiting of a
     *      burried method call stack.
     */
    public void fail( String msg )
            throws TestFailRuntimeException;
    
    
    /**
     * Add a failure associated with a Throwable to the list of current errors.
     * Validation methods that register failures will halt the current path's
     * testing.  This method will throw a <tt>TestFailRuntimeException</tt> to
     * allow for an easy exit from a burried method call stack.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     * @param t the exception associated with the error.
     * @exception TestFailRuntimeException allows for easy exiting of a
     *      burried method call stack.
     */
    public void fail( String msg, Throwable t )
            throws TestFailRuntimeException;
    
    
    /**
     * Add an error to the list of current errors.  Validation methods that
     * register errors will not halt the current path's testing, but the error
     * will be listed in the report with the associated path where the error
     * condition occured.
     * <P>
     * Errors should be registered when an error occurs in the system, but
     * the system can continue processing the path.
     *
     * @param msg a human-readable error message.
     */
    public void addError( String msg );
    
    
    /**
     * Add an error associated with a Throwable to the list of current errors.
     * Validation methods that register errors will halt the current path's
     * testing.
     * <P>
     * Errors should be registered when an error occurs in the system, but
     * the system can continue processing the path.
     *
     * @param msg a human-readable error message.
     * @param t the exception associated with the error.
     */
    public void addError( String msg, Throwable t );
    
    
    /**
     * Add a warning to the list of current warnings.  Warnings will not
     * halt the testing process, and will not register an error.
     * <P>
     * Warnings should be used when a questionable system state occurs, or if
     * the tester wants to perform debugging.
     *
     * @param msg a human-readable message.
     */
    public void addWarning( String msg );
    
    
    /**
     * Retrieve all registered errors.
     */
    public IError[] getErrors();
}

