// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: pkfont.C,v 1.21 91/02/22 15:58:17 hmgr Exp $";

// read font data from the METAFONT PK (packed font) files
//
// by Parag Patel

#include "defs.h"


// these are for "drawing" an image of a character in a font
const int WHITE = FALSE;
const int BLACK = TRUE;


// return a string containing the magnification sub-directory for the
// requested magnification
// 
char *pkmagdir(Dirlist &dirlist, long mag)
{
    // look for the closest resolution value to the one desired
    int ent = -1;
    long diff = MAXLONG;

    for (int i = 0; i < dirlist.size(); i++)
    {
	long magval = dirlist[i].val;
	if (strchr(dirlist[i].name, '.') == NULL)
	{
	    magval *= 1000;
	    magval /= RESOLUTION;
	}

	long d = mag - magval;
	long newdiff = d < 0 ? -d : d;	// absolute value
	if (newdiff < diff)
	{
	    diff = newdiff;
	    ent = i;
	}
    }
    debug(7, "PK dir ent=%s  diff=%ld  mag=%ld",
	    ent < 0 ? "" : dirlist[ent].name, diff, mag);

    // we should be within 1/32 (~3%) of the desired mag value
    if (ent < 0 || diff > (mag >> 5))
	return "";

    // return the new value as a string, if there was one
    return dirlist[ent].name;
}


// setup a PK font file in memory
//
void setuppkfont(font &f, FILE *fp)
{
    // skip the comment
    int len = (unsigned)getuval(1, fp);
    while (len-- > 0)
	(void)getuval(1, fp);

    // get and verify that the design size is ok - the PK file stores
    // this value as a "fix_word" value while the DVI file keeps it as
    // a "scaled-point" value - also note that the "basename" pointer
    // is NULL only for the "dumpfont" main program

    long dsize = getsval(4, fp);
    debug(4, "PK designsize=%ld", dsize);
    if (f.basename != NULL && dsize >> 4 != f.designsize)
	if (dochecksum && dsize != 0 && f.designsize != 0)
	    quit("Designsize in DVI and PK file %s does not match", f.path);
	else
	    warn("Designsize in DVI and PK file %s does not match", f.path);

    // check the checksum
    long check = getuval(4, fp);
    debug(5, "PK checksum=%ld", check);
    if (f.basename != NULL && check != f.checksum)
	if (dochecksum && check != 0 && f.checksum != 0)
	    quit("Checksums in DVI and PK file %s do not match", f.path);
	else
	    warn("Checksums in DVI and PK file %s do not match", f.path);

    // get the horizontal and vertical pixels per point values
    f.hppp = getuval(4, fp);
    f.vppp = getuval(4, fp);

    debug(3, "mag=%ld  hppp=%ld vppp=%ld", f.mag, f.hppp, f.vppp);

    f.minm = MAXLONG;
    f.maxm = -MAXLONG;
    f.minn = MAXLONG;
    f.maxn = -MAXLONG;

    // now initialize the info for each character in this font
    int op;
    while ((op = (int)getuval(1, fp)) != PKPOST)
    {
	switch (op)
	{
	    // special opcodes - just ignore for now
	case PKXXX1:
	    skipbytes(getuval(1, fp), fp);
	    continue;
	case PKXXX2:
	    skipbytes(getuval(2, fp), fp);
	    continue;
	case PKXXX3:
	    skipbytes(getuval(3, fp), fp);
	    continue;
	case PKXXX4:
	    skipbytes(getuval(4, fp), fp);
	    continue;

	    // special numeric opcode - also ignored
	case PKYYY:
	    (void)getuval(4, fp);
	    continue;

	case PKNOOP:
	    continue;
	default:
	    ;
	}

	// save the location of this character in the file
	long floc = ftell(fp) - 1;

	if (op > PKCHAR)
	    quit("Illegal opcode %d in PK file %s", op, f.path);

	int flag = op;
	int size = (flag & 0x04) ? 2 : 1;
	int addtolen = flag & 0x03;
	if ((flag & 0x07) == 0x07)
	    size = 4;

	long packetlen, charcode;

	switch (size)
	{
	Case 1:
	    packetlen = getuval(1, fp) + (addtolen << 8);
	    charcode = getuval(1, fp);

	Case 2:
	    packetlen = getuval(2, fp) + (addtolen << 16);
	    charcode = getuval(1, fp);

	Case 4:
	    packetlen = getuval(4, fp);
	    charcode = getuval(4, fp);

	Default:
	    quit("?!?Eh?  Internal error - size must be 1, 2, or 4.");
	}

	if (charcode >= MAXCHARS)
	    quit("Char %ld too big in PF file %s", charcode, f.path);
	fontchar & g = f.chr[charcode];

	// save the location of the character data in the file
	g.floc = floc;

	double tfmwidth;
	long width, height, hoff, voff;

	switch (size)
	{
	Case 1:
	    tfmwidth = Getuval(3, fp);
	    g.dx = getuval(1, fp) << 16;
	    g.dy = 0;
	    width = getuval(1, fp);
	    height = getuval(1, fp);
	    hoff = getsval(1, fp);
	    voff = getsval(1, fp);
	    packetlen -= 8;

	Case 2:
	    tfmwidth = Getuval(3, fp);
	    g.dx = getuval(2, fp) << 16;
	    g.dy = 0;
	    width = getuval(2, fp);
	    height = getuval(2, fp);
	    hoff = getsval(2, fp);
	    voff = getsval(2, fp);
	    packetlen -= 13;

	Case 4:
	    tfmwidth = Getsval(4, fp);
	    g.dx = getuval(4, fp);
	    g.dy = getuval(4, fp);
	    width = getuval(4, fp);
	    height = getuval(4, fp);
	    hoff = getsval(4, fp);
	    voff = getsval(4, fp);
	    packetlen -= 28;

	Default:
	    quit("?!?Eh?  Internal error - size must be 1, 2, or 4.");
	}

	// calculate character min/max boundaries
	g.minm = -hoff;
	g.maxm = width - hoff;
	g.minn = voff - height + 1;
	g.maxn = voff;

	// set the size limits of the characters in this font
	if (g.minm < f.minm)
	    f.minm = g.minm;
	if (g.maxm > f.maxm)
	    f.maxm = g.maxm;
	if (g.minn < f.minn)
	    f.minn = g.minn;
	if (g.maxn > f.maxn)
	    f.maxn = g.maxn;

	// calculate the scaled points width from the pixel width
	g.width = (double)tfmwidth / (double)16 * (double)f.designsize /
	(double)65536L * (double)f.mag / 1000.0;

	// skip the character data and go to the next char
	if (fseek(fp, packetlen, SEEK_CUR) < 0)
	    quit("Cannot seek past character data in %s", f.path);
    }
}


static long packetlen = 0;


// return a nybble (4-bit value) from a file - reset ourselves
// if we are called with a NULL file - return the upper nybble
// of a byte first and then the lower - as a side-effect, we decrement
// the packetlen count for every byte read from the file
//
static int getnyb(FILE *fp)
{
    static boolean nybsave = FALSE;
    static int nybdata = 0;

    packetlen--;
    if (fp == NULL || nybsave)
    {
	nybsave = FALSE;
	return nybdata & 0x0F;
    }
    nybsave = TRUE;
    nybdata = (unsigned)getuval(1, fp);
    return nybdata >> 4;
}



static int dynf = 0;
static int repeat = 0;


// get a packed run-count value from a file using getnyb() above - 
// assumes that dynf above is initialized to a reasonable value
//
static int getpacked(FILE *fp)
{
    int i = getnyb(fp);
    if (i == 0)
    {
	if (packetlen <= 0)
	    return 0;
	int j;
	do
	{
	    j = getnyb(fp);
	    i++;
	} while (j == 0);
	while (i-- > 0)
	    j = (j << 4) + getnyb(fp);
	return j - 15 + ((13 - dynf) << 4) + dynf;
    }
    if (i <= dynf)
	return i;
    if (i < 14)
	return ((i - dynf - 1) << 4) + getnyb(fp) + dynf + 1;
    if (i == 14)
	repeat = getpacked(fp);
    else
	repeat = 1;
    return getpacked(fp);
}



// load the specified character "ch" from the PK file - we read this
// character's PK commands to build a bitmap of the character in memory
// 
void getpkchar(font &f, fontchar &g, int ch)
{
    // go to the file where this character is defined
    if (fseek(f.fp, g.floc, SEEK_SET) < 0)
	quit("Cannot fseek to start of font in %s", f.path);
    FILE *fp = f.fp;

    // reset our get-nybble function
    (void)getnyb(NULL);

    int flag = (unsigned)getuval(1, fp);
    if (flag > PKCHAR)
	quit("Illegal flag value %d in PK file %s", flag, f.path);

    // set up globals for getpacked() above
    dynf = flag >> 4;

    int size = (flag & 0x04) ? 2 : 1;
    long addtolen = flag & 0x03;
    if ((flag & 0x07) == 0x07)
	size = 4;

    long charcode;

    switch (size)
    {
    Case 1:
	packetlen = getuval(1, fp) + (addtolen << 8);
	charcode = getuval(1, fp);

    Case 2:
	packetlen = getuval(2, fp) + (addtolen << 16);
	charcode = getuval(1, fp);

    Case 4:
	packetlen = getuval(4, fp);
	charcode = getuval(4, fp);

    Default:
	quit("?!?Eh?  Internal error - size must be 1, 2, or 4");
    }

    if (charcode != ch)
	quit("?!?Eh?  Internal error - expected charcode for %d here", ch);

    long dx, dy, tfmwidth, width, height, hoff, voff;

    switch (size)
    {
    Case 1:
	tfmwidth = getuval(3, fp);
	dx = getuval(1, fp) << 16;
	dy = 0;
	width = getuval(1, fp);
	height = getuval(1, fp);
	hoff = getsval(1, fp);
	voff = getsval(1, fp);
	packetlen -= 8;

    Case 2:
	tfmwidth = getuval(3, fp);
	dx = getuval(2, fp) << 16;
	dy = 0;
	width = getuval(2, fp);
	height = getuval(2, fp);
	hoff = getsval(2, fp);
	voff = getsval(2, fp);
	packetlen -= 13;

    Case 4:
	tfmwidth = getuval(4, fp);
	dx = getuval(4, fp);
	dy = getuval(4, fp);
	width = getuval(4, fp);
	height = getuval(4, fp);
	hoff = getsval(4, fp);
	voff = getsval(4, fp);
	packetlen -= 28;

    Default:
	quit("?!?Eh?  Internal error - size must be 1, 2, or 4.");
    }
    {
	int len;
	const char *x = dev_char2dev(ch, len);
	debug(5, "char=%d(%s)  minm=%ld maxm=%ld  minn=%ld maxn=%ld",
		ch, x, g.minm, g.maxm, g.minn, g.maxn);
	debug(5, "    dx=%ld dy=%ld  dx/=%ld dy/=%ld  width=%ld/%f  height=%ld",
		dx, dy, dx >> 16, dy >> 16, width, g.width, height);
	debug(5, "    hoff=%ld voff=%ld  len=%ld  flag=%d  dyn_f=%d",
		hoff, voff, packetlen, flag, dynf);
    }

    if (dynf < 0 || dynf > 14)
	quit("?!?Eh?  dyn_f value must be 0-14 inclusive - not %d", dynf);

    // initialize the character painting variables
    register long m = 0;
    register long n = height - 1;
    int p = (flag & 0x08) ? BLACK : WHITE;

    // clear the global fontbits for the bitmap
    for (long d = n; d >= 0; d--)
	fontbits[d]->clear();

    // paint the fontbits to build this character
    if (dynf < 14)
    {
	// packed data
	repeat = 0;
	packetlen <<= 1;
	while (packetlen > 0)
	{
	    int count = getpacked(fp);
	    while (count-- > 0)
	    {
		if (p)
		    *fontbits[n] += m;
		if (++m >= width)
		{
		    long r = n--;
		    n -= repeat;
		    m = 0;
		    for (; repeat > 0; repeat--)
			*fontbits[r - repeat] = *fontbits[r];
		    repeat = 0;
		}
	    }
	    p = !p;
	}
	for (; repeat > 0; repeat--)
	    *fontbits[n - repeat] = *fontbits[n];
    }
    else
    {
	// straight bit-map
	while (packetlen-- > 0)
	{
	    int byte = (unsigned)getuval(1, fp);
	    for (int mask = 0x0080; mask != 0; mask >>= 1)
	    {
		if (byte & mask)
		    *fontbits[n] += m;
		if (++m >= width)
		{
		    n--;
		    m = 0;
		}
	    }
	}
    }
}
