// Due to some compiler template specialisation inadequacies, we use
// separate list-of-pointers and list-of-objects classes.
#ifndef _cpptcl_olist_
#define _cpptcl_olist_

#include <stdlib.h>

// Some forward references.  We're defining three
// mutually-using classes.
template<class T>
class olist;
 
template<class T> 
class olist_pos;


// -------------------- olist_node (private class) --------------------
// olist_nodes are little more than cells containing a T-value 
// and pointers to the next and previous nodes.
// They are entirely private, except that the public classes 
// can refer to them.

template<class T> struct olist_node;

template<class T> 
struct olist_node{
private:
friend class olist<T>;
friend class olist_pos<T>;
  T t;
  olist_node<T> *next, *prev;
  olist_node(olist_node<T> *prev, const T tt,  olist_node<T> *next) : t(tt)
    {this->prev = prev; this->next=next;};
  ~olist_node(void) {} // Just delete this node
};

// --------------------  olist --------------------

template<class T> 
class olist {
public:

  // Create olists with 0-4 elements....
  olist ();
  olist (T t1); 
  olist (T t1, T t2); 
  olist (T t1, T t2, T t3); 
  olist (T t1, T t2, T t3, T t4); 
  
  // Usual Classical Form stuff...
  virtual ~olist(void);

  // Copy and assignment.  Warning; these duplicate olist structure,
  // so be a bit careful about calling them; they can be expensive.
  olist(const olist&);
  olist<T>& operator=(const olist<T>& l);

  // The following constructor creates a olist from a length n array of T's
  olist(T *at, int  n);

// MUTATORS

  // Adding and removing things to olists
 

  void append(T t);  // Add t at the end of *this.
  void remove(T t);        //rem t from *this.
// Tests...
  bool isEmpty() const { return (this->headPtr == 0 ? true : false); }
  bool isNonEmpty() const {return (this->headPtr == 0 ? false : true); }
  // Return true iff *this contains an element == to t.
  // Requires T::operator== or the like.
  bool contains(const T& t) const;

/*
  int operator==(const olist<T> l) const;
  int operator!=(const olist<T> l) const{return !(*this == l);};

*/

// Returns the position of 't' in the olist, or -1 if not there
// Requires T::operator== or similar (ok for pointers).
// Note: totally un-debugged because I thought I needed it and then
// came up with an easier solution to this problem
  int positionOf(const T& t) const;
  
  
// Getting olist_pos's to the ends of the olist.

   // Get a olist_pos (a marker to an element)
   // head and tail olist_pos's allow modification of the olist through 
   // the olist_pos.
   // headConst and tailConst can be used to look along the olist,
   // but not modify it.  They can be used in const contexts, though.

  olist_pos<T> head() ;
  olist_pos<T> headConst() const; 
  olist_pos<T> tail() ;
  olist_pos<T> tailConst() const;

// OPERATIONS: 
  
  olist<T> operator+(const olist<T>& lt) const;
  olist<T>& operator+=(const olist<T>& lt);
  // olist concatenation.  Copies both *this and lt, so it's not 
  // cheap.
  
  void appendAndDestroy(olist<T>& lt);
  // Append lt to *this.  Modifies *this (making it have lt at the end),
  // and sets lt to a null olist.  This is quite cheap.
  // Hopefully, the alarming name will warn users that
  // it's got an unusual side effect.
  

  // Reverses the olist so that the old head element is the new tail
  // element and vis versa
  void reverseInPlace();

  // toArray creates an array of T's from olist. 
  T *toArray() const;
  // returns n'th element
  // currently performs no error checking
  T nth_element(int n) const;
  
  // The length of the olist (duh).
  int length() const;

  /// Empty the list safely
  void remove_all_contents(void) {trashlist();initToNulllist();}

private: 

  // ABSTRACTION FUNCTION:
  // This implements a mutable olist of T's.
  // The elements of the olist are given, in order,
  // by starting with this->headPtr's T element, and following
  // next-pointers until the NULL marking the end of the olist.

  // REP INVARIANT:
  // If the olist is empty, both headPtr and tailPtr are null.
  // Otherwise, headPtr points to the head and tailPtr
  // to the tail.

friend class olist_node<T>;
friend class olist_pos<T>;

  olist_node<T> *headPtr;
  // Head pointer: NULL for empty olist, non-NULL for non-empty olist.

  olist_node<T> *tailPtr;
  // Tail pointer: NULL for empty olist, non-NULL for non-empty olist.



  void initToNulllist();
  // UNSAFE.
  // Turns *this into a null olist, losing track of any previous
  // contents.  That is, it NULLs both pointers.
  // Afterwards, the olist is a well-represented empty
  // olist, and (e.g.) it is safe to iterate over and use olistpos's on.

  void createFirstElement(T t);
  // UNSAFE.
  // Turns an empty olist into a singleton containing t.
  // It does no checking, and will do bad things (probably
  // losing previous olist contents) if the olist is nonempty.

  void trashlist();
  // UNSAFE.
  // Destructor body.
  // Deletes the elements of *this,
  // leaving *this in an inconsistent state.

  void copyToNullFrom(const olist<T>& l);
  // UNSAFE.
  // copy constructor and assignment component.
  // Assumes that *this is a proper null olist, and copies
  // l into it.
  virtual olist<T>& _copy_me(void) const;

  olist_node<T> * allocNode(olist_node<T> *prev, T t, olist_node<T> *next);
  // Returns a new olist_node<T> containing t, doing appropriate
  // memory allocation checks.  It doesn't really concern *this,
  // but we want to hide it inside the abstraction barrier, so it's
  // here.
};


// ------------------------------ olist_pos ------------------------------
template<class T>
class olist_pos {
public:

  // No ab nihilo constructors.
  // The ordinary way of making a olist_pos on a olist is,
  // olist_pos<T> lp = l.head();
  // (or l.tail() if you prefer);
  // or one of the other olist_pos-returning functions when and if
  // they are added.
  //
  // Basically, we consider "olist_pos<T> lp(l)" to be
  // ambiguous about which end of l lp will point to.
  // However, I don't, so I've added the constructor to the head();
  
  // Assignment and copying is normal.
  olist_pos(const olist_pos<T>& lp);
  olist_pos<T>& operator=(const olist_pos<T>& l);
  olist_pos(olist<T>& l);
  olist_pos(const olist<T>& l);
  
  // Returns 1 if this olist_pos is actually on the olist it thinks it is.
  bool reallyOnlist() const;

  // Operations
  bool operator<(const olist_pos<T>& b) const;
  bool operator==(const olist_pos<T>& b) const;
  

  // Destroying the olistpos doesn't do anything to the olist
  ~olist_pos(void);

  // get the element at this position.
  // We provide two names because (A) each made sense in
  // some phrasing, and (B) we couldn't think of a good
  // single name when we wrote that part of the code.
  // Bad style, really.
  // These functions blow up if the olist_pos is offlist.
  T here() const;
  T item() const { return this->here(); };

  // Get back to the olist that this olist_pos points to.  
  olist<T>* mylist() const {return this->tolist;};

  // The main test is, does the olist_pos point somewhere
  // on the olist.  The guarantee is,
  // if it is onlist, then it's here() will work right.
  // e.g., if mylist is empty, mylist.head() is offlist().
  // Note also that if lp points to an element of a olist, 
  // and that element is deleted, lp will still register as
  // onlist.
  bool offlist() const {return (this->cursor ? false : true);}
  bool onlist() const {return  (this->cursor ? true : false);}
  
  // so a olist_pos can be used just like a pointer in a for loop
  operator int () const  { return onlist();};
  const olist_pos<T>& operator ++ (int) {goNext(); return *this;};
  olist_pos<T>& operator ++ () {goNext(); return *this;};
  // return true if *this will refuse to allow modifying
  // operations on the underlying olist to be used through it. 
  bool isConst() const {return this->isConstPos;};

  // Is the cursor at the head of the olist (empty or not)?
  // if this->atHead() is true, this->insertPrev(t) will
  // make t the new head element of the olist.
  // For example, if the olist is empty, its olistPos's are atHead.
  bool atHead() const;

  // Dual to atHead();
  bool atTail() const;

// MOVING THE LISTPOS AROUND
  
  // go forward or backward in the olist.
  void goNext();
  void goPrev();

// MUTATING THE LIST ITSELF

  // Insert an element t after the current position.
  // Leave this olist_pos pointed at the new element.
  // I'm not sure what the right behavior should be when
  // *this is off the olist; so it causes an error.
  // However, when the olist is empty, this adds a single element to it.
  void insertNext(T t);

  // Insert t before the current position; leave *this
  // pointing to new position.
  // Like insertNext, mutatis mutandis.
  void insertPrev(T t);


  void replaceWith(T t);
  // replace referenced olist element with t.
  // Doom, if the olist_pos doesn't refer to an element
  // of a non-constant olist.

  void delete0();
  // Delete the node of the olist *this is pointing to.
  // Leaves *this pointing to nowhere; that's what the 
  // "0" in "delete0" means.  
  
  // WARNING: Dangling olist_pos's are dangerous: 
  // if you delete a olist_pos to element x, you might
  // have other olist_pos's to x -- and that's trouble.
  // MISSING: isReallyOnlist() -- scan olist and make sure 
  // that *this is a olist_pos to somewhere on the olist.
  
  void deleteAndGoNext();
  // like delete0, but moves to the next olist_pos.

  // Returns a olist consisting of this through b.
  olist<T> through(const olist_pos<T> b) const;

  
private:
friend class olist<T>;

  olist_pos(olist<T> *l, olist_node<T> * n, bool consty);
  // Construct a olist_pos to node n of olist l, with
  // constant-ness consty.
  // Use olist_pos<T> lp = l.head()   (or l.tail() if that's what you mean)
  // for internal use only.


  olist_node<T> *cursor;
  olist<T> *tolist;
  bool isConstPos; // True iff the olist_pos should not be used for mutators.
};  


#endif

  
