/* @(#) ./sparse/spmat.h 07/23/93 */

/*
   New sparse structure.

   Goals: flexible, object-oriented style.  The basic structure
   contains some publically visable data and a pointer to various
   storage formats.  

   This includes the size of an element; this can be used to allow the
   routines to work with blocks of numbers as the basic unit (for example,
   block incomplete factorizations, or complex matrices).  In these cases,
   the basic arithmetic operations must be replaced with their block
   counterparts.

   New and Improved! data-structure neutral format includes scatter/gather
   for rows of the matrix.  
 */

#include "sparse/pool.h"

#ifndef _SPARSE

/* index mapping 
   WARNING: This structure is under construction.  It may change.
 */
typedef struct _Mapping {
    int *rowmap,    *colmap,    *icolmap;
    int q_user_row, q_user_col, q_user_icol;
    } Mapping;

/* This structure contains the operations that we export outside of any 
   choice of data structure.  Note that this structure will increase in
   size as we add operations.  Be sure that you update ALL of the create
   routines when changing this structure. 

   WARNING: This structure is under construction.  It WILL change, and
   frequently.

   NOTE: This structure is optimized for operations related to solving
   sparse linear systems, and includes routines for relaxation methods,
   even though those could be implemented through regular solve followed
   by a scaling.

   We may want to add Gauss-Seidel smoothing routines to SpOps eventually.
 */
typedef struct {
    /* Matrix - vector operations */
    void (*mv)(),               /* u = Mv */
         (*mvpv)(),             /* u = Mv + w */
         (*mtv)(),              /* u = M'v (M-transpose*v) */
	 (*solve)(),            /* u = M^{-1} v */
         (*solveTranspose)(),   /* u = M^{-T} v */
         (*TsolveRelaxL)(),     /* u = omega L^{-1} v */
         (*TsolveRelaxU)(),     /* u = omega U^{-1} v */
         (*destroy)();
    void *(*create)(), *(*createsplit)();

    /* Matrix factorization operations */
    int  (*incompleteFill)(), (*fill)(), (*factor)();

    /* Matrix Subsetting operations */
    void *(*subset)(), *(*subsetrows)();

    /* Get/set elements of a matrix */
    void (*gathertorow)(),      /* (row, i[j],v[j]) -> inserts row in matrix */
         (*gatheraddtorow)(),   /* as gather, but adds into matrix */
         (*scatterfromrow)();   /* extract a row into a vector of column
				   numbers and a vector of element values */
    void (*lowersize)();        /* set an integer vector with the number of
				   elements in the lower triangle (not counting
				   the main diagonal).  Can be used to "split"
				   the matrix */

    /* Matrix - matrix operations */
    void *(*mm)();          /* A = (? b ) B * C ( ? + a D ) */
    
    /* These values characterize the behavior of the scatter/gather
       operations */
    int  ScatterGivesCols,       /* 1 if actual data */
         ScatterGivesVals,       /* 1 if actual data */
         ColOffset;              /* index of 1st column in datastructure */
    } SpOps;
/* For now, the known formats are in tools/sparse/:
   aij
   block
   default
   dense
   diag
   row

   We need to add
   ija
 */

#if defined(NO_GENERIC_SPARSE)
extern void SpMult(), SpMultAdd(), SpMultTrans();
extern int  SpComputeFill(), SpComputeFactor();
extern void SpSolve(), SpSolveTrans();
extern void SpDestroy();
extern void SpScatterFromRow(), SpGatherToRow(), SpGatherAddToRow();
#else
/* 
   If we use the macros, we can't access the functions directly (we
   need a different macro for the function and the application of the function)
   For now, code that needs the function itself must use NO_GENERIC_SPARSE;
   we need to fix this eventually.
 */
#define SpMult(m,vin,vout) (*(m)->ops->mv)( m, vin, vout )
#define SpMultAdd( m, v, v1, vout ) (*(m)->ops->mvpv)( m, v, v1, vout )
#define SpMultTrans( mat, v, vout ) (*(mat)->ops->mtv)( mat, v, vout )
#define SpComputeFill( m, mb ) (*(m)->ops->fill)( m, mb )
#define SpComputeFactor( m, mb ) (*(m)->ops->factor)( m, mb )
#define SpSolve( m, b, x ) (*(m)->factor->ops->solve)( m, b, x )
#define SpSolveTrans( BB, b, x ) \
    (*(BB)->factor->ops->solveTranspose)( BB, b, x )
#define SpTsolveRelaxL( BB, omega, b, x ) \
    (*(BB)->factor->ops->TsolveRelaxL)( BB, omega, b, x )
#define SpTsolveRelaxU( BB, omega, b, x ) \
    (*(BB)->factor->ops->TsolveRelaxU)( BB, omega, b, x )
#define SpDestroy( m ) (*(m)->ops->destroy)( m )
#define SpGatherToRow( mat, row, nz, i, v ) \
    (*(mat)->ops->gathertorow)( mat, row, nz, i, v )
#define SpGatherAddToRow( mat, row, nz, i, v ) \
    (*(mat)->ops->gatheraddtorow)( mat, row, nz, i, v );
#define SpScatterFromRow( mat, row, nz, i, v ) \
    (*(mat)->ops->scatterfromrow)( mat, row, nz, i, v );
#define SpLowersize( mat, idx ) \
    (*(mat)->ops->lowersize)( mat, idx )
#define SpMM( mat1, mat2 ) \
    (*(mat1)->ops->mm)( mat1, mat2 )
#endif

/* Initial definitions */
/*M
   SpCreate - Allocates an n x m sparse matrix (row format).

   Input Parameters:
.   n - number of rows
.   m - number of columns
.   mmax - estimated number of elements in each row (this many elements
          will be pre-allocated; this value may be set to zero, in which
	  case space will be allocated as required.)

    Synopsis:
    SpMat *SpCreate( n, m, mmax )
    int  n, m, mmax;
M*/
#define SpRCreate SpCreate

typedef struct {
    int    type;              /* type of representation */
    SpOps  *ops;              /* Pointer to structure of operations */
    int    rows, cols;        /* size of matrix */
    int    is_sorted;         /* are entries stored in increasing row/column */
    int    element_length;    /* number of bytes in an element */
    int    nz;                /* number of non-zero elements (if < 0,
    				 the value is unknown) */
    SPPool pool;              /* Storage pool */
    int    alloc_together;    /* true if column indices and data are
                                 allocated together, false otherwise.
                                 See discussion below */
    int    alloc_incr;        /* number of elements to allocate when extending
    				 a row */
    struct _Mapping *map;     /* Index mapping, if any */
    void   *data;             /* Representation */
    } SpMat;

/* Split matrix.  This is primarily used for factors or approximate
   factors.  Because it is usually prepared by an Sp routine, and because 
   the cost of determining the number of floating point operations performed
   can be high (perhaps requiring a symbolic re-factorization), we provide
   a place to store the values here.  If the value is negative, the 
   appropriate cost function should be called.
*/
typedef struct {
    int   type;
    SpMat *factor;
    int   *nzl;         /* Number of non-zeros in lower triangle
                                  (not counting diagonal) */
    int   nf;           /* Number of flops used in preparing the split 
			   matrix */
    } SpMatSplit;


/* Pivoting strategies.  PIVOT_ACTIVE not yet implemented */
typedef enum { PIVOT_NONE, PIVOT_PRE_SYM, PIVOT_PRE_NONSYM, PIVOT_ACTIVE }
    PIVOTTYPE;
/* Here are the available orderings. */
typedef enum { ORDER_RCM, ORDER_QMD, ORDER_ND, ORDER_1WD } SPORDERTYPE;
/* The refined quotient tree does not seem to work */
/* #define ORDER_RQT 4 */
extern void SpOrderRCM();
extern void SpOrderQMD();
extern void SpOrderND();
extern void SpOrder1WD();

/* routines related to ordering */
extern void SpOrderRegister();
extern void SpOrderRegisterAll();
extern void SpOrderRegisterDestroy();

/* Routine declarations */
/* Allocation: */
extern SpMat *SpCreate(), *SpUnPack();
extern SpMatSplit *SpCreateSplit();
extern void SpExtend();
extern struct _Mapping *SpCreateMap();

void SPMallocNV();

/* Manipulate Elements */
extern void SpSetInRow(), SpAddInRow(), SpZeroRows(), SpAdd(), SpAddGeneral(),
            SpSetGeneral(), SpAddSubmatrix(), SpSetSubmatrix(), SpAddValue(),
            SpSupport();

/* Subsets */
extern SpMat *SpSubset(), *SpSubsetSorted(), 
             *SpCondense(), *SpCopy(), *SpTranspose();
extern SpMat *SpSubMatrixInPlace();
extern SpMat *SpRowSubMatrixInPlace();

/* Mappings */
extern void SpSetMappingPtrs(), SpCopyMappings(), SpComposeMappings();

/* Scalings */
extern void SpScaleRowsByDiag(), SpScaleColsByDiag(), SpScaleSym();

/* Matrix-Vector multiply */
extern void SpMultTransposeAdd();

/* Factor and Solve */
extern int        SpComputeILUFill();
extern void       SpSolveAdd();
extern SpMatSplit *SpQuickFactor();
extern void       SpCopyStructure();
extern SpMatSplit *SpICC(), *SpICCJP();

/* Values */
extern double SpGetValue();
extern void   SpGetDiagonal();

/* Info */
extern int  SpGetNz(), SpGetSpaceUsed(), SpWalk();
extern char *SpOrderName();
extern void SpOrder(), SpPrint();

/* Costs */
extern int SpCostFactor(), SpCostSolve(), SpCostMult();

/* Conversions */
extern int   SpToAIJ();
extern SpMat *SpFromAIJ();
extern SpMat *SpFromIJA();
extern void SpWrite();
extern SpMat *SpRead(), *SpReadRows(), *SpReadRowsAIJ();
extern void SpToDiagonalAijSymmetric();
extern SpMat *SpFromDiagonalAijSymmetric();
extern SpMat *SpBToBlk();
extern SpMatSplit *SpSplitFromMat();
extern void SpChunckPrintUsage();

/* Other create routines.
   (If they have their own includes, include that include here) */
extern SpMat *SpAIJCreateFromData(), *SpAIJCreate();

/* Error values */
#define ERR_NO_MAP 0x101
#define ERR_NOT_VALID_WITH_MAP 0x102
#define _SPARSE
#endif

/*
    Allocation controls.  There are lots of paramters to tune here; the
    issues include the cost of a "malloc", whether to allocate column
    indices with the contents of those entries, and how much "fuzz"
    will be allocated (unit of allocation).  These continue to change
    as we discover various problems with different approaches (there is
    the classic time/space tradeoff.  The "pool" system provides
    efficient (in time) allocation by allocating large blocks and
    not trying hard to free them until the matrix is destroyed.  For some
    applications, space, not time, is the constraint; for these, the
    pool system is not appropriate.
 */    
