#ifndef lint
static char SCCSid[] = "@(#) ./sparse/solve/sltrm.c 07/23/93";
#endif

/*
    This file contains a simple sparse solve routine for solving the
    problem A^T x = b.  This is done by using the factors of A:
    A = LU; then A^T = U^TL^T and thus the solution is arrived at by
    solving U^T y = b followed by L^T x = y .  Because row storage is
    used for the regular factors, column storage is used for the transposes.

    This routine works on a block of vectors stored across vectors
    (That is, x[i*nv+j] is the i'th element of the j'th vector.
 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
#include "inline/spops.h"

/*@ 
  SpSolveTransM - Solve a system of equations using the TRANSPOSE of
                 a factored matrix

  Input Parameters:
.  BB - Split matrix already processed by Factor
.  nv - number of vectors to solve for
.  b  - Right-hand-side
.  x  - solution
 @*/
int SpSolveTransM( BB, nv, b, x )
SpMatSplit *BB;
int        nv;
double     *b, *x;
{
int err;
if (BB->factor->map) 
    err = 1;
else
    err = SpiSolveBaseTransM( BB, nv, x, b );
return err;
}

/*
   This is a solve that does not involve any mappings 
 */
int SpiSolveBaseTransM( BB, nv, x, b )
SpMatSplit  *BB;
int         nv;
double      *x, *b;
{
int      i, k, j, n, *nzs;
double   *vv;
register double       *v, *alpha, *bb, *xx, term;
register int          *vi, nz;
SpMat    *B = BB->factor;
SpVec    *row, **rs;
SpRowMat *R = (SpRowMat *)B->data;

n      = B->rows;
/* forward solve the lower triangular (actually the transpose of the
   upper triangular factor) */
nzs    = BB->nzl;
rs     = R->rs;
for (i=0; i<n; i++) {
    row  = *rs++;
    nz   = *nzs++;
    v    = row->v + nz;
    vi   = row->i + nz;
    nz   = row->nz - nz;
    alpha = x + i*nv;
    bb    = b + i*nv;
    for (k=0; k<nv; k++)
        alpha[k] = bb[k] * v[0];
    v++; vi++; nz--;
    /* Really should be: (it would be nice to eliminate these integer
       multiplies; also, perhaps we should insist for this routine
       that b == x. Also, to avoid the integer multiplies, we might
       want a different algorithm.  Barry's code unrolled this */
    for (k=0; k<nz; k++) {
    	bb   = b + vi[k] * nv;
    	term = - v[k];
        for (j=0; j<nv; j++) {
            bb[j] += term * alpha[j];
            }
    	}
    }
/* backward solve the upper triangular (actually the transpose of the
   lower triangular factor) */
nzs--; rs--;
for (i=n-1; i>=1; i--) {
    nz   = *nzs--;
    row  = *rs--;
    vv = v = row->v;
    vi   = row->i;
    alpha= x + i*nv;
    /* This is really a block sparse axpy */
    for (k=0; k<nz; k++) {
    	xx   = x + vi[k] * nv;
    	term = - v[k];
        for (j=0; j<nv; j++) {
            xx[j] += term * alpha[j];
            }
    	}
    }
return ERR_NONE;
}
