#ifndef lint
static char SCCSid[] = "@(#) ./sparse/qdir.c 07/23/93";
#endif

/*
   This file contains a simple routines for factoring a linear system
   given the matrix, optionally a method of ordering and optionally 
   a level of fill. 

 */

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"
/*@
    SpQuickFactor - Factor a matrix with a given ordering and level of fill.

    Description:
    Given a matrix, an optional level of fill and ordering an
    optional ordering, ORDER_ND, ORDER_RCM, ORDER_RQM, etc.
    returns the factored matrix. Defaults to complete 
    factorization and nested dissection ordering.
     
    Sometimes fails for nonsymmetric problems.

    Input Parameters:
.   mat   - matrix to factor
.   fill  - level of fill.  -1 for complete factor
.   ordering - ordering method to use.  -1 for no ordering
.   threshold - threshold to use for block factorization    
.   pivoting  - type of pivoting (if any) to use

    Notes:
    This works for dense matrices created with SpDnCreate or 
    SpDnCreateFromData.  In that case, LAPACK routines are used without
    ordering or selected pivoting.  The threshold parameter is ignored
    for this case as well.
 @*/
SpMatSplit *SpQuickFactor( mat, fill, ordering, threshold, pivoting )
SpMat     *mat;
int       threshold, fill, pivoting;
SPORDERTYPE ordering;
{
  SpMatSplit *factor; 
  int        i,*rperm,*cperm,*iperm, n = mat->rows;

  factor = SpCreateSplit( mat, 0 ); CHKERRV(0,0);
  
  /* compute the symbolic factorization */
  if (fill >= 0) {
    SpComputeILUFillNoMap( mat, factor, fill ); CHKERRV(0,0);
  }
  else { 
    if (mat->type != MATDENSE) {
	/* Order the matrix */
	rperm = (int *) MALLOC(n*sizeof(int)); CHKPTRV(rperm,0); 
	cperm = (int *) MALLOC(n*sizeof(int)); CHKPTRV(cperm,0); 
	iperm = (int *) MALLOC(n*sizeof(int)); CHKPTRV(iperm,0); 
	if ((int)ordering != -1) { SpOrder(mat,ordering,rperm,iperm);}
	else { for ( i=0; i<n; i++ ) rperm[i] = iperm[i] = i; }
	MEMCPY(cperm,rperm,n*sizeof(int));
	
	/* Re-ordering code does not yet work if initial ordering is not
	   the identity */
	if ((int)ordering == -1 && 
	    (pivoting == (int)PIVOT_PRE_SYM || 
	     pivoting == (int)PIVOT_PRE_NONSYM)) {
	    int *irperm;
	    double atol = 0.0;  /* Drop only exactly zero diagonal elements */
	    
	    irperm   = (int *)MALLOC( n*sizeof(int) ); CHKPTRV(irperm,0);
	    for (i=0; i<n; i++) cperm[i] = rperm[i];
	    switch (pivoting) {
		case PIVOT_PRE_SYM:
		SpSymmetricReorderForZeroDiagonal( mat, atol, rperm, cperm );
		break;
		case PIVOT_PRE_NONSYM:
		SpUnSymmetricReorderForZeroDiagonal( mat, atol, rperm, cperm );
		break;
		}
	    SpInverse( n, cperm, iperm );
	    /* SpInverse( n, rperm, irperm ); */
	    FREE( irperm );
	    }
	SpiSetMappingPtrs(factor->factor,rperm,cperm,iperm); CHKERRV(0,0);
	}
    SpComputeFill( mat, factor ); CHKERRV(0,0);
  }

  /* factor the matrix */
  if (mat->type != MATDENSE && threshold) {
     SpFindBlocks( factor->factor, threshold );  
     }
  SpComputeFactor( mat, factor ); 
  return factor;
}
