#ifndef lint
static char SCCSid[] = "@(#) ./sparse/ilu.c 07/23/93";
#endif

#include "tools.h"
#include "sparse/spmat.h"
#include "sparse/sppriv.h"

/*
   Routines to compute an ilu factorization from a given matrix
   One way is to just use the SparseFactorCompute routine, since
   non-existent fill will just be ignored in rtmp (rather than
   checking each value for an entry in rtmp).  The only possible
   problem is overflow or underflow on the values that will be
   discarded.
 */

/*@
   SpComputeILUFill - Compute ILU fill. Find storage for factor of
                      matrix B in BB (descriptors allocated but
                      not set). BB should have been obtained previously
                      by SpCreateSplit(). After this call, the numeric 
                      factorization can be done with SpComputeFactor().

   Input Parameters:
.  B    - matrix being factored
.  BB   - matrix to contain factor
.  level - level of fill.
 @*/
int SpComputeILUFill( B, BB, level )
SpMat       *B;
SpMatSplit  *BB;
int         level;
{
int     prow, row, *fill, *xi, nz, nzi, n, fm, nnz, idx, m, nzf, 
        *nzl, err, *im, *flev, incrlev;
double  *rtmp;
SpMat   *BBf = BB->factor;
SpVec   *pivot_row, *elim_row, **rs, **frs;
int     *r = BBf->map->rowmap, *ic = BBf->map->icolmap;
SpRowMat *R = (SpRowMat *)B->data, *RR = (SpRowMat *)BBf->data;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
n    = B->rows;
SPAllocV(n+1,rtmp,fill); CHKPTRV(rtmp,1);
im   = (int *)rtmp;

/* Nothing to do for the first row */
BB->nzl[0] = 0;
rs         = R->rs;
frs        = RR->rs;
nzl        = BB->nzl;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  */
    elim_row  = rs[r[prow]];
    xi        = elim_row->i;
    /* Add the initial row (fill,rtmp) */
    nz        = elim_row->nz;
    nzf       = nz;
    /* May be inserted in any order */
    fill[n] = n;
    while (nz--) {
	fm        = n;
	idx       = ic[*xi++];
	do {
	    m = fm;
	    fm= fill[m];
	    } while (fm < idx);
	fill[m]  = idx;
	fill[idx] = fm;
	im[idx]   = 0;
	}

    nzi = 0;
    /* Eliminate into (fill,rtmp) */
    row = fill[n];
    while (row < prow) {
	incrlev    = im[row] + 1;
	pivot_row  = frs[row];
	nz         = nzl[row];
	/* Get the pivot row */
	xi         = pivot_row->i + nz;
	flev       = ((int *)pivot_row->v) + nz;
	if (*xi++ != row) 
	    {SETERRC(-(row+1),"Zero pivot encountered"); 
             return -(row+1);}    /* zero (missing) pivot */
	flev++;
	nz++;
	nnz     = pivot_row->nz - nz;
	fm      = row;
	while (nnz-- > 0) {
	    idx       = *xi++;
	    if (*flev + incrlev > level) {
		flev++;
		continue;
		}
	    /* find fm such that fill[m] <= idx fill[fill[fm]] */
	    /* elements are ordered */
	    do {
		m  = fm;
		fm = fill[m];
		} while (fm < idx);
	    if (fm != idx) {
		/* insert */
		fill[m]   = idx;
		fill[idx] = fm;
		im[idx]   = *flev + incrlev;
		fm        = idx;
		nzf++;
		}
	    else {
		if (im[idx] > *flev + incrlev) im[idx] = *flev + incrlev;
		}
	    flev++;
	    }
	row   = fill[row];
	nzi++;
	};

    /* Allocate elim_row */
    row      = fill[n];
    elim_row = frs[prow];
    SPiMallocNVt(BBf,nzf,&elim_row->v,&elim_row->i,err); CHKERRV(ERR_NO_MEM,1);
    elim_row->maxn = nzf;
    elim_row->nz   = nzf;

    /* Save the number of non-zeros in the lower triangle */
    nzl[prow] = nzi;
    
    /* Store filled row */
    nnz = nzf;
    xi  = elim_row->i;
    flev= (int *)elim_row->v;
    fm  = row;
    while (nnz--) {
	*xi++ = fm;
	*flev++ = im[fm];
	fm    = fill[fm];
	}
    }

/* Recover temp used to find fill */
SPFreeVv( rtmp );
return ERR_NONE;
}

/*@
   SpComputeILUFillNoMap - Compute ILU fill for a matrix without 
                           mappings.Find storage for factor of
                           matrix B in BB (descriptors allocated but
                           not set). BB should have been obtained previously
                           by SpCreateSplit(). After this call, the numeric 
                           factorization can be done with SpComputeFactor().

   Input Parameters:
.  B    - matrix being factored
.  BB   - matrix to contain factor
.  level - level of fill.
 @*/
int SpComputeILUFillNoMap( B, BB, level )
SpMat       *B;
SpMatSplit  *BB;
int         level;
{
int     prow, row, *fill, *xi, nz, nzi, n, fm, nnz, idx, m, nzf, 
        *nzl, err, *im, *flev, incrlev;
double  *rtmp;
SpMat   *BBf = BB->factor;
SpVec   *pivot_row, *elim_row, **rs, **frs;
SpRowMat *R = (SpRowMat *)B->data, *RR = (SpRowMat *)BBf->data;

/* Allocate temporaries to compute fill in; row to compute values of fill in */
n    = B->rows;
SPAllocV(n+1,rtmp,fill); CHKERRV(ERR_NO_MEM,1);
im   = (int *)rtmp;

/* Nothing to do for the first row */
BB->nzl[0] = 0;
rs         = R->rs;
frs        = RR->rs;
nzl        = BB->nzl;
for (prow=0; prow<n; prow++) {
    /* Update row "prow" using all rows that are needed to eliminate
       entries on this row.  */
    elim_row  = rs[prow];
    xi        = elim_row->i;
    /* Add the initial row (fill,rtmp) */
    nz        = elim_row->nz;
    nzf       = nz;
    /* May be inserted in any order */
    fill[n] = n;
    while (nz--) {
	fm        = n;
	idx       = *xi++;
	do {
	    m = fm;
	    fm= fill[m];
	    } while (fm < idx);
	fill[m]  = idx;
	fill[idx] = fm;
	im[idx]   = 0;
	}

    nzi = 0;
    /* Eliminate into (fill,rtmp) */
    row = fill[n];
    while (row < prow) {
	incrlev    = im[row] + 1;
	pivot_row  = frs[row];
	nz         = nzl[row];
	/* Get the pivot row */
	xi         = pivot_row->i + nz;
	flev       = ((int *)pivot_row->v) + nz;
	if (*xi++ != row) 
	    {SETERRC(-(row+1),"Zero pivot encountered"); 
             return -(row+1);}     /* zero (missing) pivot */
	flev++;
	nz++;
	nnz     = pivot_row->nz - nz;
	fm      = row;
	while (nnz-- > 0) {
	    idx       = *xi++;
	    if (*flev + incrlev > level) {
		flev++;
		continue;
		}
	    /* find fm such that fill[m] <= idx fill[fill[fm]] */
	    /* elements are ordered */
	    do {
		m  = fm;
		fm = fill[m];
		} while (fm < idx);
	    if (fm != idx) {
		/* insert */
		fill[m]   = idx;
		fill[idx] = fm;
		im[idx]   = *flev + incrlev;
		fm        = idx;
		nzf++;
		}
	    else {
		if (im[idx] > *flev + incrlev) im[idx] = *flev + incrlev;
		}
	    flev++;
	    }
	row   = fill[row];
	nzi++;
	};

    /* Allocate elim_row */
    row      = fill[n];
    elim_row = frs[prow];
    SPiMallocNVt(BBf,nzf,&elim_row->v,&elim_row->i,err); CHKERRV(ERR_NO_MEM,1);
    elim_row->maxn = nzf;
    elim_row->nz   = nzf;

    /* Save the number of non-zeros in the lower triangle */
    nzl[prow] = nzi;
    
    /* Store filled row */
    nnz = nzf;
    xi  = elim_row->i;
    flev= (int *)elim_row->v;
    fm  = row;
    while (nnz--) {
	*xi++ = fm;
	*flev++ = im[fm];
	fm    = fill[fm];
	}
    }

/* Recover temp used to find fill */
SPFreeVv( rtmp );
return ERR_NONE;
}


