
/*
     Test routine for SVctx library. Runs two examples 

       1) Convection diffusion with upwind differencing 
          use -l value (where value > 0.0) to increase convection.

       2) Three dimensional linear elasticity on unit cube;
          use -e for this problem.

       Use -n integer (where integer > 2) to set the mesh refinement.

       The relative convergence tolerance for iterative methods may 
       be set with -rtol value (1.0e-12 is the default)
   example -n 30
   example -n 30 -l 3.0
   example -n 5  -e

*/
#include "tools.h"
#include "system/system.h"
#include "sparse/spmat.h"
#include "solvers/svctx.h"
#include <math.h>
#include <stdio.h>

SpMat  *CD2d();
double *FormSolution();
double ErrorNorm();

void ITXMonitor();
void ITXMonitorRVal();

main(argc,argv)
int  argc;
char **argv;
{
  SpMat    *mat, *cmat;
  int      cx, cy;
  SVctx    *ctx;
  double   *x,*solution, *b, omega;
  int      M = 3,N,its,i, sf, ndiv;
  int      iluFill = 0;
  double   iluDrop = 0.0;
  double   err,start,end,lambda = 0.0, setup, rtol = 1.0e-12;
  ITMETHOD method, itinner;
  SVMETHOD svmethod, svinner;
  int      innerMaxit;
  char     svname[50], itbuf[50], orbuf[50], title[80];
  int      do_display = 0, do_res = 0;
  int      noheader, Use2dDecomp = 0;
  SPORDERTYPE ordering;
  SVDB     *sd;

  /* parse the input arguments */
  if (SYArgHasName( &argc, argv, 1, "-help" )) {
      fprintf( stderr, "%s - run example programs for solvers. The method \n",
	       argv[0] );
      fprintf( stderr, "can be selected with -sv <methodname>\n" );
      fprintf( stderr, "Other options include:\n" );
      fprintf( stderr, 
	       "-itmethod <iterative> -rtol <relativetol> -dres\n" );
      fprintf( stderr, 
	       "[-itinner... for inner iterations, including -maxit n]\n" );
      fprintf( stderr, 
	       "-dres for graph of residual\n" );
      fprintf( stderr, 
	       "-dresv for display of residual value\n" );
      fprintf( stderr, 
	       "-2d for 2d decomposition in bdd or osm\n" );
      fprintf( stderr, 
	       "-2dovlp for 2d definition of overlap domains in osm\n" );
      exit(0);
      }
  /* Look for solver database */
  sd = SVDBOpen( &argc, argv );
  title[0] = 0;
  SYArgGetString( &argc, argv, 1, "-title", title, 80 );

  SYArgGetInt( &argc, argv, 1, "-n", &M );
  ndiv = M;
  SYArgGetInt( &argc, argv, 1, "-ndiv", &ndiv );

  Use2dDecomp = SYArgHasName( &argc, argv, 1, "-2d" );

#ifdef USEX
  do_display = SYArgHasName( &argc, argv, 1, "-dres" );
  do_res     = SYArgHasName( &argc, argv, 1, "-dresv" );
  if (do_res) {
      ITXMonitorMeshSize( M, M );
      }
#endif
  noheader   = SYArgHasName( &argc, argv, 1, "-noheader" );
  SYArgGetDouble( &argc, argv, 1, "-l", &lambda );
  SYArgGetDouble( &argc, argv, 1, "-rtol", &rtol );
  svmethod = SVLU;
  strcpy( svname, "lu" );
  SYArgGetString( &argc, argv, 0, "-sv", svname, 50 );
  SVGetMethod( &argc, argv, "-sv", &svmethod );

#if defined(USEX)
  /* Process X-specific arguments (-geometry) */
  XBQGetArgs( &argc, argv, 1 );
#endif

  /* Set the default method choice */
  if (lambda != 0.0 || svmethod == SVILU || svmethod == SVOSM) {
      strcpy( itbuf, "gmres" );
      method = ITGMRES; 
      }
  else {
      strcpy( itbuf, "cg" );
      method = ITCG;
      }

  SYArgGetString( &argc, argv, 0, "-itmethod", itbuf, 50 );
  ITGetMethod( &argc, argv, "-itmethod", &method );
  if (svmethod != SVLU) {
      strcat( svname, " " );
      strcat( svname, itbuf );
      }

  /* Inner iteration parameters for BDD and OSM */
  svinner = SVLU;
  SVGetMethod( &argc, argv, "-svinner", &svinner );
  itinner = ITGMRES;
  ITGetMethod( &argc, argv, "-itmethod", &itinner );
  innerMaxit = -1;
  SYArgGetInt( &argc, argv, 1, "-maxit", &innerMaxit );

  SYArgGetInt( &argc, argv, 1, "-ilufill", &iluFill );
  SYArgGetDouble( &argc, argv, 1, "-iludrop", &iluDrop );

  /* for symmetric problems use CG otherwise GMRES */
  if (lambda != 0.0) {
      if (svmethod == SVICC || svmethod == SVICCJP) exit(0);
      }

  ordering = ORDER_ND;
  if (svmethod == SVLU) {
      if (SYArgGetString( &argc, argv, 0, "-ordering", orbuf, 50 )) {
	  SpGetOrdering( &argc, argv, "-ordering", &ordering );
	  strcat( svname, " " );
	  strcat( svname, orbuf );
	  }
      }
  if (title[0] == 0) 
      strcpy( title, svname );

#if !defined(rs6000)
    SYSetFPTraps();
#endif


  /* for model 2D grid problems; this is good relaxation factor for SOR */
  if (lambda == 0.0)
       omega = 2.0/(1.0 + sin(3.14159265358979/(M + 1.0)));
  else omega = 1.20;

  /* Generate matrix */
  mat  = CD2d(M,M,lambda);                 CHKERR(1);
 
  N           = mat->rows;
  solution    = FormSolution(N);                           CHKERR(1);
  x = (double *) MALLOC(N*sizeof(double));                 CHKPTR(x);
  b = (double *) MALLOC(N*sizeof(double));                 CHKPTR(x);

  /* given a "solution", generate the right hand side */
  SpMult(mat,solution,b);                                  CHKERR(1);

if (!noheader) {
  printf("                Matrix size %d \n",N);
  printf("    Method                 Error        Setup       Time      Speed         Its\n");
  printf("------------------------------------------------------------------------------\n");
  }
  start = SYGetCPUTime(); 
  ctx   = SVCreate(mat,svmethod);              CHKERR(1);

  /* Options */
  SVSetLUOrdering(ctx,ordering);
  SVSetBDDDomainsNumber(ctx,ndiv);
  SVSetOSMDomainsNumber(ctx,ndiv);
  /* This is for the alternate multiplicative method */
  SVSetOSMDefaultUseIntersections(ctx, 
				  SYArgHasName( &argc, argv, 1, "-new" ) );

  /* Control inner iterations for OSM and BDD */
  SVSetOSMDefaultMethod(ctx,svinner);
  SVSetOSMDefaultAccelerator(ctx,itinner);
  SVSetOSMDefaultMaxIt(ctx,innerMaxit);
  SVSetBDDDefaultMethod(ctx,svinner);
  SVSetBDDDefaultAccelerator(ctx,itinner);
  SVSetBDDDefaultMaxIt(ctx,innerMaxit);

  /* Coarse grid problem.  We need more information, including the 
     decomposition to use.  One requirement is that the result of
     using the coarse-grid problem produces a result (after applying
     the restriction and interpolation operators) that is scaled the
     same way as the original problem.  Also note that for now,
     there are no automatic choices of restriction/interpolation operator.
     These must be chosen to fit the original problem.  For FE methods,
     these should usually be the test functions; for FD methods, a common
     choice is linear or bilinear interpolation.  SLES provides some sample
     routines for some common special cases.

     Note that these requirements are no more stringent that required by
     multigrid.
   */

  SVSetILUFill(ctx,iluFill);
  SVSetILUDropTol(ctx,iluDrop);

  cmat = 0;
  cx   = 2; /* ndiv */
  cy   = 2;
  SYArgGetInt( &argc, argv, 1, "-cx", &cx );
  SYArgGetInt( &argc, argv, 1, "-cy", &cy );
  if (SYArgHasName( &argc, argv, 1, "-osmglobal" ) )
      cmat = CD2d( cx-1, cy-1, lambda );
  SVOSM2dSetup( ctx, M, M, 1, cx, cy, cmat, 1 );

  if (Use2dDecomp) {
      SVSetBDDRegularDomains2d( ctx, M, M, 1 );
      }
  /* SVSetOSMDefaultMethod(ctx,SVILU); */
  SVSetAccelerator(ctx,method);
  /* SVSetSSOROmega(ctx,omega); */

  SVSetUp(ctx);                                CHKERR(1);
  /* Msh2dDumpDomains( ctx ); */

#ifdef USEX
  XBQGetArgs( &argc, argv, 1 );
  if (do_display) {
      SVSetMonitor(ctx,ITXMonitor,(void *)0);
      ITXMonitor(0,0,-2,0.0); 
      ITXMonitorLimits( 0, 1.0e-10 );
      }
  else if (do_res) {
      SVSetMonitor(ctx,ITXMonitorRVal,(void *)0);
      XBQContourSaveMapping( 1 );
      XBQContourSetMapping( 32, -0.1, 1.1 );
      ITXMonitorRVal(0,0,-2,0.0); 
      }
#endif

  SVSetIts(ctx,N);
  SVSetRelativeTol(ctx,rtol);
  SVSetGMRESRestart(ctx,2*M);

  {double *r = (double *)MALLOC( N * sizeof(double) );
  SVSaveResidualHistory( ctx, r, N );
  }

  setup                    = SYGetCPUTime()-start;
  SVGetFlops(ctx,sf);
  its = SVSolve(ctx,b,x);    end = SYGetCPUTime()-start;   CHKERR(1);
  err = ErrorNorm(N,solution,x);
  PrintInfo( ctx, title, err, setup, end, its, sf );
  /* Output database info, if requested */
  SVDBHeader( sd, ctx, (char *)0 );
  SVDBHistory( sd, ctx, (char *)0 );
  SVDBClose( sd );
  /* Recover space in solver context */
  SVDestroy(ctx);                                          CHKERR(1);

  SpDestroy(mat);

#ifdef USEX
  if (do_res) {
      XBQContour( x, (double *)0, (double *)0, M, M, 32 );
      getc( stdin );
      }
  else {
      if (SYArgHasName( &argc, argv, 1, "-wait" ))
	  getc( stdin );
      }
#endif
return 0;
}
/* ---------------------------------------------------------------- */
/* 
     Upwind finite difference discretization for
	  U_xx + U_yy + lambda U_x

     This assumes homogenous Dirichlet boundary conditions (or that the 
     boundary conditions have been folded into the right-hand-side).
*/
SpMat *CD2d( mx, my, lambda )
int    mx, my;
double lambda;
{
  SpMat *f;
  int    n, i, j, row;
  double hinvx,hinvx2,hinvy,hinvy2;

  n   = mx * my;
  f   = SpCreate( n, n, 5 );

  hinvx = 1.0/(mx + 1.0);
  hinvx2 = hinvx*hinvx;
  hinvy = 1.0/(my + 1.0);
  hinvy2 = hinvy*hinvy;

  for (j=0; j<my; j++) {
    for (i=0; i<mx; i++) {
      row = i + j * mx;
      if (j > 0) {
        SpAddValue( f, -hinvy2, row, row - mx ); CHKERRV(1,0);
      }
      if (i > 0) {
        SpAddValue( f, -hinvx2 - lambda*hinvx, row, row - 1 ); CHKERRV(1,0);
      }
      SpAddValue( f, 2.0*hinvx2 + lambda*hinvx + 2.0*hinvy2, row, row ); 
      CHKERRV(1,0);
      if (i + 1 < mx) {
        SpAddValue( f, -hinvx2, row, row + 1 ); CHKERRV(1,0);
      }
      if (j + 1 < my) {
        SpAddValue( f, -hinvy2, row, row + mx  ); CHKERRV(1,0);
      }
    }    
  }    
  return f;
}
/* ---------------------------------------------------------------- */
/*  
     Builds a random solution 
*/
double *FormSolution(n)
int n;
{
  double *x;
  int    i;
  x = (double *) MALLOC(n*sizeof(double)); CHKPTRV(x,0);
/*   for ( i=0; i<n; i++ ) x[i] = ((double) random())/2147483647.0; */
  for ( i=0; i<n; i++ ) x[i] = 1.0;
  return x;
}
/* ---------------------------------------------------------------- */
/*  
     Returns 2 norm of difference of 2 vectors 
*/
double ErrorNorm(n,solution,x)
double *solution,*x;
int    n;
{
  double err;
  DVaxpy(&n, -1.0, solution,x);
  DVnorm(&n, x, &err);
  return err;
}
  
/* Routine to print output line */
PrintInfo( ctx, name, err, ts, te, its, sf )
SVctx  *ctx;
char   *name;
int    its, sf;
double err, ts, te;
{
int    flops;
double srate, svrate;

SVGetFlops( ctx, flops );

if (strlen(name) > 24) {
    printf( "%s\n", name );
    name[0] = 0;
    }
srate  = 0.0;
svrate = 0.0;
if (ts > 0)
    srate = ((double) sf) / (1.e6 * ts);
if (te > 0)
    svrate = ((double) flops) / (1.e6 * te);
if (its >= 0) 
    printf("%24s %11.3e %11.3e %11.3e %4.1fMf %4.1fMf %d\n",
	   name, err, ts, te, srate, svrate, its );
else
    printf("%24s %11.3e %11.3e %11.3e %4.1fMf %4.1fMf\n",
	   name, err, ts, te, srate, svrate );
}
