#ifndef lint
static char SCCSid[] = "@(#) ./nonlin/min2/dogleg.c 07/23/93";
#endif

#include "tools.h"
#include "nonlin/nlall.h"
#include "nonlin/min2/dogleg.h"
#include <math.h>

#define MIN(a,b) ( ((a)<(b)) ? a : b )
#define SQUR(a) ((a)*(a))

/* 
      dogleg

     This subroutine computes an approximate solution to the problem

          min { || f+J*w|| : ||D*w|| <= delta }

     The subroutine returns the solution p and the reduction redssq
     in the sum of squares.
*/
void NLDogLeg( nlP, usrP, x, f, p, fnorm, delta, trunctol, Gpnorm, Pnorm )
NLCntx  *nlP;
void    *usrP, *x, *f, *p;
double  fnorm, delta, trunctol, *Gpnorm, *Pnorm;
{
void   *s, *g, *w1, *w2;                     /* work vectors */
void   *D;
double qnorm, gnorm, sgnorm, alpha, temp, Jgnorm;
NLStepDogleg *Sctx;
FILE   *fp = (FILE *) nlP->fp;

Sctx = (NLStepDogleg *)nlP->stepCtx;
s    = Sctx->work[0];
g    = Sctx->work[1];
w1   = Sctx->work[2];
w2   = Sctx->work[3];
D    = (void *)Sctx->D;

/* Compute an approximate solution s of Js = -f */
(*Sctx->solve)( Sctx->svctx, f, s );
VSCALE( nlP->vc, usrP, -1.0, s );      

/* Test whether the truncated Newton direction is acceptable. */
VPMULT( nlP->vc, usrP, s, D, w1 );
VNORM( nlP->vc, usrP, w1, &qnorm );  /* | s * D | */
nlP->nvectors += 3;

if (qnorm <= delta) {
    if (fp) fprintf( fp, "Direction is in Trust Region\n" );
    VCOPY( nlP->vc, usrP, s, p );
    VNORM( nlP->vc, usrP, p, Pnorm ); nlP->nvectors += 2;
    *Gpnorm = 0.0;
    return;
    }

if (fp) fprintf( fp, "Direction not in Trust Region; choosing alternate\n" );

/* 
     The truncated Newton direction is not acceptable.
     Next, calculate the scaled gradient direction.
 */
(*Sctx->JTv)( Sctx->J, f, g );     /* g <- J'*f */

/* w1 <- D * g  (point-wise scaling) */
VPDIV( nlP->vc, usrP, g, D, w1 );       

/* 
  Calculate the norm of the scaled gradient and test for
  the special case in which the scaled gradient is zero.
 */
VNORM( nlP->vc, usrP, f, &fnorm );
VNORM( nlP->vc, usrP, w1, &gnorm ); nlP->nvectors += 3;
sgnorm = 0.0;
alpha  = delta / qnorm;
if (gnorm != 0.0) {
    /* Calculate the point along the scaled gradient
       at which the quadratic is minimized. */
    /* w1 <- ( w1 / gnorm ) / D */
    VSCALE( nlP->vc, usrP, (1.0/gnorm), w1 );
    VPDIV( nlP->vc, usrP, w1, D, w1 );       nlP->nvectors += 2;
    (*Sctx->Jv)( Sctx->J, w1, w2 );        /* w2 <- J w1 */

    VNORM( nlP->vc, usrP, w2, &Jgnorm );     nlP->nvectors ++;
    sgnorm = (gnorm/Jgnorm)/Jgnorm;

    /* Test whether the scaled gradient direction is acceptable. */
    alpha = 0.0;
    if (sgnorm <= delta) {
	/* The scaled gradient direction is not acceptable.
           Finally, calculate the point along the dogleg
           at which the quadratic is minimized. */
	temp = (fnorm/gnorm)*(fnorm/qnorm)*(sgnorm/delta);
	temp = temp - (delta/qnorm)*SQUR(sgnorm/delta) +
	              sqrt( SQUR(temp-(delta/qnorm)) +
                               (1.0-SQUR(delta/qnorm)) * 
			       (1.0-SQUR(sgnorm/delta)) );
	alpha = ( (delta/qnorm)*(1.0 - SQUR(sgnorm/delta)) ) / temp;
	if (fp) fprintf( stderr, "Computing dogleg, alpha = %g\n", alpha );
	}
    }

/*
   Form appropriate convex combination of the truncated Newton
   direction and the scaled gradient direction.
 */
temp = (1.0 - alpha) * MIN(sgnorm,delta);
VCOPY( nlP->vc, usrP, w1, p );
VSCALE( nlP->vc, usrP, -temp, p );           nlP->nvectors += 2;
if (alpha != 0.0) {
    VAXPY( nlP->vc, usrP, alpha, s, p );     /* p <- - temp w1 + alpha s */
    }
VDOT( nlP->vc, usrP, s, p, &temp );
/* fprintf( stderr, "p'*s = %lg\n", temp ); */

/* DisplayPlane( usrP, nlP, usrP, x, w1, s, delta, nlP->fun, p ); */
if (Sctx->monitor)
    (*Sctx->monitor)( nlP, usrP, delta, x, f, fnorm, p );

/* Compute the scaled reduction in the sum of squares. */
Sctx->redssq = (1.0 - SQUR(1.0 - alpha)) +
                SQUR(1.0 - alpha)*(((sgnorm*gnorm)/fnorm)/fnorm);

VNORM( nlP->vc, usrP, p, Pnorm );            nlP->nvectors++;

/* If redssq is bad, don't use it.  We may instead wish to flag an error */
if (Sctx->redssq >= 1.0) {
    *Gpnorm = 0.0;
    }
else {
    *Gpnorm = fnorm * sqrt( 1.0 - Sctx->redssq );
    }
#define DEBUG
#ifdef DEBUG
if (fabs(*Pnorm - delta) > 1.0e-10) 
    fprintf( stderr, "step size %g != trust-region size %g\n", *Pnorm, delta );
/* Check redssq */
{
double tmp1;
(*Sctx->Jv)( Sctx->J, p, w2 );        /* w2 <- J p */
VAXPY( nlP->vc, usrP, 1.0, f, w2 );   /* w2 <- f + J p */
VNORM( nlP->vc, usrP, w2, &tmp1 );   
tmp1 = 1.0 - SQUR(tmp1/fnorm);
if (fabs(tmp1 - Sctx->redssq) > 1.0e-10) {
    fprintf( stderr, "redssq = %g != actual value = %g\n", Sctx->redssq, tmp1 );
    }
}

#endif
return;
}

/* 
   Compute the dscaling as the norms of the columns of the Jacobian,
   and setup the solver for applying the inverse of the Jacobian 
 */
void NLDogLegSetup( nlP, usrP, x )
NLCntx *nlP;
void   *usrP, *x;
{
NLStepDogleg *Sctx = (NLStepDogleg *)nlP->stepCtx;
int    i, n = *(int*)usrP;
double *D = Sctx->D;
        
/* Get the jacobian */
(*Sctx->GetJacobian)( nlP, usrP, x );

/* To do this, we need only scatter and add the squares of the rows
   of J into diag, then take the square-root when we are done */
if (Sctx->ScaleJ) {
    (*Sctx->ScaleJ)( nlP, usrP, D );
    }
else {
    for (i=0; i<n; i++) 
	D[i] = 1.0;  /* 0.0 later */
    }
#ifdef FOO
         do j = 1, n
            diag(j) = dnrm2(n,fjac(1,j))
         end
#endif
Sctx->nwork = 4;
Sctx->work  = VGETVECS( nlP->vc, usrP, 4 );    CHKPTR(Sctx->work);
}

void NLDogLegDestroy( nlP, usrP )
NLCntx *nlP;
void   *usrP;
{
NLStepDogleg *Sctx = (NLStepDogleg *)nlP->stepCtx;

VFREEVECS( nlP->vc, usrP, Sctx->work, Sctx->nwork );
}

void NLDogLegCreate( nlP, usrP, J, GetJ, ScaleJ, JTv, Jv, JSolve, JSolveCtx )
NLCntx *nlP;
void   *usrP;
void   *J, (*GetJ)(), (*ScaleJ)(), (*JTv)(), (*Jv)(), (*JSolve)(), *JSolveCtx;
{
NLStepDogleg *Sctx;

Sctx = NEW(NLStepDogleg);    CHKPTR(Sctx);
Sctx->J           = J;
Sctx->ScaleJ      = ScaleJ;
Sctx->GetJacobian = GetJ;
Sctx->JTv         = JTv;
Sctx->Jv          = Jv;
Sctx->solve       = JSolve;
Sctx->setupSolve  = 0;
Sctx->destroySolve= 0;
Sctx->svctx       = JSolveCtx;
Sctx->D           = (double*) VCREATE( nlP->vc, usrP );    CHKPTR(Sctx->D);
Sctx->monitor     = 0;

nlP->stepCtx      = (void *)Sctx;
}
