/*
     Test of the nonlinear solvers code. This uses several problems
 from the minpack2 test suite. The Jacobians are treated as dense.

     Thanks to Brett Averick and Jorge More' for the test problems.

*/
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "tools.h" 
#include "nonlin/nlall.h"

#if defined(FORTRANCAPS)
#define dfcn_   DFCN
#define dgetf2_ DGETF2
#define dgetrs_ DGETRS
#define dgemv_  DGEMV
#elif !defined(FORTRANUNDERSCORE)
#define dfcn_   dfcn
#define dgetf2_ dgetf2
#define dgetrs_ dgetrs
#define dgemv_  dgemv
#endif

typedef struct {
  int     n;                  
  double  *jacobian;
  double  *jacobianFtr;
  int     *pivot;
  int     Problem;
} DenseCntx;

DenseCntx    *CreateDense();
void         DestroyDense();
void         Function();
void         InitialGuess();
void         SetUpLinearSolve();
void         LinearSolve();
void         DestroyLinearSolve();

/* For the dog-leg routine */
void GetJ(), JTv(), Jv(), JSolve(), ScaleJ();

#define ERR {printf("Error %d %s \n",__LINE__,__FILE__); exit(0);}

main(argc,argv)
int  argc;
char *argv[];
{
  NLCntx    *neP;
  DenseCntx *neD;
  int       n, its, i, Problem = 1, nfunc,nsteps,nvector;
  int       stepkind = 0, dodis = 0;
  double    *x, *f, norm,xnorm;
  NLMETHOD  nlmethod = NLNEWTONLS;

  SYSetFPTraps();
  if (SYArgHasName( &argc, argv, 1, "-help")) {
   fprintf( stderr, 
      "%s: [-t <trust>] [-l <linesearch>] [-t2 <trustminpack2>]\n", argv[0] );
   fprintf( stderr, "    [-d <doglegstep> ] [-p n <problem>] [-log]\n" );
   exit(0);
   }
  if (SYArgHasName( &argc, argv, 1, "-t" )) nlmethod  = NLNEWTONTR;
  if (SYArgHasName( &argc, argv, 1, "-l" )) nlmethod  = NLNEWTONLS;
  if (SYArgHasName( &argc, argv, 1, "-t2" )) nlmethod = NLNEWTONTR2;
  if (SYArgHasName( &argc, argv, 1, "-dis")) dodis = 1;
  if (SYArgHasName( &argc, argv, 1, "-d" )) {
      stepkind  = 1;
      if (nlmethod == NLNEWTONLS)
	  nlmethod  = NLNEWTONTR;
      }
  if (SYArgGetInt( &argc, argv, 1, "-p", &Problem )) {
      if (Problem > 5) ERR;
      }

  neD         = CreateDense(Problem); if (!neD) ERR;
  n           = neD->n;

  printf("Running Problem number %d Size %d \n",Problem, n);

  neP         = NLCreate(nlmethod);       if (!neP) ERR;
  NLSetVectorOps(neP,VECreate());
  DVSetDefaultFunctions(neP->vc);
  NLSetFunction(neP,Function);
  NLSetInitialGuess(neP,InitialGuess);
  if (nlmethod == NLNEWTONTR2)
      NLSetJv(neP,Jv,neD);
  if (stepkind) {
      NLSetStepSetUp(neP,NLDogLegSetup);
      NLSetStepCompute(neP,NLDogLeg);
      NLSetStepDestroy(neP,NLDogLegDestroy);
      NLDogLegCreate( neP, neD, neD, GetJ, (void (*)())0, JTv, Jv, 
		      JSolve, neD );
      if (dodis)
	  NLSetDogLegMonitor( neP, NLDogLegMonitor );
      }
  else {
      NLSetStepSetUp(neP,SetUpLinearSolve);
      NLSetStepCompute(neP,LinearSolve);
      }
  neP->vec_sol = (void *) MALLOC(n*sizeof(double)); if (!neP->vec_sol) ERR;
  neP->residual_history = (double *) MALLOC(neP->max_it*sizeof(double));
  if (!neP->residual_history) ERR;
  if (SYArgHasName( &argc, argv, 1, "-log" )) neP->fp = stderr;

  NLSetUp(neP,neD);

  its = NLSolve(neP,neD); printf("Number of Newton iterations %d \n",its);

  for ( i=0; i<its+1; i++ ) printf("%d %g \n",i,neP->residual_history[i]);

  NLGetNumberFunctionApplicationsUsed(neP,nfunc);
  NLGetNumberStepComputationsUsed(neP,nsteps);
  NLGetNumberVectorOpsUsed(neP,nvector);

  printf("Function evaluations %d \n",nfunc);
  printf("Inverse Jacobian Applications %d \n",nsteps);
  printf("Number of vector operations %d \n",nvector);

  NLDestroy(neP,neD);
  DestroyDense(neP,neD);
}
/*-------------------------------------------------------------*/
void Function(nlP,neD,x,f)
NLCntx    *nlP;
DenseCntx *neD;
double    *x,*f;
{
  int    n = neD->n,info = 1,prob = neD->Problem;
  dfcn_(&n,&n,x,f,NULL,&n,&info,&prob); 
}
/*-------------------------------------------------------------*/
/*
      Forms and factors Jacobian at the point x
*/
void SetUpLinearSolve(nlP,neD,x)
NLCntx     *nlP;
DenseCntx  *neD;
double     *x;
{
  double *A = neD->jacobianFtr;
  double *AA= neD->jacobian;
  int    n = neD->n,info = 2,prob = neD->Problem, one = 1,i,*piv = neD->pivot;
  int    n2 = n * n;
  FILE   *fp = (FILE *) nlP->fp;

  /* get Jacobian */
  dfcn_(&n,&n,x,NULL,AA,&n,&info,&prob); 
  for ( i=0; i<n2; i++ )
      A[i] = AA[i];

  /* factor Jacobian */
  dgetf2_(&n,&n,A,&n,piv,&info); if (info < 0) ERR;

  /* if factor is sigular we use a tacky fix */
  if (info > 0) { 
    if (fp) fprintf(fp,"Zero Pivot in LU %d\n",info);
    for ( i=0; i<n; i++ ) {
      if (A[n*i + i] == 0.0) A[n*i + i] = 1.0;
    }
  }
}
/*-------------------------------------------------------------*/
/*             y = -J(x)\f  || y || <= delta
*/
void LinearSolve(nlP,neD,x,f,y,fnorm,delta,rtol,gpnorm,ynorm)
NLCntx    *nlP;
DenseCntx *neD;
double    *f,*x,*y,fnorm,*gpnorm,delta,*ynorm,rtol;
{
  double *A = neD->jacobianFtr,norm;
  int    n = neD->n,info,prob = neD->Problem, one = 1, *piv = neD->pivot;
  FILE   *fp = (FILE *) nlP->fp;

  /* determine y by solving linear system */
  VCOPY(nlP->vc,neD,f,y);
  dgetrs_("N",&n,&one,A,&n,piv,y,&n,&info);

  /* scale step so its length is less then delta */
  VNORM(nlP->vc,neD,y,&norm);
  if (norm > delta) {
    norm = delta/norm;
    *gpnorm = (1.0 - norm)*fnorm;
    VSCALE(nlP->vc,neD,-norm,y);
    if (fp) fprintf(fp,"Scaling direction by %g \n",norm);
    *ynorm = delta;
  }
  else {
    *gpnorm = 0.0;
    VSCALE(nlP->vc,neD,-1.0,y);
    if (fp) fprintf(fp,"Direction is in Trust Region \n");
    *ynorm = norm;
  }
}
/*-------------------------------------------------------------*/
void InitialGuess(nlP,neD,x)
NLCntx    *nlP;
DenseCntx *neD;
double    *x;
{
  int    n = neD->n,info = -1,prob = neD->Problem, i;
  dfcn_(&n,&n,x,NULL,NULL,&n,&info,&prob); 
}
/*-------------------------------------------------------------*/
int PrintVector(n,x)
int    *n;
double *x;
{
  int i;
  for ( i=0; i<*n; i++ ) fprintf(stderr,"%d %g \n",i,x[i]);
}
/*-------------------------------------------------------------*/
static int N[5] = {8,11,5,80,140};
DenseCntx *CreateDense(problem)
int       problem;
{
  int n;
  DenseCntx *nedense;
  n                 = N[problem-1];
  nedense           = NEW(DenseCntx);       if (!nedense) ERR;
  nedense->n        = n;
  nedense->jacobian = (double *) MALLOC(n*n*sizeof(double)); 
  if (!nedense->jacobian) ERR;
  nedense->jacobianFtr = (double *) MALLOC(n*n*sizeof(double)); 
  if (!nedense->jacobianFtr) ERR;
  nedense->pivot    = (int *) MALLOC(n*sizeof(int)); 
  if (!nedense->pivot) ERR;
  nedense->Problem  = problem;
  return nedense;
}
/*-------------------------------------------------------------*/
void DestroyDense(nlP,nedense)
NLCntx    *nlP;
DenseCntx *nedense;
{
  FREE(nedense->jacobian);
  FREE(nedense->jacobianFtr);
  FREE(nedense->pivot);
  FREE(nedense);
}

/* For the dogleg routine */
void GetJ( nlP, neD, x )
NLCntx    *nlP;
DenseCntx *neD;
double    *x;
{
SetUpLinearSolve(nlP,neD,x);
}

/* If this routine is not provided, the identity will be used */
void ScaleJ( nlP, neD, d )
NLCntx    *nlP;
DenseCntx *neD;
double    *d;
{
double *A = neD->jacobian, *a, sum;
int     n = neD->n, one = 1, i, j;

a = A;
for (i=0; i<n; i++) {
    sum = 0.0;
    for (j=0; j<n; j++) 
	sum += a[j]*a[j];
    a += n;
    d[i] = sqrt(sum);
    }
}

/* g <- J' f */
void JTv( neD, f, g )
DenseCntx *neD;
double    *f, *g;
{
double *A = neD->jacobian;
int     n = neD->n, one = 1;
double  done = 1.0, dzero = 0.0; 

dgemv_( "T", &n, &n, &done, A, &n, f, &one, &dzero, g, &one, 1 );
}

/* g <- J f */
void Jv( neD, f, g )
DenseCntx *neD;
double    *f, *g;
{
double *A = neD->jacobian;
int     n = neD->n, one = 1;
double  done = 1.0, dzero = 0.0; 

dgemv_( "N", &n, &n, &done, A, &n, f, &one, &dzero, g, &one, 1 );
}

/* g <- J \ f */
void JSolve( neD, f, g )
DenseCntx *neD;
double    *f, *g;
{
double *A = neD->jacobianFtr;
int     n = neD->n;
int     i, one = 1, info, *piv = neD->pivot;

for (i=0; i<n; i++) 
    g[i] = f[i];

dgetrs_("N", &n, &one, A, &n, piv, g, &n, &info, 1);
}
