#ifndef lint
static char SCCSid[] = "@(#) ./iter/cg/cg.c 07/23/93";
#endif

/*                       
           This implements Preconditioned Conjugate Gradients.       
*/
#include <stdio.h>
#include <math.h>
#include "tools.h"
#include "iter/itctx.h"
#include "iter/itfunc.h"
#include "iter/cgctx.h"
#include "iter/itpriv.h"

/*+
    ITCGCreate - Creates an ITCntx variable for the conjugate
    gradient method. The prefered calling sequence is ITCreate(ITCG);
+*/
ITCntx *ITCGCreate()
{
  ITCntx    *itP;
  CGCntx *cgP;

  itP = NEW(ITCntx);  CHKPTRV(itP,0);
  ITSetDefaults( itP );
  cgP = NEW(CGCntx);  CHKPTRV(cgP,0);
  itP->MethodPrivate = (void *) cgP;

  itP->method               = ITCG;
  itP->max_it               = 50;
  itP->right_inv            = 0;
  itP->calc_res             = 1;
  itP->use_pres             = 0;
  itP->guess_zero           = 0;
  itP->rtol                 = 1.e-5;
  itP->atol                 = 1.e-50;
  itP->usr_monitor          = ITDefaultMonitor;
  itP->residual_history     = NULL;

  itP->setup                = ITCGSetUp;
  itP->solver               = ITCGSolve;
  itP->adjustwork           = ITDefaultAdjustWork;
  itP->closedown            = ITCGDestroy;
  return(itP);
}
/*+
    ITCGSetUp - Called after a call to ITCGCreate() or ITCREATE(ITCG)
    allocates space needed in the conjugate gradient solution. Preferred
    calling sequence is ITSetUp(itP,usrP).

    Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
+*/
void ITCGSetUp(itP,usrP)
ITCntx *itP;
void   *usrP;
{
  CGCntx *cgP;
  int    maxit;
  cgP = (CGCntx *) itP->MethodPrivate;
  maxit = itP->max_it;

  if (itP->method != ITCG) {
      SETERRC(1,"Attempt to use CG Setup on wrong context"); return;}

  /* check user parameters and functions */
  if ( itP->right_inv ) {
      SETERRC(2,"Right-inverse preconditioning not supported for CG");return;}
  if (ITCheckDef( itP )) return;

  /* get work vectors from user code */
  ITDefaultGetWork( itP, usrP, 3 );

  if (itP->calc_eigs) {
    /* get space to store tridiagonal matrix for Lanczo */
    cgP->e = (double *) MALLOC(4*(maxit+1)*sizeof(double)); CHKPTR(cgP->e);
    cgP->d  = cgP->e + maxit + 1; 
    cgP->ee = cgP->d + maxit + 1;
    cgP->dd = cgP->ee + maxit + 1;
  }
}
/*+
    ITCGSolve - Called after a call to ITCGCreate() or ITCreate(ITCG)
    and the call to ITCGSetUp() or ITSetUp().
    Actually solves the linear system using the conjugate gradient 
    method. Prefered calling sequence ITSolve(itP,usrP).

    Input Parameters: 
.   itP  - the iterative context
.   usrP - the user context

    Returns:
    the number of iterations required or -1 on error.
+*/
int  ITCGSolve(itP, usrP)
void   *usrP;
ITCntx *itP;
{
  int       i = 0,maxit,eigs,res,pres, hist_len, cerr;
  double    a,beta,betaold,b,*e,*d,*history, dp; 
  void      *X,*B,*Z,*R,*P;
  CGCntx    *cgP;
  cgP = (CGCntx *) itP->MethodPrivate;

  eigs    = itP->calc_eigs;
  res     = itP->calc_res;
  pres    = itP->use_pres;
  maxit   = itP->max_it;
  history = itP->residual_history;
  hist_len= itP->res_hist_size;
  X       = itP->vec_sol;
  B       = itP->vec_rhs;
  R       = itP->work[0];
  Z       = itP->work[1];
  P       = itP->work[2];

  if (eigs) {e = cgP->e; d = cgP->d; e[0] = 0.0; b = 0.0; }

  if (!itP->guess_zero) {
    MM(X,R);                                  /*   r <- b - Ax      */
    DAYPX(-1.0,B,R);
  }
  else { 
    COPY(B,R);                                /*     r <- b (x is 0)*/
  }
  PRE(R,Z);                                   /*     z <- Br        */
  if (pres) {
      NORM(Z,&dp);                           /*    dp <- z'*z       */
      }
  else {
      NORM(R,&dp);                           /*    dp <- r'*r       */       
      }
  /* Test for nothing to do */
  NORM(R,&dp);
  if (CONVERGED(dp,0)) return RCONV(0);
  MONITOR(dp,0);
  if (history) history[0] = dp;

  for ( i=0; i<maxit; i++) {
     DOT(R,Z,&beta);                          /*     beta <- r'z    */
     if (i == 0) {
           if (beta == 0.0) break;
           COPY(Z,P);                         /*     p <- z         */
     }
     else {
         b = beta/betaold;
         if (eigs) e[i] = sqrt(b)/a;  
         DAYPX(b,Z,P);                        /*     p <- z + b* p   */
     }
     betaold = beta;
     MM(P,Z);                                 /*     z <- Kp         */
     DOT(P,Z,&dp);
     a = beta/dp;                             /*     a = beta/p'z    */
     if (eigs)  d[i] = sqrt(b)*e[i] + 1.0/a;
     DAXPY(a,P,X);                            /*     x <- x + ap     */
     DAXPY(-a,Z,R);                           /*     r <- r - az     */
     if (pres) {
       PRE(R,Z);                              /*     z <- Br         */
       NORM(Z,&dp);                           /*    dp <- z'*z       */
     }
     else {
       NORM(R,&dp);                           /*    dp <- r'*r       */       
     }
     if (history && hist_len > i + 1) history[i+1] = dp;
     MONITOR(dp,i+1);
     if (CONVERGED(dp,i+1)) break;
     if (!pres) PRE(R,Z);                     /*     z <- Br         */
  }
  if (i == maxit) i--;
  if (history) itP->res_act_size = (hist_len < i + 1) ? hist_len : i + 1;

  /* Update computational work */
  itP->namult += i+1;
  itP->nbinv  += i+1;
  itP->nvectors += (i+1)*10;

  return RCONV(i+1);
}

/*+
    ITCGDestroy - Destroys a iterative context variable obtained 
    by a call to ITCGCreate() or ITCreate(ITCG). Preferred calling
    sequence ITDestroy().

    Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
+*/
void ITCGDestroy(itP,usrP)
ITCntx *itP;
void   *usrP;
{
  CGCntx *cgP;
  cgP = (CGCntx *) itP->MethodPrivate;

  /* free space used for eigenvalue calculations */
  if ( itP->calc_eigs ) {
    FREE(cgP->e);
  }

  ITDefaultFreeWork( itP, usrP );
  
  /* free the context variables */
  FREE(cgP); FREE(itP);
}

/*ARGSUSED*/
/*@
    ITCGDefaultMonitor - The default iterative monitor routine for CG,
    it prints the two norm of the true residual and estimation from
    Lanczo of the extreme eigenvalues of the preconditioned problem
    at each iteration.

    Input Parameters: 
.   itP - the iterative context
.   usrP - the user context
.   n  - the iteration
.   rnorm - the two norm of the residual
@*/
void ITCGDefaultMonitor(itP,usrP,n,rnorm)
ITCntx *itP;
void   *usrP;
int    n;
double rnorm;
{
  CGCntx *cgP;
  double    c;
  if (!itP->calc_eigs) {
    printf("%d %14.12e \n",n,rnorm);
  }
  else {
    cgP = (CGCntx *) itP->MethodPrivate;
    ITCGGetEigenvalues(itP,n,&cgP->emax,&cgP->emin); CHKERR(55);
    c = cgP->emax/cgP->emin;
    printf("%d %14.12e %% %g %g %g \n",n,rnorm,cgP->emax,cgP->emin,c);
  }
}
