#ifndef LINT
static char SCCSid[] = "@(#) ./fd/laplac.c 07/23/93";
#endif

/*
   This file contains several routines to generate a matrix for the
   2-d Laplacian, using the usual 5-point finite different scheme.
   They illustrate a number of different ways of forming such a matrix.
 */

#include <stdio.h>
#include <math.h>
#include "fd/fd.h"

/*@
    FDBuildLaplacian2d - Create a sparse matrix for the 2-d Laplacian

    Input parameters:
.    mx,my - the mesh is mx x my.
.    hx,hy - mesh spacing.  If zero, spacing appropriate for the unit
             square is assumed.
.    sc    - scaling (each element is multiplied by this value.  If zero,
             hx*hx+hy*hy)/2 is used.

    Notes:
    This Laplacian does NOT include the boundaries.  The right-hand-side
    must be appropriately modified.  The discretization is the 5-point 
    2nd order finite difference stencil.
 @*/   
SpMat *FDBuildLaplacian2d( mx, my, hx, hy, sc )
int    mx, my;
double hx, hy, sc;
{
SpMat *f;
int   n, i, j, row;

n   = mx * my;
f   = SpCreate( n, n, 5 );
if (hx == 0.0) hx = 1.0 / mx;
if (hy == 0.0) hy = 1.0 / my;
if (sc == 0.0) sc = (hx*hx + hy*hy) / 2.0;

for (j=0; j<my; j++) {
    for (i=0; i<mx; i++) {
    	row = i + j * mx;
        if (j > 0) {
	    SpAddValue( f, sc / (hy*hy), row, row - mx );
            }
        if (i - 1 >= 0) {
	    SpAddValue( f, sc / (hx*hx), row, row - 1 );
	    }
        SpAddValue( f, - sc * ( 2.0 / (hx*hx) + 2.0 / (hy*hy) ), row, row );
        if (i + 1 < mx) {
	    SpAddValue( f, sc / (hx*hx), row, row + 1 );
	    } 
        if (j + 1 < my) {
	    SpAddValue( f, sc / (hy*hy), row, row + mx );
	    }
        }
    }
return f;
}

/*@
    FDBuildLaplacian2dBndy - Create a sparse matrix for the 2-d Laplacian,
    with boundary conditions included.

    Input parameters:
.    mx,my - the mesh is mx x my.
.    hx,hy - mesh spacing.  If zero, spacing appropriate for the unit
             square is assumed.
.    sc    - scaling (each element is multiplied by this value.  If zero,
             hx*hx+hy*hy)/2 is used.

    Notes:
    The discretization is the 5-point 2nd order finite difference stencil.
 @*/   
SpMat *FDBuildLaplacian2dBndy( mx, my, hx, hy, sc )
int    mx, my;
double hx, hy, sc;
{
SpMat *FDBuildLaplacian2dBndySub();

return FDBuildLaplacian2dBndySub( mx, my, hx, hy, sc, 0, mx-1, 0, my-1 );
}

/*@
    FDBuildLaplacian2dBndySub - Create a sparse matrix for the 2-d Laplacian,
    with boundary conditions included, for only the subrectangle requested.
    The column indices will be wrt the usual global natural ordering.

    Input parameters:
.    mx,my - the mesh is mx x my including the boundaries.
.    hx,hy - mesh spacing.  If zero, spacing appropriate for the unit
             square is assumed.
.    sc    - scaling (each element is multiplied by this value.  If zero,
             hx*hx+hy*hy)/2 is used.  Boundary nodes are NOT scaled.
.    [si,ei]x[sj,ej] - subrectangle to generate coefficients for

    Notes:
    The discretization is the 5-point finite difference stencil.

    This routine is included primarily as an example of forming a submatrix
    for the parallel linear solver routines.
 @*/   
SpMat *FDBuildLaplacian2dBndySub( mx, my, hx, hy, sc, si, ei, sj, ej )
int    mx, my, si, ei, sj, ej;
double hx, hy, sc;
{
SpMat *f;
int    n, i, j, row, rrow, mmx, mmy;
double hinvx, hinvy;

mmx = ei - si + 1;
mmy = ej - sj + 1;
n   = mmx * mmy;
f   = SpCreate( n, mx*my, 5 );
if (hx == 0.0) hx = 1.0 / (mx - 1.0);
if (hy == 0.0) hy = 1.0 / (my - 1.0);
if (sc == 0.0) sc = (hx*hx + hy*hy) / 2.0;
hinvx = sc / (hx * hx);
hinvy = sc / (hy * hy);

for (j=sj; j<=ej; j++) {
    for (i=si; i<=ei; i++) {
    	row = i + j * mx;
	rrow = (i - si) + (j - sj) * mmx;
	if (i == 0 || j == 0 || i == mx - 1 || j == my - 1) {
	    SpAddValue( f, 1.0, rrow, row );
	    continue;
	    }
	/* Else ... */
        if (j > 0) {
	    SpAddValue( f, -hinvy, rrow, row - mx );
            }
        if (i - 1 >= 0) {
	    SpAddValue( f, -hinvx, rrow, row - 1 );
	    }
        SpAddValue( f, 2.0*(hinvx + hinvy), rrow, row );
        if (i + 1 < mx) {
	    SpAddValue( f, -hinvx, rrow, row + 1 );
	    } 
        if (j + 1 < my) {
	    SpAddValue( f, -hinvy, rrow, row + mx );
	    }
        }
    }
return f;
}

/*@
    FDBuildFDElliptic2dBndySub - Create a sparse matrix for the 2-d elliptic
    problems, with finite differences (FD) and Dirichlet boundary conditions,
    for only the subrectangle requested.
    The column indices will be wrt the usual global natural ordering.

    Input parameters:
.    mx,my           - the mesh is mx x my.
.    [si,ei]x[sj,ej] - subrectangle to generate coefficients for
.    hx,hy - mesh spacing.  If zero, spacing appropriate for the unit
             square is assumed.
.    sc    - scaling (each element is multiplied by this value.  If zero,
             (hx*hx+hy*hy)/2 is used.
.    ax,ay             - functions for second derivative terms
.    bx,by             - functions for first derivative terms
.    c                 - function for undifferentiated term

    Notes:
    The functions have the form
$   double name(x,y)
$   A null function is taken as a 1 for (ax,ay) and a 0 for (bx,by,c).  In
    this way, using null functions will give you the Laplacian.
    
    The entries are all naturally scaled.
    The discretization is the 5-point finite difference stencil.
    For this case, using finite differences, the method is (centered
    differencing for everything):
$    bx u_x is b(x,y) (u(x+h,y) - u(x-h,y))/2h
$    (ax u_x)_x is (a(x+h/2,y)(u(x+h,y)-u(x,y))/h -
$				a(x-h/2,y)(u(x,y)-u(x-h,y))/h)
$       or
$                a(x+h/2,y)u(x+h,y)/h^2 - (a(x+h/2,y)+a(x-h/2,y))u(x,y)/h^2 +
$                a(x-h/2,y)u(x-h,y)/h^2

    This routine is included primarily as an example of forming a submatrix
    for the parallel linear solver routines, and for implementing a
    particular differencing scheme.  For other schemes (such as FEM, FV, or
    upwinding), this routine may be used as a guide.
 @*/   
SpMat *FDBuildFDElliptic2dBndySub( ax, ay, bx, by, c, mx, my, hx, hy, sc, 
				   si, ei, sj, ej )
double (*ax)(), (*ay)(), (*bx)(), (*by)(), (*c)();
int    mx, my, si, ei, sj, ej;
double hx, hy, sc;
{
SpMat *f;
int    n, i, j, row, rrow, mmx, mmy;
double hinvx, hinvy;
double val, x, y;
double AxP, AxM, AyP, AyM, Bx, By, C0;

if (hx == 0.0) hx = 1.0 / (mx - 1.0);
if (hy == 0.0) hy = 1.0 / (my - 1.0);
if (sc == 0.0) sc = (hx*hx + hy*hy) / 2.0;

mmx   = ei - si + 1;
mmy   = ej - sj + 1;
n     = mmx * mmy;
f     = SpCreate( n, n, 5 );
hinvx = 1.0 / (hx * hx);
hinvy = 1.0 / (hy * hy);

/* 
   Lots of optimizations are possible.  They include:
   Change functions to return a number of values at once
   Insert multiple items into the sparse matrix at once
 */   
for (j=sj; j<=ej; j++) {
    y = j * hy;
    x = si * hx;
    for (i=si; i<=ei; i++) {
    	row = i + j * mx;
	rrow = (i - si) + (j - sj) * mmx;
	/* Dirichlet boundary */
	if (i == 0 || j == 0 || i == mx - 1 || j == my - 1) {
	    SpAddValue( f, 1.0, rrow, row );
	    x += hx;
	    continue;
	    }
	/* Else ... */
	AxP = (ax) ? (*ax)( x + hx * 0.5, y ) : 1.0;
	AxM = (ax) ? (*ax)( x - hx * 0.5, y ) : 1.0;
	AyP = (ay) ? (*ay)( x, y + hy * 0.5 ) : 1.0;
	AyM = (ay) ? (*ay)( x, y - hy * 0.5 ) : 1.0;
	C0  = (c)  ? (*c)( x, y )             : 0.0;
	Bx  = (bx) ? (*bx)( x, y )            : 0.0;
	By  = (by) ? (*by)( x, y )            : 0.0;
        if (j > 0) {
            val = AyM * (hinvy) - By / (2.0*hy);
	    SpAddValue( f, - sc * val, rrow, row - mx );
            }
        if (i - 1 >= 0) {
            val = AxM * (hinvx) - Bx / (2.0*hx);
	    SpAddValue( f, - sc * val, rrow, row - 1 );
	    }
	val = - ( (AxP + AxM) * hinvx + (AyP + AyM) * hinvy + C0 );
        SpAddValue( f, - sc * val, rrow, row );
        if (i + 1 < mx) {
            val = AxP * (hinvx) + Bx / (2.0*hx);
	    SpAddValue( f, - sc * val, rrow, row + 1 );
	    } 
        if (j + 1 < my) {
            val = AyP * (hinvy) + By / (2.0*hy);
	    SpAddValue( f, - sc * val, rrow, row + mx );
	    }
	x += hx;
        }
    }
return f;
}
