#ifdef FOO
/*
  The following program is designed to perform a broadcast across a row or 
  column of processors.  It is done by recursively dividing 
  the line of processors 
  in half and sending to the nearest processor in the other half of the 
  processors.  It is assumed that the processors to broadcast over comprise a 
  linear line of physical processors and there are no gaps or breaks in the 
  line of processors.

  Notes for this routine can be found in the notebook of Steve Lederman
  for 6/24/92.

  Given the following input:

  bc(3, 1, 7, 1000, p_send, p_recv)

  The algorithm can be view pictorially as (assuming you 
  have nodes 0-8 allociated):

  NODE     STAGE 1     STAGE 2    STAGE 3

    0 (not involved in broadcast)

    1                              <--|
                                  -------
    2                   <--|       >--|
                       -------
    3       >--|        >--|       >--|
               |                  -------
    4          |                   <--|
           -------
    5       <--|        >--|       >--|
                           |      -------
    6                      |       <--|
                       -------
    7                   <--|      (done)

    8 (not involved in broadcast)

where the symbols means that processor 3 sends to processor 5 in stage 1.

*/

/* LIBRARIES */

#include <stdeig.h>
#include <stdio.h>
#include <global.h>

void bc (long l_node_bc, long l_first_node, long l_last_node, long l_bytes,
	 double *p_send, double *p_recv, E_TYPES e_bc_dim)
/*
  PARAMETERS RECEIVED:
  --------------------

  l_node_bc: The x or y physical node address of the node in the row or column
  to start the broadcast.  (unchanged)

  l_first_node: The x or y physical node address of the first node in the
  row or column to broadcast over.  (unchanged)

  l_last_node: The x or y physical node address of the last node in the
  row or column to broadcast over.  (unchanged)

  l_bytes: The number of bytes to broadcast.  (unchanged)

  p_send: Pointer to first value to broadcast.  The values are sizeof double
  and stored in contiguous memory locations.  (unchanged)

  p_recv: Pointer to first value to receive values in broadcast.  The
  values are sizeof double and stored in contiguous memory locations.  (The
  memory locations starting at this pointer are modified)

  e_bc_dim: Either equal to "vert" or "horz".  "vert" indicates that the
  broadcast is down rows and "horz" indicates that the broadcast is
  across columns.  (unchanged)
*/

{

#define FORCED_MIN 1600000000		/* starting number for forced message type */
#define SYNC_MIN 1000			/* starting number for synchronizing message type */

  long
    l_node,	/* what physical node you are in row or column.  If you are */
		/* broadcasting across rows then l_node is your row location */
		/* and if across columns then l_node is your column location. */
    l_scnd,	/* the other (second) dimension in your address.  If you are */
		/* broadcasting across rows then it is your column address or */
		/* your row address if across columns.  The value is the same */
		/* for all the nodes in your column in row broadcast and row */
		/* in column broadcast */
    l_max,	/* the maximum (last) node in your part of the row or column */
    l_mid,	/* the middle node in your part of the row or column */
    l_min,	/* the minimum (first) node in your part of the row or column */
    l_wr,	/* what node to receive in your part of row or column */
    l_ws,	/* what node to send in your part of row or column */
    l_node_inc, /* increment to use to convert your location in the row or */
		/* column into a physical address.  This is to apply to */
		/* l_node. */
    l_scnd_inc, /* increment to use to convert your location in the row or */
		/* column into a physical address.  This is to apply to */
		/* l_scnd */
    l_sync,	/* dummy variable for zero length synchronization send/recv */
    l_send,	/* physical address of what node to send to */
    i,		/* loop counter */
    l_msg_num	/* id for irecv */
    ;

  extern void generror(char c_error_text[]);

  /* check to see if got NULL pointers */
/*
  if (p_send == NULL)
    {
      generror("bc: the sending pointer (p_send) is NULL");
    }
*/
/*
  if (p_recv == NULL)
    {
      generror("bc: the receiving pointer (p_recv) is NULL");
    }
*/
  /* check to see if message length is > 0 */
/*
  if (l_bytes <= 0)
    {
      generror("bc: the message length not greater than 0");
    }
*/
  /* error if broadcasting node not between first and last node */
/*
  if ((l_node_bc < l_first_node) || (l_node_bc > l_last_node))
    {
      generror("bc: broadcast node not between participating nodes");
    }
*/

  /* set up values of variables for either row or column broadcast */
  /* error message if illegal option chosen */

  switch (e_bc_dim)
    {
    case horz:
      l_node = l_node_col;
      l_scnd = l_node_row;
      l_node_inc = 1;
      l_scnd_inc = l_msh_cols;
      break;
    case vert:
      l_node = l_node_row;
      l_scnd = l_node_col;
      l_node_inc = l_msh_cols;
      l_scnd_inc = 1;
      break;
    default:
      generror("bc: illegal option selected for broadcast");
      return;
    }

  /* if you are the node to broadcast and the receive buffer is 
     different from the send buffer then you need to copy the send
     buffer to the receive buffer since you don't send to yourself */

  if ((l_node == l_node_bc) && (p_send != p_recv))
    {
      for (i = 0; i < l_bytes/sizeof(double); i++)
	{
	  *(p_recv+i) = *(p_send+i);
	}
    }

  l_ws = l_node_bc;								 /* initial node to send is node to broadcast */
  l_min = l_first_node;								 /* initial min mode is first node in group */
  l_max = l_last_node;								 /* initial max mode is last node in group */

  /* while loop until you are done sending and receiving.  You know you
     are done when your max == min because this means that your
     group within the row or column only includes you so you have recursively   
     finished */

  while (l_min != l_max)
    {
      l_mid = (l_max + l_min) / 2;

      /* reset who receives in your group.  if the sending node is at or below
	 the middle then you send one past the minimum.  if the sending
	 node is above the middle then you send to the middle node */

      if (l_ws <= l_mid)
	{
	  l_wr = l_mid + 1;
	}
      else
	{
	  l_wr = l_mid;
	}

      /* do sending and receiving for node that should */
      if (l_node == l_ws)
	{
	  /* you are a sending node.  first you wait until you receive
	     a synchronization message from the node you expect to send
	     to and then you send a forced type message to that node
	     since you know it is now ready to receive */
	  
	  l_send = l_wr * l_node_inc + l_scnd * l_scnd_inc;			 /* physical address of who to send to */
	  crecv (SYNC_MIN + l_wr, &l_sync, 0);					 /* get a zero length message tagged with id of */
										 /* receiving node */
	  csend (FORCED_MIN + l_node, p_recv, l_bytes, l_send, 0);		 /* send data by forced type with id of */
										 /* sending node */
	}
      else if (l_node == l_wr)
	{
	  /* you are a receiving node.  first you post a receive for
	     the forced type message and then send a synchronizing
	     message to the node you expect to receive receive from
	     so it knows to send */

	  l_msg_num = irecv (FORCED_MIN + l_ws, p_recv, l_bytes);		 /* post receive for broadcast data with id of */
										 /* sending node */
	  l_send = l_ws * l_node_inc + l_scnd * l_scnd_inc;			 /* physical address of node to send */
										 /* synchronization message to */
	  csend (SYNC_MIN + l_node, &l_sync, 0, l_send, 0);			 /* send sychronization message with id of */
										 /* receiving node */
	  msgwait (l_msg_num);							 /* wait until data received before proceeding */
	}

      /* recalculate the min, max and send nodes as needed */

      if (l_node > l_mid)							 /* your min changes if you are above middle */
	{
	  l_min = l_mid + 1;
	  /* to get here you are above the middle.  If the sending node
	     was below the middle then your new group received the value
	     and that node becomes the sending node next time */
	  if (l_ws <= l_mid)
	    {
	      l_ws = l_mid + 1;
	    }
	}
      else									 /* your max changes if you are below the middle */
	{
	  l_max = l_mid;
	  /* to get here you are at or below the middle.  If the sending node
	     was above the middle then your new group received the value
	     and that node becomes the sending node next time */
	  if (l_ws > l_mid)
	    {
	      l_ws = l_mid;
	    }
	}
    } /* end while */
  return;
}
#endif
