/* New code based on an article by P. L. Montgomery, Modular multiplication 
   without trial division, Math. of Comp., vol. 44, n 170, 1985, pp. 519-21.
*/
 
#include <stdio.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "BigZ.h"
#include "BntoBnn.h"
 

#define NEWLINE		fprintf(stderr,"\n")
#define From(A)		BzToBigNum(BzFromString(A,10), &foonl)
#define To(A)		BzToString(BzFromBigNum(A,l),10)


double runtime() {
	struct rusage r;
	struct timeval t;
 
	getrusage(0, &r);
	t = r.ru_utime;
	return(1000*t.tv_sec + (t.tv_usec/1000.));
}
 
/* used for conversion BigZ --> BigNum */
static unsigned     foonl;

main()
{
   printf("New version with almost no division at all\n");
   printf("\n");
 
   printf("Times on a SUN 3/60 (3.4-12 Mo) \n");
   printf("\n");
   printf("C, old version                 : 2.7s \n");
   printf("C, new version, BnMultiply     : 3.4s \n");
   BnBench();
}
 
BnBench() {
	double t1, tp = 0.0;
	int i;
 
	BigNum a = From("6290843252034247832488948530889507820372534220155749337406213413141456028021015613339401109972769059704411603077661248901138314618192385808292059038302087");
	BigNum e = From("7301528981815353440186284264567853991785844716335520074738049695514504674875031196492510938054180065876409690571413838538140398885731057226412015631919490");
        BigNum n = From("8147771155789330029278033691977920011607194547896681542175548825699070281758159013667940742879342370258220466020565804322942452235743797417874170775049979");
       /* r = a**e mod n */
	BigNum r = From("6592812686517455533682192402687982584732723980364904769602645412889636956320139746048674348010313976797086419629753160088338140528581529611487558287516924");
	BigNum v = From("9763332161996754725804021909723709021344036604223177257841026250178160895962979600115909745070395839862975702510469807950977622324244179136726391164566477");
 
	int l = 16*32/BN_DIGIT_SIZE;
	BigNum p = BnCreate((BigNumType)1, l);
 
	for(i=0; i<5; i++) {
		t1 = runtime();
		MtyExpMod(p, a, l, e, l, n, l, v, l);
		tp += runtime() - t1;
		if(BnCompare(p, 0, l, r, 0, l) != BN_EQ)
		  printf("Error: the result should be %s and not %s\n",To(r),To(p));
	}
	printf("Time is %f\n", tp/5000);
}
	
/* Modular exponentiation */
/* p[0..nl-1] <- a[0..al-1]**e[0..el-1] mod n[0..nl-1] */
/* Reduction "a` la" Montgomery */
int MtyExpMod(p, a, al, e, el, n, nl, v, vl)
	BigNum p, a, e, n, v;
	int al, el, vl;
	register int nl;
	{
	    register int rz = 0;
	    register BigNum c = BnCreate((BigNumType)1, 2);
	    register int l = 1+ (nl<<1);
	    register BigNum tmp1 = BnCreate((BigNumType)1, l);
	    register BigNum tmp2 = BnCreate((BigNumType)1, l);
	    register int es,pl,tmpl,ral;
 
	 /* conversion of a */
	    ral = MtyConvertToResidue(a,a,al,n,nl,v,vl,tmp1);
	    BnAssign(p,0,a,0,ral);
	    pl = ral;
	 /* el becomes the current address of the most significant digit */
	    BnAssign(c, rz, e, --el, 1);
	    es = 1 + BnNumLeadingZeroBitsInDigit(c, rz);
            BnShiftLeft(c, rz, 1, c, 1, es);
	    es = BN_DIGIT_SIZE - es;
	    while(el >= rz){
	      while (es){
		es--;
	        /* tmp1 <- p^2 */
	        tmpl = 1 + (pl<<1);
	        BnSetToZero(tmp1, rz, l);
		BnMultiply(tmp1, rz, tmpl, p, rz, pl, p, rz, pl);
	        /* tmp1 <- MtyRedc(tmp1) */
		pl = MtyRedc(tmp1, tmpl, n, nl, v, vl, tmp2);
	        BnAssign(p, 0, tmp1, nl, pl);
                BnShiftLeft(c, rz, 1, c, 1, 1);
                if(BnIsDigitOdd(c, 1)) {
                   /* tmp1 <- p*a */
		   tmpl = 1 + ral + pl;
		   BnSetToZero(tmp1, rz, l);
		   BnMultiply(tmp1, rz, tmpl, p, rz, pl, a, rz, ral);
		   /* tmp1 <- MtyRedc(tmp1) */
		   pl = MtyRedc(tmp1, tmpl, n, nl, v, vl, tmp2);
		   BnAssign(p, 0, tmp1, nl, pl);
                }
              }
	      if(el--){
  	        BnAssign(c, rz, e, el, 1);
    		es = BN_DIGIT_SIZE;
	      }
	    }
 	    MtyConvertToInteger(a, a, ral, n, nl, v, vl, tmp1, tmp2);
	    MtyConvertToInteger(p, p, pl, n, nl, v, vl, tmp1, tmp2);
	}
 
 
/* We assume that z=z[0..zl-1], with zl <= 2*nl+1. If zl <= 2*nl, it is 
   assumed that  z[zl..2*nl] = 0; m is a buffer of size 2*nl+1. Returns the 
   number of digits of z[nl..2*nl-1].*/
int MtyRedc(z, zl, n, nl, v, vl, m)
	BigNum n,v;
 	register BigNum z,m;
	register int zl,nl,vl;
	{
	    int ml = nl+vl;
	    int pl = (nl << 1);
 
	    BnSetToZero(m, 0, ml);
	    BnMultiply(m, 0, ml, z, 0, nl, v, 0, vl);
	    BnMultiply(z, 0, pl+1, m, 0, nl, n, 0, nl);
	    zl = BnNumDigits(z, nl, 1+pl-nl);
	    if(BnCompare(z, nl, zl, n, 0, nl) == -1)
	      return(zl);
	    else{
	      BnSubtract(z, nl, zl, n, 0, nl, 1);
	      return(BnNumDigits(z, nl, zl));
	    }
	}
 
/* Converts an integer a=a[0..al-1] into an N-Residu p=p[0..pl-1] and returns
   pl; tp is a buffer of size 2*nl+1. It is possible to have p=a. */
int MtyConvertToResidue (p, a, al, n, nl, v, vl, tp)
	BigNum p,a,n,v,tp;
	int al,nl,vl;
	{
	    int l = 1+ (nl << 1);
	    BnSetToZero(tp, 0, l);
	    BnAssign(tp, nl, a, 0, al);
	    BnDivide(tp, 0, l, n, 0, nl);
	    BnAssign(p, 0, tp, 0, nl);
	    return(BnNumDigits(p, 0, nl));
	}
 
/* Converts a N-Residue p=p[0..pl-1] into an integer a=a[0..al-1] and returns
   al. It is possible to have a=p. */
int MtyConvertToInteger(a, p, pl, n, nl, v, vl, tp1, tp2)
	BigNum a,p,n,v,tp1,tp2;
	int pl,nl,vl;
	{
	    int l = 1+(nl << 1);
	    int al;
 
	    BnSetToZero(tp1, 0, l);
            BnAssign(tp1, 0, p, 0, pl);
            al = MtyRedc(tp1, pl, n, nl, v, vl, tp2);
            BnSetToZero(a, 0, nl);
            BnAssign(a, 0, tp1, nl, al);
            return(al);
	}
 

