module TayConv;

%*****************************************************************
%
%     Functions converting Taylor kernels to prefix forms
%
%*****************************************************************


exports prepTaylor!*!*, prepTaylor!*, prepTaylor!*1,
        Taylor!-gen!-big!-O;

imports

% from the REDUCE kernel:
        eqcar, lastpair, prepsq!*, replus, retimes, reval,

% from the header module:
        TayCfPl, TayCfSq, TayCoeffList, TayTemplate, TayTpElOrder,
        TayTpElPoint, TayTpElVars;


fluid '(convert!-Taylor!*
        TaylorPrintTerms
        Taylor!-truncation!-flag);


symbolic procedure prepTaylor!*1 (coefflist, template, no!-of!-terms);
  replus for each cc in coefflist join
    begin scalar x; integer count;
      if Taylor!-truncation!-flag then return nil;
      x := prepTaylor!*2 (cc, template);
      if null x or null no!-of!-terms then return x;
      no!-of!-terms := no!-of!-terms - 1;
      if no!-of!-terms < 0
        then << Taylor!-truncation!-flag := t;
                return nil >>;
      return x
    end;

symbolic procedure prepTaylor!*2 (coeff, template);
  (lambda (pc);
    if pc = 0 then nil
     else {retimes (
            (if eqcar (pc, 'quotient) and eqcar (cadr pc, 'minus)
               then {'minus, {'quotient, cadr cadr pc, caddr pc}}
              else pc) . preptaycoeff (TayCfPl coeff, template))})
    (prepsq!* TayCfSq coeff);


symbolic procedure checkdifference (var, var0);
  if var0 = 0 then var else {'difference, var, var0};

symbolic procedure checkexp (bas, exp);
  if exp = 0 then 1
   else if exp = 1 then bas
   else {'expt, bas, exp};

symbolic smacro procedure checkpower (var, var0, n);
  if var0 eq 'infinity
    then if n = 0 then 1
          else {'quotient, 1, checkexp (var, n)}
   else checkexp (checkdifference (var, reval var0), n);

symbolic procedure preptaycoeff (cc, template);
  begin scalar result;
    while not null template do begin scalar ccl;
      ccl := car cc;
      for each var in TayTpElVars car template do <<
        result := checkpower (var, TayTpElPoint car template, car ccl)
                    . result;
        ccl := cdr ccl >>;
      cc := cdr cc;
      template := cdr template
    end;
    return result
  end;

put ('taylor!*, 'prepfn2, 'preptaylor!*!*);

symbolic procedure prepTaylor!*!* u;
   if null convert!-taylor!* then u else preptaylor!* u;

symbolic procedure prepTaylor!* u;
   prepTaylor!*1 (TayCoeffList u, TayTemplate u, nil);

symbolic procedure Taylor!-gen!-big!-O tp;
  %
  % Generates a big-O notation for the Taylor template tp
  %
  'O . for each el in tp collect
          if null cdr TayTpElVars el
            then checkpower (car TayTpElVars el, TayTpElPoint el,
                             TayTpElOrder el + 1)
           else begin scalar var0; integer n;
             var0 := reval TayTpElPoint el;
             n := TayTpElOrder el + 1;
             return
               if var0 eq 'infinity
                 then {'quotient, 1,
                       checkexp ('list . TayTpElVars el, n)}
                else checkexp (
                 'list .
                   for each krnl in TayTpElVars el collect
                     checkdifference (krnl, var0),
                 n)
           end;

endmodule;

module TayPrint$

%*****************************************************************
%
%     Functions for printing Taylor kernels
%
%*****************************************************************


exports Taylor!*print$

imports

% from the REDUCE kernel:
        denr, eqcar, fmprint, kernp, lastpair, maprint, mvar, numr,
        prepsq, simp!*, typerr,

% from the header module:
        TayCoeffList, TayOrig, TayTemplate, TayTpElOrder,
        TayTpElPoint, TayTpElVars,

% from module Tayconv:
        prepTaylor!*1, Taylor!-gen!-big!-O;


fluid '(!*fort !*nat !*taylorprintorder Taylor!-truncation!-flag
        TaylorPrintTerms);

symbolic procedure check!-print!-terms u;
  begin scalar x;
    x := simp!* u;
    if kernp x and mvar numr x eq 'all then return nil
     else if denr x = 1 and fixp numr x then return numr x
     else typerr (x, "value of TaylorPrintTerms")
  end;

symbolic procedure Taylor!*print (u, p);
  begin scalar Taylor!-truncation!-flag, prepexpr, rest;
    prepexpr := prepTaylor!*1 (
                  TayCoeffList u,
                  TayTemplate u, 
                  if null !*fort and !*nat and !*taylorprintorder
                    then check!-print!-terms TaylorPrintTerms
                   else nil);
    if !*taylorprintorder then <<
      rest := {Taylor!-gen!-big!-O TayTemplate u};
      if Taylor!-truncation!-flag then rest := '!.!.!. . rest;
      >>
     else rest := {'!.!.!.};
    if !*fort then fmprint (prepexpr, 0)
     else maprint (if null !*nat then 
                     'taylor . 
                        (if TayOrig u
                           then prepsq Tayorig u
                          else prepexpr) .
                        for each el in TayTemplate u join
                          {if null cdr TayTpElVars el
                             then car TayTpElVars el
                            else 'list . TayTpElVars el,
                           TayTpElPoint el,
                           TayTpElOrder el}
                    else if not eqcar (prepexpr, 'plus)
                     then 'plus . (prepexpr or 0) . rest
                    else nconc (prepexpr, rest),
                    p)
  end;

put ('Taylor!*, 'pprifn, 'Taylor!*print);


comment We need another printing function for use with the 
        TeX-REDUCE interface; %not yet done;


endmodule;

module TayFrontend;

%*****************************************************************
%
%          User interface
%
%*****************************************************************


exports taylorcombine, taylororiginal, taylorprintorder,
        taylorseriesp, taylortemplate, taylortostandard;

imports

% from the REDUCE kernel:
        eqcar, mk!*sq, mvar, numr, prepsq, simp!*, typerr,

% from the header module:
        Taylor!-kernel!-sq!-p, TayOrig, TayTemplate,

% from module Tayintro:
        Taylor!-error,

% from module Taysimp:
        taysimpsq;


symbolic procedure taylorseriesp u;
  (Taylor!-kernel!-sq!-p sq)
      where sq := simp!* u;

symbolic procedure taylorcombine u;
  mk!*sq taysimpsq simp!* u;

symbolic procedure taylortostandard u;
  (prepsq if not eqcar (u, '!*sq) then simp!* u else cadr u)
          where convert!-Taylor!* := t;

symbolic procedure taylororiginal u;
  (if not Taylor!-kernel!-sq!-p sq
     then typerr (u, "Taylor kernel")
    else (if TayOrig tay then mk!*sq TayOrig tay
        else Taylor!-error ('no!-original, 'taylororiginal))
           where tay := mvar numr sq)
     where sq := simp!* u;

symbolic procedure taylortemplate u;
  (if not Taylor!-kernel!-sq!-p sq
     then typerr (u, "Taylor kernel")
    else 'list . for each triple in TayTemplate mvar numr sq collect
              'list . ((if null cdr car triple then car car triple
                         else 'list . car triple)
                     . cdr triple))
     where sq := simp!* u;

flag ('(taylorseriesp taylorcombine taylortostandard taylororiginal
        taylortemplate),
      'opfn);

flag ('(taylorseriesp), 'boolean);

endmodule;

module TayFns;

%*****************************************************************
%
%       Simplification functions for special functions
%
%*****************************************************************


exports taysimpexpt, taysimpatan, taysimplog, taysimpexp,
        taysimptan, taysimpsin, taysimpasin$

imports

% from the REDUCE kernel:
        !*f2q, !*k2q, !*n2f, addsq, aeval, denr, domainp, eqcar,
        evenp, invsq, lastpair, lprim, kernp, mk!*sq, mksq, multsq,
        mvar, negsq, nth, numr, over, prepd, prepsq, quotsq, reval,
        reversip, simp, simp!*, subtrsq,

% from the header module:
        !*tay2q, copy!-list, cst!-Taylor!*, find!-non!-zero,
        get!-degree, make!-cst!-coefflis, make!-cst!-powerlist,
        make!-Taylor!*, set!-TayFlags, set!-TayOrig, TayCfPl, TayCfSq,
        TayCoeffList, TayFlags, TayGetCoeff, Taylor!*p,
        Taylor!-kernel!-sq!-p, TayMakeCoeff, TayOrig, TayTemplate,
        TayTpElOrder, TayTpElPoint, TayTpElVars, TayVars,

% from the module Tayintro:
        confusion, nzerolist, smemberlp, Taylor!-error, var!-is!-nth,

% from the module Tayutils:
        replace!-nth, replace!-nth!-nth,

% from the module Taybasic:
        addtaylor, addtaylor1, invtaylor, makecoeffs, makecoeffpairs,
        makecoeffpairs1, multtaylor, multtaylor1, multtaylorsq,
        negtaylor, negtaylor1, quottaylor,

% from the module Taysimp:
        expttayrat, taysimpsq,

% from the module Taydiff:
        difftaylorwrttayvar,

% from the module Tayfronted:
        taylorcombine, taylortostandard;


fluid '(!*taylorkeeporiginal);

global '(frlis!*);

symbolic procedure taysimpexpt u;
  %
  % Argument is of the form ('expt base exponent)
  % where both base and exponent (but a least one of them)
  % may contain Taylor kernels given as prefix forms.
  % Value is the equivalent Taylor kernel.
  %
  if not (car u eq 'expt) or cdddr u then confusion 'taysimpexpt
   else if cadr u eq 'e then taysimpexp {'exp, caddr u}
   else begin scalar bas, expn;
     bas := taysimpsq simp!* cadr u;
     expn := taysimpsq simp!* caddr u;
     if null kernp bas
       then if not denr bas = 1
              then return mksq ({'expt, prepsq bas, prepsq expn}, 1)
             else if domainp numr bas
              then return taysimpexp {'exp,
                  prepsq multsq (simp!* {'log, prepd numr bas}, expn)}
             else mksq ({'expt, prepsq bas, prepsq expn}, 1);
     if fixp numr expn and fixp denr expn
       then return !*tay2q expttayrat (mvar numr bas, expn);
     if denr expn = 1 and eqcar (numr expn, '!:rn!:)
       then return !*tay2q expttayrat (mvar numr bas, cdr numr expn);
     if null kernp expn
       then return mksq ({'expt, prepsq bas, prepsq expn}, 1);
     bas := mvar numr bas;
     expn := mvar numr expn;
     return if Taylor!*p bas
       then if Taylor!*p expn
         then if TayTemplate bas = TayTemplate expn
                then taysimpexp {'exp, multtaylor (expn,
                              mvar numr taysimplog {'log, bas})}
               else mksq ({'expt, bas, expn}, 1)
        else if not smemberlp (TayVars bas, expn)
               then taysimpexp {'exp,
                 multtaylorsq(mvar numr taysimplog {'log, bas},
                              !*k2q expn)}
              else mksq ({'expt, bas, expn}, 1)
      else if Taylor!*p expn and not smemberlp (TayVars expn, bas)
       then taysimpexp {'exp, multtaylorsq (expn, simp!* {'log, bas})}
      else mksq ({'expt, bas, expn}, 1);
  end$

put ('expt, 'taylorsimpfn, 'taysimpexpt)$


symbolic procedure TayCoeffList!-union u;
  if null cdr u then car u
   else TayCoeffList!-union2 (car u, TayCoeffList!-union cdr u)$

symbolic procedure TayCoeffList!-union2 (x, y);
  %
  % returns union of TayCoeffLists x and y
  %
  << for each w in y do
       if null (assoc (car w, x)) then x := w . x;
     x >>$

symbolic procedure inttaylorwrttayvar (tay, var);
  %
  % integrates Taylor kernel tay wrt variable var
  %
  inttaylorwrttayvar1 (TayCoeffList tay, TayTemplate tay, var)$

symbolic procedure inttaylorwrttayvar1 (tcl, tp, var);
  %
  % integrates Taylor kernel with TayCoeffList tcl and template tp
  %  wrt variable var
  %
  begin scalar tt, u, w; integer n, n1, m;
    u := var!-is!-nth (tp, var);
    n := car u;
    n1 := cdr u;
    tt := nth (tp, n);
    u := for each cc in tcl collect <<
           m := nth (nth (TayCfPl cc, n), n1);
           if TayTpElPoint nth (tp, n) eq 'infinity
             then <<
               if m = 1 then Taylor!-error ('inttaylorwrttayvar, nil);
               TayMakeCoeff (replace!-nth!-nth (TayCfPl cc, n, n1, m-1),
                             multsq (TayCfSq cc,
                                     invsq ((-m + 1) ./ 1))) >>
            else <<
               if m = -1 then Taylor!-error ('inttaylorwrttayvar, nil);
               TayMakeCoeff (replace!-nth!-nth (TayCfPl cc, n, n1, m+1),
                             multsq (TayCfSq cc,
                                     invsq ((m + 1) ./ 1))) >> >>;
    w := list (TayTpElVars tt, TayTpElPoint tt,
               if null car TayTpElVars tt
                 then if TayTpElPoint tt eq 'infinity
                        then TayTpElOrder tt - 1
                       else TayTpElOrder tt + 1
                else TayTpElOrder tt);
    return make!-Taylor!* (u, replace!-nth (tp, n, w), nil, nil)
%
% The following is not needed yet
%
%     return make!-Taylor!* (
%              u,
%              replace!-nth (TayTemplate tay, n, w),
%              if !*taylorkeeporiginal and TayOrig tay
%                then simp list ('int, mk!*sq TayOrig tay, var)
%               else nil,
%              TayFlags u)
  end$


comment The inverse trigonometric and inverse hyperbolic functions
        of a Taylor kernel are calculated by first computing the
        derivative(s) with respect to the Taylor variable(s) and
        integrating the result.  The derivatives can easily be
        calculated by the manipulation functions defined above.

        The method is best illustrated with an example.  Let T(x)
        be a Taylor kernel depending on one variable x.  To compute
        the inverse tangent T1(x) = atan(T(x)) we calculate the
        derivative

                                T'(x)
                    T1'(x) = ----------- .
                                      2
                              1 + T(x)

        (If T and T1 depend on more than one variable replace
        the derivatives by gradients.)
        This is integrated again with the integration constant
        T1(x0) = atan(T(x0)) yielding the desired result.
        If there is more than variable we have to find the
        potential function T1(x1,...,xn) corresponding to
        the vector grad T1(x1,...,xn) which is always possible
        by construction.

        The prescriptions for the eight functions asin, acos, asec,
        acsc, asinh, acosh, asech, and acsch can be put together
        in one procedure since the expressions for their derivatives
        differ only in certain signs.  The same remark applies to
        the four functions atan, acot, atanh, and acoth.

        These expressions are:

         d                 1
         -- asin x = ------------- ,
         dx           sqrt(1-x^2)

         d                -1
         -- acos x = ------------- ,
         dx           sqrt(1-x^2)

         d                 1
         -- asinh x = ------------- ,
         dx            sqrt(1+x^2)

         d                 1
         -- acosh x = ------------- ,
         dx            sqrt(x^2-1)

         d               1
         -- atan x = --------- ,
         dx           1 + x^2

         d               -1
         -- acot x = --------- ,
         dx           1 + x^2

         d                1
         -- atanh x = --------- ,
         dx            1 - x^2

         d                1
         -- acoth x = --------- ,
         dx            1 - x^2

        together with the relations

                       1
         asec x = acos - ,
                       x

                       1
         acsc x = asin - ,
                       x

                         1
         asech x = acosh - ,
                         x

                         1
         acsch x = asinh -
                         x ;


symbolic procedure taysimpasin u;
  if not (car u memq '(asin acos acsc asec asinh acosh acsch asech))
     or cddr u
    then confusion 'taysimpasin
   else begin scalar l, l0, c0, v, tay, tay2, tp;
     tay := taysimpsq simp!* cadr u;
     if not Taylor!-kernel!-sq!-p tay
       then return mksq (list (car u, mk!*sq tay), 1);
     tay := mvar numr tay; % asin's argument
     if car u memq '(asec acsc asech acsch) then tay := invtaylor tay;
     tp := TayTemplate tay;
     l0 := make!-cst!-powerlist tp;
     c0 := car TayCoeffList tay;
     for each el in TayCfPl c0 do
       if get!-degree el < 0
         then Taylor!-error ('essential!-singularity, car u);
     tay2 := multtaylor (tay, tay);
     if car u memq '(asin acos acsc asec)
       then tay2 := negtaylor tay2;
     tay2 := addtaylor (
               cst!-Taylor!* (
                 !*f2q (if car u memq '(acosh asech) then -1 else 1),
                 tp),
               tay2);
     if null numr TayGetCoeff (l0, TayCoeffList tay2)
       then Taylor!-error ('essential!-singularity, car u);
     tay2 := invtaylor expttayrat (tay2, 1 ./ 2);
     if car u eq '(acos asec) then tay2 := negtaylor tay2;
     l := for each krnl in TayVars tay collect
            TayCoeffList inttaylorwrttayvar (
              multtaylor (difftaylorwrttayvar (tay, krnl), tay2),
              krnl);
     v := TayCoeffList!-union l;
     %
     % special treatment for zeroth coefficient
     %
     c0 := simp list (car u,
                      mk!*sq TayGetCoeff (l0, TayCoeffList tay));
     v := TayMakeCoeff (l0, c0) . v;
     return !*tay2q make!-Taylor!* (v, tp,
                  if !*taylorkeeporiginal and TayOrig tay
                    then simp list (car u, mk!*sq TayOrig tay)
                   else nil,
                  TayFlags tay)
  end$

put('asin, 'taylorsimpfn, 'taysimpasin)$
put('acos, 'taylorsimpfn, 'taysimpasin)$
put('asec, 'taylorsimpfn, 'taysimpasin)$
put('acsc, 'taylorsimpfn, 'taysimpasin)$
put('asinh, 'taylorsimpfn, 'taysimpasin)$
put('acosh, 'taylorsimpfn, 'taysimpasin)$
put('asech, 'taylorsimpfn, 'taysimpasin)$
put('acsch, 'taylorsimpfn, 'taysimpasin)$


symbolic procedure taysimpatan u;
  if not (car u memq '(atan acot atanh acoth)) or cddr u
    then confusion 'taysimpatan
   else begin scalar l, l0, c0, v, tay, tay2, tp;
     tay := taysimpsq simp!* cadr u;
     if not Taylor!-kernel!-sq!-p tay
       then return mksq (list (car u, mk!*sq tay), 1);
     tay := mvar numr tay; % atan's argument
     tp := TayTemplate tay;
     l0 := make!-cst!-powerlist tp;
     c0 := car TayCoeffList tay;
     for each el in TayCfPl c0 do
       if get!-degree el < 0
         then Taylor!-error ('branch!-cut, car u);
     tay2 := multtaylor (tay, tay);
     if car u memq '(atanh acoth) then tay2 := negtaylor tay2;
     tay2 := addtaylor (cst!-Taylor!* (1 ./ 1, tp), tay2);
     if null numr TayGetCoeff (l0, TayCoeffList tay2)
       then Taylor!-error ('essential!-singularity, car u);
     tay2 := invtaylor tay2;
     if car u eq 'acot then tay2 := negtaylor tay2;
     l := for each krnl in TayVars tay collect
            TayCoeffList inttaylorwrttayvar (
              multtaylor (difftaylorwrttayvar (tay, krnl), tay2),
              krnl);
     v := TayCoeffList!-union l;
     %
     % special treatment for zeroth coefficient
     %
     c0 := simp list (car u,
                      mk!*sq TayGetCoeff (l0, TayCoeffList tay));
     v := TayMakeCoeff (l0, c0) . v;
     return !*tay2q make!-Taylor!* (v, tp,
                  if !*taylorkeeporiginal and TayOrig tay
                    then simp list (car u, mk!*sq TayOrig tay)
                   else nil,
                  TayFlags tay)
  end$

put('atan, 'taylorsimpfn, 'taysimpatan)$
put('acot, 'taylorsimpfn, 'taysimpatan)$
put('atanh, 'taylorsimpfn, 'taysimpatan)$
put('acoth, 'taylorsimpfn, 'taysimpatan)$


comment For the logarithm and exponential we use the extension of
        an algorithm quoted by Knuth who shows how to do this for
        series in one expansion variable.

        We extended this to the case of several variables which is
        straightforward except for one point, see below.
        Knuth's algorithm works as follows: Assume you want to compute
        the series W(x) where

            W(x) = log V(x)

        Differentiation of this equation gives

                    V'(x)
            W'(x) = ----- ,   or V'(x) = W'(x)V(x) .
                     V(x)

        You make now the ansatz

                    -----
                    \           n
            W(x) =   >      W  x  ,
                    /        n
                    -----

        substitute this into the above equation and compare
        powers of x.  This yields the recursion formula

                               n-1
                 V            -----
                  n       1   \
           W  = ---- - ------  >    m W  V     .
            n    V      n V   /        m  n-m
                  0        0  -----
                               m=0

        The first coefficient must be calculated directly, it is

           W   = log V  .
            0         0

        To use this for series in more than one variable you have to
        calculate all partial derivatives: n and m refer then to the
        corresponding component of the multi index.  Looking closely
        one finds that there is an ambiguity: the same coefficient
        can be calculated using any of the partial derivatives.  The
        only restriction is that the corresponding component of the
        multi index must not be zero, since we have to divide by it.

        We resolve this ambiguity by simply taking the first nonzero
        component.

        The case of the exponential is nearly the same: differentiation
        gives

            W'(x) = V'(x) W(x) ,

        from which we derive the recursion formula

                   n-1
                  -----
                  \     n-m
            W  =   >    --- W  V     , W  = exp V  .
             n    /      m   m  n-m     0        0
                  -----
                   m=0

        ;


symbolic procedure taysimplog u;
  %
  % Special Taylor expansion function for logarithms
  %
  if not (car u eq 'log) or cddr u then confusion 'taysimplog
   else begin scalar a0, clist, coefflis, l0, l, tay, tp;
            u := taysimpsq simp!* cadr u;
    if not Taylor!-kernel!-sq!-p u
      then return mksq (list ('log, mk!*sq u), 1);
    tay := mvar numr u;
    tp := TayTemplate tay;
    l0 := make!-cst!-powerlist tp;
    %
    % The following relies on the standard ordering of the
    % TayCoeffList.
    %
    l := TayCoeffList tay;
    for each el in TayCfPl car l do
      if get!-degree el < 0
        then Taylor!-error ('essential!-singularity, 'logarithm);
    a0 := TayGetCoeff (l0, l);
    if null numr a0 then Taylor!-error ('not!-a!-unit, 'logarithm);
    clist := list TayMakeCoeff (l0, simp!* list ('log, mk!*sq a0));
    coefflis := makecoeffs for each term in tp collect
               (nzerolist length TayTpElVars term . TayTpElOrder term);
    for each cc in cdr coefflis do
      begin scalar s, pos, pp; integer n, n1;
        s := nil ./ 1;
        pos := find!-non!-zero cc;
        n := nth (nth (cc, car pos), cdr pos);
        pp := makecoeffpairs (pair (l0, cc), TayCfPl car l);
        for each p in pp do <<
          n1 := nth (nth (car p, car pos), cdr pos);
          s := addsq (s,
                      multsq (!*f2q !*n2f n1,
                              multsq (TayGetCoeff (car p, clist),
                                      TayGetCoeff (cdr p, l))))>>;
        clist :=
          TayMakeCoeff (cc,
                        quotsq (subtrsq (TayGetCoeff (cc, l),
                                         multsq (s, invsq (n ./ 1))),
                                a0))
                  . clist
      end;
    return !*tay2q make!-Taylor!* (
      reversip clist,
      tp,
      if !*taylorkeeporiginal and TayOrig tay
        then simp list ('log, mk!*sq TayOrig tay)
       else nil,
      TayFlags tay)
  end$

put('log, 'taylorsimpfn, 'taysimplog)$


symbolic procedure taysimpexp u;
  %
  % Special Taylor expansion function for exponentials
  %
  if not (car u eq 'exp) or cddr u then confusion 'taysimpexp
   else begin scalar a0, clist, coefflis, l0, l, tay, tp;
    u := taysimpsq simp!* cadr u;
    if not Taylor!-kernel!-sq!-p u
      then return mksq (list ('exp, mk!*sq u), 1);
    tay := mvar numr u;
    tp := TayTemplate tay;
    l0 := make!-cst!-powerlist tp;
    %
    % The following relies on the standard ordering of the
    % TayCoeffList.
    %
    l := TayCoeffList tay;
    for each el in TayCfPl car l do
      if get!-degree el < 0
        then Taylor!-error ('essential!-singularity, 'exp);
    a0 := TayGetCoeff (l0, l);
    clist := list TayMakeCoeff (l0, simp!* list ('exp, mk!*sq a0));
    coefflis := makecoeffs for each term in tp collect
               (nzerolist length TayTpElVars term . TayTpElOrder term);
    for each cc in cdr coefflis do
      begin scalar s, pos, pp; integer n, n1;
        s := nil ./ 1;
        pos := find!-non!-zero cc;
        n := nth (nth (cc, car pos), cdr pos);
        pp := makecoeffpairs (pair (l0, cc), TayCfPl car l);
        for each p in pp do <<
          n1 := nth (nth (car p, car pos), cdr pos);
          s := addsq (s,
                      multsq (!*f2q !*n2f (n - n1),
                              multsq (TayGetCoeff (car p, clist),
                                      TayGetCoeff (cdr p, l))))>>;
        clist := TayMakeCoeff (cc, multsq (s, invsq (n ./ 1))) . clist
      end;
    return !*tay2q make!-Taylor!* (
      reversip clist,
      tp,
      if !*taylorkeeporiginal and TayOrig tay
        then simp list ('exp, mk!*sq TayOrig tay)
       else nil,
      TayFlags tay)
  end$

put('exp, 'taylorsimpfn, 'taysimpexp)$


comment The algorithm for the trigonometric functions is also
        derived from their differential equation.
        The simplest case is that of tangent whose equation is

                            2
           tan'(x) = 1 + tan (x) .          (*)

        For the others we have

                               2
           cot'(x) = - (1 + cot (x)),

                              2
           tanh'(x) = 1 - tanh (x),

                               2
           coth'(x) = -1 + coth (x) .



        Let T(x) be a Taylor series,

                  -----
                  \         N
           T(x) =  >    a  x
                  /      N
                  -----
                   N=0

        Now, let

                              -----
                              \         N
           T1(x) = tan T(x) =  >    b  x
                              /      N
                              -----
                               N=0

        from which we immediately deduce that b  = tan a .
                                               0        0

        From (*) we get
                              2
           T1'(x) = (1 + T1(x) ) T'(x) ,

        or, written in terms of the series:

        Inserting this into (*) we get

          -----              /     /  -----       \ 2 \  -----
          \           N-1    |     |  \         N |   |  \           L
           >    N b  x    =  | 1 + |   >    b  x  |   |   >    L a  x
          /        N         |     |  /      N    |   |  /        L
          -----              \     \  -----       /   /  -----
           N=1                         N=0                L=1

        We perform the square on the r.h.s. using Cauchy's rule
        and obtain:


           -----
           \           N-1
            >    N b  x    =
           /        N
           -----
            N=1

                              N
               /     -----  -----            \  -----
               |     \      \              N |  \           L
               | 1 +  >      >    b    b  x  |   >    L a  x  .
               |     /      /      N-M  M    |  /        L
               \     -----  -----            /  -----
                      N=0    M=0                 L=1

        Expanding this once again with Cauchy's product rule we get

           -----
           \           N-1
            >    N b  x    =
           /        N
           -----
            N=1

                                L-1     N
           -----      /        -----  -----                    \
           \      L-1 |        \      \                        |
            >    x    | L a  +  >      >    b    b  (L-N) a    | .
           /          |    L   /      /      N-M  M        L-N |
           -----      \        -----  -----                    /
            L=1                 N=0    M=0

        From this we immediately deduce the recursion relation

                      L-1                 N
                     -----              -----
                     \       L-N        \
           b  = a  +  >     ----- a      >    b    b  .  (**)
            L    L   /        L    L-N  /      N-M  M
                     -----              -----
                      N=0                M=0

        This relation is easily generalized to the case of a
        series in more than one variable, where the same comments
        apply as in the case of log and exp above.

        For the hyperbolic tangent the relation is nearly the same.
        Since its differential equation has a `-' where that of
        tangent has a `+' we simply have to do the same substitution
        in the relation (**).  For the cotangent we get an additional
        overall minus sign.

        ;


symbolic procedure taysimptan u;
  %
  % Special Taylor expansion function for circular and hyperbolic
  %  tangent and cotangent
  %
  if not (car u memq '(tan cot tanh coth)) or cddr u
    then confusion 'taysimptan
   else begin scalar a0, clist, coefflis, l0, l, tay, tp;
    tay := taysimpsq simp!* cadr u;
    if not Taylor!-kernel!-sq!-p tay
      then return mksq (list (car u, mk!*sq tay), 1);
    tay := mvar numr tay;
    tp := TayTemplate tay;
    l0 := make!-cst!-powerlist tp;
    %
    % The following relies on the standard ordering of the
    % TayCoeffList.
    %
    l := TayCoeffList tay;
    for each el in TayCfPl car l do
      if get!-degree el < 0
        then Taylor!-error ('essential!-singularity, car u);
    a0 := TayGetCoeff (l0, l);
    clist := list TayMakeCoeff (l0, simp!* list (car u, mk!*sq a0));
    coefflis := makecoeffs for each term in tp collect
               (nzerolist length TayTpElVars term . TayTpElOrder term);
    for each cc in cdr coefflis do
      begin scalar cf, s, pos, pp, x, y; integer n, n1;
        s := nil ./ 1;
        pos := find!-non!-zero cc;
        n := nth (nth (cc, car pos), cdr pos);
        pp := makecoeffpairs (pair (l0, cc), l0);
        for each p in pp do <<
          x := reversip makecoeffpairs1 (pair (l0, car p), l0);
          y := nil ./ 1;
          for each z in x do
            y := addsq (y, multsq (TayGetCoeff (car z, clist),
                                   TayGetCoeff (cdr z, clist)));
          n1 := nth (nth (car p, car pos), cdr pos);
          s := addsq (s,
                      multsq (!*f2q !*n2f (n - n1),
                              multsq (y, TayGetCoeff (cdr p, l))))>>;
        cf := multsq (s, invsq (n ./ 1));
        if car u memq '(tanh coth) then cf := negsq cf;
        cf := addsq (TayGetCoeff (cc, l), cf);
        if car u memq '(cot coth) then cf := negsq cf;
        clist := TayMakeCoeff (cc, cf) . clist
      end;
    return !*tay2q make!-Taylor!* (
      reversip clist,
      tp,
      if !*taylorkeeporiginal and TayOrig tay
        then simp list (car u, mk!*sq TayOrig tay)
       else nil,
      TayFlags tay)
  end$

put('tan, 'taylorsimpfn, 'taysimptan)$
put('cot, 'taylorsimpfn, 'taysimptan)$
put('tanh, 'taylorsimpfn, 'taysimptan)$
put('coth, 'taylorsimpfn, 'taysimptan)$


comment For the sine, cosine (circular and hyperbolic) and their
        reciprocals we calculate the tangent and use the formulas

                            1
           cos x = ------------------- ,
                                   2
                    sqrt(1 + tan(x) )


                          tan x
           sin x = ------------------- ,
                                   2
                    sqrt(1 + tan(x) )


        etc.  This is not the most elegant way, but it's simpler
        to implement.

        ;


symbolic procedure taysimpsin u;
  %
  % Special Taylor expansion function for circular and hyperbolic
  %  sine, cosine and their reciprocals
  %
  if not (car u memq '(sin cos sec cosec sinh cosh sech cosech))
     or cddr u
    then confusion 'taysimpsin
   else begin scalar l, tay, tay2, tp;
    tay := taysimpsq simp!* cadr u;
    if not Taylor!-kernel!-sq!-p tay
      then return mksq (list (car u, mk!*sq tay), 1);
    tay := mvar numr tay;
    tp := TayTemplate tay;
    for each el in TayCfPl car TayCoeffList tay do
      if get!-degree el < 0
        then Taylor!-error ('essential!-singularity, car u);
    if car u memq '(sin cos sec cosec)
      then tay := taysimptan list ('tan, tay)
     else tay := taysimptan list ('tanh, tay);
    tay := mvar numr tay;
    l := TayCoeffList tay;
    l := multtaylor1 (tp, l, l);
    if car u memq '(sinh cosh sech cosech) then l := negtaylor1 l;
    tay2 := addtaylor1 (make!-cst!-coefflis (1 ./ 1, tp), l);
    %
    % check if you can calculate the root
    %
    for each ll in TayCfPl car tay2 do
       for each p in ll do
         if not evenp p
           then Taylor!-error ('branch!-point, 'taysimpsin);
    %
    % build the Taylor kernel for expttayrat
    %
    tay2 := make!-Taylor!* (tay2, tp, nil, nil);
    tay2 := expttayrat (tay2, 1 ./ 2);
    if car u memq '(cos cosh)
      then tay2 := invtaylor tay2
     else if car u memq '(sin sinh)
      then tay2 := quottaylor (tay, tay2)
     else if car u memq '(cosec cosech)
      then tay2 := quottaylor (tay2, tay);
    set!-TayOrig (tay2,
                  if !*taylorkeeporiginal and TayOrig tay
                    then simp list (car u, mk!*sq TayOrig tay));
    set!-TayFlags (tay2, TayFlags tay);
    return !*tay2q tay2
  end$

put('sin,    'taylorsimpfn, 'taysimpsin)$
put('cos,    'taylorsimpfn, 'taysimpsin)$
put('sec,    'taylorsimpfn, 'taysimpsin)$
put('cosec,  'taylorsimpfn, 'taysimpsin)$
put('sinh,   'taylorsimpfn, 'taysimpsin)$
put('cosh,   'taylorsimpfn, 'taysimpsin)$
put('sech,   'taylorsimpfn, 'taysimpsin)$
put('cosech, 'taylorsimpfn, 'taysimpsin)$

comment Support for the integration of Taylor kernels.
        Unfortunately, with the current interface, only Taylor kernels
        on toplevel can be treated successfully.

        The way it is down means stretching certain interfaces beyond
        what they were designed for...but it works!

        First we add a rule that replaces a call to INT with a Taylor
        kernel as argument by a call to TAYLORINT, then we define
        REVALTAYLORINT as simplification function for that;


put ('int, 'opmtch,
     '(((taylor!* !=x !=y !=z !=w) !=u) (nil . t)
       (taylorint !=x !=y !=z !=w !=u) nil) .
      '((!=x !=y) (nil . (smember 'taylor!* (aeval '!=x)))
        (taylorint1 !=x !=y) nil) .
      get ('int, 'opmtch));

for each x in '(!=x !=y !=z !=w !=u) do
  if not (x memq frlis!*) then frlis!* := x . frlis!*;

put('taylorint1, 'psopfn, 'revaltaylorint1);

symbolic procedure revaltaylorint1 x;
  begin scalar u, v;
    u := car x;
    v := cadr x;
    if Taylor!*p u then return revaltaylorint append (cdr u, list v);
    u := reval taylorcombine u;
    if Taylor!*p u then return revaltaylorint append (cdr u, list v);
    lprim "Converting Taylor kernels to standard representation";
    return aeval {'int, taylortostandard car x, v}
  end;

put('taylorint, 'psopfn, 'revaltaylorint);

symbolic procedure revaltaylorint u;
  begin scalar taycfl, taytp, tayorig, tayflags, var;
    taycfl := car u;
    taytp := cadr u;
    tayorig := caddr u;
    tayflags := cadddr u;
    var := car cddddr u;
    return mk!*sq !*tay2q
      if var member (for each x in taytp join copy!-list car x)
        then inttaylorwrttayvar1 (taycfl, taytp, var)
       else 
        make!-taylor!* (
          for each pp in taycfl collect 
            car pp . simp!* {'int, mk!*sq cdr pp, var},
          taytp,
          if not null tayorig
            then reval {'int, mk!*sq tayorig, var}
           else nil,
          nil)
   end;

endmodule;

module TayRevert;

%*****************************************************************
%
%       Functions for reversion of Taylor kernels
%
%*****************************************************************


exports taylorrevert;

imports

% from the REDUCE kernel:
        !*a2k, !*f2q, !*n2f, !*q2a, lastpair, mvar, neq, nth, numr,
        over, simp!*, typerr,

% from the header module:
        make!-cst!-coefflis, make!-taylor!*, multintocoefflist,
        set!-TayTemplate, TayCfPl, TayCfSq, TayCoeffList,
        TayMakeCoeff, Taylor!-kernel!-sq!-p, TayTemplate,
        TayTpElOrder,TayTpElPoint, TayTpElVars,

% from module Tayintro:
        confusion, delete!-nth, Taylor!-error,

% from module Taybasic:
        addtaylor1, invtaylor1, multtaylor1, negtaylor1,

% from module Tayutils:
        enter!-sorted;


symbolic procedure tayrevert (tay, okrnl, krnl);
  %
  % Reverts Taylor kernel tay that has okrnl as variable
  %  into a Taylor kernel that has krnl as variable.
  %
  % This is the driver procedure; it check whether okrnl
  %  is valid for this operation and calls tayrevert1 to do the work.
  %
  begin scalar tp, cfl, x; integer i;
    cfl := TayCoeffList tay;
    tp := TayTemplate tay;
    x := tp;
    i := 1;
    %
    % Loop through the template and make sure that the kernel
    %  okrnl is present and not part of a homogeneous template.
    %
   loop:
    if okrnl member TayTpElVars car x then <<
      if not null cdr TayTpElVars car x then <<
        Taylor!-error ('tayrevert,
                       {"Kernel", okrnl,
                        "appears in homogenous template", car x});
        return
        >>
       else goto found;
      >>
     else <<
       x := cdr x;
       i := i + 1;
       if not null x then goto loop
       >>;
    Taylor!-error
      ('tayrevert, {"Kernel", okrnl, "not found in template"});
    return;
   found:
    return tayrevert1 (tp, cfl, car x, i, okrnl, krnl)
  end;


symbolic procedure tayrevertreorder (cf, i);
  %
  % reorders coefflist cf so that
  %  a) part i of template is put first
  %  b) the resulting coefflist is again ordered properly
  %
  begin scalar cf1, pl, sq;
    for each pp in cf do <<
      pl := TayCfPl pp;
      sq := TayCfSq pp;
      pl := nth (pl, i) . delete!-nth (pl, i);
      cf1 := enter!-sorted (TayMakeCoeff (pl, sq), cf1)
      >>;
    return cf1
  end;

symbolic procedure tayrevertfirstdegreecoeffs (cf, n);
  %
  % Returns a coefflist that corresponds to the coefficient
  %  of (the first kernel in the template) ** n.
  %
  for each el in cf join
    if car car TayCfPl el = n and not null numr TayCfSq el
     then {TayMakeCoeff (cdr TayCfPl el, TayCfSq el)} else nil;


symbolic procedure tayrevert1 (tp, cf, el, i, okrnl, krnl);
  %
  % This is the procedure that does the real work.
  %  tp is the old template,
  %  cf the old coefflist,
  %  el the element of the template that contains the "old" kernel,
  %  i its position in the template,
  %  okrnl the old kernel,
  %  krnl the new kernel.
  %
  begin scalar el, newtp, newcf, newpoint, newel, u, u!-k, v, w, x, x1;
        integer n;
    %
    % First step: reorder the coefflist as if the okrnl appears
    %              at the beginning of the template and construct a
    %              new template by first deleting this element from it.
    %
    newcf := tayrevertreorder (cf, i);
    newtp := delete!-nth (tp, i);
    %
    % Remove zero coefficients
    %
    while null numr TayCfSq car newcf do newcf := cdr newcf;
    %
    % Check that the lowest degree of okrnl is -1, 0, or +1.
    %  For -1, we have a first order pole.
    %  For 0, reversion is possible only if the coefficient
    %   of okrnl is a constant w.r.t. the other Taylor variables.
    %  For +1, we use the algorithm quoted by Knuth,
    %   in: The Art of Computer Programming, vol2. p. 508.
    %
    n := car car TayCfPl car newcf;
    if n = -1
      then tayrevert1pole (tp, cf, el, i, okrnl, krnl, newcf, newtp);
    if n = 0
      then if not null newtp
             then begin scalar xx;
               xx := tayrevertfirstdegreecoeffs (newcf, 0);
               if length xx > 1
                 then Taylor!-error
                       ('tayrevert,
                        "Term with power 0 is a Taylor series");
               xx := car xx;
               for each el in TayCfPl xx do
                 for each el2 in el do
                   if not (el2 = 0)
                     then Taylor!-error
                           ('tayrevert,
                            "Term with power 0 is a Taylor series");
               newpoint := !*q2a TayCfSq xx;
              end
            else << newpoint := !*q2a TayCfSq car newcf;
                    newcf := cdr newcf >>
     else if n = 1
      then newpoint := 0
     else return Taylor!-error
                   ('tayrevert, "Lowest order term is not linear");
    x := tayrevertfirstdegreecoeffs (newcf, 1);
    x1 := x := invtaylor1 (newtp, x);
    w := for each pp in x1 collect
           TayMakeCoeff ({1} . TayCfPl pp, TayCfSq pp);
    v := for j := 2 : TayTpElOrder el collect
           (j . multtaylor1 (newtp,
                             tayrevertfirstdegreecoeffs (newcf, j), x));
%
    u := (0 . make!-cst!-coefflis (1 ./ 1, newtp)) . nil;
    for j := 2 : TayTpElOrder el do <<
      for k := 1 : j - 2 do begin scalar xx;
        xx := assoc (k, u);
        if null xx then confusion "revert" else u!-k := cdr xx;
        for l := k - 1 step -1 until 0 do
          u!-k := addtaylor1
                   (u!-k,
                    negtaylor1 multtaylor1 (newtp,
                                            cdr assoc (l, u),
                                            cdr assoc (k - l + 1, v)));
        rplacd (xx, u!-k);
        end;
      u!-k := multintocoefflist (cdr assoc (j, v), !*f2q !*n2f j);
      for k := 1 : j - 2 do
        u!-k := addtaylor1
                 (multintocoefflist (multtaylor1 (newtp,
                                                  cdr assoc (k, u),
                                                  cdr assoc (j - k, v)),
                                     !*f2q !*n2f (j - k)),
                  u!-k);
      u!-k := negtaylor1 u!-k;
      u := ((j - 1) . u!-k) . u;
%
      x1 := multtaylor1 (newtp, x1, x);            % x1 is now x ** j
      for each pp in
             multtaylor1 (newtp,
                          multintocoefflist
                           (cdr assoc (j - 1, u), 1 ./ !*n2f j), x1) do
        w := enter!-sorted (TayMakeCoeff ({j} . TayCfPl pp, TayCfSq pp),
                            w);
      >>;
%
      newtp := {{krnl}, newpoint, TayTpElOrder el} . newtp;
      w := enter!-sorted (
             make!-cst!-coefficient (simp!* TayTpElPoint el, newtp),
             w);

      return Make!-taylor!* (w, newtp, nil, nil)
%
  end;

comment The mechanism for a first order pole is very simple:
        This corresponds to a first order zero at infinity,
        so we invert the original kernel and revert the result;

symbolic procedure tayrevert1pole (tp, cf, el, i, okrnl, krnl,
                                   newcf, newtp);
  begin scalar x, y, z;
    cf := invtaylor1 (tp, cf);
    x := tayrevert1 (tp, cf, el, i, okrnl, krnl);
    y := TayTemplate x;
    if TayTpElPoint car y neq 0
      then Taylor!-error ('not!-implemented,
                          "(Taylor series reversion)")
     else <<
       set!-TayTemplate (x, {{krnl}, 'infinity, TayTpElOrder car y}
                              . cdr y);
       return x >>
  end;   

comment The driver routine;

symbolic procedure TaylorRevert (u, okrnl, nkrnl);
  (if not Taylor!-kernel!-sq!-p sq
     then typerr (u, "Taylor kernel")
    else tayrevert (mvar numr sq, !*a2k okrnl, !*a2k nkrnl))
     where sq := simp!* u$
  
flag ('(TaylorRevert), 'opfn);

endmodule;

end;
