/*
 *	mp.e:
 *		Public declarations of types and functions for the user
 *		of the MP package.
 */


#ifndef MP_SPEC_INCLUDED
#define MP_SPEC_INCLUDED

/******************************************************************************
		THE FOLLOWING DEFINITIONS ARE NON-PORTABLE
******************************************************************************/

#define MAX_DIGIT	0x7fff
#define MAX_INT		2147483647


#define MAX_EXPT	(MAX_INT / 4)
#define MIN_EXPT	(-MAX_EXPT)


#if 0
#ifndef MP_BLOCK_TYPE



#endif
#endif

/******************************************************************************
			    PUBLIC TYPES
******************************************************************************/

#if 0
typedef t_word		mp_int;
typedef mp_int		mp_bool;
typedef mp_int		mp_base_type;
typedef mp_int		mp_length;
typedef mp_int		mp_expt_type;
typedef mp_int		mp_sign_type;
typedef short		mp_digit_type;

typedef mp_int		mp_round_type;
typedef long		mp_debug_mask;
typedef void		(*void_func)();

#endif


/*
The public mp_float type (the fields should never be accessed by the user):
*/

#if 0
typedef struct
{
    t_handle	mp_flt_handle;
    mp_length	mp_flt_offset;
} mp_float;
#endif


#if 0
typedef t_handle		mp_float_array;
typedef t_handle		mp_vector;
typedef t_handle		mp_build_context;
#endif

/*
The internal structure of the mp_vector:
*/

typedef struct
{
    t_block_header		block_header;
    mp_float		el[VARIABLE_LENGTH];
} mp_vector_hdr;

#define MP_VECTOR_HDR_LEN	(sizeof(mp_vector_hdr) - sizeof(mp_float))

#define mp_vector_el(h, i) \
	(((mp_vector_hdr *)mem_access(h))->el[i])


/******************************************************************************
			ROUNDING TYPE CONSTANTS
******************************************************************************/

#define MP_TRUNC	0
#define MP_RND		1
#define MP_RND_DOWN	2
#define MP_RND_UP	3



/******************************************************************************
			    DEBUGGING MASKS
******************************************************************************/

#define DEBUG_BH	1
#define DEBUG_ACC	2
#define DEBUG_OTHER	4
#define DEBUG_ATAN	8
#define DEBUG_ADD	16
#define DEBUG_MUL	32
#define DEBUG_DIV	64
#define DEBUG_MULI	128
#define DEBUG_DIVI	256
#define DEBUG_BUILD	512
#define DEBUG_BREAK	1024
#define DEBUG_LOG	2048
#define DEBUG_ROOT	4096
#define DEBUG_GAMMA	8192
#define DEBUG_POWER	16384
#define DEBUG_BERN	32768
#define DEBUG_ZETA	65536
#define DEBUG_BESSEL	(1 << 16)
#define DEBUG_ERF	(1 << 17)
#define DEBUG_DAW	(1 << 18)
#define DEBUG_KARA	(1 << 19)




/******************************************************************************
			PUBLIC FUNCTIONS
******************************************************************************/


mp_float
mp_add	HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Sets z = x + y and returns z.  Accumulator operations are performed.
*/


mp_float
mp_a_to_mp	HAS_2_ARGS(
	char *,		s,
	mp_float,	z
);
/*
Scans the string s for a decimal floating point number which is placed in z
and then returned.  The number is placed in z in z's internal base.
Only decimal digits in the string are considered to be legal.  A UNIX regular
expression denoting the legal input strings is:

	-?[0-9]*(\.[0-9]*)?(e-?[0-9]*)?
*/


void
mp_bern		HAS_2_ARGS(
	mp_float_array, x,
	mp_int,		n
);
/*
Computes the Bernoulli numbers b2 = 1/6, b4 = -1/30, b6 = 1/42, b8 = -1/30,
b10 = 5/66, b12 = -691/2730, etc., defined by the generating function
y/(exp(y) - 1).	 The Bernoulli numbers b2, .., b(2n) are returned in the
array x, with b(2j) placed in x[j - 1].	 Rounding options are not implemented,
and no guard digits are used; the relative error in b(2j) is O(j^2 * b^(1 - t)).
If n is negative, abs(n) Bernoulli numbers are returned, but the comment above
about relative error no longer applies - instead the precision decreases
linearly from the first to the last bernoulli number (this is usually
sufficient if the bernoulli numbers are to be used as coefficients in an
Euler-Maclaurin expansion).  The time taken is O(t * min(n, t)^2 + n * M(t)).
*/


mp_float
mp_besj		HAS_3_ARGS(
	mp_float,	x,
	mp_int,		nu,
	mp_float,	y
);
/*
Returns y = J(nu, x), the first-kind Bessel function of order nu, for
small integer nu, and mp x and y.  Results for negative arguments are
defined by: J(-nu, x) = J(nu, -x) = (-)^nu * J(nu, x).	The method used
is Hankel's asymptotic expansion if abs(x) is large, the power series if
abs(x) is small, and the backward recurrence method otherwise.
Error could be induced by O(b^(1-t)) perturbations in x and y.
The time taken is O(t * M(t)) for fixed x and nu and increases as x and
nu increase unless x is large enough for the asymptotic series to be used.
*/




void
mp_break	HAS_5_ARGS(
	mp_float,	x,
	mp_base_type,	outbase,
	mp_length,	width,
	mp_length,	places,
	void_func,	act /* (char) */
);
/*
Breaks x into characters for output, and sends them through act().  If places
is less than zero, as many fractional places as possible are sent, otherwise
places number of places are sent.  Accumulator operations are performed.
*/


void
mp_break_expt	HAS_5_ARGS(
	mp_float,	x,
	mp_base_type,	outbase,
	mp_length,	width,
	mp_length,	places,
	void_func,	act /* (char) */
);
/*
Breaks x into characters for output in exponent form, and sends
them through act().  The exponent is printed in decimal.  The parameters
width and places are the same as for mp_break().
Accumulator operations are performed.
*/


mp_build_context
mp_build_start		HAS_2_ARGS(
	mp_float,	z,
	mp_base_type,	inbase
);
/*
Sets up a build_context for use with z.	 inbase is the base used
for subsequent build input calls.
*/


void
mp_build_point		HAS_1_ARG(
	mp_build_context,	bc
);
/*
Build process only: inserts a fractional point in the digits of the mp
number being built.  See the documentation for details.
*/


void
mp_build_minus		HAS_1_ARG(
	mp_build_context,	bc
);
/*
Build process only: inserts a minus sign in front of the mantissa/exponent
of the mp number being built.  See the documentation for details.
*/


void
mp_build_expt		HAS_1_ARG(
	mp_build_context,	bc
);
/*
Build process only: signifies the beginning of the exponent in the mp
number which is being built.  See the documentation for details.
*/


void
mp_build_digit		HAS_2_ARGS(
	mp_build_context,	bc,
	mp_digit_type,		dig
);
/*
Build process only: inserts the next digit into the mp number which is being
built.	This is called for digits of both the mantissa and the exponent.
*/


mp_float
mp_build_finish		HAS_1_ARG(
	mp_build_context *,	bptr
);
/*
Finishes a build process and returns the completed mp number.
*/


mp_float
mp_catalan	HAS_1_ARG(
	mp_float,	x
);
/*
Sets x to Catalan's constant according to the precision of x.
*/


void
mp_chebyshev	HAS_3_ARGS(
	mp_float_array,		c,
	mp_int,			n,
	mp_int,			flag
);
/*
Converts the power series coefficients c[0], ..., c[n - 1] to
Chebyshev series coefficients.	(It is assumed that the constant
term in the chebyshev sum is halved.)

flag = 0 means that c[i] is the coefficient of x^i on input,
	of T(i)(x) on output,
flag = -1 means that c[i] is the coefficient of x^(2i + 1) on input,
	of T(2i + 1)(x) on output,
flag = +1 means that c[i] is the coefficient of x^(2i) on input,
	of T(2i) on output.
*/


mp_float
mp_chev		HAS_5_ARGS(
	mp_float_array,		c,
	mp_int,			n,
	mp_int,			flag,
	mp_float,		x,
	mp_float,		y
);
/*
Returns y = chebyshev series evaluated at x.  The n coefficients are in c.
For a description of c, n, and flag, see mp_chebyshev().
*/


mp_int
mp_cmp		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Compares the mp numbers x and y, returning:

	-1	if x <	y,
	 0	if x == y,
	+1	if x >	y.
*/


mp_int
mp_cmp_abs	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Compares mp numbers abs(x) and abs(y), returning:

	-1	if abs(x) <  abs(y),
	 0	if abs(x) == abs(y),
	+1	if abs(x) >  abs(y).
*/


mp_int
mp_cmp_int	HAS_2_ARGS(
	mp_float,	x,
	mp_int,		i
);
/*
Compares the mp number x with the integer i, returning:

	-1	if x <	i,
	 0	if x == i,
	+1	if x >	i.

Accumulator operations are performed.
*/


mp_int
mp_cmp_q	HAS_3_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j
);
/*
Compares the mp number x with the rational number i/j, returning:

	-1	if x <	i/j,
	 0	if x == i/j,
	+1	if x >	i/j.

Accumulator operations are performed.
*/


mp_int
mp_cmp_double	HAS_2_ARGS(
	mp_float,	x,
	double,		d
);
/*
Compares the mp number x with the (machine) double d, returning:

	-1	if x <	d,
	 0	if x == d,
	+1	if x >	d.

Accumulator operations are performed.
*/


mp_float
mp_daw		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = Dawson's integral(x)
	  = exp(-x^2) * (integral from 0 to x of exp(u^2) du).
*/


mp_debug_mask
mp_debug_on HAS_1_ARG(
	mp_debug_mask, mask
);
/*
Turns the debug type on.
*/


mp_debug_mask
mp_debug_off HAS_1_ARG(
	mp_debug_mask, mask
);
/*
Turns the debug type off.
*/


mp_float
mp_div		HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Sets z = x / y and returns z.  Accumulator operations are performed.
*/


mp_float
mp_rec		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = 1/x and returns y.  The time taken is O(M(t)).
Accumulator operations are performed.
*/


mp_float
mp_div_int	HAS_3_ARGS(
	mp_float,	x,
	mp_int,		iy,
	mp_float,	z
);
/*
Sets z = x / (int)iy and returns z.  This is much faster than division
by an mp number.  Accumulator operations are performed.
*/


mp_float
mp_double_to_mp		HAS_2_ARGS(
	double,		d,
	mp_float,	x
);
/*
Converts the (machine) double d to the mp float x and returns x.
*/


double
mp_to_double	HAS_1_ARG(
	mp_float,	x
);
/*
Returns the (machine) double value of the mp number x.
*/


mp_float
mp_ei	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = ei(x) = -e1(-x) 
	  = principal value of integral from - infinity to x of exp(u)/u du.

The time taken is O(t * M(t)).
*/


mp_float
mp_li	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = li(x) = logarithmic integral of x
	  = principal value of integral from 0 to x of du/log(u).

The time taken is O(t * M(t)).	The condition on x is that x >= 0 && x != 1.
Error in y could be induced by an O(b^(1 - t)) relative perturbation in
x followed by similar perturbation in y.  Thus relative error in y is
small unless x is close to 1 or to the zero 1.45136923488338105028...
of li(x).
*/


mp_float
mp_erf		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = erf(x) = sqrt(4/pi) * (integral from 0 to x of exp(-u^2) du).
*/


mp_float
mp_erfc		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = erfc(x) = 1 - erf(x).
*/


mp_float
mp_eul		HAS_1_ARG(
	mp_float,	x
);
/*
Returns x = Euler's constant (gamma = 0.57721566...) to almost full
multi-precision accuracy.  The time taken is O(t^2).
*/


mp_float
mp_exp		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = exp(x) and returns y.	The time taken is O(sqrt(t)*M(t)).
*/


mp_float
mp_gam_q		HAS_3_ARGS(
	mp_int,		i,
	mp_int,		j,
	mp_float,	x
);
/*
Returns x = Gamma(i/j) for small integers i and j.  The time taken is O(t^2)
if i/j is not too large.  If i/j > 100 (approximately) it is faster to use
mp_gamma().
*/


mp_float
mp_log_gam	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = log(Gamma(x)), for positive x, using Stirling's aymptotic
expansion.  This is slower than mp_gam_q() (unless x is large), and
uses more space, so use mp_gam_q() and mp_log() if x is rational and
not too large (say <= 200).  The space required is O(t^2) (more precisely:
(t * log2(b)/12 + O(1)) * t/2 + O(t)).	The time taken is O(t^3).
*/


mp_float
mp_gamma	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = Gamma(x) using mp_gam_q() if abs(x) <= MAX_INT / 240 - 1 and
240 * x is an integer, otherwise using mp_log_gam().  The space required
is O(t^2), time O(t^3).
*/


mp_float
mp_gcd		HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Returns z = gcd(x, y) where x, y, and z are integers represented as mp
numbers and must satisfy: abs(x), abs(y) < b^t.	 gcd(x, 0) = gcd(0, x) = abs(x).
Also, gcd(x, y) >= 0.  The time taken is O(t^2).
*/


void
mp_gcd2		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns (x, y) as (x/z, y/z), where z is the gcd of x and y.  x and y
must satisy the same conditions of those in mp_gcd().
*/


mp_float
mp_cosh		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = cosh(x), x not too large.  No guard digits are used.
Accumulator operations are performed.
*/


mp_float
mp_sinh		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = sinh(x), x not too large.  No guard digits are used.
Accumulator operations are performed.
*/


mp_float
mp_tanh		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = tanh(x).  Accumulator operations are performed.
*/


mp_float
mp_int_to_mp	HAS_2_ARGS(
	mp_int, ix,
	mp_float,	z
);
/*
Converts integer ix to mp float z and returns z (called mpcim in Brent).
Assumes that abs(ix) <= b^t (otherwise ix can not usually be represented
exactly as an mp number).
*/


mp_float
mp_atan		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = arctan(x) using an O(t * M(t)) method which could easily be
modified to an O(sqrt(t) * M(t)) method (as in mp_exp1()).  y is in the
range -pi/2 to pi/2.
*/


mp_float
mp_acos		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = arccos(x), assuming that abs(x) <= 1.  y is in the range 0
to pi.	The method is to use mp_asin(), so the time taken is O(M(t) * t).
No rounding options are considered and no guard digits are used.
*/


mp_float
mp_asin		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = arcsin(x), assuming that abs(x) <= 1.  y is in the range -pi/2
to pi/2.  The method is to use mp_atan(), so the time taken is O(M(t) * t).
No rounding options are considered and no guard digits are used.
*/


mp_float
mp_atan2	HAS_3_ARGS(
	mp_float,	y,
	mp_float,	x,
	mp_float,	z
);
/*
Returns z = arctan(y/x) in the appropriate quadrant.  If both x and y are zero,
an error occurs.  If x is zero and y non-zero, z is sign(y) * pi/2.  Otherwise
z is in the range (-pi, pi] according to the quadrant which x and y indicate.
*/


mp_float
mp_log_int	HAS_2_ARGS(
	mp_int, n,
	mp_float,	x
);
/*
Returns x = log(n) for small positive integer n (2 * n must be representable).
The time taken is O(t^2).  Accumulator operations are performed.
*/


mp_float
mp_logs		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = log(1 + x) if x satisfies the condition that abs(x) < 1/b.
Otherwise there is an error.  The time taken is O(sqrt(t) * M(t))
as for mp_exp1().  Accumulator operations are performed.
*/


mp_float
mp_log		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = log(x) with the restriction that the integer part of log(x)
must be representable as an integer.  The time taken is O(sqrt(t)*M(t)
+ t^2).	 For small integer x, mp_log_int() is faster.  Asymptotically
faster methods exist (e.g. the gauss-salamin method, see mp_log_gs()),
but are not useful unless t is large.  See the comments to mp_exp1(),
and mp_pi_gl().	 Accumulator operations are performed.
*/


mp_float
mp_log_gs	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = log(x) using the Gauss-Salamin algorithm based on the arithmetic-
geometric mean iteration unless x is close to 1.  The time taken is O(log(t) *
M(t)) + O(t^2) if abs(x-1) >= 1/b, and as for mp_logs() otherwise.  This is
slower than mp_log() unless t is large (>= about 500), so is mainly useful
for testing purposes.
*/


mp_float
mp_abs		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y to the absolute value of x and returns y.
*/


mp_float
mp_accuracy	HAS_1_ARG(
	mp_float,	x
);
/*
Returns x which is set to the (multiple-precision) machine precision,
that is

	x = 1.01 * (b^(1 - t))	 (rounded up) if round = MP_TRUNC,
	     0.5 * (b^(1 - t))	 (rounded up) if round = MP_RND,
		    b^(1 - t)		      if round = MP_RND_UP or
							 MP_RND_DOWN,

where round is the current rounding type.
x is an upper bound on the smallest positive representable
number such that the relative error in the basic mp operations
(addition, subtraction, multiplication and division) is at
most x (unless the result underflows).
*/


mp_float
mp_add_int		HAS_3_ARGS(
	mp_float,	x,
	mp_int, iy,
	mp_float,	z
);
/*
Adds multiple-precision x to integer iy giving multi-precision z
and returns z.	Accumulator operations are performed.
*/


mp_float
mp_ceil		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = ceiling(x), i.e. the smallest integer not less than x.
*/


void
mp_decimal	HAS_3_ARGS(
	mp_int,		dec,
	mp_base_type *, b,
	mp_length *,	t
);
/*
Sets base *b and number of digits *t to appropriate values for use with
mp numbers with dec decimal places.
*/


mp_float
mp_epsilon	HAS_1_ARG(
	mp_float,	x
);
/*
Returns x which is set to epsilon: the smallest number such that
the sum of it and 1 is greater than 1.	This depends on the current
rounding type (as well as the base and number of digits of x).
*/


mp_sign_type
mp_float_sign HAS_1_ARG(
	mp_float,	x
);
/*
Returns the sign of the float x.
*/


mp_float
mp_floor	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = floor(x), i.e. the largest integer not greater than x.
*/


mp_float
mp_huge		HAS_1_ARG(
	mp_float,	x
);
/*
Returns x = largest possible mp number representable in x.
*/


mp_bool
mp_is_int	HAS_1_ARG(
	mp_float,	x
);
/*
Returns whether the mp_float x is an exact integer.
*/


mp_float
mp_max		HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Returns z = max(x, y).
*/


void
mp_set_round	HAS_1_ARG(
	mp_round_type,	r
);
/*
Sets the current mp rounding type to r.	 All subsequent calculations
will be performed according to this rounding type.
*/


mp_float
mp_tiny		HAS_1_ARG(
	mp_float,	x
);
/*
Returns x = smallest possible mp number representable in x.
*/


void
mp_move		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Copies the sign, exponent, and digits of x into y and either pads with zeros
or rounds appropriately.  x & y need not have the same number of digits, but
their base must be equal.This routine is mainly used to copy one mp number
to another which has a different precision (e.g. a temporary variable).
Accumulator operations are performed if t of y is less than t of x.
*/


void
mp_move_round	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Moves x + s * (b^-(t of y)) * abs(x) into y using mp_move(), where s is:
	 0	if rounding is truncating or round to nearest,
	-1	if rounding is round down,
	+1	if rounding is round up.

The restrictions on x and y are the same as those in mp_move() (q.v.).
*/


mp_float
mp_to_frac	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y to the fractional part of mp x and returns y.
i.e. y = x - integer part of x (truncated towards 0).
The current rounding type is irrelevant.
*/


mp_int
mp_to_expt_frac		HAS_3_ARGS(
	mp_float,	x,
	mp_base_type,	base,
	mp_float,	y
);
/*
Given mp x, sets mp y and returns integer n such that x = base^n * y and
1 <= abs(y) < base.  It is assumed that x is not so large or small that
n overflows.
*/


mp_int
mp_to_int	HAS_1_ARG(
	mp_float,	x
);
/*
Returns the integer part of multi-precision x, assuming that x is not too
large (else use mp_to_int_mp()).  The result is truncated towards zero,
regardless of the current rounding type.
*/


mp_float
mp_to_int_mp	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = integer part of x (truncated towards 0).  This
function should be used if x is too large to be represented as a
single-precision integer.  The current rounding type is irrelevant.
*/


mp_float
mp_mul		HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Sets z = x * y and returns z.  The simple O(t^2) algorithm is used,
with rounding performed on the digits of the product which cannot fit
into z.	 The routine is faster if x has more zero digits.  Accumulator
operations are performed.
*/


mp_float
mp_mul_int		HAS_3_ARGS(
	mp_float,	x,
	mp_int,		iy,
	mp_float,	z
);
/*
Sets z = x * (int)iy and returns z.  This is faster than multiplication
of two mp numbers.  Accumulator operations are performed.
*/


mp_float
mp_pi		HAS_1_ARG(
	mp_float,	x
);
/*
Sets x = pi to the available precision.	 The function uses Machin's identity:

	pi/4 = 4 * arctan(1/5) - arctan(1/239).

Accumulator operations are performed.
*/


mp_float
mp_pi_gl		HAS_1_ARG(
	mp_float,	x
);
/*
Returns x = pi to the available precision.  The function uses the
Gauss-Legendre algorithm which takes time O(log(t) * M(t)), so it is
slower than mp_pi() if m(t) = O(t^2), but would be faster for large t
if a faster multiplication algorithm were used.
*/


t_void
mp_poly		HAS_5_ARGS(
	mp_int,		n,
	mp_vector,	Ar,
	mp_vector,	Ai,
	mp_vector,	Xr,
	mp_vector,	Xi
);
/*
*/


void
mp_init_roots	HAS_3_ARGS(
	mp_int,		n,
	mp_vector,	Xr,
	mp_vector,	Xi
);
/*
*/


mp_float
mp_int_power	HAS_3_ARGS(
	mp_float,	x,
	mp_int,		n,
	mp_float,	y
);
/*
Sets y = x ^ n for mp x and y, and int n, and returns y.  0^0 is defined to
be 1.  Accumulator operations are performed.
*/


mp_float
mp_power	HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Returns z = x^y, where x is positive (x == 0 allowed if y > 0).
This is slower than mp_int_power() and mp_q_power(), so use them
if possible.  Accumulator operations are performed.
*/


mp_float
mp_q_power	HAS_5_ARGS(
	mp_int, i,
	mp_int, j,
	mp_int, k,
	mp_int, l,
	mp_float,	x
);
/*
Returns x = (i/j)^(k/l).  The time taken is O(t^2).  Accumulator operations
are performed.
*/


mp_float
mp_q_to_mp		HAS_3_ARGS(
	mp_int,		i,
	mp_int,		j,
	mp_float,	z
);
/*
Converts the rational number i/j to multi-precision z and returns z.
*/


mp_float
mp_add_q		HAS_4_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j,
	mp_float,	z
);
/*
Adds multiple-precision x to rational number i/j giving multi-precision z
and returns z.	The effect is the same as converting i/j to multi-precision
using mp_q_to_mp() and then adding to x using mp_add(), so see the comments
in these routines.  Accumulator operations are performed.
*/


mp_float
mp_mul_q	HAS_4_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j,
	mp_float,	y
);
/*
Multiplies mp x by i / j, giving mp result y.  Accumulator operations
are performed.
*/


mp_float
mp_mul_double_q		HAS_5_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j,
	mp_int,		k,
	mp_int,		l
);
/*
Sets x *= i * j / (k * l).  The routine calls mp_mul_q() once if i *
j and k * l are not too large, otherwise mp_mul_q() is called twice.
Rounding is not the best possible, but directed rounding is in the
correct direction.
*/


mp_float
mp_root		HAS_3_ARGS(
	mp_float,	x,
	mp_int,		n,
	mp_float,	y
);
/*
Returns y = x^(1/n) for integer n, mp x and y using Newton's method without
divisions.  The time taken is O(M(t)) unless abs(n) is large (in which
case mp_power() is used instead).  Accumulator operations are performed.
*/


mp_float
mp_sqrt		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = sqrt(x), using mp_root().
*/


mp_float
mp_sub		HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Sets z = x - y and returns z.
*/


mp_float
mp_neg		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = -x and returns y.
*/


mp_float
mp_cos		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = cos(x) and returns y.
*/


mp_float
mp_sin		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = sin(x) and returns y.
*/


mp_float
mp_tan		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = tan(x) and returns y.	Uses mp_cis(), so the time taken is
O(sqrt(t) * M(t)).  Accumulator operations are performed.
*/


mp_float
mp_zeta		HAS_2_ARGS(
	mp_int,		n,
	mp_float,	x
);
/*
Returns x = Riemann zeta(n) for n > 1 (the sum from i = 1 to infinity of i^-n). 
Uses the Euler-Maclaurin series unless n = 2, 3, 4, 6, or 8, so the
space required is O(t^2).  (More accurately, space = nl * t/2 + O(t) words,
where nl is the number of terms used in the asymptotic expansion,
nl = O(t).)  The time taken is O(t^3).
*/


void
mp_start HAS_2_ARGS(
	mp_round_type,	r,
	void_func,	callback /*(char *)*/
);
/*
Initializes the mp package and sets the internal rounding type to r.
The mp callback function is set to the parameter callback.  This function will
be called with a string containing a message when an error occurs.
*/


mp_float
mp_alloc	HAS_2_ARGS(
	mp_base_type,	b,
	mp_length,	t
);
/*
Returns an allocated mp_float with its internal base set to b, and the
number of digits to be held in it set to t.
*/


void
mp_delete_float HAS_1_ARG(
	mp_float,	z
);
/*
Deletes the float z.
*/


mp_float_array
mp_float_array_alloc	HAS_3_ARGS(
	mp_int,		dim,
	mp_base_type,	b,
	mp_length,	t
);
/*
Returns an allocated mp_float_array with room for dim (float) elements, each of
which having its internal base set to b, and the number of digits to be held
in it set to t.
*/


mp_float
mp_array_element	HAS_2_ARGS(
	mp_float_array,		a,
	mp_int,			i
);
/*
Returns the i'th element of the array a (origin 0).
*/


void
mp_delete_float_array	HAS_1_ARG(
	mp_float_array,		a
);
/*
Deletes the float array a.
*/


mp_vector
mp_vector_alloc HAS_1_ARG(
	mp_int,		dim
);
/*
Returns an allocated mp_vector with room for dim floats.
*/


void
mp_delete_vector	HAS_1_ARG(
	mp_vector,		a
);
/*
Deletes the vector a.
*/

#endif
