/*******************************************************************************
                   
anf_rel_sort.h
   
Header file for the sorting of the conjugaets of an field over an 
subfield.
    
A anf_rel_sort is a C-structure which includes three vectors:

  - m= [K:F]

  - an integer vector "ordering" containing the sorting of the conjugates:

	F^(1), ..., F^(n) the conjugated fields of F= F^(1)

	K^(1), ..., K^(n*m)  the conjugated fields of K= K^(1)

	F^(j) contained in   K^(ordering((j-1)*m+i)   (1<= i<= m)



  - an integer vector "real_zeroes(j)" containing the number of the real
    fields K^(i) over F^(j) (j fixed) for F^(j) a real field
    (for complex field their are no real extension fields)

  - an integer vector "comp_zeroes(j)" containing the number of complex and
    complex conjugates fields  K^(i) over F^(j) (j fixed) for F^(j) a real field 
    (for complex field their are not the complex and the complex conjugated 
     extension field K^(i) over F(j) )


If F^(j) a real field the conjugated fields over F^(j) are sorted
in the following way: first the real fields, then the complex fields,
and last the complex conjugated fields.
 
History:
        92-06-25 AJ        first version

*******************************************************************************/

typedef t_handle  anf_rel_sort;

#define ANF_REL_SORT_BLOCK_TYPE       11667


/*******************************************************************************
/
/  Structure definition for anf_rel_sort
/
*******************************************************************************/
 
typedef struct
	{
		t_block_header	trbl_hdr;
                                          
	 	integer_small   m;

		vector		ordering;     /* length: [K:Q] */
		vector		real_zeroes;  /* length: r1 of [F:Q] */
		vector		comp_zeroes;  /* length: r1 of [F:Q] */
	} 	
	anf_rel_sort_struct;

 
/*******************************************************************************
/
/   Macros for accessing structure items of anf_rel_sort 
/
*******************************************************************************/
  
 
/*
    access macro for table element (internal use)
    accessing the ordering (internal use)
    accessing the real_zeroes (internal use)
    accessing the comp_zeroes (internal use)
*/
  
#define anf_rel_sort_access(h)	     ((anf_rel_sort_struct *) mem_access(h))
#define anf_rel_sort_m(h)     	     (anf_rel_sort_access(h) -> m)
#define anf_rel_sort_ordering(h)     (anf_rel_sort_access(h) -> ordering)
#define anf_rel_sort_real_zeroes(h)  (anf_rel_sort_access(h) -> real_zeroes)
#define anf_rel_sort_comp_zeroes(h)  (anf_rel_sort_access(h) -> comp_zeroes)
 
/*   
    accessing a particular ordering (m is the relative index [K:F]
    accessing a particular real_zeroes
    accessing a particular comp_zeroes
*/        
 
#define anf_rel_sort_ordering_elt(h,j,i)  \
	     (vec_entry(anf_rel_sort_ordering(h), (j-1)*anf_rel_sort_m(h) +i))
#define anf_rel_sort_real_zeroes_elt(h,j) \
	     (vec_entry(anf_rel_sort_real_zeroes(h), j))
#define anf_rel_sort_comp_zeroes_elt(h,j) \
	     (vec_entry(anf_rel_sort_comp_zeroes(h), j))
  
 
/*  
    incrementing the reference counter of an ceiling
*/
  
#define anf_rel_sort_incref(h)		block_incref(h)
