#include "defs.h"
#include "integer.e"
#include "poly.h"
#include "dyn_arr.h"

#define MINEXTN 3
#define min(a, b) (a < b ? a : b)

static t_void poly_degree_n_vector_rec P_((dyn_arr_handle,t_poly,t_int));
static t_void poly_degree_vector_rec P_((dyn_arr_handle,t_poly));

t_int
poly_deg (apoly)
t_poly		apoly;
/*
** poly_deg: Returns the degree of the polynomial apoly
*/
{
	t_polyp		ap;
	t_int		nterms;

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );
	DENY(nterms == 0);

	/* exponent of the most significant term */
	return m_polyp_expt( ap, nterms - 1 );
}

t_int
poly_total_degree (apoly)
t_poly		apoly;
/* 
** Returns the total degree 
** which is defined to be the greatest sum of
** indices for each term.
*/
{
	t_polyp          ap;
        t_int   nterms;
	t_int		deg, i;
 
        ap = m_poly_to_ptr( apoly );
 	nterms = m_polyp_nterms( ap );
        DENY ( nterms == 0 );
 
	/* In the univariate case it is just the degree */
	if (m_polyp_univariate( ap ))
		return m_polyp_expt(ap, nterms - 1);

	/* In the general case, it is the maximum of the tot degrees 
	** of all the terms.
	*/

	deg = 0;
	for (i = 0; i < nterms; i++)
	{
		deg = int_max(deg, (m_polyp_expt(ap, i) + 
			poly_total_degree(m_polyp_coefft(ap, i))));
	}
	return deg;
}
		

t_int
poly_trailing_degree (apoly)
t_poly		apoly;
{
	t_polyp        ap;
	t_int   	nterms;

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );
	DENY (nterms  == 0 );
	return m_polyp_expt(ap, 0);
}

t_int 
poly_trailing_total_degree (apoly)
t_poly		apoly;
{
        t_polyp        ap;
        t_int		nterms;
        t_int           deg, i;

        ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );
       	DENY( nterms == 0 );

        /* In the univariate case it is just the degree of trailing coeff. */
        if (m_polyp_univariate( ap ))
                return m_polyp_expt(ap, 0);

        /* In the general case, it is the minimum of the tot degrees
        ** of all the terms.
        */

        deg = integer_BETA - 1;
        for (i = 0; i < nterms; i++)
        {
                deg = min(deg, (m_polyp_expt(ap, i) +
                        poly_trailing_total_degree(m_polyp_coefft(ap, i))));
        }
        return deg;

}


t_int
poly_deg_sp_var (apoly, i)
t_poly		apoly;
t_int		i;
/*
** poly_deg_sp_var: Returns the degree of apoly in the i-th var.
*/
{
	t_polyp		ap;
	t_int	nterms;
	t_int	termno;
	integer_big	res;
	t_int	temp;

	ap = m_poly_to_ptr( apoly );

	if ( i < m_polyp_princvar( ap ) )
	{
		return 0;
	}

	if ( i > m_polyp_least_pvar( ap ) )
	{
		return 0;
	}

	if ( i == m_polyp_princvar( ap ) )
	{
		/* This is the univariate case. */
		return poly_deg(apoly );
	}

	/* i less principal than m_poly_princvar( ap ) */
	nterms = m_polyp_nterms( ap );
	DENY ( nterms == 0 );

	res = 0;

	for ( termno = 0; termno < nterms; termno++ )
	{
		temp = poly_deg_sp_var( m_polyp_coefft( ap, termno ), i );
		if ( temp > res )
			res = temp;
	}

	return res;
}


static t_void
poly_degree_vector_rec (dhdl, apoly)
dyn_arr_handle		dhdl;
t_poly			apoly;
/*
** recursive element of poly_degree_vector(), assumes apoly not const
*/
{
	t_polyp		ap;
	t_int	i;
	t_int	nterms;
	t_int	var;
	t_poly	acoefft;
	t_int	len;
	t_int	degree;

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );
	var = m_polyp_princvar( ap );

	/* extend dhdl to var terms if necessary, zeroing extra terms */

	len = dyn_arr_curr_length( dhdl );

	if ( len < var )
	{
		dyn_arr_assure_space( dhdl, var, MINEXTN );

		for ( i=len; i<var; ++i )
		{
			dyn_arr_element( dhdl, i ) = 0;
		}

		dyn_arr_curr_length( dhdl ) = var;
	}

	/* check degree of apoly against current maximum */

	degree = m_polyp_expt( ap, nterms-1 );

	if ( degree > dyn_arr_element( dhdl, var-1 ) )
	{
		dyn_arr_element( dhdl, var-1 ) = degree;
	}

	/* perform same calculations for each coefficient */

	if (!m_polyp_univariate (ap))
	{
		for ( i=0; i<nterms; ++i )
		{
			acoefft = m_polyp_coefft( ap, i );
			poly_degree_vector_rec(dhdl, acoefft );
		}
	}
}



dyn_arr_handle
poly_degree_vector (apoly)
t_poly		apoly;
/*
** return a dynamic array of the degree of apoly in each variable
*/
{
	dyn_arr_handle	dhdl;

	dhdl = dyn_arr_alloc( MINEXTN );
	dyn_arr_curr_length( dhdl ) = 0;

	DENY ( m_poly_const( apoly ) );
	poly_degree_vector_rec(dhdl, apoly );

	return dhdl;
}



static t_void
poly_degree_n_vector_rec (dhdl, apoly, n)
dyn_arr_handle		dhdl;
t_poly			apoly;
t_int			n;
/*
** recursive element of poly_degree_n_vector(), assumes apoly not const
*/
{
	t_polyp	ap;
	t_int	i;
	t_int	nterms;
	t_int	var;
	t_poly	acoefft;
	t_int	degree;

	ap = m_poly_to_ptr( apoly );

	var = m_polyp_princvar( ap );
	if (var >= n)
		return;

	nterms = m_polyp_nterms( ap );

	/* check degree of apoly against current maximum */

	degree = m_polyp_expt( ap, nterms-1 );

	if ( degree > dyn_arr_element( dhdl, var ) )
	{
		dyn_arr_element( dhdl, var ) = degree;
	}

	/* perform same calculations for each coefficient */

	if (!m_polyp_univariate(ap))
	{
		for ( i=0; i<nterms; ++i )
		{
			acoefft = m_polyp_coefft( ap, i );

			poly_degree_n_vector_rec(dhdl, acoefft, n );
		}
	}
}


dyn_arr_handle
poly_degree_n_vector(apoly, n)
t_poly		apoly;
t_int		n;
/*
** Return a dynamic array of the degree of apoly in each variable up to but not
** including variable number n.  The array is padded with zeros if apoly
** doesn't have that many variables, so the array is *always* of length n.
*/
{
	dyn_arr_handle	dhdl;
	t_int	i;

	dhdl = dyn_arr_alloc( n );
	dyn_arr_curr_length(dhdl) = n;

	for (i = 0; i < n; i++)
		dyn_arr_element(dhdl, i) = 0;

	DENY ( m_poly_const( apoly ) );
	poly_degree_n_vector_rec(dhdl, apoly, n );

	return dhdl;
}

