#include "poly.h"

t_handle
poly_lbase_coefft (pring, apoly)
t_handle	pring;
t_poly   	apoly;
/*
*    POLY_LBASE_COEFFT: Returns the leading base coefficient of the
*    polynomial 
*/
{
	t_poly_context	context;

	poly_init_context(pring, &context);
	return poly_lbase_coefft_crf(&context, apoly);
}

t_handle
poly_lbase_coefft_crf (context, apoly)
t_poly_ctx	context;
t_poly   	apoly;
{
	t_polyp		ap;
	t_poly		acoefft;
	t_int		nterms;
	t_handle	cring;
	t_pfh		elt_incref;


	cring = m_poly_ctx_cring(context);
	elt_incref = m_poly_ctx_elt_incref(context);

	DENY ( m_poly_const( apoly ) );

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );


	if ( nterms == 0 )
	{
		return ring_zero(cring);
	}

	acoefft = m_polyp_coefft( ap, nterms - 1 );

	if (m_polyp_univariate(ap))
	{
		/* constant acoefft */
		return (* elt_incref)(cring, acoefft);
	}
	return poly_lbase_coefft_crf( context, acoefft );

}



t_handle
poly_tbase_coefft (pring, apoly)
t_handle	pring;
t_poly		apoly;
/*
*    POLY_TBASE_COEFFT: Returns the trailing base coefficient of the
*    polynomial
*/
{
	t_poly_context	context;

	poly_init_context(pring, &context);
	return poly_tbase_coefft_crf(&context, apoly);
}

t_handle
poly_tbase_coefft_crf (context, apoly)
t_poly_ctx	context;
t_poly   	apoly;
{
	t_polyp		ap;
	t_poly		acoefft;
	t_int		nterms;
	t_handle	cring;
	t_pfh		elt_incref;

	elt_incref = m_poly_ctx_elt_incref(context);
	cring = m_poly_ctx_cring(context);

	DENY ( m_poly_const( apoly ) );

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms(ap);

	if ( nterms == 0 )
	{
		return ring_zero(cring);
	}

	acoefft = m_polyp_coefft( ap, 0 );

	if (m_polyp_univariate(ap))
	{
		/* constant acoefft */
		return (* elt_incref)(cring, acoefft);
	}
	return poly_tbase_coefft_crf( context, acoefft );

}

t_poly
poly_ith_coefft (pring, apoly, i)
t_handle	pring;
t_poly		apoly;
t_int		i;
/*
*    POLY_ith_COEFFT: Returns the coefficient of the term of degree i
*    when the polynomial is written in recursive form. 
*/
{
	t_poly_context	context;

	poly_init_context(pring, &context);
	return poly_ith_coefft_crf(&context, apoly,i);
}

t_poly
poly_ith_coefft_crf(context, apoly,i )
t_poly_ctx	context;
t_poly		apoly;
t_int i;
{
	t_polyp		ap;
	t_handle	cring;
	t_int  		nterms, j, univ;
	t_pfh		elt_incref;


	elt_incref = m_poly_ctx_elt_incref(context);
	cring = m_poly_ctx_cring(context);


	DENY ( m_poly_const( apoly ));

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );
	univ = m_polyp_univariate(ap);

	DENY ( nterms == 0 );

	for (j = 0; j < nterms; j++)
	{
		if (m_polyp_expt(ap, j) == i)
		{
			if (univ)
			{
				return (*elt_incref)(cring, m_polyp_coefft(ap, j));
			}
			else
			{
				return  poly_elt_incref_crf(context, m_polyp_coefft(ap, j));
			}
		}
		if (m_polyp_expt(ap, j) > i)
		{
			break;
		}
	}
	if (univ)
	{
		return ring_zero(cring);
	}
	return poly_zero_poly_crf(context, m_polyp_coefft(ap, j));
}

t_poly
poly_lead_coefft (pring, apoly)
t_handle	pring;
t_poly		apoly;
/*
*    POLY_lead_COEFFT: Returns the leading coefficient of the polynomial
*    as a polynomial of pring. NOTE : this is NOT suitable as a top level function
*    as it does not wrap the x^0 on the outside. 
*
*    Don't do it with ith coefficient - takes a long time.
*/
{
	t_poly_context	context;

	poly_init_context(pring, &context);
	return poly_lead_coefft_crf(&context, apoly);
}

t_poly
poly_lead_coefft_crf (context, apoly)
t_poly_ctx	context;
t_poly		apoly;
{
	t_polyp		ap;
	t_handle	cring;
	t_int		nterms;
	t_pfh		elt_incref;


	elt_incref = m_poly_ctx_elt_incref(context);
	cring = m_poly_ctx_cring(context);

	DENY ( m_poly_const( apoly ) );

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );

	DENY ( nterms == 0 );

	if (m_polyp_univariate(ap))
	{
		return (*elt_incref)(cring, m_polyp_coefft( ap, nterms - 1 ));
	}
	return  poly_elt_incref_crf( context, m_polyp_coefft( ap, nterms - 1 ) );
}
 


t_poly
poly_trailing_coefft (pring, apoly)
t_handle	pring;
t_poly		apoly;
/*
*    POLY_TRAILING_COEFFT: Returns the TRAILING coefficient of the polynomial
*    as a polynomial of pring. NOTE : this is NOT suitable as a top level function
*    as it does not wrap the x^0 on the outside. 
*
* This can't be done with the ith degree - since the trailing coefft may 
* have any degree
*/
{
	t_poly_context	context;

	poly_init_context(pring, &context);
	return poly_trailing_coefft_crf(&context, apoly);
}

t_poly
poly_trailing_coefft_crf (context, apoly)
t_poly_ctx	context;
t_poly		apoly;
{
	t_polyp		ap;
	t_int		nterms;
	t_pfh		elt_incref;
	t_handle	cring;


	elt_incref = m_poly_ctx_elt_incref(context);
	cring = m_poly_ctx_cring(context);

	DENY ( m_poly_const( apoly ) );

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );

	DENY ( nterms == 0 );
	if (m_polyp_univariate(ap))
	{
		return (*elt_incref)(cring, m_polyp_coefft( ap,0));
	}
	return  poly_elt_incref_crf( context, m_polyp_coefft( ap, 0 ) );
}

t_handle 
poly_monomial_coefft (pring, apoly, monomial)
t_handle	pring;
t_poly		apoly;
t_poly		monomial;
/*
** Matches the degrees recursively until we 
** get to the constant level. Returns the constant as an 
** element of the coefficient ring. Doesn't incref the 
** coefficient ring
*/
{
	t_poly_context	context;

	poly_init_context(pring, &context);
	return poly_monomial_coefft_crf(&context, apoly, monomial);
}


t_handle 
poly_monomial_coefft_crf (context, apoly, monomial)
t_poly_ctx	context;
t_poly		apoly;
t_poly		monomial;
{
	t_polyp		ap, mp;
	t_int		nterms, mterms;
	t_int		lpvar, pvar, i, deg;
	t_poly		c;
	t_handle	cring;
	t_pfh		elt_incref;

	elt_incref = m_poly_ctx_elt_incref(context);
	cring = m_poly_ctx_cring(context);

	DENY ( m_poly_const( apoly ) );

	ap = m_poly_to_ptr( apoly );
	nterms = m_polyp_nterms( ap );

	DENY ( nterms == 0 );

	DENY (m_poly_const(monomial));

	mp = m_poly_to_ptr( monomial);
	mterms = m_polyp_nterms(mp);

	DENY (mterms == 0);

	/* Now we know that neither of them are constants. Now make sure they are at 
	** the same level.  least_princvars will be the same, as they are in the same
	** ring, but most pvars might not.
	*/
	if (m_polyp_princvar(ap) > m_polyp_princvar(mp))
	{
		if (m_polyp_expt(ap, 0) == 0)
			return poly_monomial_coefft_crf(context,m_polyp_coefft(ap,0),monomial);
		else
			return ring_zero(cring);
	}

	if (m_polyp_princvar(ap) < m_polyp_princvar(mp))
	{
		if (m_polyp_expt(mp, 0) == 0)
			return poly_monomial_coefft_crf(context,apoly,m_polyp_coefft(mp, 0));
		else
			return ring_zero(cring);
	}


	/* simply peel off the layers matching degrees. note - 
	** the polys should be at the same level 
	*/
	pvar =  m_polyp_princvar(ap); 
	lpvar =  m_polyp_least_pvar(ap); 
	poly_elt_incref_crf(context, apoly);
	for (i = pvar; i < lpvar; i++)
	{
		DENY(m_poly_const(apoly));
#if DEVELOP
		ap = m_poly_to_ptr(apoly);
		DENY(m_polyp_univariate(ap));
#endif
		deg = poly_deg(monomial);

		c = poly_ith_coefft_crf(context, apoly, deg);
		poly_elt_delete_crf(context, &apoly);
		apoly = c;
		/* If there is no such term, return whatever c is
		** dont increff it as poly_ith_coefft makes it and increffs it
		*/

		monomial = m_polyp_coefft(mp, 0);
		mp = m_poly_to_ptr(monomial);
	}
	/* Now we are down to the last layer -
	** i == lpvar.
	*/
	
	deg = poly_deg(monomial);
 
	c = poly_ith_coefft_crf(context, apoly, deg);
	poly_delref_crf(context, apoly);
	return c;
}
	
