#include "defs.h"
#include "mp.e"
#include "mp.h"


void
mp_priv_cis	WITH_5_ARGS(
	mp_float,	x,
	mp_float,	c,
	mp_float,	s,
	mp_bool,	do_cos,
	mp_bool,	do_sin
)
/*
Sets c = cos(x) if do_cos is true, and s = sin(x) if do_sin is true.
If do_cos is false, the cos argument is completely ignored, and likewise
for do_sin.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_expt_type	expt;
    mp_int		log2_b, j, num_divs;
    mp_acc_float	x_copy, cos1, sin, term;
    mp_ptr_type		x_copy_ptr, term_ptr;
    
    if (mp_is_zero(xp))
    {
	if (do_cos)
	    mp_int_to_mp(1, c);

	if (do_sin)
	    mp_int_to_mp(0, s);

	return;
    }

    /*
    Copy parameters of x.  Use truncated arithmetic internally.
    */

    b = mp_b(xp);
    t = mp_t(xp);
    expt = mp_expt(xp);
    round = MP_TRUNC;

    /*
    Set j to the optimum number for dividing x by a power of 2.
    */

    log2_b = mp_times_log2_b(1, b);

    j = 0;

    do
	j++;
    while (log2_b * j * j < 5 * t);


    /*
    Calculate sufficient guard digits.
    */

    new_t = t;

    if (expt > 0)
	new_t += expt;

    new_t += 1 + mp_extra_guard_digits(mp_times_log2_b(j, b), b);


    /*
    Even if the accumulator float sin is not used, we allocate it, since
    the code to allocate four floats together is very simple and more efficient
    than allocating three then possibly an extra one.
    */

    mp_acc_float_alloc_4(b, new_t, x_copy, cos1, sin, term);
    mp_change_up();

    mp_move(x, x_copy);


    /*
    Divide x_copy by suitable power of two (it will be 2^num_divs).
    */

    x_copy_ptr = mp_acc_float_ptr(x_copy);
    term_ptr = mp_acc_float_ptr(term);


#define fix_pointers()	if (mp_has_changed())				\
			{						\
			    x_copy_ptr = mp_acc_float_ptr(x_copy);	\
			    term_ptr = mp_acc_float_ptr(term);		\
			}


    for (num_divs = 0; mp_expt(x_copy_ptr) > -j; num_divs++)
    {
	mp_div_int_eq(x_copy, 2);
	fix_pointers();
    }

    /*
    If only cos necessary, square x first since we use only even terms.
    */

    if (!do_sin)
	mp_mul_eq(x_copy, x_copy);


    /*
    In the power series, we only set cos1 = cos(x) - 1, so we set cos1 = 0
    first.  Set sin = 0 if we are calculating it.   Set term = 1.
    */
    
    mp_set_sign(mp_acc_float_ptr(cos1), 0);

    if (do_sin)
	mp_set_sign(mp_acc_float_ptr(sin), 0);


    mp_int_to_mp(1, term);

    j = 0;


    /*
    Power series loop.  After n iterations, term will equal
    (-1)^n * x^2n / (2n)!.  In each iteration, term is divided by -j/(j + 1),
    and j is incremented by 2, to obtain the factorial denominator.
    */

    do
    {
	/*
	For multiplications only, we can reduce t to mul_t:
	*/

	mp_length	mul_t = mp_expt(term_ptr) + new_t;

	if (mul_t < 2)
	    mul_t = 2;
	else if (mul_t > new_t)
	    mul_t = new_t;


	/*
	Set t of x_copy and term to mul_t for multiplications.
	*/

	mp_t(x_copy_ptr) = mp_t(term_ptr) = mul_t;

	j += 2;

	mp_mul_eq(term, x_copy);
	mp_div_int_eq(term, j - 1);


	/*
	If calculating sin, add in the term.
	*/

	if (do_sin)
	{
	    /*
	    Reset t of term for adding.
	    */

	    fix_pointers();
	    mp_t(term_ptr) = new_t;

	    mp_add_eq(sin, term);


	    /*
	    Restore t of term for multiplication.
	    */

	    fix_pointers();
	    mp_t(term_ptr) = mul_t;
	}


	/*
	If calculating sin, multiply term by x_copy.  If not, x_copy has
	been set to x^2, so this is unnecessary.  Then divide by -j, so
	term = (-1)^n * x^2n / (2n)! for the nth iteration.
	*/

	if (do_sin)
	    mp_mul_eq(term, x_copy);

	mp_div_int_eq(term, -j);


	/*
	Restore t of term for addition.
	*/

	fix_pointers();
	mp_t(term_ptr) = new_t;

	mp_add_eq(cos1, term);


	/*
	Fix pointers if necessary for test of while loop and next
	iteration.  Also check for underflow of term in the test of the while.
	*/

	fix_pointers();

    } while (!mp_is_zero(term_ptr) && mp_expt(term_ptr) >= -new_t);


    /*
    From now on x_copy is used as a temporary scratch variable.
    */

    if (num_divs > 0)
    {
	/*
	Use the doubling identities to get the final result
	(i.e. sin(2x) = 2sin(x)cos(x) & cos(2x) = 2cos(x)^2 - 1).
	Since we have cos(x) - 1 instead of cos(x), we must use modified
	identities:

		sin(x) =  2sin(x/2)(1 + cos1(x/2))

		cos1(x) = cos(x) - 1
		        = 2cos(x/2)^2 - 1
		        = 2[cos1(x/2) + 1]^2 -1
		        = 2cos1(x/2)[cos1(x/2) + 2].
	*/

	mp_acc_float	two;
	
	mp_acc_float_alloc(b, new_t, two);
	mp_int_to_mp(2, two);

	mp_t(x_copy_ptr) = new_t;

	/*
	We need to apply the identities the number of times we divided x by 2.
	*/

	while (num_divs-- > 0)
	{
	    if (do_sin)
	    {
		mp_mul(cos1, sin, x_copy);
		mp_add_eq(sin, x_copy);
		mp_mul_int_eq(sin, 2);
	    }

	    /*
	    Don't bother with cos if only sin is wanted and this is the
	    last iteration.
	    */

	    if (num_divs || do_cos)
	    {
		mp_add(cos1, two, x_copy);
		mp_mul_eq(cos1, x_copy);
		mp_mul_int_eq(cos1, 2);
	    }
	}
	mp_acc_float_delete(two);
    }


    /*
    Finally we add 1 to cos1 to get cos.
    */

    if (do_cos)
	mp_add_int_eq(cos1, 1);


    /*
    Adjust for directed roundings.
    */

    if (save_round == MP_RND_UP || save_round == MP_RND_DOWN)
    {
	round = save_round;

	mp_int_to_mp(round == MP_RND_UP? 1: -1, x_copy);

	mp_acc_expt(x_copy) = 1 - t;

	if (do_cos)
	    mp_add_eq(cos1, x_copy);

	if (do_sin)
	    mp_add_eq(sin, x_copy);
    }

    if (do_cos)
    {
	/*
	Copy final cos result into c, making sure that int_abs(c) <= 1.
	*/

	mp_ptr_type	c_ptr = mp_ptr(c);

	mp_move(cos1, c);

	if (!mp_is_zero(c_ptr) && mp_expt(c_ptr) > 0)
	    mp_int_to_mp(mp_sign(c_ptr), c);
    }

    if (do_sin)
    {
	/*
	Copy final sin result into s, making sure that int_abs(s) <= 1.
	*/

	mp_ptr_type	s_ptr = mp_ptr(s);

	mp_move(sin, s);

	if (!mp_is_zero(s_ptr) && mp_expt(s_ptr) > 0)
	    mp_int_to_mp(mp_sign(s_ptr), s);
    }

    round = save_round;

    /*
    Restore accumulator stack pointer and level.
    */

    mp_acc_float_delete(term);
    mp_acc_float_delete(sin);
    mp_acc_float_delete(cos1);
    mp_acc_float_delete(x_copy);
    mp_change_down();
}



mp_float
mp_cos		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y = cos(x) and returns y.
*/
{
    mp_check_2("mp_cos", mp_ptr(x), mp_ptr(y));

    mp_cis(x, y, y, TRUE, FALSE);

    return y;
}



mp_float
mp_sin		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y = sin(x) and returns y.
*/
{
    mp_check_2("mp_sin", mp_ptr(x), mp_ptr(y));

    mp_cis(x, y, y, FALSE, TRUE);

    return y;
}



mp_float
mp_tan		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y = tan(x) and returns y.  Uses mp_cis(), so the time taken is
O(sqrt(t) * M(t)).  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	cos, sin;


    mp_check_2("mp_tan", xp, yp);

    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);
	return y;
    }

    b = mp_b(xp);
    t = mp_t(xp);

    /*
    Allocate temporary floats with extra guard digits and set internal
    rounding to truncating.
    */

    mp_acc_float_alloc_2(b, t + 1 + mp_extra_guard_digits(1, b), cos, sin);
    round = MP_TRUNC;


    /*
    Copy x into cos and then calculate cos(x) and sin(x).
    */

    mp_move(x, cos);
    mp_cis(cos, cos, sin, TRUE, TRUE);

    if (mp_is_zero(mp_acc_float_ptr(cos)))
	mp_error("mp_tan: tan(x) is too large");

    mp_div(sin, cos, cos);

    /*
    Copy result into y and restore rounding type and accumulator stack pointer.
    */

    mp_move(cos, y);
    round = save_round;

    mp_acc_float_delete(sin);
    mp_acc_float_delete(cos);

    return y;
}
