#include "defs.h"
#include "mp.e"
#include "mp.h"


void
mp_priv_int_gcd		WITH_2_ARGS(
	mp_int *,	k,
	mp_int *,	l
)
/*
Sets *k /= gcd and *l /= gcd where gcd is the greatest common divisor
of *k and *l.
*/
{
    mp_int	i = int_abs(*k), j = int_abs(*l);


    if (!j)
    {
	/*
	If i also == 0, return (0, 0), else just (1, 0).
	*/

	*l = 0;
	*k = i? 1: 0;
    }

    else
    {
	/*
	Perform Euclidean algorithm loop.
	*/

	while (i && j)
	{
	    i %= j;

	    if (i)
		j %= i;
	}

	/*
	One of i & j is zero, the other non-zero.  Set j to the non-zero one.
	*/

	if (i)
	    j = i;


	/*
	Now j is the gcd of *k and *l.
	*/

	*k /= j;
	*l /= j;
    }
}


mp_float
mp_gcd		WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
)
/*
Returns z = gcd(x, y) where x, y, and z are integers represented as mp
numbers and must satisfy: int_abs(x), int_abs(y) < b^t.  gcd(x, 0) = gcd(0, x) = int_abs(x).
Also, gcd(x, y) >= 0.  The time taken is O(t^2).
*/
{
    mp_round_type	save_round = round;
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y), zp = mp_ptr(z);
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	temp1, temp2, temp3, bottom;
    mp_ptr_type		temp1_ptr, temp2_ptr, temp3_ptr;
    mp_int		i, j, iq;


    mp_check_3("mp_gcd", xp, yp, zp);

    round = MP_TRUNC;
    b = mp_b(xp);
    t = mp_t(xp);

    mp_acc_float_alloc_3(b, t, temp1, temp2, temp3);
    bottom = temp1;

    mp_abs(x, temp1);
    mp_abs(y, temp2);

    if (!mp_is_int(temp1) || !mp_is_int(temp2))
	mp_error("mp_gcd: x or y non-integer");

    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);
    temp3_ptr = mp_acc_float_ptr(temp3);

    if (mp_is_zero(temp1_ptr))
    {
	mp_copy(temp2, z);
	goto FINISH;
    }

    if (mp_is_zero(temp2_ptr))
    {
	mp_copy(temp1, z);
	goto FINISH;
    }

    DEBUG_1("temp1 = ", temp1_ptr);
    DEBUG_1("temp2 = ", temp2_ptr);

    /*
    Check that int_abs(x), int_abs(y) < b^t.
    */

    if (mp_expt(temp1_ptr) > t || mp_expt(temp2_ptr) > t)
	mp_error("mp_gcd: x or y too large");
    
    mp_change_up();

    while (!mp_is_zero(temp2_ptr))
    {
	mp_int	diff;

	if ((diff = mp_cmp(temp1, temp2)) == 0)
	    break;

	else if (diff < 0)
	{
	    mp_acc_float	swap_temp;

	    mp_swap_vars(temp1, temp2, swap_temp);
	}

	/*
	Check for small exponent.
	*/

	if (mp_expt(temp1_ptr) <= 2)
	    break;

	/*
	Reduce t (trailing digits must be zero).
	*/

	mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) =
							mp_expt(temp1_ptr);

	mp_copy(temp2, temp3);

	/*
	Force exponents to be equal.
	*/

	mp_expt(temp3_ptr) = mp_expt(temp1_ptr);

	/*
	Get first 2 digits.
	*/

	iq = b * mp_digit(temp1_ptr, 0) + mp_digit(temp1_ptr, 1);

	if (mp_cmp(temp1, temp3) >= 0)
	{
	    iq /= (b * mp_digit(temp3_ptr, 0) + mp_digit(temp3_ptr, 1));

	    if (iq < 1)
		iq = 1;
	}

	else
	{
	    /*
	    Reduce exponent by one and underestimate quotient.
	    */

	    mp_expt(temp3_ptr)--;
	    iq /= (mp_digit(temp3_ptr, 0) + 1);
	}

	mp_mul_int_eq(temp3, iq);
	mp_sub_eq(temp2, temp3);

	if (mp_has_changed())
	{
	    temp1_ptr = mp_acc_float_ptr(temp1);
	    temp2_ptr = mp_acc_float_ptr(temp2);
	    temp3_ptr = mp_acc_float_ptr(temp3);
	}
    }

    /*
    It's safe here to use integer arithmetic.
    */

    i = mp_to_int(temp1);
    j = mp_to_int(temp2);

    while (i && j)
    {
	i %= j;

	if (i)
	    j %= i;
    }

    /*
    One and only one of i and j now has the gcd; the other is zero.
    */

    mp_int_to_mp(i + j, z);

    round = save_round;

FINISH:
    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_change_down();

    return z;
}


void
mp_gcd2		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns (x, y) as (x/z, y/z), where z is the gcd of x and y.  x and y
must satisy the same conditions of those in mp_gcd().
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round = round;
    mp_acc_float	gcd;


    round = MP_TRUNC;
    mp_acc_float_alloc(mp_b(xp), mp_t(xp), gcd);

    mp_gcd(x, y, gcd);

    /*
    Check whether x and y are equal.
    */

    if (mp_cmp(x, y) == 0)
    {
	mp_int_to_mp(mp_sign(mp_ptr(x)), x);
	mp_copy(x, y);
    }

    else
    {
	if (mp_cmp_int(gcd, MAX_INT) > 0)
	{
	    /*
	    gcd is large.
	    */

	    mp_div_eq(x, gcd);
	    mp_div_eq(y, gcd);
	}

	else
	{
	    mp_int	g = mp_to_int(gcd);

	    if (g != 1)
	    {
		mp_div_int_eq(x, g);
		mp_div_int_eq(y, g);
	    }
	}
    }

    mp_acc_float_delete(gcd);
    round = save_round;
}
