 
#include "defs.h"
#include "real.e"
#include "integer.e"
#include "dmat.h"
#include "mat.h"
#include <math.h>
#include "conv.e"
#include "debug.h"
 
#define DEBUG_LLL_INT_D_REDUCE 46
#undef DEBUG_FLAG
#define DEBUG_FLAG( n ) (v3_df[ (n) ])
 
t_void
lll_real_d_reduce WITH_5_ARGS(
        t_handle,         rrng,
        matrix,         b_orig,
        double,         delta,
        matrix *,       bh_orig,
        matrix *,       trans_orig
)
/*
** The real matrix b is LLL-reduced into *bh, using the LLLFP algorithm
** from ``Lattice Basis Reduction: Improved Practical Algorithms and Solving
** Subset Sum Problems.'' by Schnorr and Euchner.
*/
{
        double          *dmptr1;
        double          *dmptr2;
        double          *dmptr3;
        t_real          *imptr1;
        t_real          *imptr2;
        integer_small   k;
        integer_small   m;
        integer_small   n;
        matrix          b;
        Logical         bh_pack;
        matrix          bh;
        Logical         trans_pack;
        matrix          trans;
        t_dmat          c;
        t_dmat          bhprime;
        t_dmat          bhprimesn;
        double          twotomtau;
        double          twotohtau;
        double          d1;
        double          d2;
        double          d3;
        t_real          i1;
        t_real          i2;
        t_real          i3;
        integer_small   i;
        integer_small   j;
        double          muprime;
        t_real          mu;
        t_dmat          mumat;
        double          s;
        Logical         flag_c;
        Logical         flag_r;
 
#ifdef DEBUG
        if ( delta <= 0.25 || delta >= 1.00 )
        {
                error_internal( "delta not in (1/4, 1) in lll_real_d_reduce()" );
        }
#endif 
 
        m = mat_col( b_orig );
        n = mat_row( b_orig );
 
        mat_create_unpkd( rrng, b_orig, b, n, m );
 
        bh_pack = mat_result_pkd( rrng, *bh_orig );
        mat_alloc_result_unpkd( bh_pack, *bh_orig, bh, n, m );
        mat_ring_copy_sub( rrng, b, &bh );
 
        mat_free_unpkd( b_orig, b );
 
        trans_pack  = mat_result_pkd( rrng, *trans_orig );
        mat_alloc_result_unpkd( trans_pack, *trans_orig, trans, m, m );
        mat_ring_create_id_sub( rrng, trans );
 
        bhprime = dmat_new( n, m );
        c = dmat_new( 1, m );
        bhprimesn = dmat_new( 1, m );
        mumat = dmat_new( m, m );
 
/*
** HACK ALERT !!! HACK ALERT !!! 32 should be replaced by accuracy of
** double calculation accuracy in binary digits.
*/
 
        twotohtau = pow( (double) 2, (double) 32 / 2 );
        twotomtau = pow( (double) 2, (double) -32 );
 
/*
** STEP 1
*/
 
        k = 2;
 
        dmptr1 = dmat_eltptr( bhprime );
        imptr1 = mat_elt0_ptr( bh );
 
        for ( i=1; i<=m*n; ++i )
        {
                dmptr1[i] = conv_real_to_double(rrng, imptr1[i] );
        }
 
/*
** STEP 2
*/
 
        while ( k <= m )
        {
                if ( k == 2 )
                {
                        dmptr1 = dmat_eltptr( bhprime );
 
                        d1 = dmptr1[1] * dmptr1[1];
                        for ( i=m+1; i<=m*n; i+=m )
                        {
                                d1 += ( dmptr1[i] * dmptr1[i] );
                        }
                        dmat_entry( bhprimesn, 1 ) = dmat_entry( c, 1 ) = d1;
                }
 
                dmptr1 = dmat_eltptr( bhprime );
 
                d1 = dmptr1[k] * dmptr1[k];
                for ( i=m+k; i<=m*n; i+=m )
                {
                        d1 += ( dmptr1[i] * dmptr1[i] );
                }
                dmat_entry( bhprimesn, k ) = dmat_entry( c, k ) = d1;
 
                for ( j=1; j<k; ++j )
                {
                        dmptr1 = dmat_eltptr( bhprime ) + k;
                        dmptr2 = dmat_eltptr( bhprime ) + j;
 
                        d1 = dmptr1[0] * dmptr2[0];
                        for ( i=m; i<m*n; i+=m )
                        {
                                d1 += ( dmptr1[i] * dmptr2[i] );
                        }
 
                        d2 = d1 * d1;
 
                        if ( d2 < twotomtau * dmat_entry( bhprimesn, k ) * dmat_entry( bhprimesn, j
 ))
                        {
                                i1 = real_mult(rrng, mat_entry( bh, j ), mat_entry( bh, k ));
                                for ( i=m; i<m*n; i+=m )
                                {
                                        i2 = real_mult(rrng, mat_entry( bh, j+i ), mat_entry( bh, k+i ));
                                        i3 = real_add(rrng, i1, i2 );
                                        real_delete(&i2 );
                                        real_delete(&i1 );
 
                                        i1 = i3;
                                }
 
                                s = conv_real_to_double(rrng, i1 );
                                real_delete(&i1 );
                        }
                        else
                        {
                                s = d1;
                        }
 
                        dmptr1 = dmat_eltptr( mumat ) + (k-1)*m;
                        dmptr2 = dmat_eltptr( mumat ) + (j-1)*m;
                        dmptr3 = dmat_eltptr( c );
 
                        for ( i=1; i<j; i++ )
                        {
                                s -= ( dmptr2[ i ] * dmptr1[ i ] * dmptr3[ i ] );
                        }
                        d1 = dmptr1[ j ] = s / dmptr3[ j ];
 
                        dmptr3[ k ] -= ( d1 * s );
                }
 
#ifdef DEBUG
                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                {
                        cay_print( "Recalculated for k = %d\n", k );
                        cay_print( "bh =\n" );
                        print_matrix( rrng, bh, 0 );
                        cay_print( "mu =\n" );
                        print_dmat( mumat );
                        cay_print( "bhprime =\n" );
                        print_dmat( bhprime );
                        cay_print( "bhprimesn =\n" );
                        print_dmat( bhprimesn );
                        cay_print( "c =\n" );
                        print_dmat( c );
                        cay_print( "\n" );
                }
#endif 
 
/*
** STEP 3
*/
 
                flag_c = FALSE;
                flag_r = FALSE;
 
                for ( j=k-1; j>=1; --j )
                {
                        muprime = dmat_entry( mumat, (k-1)*m + j );
#ifdef DEBUG
                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                        {
                                cay_print( "muprime = %f\n", muprime );
                        }
#endif 
                        if ( muprime > 0.5 || muprime < -0.5 )
                        {
                                flag_r = TRUE;
 
                                if ( muprime > twotohtau || muprime < -twotohtau )
                                {
#ifdef DEBUG
                                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                        {
                                                cay_print( "muprime = %f, twotohtau = %f, set F_c true\n", muprime,
 twotohtau );
                                        }
#endif 
 
                                        flag_c = TRUE;
                                }
 
                                mu = conv_int_to_real(rrng, conv_double_to_int_round(muprime));
                                muprime = conv_real_to_double(rrng, mu );
#ifdef DEBUG
                                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                {
                                        cay_print( "subtracting %d by col %d from col %d\n", mu, j, k );
                                }
#endif 
 
                                dmptr1 = dmat_eltptr( mumat ) + (j-1)*m;
                                dmptr2 = dmat_eltptr( mumat ) + (k-1)*m;
 
                                for ( i=1; i<j; ++i )
                                {
                                        dmptr2[ i ] -= ( muprime * dmptr1[ i ] );
                                }
 
                                dmat_entry( mumat, (k-1)*m + j ) -= muprime;
 
                                for ( i=0; i<m*n; i+=m )
                                {
                                        i1 = mat_entry( bh, i+k );
                                        i2 = real_mult(rrng, mat_entry( bh, i+j ), mu );
                                        mat_entry( bh, i+k ) = real_subtract(rrng, i1, i2 );
                                        real_delete(&i1 );
                                        real_delete(&i2 );
                                }
 
                                for ( i=0; i<m*m; i+=m )
                                {
                                        i1 = mat_entry( trans, i+k );
                                        i2 = real_mult(rrng, mat_entry( trans, i+j ), mu );
                                        mat_entry( trans, i+k ) = real_subtract(rrng, i1, i2 );
                                        real_delete(&i1 );
                                        real_delete(&i2 );
                                }
 
                                real_delete(&mu );
#ifdef DEBUG
                                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                {
                                        cay_print( "After column addition\n" );
                                        cay_print( "bh =\n" );
                                        print_matrix( rrng, bh, 0 );
                                        cay_print( "mu =\n" );
                                        print_dmat( mumat );
                                        cay_print( "bhprime =\n" );
                                        print_dmat( bhprime );
                                        cay_print( "bhprimesn =\n" );
                                        print_dmat( bhprimesn );
                                        cay_print( "c =\n" );
                                        print_dmat( c );
                                        cay_print( "\n" );
                                }
#endif 
                        }
                }
 
                if ( flag_r )
                {
#ifdef DEBUG
                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                        {
                                cay_print( "F_r true, recalculating\n" );
                        }
#endif 
 
                        imptr1 = mat_elt0_ptr( bh );
                        dmptr1 = dmat_eltptr( bhprime );
                        d1 = 0.0;
                        for ( i=k; i<=m*n; i+=m )
                        {
                                d2 = dmptr1[ i ] = conv_real_to_double(rrng, imptr1[ i ] );
                                d1 += d2 * d2;
                        }
                        dmat_entry( bhprimesn, k ) = d1;
                }
 
                if ( flag_c )
                {
#ifdef DEBUG
                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                        {
                                cay_print( "F_c true\n" );
                        }
#endif 
 
                        if ( k > 2 )
                        {
#ifdef DEBUG
                                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                {
                                        cay_print( "decreasing k from %d to %d\n", k, k-1 );
                                }
#endif 
                                k--;
                        }
                        else
 
                        {
#ifdef DEBUG
                                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                {
                                        cay_print( "decreasing k impossible, keep at 2\n" );
                                }
#endif 
                        }
 
                        continue;
                }
 
/*
** STEP 4 */
 
                d1 = dmat_entry( mumat, (k-1)*(m+1) );  /* (k-1)*m + (k-1) */
                d2 = dmat_entry( c, k-1 );
                d3 = dmat_entry( c, k );
 
                if ( d3 < (delta - d1 * d1) * d2 )
                {
#ifdef DEBUG
                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                        {
                                cay_print( "swapping %d and %d\n", k, k-1 );
                        }
#endif 
 
                        imptr1 = mat_elt0_ptr( bh ) + k;
                        imptr2 = mat_elt0_ptr( trans ) + k;
                        dmptr1 = dmat_eltptr( bhprime ) + k;
 
                        for ( i=0; i<m*n; i+=m )
                        {
                                i1 = imptr1[ i-1 ];
                                imptr1[ i-1 ] = imptr1[ i ];
                                imptr1[ i ] = i1;
 
                                d1 = dmptr1[ i-1 ];
                                dmptr1[ i-1 ] = dmptr1[ i ];
                                dmptr1[ i ] = d1;
                        }
 
                        for ( i=0; i<m*m; i+=m )
                        {
                                i1 = imptr2[ i-1 ];
                                imptr2[ i-1 ] = imptr2[ i ];
                                imptr2[ i ] = i1;
                        }
 
                        dmptr1 = dmat_eltptr( bhprimesn );
                        d1 = dmptr1[ k ];
                        dmptr1[ k ] = dmptr1[ k-1 ];
                        dmptr1[ k-1 ] = d1;
 
#ifdef DEBUG
                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                        {
                                cay_print( "After late swap of %d and %d\n", k, k-1 );
                                cay_print( "bh =\n" );
                                print_matrix( rrng, bh, 0 );
                                cay_print( "mu =\n" );
                                print_dmat( mumat );
                                cay_print( "bhprime =\n" );
                                print_dmat( bhprime );
                                cay_print( "bhprimesn =\n" );
                                print_dmat( bhprimesn );
                                cay_print( "c =\n" );
                                print_dmat( c );
                                cay_print( "\n" );
                        }
#endif 
 
                        if ( k > 2 )
                        {
#ifdef DEBUG
                                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                {
                                        cay_print( "decreasing k from %d to %d\n", k, k-1 );
                                }
#endif 
 
                                k--;
                        }
#ifdef DEBUG
                        else
                        {
                                if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                                {
                                        cay_print( "decreasing k impossible, keep k at 2\n" );
                                }
                        }
#endif 
                }
                else
                {
#ifdef DEBUG
                        if ( DEBUG_FLAG( DEBUG_LLL_INT_D_REDUCE ))
                        {
                                cay_print( "increasing k from %d to %d\n", k, k+1 );
                        }
#endif 
 
                        k++;
                }
        }
 
        mat_create_result( rrng, bh_pack, *bh_orig, bh );
        mat_create_result( rrng, trans_pack, *trans_orig, trans );
 
        dmat_delete( &c );
        dmat_delete( &bhprime );
        dmat_delete( &bhprimesn );
        dmat_delete( &mumat );
}
