/*  inthdl_mult.c
*/

#include "defs.h"
#include "inthdl.e"
#include "intbig.h"

void
inthdl_mult_beta    WITH_3_ARGS(
    inthdl_handle,  xhdl,
    t_int,  b,
    inthdl_handle,  prod
)
/*
Multiply the integer in block with t_handle xhdl by the beta-digit b, returning
the result in block with t_handle prod, which must be a pre-allocated integer
block different from xhdl and with room for at least intbig_curr_size(xhdl) + 1
beta-digits.
*/
{
    inthdl_sign		xsign;
    t_int	xlen;

    DEBUG_INTHDL_BETA("+inthdl_mult_beta", xhdl, b);

    xsign = intbig_sign(xhdl);
    if (xsign == 0 || b == 0)
    {
	intbig_sign(prod) = 0;
	DEBUG_INTHDL_1("-inthdl_mult_beta", prod);
	return;
    }

    if (b > 0)
	intbig_sign(prod) = xsign;
    else
    {
	intbig_sign(prod) = -xsign;
	b = -b;
    }

    xlen = intbig_curr_size(xhdl);

    if (b == 1)
    {
	intbig_copy_digits(xhdl, 0, xlen, prod, 0);
	intbig_curr_size(prod) = xlen;
	DEBUG_INTHDL_1("-inthdl_mult_beta", prod);
	return;
    }

    if (xlen == 1)
    {
	/*
	xhdl is single-precision
	*/

	intbig_medium		carry, result;

	ib_mult(intbig_digit(xhdl, 0), b, &carry, &result);
	intbig_digit(prod, 0) = result;

	if (carry == 0)
	    intbig_curr_size(prod) = 1;
	else
	{
	    intbig_digit(prod, 1) = carry;
	    intbig_curr_size(prod) = 2;
	}
    }
    else
    {
	/*
	xhdl is a multi-precision number.
	*/

	register unsigned long	carry;
	register t_int	*xp, *pp, *limit;

	carry = 0;

	xp = intbig_dig0_ptr(xhdl);
	pp = intbig_dig0_ptr(prod);
	limit = xp + xlen;

	while (xp < limit)
	{
	    unsigned long	tmpcarry, result;

	    ib_mult(*xp++, b, &tmpcarry, &result);
	    result += carry;
	    carry = tmpcarry;

	    if (result >= BETA)
	    {
		result -= BETA;
		carry++;
	    }

	    *pp++ = result;
	}

	if (carry)
	    *pp++ = carry;

	intbig_curr_size(prod) = pp - intbig_dig0_ptr(prod);
    }

    DEBUG_INTHDL_1("-inthdl_mult_beta", prod);
}



void
inthdl_square	WITH_2_ARGS(
    inthdl_handle,  ahdl,
    inthdl_handle,  result
)
/*
Multiply the integer in block with t_handle ahdl by itself, returning the
result in block with t_handle result, which must be a pre-allocated integer
block different from ahdl and with room for 2 * intbig_curr_size(ahdl)
beta-digits.  This is faster than calling inthdl_mult() with ahdl passed twice.
*/
{
    inthdl_length	alen, i, j, two_i;
    t_int	*ap, *rp;
    unsigned long	carry, newcarry, prod, sum;

    DEBUG_INTHDL_1("+inthdl_sqr", ahdl);

    if (intbig_sign(ahdl) == 0)
    {
	intbig_sign(result) = 0;
	DEBUG_INTHDL_1("-inthdl_sqr", result);
	return;
    }

    intbig_sign(result) = 1;

    alen = intbig_curr_size(ahdl);

    if (alen == 1)
    {
	ib_sqr(intbig_digit(ahdl, 0), &carry, intbig_dig0_ptr(result));
	if (carry)
	{
	    intbig_digit(result, 1) = carry;
	    intbig_curr_size(result) = 2;
	}
	else
	    intbig_curr_size(result) = 1;
	DEBUG_INTHDL_1("-inthdl_sqr", result);
	return;
    }

    intbig_set_zero(result, alen << 1);

    ap = intbig_dig0_ptr(ahdl);
    rp = intbig_dig0_ptr(result);

    for (i = 0; i < alen; i++)
    {
	ib_sqr(ap[i], &carry, &prod);
	two_i = i << 1;
	sum = rp[two_i] + prod;
	carry += (sum >> ZETA);
	rp[two_i] = sum & BETA_MASK;

	for (j = i + 1; j < alen; j++)
	{
	    ib_mult(ap[i], ap[j], &newcarry, &prod);
	    sum = rp[i + j] + (prod << 1) + carry;
	    carry = (sum >> ZETA) + (newcarry << 1);
	    rp[i + j] = sum & BETA_MASK;
	}

	/*
	Store the carry in the next most significant digit
	*/

	rp[i + j] = carry;
    }

    /*
    Now check to see if the most significant digit is zero, in which case
    the result length is one shorter.
    */

    if (carry)
	intbig_curr_size(result) = (alen << 1);
    else
	intbig_curr_size(result) = (alen << 1) - 1;

    DEBUG_INTHDL_1("-inthdl_sqr", result);
}



void
inthdl_mult	WITH_3_ARGS(
    inthdl_handle,  ahdl,
    inthdl_handle,  bhdl,
    inthdl_handle,  result
)
/*
Multiply the integer in block with t_handle ahdl by the integer in block
bhdl, returning the result in block with t_handle result, which must be a
pre-allocated integer block different from ahdl and bhdl and with room for
at least intbig_curr_size(ahdl) + inhdl_curr_size(bhdl) beta-digits.
*/
{
    inthdl_sign		asign, bsign;
    inthdl_length	alen, blen;
    t_int	*ap, *bp, *bbase, *rbase, *rp, *alimit, *blimit;
    unsigned long	carry;

    DEBUG_INTHDL_2("+inthdl_mult", ahdl, bhdl);

    asign = intbig_sign(ahdl);
    bsign = intbig_sign(bhdl);

    if (asign == 0 || bsign == 0)
    {
	intbig_sign(result) = 0;
	DEBUG_INTHDL_1("-inthdl_mult", result);
	return;
    }

    if (ahdl == bhdl)
    {
	inthdl_square(ahdl, result);
	DEBUG_INTHDL_1("-inthdl_mult", result);
	return;
    }

    alen = intbig_curr_size(ahdl);
    blen = intbig_curr_size(bhdl);

    if (alen == 1)
    {
	inthdl_mult_beta(bhdl, asign * intbig_digit(ahdl, 0), result);
	DEBUG_INTHDL_1("-inthdl_mult", result);
	return;
    }

    if (blen == 1)
    {
	inthdl_mult_beta(ahdl, bsign * intbig_digit(bhdl, 0), result);
	DEBUG_INTHDL_1("-inthdl_mult", result);
	return;
    }

    /*
    Both a and b are multi-precision.  Let a have the shorter length.
    */

    if (alen > blen)
    {
	inthdl_handle	temph;
	inthdl_length	templ;

	temph = ahdl; ahdl = bhdl; bhdl = temph;
	templ = alen; alen = blen; blen = templ;
    }

    intbig_set_zero(result, alen + blen);

    /*
    Perform the long multiplication: traverse the digits of a and on
    each iteration multiply the current digit of a by each digit of b.
    */

    ap = intbig_dig0_ptr(ahdl);
    bbase = intbig_dig0_ptr(bhdl);
    rp = rbase = intbig_dig0_ptr(result);

    alimit = ap + alen;
    blimit = bbase + blen;

    while (ap < alimit)
    {
	unsigned long	adig;

	if (adig = *ap++)
	{
	    bp = bbase;
	    rp = rbase;
	    carry = 0;

	    while (bp < blimit)
	    {
		unsigned long	nextr, tmpcarry, tmpresult;

		ib_mult(adig, *bp++, &tmpcarry, &tmpresult);

		/*
		Add the result of the digits just multiplied to the
		digit in the result so far.
		*/

		nextr = *rp + tmpresult + carry;
		carry = (nextr >> ZETA) + tmpcarry;
		*rp++ = nextr & BETA_MASK;
	    }

	    /*
	    Store the carry in the next most significant digit
	    */

	    *rp++ = carry;

	}

	rbase++;

	/*
	Given that i iterations of this loop have been performed, result now
	has the product of b by the i least significant beta-digits of a.
	The result has length (i + blen) except that the most significant
	digit could be zero if there was no carry.
	*/
    }

    /*
    Now check to see if the most significant digit is zero, in which case
    the result length is one shorter.
    */

    if (carry)
	intbig_curr_size(result) = rp - intbig_dig0_ptr(result);
    else
	intbig_curr_size(result) = rp - intbig_dig0_ptr(result) - 1;

    intbig_sign(result) = asign * bsign;

    DEBUG_INTHDL_1("-inthdl_mult", result);
}
