/*  build.c

Build up an integer (perhaps a big one) presented one decimal digit at a time,
starting with the most significant digit
*/

#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"

#define	INTBIG_POWER10_MASTER
#include "power10.h"
#undef	INTBIG_POWER10_MASTER

typedef
    struct
	{
	    t_block_header		tbl_hdr;
	    integer_small	tht_dgt;
	    t_word		dc_dg_ct;
	    inthdl_handle	rs_sfr;
	    inthdl_handle	tmp_wrk;
	}
    context_data;

#define CONTEXT_LEN	MEM_BYTES_TO_WORDS(sizeof(context_data))

/*
The tracked-block type may be predefined in an include file if desired
*/
#ifndef INTEGER_BUILD_CONTEXT_TYPE
#define INTEGER_BUILD_CONTEXT_TYPE 0
#endif

#define theta_digit(h)		(((context_data *)mem_access(h))->tht_dgt)
#define dec_dig_count(h)	(((context_data *)mem_access(h))->dc_dg_ct)
#define result_sofar(h)		(((context_data *)mem_access(h))->rs_sfr)
#define temp_work(h)		(((context_data *)mem_access(h))->tmp_wrk)
	
/*
How many beta-digits to allocate initially, and to extend by when necessary:
*/

#define BETA_DIGIT_CHUNK	10


t_handle
integer_build_first	WITH_1_ARG(
	decimal_digit,	dig
)
/*
Initialize a "build context" to store working data during construction
of an integer from its decimal digits, and enter the first (i.e. most
significant or leftmost) decimal digit into the context.  Return the
t_handle of the context to the caller: the caller will pass this t_handle
back in future calls to integer_build_next and integer_build_end.

The overall calling sequence is:

	t_handle		build_context;
	integer_big	result;

	build_context = integer_build_first(leftmost_decimal_digit);
	integer_build_next(build_context, next_decimal_digit);
	integer_build_next(build_context, next_decimal_digit);
	...
	result = integer_build_end(build_context);

Note that integer_build_end deletes the build context.  The caller should
not access the build context in any way except through the integer_build_
functions provided.
*/
{
    t_handle	ctx;

    DEBUG_INTEGER_0("+integer_build_first");

    ctx = mem_alloc_words_zero(CONTEXT_LEN);
    block_init(ctx, INTEGER_BUILD_CONTEXT_TYPE);
    theta_digit(ctx) = dig;
    dec_dig_count(ctx) = 1;
    result_sofar(ctx) = temp_work(ctx) = 0;
    return ctx;
}


void
integer_build_next	WITH_2_ARGS(
	t_handle,		ctx,
	decimal_digit,	dig
)
/*
Incorporate another decimal digit in the integer being built
*/
{
    inthdl_handle	res;
    inthdl_handle	temp;

    if (++dec_dig_count(ctx) <= ETA)
    {
	theta_digit(ctx) = 10 * theta_digit(ctx) + dig;
	return;
    }

    res = result_sofar(ctx);

    if (res == 0)
    {
	res = result_sofar(ctx) = inthdl_alloc(BETA_DIGIT_CHUNK);
	intbig_sign(res) = 1;
	intbig_digit(res, 0) = theta_digit(ctx);
	intbig_curr_size(res) = 1;
    }
    else
    {
	/*
	Multiplying by THETA (<= BETA) and adding a theta-digit cannot
	increase the number of beta-digits in res by more than 1:
	*/

	intbig_assure_space(res, intbig_curr_size(res)+1, BETA_DIGIT_CHUNK);

	temp = temp_work(ctx);

	if (temp == 0)
	    temp = temp_work(ctx) = inthdl_alloc(BETA_DIGIT_CHUNK);
	else
	    intbig_assure_space(temp,
		intbig_curr_size(res)+1, BETA_DIGIT_CHUNK);

	inthdl_mult_beta(res, THETA, temp);

	inthdl_add_beta(temp, theta_digit(ctx), res);
    }

    theta_digit(ctx) = dig;
    dec_dig_count(ctx) = 1;
}


integer_big
integer_build_end	WITH_1_ARG(
	t_handle,	ctx
)
/*
Finish building the integer, then delete working data and return the value of
the integer (small or big)
*/
{
    inthdl_handle	res = result_sofar(ctx);
    inthdl_handle	temp = temp_work(ctx);
    integer_big		value;

    if (res == 0)
	value = theta_digit(ctx);
    else
    {
	intbig_assure_space(res, intbig_curr_size(res)+1, BETA_DIGIT_CHUNK);

	if (temp == 0)
	    temp = temp_work(ctx) = inthdl_alloc(intbig_curr_size(res)+1);
	else
	    intbig_assure_space(temp, intbig_curr_size(res)+1, 0);

	inthdl_mult_beta(res, intbig_power10[ dec_dig_count(ctx) ], temp);

	inthdl_add_beta(temp, theta_digit(ctx), res);
                                          
	/*
	    put the result in standard form, and delete the block if necessary
	*/
 	value = inthdl_standardize(res);
    }

    if (temp != 0)
	mem_delete_hptr(&temp);
   
    mem_delete_hptr(&ctx);

    DEBUG_INTEGER_1("-integer_build_end", value);

    return value;
}
