/*  break.c
*/

#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"
#include "power10.h"


static void
intbig_break_theta_nolead0  WITH_2_ARGS(
    t_int,  thetadigit,
    integer_break_action,	act
)
/*
Break up the given thetadigit into decimal digits.  Apply function (*act)
to each decimal digit, from most significant to least significant, except for
leading zeros.

NB: thetadigit must be non-negative; if thetadigit is zero (*act) is never
called.
*/
{
    register t_int  place;

    place = ETA - 1;

    /*
    Skip over the leading zeros
    */

    while (place >= 0 && (thetadigit / intbig_power10[place]) == 0)
	place--;

    for (; place >= 0; place--)
    {
	(*act)((thetadigit / intbig_power10[place]) + '0');
	thetadigit %= intbig_power10[place];
    } 
}


static void
intbig_break_theta_lead0    WITH_2_ARGS(
    t_int,  thetadigit,
    integer_break_action,	act
)
/*
Break up the given thetadigit into decimal digits.  Apply function (*act)
to each decimal digit, from most significant to least significant, including
leading zeros.

NB: thetadigit must be non-negative.
*/
{
    register t_int  place;
    register t_int  dig;

    for (place = ETA - 1; place >= 0; place--)
    {
	dig = thetadigit / intbig_power10[place];

	(*act)(dig + '0');
	thetadigit -= dig * intbig_power10[place];
    }
} 


void
inthdl_break    WITH_2_ARGS(
    inthdl_handle,  ahdl,
    integer_break_action,	act
)
/*
Call the action routine *act repeatedly with an argument consisting of a
character represented as a long integer, to generate a printed representation
of the big integer ('a') with t_handle ahdl.  Firstly, if a is negative, *act is
called with the character (long) '-' as argument.  Then *act is called repeatedly
with argument the characters representing the successive decimal digits of a,
starting with the most significant digit and finishing with the least.  If a
is zero, *act is called just once with '0' as argument.
*/
{
    register t_int  digit_count;
    t_int       alen;
    inthdl_handle       betanumhdl;
    inthdl_handle       tmphdl;
    inthdl_handle       tmp2hdl;
    t_int	theta_digit;

    /*
    The following variable declaration of thetanum is NOT really an integer_big.
    It is instead a sequence of THETA digits (NOT BETA digits as is the case
    for an integer_big).  The input number is split up into THETA digits, which
    are stored here, and then subsequently the THETA digits are split up into
    decimal digits.
    */

    inthdl_handle   thetanumhdl;

    DEBUG_INTHDL_1("+inthdl_break", ahdl);

    if (intbig_sign(ahdl) == 0)
    {
	(*act)('0');
	DEBUG_INTHDL_0("-inthdl_break");
	return;
    }

    alen = intbig_curr_size(ahdl);
    tmphdl = inthdl_alloc(alen);

    /*
    Deal with negatives and take a copy of the input number as the process
    is destructive
    */

    betanumhdl = inthdl_alloc(alen);

    if (intbig_sign(ahdl) < 0)
    {
	(*act)('-');

	/*
	If debugging output is on, the '-' just displayed is likely
	to be lost in the verbiage, so we draw attention to it:
	*/

	DEBUG_INTHDL_0(".inthdl_break: minus sign was output");

	inthdl_negate(ahdl, betanumhdl);
    }
    else
    { 
	intbig_sign(betanumhdl) = 1;
	intbig_copy_digits(ahdl, 0, alen, betanumhdl, 0);
	intbig_curr_size(betanumhdl) = alen;
    }

    /*
    Since the number can be represented in alen beta-digits, it is less
    than BETA to the power alen, which in turn is less than THETA to the
    power

	    ceil(alen * log10(BETA) / log10(THETA))

	<=  ceil(alen * ceil(log10(BETA)) / ETA)

	<=  ceil(alen * (ETA + 1) / ETA)

	<=  floor(alen * (ETA + 1) / ETA) + 1

    Hence if we set

	max_theta_len = alen * (ETA+1) / ETA + 1;

    (noting that the C integer division yields floor), then we can be sure
    that our number is less than THETA to the power max_theta_len, that is,
    has at most max_theta_len digits.
    */

    thetanumhdl = inthdl_alloc(alen * (ETA+1) / ETA + 1);

    /*
    Use successive division by THETA to convert the integer_big from base
    BETA to base THETA
    */

    digit_count = 0;

    while (intbig_sign(betanumhdl))
    {
	inthdl_quot_rem_beta(betanumhdl, THETA, tmphdl, &theta_digit);
	intbig_digit(thetanumhdl, digit_count) = theta_digit; 
	digit_count++;
	tmp2hdl = betanumhdl;
	betanumhdl = tmphdl;
	tmphdl = tmp2hdl;
    }

    /*
    Now convert the THETA digits to digits to the base 10
    */

    if (digit_count == 1 && intbig_digit(thetanumhdl, 0) == 0)
	(*act)('0');
    else
    {
	--digit_count;
	intbig_break_theta_nolead0(intbig_digit(thetanumhdl, digit_count), act);

	while (--digit_count >= 0)
	    intbig_break_theta_lead0(intbig_digit(thetanumhdl, digit_count), act); 
    }

    /*
    Delete working storage
    */

    mem_delete_hptr(&thetanumhdl);
    mem_delete_hptr(&betanumhdl); 
    mem_delete_hptr(&tmphdl); 

    DEBUG_INTHDL_0("-inthdl_break");
}


void
integer_break   WITH_2_ARGS(
    integer_big,    a,
    integer_break_action,	act
)
/*
Call the action routine *act repeatedly with an argument consisting of a
character represented as a long integer, to generate a printed representation
of the big integer a.  Firstly, if a is negative, *act is called with the
character (long) '-' as argument.  Then *act is called repeatedly with
argument the characters representing the successive decimal digits of a,
starting with the most significant digit and finishing with the least.  If a
is zero, *act is called just once with '0' as argument.
*/
{
    inthdl_handle	ahdl;

    if (integer_is_single(a))
    {
	if (a < 0)
	{
	    (*act)('-');
	    a = -a;
	}

	if (a < THETA)
	{
	    if (a == 0)
		(*act)('0');
	    else
		intbig_break_theta_nolead0(a, act);
	}
	else
	{
	    /* THETA <= a < BETA */
	    intbig_break_theta_nolead0(a / THETA , act);
	    intbig_break_theta_lead0(a % THETA, act);
	}
    }
    else
    {

	/* a is infinite precision */
    
	ahdl = inthdl_big_to_handle(a);
    
	inthdl_break(ahdl, act);
    }
}
