/******************************************************************************
  order_trans.c                                                           
******************************************************************************/

#include "kant.h"
#include "integer.e"
#include "mat.h"
#include "ring.h"
#include "anf.h"

order
order_trans WITH_3_ARGS(
	order,		subord,
	matrix,		trans,
	integer_big,	den
)
/*******************************************************************************
 
Description:
 
       Computes the order given by a suborder, a transformation matrix and a
       denominator. All order values of the new order are transformed to
       representations referring to the new order basis.
 
       If (and only if) the index of the new order in the old order is one
       the units (if known) are moved to the new order.
 
       The handling of ideal class group generators is not yet supported.
 
 
Calling sequence:
 
	ordnew = order_trans(ordold, trans, den);
 
      	order       ordold   = t_handle of suborder 
      	trans       gamma    = transformation matrix
        integer_big den      = denominator of transformation
      
 
History:
 
	92-03-10 JS    using anf_elt_move, handling of units, disc, maximal,
                       check for trans = identity
	91-11-19 KW    mat_z_inverse_den instead of _dirty
	91-10-21 JS    minor changes
	91-10-01 JS    written
 
*******************************************************************************/
{
	block_declarations;
	
	order		ord, ordcoef;
	integer_small	deg, degsq, one1, one2, ucnt;
	integer_big	deninv, temp1, temp2;
	matrix		invtrans;
	integer_small	i;
  

	order_must_be_over_z(subord);
 
        ordcoef = order_coef_order(subord);
	deg 	= order_rel_degree(subord);
	degsq 	= deg*deg;
                       
/* 
    perhaps the matrix is identity? 
*/
 
        if (den == 1 && mat_ring_is_one(ordcoef, trans)) 
		return order_incref(subord);
 
/*
    setting up parameters of the new order
*/
	ord = order_create(order_anf(subord), ordcoef, deg, 0);
	order_set_basis_is_rel(ord);
	order_suborder(ord) = ring_incref(subord);
	order_tran(ord)     = mat_incref(trans);
	order_tran_den(ord) = integer_incref(den);
 
	if (den == 1)
	{                    
    		/* easy case, there is no denominator */
 
               	mat_z_inverse_sub(ordcoef, trans, &order_invtran(ord));
		order_index(ord) = 1;
                if (order_is_maximal(subord)) order_set_is_maximal(ord);
	}
	else
	{
                /*    from now on we have to care for denominators  */
 
		mat_z_inverse_den(ordcoef, trans, &invtrans, &deninv);
 
		order_invtran(ord) = mat_new(deg, deg);
 
		for (i=1; i<=degsq; ++i)
		{    
			temp1 = integer_mult(mat_entry(invtrans, i), den);
			mat_entry(order_invtran(ord), i) = 
					integer_div(temp1, deninv); 
			integer_delref(temp1);
		}
 
                /*    the index is the determinant of invtrans   */
	                                
		temp1 = integer_power(den, deg);
		temp2 = integer_div(temp1, deninv);
		order_index(ord) = integer_abs(temp2);
       	
		integer_delref(temp1);
		integer_delref(temp2);
		integer_delref(deninv);
	        mat_delref(ordcoef, &invtrans);
	}
 
/*
    Do we know something about the position of the "one"?
    ... we make it easy and for the moment just check whether the "one"
    is on the same position.
*/
 
	one1 = order_one_position(subord);

	if (one1)
	{
		one2 = one1;
		for (i=1; i<=deg; ++i)
		{               
			if (i==one1)
			{
				if (integer_compare(mat_elt(trans, i, i), den))
					one2 = 0;
			}
			else
			{
				if (mat_elt(trans, i, one1))	
					one2 = 0;
			}
			if (!one2) break;
		}
		order_one_position(ord) = one2;
	}
 
/*
    handling of units and torsion unit
*/
 
        if (order_index(ord) == 1)
        {
	        ucnt=order_units_count(subord);
        	if (ucnt)
	        {
        	        order_units_count_set(ord, ucnt);
                	for (i=1; i<=ucnt; ++i)
	                {                         
        	        	order_unit(ord, i) = anf_elt_move(subord,
	                                             order_unit(subord,i),ord);
        	        }
	                if (order_units_are_fund(subord)) order_set_units_are_fund(ord);
                }
	        if (order_torsion_unit_known(subord))
	        {
                	order_torsion_rank(ord) = order_torsion_rank(subord);
	                order_torsion_unit(ord) = anf_elt_incref(order_torsion_unit(subord));
                }
        	
        }
 
 
/*
    discriminant 
*/
        
        if (order_disc_known(subord)) order_disc_calc(ord);
 
	return ord;
}
