#include "stdio.h"        
#include "kant.h"
#include "poly.e"
#include "mp.e"
#include "mp.h"
#include "real.e"
#include "anf.h"
#include "conv.e"
#include "dyn_arr.h"
  
void
anf_poly_zeroes WITH_3_ARGS(
	anf,		field,
	t_handle,		reals,
	integer_small,	ind
)
/*******************************************************************************
 
Description:
 
       Computes (or recalculates) the zeroes of Z-polynomial No. "ind" in the 
       table of minimal polynomials of the number field "field".
       The computation is done over the field "reals".        
 
       If the signature of the field was not known it is set.
 
 
Calling sequence:
 
	anf_poly_zeroes(field, R, ind)

	anf 		field    = number field 
	t_handle 		reals    = real field over which the zeroes are to be
                                   computed
        integer_small   ind      = the index of the polynomial in the 
                                   polynomial table of field
 
History:
 
	91-10-21 JS    minor changes
	91-10-01 JS    first version
 
*******************************************************************************/
{
	block_declarations;
 
	mp_float	zero, temp, tempr, tempi, temp1, eps;
	mp_vector	Ar, Xr, Xi;
	mp_length	base_places, old_base_places;
	integer_small	deg, i, j, j1, j2, r1, r2, r12, r;
	t_poly	pol;
	dyn_arr_handle	arr;
	order		ordcoef, z;
 
	deg  = anf_abs_degree(field);
	pol  = anf_poly_z_poly(field, ind);
	base_places = real_beta_prec(reals);
	
/*
    Have we to do anything?
*/
	if (anf_poly_z_zeroes_known(field, ind))
	{
	   old_base_places = real_beta_prec(anf_poly_z_zeroes_reals(field,ind));
 	   if (old_base_places == base_places)	return;
	}
 
/*
    Casting coefficients of polynomial into mp_vectors for mp_poly
*/
	Ar = mp_vector_alloc(deg+1);
             
	z  = m_z_str_incref(structure_z);
	arr = poly_to_dyn_arr(z, pol);
 
	for (i=0; i<=deg; ++i)
	{                  
		mp_vector_el(Ar,i) = mp_alloc(real_base, base_places);
		conv_int_to_mp(dyn_arr_element(arr, i), mp_vector_el(Ar, i));
		 
		integer_delref(dyn_arr_element(arr, i));
	}

	dyn_arr_delete(&arr); 
	ring_delete(&z);                               

/*
    starting values for iteration
*/
 	
	Xr = mp_vector_alloc(deg);
	Xi = mp_vector_alloc(deg);
	for (i=0; i<deg; ++i)
	{                  
		mp_vector_el(Xr,i) = mp_alloc(real_base, base_places);
		mp_vector_el(Xi,i) = mp_alloc(real_base, base_places);
	}
	if (anf_poly_z_zeroes_known(field, ind))
	{
		r1  = anf_r1(field);
		r2  = anf_r2(field);
		r12 = r1 + r2;
		ring_delref(&anf_poly_z_zeroes_reals(field, ind));
 
		for (i=1; i<=r1; ++i)
		{
			real_make_mp(anf_poly_z_zero(field, ind, i), temp);
			mp_priv_move(temp, mp_vector_el(Xr, i-1));
		} 
  
		for (i=r1+1, j=r12+1; i<=r12; ++i, ++j)
		{     
			real_make_mp(anf_poly_z_zero(field, ind, i), tempr);
			real_make_mp(anf_poly_z_zero(field, ind, j), tempi);
 
			mp_priv_move(tempr, mp_vector_el(Xr, i-1));
			mp_priv_move(tempr, mp_vector_el(Xr, j-1));
			mp_priv_move(tempi, mp_vector_el(Xi, i-1));
			mp_neg(mp_vector_el(Xi, i-1), mp_vector_el(Xi, j-1));
		}
	}
 
	else
 
	{
		mp_init_roots(deg, Xr, Xi);
	}
 
	anf_poly_z_zeroes_reals(field, ind) = ring_incref(reals);

 
/*
    Now we compute the zeroes
*/
	mp_poly(deg, Ar, 0, Xr, Xi);
         
/*
    If we had the zeroes before: we know were to find real and imaginary parts
*/
  
	if(anf_poly_z_zeroes_known(field, ind))
	{
		for (i=1; i<=r1; ++i)
		{            
			real_delete(&anf_poly_z_zero(field, ind, i));
 
			temp1 = mp_alloc(real_base, base_places);
 
			mp_copy(mp_vector_el(Xr, i-1), temp1);
 
			anf_poly_z_zero(field, ind, i) = mp_to_real(temp1);			
		} 
  
		for (i=r1+1, j=r12+1; i<=r12; ++i, ++j)
		{     
			real_delete(&anf_poly_z_zero(field, ind, i));
			real_delete(&anf_poly_z_zero(field, ind, j));
 
			tempr = mp_alloc(real_base, base_places);
			tempi = mp_alloc(real_base, base_places);
 
			mp_copy(mp_vector_el(Xr, i-1), tempr);
			mp_copy(mp_vector_el(Xi, i-1), tempi);                 
 
			anf_poly_z_zero(field, ind, i) = mp_to_real(tempr);			
			anf_poly_z_zero(field, ind, j) = mp_to_real(tempi);			
		}
	}
	
   	else
 
/*
   the difficult case: we have to find out r1 and r2!
*/      
 
	{
		anf_poly_z_zeroes_alloc(field, ind);
 
/*
    Everything below the square root of epsilon is regarded as zero.
*/
 
		eps   = mp_alloc(real_base, base_places);
                zero  = mp_alloc(real_base, base_places);
 
		mp_epsilon(eps);
		mp_sqrt(eps, eps);
		mp_int_to_mp(0, zero);
 
		for (i=0; i<deg; ++i)
		{
			if (mp_cmp_abs(mp_vector_el(Xr, i), eps) < 0)
				mp_copy(zero, mp_vector_el(Xr, i));
 
			if (mp_cmp_abs(mp_vector_el(Xi, i), eps) < 0)
				mp_copy(zero, mp_vector_el(Xi, i));
		}
 
/*
     First we identify the real zeroes
*/
		r1 = 0;
 
		for (i=0; i<deg; ++i)
		{
			if (!mp_cmp(mp_vector_el(Xi, i), zero))
			{
			   temp1 = mp_alloc(real_base, base_places);
			   mp_copy(mp_vector_el(Xr, i), temp1);
			   anf_poly_z_zero(field,ind,++r1) = mp_to_real(temp1);
			}
		}

/*
     Now the complex ones...
*/
		r2 = 0;
		j1 = r1;
		j2 = r1 + (deg-r1)/2;
 
		if (r1<deg)
		{
			for (i=0; i<deg; ++i)
			{
				if (mp_cmp(mp_vector_el(Xi, i), zero) > 0)
				{
					tempr = mp_alloc(real_base,base_places);
					mp_copy(mp_vector_el(Xr, i), tempr);
					anf_poly_z_zero(field,ind,++j1) = 
						mp_to_real(tempr);
	
					tempi = mp_alloc(real_base,base_places);
					mp_copy(mp_vector_el(Xi, i), tempi);
					anf_poly_z_zero(field,ind,++j2) = 
						mp_to_real(tempi);
				   	++r2;
				}
			}
		}  
 
		if(r1 + r2 + r2 != deg)
		    error_internal("anf_poly_zeroes: Something went wrong."); 
 
/*
    We can initialize the signature parameters for the field and all orders
*/ 
 
		anf_order_set_sig(field, r1, r2);
 
		mp_delete_float(eps);
		mp_delete_float(zero);
 
		if (anf_print_level > 2)
		  printf("Signature of field: %d = %d + 2*%d\n", deg, r1, r2);
	}   
 
	for (i=0; i<deg; ++i)
	{
		mp_delete_float(mp_vector_el(Ar, i));
		mp_delete_float(mp_vector_el(Xr, i));
		mp_delete_float(mp_vector_el(Xi, i));
	}
	mp_delete_float(mp_vector_el(Ar, deg));
 
	mp_delete_vector(Ar);
	mp_delete_vector(Xr);
	mp_delete_vector(Xi);
 
	if(anf_print_level > 3)
	{
	 	printf("Zero-vector:\n");
		for (i=1; i<=deg; ++i)
		{
			real_write(reals, anf_poly_z_zero(field, ind, i), 120);
			printf("\n");
		}
	}
 
	return;
}
