/******************************************************************************
  anf_elt_con.c                                                           
******************************************************************************/
 
#include "kant.h"

anf_elt
anf_elt_con WITH_2_ARGS(
	order,		ord,
	anf_elt,	alpha
)
/*******************************************************************************

Description: 

	Computes the conjugate vector of an algebraic number in the actual 
	precision. Calls mat_real_mat_z_mult.
 
	If some of the resulting conjugates are very near to zero errors
        because of extinction might occur (which are not detected by this
        routine. You have to increase the precision of the conjugate base of
        the order for better success.
 
Calling sequence: 
	
	alphac = anf_elt_con(ord, alpha);
                               
	order		ord    = t_handle to the order 
	anf_elt		alpha  = t_handle to algebraic number
	anf_elt		alphac = t_handle to conjugates of alpha
 
History:

	92-09-08 JS	ring_create_zero replaced by conv_int_to_real
	92-03-31 JS 	incref if alpha is already in conjugates
	91-10-01 JS 	first version
  
*******************************************************************************/
{
	block_declarations;
 
	anf_elt		alphac;
	order		ordcoef;
	integer_small	deg;
	integer_small	i, j, r1, r2, r12;
	t_handle		reals;
	t_real		ra, ra2, sq2, zero, rden;
	matrix		mata, matprod;
	matrix		mat;
 
	order_must_be_over_z(ord);
 
	ordcoef = order_coef_order(ord);
	deg     = order_abs_degree(ord);
	r1      = order_r1(ord);
	r2      = order_r2(ord);
	r12     = r1 + r2;
	reals   = order_reals(ord);
 
 
	if (anf_elt_is_integer(alpha))
	{
/*
    trivial case
*/ 
		anf_con_alloc(alphac, deg);
		ra = conv_int_to_real(reals, alpha);
 
		for (i=1; i<=r1; ++i)
			anf_con(alphac, i) = real_incref(ra);
 
		if(r2)
		{
			sq2  = order_sqrt_2(ord);
			ra2  = real_mult(reals, ra, sq2);
			zero = conv_int_to_real(reals, 0);
 
			for (i=r1+1; i<=r12; ++i)
				anf_con(alphac, i) = real_incref(ra2);
 
			for (i=r12+1; i<=deg; ++i)
				anf_con(alphac, i) = real_incref(zero);
  
			real_delete(&zero);
			real_delete(&ra2);
		}
		 
		real_delete(&ra);
	}
 
	else if anf_elt_is_con(alpha)
	{                             
                alphac = anf_elt_incref(alpha);
        }

	else
 
	{                             
/*
    general case: Multiplication of the real basis by the
    coefficient vector of alpha
*/      
		mata = mat_new(deg, 1); 
		anf_elt_to_mat_order_col(ord, alpha, mata, 1);
 
		matprod = mat_real_mat_z_mult(reals,order_basis_real(ord),mata);
  
		if (anf_elt_den(alpha) == 1)
		{
			alphac = mat_real_col_to_anf_con(ord, matprod, 1);
		}
		else
		{
			anf_con_alloc(alphac, deg);
			rden  = conv_int_to_real(reals, anf_elt_den(alpha));
			for (i=1; i<=deg; ++i)
			{
				anf_con(alphac, i) = 
				real_divide(reals, mat_entry(matprod, i), rden);
			}
			real_delete(&rden);
		}
 
		mat_delref(ordcoef, &mata);
		mat_delref(reals, &matprod);
	}
 
	return alphac;
}
