;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald THEORY)


;;; THEORY STRUCTURE TYPE

(define-structure-type THEORY
  name					; symbol
  component-theories			; set of immediate structural sub-theories
  language				; a language
  original-language-name 
  axioms				; set of theorems, 
					; i.e. formulas with names and usage lists
  top-level-axioms			; subset of axioms
  definitions				; list of definitions
  recursive-definitions			; list of recursive definitions
  sort-definitions			; list of sort definitions
  mc-extensions				; list of model conservative extensions
  theorem-hash-table			; table of known theorems (including axioms)
  transform-table			; object to expand, simplify, etc.
  valid-processors                      ; list of valid processors
  domain-range-handler			; object with info on domain and range
  distinct-constant-table		; table of constants
					; indicating pairwise distinctness
  equivalence-relations			; list of relation constants known to
					; express equivalence relations
  distinct-constant-list		; same info in list-of-lists form 
  history				; a sequence of history-events
  subsorting-structure			;caches transitive subsort relations 
  table-hash 
  
  (((name self)
    (theory-name self))
   ((print self port)
    (format port "#{IMPS-theory ~A: ~S}"
	    (object-hash self)
	    (theory-name self)))
   ((two-d-table-hash self)
    (let ((hash (theory-table-hash self)))
      (if (uncomputed? hash)
	  (let ((hash (descriptor-hash self)))
	    (set (theory-table-hash self) hash)
	    hash)
	  hash)))))

(block
  (set (theory-definitions (stype-master theory-stype)) '())
  (set (theory-recursive-definitions (stype-master theory-stype)) '())
  (set (theory-sort-definitions (stype-master theory-stype)) '())
  (set (theory-mc-extensions (stype-master theory-stype)) '())
  ;;
  ;;  (set (theory-theorem-hash-table (stype-master theory-stype)) '#f)
  ;; 
  (set (theory-transform-table (stype-master theory-stype))
       (make-table 'transform-table))
  (set (theory-valid-processors (stype-master theory-stype)) the-empty-set)
  (set (theory-domain-range-handler (stype-master theory-stype)) '#f)
  (set (theory-equivalence-relations (stype-master theory-stype)) '())
  (set (theory-table-hash (stype-master theory-stype)) (uncomputed))
  (set (theory-history (stype-master theory-stype)) nil))


(define (FIND-STRUCTURAL-SUB-THEORIES theory)
  (set-union (theory-component-theories theory)
	     (collect-set find-structural-sub-theories
			  (theory-component-theories theory))))

(define (STRUCTURAL-SUB-THEORY? sub-theory theory)
  (or (eq? sub-theory theory)
      (any?
       (lambda (th)
	 (structural-sub-theory? sub-theory th))
       (theory-component-theories theory))))

;;; Previously:
;;;
;;;(define (STRUCTURAL-SUB-THEORY? sub-theory theory)
;;;  (let ((sub-theories (find-structural-sub-theories theory)))
;;;    (and sub-theories
;;;	 (element-of-set? sub-theory sub-theories))))


;;; MISC.

;;; (define (CLEAR-CONTEXTS-&-HANDLERS)
;;;   (clear-contexts)
;;;   (clear-transform-handlers-&-transforms))

(define (CLEAR-EM)
  (reset-quasi-constructors)
  (clear-contexts)
  (clear-processors))

; A value DCL or DISTINCT-CONSTANTS is assumed to be a list of disjoint sets of
; constants

(define (distinct-constant-lists-equivalent? dcl-1 dcl-2)
  (let ((halfway?
	 (lambda (dcl-1 dcl-2)
	   (iterate iter ((dcl-1 dcl-1))
	     (cond ((null? dcl-1) '#t)
		   ((let ((exemplar (caar dcl-1)))
		      (any
		       (lambda (constant-set)
			 (and (memq? exemplar constant-set)
			      constant-set))
		       dcl-2))
		    => (lambda (constant-set)
			 (if (set-equal? (car dcl-1) constant-set)
			     (iter (cdr dcl-1))
			     '#f)))
		   (else '#f))))))
    (and (halfway? dcl-1 dcl-2)
	 (halfway? dcl-2 dcl-1))))

(define (theory-respects-distinct-constants? theory distinct-constants)
  (let ((table (theory-distinct-constant-table theory))) 
    (iterate iter ((distinct-constants distinct-constants))
      (or
       (null? distinct-constants)
       (and
	(let ((exemplar (table-entry table (caar distinct-constants))))
	  (and exemplar
	       (every?
		(lambda (c)
		  (eq? exemplar (table-entry table c)))
		(cdar distinct-constants))))
	(iter (cdr distinct-constants)))))))

;;; GLOBAL THEORY TABLE
;;;
;;; The GLOBAL-THEORY-TABLE has an entry for every theory active in the system.
;;; These are organized by languages and a hash constructed from the set of
;;; axioms.

(define GLOBAL-THEORY-TABLE (make-table 'GLOBAL-THEORY-TABLE))

(define (ENTER-THEORY theory)
  (push
   (two-d-table-entry global-theory-table
		      (theory-language theory)
		      (reduce-map
		       fx+ 0
		       expression-descriptor-hash (theory-axioms theory)))
   theory))

(define (THEORIES-IN-GLOBAL-THEORY-TABLE)
  (let ((accum '()))
    (walk-table
     (lambda (key value)
       (ignore key)
       (walk-table
	(lambda (key value)
	  (ignore key)
	  (set accum (append value accum)))
	value))
     global-theory-table)
    (make-set accum)))
  
;;; (define (THEORY-NAMES-IN-GLOBAL-THEORY-ALIST)
;;;   (iterate loop ((theory-list (theories-in-global-theory-alist))
;;; 		 (names '()))
;;;     (cond ((null? theory-list)
;;; 	   (map (lambda (x) (string-downcase (symbol->string x))) (make-set names)))
;;; 	  ((name (car theory-list))
;;; 	   =>
;;; 	   (lambda (x) (loop (cdr theory-list) (cons x names))))
;;; 	  (else 
;;; 	   (loop (cdr theory-list) names)))))

(define (THEORY-NAMES-IN-GLOBAL-THEORY-TABLE)
  (map (lambda (x) (string-downcase (symbol->string x)))
       (disgorge-names name->theory)))

(define (COUNT-THEORIES)
  (length (set-separate name (theories-in-global-theory-table))))

(define (FIND-THEORY language axioms distinct-constants)
  (let ((candidates
	 (two-d-table-entry
	  global-theory-table
	  language
	  (reduce-map
	   fx+ 0
	   expression-descriptor-hash axioms))))
    (any
     (lambda (th)
       (and (set-equal? (theory-axioms th) axioms)
	    (distinct-constant-lists-equivalent?
	     (theory-distinct-constant-list th)
	     distinct-constants)
	    th))
     candidates)))

; (define (FIND-SUB-THEORIES language axioms)
;   (let* ((sublist (collect-set
; 		   (lambda (pair) 
; 		     (if (sub-language? (car pair) language)
; 			 (cdr pair)
; 			 the-empty-set))
; 		   (cdr *global-theory-alist*))))
;     (set-map
;      cdr
;      (set-separate
;       (lambda (pair) (subset? (car pair) axioms))
;       sublist))))


(define (DELETE-THEORY theory)
  (set
   (two-d-table-entry global-theory-table
		      (theory-language theory)
		      (reduce-map
		       fx+ 0
		       expression-descriptor-hash (theory-axioms theory)))
   (set-difference
    (two-d-table-entry global-theory-table
		       (theory-language theory)
		       (reduce-map
			fx+ 0
			expression-descriptor-hash (theory-axioms theory)))
    theory)))

(define-operation (disgorge-names name->object-fn))

(define NAME->THEORY
  (let ((*theory-name-alist* nil))
    (operation
	(lambda (symbol-form)
	  (cond ((assq symbol-form *theory-name-alist*)
		 => cdr)
		(else nil)))
      ((disgorge-names self)
       (iterate iter ((names '())
		      (l *theory-name-alist*))
	 (if (null? l)
	     (reverse! names)
	     (iter (cons (caar l) names) (cdr l)))))
      ((setter self)
       (lambda (symbol-form new-value)
	 (imps-enforce theory? new-value)
	 (imps-enforce possible-symbol-form? symbol-form)
	 (cond ((eq? symbol-form 'nil) nil)
	       ((assq symbol-form *theory-name-alist*)
		=> (lambda (pair)
		     (format '#t
			     "~%; WARNING: redefining theory named ~S.~%" 
			     symbol-form)
		     (set (cdr pair) new-value)))
	       (else (set *theory-name-alist*
			  (cons (cons symbol-form new-value)
				*theory-name-alist*)))))))))

(define (THEORY-NAME-STRING theory)
  (string-downcase! (symbol->string (theory-name theory))))


;;; GLOBAL SUPER-THEORIES ALIST 
;;; 
;;; Each entry in the *global-super-theories-alist* is a pair consisting 
;;; of a theory and the set of immediate structural super-theories of the 
;;; theory.  (Note: A theory is a not considered a structural super-theory
;;; of itself.)  There is an entry of this type for every active theory.

(lset *GLOBAL-SUPER-THEORIES-ALIST* (list '*GLOBAL-SUPER-THEORIES-ALIST*))

(define (FIND-STRUCTURAL-SUPER-THEORIES theory)
  (let ((immed-super-ths 
	 (cdr (assq theory (cdr *global-super-theories-alist*)))))
    (set-union immed-super-ths
	       (collect-set find-structural-super-theories immed-super-ths))))

(define (STRUCTURAL-SUPER-THEORY? super-theory theory)
  (structural-sub-theory? theory super-theory))

(define (ENTER-SUPER-THEORY theory super-theory)
  (if (not (element-of-set? theory (theory-component-theories super-theory)))
      (imps-error
       "ENTER-SUPER-THEORY: ~S is not an immediate structural super-theory of ~S."
       super-theory theory)
      (let ((rem (cdr *global-super-theories-alist*)))
	(cond ((assq theory rem)
	       => (lambda (pair)
		    (set (cdr pair) 
			 (add-set-element super-theory (cdr pair)))))
	      (else 
	       (set (cdr *global-super-theories-alist*)
		    (cons (cons theory (make-set (list super-theory)))
			  rem)))))))

(define (FIND-SUPER-THEORY-HAVING-CONSTANT theory name)
  (iterate iter ((super-theories (find-structural-super-theories theory)))
    (cond ((empty-set? super-theories)
	   '#f)
	  ((find-constant (theory-language (first-set-element super-theories))
			  name)
	   (first-set-element super-theories))
	  (else
	   (iter (rest-of-set super-theories))))))

(define (FIND-SUPER-THEORY-HAVING-SORT theory name)
  (iterate iter ((super-theories (find-structural-super-theories theory)))
    (cond ((empty-set? super-theories)
	   '#f)
	  ((name->sort (theory-language (first-set-element super-theories))
		       name)
	   (first-set-element super-theories))
	  (else
	   (iter (rest-of-set super-theories))))))


;;;  BUILDING THEORIES
;;;
;;;  DISTINCT-CONSTANTS is a list of lists of constants.  Any two different
;;;  constants occurring in the same list will be recognized to be unequal.  

(define (BUILD-THEORY 
	 component-theories new-language new-axioms distinct-constants . symbol-form)

  ;; check arguments

  (cond ((and (not (null? symbol-form))
	      (or (not (possible-symbol-form? (car symbol-form)))
		  (eq? nil (car symbol-form))))
	 (imps-error "BUILD-THEORY: ~A is an unsuitable name." (car symbol-form)))
	((not (every? theory? component-theories))
	 (imps-error "BUILD-THEORY: ~A is not set of theories." component-theories))
	((not (language? new-language))
	 (imps-error "BUILD-THEORY: ~A is not a language." new-language))
	((not (every-set-element? theorem? new-axioms))
	 (imps-error "BUILD-THEORY: ~A is not a set of theorems." new-axioms))
	((any?
	  (lambda (l)
	    (any? (lambda (c) (not (constant? c))) l))
	  distinct-constants)
	 (imps-error "BUILD-THEORY: ~A contains non-constants." distinct-constants))
	((any
	  (lambda (l)
	    (and (not (is-set? l))
		 l))
	  distinct-constants)
	 =>
	 (lambda (non-set)
	   (imps-warning
	    "BUILD-THEORY: Distinct constant list~&~A~&~A~&~A"
	    non-set
	    "contains a multiple occurrence."
	    "The resulting theory will be INCONSISTENT."))))
	    
  ;; make structure

  (let* ((component-theories (add-set-element the-kernel-theory component-theories))
	 ;; The kernel theory is a component theory of every theory.
	 (language
	  (language-union
	   (add-set-element new-language
			    (set-map theory-language component-theories))))
	 (distinct-constants 
	  (apply append (cons distinct-constants 
			      (map theory-distinct-constant-list component-theories))))
	 (old-axioms (collect-set theory-axioms component-theories))
	 (axioms (set-union old-axioms new-axioms)))
    (if (and symbol-form
	     (compound-language? language)
	     (not (compound-language-name language)))
	(let ((lang-name (concatenate-symbol (car symbol-form) '-language)))
	  (set (compound-language-name language) lang-name)
	  (set (name->language lang-name) language)))
    (cond ((find-theory language (set-map theorem-formula axioms) distinct-constants)
	   =>
	   (lambda (th)
	     (if (and symbol-form
		      (not (eq? (name->theory (car symbol-form)) th)))
		 (set (name->theory (car symbol-form)) th))
	     (if (and symbol-form
		      (not (theory-name th)))
		 (set (theory-name th) (car symbol-form)))
	     th))
	  (else
	   (let ((theory (make-theory))
		 (theorems (collect-set theory-theorems component-theories))

		 ;; The order of (direct, recursive, and sort) definitions
		 ;; and model conservative extensions in the component theories 
		 ;; must be preserved in the theory being built.  Hence 
		 ;; REMOVE-DUPLICATES-BACK-TO-FRONT is used to remove duplicates 
		 ;; in the definition lists without upsetting their order.

		 (definitions 
		   (remove-duplicates-back-to-front
		    eq? 
		    (apply append (map theory-definitions component-theories))))
		 (recursive-definitions
		  (remove-duplicates-back-to-front
		   eq? 
		   (apply append (map theory-recursive-definitions component-theories))))
		 (sort-definitions 
		  (remove-duplicates-back-to-front 
		   eq? 
		   (apply append (map theory-sort-definitions component-theories))))
		 (mc-extensions
		  (remove-duplicates-back-to-front 
		   eq? 
		   (apply append (map theory-mc-extensions component-theories))))

		 ;; The events of the theory's history are ordered in reverse 
		 ;; chronological order.

		 (history 
		  (append-histories (map theory-history component-theories))))
	     (set-theory-name theory (car symbol-form))
	     (set-theory-component-theories theory component-theories)
	     (set (theory-language theory) language)
	     (set (theory-original-language-name theory) (name new-language))
	     (set (theory-axioms theory) axioms)
	     (set (theory-top-level-axioms theory) (set-diff new-axioms old-axioms))
	     (set-walk
	      (lambda (ax)
		(update-theory-usage-alist-for-theorem theory ax))
	      new-axioms)
	     (set (theory-distinct-constant-list theory) distinct-constants)
	     (set (theory-distinct-constant-table theory) (make-table))
	     (let ((table (theory-distinct-constant-table theory)))
	       (walk
		(lambda (distinct-list)
		  (let ((exemplar (car distinct-list)))
		    (walk
		     (lambda (c) (set (table-entry table c) exemplar))
		     distinct-list)))
		distinct-constants))
	     (enter-theory theory)
	     (set (theory-history theory) history)
	     (set (theory-definitions theory) definitions)
	     (set (theory-recursive-definitions theory) recursive-definitions)
	     (set (theory-sort-definitions theory) sort-definitions)
	     (set (theory-mc-extensions theory) mc-extensions)
	     (set-theory-transform-table theory)
	     (set-theory-domain-range-handler theory)
	     ;; The transform handler must be installed before theorems may
	     ;; be installed.
	     (set-theory-theorems theory theorems new-axioms)
	     (set (theory-subsorting-structure theory)
		  (build-theory-subsorting theory))
	     (maybe-register-imps-obarray-entry (name theory) 'theory)
	     theory)))))

(define (SET-THEORY-NAME theory symbol-form)
  (set (theory-name theory) symbol-form)
  (if symbol-form 
      (set (name->theory symbol-form) theory)))

(define (SET-THEORY-COMPONENT-THEORIES theory component-theories)
  (set (theory-component-theories theory) component-theories)
  (set-walk
   (lambda (component-theory)
     (enter-super-theory component-theory theory))
   (theory-component-theories theory)))

(define (UPDATE-THEORY-USAGE-ALIST-FOR-THEOREM theory theorem)
  (let ((pair (assq theory (theorem-theory-usage-alist theorem))))
    (set (theorem-theory-usage-alist theorem)
	 (if pair
	     (cons (cons theory (set-union (theorem-usage-list theorem)
					   (cdr pair)))
		   (delq pair (theorem-theory-usage-alist theorem)))
	     (cons (cons theory (theorem-usage-list theorem))
		   (theorem-theory-usage-alist theorem))))))

(define (SET-THEORY-TRANSFORM-TABLE theory)
  (set (theory-transform-table theory)
       (join-theory-transform-tables
	(map theory-transform-table (theory-component-theories theory)))))

(define (SET-THEORY-DOMAIN-RANGE-HANDLER theory)
  (set (theory-domain-range-handler theory)
       (join-d-r-handler-list
	theory
	(map theory-domain-range-handler (theory-component-theories theory))))
  ;; this d-r-handler is as yet nameless:
  (set (d-r-handler-name (theory-domain-range-handler theory))
       (if (theory-name theory)
	   (concatenate-symbol (theory-name theory) '-default-d-r-handler)
	   'default-d-r-handler))
  (return))

(define (SET-THEORY-THEOREMS theory theorems new-axioms)
  (set (theory-theorem-hash-table theory) (make-table))
  (bind (((omit-theorem-usage-hooks) '#t))
    ;; these theorems (from component theories) have already been added to the
    ;; transform handler or domain range handler  
    (set-walk
     (lambda (thm)		
       (theory-add-theorem-without-event theory thm '#f))
     theorems))
  (set-walk				; Add new axioms as theorems
   (lambda (ax)		
     (theory-add-theorem theory ax (theorem-name ax)))
   new-axioms))

(define (EXTEND-THEORY theory new-language new-axioms . symbol-form)
  (apply
    build-theory
    (list theory)
    new-language
    new-axioms
    (theory-distinct-constant-list theory)
    symbol-form))

(define (THEORY-UNION theories . symbol-form)
  (if (null? (cdr theories))
      (car theories)
      (apply
       build-theory
       theories
       the-null-language
       the-empty-set
       (apply append (map theory-distinct-constant-list theories))
       symbol-form)))

(define (LANGUAGE->THEORY language)
  (or (find-theory language the-kernel-theory-axioms '())
      (build-theory '() language '() '())))


;;; OPERATIONS ON THEORIES

(define (SUB-THEORY? sub-theory theory)
  (or (structural-sub-theory? sub-theory theory)
      (and (sub-language? (theory-language sub-theory)
			  (theory-language theory))
	   (subset? (theory-axioms sub-theory)
		    (theory-theorems theory))
	   (theory-respects-distinct-constants?
	    theory
	    (theory-distinct-constant-list sub-theory)))))

(define SUBTHEORY? SUB-THEORY?)

(define (EXTEND-THEORY-LANGUAGE theory constants resolver . symbol-form)
  (let* ((new-name 
	  (if symbol-form
	      (car symbol-form)
	      (retrieve-unused-name 
	       name->language 
	       (theory-name theory) 
	       '-extended-language)))
	 (new (extend-language (theory-language theory) constants resolver new-name)))
    (set (language-default-sortings new)
	 (language-default-sortings (theory-language theory)))
    ;; (delete-theory theory)
    (set (theory-language theory) new)
    (enter-theory theory)
    new))

(define (ADD-COMPONENT-THEORY comp-theory theory)
  (or (sub-theory? comp-theory theory)
      (imps-error 
       "ADD-COMPONENT-THEORY: ~A ~A ~A."
       comp-theory "is not a sub-theory of" theory))
  (set (theory-component-theories theory)
       (add-set-element comp-theory (theory-component-theories theory)))
  (enter-super-theory comp-theory theory)
  theory)

(define (THEORY-TOP-LEVEL-THEOREMS theory)
  (enforce theory? theory)
  (let ((lower-level-theorems 
	 (big-u (set-map theory-theorems (theory-component-theories theory)))))
    (set-diff (theory-theorems theory) lower-level-theorems)))

(define (THEORY-AXIOMS-AND-DEFINITION-AXIOMS theory)
  (set-union
   (big-u (list
	   (theory-axioms theory)
	   (map sort-definition-axiom (theory-sort-definitions theory))
	   (map definition-axiom (theory-definitions theory))
	   (map recursive-definition-minimality-axiom 
		(theory-recursive-definitions theory))))
   (big-u (map recursive-definition-equation-axioms 
	       (theory-recursive-definitions theory)))))
	 
(define (THEORY-passive-THEOREM? theory formula)
  (catch found
    (walk-possible-matching-theorems
     (lambda (theorem)
       (if (eq? theorem formula)
	   (found '#t)))
     theory
     formula)
    '#f))

(define (THEORY-THEOREM? theory formula-or-theorem)
  (or (trivial-theorem? formula-or-theorem)
      (cond
       ((expression? formula-or-theorem)
	(let ((expr (universal-matrix formula-or-theorem '())))
	  (catch found
		 (walk-possible-matching-theorems
		  (lambda (theorem)
		    (let* ((pattern (universal-matrix (theorem-formula theorem) '()))
			   (subst (match expr pattern)))
		      (if (and
			   (succeed? subst)
			   (every?
			    (lambda (component)
			      (let ((from (target component))
				    (to (replacement component)))
				(context-immediately-entails-defined-in-sort?
				 (theory-null-context theory)
				 to
				 (expression-sorting from))))
			    subst))
			  (found '#t))))
		  theory
		  expr)
		 '#f)))
       ((theorem? formula-or-theorem)
	(theory-theorem? theory (theorem-formula formula-or-theorem)))
       (else '#f))))
      
     

(define (THEORY-GET-THEOREM-FROM-FORMULA theory formula)
  (let ((expr (universal-matrix formula '())))
    (catch found
      (walk-possible-matching-theorems
       (lambda (theorem)
	 (let* ((pattern (universal-matrix (theorem-formula theorem) '()))
		(subst (match expr pattern)))
	   (if (and (succeed? subst)
		    (every? (lambda (component)
			      (let ((from (target component))
				    (to (replacement component)))
				(and (formal-symbol? to)
				     (eq? (expression-sorting from)
					  (expression-sorting to)))))
			    subst))
	       (found theorem))))
       theory
       expr)
      '#f)))



(define (THEORY-SORTS-RESOLVED theory)
  (sorts-resolved (theory-language theory)))

(define (THEORY-PRIMITIVE-SORTS theory)
  (set-diff (theory-sorts-resolved theory)
	    (theory-defined-sorts theory)))

(define (THEORY-RESOLVED-SORT? theory sort)
  (element-of-set? sort (sorts-resolved (theory-language theory))))

(define (THEORY-PRIMITIVE-SORT? theory sort)
  (element-of-set? sort (theory-primitive-sorts theory)))

(define (THEORY-IND-SORTS-RESOLVED theory)
  (let ((sorts (sorts-resolved (theory-language theory))))
    (delete-set-element prop sorts)))

(define (THEORY-CONSTANTS theory)
  (language-constants (theory-language theory)))

(define (THEORY-PRIMITIVE-CONSTANTS theory)
  (set-diff (theory-constants theory)
	    (theory-defined-constants theory)))

(define (THEORY-CONSTANT? theory constant)
  (element-of-set? constant (language-constants (theory-language theory))))

(define (THEORY-PRIMITIVE-CONSTANT? theory constant)
  (element-of-set? constant (theory-primitive-constants theory)))

(define (THEORY-PRINT-NAMES theory kind port)
  (print
   (xcase kind
     ((definitions) (map definition-constant (theory-definitions theory)))
     ((theorems) (map name (theory-theorems theory))))
   port))


;;; ADDING THEOREMS TO THEORIES

(define omit-theorem-usage-hooks
  (make-simple-switch 'omit-theorem-usage-hooks boolean? '#f))

(define (THEORY-ADD-THEOREM theory theorem the-name . usage-list)
  (theory-add-theorem-and-event theory theorem the-name '#t usage-list))

(define (THEORY-ADD-THEOREM-WITHOUT-EVENT theory theorem the-name . usage-list)
  (theory-add-theorem-and-event theory theorem the-name '#f usage-list))

(define (THEORY-ADD-THEOREM-AND-EVENT theory theorem the-name add-event? usage-list)

  ;; Build theorem

  (let ((old (table-entry *theorem-info-table* theorem)))
    (if (not old)
	(build-theorem theory theorem the-name usage-list)
	(block 
	  (or (eq? (theorem-name theorem) the-name)
	      (and (set (table-entry *name-theorem-table* the-name) theorem)
		   (set (theorem-name theorem) the-name)))
	  (set (theorem-usage-list theorem) 
	       (set-union (theorem-usage-list theorem) usage-list))
	  (or (null? the-name)
	      (set (theorem-usage-list theorem)
		   (add-set-element 'elementary-macete
				    (theorem-usage-list theorem))))
	      
	  )))
  
;;;    (cond ((not old)
;;;           (build-theorem theory theorem the-name usage-list))
;;;          ((not (eq? (theorem-name theorem)
;;;                     the-name))
;;;           (set (table-entry *name-theorem-table* the-name) theorem)
;;;           (set (theorem-name theorem) the-name))
;;;           (else nil))

  (let ((formula (theorem-formula theorem)))

    ;; Checks

    (or (closed? formula)
	(imps-error "THEORY-ADD-THEOREM-AND-EVENT: The theorem formula ~S is not closed." 
		    formula))
    (or (contains-expression? (theory-language theory) formula)
	(imps-error "THEORY-ADD-THEOREM-AND-EVENT: ~S ~A ~S"
		    theorem
		    "is not in the language of"
		    theory))

    ;; Update home theory of THEOREM

    (if (or (null? (theorem-home-theory theorem))
	    (not (theory-literal-theorem? (theorem-home-theory theorem)	theorem)))
	(set (theorem-home-theory theorem) theory))

    (update-theory-usage-alist-for-theorem theory theorem)
    ;;
    ;; Done in build-theorem.
    ;;     (maybe-register-imps-obarray-entry the-name 'theorem)
    ;; 
    ;; Add THEOREM to THEORY and all super-theories of THEORY
    
    (let ((theories (add-set-element
		     theory
		     (find-structural-super-theories theory))))
      (set-walk
       (lambda (theory-x)
	 (theory-add-theorem-aux theory-x theorem add-event?))
       theories)
      theorem)))

(define (THEORY-ADD-THEOREM-AUX theory theorem add-event?)
  (let ((usage-list (theorem-usage-list theorem)))
    (theory-install-theorem theory theorem)		; install theorem
    (if add-event?
	(theory-push-theorem-event theory theorem))     ; push event
    (or (omit-theorem-usage-hooks)
	(theorem-usage-hooks theory theorem usage-list))
    theorem))

(define (THEOREM-ADD-USAGE theory theorem new-usage)
  (and (theory-theorem? theory theorem)
       (block
	 (theorem-usage-hooks theory theorem (list new-usage))
	 (push (theorem-usage-list theorem) new-usage)
	 new-usage)))

(define (THEOREM-USAGE-HOOKS theory theorem usage-list)
  (walk
   (lambda (usage)
     (let ((hook (cdr (assq usage theorem-usage-hook-alist))))
       (and hook 
	    (hook theory theorem))))
   usage-list))

(define (THEORY-INSTALL-SIMPLIFY-LOGICALLY-FIRST theory theorem)
  (ignore theory)
  (and (memq 'rewrite (theorem-usage-list theorem))
       (rewrite-usage-simplog1st theorem))
  (and (memq 'transportable-rewrite (theorem-usage-list theorem))
       (transportable-rewrite-usage-simplog1st theorem)))

(lset THEORY-INSTALL-TRIGGER (lambda (theory theorem) (ignore theory theorem) nil))

(let ((pseudo-recursive-definitions the-empty-set))

  (define (BUILD-PSEUDO-RECURSIVE-DEFINITION theory theorem)
    (ignore theory)
    (let ((formula (theorem-formula theorem)))
      (set pseudo-recursive-definitions
	   (add-set-element
	    (retrieve-macete-from-formula formula)
	    pseudo-recursive-definitions))
      (let ((transportable-macete (retrieve-transportable-macete-from-formula formula)))
	(if transportable-macete
	    (set pseudo-recursive-definitions
		 (add-set-element
		  (retrieve-macete-from-formula formula)
		  pseudo-recursive-definitions))))

      (set (global-induction-step-hook)
	   (let ((macete
		  ((constructor-name->macete-constructor 'series)
		   pseudo-recursive-definitions)))
	     (set (macete-name macete) 'recursive-unfolding)
	     macete)))
    theorem))


;; Order matters in the following alist

(define THEOREM-USAGE-HOOK-ALIST
  (list (cons 'elementary-macete theory-install-elementary-macete)
	(cons 'transportable-macete theory-install-transportable-macete)
	(cons 'rewrite theory-install-rewrite-rule)
	(cons 'transportable-rewrite theory-install-transportable-rewrite-rule)
	(cons 'd-r-value theory-install-value-condition)
	(cons 'd-r-convergence theory-install-convergence-condition)
	(cons 'simplify-logically-first theory-install-simplify-logically-first)
;;;	(cons 'processor theory-install-processor-condition)
;;;	(cons 'recursive-unfolding build-pseudo-recursive-definition)
	(cons 'trigger theory-install-trigger)))

(define (GET-THEOREM-FORMULA the-name)
  (let ((thm (name->theorem the-name)))
    (if (theorem? thm)
	(theorem-formula thm)
	'#f)))

;;;(define (THEORY-GET-THEOREM theory the-name) ; This is obsolete!
;;;  (name->theorem the-name))
;;;
;;;(define (THEORY-GET-THEOREM-FORMULA theory the-name) ; This is obsolete!
;;;  (let ((thm (name->theorem the-name)))
;;;    (if (theorem? thm)
;;;	(theorem-formula thm)
;;;	'#f)))


(define (THEORY-ENTAILS-IMMEDIATELY? theory formula)
  (let ((seq (theory-assertion->sequent theory formula)))
    (theory-entails-sequent-immediately? theory seq)))

(define (THEORY-ENTAILS-SEQUENT-IMMEDIATELY? theory sequent)
  (or (sub-theory? (context-theory (sequent-context sequent)) theory)
      (imps-error
       "THEORY-ENTAILS-SEQUENT-IMMEDIATELY?: ~S not included within ~S."
       sequent theory))
  (context-syllogistically-entails?
   (sequent-context sequent)
   (sequent-assertion sequent)))

;;; The following is unworkable.  

;;; (any?
;;;    (lambda (theorem)
;;;	 (theorem-instance? theorem sequent 1))		;not much persistence
;;;    (theory-theorems theory))


;;; THEORY FROM DEFINITION

;;; Routines for reading theories from forms like:
;;; (peano-arithmetic
;;;  (component-theories ...)
;;;  (language arithmetic)		Really union of this and langs of component-theories
;;;  (distinct-constants (hem shem japheth)
;;;			 (mishrak amtrak abednigo))
;;;  (axioms
;;;   theorem1				Theorem has the form (name formula usage-list)
;;;   theorem2...))
;;;   

(define (THEORY-FROM-DEFINITION form)
  (receive (theory-name component-theories new-language new-axioms distinct-constants)
    (destructure-theory-definition form)
    (build-theory component-theories new-language new-axioms distinct-constants theory-name)))

(define (DESTRUCTURE-THEORY-DEFINITION form)
  (let ((theory-name (imps-enforce symbol? (car form)))
	(form (cdr form)))
    (let ((component-theories 
	   (destructure-theory-component-theories
	    (assq-val 'component-theories form)))
	  (language
	   (cond ((assq-val 'language form)
		  =>
		  (lambda (l)
		    (if (language? (name->language (car l)))
			(name->language (car l))
			(theory-language (enforce theory? (name->theory (car l))))))) 
		 (else the-null-language))))
      (let ((full-language
	     (language-union
	      (add-set-element language (set-map theory-language component-theories)))))

      (return
       theory-name
       component-theories
       language
       (destructure-theory-axioms
	(assq-val 'axioms form)
	full-language)
       (destructure-distinct-constants
	(assq-val 'distinct-constants form)
	full-language))))))

(define (DESTRUCTURE-THEORY-COMPONENT-THEORIES clause)
  (iterate iter ((names clause)
		 (theories '()))
    (if (null? names)
	(reverse! theories)
	(let ((th (name->theory (car names))))
	  (if (theory? th)
	      (iter (cdr names)
		    (cons th theories))
	      (imps-error "DESTRUCTURE-THEORY-COMPONENT-THEORIES: Bogus theory name ~S"
			  (car names)))))))

(define (DESTRUCTURE-DISTINCT-CONSTANTS clause language)
  (map
   (lambda (l)
     (map
      (lambda (cn)
	(imps-enforce constant? (find-constant language cn)))
      l))
   clause))
     
(define (DESTRUCTURE-THEORY-AXIOMS clause language)
  (destructure-theorems clause language))

(define (DESTRUCTURE-THEOREMS clause language)
  (map
   (lambda (sexp)
     (destructure (((the-name formula usage) sexp))
       (build-theorem nil
		      (qr formula language)
		      the-name 
		      usage)))
   clause))

(lset *DESTRUCTURE-THEORY-READ-PROC*
      (lambda (language input)
	((imps-read-proc) input language)))

(lset *DESTRUCTURE-THEORY-PRINT-PROC*			;inverse to sexp->expression in
      imps-expression->string-procedure)		;appropriate language.  


;;; THEORY THEOREMS FROM DEFINITION

;;; Routines for reading theory theorems from forms like:
;;; (peano-arithmetic
;;;  (theorems
;;;   theorem1				Theorem has the form (name formula usage-list)
;;;   theorem2...))
;;;   

(define (THEORY-THEOREMS-FROM-DEFINITION form)
  (receive (theory-name new-theorems)
    (destructure-theory-theorems-definition form)
    (set-walk
     (lambda (theorem)
       (theory-add-theorem (name->theory theory-name) theorem '#f))
     new-theorems)))

(define (DESTRUCTURE-THEORY-THEOREMS-DEFINITION form)
  (let ((theory-name (imps-enforce symbol? (car form)))
	(form (cdr form)))
    (return
     theory-name
     (destructure-theorems (assq-val 'theorems form) (theory-language theory)))))

;;; (define (THEORY-INSTALL-SYMBOLIC-DEFINITIONS theory def-specs)
;;;   (iterate iter ((language (theory-language theory))
;;;		 (def-specs def-specs))
;;;	(if (null? def-specs)
;;;	'#t
;;;	(destructure (((name sorting-info defining-form) (car def-specs)))
;;;	  (let ((kind (cond ((symbol? sorting-info) term?)
;;;			    ((eq? prop (tree-last-atom sorting-info)) predicate?)
;;;			    (else function?))))
;;;	    (theory-build-definition
;;;	     theory
;;;	     name
;;;	     (*destructure-theory-read-proc* language defining-form)
;;;	     kind))
;;;	  (iter (theory-language theory)
;;;		(cdr def-specs))))))



;        (axioms
;	 (iterate iter ((theories component-theories)
;			(axioms new-axioms))
;	   (if (null? theories)
;	       axioms
;	       (iter (cdr theories)
;		     (set-union (theory-axioms (car theories))
;				axioms)))))
;	(theorems
;	 (iterate iter ((theories component-theories)
;			(theorems the-empty-set))
;	   (if (null? theories)
;	       theorems
;	       (iter (cdr theories)
;		     (set-union (theory-theorems (car theories))
;				theorems)))))

(define (THEORY-VERIFY-AND-ADD-THEOREM theory formula the-name usage-list . file-name)
  (theory-verify-theorem theory formula (car file-name) the-name)
  (or (proper-list? usage-list)
      (imps-error :"THEORY-VERIFY-AND-ADD-THEOREM: Usage must be given as a list."))
  (apply
   theory-add-theorem theory formula the-name usage-list))

(define (THEORY-VERIFY-MODIFY-AND-ADD-THEOREM 
	 theory formula the-name usage-list translation macete home-theory proof-spec . auxiliary?)

  ;; Auxiliary means the theorem is being installed as a variant of an existing theorem, for
  ;; example as the reverse.

  (if (not (car auxiliary?))
      (theory-verify-theorem home-theory formula proof-spec the-name))
  (cond ((and translation macete)
	 (theory-add-maceted-transported-theorem 
	  theory formula the-name usage-list translation macete))
	((and translation (not macete))
	 (theory-add-transported-theorem
	  theory formula the-name usage-list translation))
	((and (not translation) macete)
	 (theory-add-maceted-theorem
	  theory formula the-name usage-list macete))
	((not (eq? theory home-theory))
	 (theory-add-generalized-theorem 
	  theory formula the-name usage-list home-theory))
	(else
	 (apply 
	  theory-add-theorem theory formula the-name usage-list))))

(define (THEORY-VERIFY-MODIFY-AND-ADD-THEOREM-AUX
	 theory-name formula-spec the-name usage-list 
	 translation-name macete-name home-theory-name proof-spec . reverse?)
  (let ((formula 
	 (if (string? formula-spec)
	     (qr formula-spec (theory-language (name->theory home-theory-name)))
	     (theorem-formula (name->theorem formula-spec)))))
    (theory-verify-modify-and-add-theorem
     (name->theory theory-name)
     (if (null? reverse?)
	 formula
	 (reverse-formula formula))
     (if (or (null? reverse?)
	     (not the-name))
	 the-name
	 (concatenate-symbol 'rev% the-name))
     (if (null? reverse?)
	 usage-list
	 (set-difference usage-list '(rewrite transportable-rewrite)))
     (and translation-name (name->translation translation-name))
     (and macete-name (name->macete macete-name))
     (name->theory home-theory-name)
     proof-spec
     (car reverse?))))

(define (THEORY-VERIFY-THEOREM theory formula proof-spec the-name)
  (imps-enforce theory? theory)
  (imps-enforce formula? formula)
  (or (contains-expression? (theory-language theory) formula)
      (imps-error "THEORY-VERIFY-THEOREM: ~S ~A ~S."
		  formula "is not a formula in the of" theory))
  (cond 
   ((proof-log-port)
    (log-imps-proof theory formula proof-spec the-name))


   ((eq? proof-spec 'existing-theorem)
    (or (theory-theorem? theory formula)
	(imps-error "THEORY-VERIFY-THEOREM: ~S ~A ~S."
		    formula "is not known to a theorem of" theory)))

   (else '#t)))

;;;(define (locate-valid-proof theory formula proof-spec)
;;;  (or (theory-theorem? theory formula)
;;;      (any? (lambda (dg)
;;;		    (and (deduction-graph-grounded? dg)
;;;			 (structural-sub-theory? (deduction-graph-theory dg) theory)
;;;			 (alpha-equivalent?
;;;			  formula
;;;			  (sequent-node-assertion (deduction-graph-goal dg)))))
;;;		  *dgs*)
;;;      (unwind-protect
;;;       (let ((uncompressed? (system-please-uncompress proof-spec)))
;;;	 (if uncompressed?
;;;	     (with-open-ports ((p (maybe-open proof-spec '(in))))
;;;	       (iterate loop ()
;;;		 (let ((form (if (eof? p) '() (read p))))
;;;		   (if (null? form)
;;;		       '#f
;;;		       (let ((evaled (eval form imps-implementation-env)))
;;;			 (if (eq? (car form) 'read-sequent-and-start-emacs-deduction) 
;;;			     (alpha-equivalent?
;;;			      (sequent-node-assertion evaled)
;;;			      formula)
;;;			     (loop)))))))
;;;	     '#f))
;;;       (system-please-compress proof-spec))
;;;      (block
;;;	(emacs-eval "(message \"No valid proof. Starting deduction.\"")
;;;	(start-emacs-deduction formula 1)
;;;	nil)))

(define (define-theory-from-definition theory-name language
	  component-theories distinct-constants new-axioms)
  (let* ((theory-name (imps-enforce symbol? theory-name))
	 (component-theories
	  (destructure-theory-component-theories component-theories))
	 (language (if (null? language)
		       the-null-language
		       (imps-enforce language? (name->language language))))
	 (full-language
	  (language-union
	   (add-set-element language (set-map
				      theory-language
				      component-theories))))
	 (distinct-constants
	  (destructure-distinct-constants distinct-constants full-language))
	 (new-axioms
	  (map (lambda (sexp) (sexp&language->theorem sexp full-language))
	       new-axioms)))
    (build-theory component-theories language
		  new-axioms distinct-constants theory-name)))

(define (sexp&language->theorem sexp language)
  (destructure (((the-name formula usage) sexp))
    (build-theorem '()
		   (sexp->expression formula language)
		   the-name 
		   usage)))

(define (define-theorem theorem-name theory-name formula usage-list proof-file)
  (let* ((theory (imps-enforce theory? (name->theory theory-name)))
	 (language (imps-enforce language? (theory-language theory))))
    (theory-verify-and-add-theorem
     theory (sexp->expression formula language)
     theorem-name usage-list proof-file)))

(define (define-definition definition-name theory-name formula usage-list)
  (let* ((theory (imps-enforce theory? (name->theory theory-name)))
	 (language (imps-enforce language? (theory-language theory)))
	 (defining-expr (sexp->expression formula language)))
    (theory-build-definition
     theory
     definition-name
     defining-expr
     (expression-sorting defining-expr)
     usage-list)))

(define (COMPRESS-THEORIES theories)
  (iterate loop ((unchecked-theories theories) (new-theories theories))
    (if (empty-set? unchecked-theories)
	new-theories
	(let ((next-th (car unchecked-theories)))
	  (if (any? 
	       (lambda (th)
		 (and (not (eq? next-th th))
		      (sub-theory? next-th th)))
	       new-theories)
	      (loop (cdr unchecked-theories) 
		    (delete-set-element next-th new-theories))
	      (loop (cdr unchecked-theories) new-theories))))))

(define (SELECT-COMMON-SUB-THEORIES theory-1 theory-2 candidate-subtheories)
  (set-separate 
   (lambda (th) 
     (and (sub-theory? th theory-1)
	  (sub-theory? th theory-2)))
   candidate-subtheories))

;;; The following procedure orders a set of (sort, direct, and recursive)
;;; definitions of a theory from oldest to newest.

(define (THEORY-ORDER-DEFINITIONS theory def-set)
  (iterate loop ((events (event-history-events (theory-history theory)))
		 (def-list `()))
   (if (null? events)
       def-list
       (let ((item (history-event-item (car events))))
	 (if (and (or (sort-definition? item)
		      (definition? item)
		      (recursive-definition? item))
		  (element-of-set? item def-set))
	     (loop (cdr events) (cons item def-list))
	     (loop (cdr events) def-list))))))

(define (theory-specific-axioms theory)
  (set-separate
   (lambda (ax)
     (every?
      (lambda (component-theory)
	(not
	 (theory-theorem? component-theory ax)))
      (theory-component-theories theory)))
   (theory-axioms theory)))
  
(define (theory-specific-theorems theory)
  (let ((components (find-structural-sub-theories theory))
	(thm-table (theory-theorem-hash-table theory))
	(accum '()))
    (walk-table
     (lambda (k v)
       (ignore k)
       (walk-table
	(lambda (k1 thms)
	  (ignore k1)
	  (do ((thms thms (cdr thms)))
	      ((null? thms))
	    (let ((thm (car thms)))
	      (or (any?
		   (lambda (component-theory)
		     (theory-passive-theorem? component-theory thm))
		   components)
		  (push accum thm)))))
	v))
     thm-table)
    (make-set accum)))


(define (theory-specific-theorem-names theory)
  (map theorem-name (theory-specific-theorems theory)))

(define (THEORY-SORT? theory sort)
  (and (sort? sort)
       (contains-sort? (theory-language theory) sort)))

(define (THEORY-EXPRESSION? theory expr)
  (and (expression? expr)
       (contains-expression? (theory-language theory) expr)))

