#############################################################################
##
#A  sheet.g                   	XGAP library                     Frank Celler
##
#H  @(#)$Id: sheet.g,v 1.8 1993/10/28 15:12:12 fceller Exp $
##
#Y  Copyright (C) 1993,  Lehrstuhl D fuer Mathematik,  RWTH, Aachen,  Germany
##
##  This file contains all functions for graphic sheets, the low level window
##  functions are in "window.g".  The menu functions are in "menu.g".
##
#H  $Log: sheet.g,v $
#H  Revision 1.8  1993/10/28  15:12:12  fceller
#H  fixed a bug in 'SavePSFile'
#H
#H  Revision 1.7  1993/10/06  16:19:23  fceller
#H  fixed 'Move' of vertex
#H
#H  Revision 1.6  1993/10/05  12:33:26  fceller
#H  added '.isAlive'
#H
#H  Revision 1.5  93/09/22  10:19:51  fceller
#H  added 'VertexOps.Destroy'
#H  
#H  Revision 1.4  1993/08/18  10:59:49  fceller
#H  removed emacs variables
#H
#H  Revision 1.3  1993/08/13  13:38:39  fceller
#H  added 'VertexOps.MoveDelta'
#H
#H  Revision 1.2  1993/07/22  11:24:32  fceller
#H  split files into three: "window.g", "sheet.g", "menu.g"
#H
#H  Revision 1.1  1993/07/21  12:32:42  fceller
#H  Initial revision
##


#############################################################################
##
#F  Close( <sheet> )  . . . . . . . . . . . . . . . . . . . . .  close window
##
Close := function( obj )
    return obj.operations.Close(obj);
end;


#############################################################################
##
#F  Delete( <sheet>, <obj> )  . . . . . . . . . . . . . . .  undraw an object
##
Delete := function( sheet, obj )
    sheet.operations.Delete( sheet, obj );
end;


#############################################################################
##
#F  SetTitle( <sheet>, <title> )  . . . . . . . . . . . . . . . . add a title
##
SetTitle := function( sheet, text )
    return sheet.operations.SetTitle( sheet, text );
end;


#############################################################################
##

#F  GraphicSheet( <name>, <width>, <height> ) . . . . . . a new graphic sheet
##
GraphicSheetOps := rec( name := "GraphicSheetOps" );

GraphicSheet := function( name, width, height )
    local   w,  txt,  m;

    # open a new graphic sheet and store the identifier
    w := WcOpenWindow( name, width, height );

    # construct a new graphic sheet record
    w.isGraphicSheet := true;
    w.name           := Copy(name);
    w.title          := w.name;
    w.width          := width;
    w.height         := height;
    w.objects        := [];
    w.free           := [];
    w.operations     := GraphicSheetOps;

    # add menu to close GraphicSheet
    m := Menu( w, "GAP",
               [ "save as PS",,
                 "show object ps","show object id","show window id",,
                 "close window"],
          w.operations.GapMenu );

    # add mouse click functions
    w.leftPointerButtonDown  := Ignore;
    w.rightPointerButtonDown := Ignore;
    
    # add close function
    w.close := Ignore;

    # return the graphic sheet <w>
    return w;

end;

GraphicSheetOps.Print := function( sheet )
    Print( "GraphicSheet( \"", sheet.name, "\", ", sheet.width,
           ", ", sheet.height," )" );
end;
    

#############################################################################
##
#F  GraphicSheetOps.Close( <sheet> )  . . . . . . . . . . close graphic sheet
##
GraphicSheetOps.Close := function( sheet )
    sheet.close(sheet);
    WcCloseWindow(sheet.id);
end;


#############################################################################
##
#F  GraphicSheetOps.CreateObject( <S>, <ops> )  . . . . . . create a template
##
GraphicSheetOps.CreateObject := function( S, ops )
    local   obj;

    # create a template
    obj            := rec();
    obj.sheet      := S;
    obj.isAlive    := true;
    obj.operations := ops;
    
    # add object to list of objects stored in <S>    
    if 0 = Length(S.free)  then
        Add( S.objects, obj );
    else
        S.objects[S.free[Length(S.free)]] := obj;
        Unbind(S.free[Length(S.free)]);
    fi;
    
    # and return
    return obj;

end;


#############################################################################
##
#F  GraphicSheetOps.Delete( <sheet>, <obj> )  . . . . delete <obj> in <sheet>
##
GraphicSheetOps.Delete := function( sheet, obj )
    local   pos;
    
    # find position of object
    pos := Position( sheet.objects, obj );
    
    # destroy object
    obj.operations.Destroy(obj);
    
    # and remove it from the list of objects
    Unbind(sheet.objects[pos]);
    Add( sheet.free, pos );

end;


#############################################################################
##
#F  GraphicSheetOps.GapMenu( <sheet>, <menu>, <entry> ) . . gap menu selected
##
GraphicSheetOps.GapMenu := function( sheet, menu, entry )
    local   res,  obj;
    
    # set pointer func and title,  save old func and title
    if entry = "show object id"  then
        if not IsBound( sheet.saveTitle )  then
            sheet.saveTitle := sheet.title;
        fi;
        if not IsBound(sheet.saveLeftPointerButtonDown)  then
            sheet.saveLeftPointerButtonDown := sheet.leftPointerButtonDown;
        fi;
    	sheet.leftPointerButtonDown := sheet.operations.ShowObjectId;
        sheet.operations.SetTitle( sheet, "Select Object" );
        
    # set pointer func and title,  save old func and title
    elif entry = "show object ps"  then
        if not IsBound( sheet.saveTitle )  then
            sheet.saveTitle := sheet.title;
        fi;
        if not IsBound(sheet.saveLeftPointerButtonDown)  then
            sheet.saveLeftPointerButtonDown := sheet.leftPointerButtonDown;
        fi;
    	sheet.leftPointerButtonDown := sheet.operations.ShowObjectPS;
        sheet.operations.SetTitle( sheet, "Select Object" );
        
    # print info line
    elif entry = "show window id"  then
        Print("#I  ",sheet," = ",sheet.id,".",Position(WINDOWS,sheet),"\n");
        
    # close the graphic sheet
    elif entry = "close window"  then
        sheet.operations.Close(sheet);
        
    # print PS file
    elif entry = "save as PS"  then
        if IsBound(sheet.filename)  then
            res := Query( FILENAME_DIALOG, sheet.filename );
        else
            res := Query( FILENAME_DIALOG );
        fi;
        if res = false  then
            return;
        fi;
        sheet.operations.SavePSFile( sheet, res );
        
    # something is wrong
    else
        Print( "#W  unknown menu entry '", entry, "'\n" );
    fi;
end;


#############################################################################
##
#F  GraphicSheetOps.LeftPointerButtonDown( <sheet>, <x>, <y> )  . . . . click
##
GraphicSheetOps.LeftPointerButtonDown := function( sheet, x, y )
    return sheet.leftPointerButtonDown( sheet, x, y );
end;


#############################################################################
##
#F  GraphicSheetOps.Resize( <sheet>, <width>, <height> )  . . .  resize sheet
##
GraphicSheetOps.Resize := function( sheet, width, height )
    WcResizeWindow( sheet.id, width, height );
    sheet.height := height;
    sheet.width  := width;
end;

    
#############################################################################
##
#F  GraphicSheetOps.RightPointerButtonDown( <sheet>, <x>, <y> ) . . . . click
##
GraphicSheetOps.RightPointerButtonDown := function( sheet, x, y )
    return sheet.rightPointerButtonDown( sheet, x, y );
end;


#############################################################################
##
#F  GraphicSheetOps.SavePSFile( <sheet>, <file> ) . . . .  save as PostScript
##
GraphicSheetOps.SavePSFile := function( sheet, file )
    local   a,  b,  obj,  str;

    # set filename and create file
    sheet.filename := file;
    PrintTo( file, "%!PS-Adobe-2.0\n" );

    # collect string in <str>
    str := "";
    
    # landscape or portrait
    if sheet.height <= sheet.width  then
        Append( str, "90 rotate\n" );
        a := QuoInt( sheet.height*1000, 6 );
        b := QuoInt( sheet.width*1000,  8 );
        a := Maximum(a,b);
        Append( str, "100000 " );
        Append( str, String(a) );
        Append( str, " div 100000 " );
        Append( str, String(a) );
        Append( str, " div scale\n" );
        Append( str, "0 " );
        Append( str, String(-sheet.height) );
        Append( str, " translate\n" );
    else
        a := QuoInt( sheet.height*1000, 8 );
        b := QuoInt( sheet.width*1000,  6 );
        a := Maximum(a,b);
        Append( str, "100000 " );
        Append( str, String(a) );
        Append( str, " div 100000 " );
        Append( str, String(a) );
        Append( str, " div scale\n" );
    fi;
    for obj  in sheet.objects  do
        Append( str, obj.operations.PSString(obj) );
    od;
    Append( str, "showpage\n" );
    AppendTo( file, str );

end;


#############################################################################
##
#F  GraphicSheetOps.SetTitle( <sheet>, <text> ) . . . . . . .  set title text
##
GraphicSheetOps.SetTitle := function( sheet, text )
    sheet.title := text;
    WcSetTitle( sheet.id, text );
end;


#############################################################################
##
#F  GraphicSheetOps.ShowObjectId( <sheet>, <x>, <y> ) print object at <x>,<y>
##
GraphicSheetOps.ShowObjectId := function( sheet, x, y )
    local   x,  y,  pos,  elm,  one;

    # try to locate a graphic object at position <x>, <y>
    pos := [ x, y ];
    one := false;
    for elm  in sheet.objects  do
    	if pos in elm  then
            elm.operations.PrintInfo(elm);
            one := true;
    	fi;
    od;
    
    # tell the user if there are no objects at this position
    if not one  then
        Print("#I  there are no objects at position [",x,", ",y,"]\n");
    fi;

    # restore old title and button function
    sheet.operations.SetTitle( sheet, sheet.saveTitle );
    sheet.leftPointerButtonDown := sheet.saveLeftPointerButtonDown;
    Unbind( sheet.saveLeftPointerButtonDown );
    Unbind( sheet.saveTitle );
    
end;


#############################################################################
##
#F  GraphicSheetOps.ShowObjectPS( <sheet>, <x>, <y> ) . . show PostScript def
##
GraphicSheetOps.ShowObjectPS := function( sheet, x, y )
    local   x,  y,  pos,  elm,  one;

    # try to locate a graphic object at position <x>, <y>
    pos := [ x, y ];
    one := false;
    for elm  in sheet.objects  do
    	if pos in elm  then
            elm.operations.PrintInfo(elm);
            Print( elm.operations.PSString(elm) );
            one := true;
    	fi;
    od;
    
    # tell the user if there are no objects at this position
    if not one  then
        Print("#I  there are no objects at position [",x,", ",y,"]\n");
    fi;

    # restore old title and button function
    sheet.operations.SetTitle( sheet, sheet.saveTitle );
    sheet.leftPointerButtonDown := sheet.saveLeftPointerButtonDown;
    Unbind( sheet.saveLeftPointerButtonDown );
    Unbind( sheet.saveTitle );
    
end;


#############################################################################
##

#F  Highlight( <obj> )  . . . . . . . . . . . . . . . . . highlight an object
##
Highlight := function( obj )
    if not obj.isAlive  then Error( "<obj> must be alive" );  fi;
    return obj.operations.Highlight( obj );
end;


#############################################################################
##
#F  Move( <obj>, <x>, <y> ) . . . . . . . . . . .  move <obj> to new position
##
Move := function( obj, x, y )
    if not obj.isAlive  then Error( "<obj> must be alive" );  fi;
    return obj.operations.Move( obj, x, y );
end;


#############################################################################
##
#F  Relabel( <obj>, <text> )  . . . . . . . . . . . . . . . relabel an object
##
Relabel := function( obj, text )
    if not obj.isAlive  then Error( "<obj> must be alive" );  fi;
    return obj.operations.Relabel( obj, text );
end;


#############################################################################
##
#F  Reshape( <obj>, ... ) . . . . . . . . . . . . . . . . . reshape an object
##
Reshape := function( arg )
    if not arg[1].isAlive  then Error( "<obj> must be alive" );  fi;
    return arg[1].operations.Reshape(arg);
end;


#############################################################################
##
#F  Unhighlight( <obj> )  . . . . . . . . . . . . . . . .  lowlight an object
##
Unhighlight := function( obj )
    if not obj.isAlive  then Error( "<obj> must be alive" );  fi;
    return obj.operations.Unhighlight( obj );
end;


#############################################################################
##

#F  Line( <sheet>, <x1>, <y1>, <x2>, <y2> ) . . . . .  draw a line in a sheet
##
LineOps := rec( name := "LineOps" );

Line := function( sheet, x1, y1, x2, y2 )
    local   line;

    # create a line object in <sheet>
    line        := sheet.operations.CreateObject( sheet, LineOps );
    line.isLine := true;
    line.x1     := x1;
    line.x2     := x2;
    line.y1     := y1;
    line.y2     := y2;
    line.width  := 1;

    # draw the line and get the identifier
    WcSetLineWidth( sheet.id, line.width );
    line.id := WcDrawLine( sheet.id, x1, y1, x2, y2 );

    # and return
    return line;

end;

LineOps.PrintInfo := function( line )
    Print( "#I  Line( W, ", line.x1, ", ", line.y1, ", ",
           line.x2, ", ", line.y2, " ) = ", line.id, ".",
    	   Position(line.sheet.objects,line), "\n" );
end;

LineOps.Print := function( line )
    Print( "Line( ", line.sheet, ", ", line.x1, ", ", line.y1, ", ",
           line.x2, ", ", line.y2, " )" );
end;


#############################################################################
##
#F  LineOps.Destroy( <line> ) . . . . . . . . . . . . . . . .  destroy <line>
##
LineOps.Destroy := function( line )
    WcDestroy( line.sheet.id, line.id );
    line.isAlive := false;
end;


#############################################################################
##
#F  LineOps.MoveDelta( <line>, <dx>, <dy> ) . . . . . . . . . . .  delta move
##
LineOps.MoveDelta := function( line, dx, dy )
    
    # make sure that we really have to move
    if dx = 0 and dy = 0  then return;  fi;

    # use 'Reshape'
    line.operations.Reshape([ line, line.x1+dx, line.y1+dy,
                                    line.x2+dx, line.y2+dy ]);
end;


#############################################################################
##
#F  LineOps.PSString( <line> )	. . . . . . . . . . . . . . PostScript string
##
LineOps.PSString := function( line )
    return Concatenation(
        "newpath\n",
        String(line.x1), " ", String(line.sheet.height-line.y1), " moveto\n",
        String(line.x2), " ", String(line.sheet.height-line.y2), " lineto\n",
        String(line.width), " setlinewidth\n",
        "stroke\n" );
end;


#############################################################################
##
#F  LineOps.Reshape( <line>, <x1>, <y1>, <x2>, <y2> ) . . . . . change <line>
##
LineOps.Reshape := function( l )

    # remove old line
    WcDestroy( l[1].sheet.id, l[1].id );

    # and create a new one
    WcSetLineWidth( l[1].sheet.id, l[1].width );
    l[1].id := WcDrawLine( l[1].sheet.id, l[2], l[3], l[4], l[5] );
    l[1].isAlive := true;
    
    # update line coordinates
    l[1].x1 := l[2];
    l[1].x2 := l[4];
    l[1].y1 := l[3];
    l[1].y2 := l[5];
    
end;


#############################################################################
##
#F  LineOps.SetWidth( <line>, <width> )	. . . . . . . . . . . .  change width
##
LineOps.SetWidth := function( line, width )
    if line.width <> width  then
        WcDestroy( line.sheet.id, line.id );
        WcSetLineWidth( line.sheet.id, width );
        line.id := WcDrawLine( line.sheet.id,
                               line.x1, line.y1, line.x2, line.y2 );
        line.isAlive := true;
        line.width := width;
    fi;
end;


#############################################################################
##
#F  LineOps.in( <pos>, <line> )	. . . . . . . . . . . . . . . <pos> on <line>
##
LineOps.\in := function( pos, line )
    local   x,  y,  ax,  ay,  ix,  iy;
    
    x  := pos[1];
    y  := pos[2];
    ax := Maximum( line.x1, line.x2 );
    ix := Minimum( line.x1, line.x2 );
    ay := Maximum( line.y1, line.y2 );
    iy := Minimum( line.y1, line.y2 );
    if 5 < x-ax or 5 < ix-x  then
    	return false;
    elif 5 < y-ay or 5 < iy-y  then
    	return false;
    elif ax = ix or ay = iy  then
    	return true;
    else
    	return AbsInt((x-line.x1)*(line.y2-line.y1)
               /(line.x2-line.x1)-(y-line.y1)) < 5;
    fi;
    
end;


#############################################################################
##

#F  Box( <sheet>, <x1>, <y1>, <x2>, <y2> )  . . . . . . draw a box in a sheet
##
BoxOps := rec( name := "BoxOps" );

Box := function( sheet, x1, y1, x2, y2 )
    local   box;

    # create a box object in <sheet>
    box       := sheet.operations.CreateObject( sheet, BoxOps );
    box.isBox := true;
    box.x1    := x1;
    box.x2    := x2;
    box.y1    := y1;
    box.y2    := y2;

    # draw the box and get the identifier
    box.id := WcDrawBox( sheet.id, x1, y1, x2, y2 );

    # and return
    return box;

end;

BoxOps.PrintInfo := function( box )
    Print( "#I  Box( W, ", box.x1, ", ", box.y1, ", ",
           box.x2, ", ", box.y2, " ) = ", box.id, ".",
    	   Position(box.sheet.objects,box), "\n" );
end;

BoxOps.Print := function( box )
    Print( "Box( ", box.sheet, ", ", box.x1, ", ", box.y1, ", ",
           box.x2, ", ", box.y2, " )" );
end;


#############################################################################
##
#F  BoxOps.Destroy( <box> ) . . . . . . . . . . . . . . . . . .  detroy <box>
##
BoxOps.Destroy := function( box )
    WcDestroy( box.sheet.id, box.id );
    box.isAlive := false;
end;


#############################################################################
##
#F  BoxOps.MoveDelta( <box>, <dx>, <dy> ) . . . . . . . . . . . .  delta move
##
BoxOps.MoveDelta := function( box, dx, dy )
    
    # make sure that we really have to move
    if dx = 0 and dy = 0  then return;  fi;

    # use 'Reshape'
    box.operations.Reshape([box,box.x1+dx,box.y1+dy,box.x2+dx,box.y2+dy]);
    
end;


#############################################################################
##
#F  BoxOps.PSString( <box> )  . . . . . . . . . . . . . . . PostScript string
##
BoxOps.PSString := function( box )
    return Concatenation(
        "newpath\n",
        String(box.x1), " ", String(box.sheet.height-box.y1), " moveto\n",
        String(box.x1), " ", String(box.sheet.height-box.y2), " lineto\n",
        String(box.x2), " ", String(box.sheet.height-box.y2), " lineto\n",
        String(box.x2), " ", String(box.sheet.height-box.y1), " lineto\n",
        String(box.x1), " ", String(box.sheet.height-box.y1), " lineto\n",
        "closepath\nfill\n" );
end;


#############################################################################
##
#F  BoxOps.Reshape( <line>, <x1>, <y1>, <x2>, <y2> )  . . . . .  change <box>
##
BoxOps.Reshape := function( b )

    # remove old box
    WcDestroy( b[1].sheet.id, b[1].id );
    
    # and create new one
    b[1].id := WcDrawBox( b[1].sheet.id, b[2], b[3], b[4], b[5] );
    b[1].isAlive := true;
    
    # update box coordinates
    b[1].x1 := b[2];
    b[1].x2 := b[4];
    b[1].y1 := b[3];
    b[1].y2 := b[5];
    
end;


#############################################################################
##
#F  BoxOps.in( <pos>, <box> ) . . . . . . . . . . . . . . . .  <pos> in <box>
##
BoxOps.\in := function( pos, box )
    local   x,  y,  ax,  ay,  ix,  iy;

    x  := pos[1];
    y  := pos[2];
    ax := Maximum( box.x1, box.x2 );
    ix := Minimum( box.x1, box.x2 );
    ay := Maximum( box.y1, box.y2 );
    iy := Minimum( box.y1, box.y2 );
    return ix-5 < x and x < ax+5 and iy-5 < y and y < ay+5;
    
end;


#############################################################################
##

#F  Circle( <sheet>, <x>, <y>, <r> )  . . . . . . .  draw a circle in a sheet
##
CircleOps := rec( name := "CircleOps" );

Circle := function( sheet, x, y, r )
    local   circle;

    # create a circle record
    circle          := sheet.operations.CreateObject( sheet, CircleOps );
    circle.isCircle := true;
    circle.x        := x;
    circle.y        := y;
    circle.r        := r;
    circle.width    := 1;

    # draw the circle and get the identifier
    WcSetLineWidth( sheet.id, circle.width );
    circle.id := WcDrawCircle( sheet.id, x, y, r );

    # and return
    return circle;

end;

CircleOps.PrintInfo := function( circle )
    Print( "#I  Circle( W, ", circle.x, ", ", circle.y, ", ",
           circle.r, " ) = ", circle.id, ".",
    	   Position(circle.sheet.objects,circle), "\n" );
end;

CircleOps.Print := function( circle )
    Print( "Circle( ", circle.sheet, ", ", circle.x, ", ", circle.y, 
           ", ", circle.r, " )" );
end;


#############################################################################
##
#F  CircleOps.Destroy( <circle> ) . . . . . . . . . . . . .  destroy <circle>
##
CircleOps.Destroy := function( circle )
    WcDestroy( circle.sheet.id, circle.id );
    circle.isAlive := false;
end;


#############################################################################
##
#F  CircleOps.MoveDelta( <circle>, <dx>, <dy> ) . . . . . . . . .  delta move
##
CircleOps.MoveDelta := function( circle, dx, dy )

    # make sure that we really have to move
    if dx = 0 and dy = 0  then return;  fi;

    # remove old circle
    WcDestroy( circle.sheet.id, circle.id );

    # update coordinates
    circle.x := circle.x+dx;
    circle.y := circle.y+dy;

    # and draw a new circle
    WcSetLineWidth( circle.sheet.id, circle.width );
    circle.id := WcDrawCircle(circle.sheet.id, circle.x, circle.y, circle.r);
    circle.isAlive := true;

end;


#############################################################################
##
#F  CircleOps.PSString( <circle> )  . . . . . . . . . . . . PostScript string
##
CircleOps.PSString := function( circle )
    return Concatenation(
        "newpath\n",
        String(circle.x), " ", String(circle.sheet.height-circle.y), " ",
        String(circle.r), " 0 360 arc\n",
        String(circle.width), " setlinewidth\n",
        "stroke\n" );
end;


#############################################################################
##
#F  CircleOps.Reshape( <circle>, <r> )  . . . . . . . . . . . . change radius
##
CircleOps.Reshape := function( c )
    
    # remove old circle
    WcDestroy( c[1].sheet.id, c[1].id );
    
    # and create a new one
    WcSetLineWidth( c[1].sheet.id, c[1].width );
    c[1].id := WcDrawCircle( c[1].sheet.id, c[1].x, c[1].y, c[2] );
    c[1].isAlive := true;

    # and update radius
    c[1].r := c[2];

end;


#############################################################################
##
#F  CircleOps.SetWidth( <circle>, <with> )  . . . . . . . . . .  change width
##
CircleOps.SetWidth := function( circle, width )
    
    # remove old circle
    WcDestroy( circle.sheet.id, circle.id );
    
    # change line width
    circle.width := width;

    # and draw a new circle
    WcSetLineWidth( circle.sheet.id, circle.width );
    circle.id := WcDrawCircle(circle.sheet.id, circle.x, circle.y, circle.r);
    circle.isAlive := true;

end;


#############################################################################
##
#F  CircleOps.in( <pos>, <circle> ) . . . . . . . . . . . . <pos> in <circle>
##
CircleOps.\in := function( pos, circle )
    return (pos[1]-circle.x)^2+(pos[2]-circle.y)^2 < (circle.r+3)^2;
end;


#############################################################################
##

#F  Disc( <sheet>, <x>, <y>, <r> )  . . . . . . . . .  draw a disc in a sheet
##
DiscOps := rec( name := "DiscOps" );

Disc := function( sheet, x, y, r )
    local   disc;

    # create a disc record
    disc        := sheet.operations.CreateObject( sheet, DiscOps );
    disc.isDisc := true;
    disc.x      := x;
    disc.y      := y;
    disc.r      := r;

    # draw the disc and get the identifier
    disc.id := WcDrawDisc( sheet.id, x, y, r );

    # and return
    return disc;

end;

DiscOps.PrintInfo := function( disc )
    Print( "#I  Disc( W, ", disc.x, ", ", disc.y, ", ",
           disc.r, " ) = ", disc.id, ".",
    	   Position(disc.sheet.objects,disc), "\n" );
end;

DiscOps.Print := function( disc )
    Print( "Disc( ", disc.sheet, ", ", disc.x, ", ", disc.y, 
           ", ", disc.r, " )" );
end;


#############################################################################
##
#F  DiscOps.Destroy( <disc> ) . . . . . . . . . . . . . . . .  destroy <disc>
##
DiscOps.Destroy := function( disc )
    WcDestroy( disc.sheet.id, disc.id );
    disc.isAlive := false;
end;


#############################################################################
##
#F  DiscOps.MoveDelta( <disc>, <dx>, <dy> ) . . . . . . . . . . .  delta move
##
DiscOps.MoveDelta := function( disc, dx, dy )

    # make sure that we really have to move
    if dx = 0 and dy = 0  then return;  fi;

    # remove old circle
    WcDestroy( disc.sheet.id, disc.id );

    # update coordinates
    disc.x := disc.x+dx;
    disc.y := disc.y+dy;

    # and draw a new circle
    disc.id := WcDrawDisc( disc.sheet.id, disc.x, disc.y, disc.r );
    disc.isAlive := true;

end;


#############################################################################
##
#F  DiscOps.PSString( <disc> )  . . . . . . . . . . . . . . PostScript string
##
DiscOps.PSString := function( disc )
    return Concatenation(
        "newpath\n",
        String(disc.x), " ", String(disc.sheet.height-disc.y), " ",
        String(disc.r), " 0 360 arc\nfill\n" );
end;


#############################################################################
##
#F  DiscOps.Reshape( <disc>, <r> )  . . . . . . . . . . . . . . change radius
##
DiscOps.Reshape := function( d )
    
    # remove old disc
    WcDestroy( d[1].sheet.id, d[1].id );
    
    # and create a new one
    d[1].id := WcDrawDisc( d[1].sheet.id, d[1].x, d[1].y, d[2] );
    d[1].isAlive := true;

    # and update radius
    d[1].r := d[2];

end;


#############################################################################
##
#F  DiscOps.in( <pos>, <disc> ) . . . . . . . . . . . . . . . <pos> in <disc>
##
DiscOps.\in := function( pos, disc )
    return (pos[1]-disc.x)^2+(pos[2]-disc.y)^2 < (disc.r+3)^2;
end;


#############################################################################
##

#F  Diamond( <sheet>, <x1>, <y1>, <x2>, <y2> )  . . draw a diamond in a sheet
##
DiamondOps := rec( name := "DiamondOps" );

Diamond := function( sheet, x1, y1, x2, y2 )
    local   dia,  x3,  y3,  x4,  y4;

    # create the other two points
    x3 := 2*x2-x1;
    y3 := y1;
    x4 := x2;
    y4 := 2*y1-y2;
    
    # create a diamond record
    dia           := sheet.operations.CreateObject( sheet, DiamondOps );
    dia.isDiamond := true;
    dia.x1        := x1;
    dia.x2        := x2;
    dia.y1        := y1;
    dia.y2        := y2;
    dia.x3        := x3;
    dia.y3        := y3;
    dia.x4        := x4;
    dia.y4        := y4;
    dia.width     := 1;

    # draw the diamond and get the identifier
    WcSetLineWidth( sheet.id, dia.width );
    dia.id1 := WcDrawLine( sheet.id, x1, y1, x2, y2 );
    dia.id2 := WcDrawLine( sheet.id, x2, y2, x3, y3 );
    dia.id3 := WcDrawLine( sheet.id, x3, y3, x4, y4 );
    dia.id4 := WcDrawLine( sheet.id, x4, y4, x1, y1 );

    # and return
    return dia;

end;

DiamondOps.PrintInfo := function( dia )
    Print( "#I  Diamond( W, ", dia.x1, ", ", dia.y1, ", ",
           dia.x2, ", ", dia.y2, " ) = ", dia.id1, "+",
           dia.id2, "+", dia.id3, "+", dia.id4, ".",
    	   Position(dia.sheet.objects,dia), "\n" );
end;

DiamondOps.Print := function( dia )
    Print( "Diamond( ", dia.sheet, ", ", dia.x1, ", ", dia.y1,
           ", ", dia.x2, ", ", dia.y2, " )" );
end;


#############################################################################
##
#F  DiamondOps.Destroy( <dia> ) . . . . . . . . . . . . . . . . destroy <dia>
##
DiamondOps.Destroy := function( dia )
    WindowCmd([ "XRO", dia.sheet.id, dia.id1, dia.id2,
                dia.id3, dia.id4 ]);
    dia.isAlive := false;
end;


#############################################################################
##
#F  DiamondOps.MoveDelta( <dia>, <dx>, <dy> ) . . . . . . . . . .  delta move
##
DiamondOps.MoveDelta := function( dia, dx, dy )
    local   x1,  x2,  x3,  x4,  y1,  y2,  y3,  y4;
    
    # make sure that we really have to move
    if dx = 0 and dy = 0  then return;  fi;

    # delete old lines
    WcDestroy( dia.sheet.id, dia.id1, dia.id2, dia.id3, dia.id4 );
    
    # change coordinates
    x1 := dia.x1 + dx;  dia.x1 := x1;
    x2 := dia.x2 + dx;  dia.x2 := x2;
    x3 := dia.x3 + dx;  dia.x3 := x3;
    x4 := dia.x4 + dx;  dia.x4 := x4;
    y1 := dia.y1 + dy;  dia.y1 := y1;
    y2 := dia.y2 + dy;  dia.y2 := y2;
    y3 := dia.y3 + dy;  dia.y3 := y3;
    y4 := dia.y4 + dy;  dia.y4 := y4;
          
    # readraw the lines
    WcSetLineWidth( dia.sheet.id, dia.width );
    dia.id1 := WcDrawLine( dia.sheet.id, x1, y1, x2, y2 );
    dia.id2 := WcDrawLine( dia.sheet.id, x2, y2, x3, y3 );
    dia.id3 := WcDrawLine( dia.sheet.id, x3, y3, x4, y4 );
    dia.id4 := WcDrawLine( dia.sheet.id, x4, y4, x1, y1 );
    dia.isAlive := true;
    
end;


#############################################################################
##
#F  DiamondOps.PSString( <dia> )  . . . . . . . . . . . . . PostScript string
##
DiamondOps.PSString := function( dia )
    local   x1,  x2,  x3,  x4,  y1,  y2,  y3,  y4;
    
    x1 := dia.x1;  y1 := dia.sheet.height - dia.y1;
    x2 := dia.x2;  y2 := dia.sheet.height - dia.y2;
    x3 := dia.x3;  y3 := dia.sheet.height - dia.y3;
    x4 := dia.x4;  y4 := dia.sheet.height - dia.y4;
    return Concatenation(
        "newpath\n",
        String(x1), " ", String(y1), " moveto\n",
        String(x2), " ", String(y2), " lineto\n",
        String(x3), " ", String(y3), " lineto\n",
        String(x4), " ", String(y4), " lineto\n",
        String(x1), " ", String(y1), " lineto\n",
        String(dia.width), " setlinewidth\n",
        "closepath\nstroke\n" );
end;


#############################################################################
##
#F  DiamondOps.SetWidth( <dia>, <width> ) . . . . . . . . . . .  change width
##
DiamondOps.SetWidth := function( dia, width )
    local   x1,  x2,  x3,  x4,  y1,  y2,  y3,  y4;
    
    x1 := dia.x1;  x2 := dia.x2;  x3 := dia.x3;
    x4 := dia.x4;  y1 := dia.y1;  y2 := dia.y2;
    y3 := dia.y3;  y4 := dia.y4;
    
    # delete old lines
    WcDestroy( dia.sheet.id, dia.id1, dia.id2, dia.id3, dia.id4 );
    
    # set new width
    dia.width := width;

    # readraw the lines
    WcSetLineWidth( dia.sheet.id, dia.width );
    dia.id1 := WcDrawLine( dia.sheet.id, x1, y1, x2, y2 );
    dia.id2 := WcDrawLine( dia.sheet.id, x2, y2, x3, y3 );
    dia.id3 := WcDrawLine( dia.sheet.id, x3, y3, x4, y4 );
    dia.id4 := WcDrawLine( dia.sheet.id, x4, y4, x1, y1 );
    dia.isAlive := true;

end;


#############################################################################
##
#F  DiamondOps.in( <pos>, <dia> ) . . . . . . . . . . . . . .  <pos> in <dia>
##
DiamondOps.\in := function( pos, dia )
    return     Minimum( dia.x1, dia.x3 ) <= pos[1]
           and pos[1] <= Maximum( dia.x1, dia.x3 )
           and Minimum( dia.y2, dia.y4 ) <= pos[2]
           and pos[2] <= Maximum( dia.y2, dia.y4 );    
end;


#############################################################################
##

#F  Text( <sheet>, <font>, <x>, <y>, <str> )  . . . . write a text in a sheet
##
TextOps := rec( name := "TextOps" );

Text := function( sheet, font, x, y, str )
    local   text;

    # create a text object in <sheet>
    text        := sheet.operations.CreateObject( sheet, TextOps );
    text.isText := true;
    text.x      := x;
    text.y      := y;
    text.font   := font;
    text.text   := Copy(str);

    # draw the text and get the identifier
    text.id := WcDrawText( sheet.id, font, x, y, str );
    
    # and return
    return text;

end;

TextOps.PrintInfo := function( text )
    Print( "#I  Text( W, ", text.font, ", ", text.x, ", ",
           text.y, ", \"", text.text, "\" ) = ", text.id, ".",
           Position(text.sheet.objects,text), "\n" );
end;

TextOps.Print := function( text )
    Print( "Text( ", text.sheet, ", ", text.font, ", ", text.x, ", ",
           text.y, ", \"", text.text, "\" )" );
end;


#############################################################################
##
#F  TextOps.Destroy( <text> ) . . . . . . . . . . . . . . . .  destroy <text>
##
TextOps.Destroy := function( text )
    WindowCmd([ "XRO", text.sheet.id, text.id ]);
    text.isAlive := false;
end;


#############################################################################
##
#F  TextOps.MoveDelta( <text>, <dx>, <dy> ) . . . . . . . . . . .  delta move
##
TextOps.MoveDelta := function( text, dx, dy )
    
    # make sure that we really have to move
    if dx = 0 and dy = 0  then return;  fi;

    # destroy old text
    WcDestroy( text.sheet.id, text.id );
    
    # and redraw it at the new position
    text.x := text.x + dx;
    text.y := text.y + dy;
    text.id := WcDrawText(text.sheet.id,text.font,text.x,text.y,text.text);
    text.isAlive := true;

end;


#############################################################################
##
#F  TextOps.PSString( <text> )  . . . . . . . . . . . . . . PostScript string
##
TextOps.PSString := function( text )
    local   save_text,  c,  a,  b;

    save_text := "";
    for c  in text.text  do
        if c = ')'  then
            Add( save_text, '\\' );
        fi;
        Add( save_text, c );
    od;
    a := QuoInt( FONTS[text.font][1] * 150, 100 );
    b := QuoInt( FONTS[text.font][3] * 175, 100 );
    return Concatenation(
        "/Courier findfont [", String(a), " 0 0 ", String(b),
        " 0 0] makefont setfont\n",
        String(text.x), " ", String(text.sheet.height-text.y), " moveto\n",
        "(", save_text, ") show\n" );
end;


#############################################################################
##
#F  TextOps.in( <pos>, <text> ) . . . . . . . . . . . . . . . <text> in <pos>
##
TextOps.\in := function( pos, text )
    local   d,  x1,  x2,  y1,  y2;

    d  := FONTS[text.font];
    y1 := text.y - d[1];
    y2 := text.y + d[2];
    x1 := text.x;
    x2 := text.x + Length(text.text) * d[3];
    return x1 <= pos[1] and pos[1] <= x2 and y1 <= pos[2] and pos[2] <= y2;
    
end;


#############################################################################
##

#F  Vertex( <sheet>, <x>, <y> )	. . . . . . . . . . . . . . . . draw a vertex
##
VertexOps := rec( name := "VertexOps" );

Vertex := function( sheet, x, y )
    local   r,  ver;
    
    # compute the radius
    r := QuoInt( TINY_FONT[3]+4*(TINY_FONT[1]+TINY_FONT[2])+5, 3 );

    # create a vertex record in <sheet>
    ver           := sheet.operations.CreateObject( sheet, VertexOps );
    ver.isVertex  := true;
    ver.x         := x;
    ver.y         := y;
    ver.r         := r;
    ver.ty        := QuoInt( 2*y+TINY_FONT[1]-TINY_FONT[2]+1, 2 );
    ver.tx        := [ x-QuoInt(TINY_FONT[3],2),   x-TINY_FONT[3],
                        x-QuoInt(3*TINY_FONT[3],2), x-2*TINY_FONT[3] ];
    ver.label     := Text( sheet, 1, x, y, "" );
    ver.outline   := [ Circle(sheet,x,y,r) ];
    ver.highlight := false;
    
    # add list of connections
    ver.connections     := [];
    ver.connectingLines := [];
    
    # and return
    return ver;

end;

VertexOps.Print := function( ver )
    Print( "Vertex( ", ver.sheet, ", ", ver.x, ", ", ver.y, " )" );
end;

VertexOps.PrintInfo := function( ver )
    Print( "#I  Vertex( W, ", ver.x, ", ", ver.y,
           " ) = -.", Position(ver.sheet.objects,ver), "\n" );
end;


#############################################################################
##
#F  VertexOps.Destroy( <ver> )  . . . . . . . . . . . . . . . . destroy <ver>
##
VertexOps.Destroy := function( ver )
    local   l;

    for l  in ver.connections  do
        Disconnect( ver, l );
    od;
    for l  in ver.outline  do
        Delete( ver.sheet, l );
    od;
    Delete( ver.sheet, ver.label );
end;


#############################################################################
##
#F  VertexOps.Highlight( <ver> )  . . . . . . . . . . . . .  highlight vertex
##
VertexOps.Highlight := function( ver )
    local   obj;
    
    ver.highlight := true;
    for obj  in ver.outline  do
        obj.operations.SetWidth( obj, 2 );
    od;

end;


#############################################################################
##
#F  VertexOps.Move( <ver>, <x>, <y> ) . . . . . . . . . . . . . absolute move
##
VertexOps.Move := function( ver, x, y )
    local   dx,  dy,  obj,  ver2,  pos1,  pos2,  i;
    
    # compute delta move
    dx := x-ver.x;
    dy := y-ver.y;
    if dx = 0 and dy = 0  then return;  fi;
    ver.x  := x;
    ver.y  := y;
    ver.tx := ver.tx + dx;
    ver.ty := ver.ty + dy;
    
    # move all objects
    for obj  in ver.outline  do
        obj.operations.MoveDelta( obj, dx, dy );
    od;
    ver.label.operations.MoveDelta( ver.label, dx, dy );
    
    # move all connections
    for i  in [ 1 .. Length(ver.connections) ]  do
        ver2 := ver.connections[i];
        pos1 := ver.operations.PositionConnection( ver, ver2.x, ver2.y );
        pos2 := ver2.operations.PositionConnection( ver2, ver.x, ver.y );
        obj  := ver.connectingLines[i];
        obj.operations.Reshape([ obj, pos1[1], pos1[2], pos2[1], pos2[2] ]);
    od;

end;


#############################################################################
##
#F  VertexOps.MoveDelta( <ver>, <dx>, <dy> )  . . . . . . . . . .  delta move
##
VertexOps.MoveDelta := function( ver, dx, dy )
    local   dx,  dy,  obj,  ver2,  pos1,  pos2,  i;
    
    # make sure we have to move
    if dx = 0 and dy = 0  then return;  fi;
    ver.x  := ver.x + dx;
    ver.y  := ver.y + dy;
    ver.tx := ver.tx + dx;
    ver.ty := ver.ty + dy;
    
    # move all objects
    for obj  in ver.outline  do
        obj.operations.MoveDelta( obj, dx, dy );
    od;
    ver.label.operations.MoveDelta( ver.label, dx, dy );
    
    # move all connections
    for i  in [ 1 .. Length(ver.connections) ]  do
        ver2 := ver.connections[i];
        pos1 := ver.operations.PositionConnection( ver, ver2.x, ver2.y );
        pos2 := ver2.operations.PositionConnection( ver2, ver.x, ver.y );
        obj  := ver.connectingLines[i];
        obj.operations.Reshape([ obj, pos1[1], pos1[2], pos2[1], pos2[2] ]);
    od;

end;


#############################################################################
##
#F  VertexOps.PSString( <ver> ) . . . . . . . . . . . . . . . . .  do nothing
##
VertexOps.PSString := function( ver )
    return "";
end;


#############################################################################
##
#F  VertexOps.PositionConnection( <ver>, <x>, <y> ) .  connection to <x>, <y>
##
VertexOps.PositionConnection := function( ver, x, y )
    
    # on the same line connect horizontal
    if AbsInt( ver.y - y ) < ver.r  then
        if x < ver.x  then
            return [ ver.x - ver.r, ver.y ];
        else
            return [ ver.x + ver.r, ver.y ];
        fi;
        
    # is it above
    elif y < ver.y  then
        return [ ver.x, ver.y - ver.r ];
        
    # otherwise it is below
    else
        return [ ver.x, ver.y + ver.r ];
    fi;
    
end;


#############################################################################
##
#F  VertexOps.Relabel( <ver>, <text> )  . . . . . . . . change label of <ver>
##
VertexOps.Relabel := function( ver, text )
    Delete( ver.sheet, ver.label );
    if 4 < Length(text)  then text := text{[1..4]};  fi;
    ver.label := Text( ver.sheet, 1, ver.tx[Length(text)], ver.ty, text );
end;


#############################################################################
##
#F  VertexOps.Reshape( <ver>, <shape> ) . . . . . . . . . . . . . . new shape
##
VertexOps.Reshape := function( args )
    local   obj,  ver,  shape;
    
    # get arguments
    ver   := args[1];
    shape := args[2];
    
    # delete old outline
    for obj  in ver.outline  do
        Delete( ver.sheet, obj );
    od;
    ver.outline := [];
    
    # and create new ones
    if shape = 1 or shape = 3  then
        Add( ver.outline, Circle( ver.sheet, ver.x, ver.y, ver.r ) );
    fi;
    if shape = 2 or shape = 3  then
        Add( ver.outline, Diamond( ver.sheet, ver.x-ver.r, ver.y,
                                   ver.x, ver.y-ver.r ) );
    fi;
    
    # high light objects again
    if ver.highlight  then
        for obj  in ver.outline  do
            obj.operations.SetWidth( obj, 2 );
        od;
    fi;
    
end;


#############################################################################
##
#F  VertexOps.Unhighlight( <ver> )  . . . . . . . . . . .  unhighlight vertex
##
VertexOps.Unhighlight := function( ver )
    local   obj;
    
    ver.highlight := false;
    for obj  in ver.outline  do
        obj.operations.SetWidth( obj, 1 );
    od;

end;


#############################################################################
##
#F  VertexOps.in( <pos>, <ver> )  . . . . . . . . . . . . . .  <pos> in <ver>
##
VertexOps.\in := function( pos, ver )
    return (pos[1]-ver.x)^2+(pos[2]-ver.y)^2 < (ver.r+3)^2;
end;


#############################################################################
##

#F  Connection( <C>, <D> )  . . . . . . . . . . . . . .  connect two vertices
##
Connection := function( C, D )
    local   L,  pos1,  pos2;
    
    # check if <C> and <D> are already connected
    if C in D.connections  then
    	return D.connectingLines[ Position( D.connections, C ) ];
    fi;

    # compute position
    pos1 := C.operations.PositionConnection( C, D.x, D.y );
    pos2 := D.operations.PositionConnection( D, C.x, C.y );

    # create a line between <C> and <D>
    L := Line( C.sheet, pos1[1], pos1[2], pos2[1], pos2[2] );

    # add line to connections of <C> and <D>
    Add( C.connections, D );  Add( C.connectingLines, L );
    Add( D.connections, C );  Add( D.connectingLines, L );

    # and return the line
    return L;

end;


#############################################################################
##
#F  Disconnect( <C>, <D> )  . . . . . . . . . . . . . disconnect two vertices
##
Disconnect := function( C, D )
    local   pos,  L;
    
    # <C> and <D> must be connected
    pos := Position( D.connections, C );
    if pos = false  then
        Error( "<C> and <D> must be connected" );
    fi;
    
    # remove connection from <C> and <D>
    L := D.connectingLines[pos];
    D.connections := Concatenation(
        D.connections{[1..pos-1]},
        D.connections{[pos+1..Length(D.connections)]} );
    D.connectingLines := Concatenation(
        D.connectingLines{[1..pos-1]},
        D.connectingLines{[pos+1..Length(D.connectingLines)]} );
    pos := Position( C.connections, D );
    C.connections := Concatenation(
        C.connections{[1..pos-1]},
        C.connections{[pos+1..Length(C.connections)]} );
    C.connectingLines := Concatenation(
        C.connectingLines{[1..pos-1]},
        C.connectingLines{[pos+1..Length(C.connectingLines)]} );
    
    # finally delete <L>
    Delete( C.sheet, L );
    
end;
