#############################################################################
##
##  Interface to {\SISYPHOS}
##

#############################################################################
##
#V  SISYPHOS
#V  p
##
##  These are global variables.
##  A perhaps existing variable 'p' will be saved always (except if the
##  computation is interrupted during the run of one of the programs in this
##  file).
##
if not IsBound( SISYPHOS ) then
  SISYPHOS:= rec( SISISO  := false,
                  SISBOOL := false,
                  SISCODE := false,
                  SISTMEM := "1000000",
                  SISPMEM :=  "300000",
                  SISCALL := "bin/sis -b -q -s gap",
                  SISOps  := rec( Print:= function( r )
                                 Print( "rec(\n",
                                        "sizeAutG := ", r.sizeAutG, ",\n",
                                        "sizeInnG := ", r.sizeInnG, ",\n",
                                        "sizeOutG := ", r.sizeOutG, ",\n" );
                                 if IsBound( r.epimorphism ) then
                                   Print( "epimorphism := ",
                                           r.epimorphism, ",\n" );
                                 fi;
                                 Print( "generators := \n", r.generators,
                                         " )" );
                                  end ) );
fi;
if not IsBound( p ) then p:= false; fi;


#############################################################################
##
#F  OrderGL( <n>, <q> ) . . . . . . . . . . . . order of general linear group
##
##  computes the order of $GL(n,q)$, where $q$ is a power of a prime $p$
##
OrderGL := function ( n, q )

    local pi,   # q^i
          ord,  # order of GL(i,q)
          i;    # loop variable

    pi := 1; ord := 1;

    for i in [1..n] do
      ord := ord*pi;
      pi := pi*q;
      ord := ord * (pi-1);
    od;
    return ord;
    end;


#############################################################################
##
#F  IsCompatiblePCentralSeries( <G> ) . . . .  compatible to p-central series
##
##  returns whether the presentation of the polycyclically presented
##  $p$-group <G> is compatible to its exponent-$p$-central series.
##  If this is the case, the component '<G>.isCompatiblePCentralSeries'
##  is set to 'true'.
##
IsCompatiblePCentralSeries := function ( G )

    local x,             # element of generator list
          iscompatible,  # boolean variable that is returned
          i,             # loop variable
          s,             # $p$-central series of <G>
          p;             # prime dividing the order of <G>

    if not IsBound( G.isCompatiblePCentralSeries ) then

      p:= Factors( Size( G ) )[1];
      s:= PCentralSeries( G, p );
      iscompatible:= true;
      i:= 1;
      while iscompatible and ( i < Length( s ) ) do
        for x in s[i].generators do
          iscompatible:= iscompatible and x in G.generators;
        od;
        i:= i+1;
      od;
      G.isCompatiblePCentralSeries:= iscompatible;

    fi;
    return G.isCompatiblePCentralSeries;
    end;


#############################################################################
##
#F  EstimateAmount( <G>, <flag> ) . .  amount of memory needed in {\SISYPHOS}
##
##  estimate amount of temporary memory needed by {\SISYPHOS} to compute the
##  automorphism group of <G>. The calculation is based on the well known
##  upper bound for $Aut(<G>)$, the size of the {\SISYPHOS} data structures
##  that hold a description of $<G>$ and $Aut(<G>)$, and several scalar
##  values.
##  The minimal value returned is 200000, the maximal value 12000000 (bytes).
##
##  If <flag> is 'true', the amount of memory needed to compute an element
##  list for $Aut(<G>)$ is added.
##
EstimateAmount := function( G, flag )

    local d,       # rank of <G>
          p,       # prime dividing the order of <G>
          sg,      # amount needed to store group presentation
          s,       # total amount
          ogl,     # order of $Gl('d','p')$
          n;       # $\log_{'p'}(\|<G>\|)$

    n:= Factors( Size( G ) );
    p:= n[1];
    n:= Length( n );
    d:= n - Length( Factors( Size( FrattiniSubgroup( G ) ) ) );
    sg := 4*n^2*(n+1);
    ogl := OrderGL(d,p);
    s := ogl*(d^2+50) + (20+d*(n-d) )*d*n + 2*sg;

    if flag then
      s := s + 3*ogl*p^(d*(n-d))*d*n;
    fi;
    return Minimum ( 12000000, Maximum( 200000, s ) );
    end;


#############################################################################
##
#F  AgGroupNormalizedAgGroup( <G> ) . . .  convert to normalized presentation
##
##  given a polycyclically presented $p$-group <G>, a normalized (weighted)
##  presentation for <G> and an isomorphism from the old to the new
##  presentation are computed using 'pQuotient'. The function returns
##  a record with components 'P' and 'isomorphism' denoting the new
##  presentation and the isomorphism, respectively.
##
AgGroupNormalizedAgGroup := function ( G )

    local nG,  # record to be returned
          pq,  # pQ structure
          p,   # prime dividing order of <G>
          l,   # list of images of old generators
          i;   # loop variable

    nG := rec();
    p := Factors( Size( G ) )[1];
    pq := pQuotient( FpGroup( G ), p, 0 );
    nG.P := AgGroupFpGroup( FpGroup( pq ) );
    l := pq.epimorphism;
    for i in [ 1 .. Length(l) ] do
      if IsInt( l[i] ) then
        l[i]:= pq.generators[l[i]];
      fi;
    od;

    nG.isomorphism:= GroupHomomorphismByImages( G, nG.P, G.generators,
         List( l, x -> MappedWord( x, pq.generators, nG.P.generators ) ) );

    return nG;
    end;


#############################################################################
##
#F  PrintSISYPHOSWord( <P>, <a> ) . . . .  convert agword to {\SISYPHOS} word
##
##  For a polycyclically presented group <P> and an element <a> of <P>,
##  'PrintSISYPHOSWord( <P> ,<a> )' prints a string that encodes <a> in the
##  input format of the {\SISYPHOS} system.
##
##  The string '\"1\"' means the identity element, the other elements are
##  products of powers of generators, the <i>-th generator is given the
##  name 'g<i>'.
##
PrintSISYPHOSWord := function( P, a )

    local list,   # list of exponents of 'a' w. r. to the IGS of 'P'
          k,      # position of first nonzero entry in 'list', if exists
          l,      # loop variable, actual position in 'list'
          count;  # number of printed characters in actual output line

    list:= P.operations.Exponents( P, a, Integers );
    k:= 1;
    while k <= Length( list ) and list[k] = 0 do k:= k + 1;  od;

    # special case of the identity element
    if k > Length( list ) then
      Print( "1" );
      return;
    fi;

    count:= 15;
    if list[ k ] <> 1  then
      Print( "g", k, "^", list[k] );
      count:= count + 2 + Length( String( list[k] ) );
    else
      Print( "g", k );
      count:= count + 1 + Length( String( k ) );
    fi;
    for l in [ k + 1 .. Length( list ) ] do
      if count > 60 then
        Print( "\n " );
        count:= 4;
      fi;
      if list[ l ] > 1  then
        Print( "*g", l, "^", list[l] );
        count:= count + 3 + Length( String( l ) )
                          + Length( String( list[l] ) );
      elif list[ l ] = 1  then
        Print( "*g", l );
        count:= count + 2 + Length( String( l ) );
      fi;
    od;
    end;


#############################################################################
##
#F  PrintSisyphosInputPGroup( <P>, <name>, <type> )  . . . . . sisyphos input
##
##  prints the presentation of the finite $p$-group <P> in a format readable
##  by the {\SISYPHOS} system.  <P> must be a polycyclically or freely
##  presented group.
##
##  In {\SISYPHOS}, the group will be named <name>.
##  If <P> is polycyclically presented the <i>-th generator gets the name
##  'g<i>'.
##  In the case of a free presentation the names of the generators are not
##  changed; note that {\SISYPHOS} accepts only generators names beginning
##  with a letter followed by a sequence of letters and digits.
##
##  <type> must be either '\"pcgroup\"' or the prime dividing the order of
##  <P>.
##  In the former case the {\SISYPHOS} object has type 'pcgroup', <P> must
##  be polycyclically presented for that.
##  In the latter case a {\SISYPHOS} object of type 'group' is created.
##  For avoiding computations in freely presented groups, is *neither*
##  checked that the presentation describes a $p$-group, *nor* that the
##  given prime really divides the group order.
##
##  See the {\SISYPHOS} manual~\cite{Wu93} for details.
##
PrintSisyphosInputPGroup := function( P, name, type )

    local   gens,         # list of generators for <P>
            prime,        # prime dividing the order of <P>
            rank,         # rank of 'P'
            rels,         # relators (of free presentation)
            i, j,         # loop variables
            w,            # word in group <P>
            l;            # length of word w

    if not ( type = "pcgroup" or IsPrimeInt( type ) ) then
      Error( "<type> must be \"pcgroup\" resp. a prime number" );
    fi;

    if IsFpGroup( P ) then

      if not IsPrimeInt( type ) then
        Error( "<type> must be a prime number in case of free presentation" );
      fi;

      prime:= type;
      rank:= pQuotient( P, prime, 1 ).dimensions[1];

      # Get the generators and relators for the group <P>.
      gens:= P.generators;
      rels:= P.relators;

      # Initialize group and generators.
      Print( name, " = group (" );
      if Length( gens ) = rank then
        Print( "minimal,\n" );
      fi;
      Print( prime, ",\ngens(\n" );
      for i in [ 1 .. Length( gens ) - 1 ] do
        Print( gens[i], ",\n" );
      od;
      Print( gens[ Length( gens ) ], "),\n" );
      Print( "rels(\n" );

      for i in [ 1 .. Length( rels ) ] do
        w:= rels[i];
        l:= LengthWord( w );
        while l > 12 do
          Print( Subword( w, 1, 12 ), "*\n" );
          w:= Subword( w, 13, l );
          l:= l - 12;
        od;
        if i < Length( rels ) then
          Print( w, ",\n" );
        else
          Print( w, "));\n" );
        fi;
      od;

    elif IsAgGroup( P ) then

      if type <> "pcgroup" then
        type:= "group";
      fi;

      prime:= Set( Factors( Size( P ) ) );
      if Length( prime ) = 1 then
        prime:= prime[1];
      else
        Error( "SISYPHOS allows p-groups only" );
      fi;

      # Get the generators for the group <P>.
      gens:= Igs( P );

      # Initialize group and generators.
      Print( name, " = ", type, "(", prime, ",\ngens(\n" );
      for i  in [ 1 .. Length( gens ) - 1 ]  do
        Print( "g", i, ",\n" );
      od;
      Print( "g", Length( gens ), "),\n" );
      Print( "rels(\n" );


      # Add the power presentation part.

      Print( "g", 1, "^", RelativeOrderAgWord( gens[ 1 ] ), " = " );
      PrintSISYPHOSWord( P, gens[ 1 ] ^ RelativeOrderAgWord( gens[ 1 ] ) );

      for i  in [ 2 .. Length( gens ) ]  do
        Print( ",\ng", i, "^", RelativeOrderAgWord( gens[ i ] ), " = " );
        PrintSISYPHOSWord( P, gens[ i ] ^ RelativeOrderAgWord( gens[ i ] ) );
      od;

      # Add the commutator presentation part.
      for i  in [ 1 .. Length( gens ) - 1 ]  do
        for j  in [ i + 1 .. Length( gens ) ]  do
          w:= Comm( gens[j], gens[i] );
          if w <> P.identity then
            Print( ",\n[g", j, ",g", i, "] = " );
            PrintSISYPHOSWord( P, Comm( gens[ j ], gens[ i ] ) );
          fi;
        od;
      od;

      # Postamble.
      Print( "));\n" );

    else
      Error( "<P> must be a polycyclically or freely presented p-group" );
    fi;

    end;


#############################################################################
##
#F  SisyphosAutomorphisms( <P>, <flags> ) . . . automorphism group of p-group
##
##  general interface to SISYPHOS's 'automorphisms' function
##
##  *Note*\:\ If the component '<P>.isCompatiblePCentralSeries' is not bound
##  it is computed.
##
SisyphosAutomorphisms := function( P, flags )

    local f1, f2,  # files for input and output
          globp,   # save global variable 'p'
          isoP,    # record containing normalized presentation 'isoP.P'
                   # and isomorphism <P> -> 'isoP.P'
          iiso,    # inverse isomorphism
          i;       # loop variable

    f1:= TmpName(); PrintTo( f1, " " );
    f2:= TmpName();

    if IsBound( p ) then
      globp:= p;
    fi;
    p:= P;
    if not IsBound( p.1 ) then
      for i in [ 1 .. Length( p.generators ) ] do
        p.(i):= p.generators[i];
      od;
    fi;

    # check if presentation is normalized, if not
    # compute normalized presentation and isomorphism onto
    # that presentation
    isoP:= rec();
    if not IsCompatiblePCentralSeries( P ) then
      isoP:= AgGroupNormalizedAgGroup( P );
      p:= isoP.P;
    fi;

    # at this point the group 'p', that will be  passed to {\SISYPHOS},
    # is in any case given via a normalized pc-presentation.

    # prepare the input file for {\SISYPHOS}
    PrintTo( f1, PrintSisyphosInputPGroup( p, "p", "pcgroup" ), "\n",
                 "set flags = ", flags, ";\n",
                 "au = auto( p );\n",
                 "print( au, images );\n",
                 "makecode ( au );\n",
                 "quit;\n" );

    # compute amount of memory needed
    i := EstimateAmount ( p, false );
    SISYPHOS.SISTMEM := String ( i );
    SISYPHOS.SISPMEM := String ( Int(i/3) );

    SISYPHOS.SISISO:= 0;

    # call {\SISYPHOS}, read the output, make clean
    ExecPkg( "sisyphos", SISYPHOS.SISCALL,
             Concatenation( " -t ", SISYPHOS.SISTMEM,
                            " -m ", SISYPHOS.SISPMEM,
                            " <", f1, " >", f2 ), "." );
    Read( f2 );
    Exec( Concatenation( "rm ", f1, " ", f2 ) );

    # check whether the output file contained the result
    if SISYPHOS.SISISO = 0 then
      Error( "output file was not readable" );
    fi;

    SISYPHOS.SISISO.generators:= List( SISYPHOS.SISISO.generators, x ->
                    GroupHomomorphismByImages( p, p, p.generators, x ) );

    # pull back automorphisms to original group if necessary
    if IsBound( isoP.isomorphism ) then
      iiso:= InverseMapping( isoP.isomorphism );
      SISYPHOS.SISISO.generators:= List( SISYPHOS.SISISO.generators, x ->
                     isoP.isomorphism * x * iiso );
    fi;

    # restore global variable 'p'
    if IsBound( globp ) then p:= globp; fi;

    # is this a group of outer automorphisms ?
    if flags[5] = 1 then
      SISYPHOS.SISISO.outer := false;
    else
      SISYPHOS.SISISO.outer := true;
    fi;

    # store coded description
    SISYPHOS.SISISO.SIScode := SISYPHOS.SISCODE;

    # supply special 'Print' function
    SISYPHOS.SISISO.operations := SISYPHOS.SISOps;

    # return the result
    return SISYPHOS.SISISO;

    end;


##############################################################################
##
#F  Automorphisms( <P> ) . . . . . . . . .  full automorphism group of p-group
#F  OuterAutomorphisms( <P> )  . . . . . . outer automorphism group of p-group
#F  NormalizedAutomorphisms( <P> ) . . . . normalized automorphisms of p-group
#F  NormalizedOuterAutomorphisms(<P>)  .  norm. outer automorphisms of p-group
##
Automorphisms :=                P -> SisyphosAutomorphisms( P, [1,0,0,1,1] );
OuterAutomorphisms :=           P -> SisyphosAutomorphisms( P, [1,0,0,1,0] );
NormalizedAutomorphisms :=      P -> SisyphosAutomorphisms( P, [1,0,1,1,1] );
NormalizedOuterAutomorphisms := P -> SisyphosAutomorphisms( P, [1,0,1,1,0] );


##############################################################################
##
#F  PresentationAutomorphisms( <P>, <flag> ) . . automorphism group of p-group
##
##  returns a polycyclically presented group isomorphic to the normalized
##  automorphisms of the polycyclically presented $p$-group <P>.
##  'flag' may have the values '\"all\"' or '\"outer\"'; in the latter case
##  only the group of normalized outer automorphisms is returned.
##
##  The group has a component 'SISAuts' whose generators correspond to the
##  generators of the returned group.
##
##  *Note*\:\ If the component '<P>.isCompatiblePCentralSeries' is not bound
##  it is computed.
##
PresentationAutomorphisms := function( P, flag )

    local f1, f2,  # files for input and output
          globp,   # save global variable 'p'
          isoP,    # record containing normalized presentation isoP.P
                   # and isomorphism <P> -> 'isoP.P'
          iiso,    # inverse isomorphism
          i,       # loop variable
          SISflags;

    f1:= TmpName(); PrintTo( f1, " " );
    f2:= TmpName();

    if IsBound( p ) then globp:= p; fi;
    p:= P;

    if not IsBound( p.1 ) then
      for i in [ 1 .. Length( p.generators ) ] do
        p.(i):= p.generators[i];
      od;
    fi;

    SISflags:= [ 1, 0, 1, 1, 1 ];

    if flag <> "outer" then
      flag:= "all";
    fi;

    # check if presentation is normalized, if not
    # compute normalized presentation and isomorphism onto
    # that presentation
    isoP := rec();
    if not IsCompatiblePCentralSeries ( P ) then
         isoP := AgGroupNormalizedAgGroup ( P );
         p := isoP.P;
    fi;


    # at this point the group p, that will be  passed to {\SISYPHOS},
    # is in any case given via a normalized pc-presentation.

    # prepare the input file for {\SISYPHOS}
    PrintTo( f1, PrintSisyphosInputPGroup( p, "p", "pcgroup" ), "\n",
                 "set flags = ", SISflags, ";\n",
                 "au = auto( p );\n",
                 "presentation ( au, ", flag, " );\n",
                 "quit;\n" );

    # compute amount of memory needed
    i := EstimateAmount ( p, false );
    SISYPHOS.SISTMEM := String ( i );
    SISYPHOS.SISPMEM := String ( Int(i/3) );

    SISYPHOS.SISISO:= 0;

    # call {\SISYPHOS}, read the output, make clean
    ExecPkg( "sisyphos", SISYPHOS.SISCALL,
             Concatenation( " -t ", SISYPHOS.SISTMEM,
                            " -m ", SISYPHOS.SISPMEM,
                            " <", f1, " >", f2 ), "." );
    Read( f2 );
    Exec( Concatenation( "rm ", f1, " ", f2 ) );

    # check whether the output file contained the result
    if SISYPHOS.SISISO = 0 then
      Error( "output file was not readable" );
    fi;

    SISYPHOS.SISISO.SISAuts.generators:=
        List( SISYPHOS.SISISO.SISAuts.generators, x ->
              GroupHomomorphismByImages( p, p, p.generators, x ) );

    # pull back automorphisms to original group if necessary
    if IsBound( isoP.isomorphism ) then
      iiso:= InverseMapping( isoP.isomorphism );
      SISYPHOS.SISISO.SISAuts.generators:=
              List( SISYPHOS.SISISO.SISAuts.generators,
                    x -> isoP.isomorphism * x * iiso );
    fi;

    # restore global variable 'p'
    if IsBound( globp ) then p:= globp; fi;

    # return the result
    return SISYPHOS.SISISO;

    end;


##############################################################################
##
#F  AgNormalizedAutomorphisms( <P> ) . . . . . . . normalized automorphisms of
#F                                                      p-group <P> as AgGroup
#F  AgNormalizedOuterAutomorphisms( <P> )  . normalized outer automorphisms of
#F                                                      p-group <P> as AgGroup
##
AgNormalizedAutomorphisms :=      P -> PresentationAutomorphisms( P, "all"  );
AgNormalizedOuterAutomorphisms := P -> PresentationAutomorphisms( P, "outer");


##############################################################################
##
#F  IsIsomorphic( <P1>, <P2> ) . . . . . . . .  isomorphism check for p-groups
##
##  returns 'true' if the $p$-groups <P1>, <P2> are isomorphic, 'false'
##  otherwise.  <P2> must be polycyclically presented, <P1> must be freely or
##  polycyclically presented.
##
##  *Note*\:\ If the component '<P2>.isCompatiblePCentralSeries' is not bound
##  it is computed.
##
IsIsomorphic := function( P1, P2 )

    local f1, f2,  # files for input and output
          globp,   # save global variable 'p'
          isoP,    # record containing normalized presentation isoP.P
                   # and isomorphism <P> -> 'isoP.P'
          iiso,    # inverse isomorphism
          type,    # string '\"pcgroup\"' or a prime
          i;       # loop variable

    f1:= TmpName(); PrintTo( f1, " " );
    f2:= TmpName();

    # check type of P2
    if not IsAgGroup ( P2 ) then
       Error ( "<P2> must be a polycyclically presented p-group" );
    fi;

    # check type of P1
    if   IsAgGroup( P1 ) then

      if not IsBound( P1.1 ) then
        for i in [ 1 .. Length( P1.generators ) ] do
          P1.(i):= P1.generators[i];
        od;
      fi;
      type:= "pcgroup";

    elif IsFpGroup( P1 ) then

      type:= FactorsInt( Order( P2, P2.1 ) )[1];

    else
      Error( "<P1> must be a polycyclically or freely presented p-group" );
    fi;

    if not IsBound( P2.1 ) then
      for i in [ 1 .. Length( P2.generators ) ] do
        P2.(i):= P2.generators[i];
      od;
    fi;

    if IsBound( p ) then globp:= p; fi;
    p:= P2;

    # check if presentation is normalized, if not
    # compute normalized presentation and isomorphism onto
    # that presentation
    isoP := rec();
    if not IsCompatiblePCentralSeries ( P2 ) then
         isoP := AgGroupNormalizedAgGroup ( P2 );
         p := isoP.P;
    fi;

    # at this point the group p, that will be  passed to {\SISYPHOS},
    # is in any case given via a normalized pc-presentation.

    # prepare the input file for {\SISYPHOS}
    PrintTo( f1, PrintSisyphosInputPGroup( P1, "q", type ), "\n",
                 PrintSisyphosInputPGroup( p, "p", "pcgroup" ), "\n",
                 "set displaystyle gap;\n",
                 "isomorphic(p,q);\n",
                 "quit;\n" );

    # compute amount of memory needed
    i := EstimateAmount ( p, false );
    SISYPHOS.SISTMEM := String ( i );
    SISYPHOS.SISPMEM := String ( Int(i/3) );

    SISYPHOS.SISBOOL:= 0;

    # call {\SISYPHOS}, read the output, make clean
    ExecPkg( "sisyphos", SISYPHOS.SISCALL,
             Concatenation( " -t ", SISYPHOS.SISTMEM,
                            " -m ", SISYPHOS.SISPMEM,
                            " <", f1, " >", f2 ), "." );
    Read( f2 );
    Exec( Concatenation( "rm ", f1, " ", f2 ) );

    # check whether the output file contained the result
    if SISYPHOS.SISBOOL = 0 then
      Error( "output file was not readable" );
    fi;

    # restore global variable 'p'
    if IsBound( globp ) then p:= globp; fi;

    # return the result
    return SISYPHOS.SISBOOL;
    end;


##############################################################################
##
#F  Isomorphisms( <P1>, <P2> ) . . . . . . . . . isomorphisms between p-groups
##
##  If the polycyclically presented $p$-groups <P1>, <P2> are not isomorphic,
##  'Isomorphisms' returns 'false'.
##  Otherwise a record is returned that encodes the isomorphisms from <P1> to
##  <P2>; its components are
##
##  'epimorphism':\\  a list of images of '<P1>.generators' that defines an
##                    isomorphism from <P1> to <P2>,
##
##  'generators':\\   a list of automorphisms that
##                    together with the inner automorphisms generate the full
##                    automorphism group of <P2>
##
##  'sizeOutG':\\     size of the group of outer automorphisms of <P2>,
##
##  'sizeInnG':\\     size of the group of inner automorphisms of <P2>,
##
##  'sizeAutG':\\     size of the full automorphism group of <P2>.
##
##  (The function 'IsIsomorphic' tests for isomorphism of $p$-groups.)
##
##  *Note*\:\ If the component '<P2>.isCompatiblePCentralSeries' is not bound
##  it is computed.
##
Isomorphisms := function( P1, P2 )

    local f1, f2,  # files for input and output
          globp,   # save global variable 'p'
          isoP,    # record containing normalized presentation isoP.P
                   # and isomorphism <P> -> 'isoP.P'
          iiso,    # inverse isomorphism
          type,    # string '\"pcgroup\"' or a prime
          i;       # loop variable

    f1:= TmpName(); PrintTo( f1, " " );
    f2:= TmpName();

    # check type of P2
    if not IsAgGroup( P2 ) then
      Error( "<P2> must be a polycyclically presented p-group" );
    fi;

    # check type of P1
    if   IsAgGroup( P1 ) then

      if not IsBound( P1.1 ) then
        for i in [ 1 .. Length( P1.generators ) ] do
          P1.(i):= P1.generators[i];
        od;
      fi;
      type:= "pcgroup";

    elif IsFpGroup( P1 ) then

      type:= FactorsInt( Order( P2, P2.1 ) )[1];

    else
      Error( "<P1> must be a polycyclically or freely presented p-group" );
    fi;

    if not IsBound( P2.1 ) then
      for i in [ 1 .. Length( P2.generators ) ] do
        P2.(i):= P2.generators[i];
      od;
    fi;

    if IsBound( p ) then globp:= p; fi;
    p:= P2;

    # check if presentation is normalized, if not
    # compute normalized presentation and isomorphism onto
    # that presentation
    isoP := rec();
    if not IsCompatiblePCentralSeries ( P2 ) then
         isoP := AgGroupNormalizedAgGroup ( P2 );
         p := isoP.P;
    fi;

    # at this point the group p, that will be  passed to {\SISYPHOS},
    # is in any case given via a normalized pc-presentation.

    # prepare the input file for {\SISYPHOS}
    PrintTo( f1, PrintSisyphosInputPGroup( P1, "q", type ), "\n",
                 PrintSisyphosInputPGroup( p, "p", "pcgroup" ), "\n",
                 "is = isomorphisms(p,q);\n",
                 "print( is, images );\n",
                 "makecode ( is );\n",
                 "quit;\n" );

    # compute amount of memory needed
    i := EstimateAmount ( p, false );
    SISYPHOS.SISTMEM := String ( i );
    SISYPHOS.SISPMEM := String ( Int(i/3) );

    SISYPHOS.SISISO:= 0;

    # call {\SISYPHOS}, read the output, make clean
    ExecPkg( "sisyphos", SISYPHOS.SISCALL,
             Concatenation( " -t ", SISYPHOS.SISTMEM,
                            " -m ", SISYPHOS.SISPMEM,
                            " <", f1, " >", f2 ), "." );
    Read( f2 );
    Exec( Concatenation( "rm ", f1, " ", f2 ) );

    # check whether the output file contained the result
    if SISYPHOS.SISISO = 0 then
      Error( "output file was not readable" );
    fi;

    # check whether groups are isomorphic
    if SISYPHOS.SISISO <> false then

      SISYPHOS.SISISO.generators:= List( SISYPHOS.SISISO.generators, x ->
                      GroupHomomorphismByImages( p, p, p.generators, x ) );

      # pull back automorphisms and epimorphism to original group if necessary
      if IsBound( isoP.isomorphism ) then
        iiso:= InverseMapping( isoP.isomorphism );
        SISYPHOS.SISISO.generators:= List( SISYPHOS.SISISO.generators, x ->
                       isoP.isomorphism * x * iiso );
        SISYPHOS.SISISO.epimorphism:= List( SISYPHOS.SISISO.epimorphism,
           x->Image ( iiso, x ) );
      fi;

      SISYPHOS.SISISO.outer := true;

      # store coded description
      SISYPHOS.SISISO.SIScode := SISYPHOS.SISCODE;

      # supply special 'Print' function
      SISYPHOS.SISISO.operations := SISYPHOS.SISOps;

    fi;

    # restore global variable 'p'
    if IsBound( globp ) then p:= globp; fi;

    # return the result
    return SISYPHOS.SISISO;

    end;


##############################################################################
##
#F  CorrespondingAutomorphism( <A>, <w> ) . .  automorphism corresp. to agword
##
##  If <A> is a polycyclically presented group of automorphisms of a group $P$
##  (as returned by "AgNormalizedAutomorphisms" 'AgNormalizedAutomorphisms' or
##  "AgNormalizedOuterAutomorphisms" 'AgNormalizedOuterAutomorphisms'),
##  and <w> is an element of <A> then 'CorrespondingAutomorphism( <A>, <w> )'
##  returns the automorphism of $P$ corresponding to <w>.
##
##  *Note*\:\ If the component '$P$.isCompatiblePCentralSeries' is not bound
##  it is computed.
##
CorrespondingAutomorphism := function( A, w )

    local f1, f2,  # files for input and output
          globp,   # save global variable 'p'
          isoP,    # record containing normalized presentation isoP.P
                   # and isomorphism <P> -> 'isoP.P'
          iiso,    # inverse isomorphism
          func,    # local function (to avoid using 'AppendTo')
          i, j, l; # loop variables used in 'func'

    f1 := TmpName(); PrintTo( f1, " " );
    f2 := TmpName();

    # check argument types
    if not IsAgGroup( A ) then
      Error( "<A> must be a a polycyclically presented p-group" );
    fi;
    if not IsBound( A.SISAuts ) then
      Error( "<A> is not an automorphism group" );
    fi;
    if not IsAgWord( w ) or not ( w in A ) then
      Error( "<w> is not a word in <A>" );
    fi;

    if IsBound( p ) then globp:= p; fi;

    p:= A.SISAuts.generators[1].source;

    func:= function()
      for i in [ 1 .. Length( A.SISAuts.generators ) ] do
        l:= A.SISAuts.generators[i].genimages;
        Print( "[" );
        for j in [ 1 .. Length( l ) - 1 ] do
          PrintSISYPHOSWord( p, Image ( iiso, l[j] ) );
          Print( "," );
        od;
        PrintSISYPHOSWord( p, Image ( iiso, l[ Length( l ) ] ) );
        if i < Length( A.SISAuts.generators ) then
          Print( "],\n" );
        fi;
      od;
      Print( "]] );\n" );
    end;

    # check if presentation is normalized, if not
    # compute normalized presentation and isomorphism onto
    # that presentation
    isoP := rec();
    if not IsCompatiblePCentralSeries ( p ) then
         isoP := AgGroupNormalizedAgGroup ( p );
         p := isoP.P;
         iiso := isoP.isomorphism;
     else
         iiso := IdentityMapping ( p );
    fi;

    # at this point the group p, that will be  passed to {\SISYPHOS},
    # is in any case given via a normalized pc-presentation.

    PrintTo( f1, PrintSisyphosInputPGroup ( p, "p", "pcgroup" ), "\n",
                 "autlist = automorphism list ( p, [",
                 func(),
                 "asauto ( autlist, ",
                 ExponentsAgWord ( w ), ", images );\n",
                 "quit;\n" );

    # compute amount of memory needed
    i := EstimateAmount ( p, false );
    SISYPHOS.SISTMEM := String ( i );
    SISYPHOS.SISPMEM := String ( Int(i/3) );

    SISYPHOS.SISISO:= 0;

    # call {\SISYPHOS}, read the output, make clean
    ExecPkg( "sisyphos", SISYPHOS.SISCALL,
             Concatenation( " -t ", SISYPHOS.SISTMEM,
                            " -m ", SISYPHOS.SISPMEM,
                            " <", f1, " >", f2 ), "." );
    Read( f2 );
    Exec( Concatenation( "rm ", f1, " ", f2 ) );

    # check whether the output file contained the result
    if SISYPHOS.SISISO = 0 then
      Error( "output file was not readable" );
    fi;

    SISYPHOS.SISISO:=
          GroupHomomorphismByImages( p, p, p.generators, SISYPHOS.SISISO );

    # pull back automorphisms to original group if necessary
    if IsBound( isoP.isomorphism ) then
      iiso:= InverseMapping( isoP.isomorphism );
      SISYPHOS.SISISO:= isoP.isomorphism * SISYPHOS.SISISO * iiso;
    fi;

    # restore global variable 'p'
    if IsBound( globp ) then p:= globp; fi;

    return SISYPHOS.SISISO;

    end;

##############################################################################
##
#F  AutomorphismGroupElements( <A> ) . . .  element list of automorphism group
##
##  <A> has to be an automorphism record as returned by one of the
##  automorphism routines or a list consisting of automorphisms of a $p$-group
##  $P$.  In the first case a list of all elements of $Aut(P)$ or $Aut_n(P)$
##  is returned, if <A> has been created by 'Automorphisms' or
##  'NormalizedAutomorphisms' respectively, or a list of coset representatives
##  of $Aut(P)$ or $Aut_n(P)$ modulo $Inn(P)$, if <A> has been created by
##  'OuterAutomorphisms' or 'NormalizedOuterAutomorphisms', respectively.
##  In the second case the list of all elements of the subgroup of $Aut(P)$
##  generated by <A> is returned.
##
##  *Note*\:\ If the component '$P$.isCompatiblePCentralSeries' is not bound
##  it is computed.
##
AutomorphismGroupElements := function( A )

    local f1, f2,       # files for input and output
          globp,        # save global variable 'p'
          isoP,         # record containing normalized presentation isoP.P
                        # and isomorphism <P> -> 'isoP.P'
          iiso,         # (inverse) isomorphism
          i, j, l,      # loop variables
          func,         # local function (to avoid 'AppendTo')
          type;         # either "outer" or "all"

    f1 := TmpName(); PrintTo( f1, " " );
    f2 := TmpName();

    if IsBound ( p ) then
      globp := p;
    fi;

    # obtain $p$-group to which <A> belongs
    if IsList ( A ) then
        p := A[1].source;
    else
        if  IsBound ( A.elements ) then
            # nothing to do
            return A.elements;
        fi;
        p := A.generators[1].source;
    fi;

    # check if presentation is normalized, if not
    # compute normalized presentation and isomorphism onto
    # that presentation
    isoP := rec();
    if not IsCompatiblePCentralSeries ( p ) then
         isoP := AgGroupNormalizedAgGroup ( p );
         p := isoP.P;
         iiso := isoP.isomorphism;
     else
         iiso := IdentityMapping ( p );
    fi;

    # at this point the group p, that will be  passed to {\SISYPHOS},
    # is in any case given via a normalized pc-presentation.

    # check if <A> is just a list
    if IsList ( A ) then

        func:= function()
          for i in [ 1 .. Length( A ) ] do
            l:= A[i].genimages;
            Print( "[" );
            for j in [ 1 .. Length( l ) - 1 ] do
              PrintSISYPHOSWord( p, Image ( iiso, l[j] ) );
              Print( ",\n" );
            od;
            PrintSISYPHOSWord( p, Image ( iiso, l[ Length( l ) ] ) );
            if i < Length( A ) then
              Print( "],\n" );
            fi;
          od;
          Print( "]] );\n" );
        end;

        PrintTo( f1, PrintSisyphosInputPGroup ( p, "p", "pcgroup" ), "\n",
                "autlist = automorphism list ( p, [",
                func(),
                "auts = elements ( autlist, outer );\n",
                "print ( auts, images );\n",
                "quit;\n" );
    else

        if ( A.outer ) then type := "outer"; else type := "all"; fi;

        PrintTo( f1, "auts = code (", A.SIScode, ");\n",
                "autl = elements ( auts, ", type, ");\n",
                "print ( autl, images, ", type, ");\n",
                "quit;\n" );
    fi;

    # compute amount of memory needed
    i := EstimateAmount ( p, true );
    SISYPHOS.SISTMEM := String ( i );
    SISYPHOS.SISPMEM := String ( Int(i/2) );

    SISYPHOS.SISISO := 0;

    # call {\SISYPHOS}, read the output, make clean
    ExecPkg( "sisyphos", SISYPHOS.SISCALL,
             Concatenation( " -t ", SISYPHOS.SISTMEM,
                            " -m ", SISYPHOS.SISPMEM,
                            " <", f1, " >", f2 ), "." );
    Read( f2 );
    Exec( Concatenation( "rm ", f1, " ", f2 ) );

    # check whether the output file contained the result
    if SISYPHOS.SISISO = 0 then
        Error( "output file was not readable" );
    fi;

    SISYPHOS.SISISO.generators:= List( SISYPHOS.SISISO.generators, x ->
             GroupHomomorphismByImages( p, p, p.generators, x ) );

    # pull back automorphisms to original group if necessary
    if IsBound( isoP.isomorphism ) then
      iiso:= InverseMapping( isoP.isomorphism );
      SISYPHOS.SISISO.generators:= List( SISYPHOS.SISISO.generators, x ->
                     isoP.isomorphism * x * iiso );
    fi;

    if not IsList ( A ) then
        A.elements := SISYPHOS.SISISO.generators;
    fi;

    if IsBound ( globp ) then p := globp; fi;

    return SISYPHOS.SISISO.generators;

    end;


