/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/

#include <strings.h>
#include <stdio.h>
#include <math.h>
 
#include "kask.h"
#include "kasktri.h"
#include "kaskass.h"
#include "kasksol.h"
 
#define	PI		(3.14159265358979323846)
 
TRIANGULATION *actTriang = nil, *firstTriang = nil;
TR *tNextApply;

static TR greenT1, greenT2;
static EDG greenE;
static int level;
static int countP, countE, countT;

static void DeleteSons();

static DummyP() { return; }
 
int InitTri(ptD,edgD,trD)
  int ptD, edgD, trD;
  {
    if(!InitFixedLists(MAXLISTS)) return false;
	InitList(PTVEC, ptD*sizeof(REAL), 100);
	InitList(EDGVEC, edgD*sizeof(REAL), 100);
	InitList(TRVEC, trD*sizeof(REAL), 100);
	InitList(PTELEM, sizeof(PT), 100);
	InitList(EDGELEM, sizeof(EDG), 100);
	InitList(TRELEM, 4*sizeof(TR), 100);
	InitList(LOCALSTIFF, 9*sizeof(REAL), 100);
	InitList(BOUNDELEM, sizeof(BOUNDARY), 10);
	
	return true;
  }

/* ************************************************************
 StdMidPoint   computes midpoint of a line edge
 -----------
 EDG *ed       adress of edge
 PT *p         if not nil, adress of point given by user
 return *PT    adress of new point, nil if failed
************************************************************ */
PT *StdMidPoint(ed,p)
  EDG *ed;
  PT  *p;
  {
   REAL *dirVals=actProblem->dirVals;

    PT *P1 = ed->p1, *P2 = ed->p2;
 
    if (p==nil)
      {
        p = NewP();
        if (p==nil) return nil;
                                 /*  Set boundary point */
        if ((ed->boundP)!=false) p->boundP=ed->boundP;
                                 /* index of the new point */
        p->indexP = (actTriang->noOfPoints)++;
        p->mark = false;
      }
    p->x = (P1->x)+((P2->x)-(P1->x))*HALF;
    p->y = (P1->y)+((P2->y)-(P1->y))*HALF;
	if ((p->boundP)==DIRICHLET)
	   {
	     (actProblem->DirichF)(p->x,p->y,ed->class,dirVals);
	     RA(p,R_SOL) = dirVals[0];
	   }
        else
	  { 
	    RA(p,R_SOL) = (RA(P1,R_SOL)+RA(P2,R_SOL))*0.5;
		if (actSolve!=nil)
		  if ((actSolve->estiP) && (actSolve->estiQuadraticP))
		    RA(p,R_SOL) += RA(ed,R_RQ)/RA(ed,R_DIAG);	
	  }
    return p;
  }
 
REAL Arcus(x,y)   /* utility for ArcMidPoint */
  REAL x,y;
  {
    int quadrant;
    REAL phi,xx,yy;
 
    if (x>=ZERO)
      quadrant = (y>=0)?0:3;
    else
      quadrant = (y>=0)?1:2;
    xx = fabs((quadrant==0)||(quadrant==2)?x:y);
    yy = fabs((quadrant==0)||(quadrant==2)?y:x);
    phi = atan2(yy,xx)+quadrant*REALPI2;
    return phi;
  }
 
/* ************************************************************
 ArcMidPoint   computes midpoint of an arc edge
 -----------
 EDG *ed       adress of edge
 PT *p         if not nil, adress of point given by user
 return *PT    adress of new point, nil if failed
************************************************************ */
PT *ArcMidPoint(ed,p)
  EDG *ed;
  PT  *p;
  {
    REAL *dirVals=actProblem->dirVals;
    REAL radius,phi,phi1,phi2,hx1,hy1,hx2,hy2;
    PT *P1  =ed->p1, *P2 = ed->p2;
 
    if (p==nil)
      {
        p = NewP();
        if (p==nil) return nil;
                                 /*  Set boundary point */
        if ((ed->boundP)!=nil) p->boundP=ed->boundP;
                                 /* index of the new point */
        p->indexP = (actTriang->noOfPoints)++;
        p->mark = false;
      }
    hx1 = P1->x-(ed->bound)->x; hy1 = P1->y-(ed->bound)->y;
    hx2 = P2->x-(ed->bound)->x; hy2 = P2->y-(ed->bound)->y;
    radius = sqrt(hx1*hx1+hy1*hy1);
    phi1 = Arcus(hx1,hy1);
    phi2 = Arcus(hx2,hy2);
    phi = phi2-phi1;
    if (phi>PI) phi -= TWO*REALPI;
    if (phi<-PI) phi += TWO*REALPI;
    phi = phi1+phi*HALF;
    p->x = ((ed->bound)->x)+radius*cos(phi);
    p->y = ((ed->bound)->y)+radius*sin(phi);
	if ((p->boundP)==DIRICHLET)
 	  {
	   (actProblem->DirichF)(p->x,p->y,ed->class,dirVals);
	    RA(p,R_SOL) = dirVals[0];
	   }
      else
	  {
	    RA(p,R_SOL) = (RA(P1,R_SOL)+RA(P2,R_SOL))*0.5; 
		if (actSolve!=nil)
		  if ((actSolve->estiP) && (actSolve->estiQuadraticP))
		    RA(p,R_SOL) += RA(ed,R_RQ)/RA(ed,R_DIAG);
	  }
    return p;
  }
 
/* ************************************************************
 NewP          allocates new point, presets most fields
 ---------
 return *PT    address of point, nil if failed
************************************************************ */
PT* NewP()
  {
    PT *p = (PT*)GetElem(PTELEM);
 
    if (p==nil) return nil;
    p->last = actTriang->lastPoint;
	p->vec = (REAL*)GetElem(PTVEC);
	if ((p->vec)==nil) return nil;
	p->level = actTriang->refLevel;

    if ((actTriang->lastPoint)!=nil) (actTriang->lastPoint)->next = p;
    actTriang->lastPoint = p;
 
    return p;
  }
 
/* ************************************************************
 returnP       releases memory of a point
 ---------
 PT *p         address of the point
************************************************************ */
void ReturnP(p)
  PT *p;
  {
	if ((p->vec) != nil) ReturnElem(PTVEC,(PTR)p->vec);
    if ((p->next)==nil) (actTriang->lastPoint) = p->last;
    else (p->next)->last = p->last;
    if ((p->last)==nil) (actTriang->firstPoint) = p->next;
    else (p->last)->next = p->next;
 
    ReturnElem(PTELEM, (PTR)p);
    return;
  }
 
/* ************************************************************
 NewE          allocates new edge, presets most fields
 ---------
 return *EDG   address of edge, nil if failed
************************************************************ */
EDG* NewE()
  {
    EDG *ed = (EDG*)GetElem(EDGELEM);
 
    if (ed==nil) return nil;
    ed->MidPoint = StdMidPoint;
	ed->boundP = INTERIOR;
	ed->level = actTriang->refLevel;
    ed->last = actTriang->lastEdge;
	ed->vec = (REAL*)GetElem(EDGVEC);
	if ((ed->vec)==nil) return nil;

    if ((actTriang->lastEdge)!=nil) (actTriang->lastEdge)->next = ed;
    actTriang->lastEdge = ed;
 
    return ed;
  }
 
/* ************************************************************
 returnE       releases memory of an edge
 ---------
 EDG *p        address of the edge
************************************************************ */
void ReturnE(ed)
  EDG *ed;
  {

	if ((ed->vec) !=nil) ReturnElem(EDGVEC,(PTR)ed->vec);
    if ((ed->next)==nil) (actTriang->lastEdge) = ed->last;
    else (ed->next)->last = ed->last;
    if ((ed->last)==nil) (actTriang->firstEdge) = ed->next;
    else (ed->last)->next = ed->next;
 
    ReturnElem(EDGELEM, (PTR)ed);
    return;
  }
 
/* ************************************************************
 NewT          allocates new triangle, presets most fields
 ---------
 return *TR    address of triangle, nil if failed
************************************************************ */
TR* New4T()
  {
    TR *t4 = (TR*)GetElem(TRELEM), *t;
	int k;
 
	if (t4==nil) return nil; 
 	for(k = 0; k<4; k++)
	  {
		t = &(t4[k]);
		t->user = nil;
		t->next = &(t4[k+1]);
		t->last = &(t4[k-1]);
		t->level = actTriang->refLevel;
		t->vec = (REAL*)GetElem(TRVEC);
	  }
 	t4[0].last = nil;
	t4[3].next = nil;
 
    return t4;
  }
 
/* ************************************************************
 returnT       releases memory of a triangle
 ---------
 TR *p         address of the triangle
************************************************************ */
void Return4T(t)
  TR *t;
  {
	int k;
    TR *tk;
 
 	for(k = 0; k<4; k++)
	  {
		tk = &(t[k]);
		if ((tk->vec)  !=nil) ReturnElem(TRVEC,(PTR)tk->vec);
		if ((tk->user) !=nil) ReturnElem(LOCALSTIFF,(PTR)tk->user);
	  }  
    ReturnElem(TRELEM, (PTR)t);
    return;
  }
 
/* ***************************************************************
   CrTri              create Triangulation
   --------------
   char*  name        Name of the Triangulation
   Return
      TRIANGULATION*  Adress of the Description of a Triangulation
*************************************************************** */    
 
TRIANGULATION *CrTri(name)
  char *name;
  {
    TRIANGULATION *new;
    int length;
 
    new = (TRIANGULATION*) ZIBAlloc ((long)sizeof(TRIANGULATION));
    if (new==nil)
	  { ZIBStdOut("Not enough memory (CrTri)\n"); return nil; }
 
    if (name!=nil)
      {
        length = strlen(name);
        new->name = (char*)ZIBAlloc((long)(length+1));
        if ((new->name)==nil)
		  { ZIBStdOut("Not enough memory (CrTri)\n"); return nil; }
        strcpy(new->name,name);
      }
    else new->name = nil;
    new->fileName = nil;

    new->noOfPoints = 0;
    new->noOfEdges = 0;
    new->noOfTriangles = 0;
    new->refLevel = 0;
	new->maxDepth = 0;
    new->noOfInitPoints = 0;
    new->noOfInitEdges = 0;
    new->noOfInitTriangles = 0;
	new->iluAvailable = false;

	new->problem = nil;
	new->assemble = nil;
	new->matrix = nil;
	new->solve = nil;
    new->plot = nil;
	new->windows = nil;

    new->NewP = DummyP;
    new->NewE = DummyP;
    new->NewT = DummyP;
    new->ReturnP = DummyP;
    new->ReturnE = DummyP;
    new->ReturnT = DummyP;

    new->firstPoint = nil;
    new->lastPoint = nil;
	new->initPoints = nil;
    new->lastDirectPoint = nil;

    new->firstEdge = nil;
    new->lastEdge = nil;
    new->firstGreenEdge = nil;
	new->initEdges = nil;

    new->firstTriangle = nil;
    new->lastTriangle = nil;
    new->firstGreenTriangle = nil;
	new->initTriangles = nil;

	new->next = firstTriang;

    actTriang = new;
	firstTriang = new;
    return new;
  }
 
/* **************************************************************
   SelTri                  Select Triangulation
   ---------------------
   TRIANGULATION* trigul   Triangulation descriptor adress
   Return int              true or false
************************************************************** */    
int SelTri(trgul)
  TRIANGULATION *trgul;
  {
    actTriang = trgul;
    return true;
  }
 
static PROC retT;
/* ***************************************************************
   CloseTri                 Close Triangulation
   --------------
   TRIANGULATION** trigul   Adress of triangulation descriptor adress
   Return int               true or false
*************************************************************** */    
int CloseTri(trgul)
  TRIANGULATION **trgul;
  {
    EDG *ed,*edNext;
    PT *p,*pNext;
    TR *t;
	TRIANGULATION *triang;
    PROC retP = (*trgul)->ReturnP, retE = (*trgul)->ReturnE;
	int k;
 
    countP  =0; countE = 0; countT = 0;
    retT = (*trgul)->ReturnT;
/*
   Return all Triangles, top-down from the initial triangulation
*/
    t = (*trgul)->initTriangles;
    for (k = 0; k<((*trgul)->noOfInitTriangles); k++)
      DeleteSons(&(t[k]));
/*
   Return all Edges
*/
    ed = (*trgul)->firstEdge;
    while (ed!=nil)
      {
        edNext = ed->next;
        (*retE)(ed);
        ReturnE(ed);
        ed = edNext;
        countE++;
      }
/*
   Return all Points
*/
    p = (*trgul)->firstPoint;
    while (p!=nil)
      {
        pNext  =p->next;
        (*retP)(p);
        ReturnP(p);
        p = pNext;
        countP++;
      }
/*
   Return List Memory
*/
    ZIBFree((PTR)((*trgul)->initTriangles));
	countT += (*trgul)->noOfInitTriangles;
	sprintf(globBuf,"Tri: %4d points, %4d edges, %4d triangles released\n",
	        countP, countE, countT);
	ZIBStdOut(globBuf);

	if ((*trgul)==firstTriang) firstTriang = (*trgul)->next;
	else
	  {
	    triang = firstTriang;
		while ((triang->next)!=(*trgul)) triang = triang->next;
		triang->next = (*trgul)->next;
	  }
    actTriang = firstTriang;
	ZIBFree((PTR)(*trgul));
	return true;
  }

static void DeleteSons(t)
  TR *t;
  {
    TR *ts = t->firstSon;
    int k, lng;

    if (ts==nil) return;
    lng = (ts->type)==T_RED?4:2;
	for(k = 0; k<lng; k++) DeleteSons(&(ts[k]));
    Return4T(ts);
    countT += 4;
	return;
  }
 
/* ***************************************************************
   ApplyP          Apply a routine to a selection of points
   -----------
   PROC Routine    Routine to apply
   int to          selection, see "kasktri.h"
   return int      true or not true, if Routine returned not true
*************************************************************** */
int ApplyP(Routine,to)
  PROC Routine;
  int to;
  {
    PT *p,*pNext;
    int rc, k;
 
    switch (to)
    {
      case boundHier:
      case boundNodal:
        p = actTriang->firstPoint;
        while (p!=nil)
          {
            pNext = p->next;
            if ((p->boundP)!=INTERIOR)
              {
                rc = (*Routine)(p);
                if (rc==false) return rc;
              }
            p = pNext;
          }
        break;
      case nullBound:
        p = actTriang->firstPoint;
        while (p!=nil)
          {
            pNext = p->next;
            if ((p->boundP)==DIRICHLET)
              {
                rc = (*Routine)(p);
                if (rc==false) return rc;
              }
            p = pNext;
          }
        p = actTriang->initPoints;
        for (k = 0; k<(actTriang->noOfInitPoints); k++) 
          {
            if ((p[k].boundP)==NEUMANN)
              {
				rc = (*Routine)(&(p[k]));
				if (rc==false) return rc;
              }
          }
        break;
      case dirichlet:
        p = actTriang->firstPoint;
        while (p!=nil)
          {
            pNext = p->next;
            if ((p->boundP)==DIRICHLET)
              {
                rc = (*Routine)(p);
                if (rc==false) return rc;
              }
            p = pNext;
          }
        break;
      case all:
        p = actTriang->firstPoint;
        while (p!=nil)
          {
            pNext = p->next;
            rc = (*Routine)(p);
            if (rc==false) return rc;
            p = pNext;
          }
		break;
      case allBackward:
        p = actTriang->lastPoint;
        while (p!=nil)
          {
            pNext = p->last;
            rc = (*Routine)(p);
            if (rc==false) return rc;
            p = pNext;
          }
		break;
      case nonBound:
        p = actTriang->firstPoint;
        while (p!=nil)
          {
            pNext = p->next;
			if ((p->boundP)==INTERIOR)
			  {
                rc = (*Routine)(p);
                if (rc==false) return rc;
		      }
            p = pNext;
          }
        break;
      case initial:
        p = actTriang->initPoints;
        for (k = 0; k<(actTriang->noOfInitPoints); k++) 
          {
            rc = (*Routine)(&(p[k]));
            if (rc==false) return rc;
          }
        break;
      case boundInit:
        p = actTriang->initPoints;
        for (k = 0; k<(actTriang->noOfInitPoints); k++) 
          {
            if ((p[k].boundP)!=INTERIOR)
              {
				rc = (*Routine)(&(p[k]));
				if (rc==false) return rc;
              }
          }
        break;
      case direct:
	    if ((actTriang->lastDirectPoint)==nil) break;
        p = actTriang->firstPoint;
        while (p!=nil)
          {
            pNext = p->next;
            rc = (*Routine)(p);
            if (rc==false) return rc;
			if (p==(actTriang->lastDirectPoint)) break;
            p = pNext;
          }
		break;
      case nonDirect:
        p = actTriang->firstPoint;
		if ((actTriang->lastDirectPoint)!=nil)
		  p = (actTriang->lastDirectPoint)->next;
        while (p!=nil)
          {
            pNext = p->next;
            rc = (*Routine)(p);
            if (rc==false) return rc;
            p = pNext;
          }
		break;
    }
    return true;
  }

/* ***************************************************************
   ApplyE          Apply a routine to a selection of edges
   -----------
   PROC Routine    Routine to apply
   int to          selection, see "kasktri.h"
   return int      true or not true, if Routine returned not true
*************************************************************** */
int ApplyE(Routine,to)
  PROC Routine;
  int to;
  {
    EDG *ed,*edNext;
    int rc, k;
 
   switch (to)
    {
      case boundHier:
        ed = actTriang->firstEdge;
        while (ed!=nil)
          {
            edNext = ed->next;
            if ((ed->boundP)!=INTERIOR)
              {
                rc = (*Routine)(ed);
                if (rc==false) return rc;
              }
            ed = edNext;
          }
        break;
      case boundNodal:
        ed = actTriang->firstEdge;
        while (ed!=nil)
          {
            edNext = ed->next;
            if (((ed->boundP)!=INTERIOR)&&((ed->firstSon)==nil))
              {
                rc = (*Routine)(ed);
                if (rc==false) return rc;
              }
            ed = edNext;
          }
        break;
      case dirichlet:
        ed = actTriang->firstEdge;
        while (ed!=nil)
          {
            edNext = ed->next;
            if (((ed->boundP)==DIRICHLET)&&((ed->firstSon)==nil))
              {
                rc = (*Routine)(ed);
                if (rc==false) return rc;
              }
            ed = edNext;
          }
        break;
      case initial:
        ed = actTriang->initEdges;
        for (k = 0; k<(actTriang->noOfInitEdges); k++) 
          {
            rc = (*Routine)(&(ed[k]));
            if (rc==false) return rc;
          }
        break;
      case boundInit:
        ed = actTriang->initEdges;
        for (k = 0; k<(actTriang->noOfInitEdges); k++) 
          {
            if ((ed[k].boundP)!=INTERIOR)
              {
				rc = (*Routine)(&(ed[k]));
				if (rc==false) return rc;
              }
          }
        break;
      case all:
        ed = actTriang->firstEdge;
        while (ed!=nil)
          {
            edNext = ed->next;
    	    rc = (*Routine)(ed);
    	    if (rc==false) return rc;
            ed = edNext;
          }
        break;
      case nodal:
        ed = actTriang->firstEdge;
        while (ed!=nil)
          {
            edNext = ed->next;
			if (((ed->firstSon)==nil)&&((ed->type)!=T_NOTHING))
			  {
                rc = (*Routine)(ed);
                if (rc==false) return rc;
			  }
            ed = edNext;
          }
        break;
      case allBackward:
        ed = actTriang->lastEdge;
        while (ed!=nil)
          {
            edNext = ed->last;
            rc = (*Routine)(ed);
            if (rc==false) return rc;
            ed = edNext;
          }
        break;
      case allGreens:
        ed = actTriang->firstEdge;
        while (ed!=nil)
          {
            if ((ed->type)!=T_GREEN) { ed = ed->next; continue; }
            edNext = ed->next;
            rc = (*Routine)(ed);
            if (rc==false) return rc;
            ed = edNext;
          }
        break;
    }
    return true;
  }
 
/* ***************************************************************
   ApplyT          Apply a routine to a selection of triangles
   -----------
   PROC Routine    Routine to apply
   int to          selection, see "kasktri.h"
   return int      true or not true, if Routine returned not true
*************************************************************** */
int ApplyT(Routine,to)
  PROC Routine;
  int to;
  {
    TR *t, *tNext;
    int rc, k;

	switch (to)
    {
      case all:
        t = actTriang->firstTriangle;
		while (t!=nil)
          {
            tNextApply = t->next;
            rc = (*Routine)(t);
            if (rc==false) return rc;
            t = tNextApply;
          }
		break;
      case initial:
        t = actTriang->initTriangles;
        for (k = 0; k<(actTriang->noOfInitTriangles); k++) 
          {
            rc = (*Routine)(&(t[k]));
            if (rc==false) return rc;
          }
        break;
      case allGreens:
        t = actTriang->firstTriangle;
        while (t!=nil)
          {
            if ((t->type)!=T_GREEN) { t = t->next; continue; }
            tNext = t->next;
            rc = (*Routine)(t);
            if (rc==false) return rc;
            t = tNext;
          }
        break;
    }
    return true;
  }

/* ***************************************************************
   ApplyPLevel     Apply a routine to points of a level
   -----------
   PROC Routine    Routine to apply
   int lev         level of triangulation
   return int      true or false
*************************************************************** */
int ApplyPLevel(Routine, lev)
  PROC Routine;
  int lev;
  {
    PT *p = actTriang->firstPoint;
    int rc;
 
    while (p!=nil)
	  {
		if ((p->level)<=lev)
		  {
			rc = (*Routine)(p);
			if (rc==false) return rc;
		  }
		p = p->next;
	  }
     return true;
  }


/* ***************************************************************
   ApplyELevel     Apply a routine to edges of a level
   -----------
   PROC Routine    Routine to apply
   int lev         level of triangulation
   return int      true or false
*************************************************************** */
int ApplyELevel(Routine, lev)
  PROC Routine;
  int lev;
  {
    EDG *ed = actTriang->firstEdge;
    int rc;

    while (ed!=nil)
	  {
		if ((ed->level)<=lev)
		  {
			if ((ed->firstSon)==nil)
			  {
				if ((ed->type)!=T_NOTHING)
				  {
					rc = (*Routine)(ed);
					if (rc==false) return rc;
				  }
			  }
			else
			  {
				if (((ed->firstSon)->level)>lev)
				  {
					if ((ed->type)!=T_NOTHING)
					  {
						rc = (*Routine)(ed);
						if (rc==false) return rc;
					  }
				  }
			  }
		  }
		ed = ed->next;
	  }
    return true;
  }


/* ***************************************************************
   ApplyTLevel      Apply a routine to triangles of a level
   -----------
   PROC Routine    Routine to apply
   int lev         level of triangulation
   return int      true or false
*************************************************************** */
static int InLevel(t)
  TR *t;
  {
	if ((t->level)>level) return false;
	if ((t->firstSon)==nil) return true;
	else return (((t->firstSon)->level)>level);
  }

int GreenRefinedP(t,lev)
  TR *t;
  int lev;
  {
    EDG *ed = t->e1;

	if ((ed->firstSon)!=nil)
	  if (((ed->firstSon)->level)<=lev) return 0;
	ed = t->e2;
	if ((ed->firstSon)!=nil)
	  if (((ed->firstSon)->level)<=lev) return 1;
	ed = t->e3;
	if ((ed->firstSon)!=nil)
	  if (((ed->firstSon)->level)<=lev) return 2;
	return -1;
  }

static int ApplyTLevRec(Routine, t)
  PROC Routine;
  TR *t;
  {
    TR *tt;
	int rc, refEdgNo;

	if (InLevel(t))
	  {
        refEdgNo = GreenRefinedP(t, level);
		if (refEdgNo>=0)
		  {
		    PT **pts = &(t->p1);
			EDG **eds = &(t->e1);
			int i1 = refEdgNo, i2 = refEdgNo+1 , i3 = refEdgNo+2;

			if (i2>2) i2 -= 3;
			if (i3>2) i3 -= 3;

			greenE.p1 = PM(eds[i1]);
			greenE.p2 = pts[i1];
			greenE.t1 = &greenT1;
			greenE.t2 = &greenT2;
			greenE.level = level;

			greenT1.p1 = pts[i1]; 
			greenT1.p2 = pts[i2];
			greenT1.p3 = PM(eds[i1]);
			greenT1.e2 = &greenE;
			greenT1.e3 = eds[i3];
			greenT1.level = level;
			greenT1.depth = (t->depth)+1;

		    greenT2.p1 = pts[i1];
			greenT2.p2 = PM(eds[i1]);
			greenT2.p3 = pts[i3];
			greenT2.e2 = eds[i2];
			greenT2.e3 = &greenE;
			greenT2.level = level;
			greenT2.depth = (t->depth)+1;

			if (pts[i2]==(eds[i1]->p1))
			  {
			    greenT1.e1 = eds[i1]->firstSon;
				greenT2.e1 = (greenT1.e1)->next;
			  }
			else
			  {
				greenT2.e1 = eds[i1]->firstSon;
				greenT1.e1 = (greenT2.e1)->next;
			  }
			rc = (*Routine)(&greenT1);
			if (!rc) return rc;
			return (*Routine)(&greenT2);
		  }
		else return (*Routine)(t);
	  }
	tt = TRED1SON(t);
	if (!ApplyTLevRec(Routine, tt)) return false;
	tt = TRED2SON(t);
	if (!ApplyTLevRec(Routine, tt)) return false;
	if ((t->type)!=T_RED) return true;
	tt = TRED3SON(t);
	if (!ApplyTLevRec(Routine, tt)) return false;
	tt = TRED4SON(t);
	return ApplyTLevRec(Routine, tt);
  }

int ApplyTLevel(Routine, lev)
  PROC Routine;
  int lev;
  {
    TR *t = actTriang->initTriangles;
    int rc, k;

    level = lev;
	for (k = 0; k<(actTriang->noOfInitTriangles); k++) 
      {
        rc = ApplyTLevRec(Routine, &t[k]);
        if (!rc) return rc;
      }
    return true;
  }
