/* linco -- progam to form a linear combination of two rasterfile images */
/* version 1.1    21 Aug 1990 */

/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */


#include <math.h>      
#include <stdio.h>
#include <strings.h>
#include <values.h>
#include "rasterio.h"
#include "linco.h"

#define ALLOC 1

main( argc, argv ) 
   unsigned int argc;    /* count of arguments parsed by operating system    */
   char *argv[];         /* pointers to arguments parsed by operating system */
   {
   FILE *fp;             /* file pointer */
   struct rasterfile H1,H2;   /* rasterfile headers */
   byte *cmap1,*cmap2;   /* colormaps */
   byte *I1,*I2;         /* byte image pointers */
   byte *u,*v;           /* byte image pointers */
   float *Out;           /* float image pointer */
   float *z;             /* float image pointer */
   float a,b;            /* multipliers */
   float min,max,dif;    /* extrema of calculation */
   float slope,incept;   /* slope and intercept scale factors */
   int imax,imin;        /* output extrema */
   int X,Y;              /* image dimensions */
   int i,j,k;            /* indices */
   int absval = FALSE;   /* T => take absolute value of result */
   int affine = FALSE;   /* T => do affine trnsf. w/ slope, intercept */
   int compress = FALSE; /* T => compression scaling */
   int expand = FALSE;   /* T => expansion or compression scaling */
   int scale = FALSE;    /* T => scale pixels; F => don't scale */
   int translate = FALSE;/* T => translate pixels up or down */
   int up = FALSE;       /* T => translate up; F => translate down */
   int nonzero;          /* operate on nonzero pixels only */
   int n = argc;         /* number of arguments */
   char *name;           /* output file name */


   if (argc < 6)
      {
      fprintf(stderr,
         "usage: linco a ras1 b [ras2|nonzero] -o outimg [-c | -f | -u | -d | -a slope intercept] -b\n");
      exit(0);
      }


   /* get multipliers */
   a = atof( argv[1] );
   b = atof( argv[3] );


   /* get scale / translate flags */
   while ( --n ) if ( argv[n][0] == '-' )
      switch ( argv[n][1] )
         {
         case 'c' :
            { 
            scale  = TRUE;
            compress = TRUE;
            fprintf(stderr,"will compress output if necessary\n");
            break;
            }
         case 'f' :
            { 
            scale  = TRUE;
            expand = TRUE;
            fprintf(stderr,"will force output to have extrema (BLACK,WHITE)\n");
            break;
            }
         case 'u' :
            {
            scale  = TRUE;
            translate = TRUE;
            up = TRUE;
            fprintf(stderr,"will translate output so max == WHITE\n");
            break;
            }
         case 'd' :  
            {
            scale  = TRUE;
            translate = TRUE;
            up = FALSE;
            fprintf(stderr,"will translate output so min == BLACK\n");
            break;
            }
         case 'b' :  
            {
            absval = TRUE;
            fprintf(stderr,"will output absolute value of result\n");
            break;
            }
         case 'a' :
            { 
            scale  = TRUE;
            affine = TRUE;
            slope  = atof( argv[n+1] );
            incept = atof( argv[n+2] );
            fprintf(stderr,"output will be scaled as %f*pixel + %f\n",
                    slope,incept);
            break;
            }
         case 'o' :
            {
            name = argv[n+1];
            break;
            }
         }


   /* open input file 1 */
   fp = OpenFile( argv[2], "IMGPATHI", "r" );
   if (  ReadRasterFile( fp, &H1, &cmap1, &I1, 0, 0, 0, 0, ALLOC )  ) exit( 0 );
   fclose( fp );

   /* remap the image */
   ExtractLuminance( &H1, cmap1, I1, 0, 0, 0, 0 );

   /* get the dimensions */
   X = H1.ras_width;
   Y = H1.ras_height;


   /* open file 2 if there is one */
   nonzero = !strcasecmp(argv[4],"nonzero");
   if ( !nonzero  &&  argv[4][0] != '-' )
      {
      fp = OpenFile( argv[4], "IMGPATHI", "r" );
      if ( ReadRasterFile( fp, &H2, &cmap2, &I2, 0, 0, 0, 0, ALLOC ) ) exit(0);
      fclose( fp );

      /* remap the image */
      ExtractLuminance( &H2, cmap2, I2, 0, 0, 0, 0 );

      if ( (H2.ras_width != X)  ||  (H2.ras_height != Y) )
         {
         fprintf(stderr, "The two input images have unequal dimensions\n");
         exit( 0 );
         }
      }
   else /* there is not a second file */
      I2 = NULL;


   /* make floating point output image. */
   if ( !(Out = (float *)calloc( X*Y, sizeof(float) )) )
         {
         fprintf(stderr, "Unable to allocate float image.");
         exit( 0 );
         }


   /* do the linear combination */
   u = I1;
   v = I2;
   z = Out;
   min = MAXFLOAT;
   max = -MAXFLOAT;
   if ( I2 )  /* then there are two images */
      for ( j=0; j<Y; ++j )
         for ( i=0; i<X; ++i)
            {
            *z = a * (float)*(u++) + b * (float)*(v++);
            if (absval) *z = fabs(*z);
            min = MIN(*z,min);
            max = MAX(*z,max);
            ++z;
            }
   else if ( nonzero )
      for ( j=0; j<Y; ++j )
         for ( i=0; i<X; ++i)
            {
            if (*u) 
               { 
               *z = a * (float)*u + b;
               if (absval) *z = fabs(*z);
               }
            ++u;
            min = MIN(*z,min);
            max = MAX(*z,max);
            ++z;
            }
   else
      for ( j=0; j<Y; ++j )
         for ( i=0; i<X; ++i)
            {
            *z = a * (float)*(u++) + b;
            if (absval) *z = fabs(*z);
            min = MIN(*z,min);
            max = MAX(*z,max);
            ++z;
            }
   dif = max-min;


   /* create output file */
   fp = OpenFile( name, "IMGPATHO", "w" );


   /* scale the image if requested and put in byte image */
   u = I1;
   z = Out;

   if ( scale )
      {
      if ( expand || (compress && (fabs(dif) > (float)WHITE)) )
         {
         a = ((float)WHITE) / dif;
         for ( j=0; j<Y; ++j )        /* compress to fit in byte */
            for ( i=0; i<X; ++i)
               *(u++) = (byte)( ((*(z++) - min) * a) + 0.5 );
         }
      else if ( compress )
         {
         if      ( max > (float)WHITE ) { translate = TRUE;  up = TRUE;  }
         else if ( min < (float)BLACK ) { translate = TRUE;  up = FALSE; }
         else                           { translate = FALSE; scale = FALSE; }
                                        /* --> RESET scale !!! <--- */
         }
      else if ( affine )
         {
         for ( j=0; j<Y; ++j )
            for ( i=0; i<X; ++i)
               {
               k = (int)( (slope * (*(z++)) + incept) + 0.5 );
               *(u++) = (byte)((k > WHITE) ? WHITE : ((k < BLACK) ? BLACK : k));
               }
         }
      u = I1;
      z = Out;
      if ( translate )
         {
         if ( up ) 
            a = ((float)WHITE) - max;  /* translate up (max out == WHITE) */
         else
            a = -min;                  /* translate down (min out == BLACK) */
         for ( j=0; j<Y; ++j )
            for ( i=0; i<X; ++i)
               {                       /* clip in other direction */
               k = (int)( *(z++) + a + 0.5 );
               *(u++) = (byte)((k > WHITE) ? WHITE : ((k < BLACK) ? BLACK : k));
               }
         }
      }

   if ( !scale )                   /* no scaling is specified, clip */
      for ( j=0; j<Y; ++j )
         for ( i=0; i<X; ++i)
            {
            k = (int)(*(z++) + 0.5);
            *(u++) = (byte)((k > WHITE) ? WHITE : ( (k < BLACK) ? BLACK : k ));
            }   

   u = I1;
   imin = WHITE;
   imax = BLACK;
   for ( j=0; j<Y; ++j )
      for ( i=0; i<X; ++i)
         {
         imin = MIN(((int)(*u)),imin);
         imax = MAX(((int)(*u)),imax);
         ++u;
         }

   fprintf( stderr, "%s extrema:  calc - (%.2f,%.2f)  out - (%d,%d)\n",
            name, min, max, imin, imax);

   WriteRasterFile( fp, &H1, cmap1, I1, 0, 0, 0, 0 ); 

   fclose( fp );
   free( I1 );
   free( I2 );
   free( Out );
   if ( cmap1 ) free( cmap1 );
   if ( cmap2 ) free( cmap2 );
               
   }

