/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/vcrEdit/RCS/vcrEditProtocol.c,v 0.35 92/09/29 18:10:43 drapeau Exp $ */
/* $Log:	vcrEditProtocol.c,v $
 * Revision 0.35  92/09/29  18:10:43  drapeau
 * Made slight modifications to PerformSelection(), mostly in the semantics used
 * for the PlayEdit() function.  When calling PlayEdit(), the application now
 * uses case C of the VideoObject's DevPlayFromTo semantics.  Case c states
 * that start == 0 and end != 0, meaning that the device should begin
 * playback from its current point and continue until the address "end"
 * has been reached.  Using this case instead of case D (specifying both start
 * and end addresses, causing an extra search) reduces delay time for most
 * devices.
 * Also, made cosmetic changes to enhance readability and ANSI compliance.
 * 
 * Revision 0.34  92/09/24  16:41:06  drapeau
 * Made changes to the way the VideoObject methods were used.  Part of the
 * reason for the changes is that the semantics of the VideoObject itself
 * have been changed; part of the reason is that vcrEdit was using some
 * bad semantics when using the PlayFromTo method.
 * Also, made a number of cosmetic changes for better readability and
 * ANSI compliance.
 * 
 * Revision 0.33  92/09/08  14:31:16  drapeau
 * Replaced references to NTSC-specific frame rate with the newly-defined
 * constant "FrameRate", which is set at compile time through a Makefile
 * variable.
 * 
 * Revision 0.32  92/05/29  12:35:52  drapeau
 * Modified code to track new name of the MAEstro "Selection" structure;
 * it is now named "MAESelection".
 * 
 * Revision 0.31  92/05/14  00:08:49  drapeau
 * A number of changes were made, the most significant having to do with the
 * implementation of the pre-search mechanism.
 * * Pre-search was flawed in that it assumed that playback always happened
 *   right after authorship of a document (i.e., a document could not be played
 *   repeatedly and still have pre-search work).  This error has been fixed,
 *   by removing the "numOnTL" variable, which used to indicate how many
 *   edits had been placed on the TimeLine (this was done by counting the
 *   number of times that GetSelection() was called, but the number was
 *   only saved during the time the application was open; the next launch
 *   of vcrEdit would cause numOnTL to be reset).
 * 
 *   Also, removed the extraneous variable "nextTLSelection", the function
 *   of which is just as well served by prudent use of the "editnum"
 *   variable, which keeps track of the currently-selected edit.
 * 
 *   Also, the application no longer turns off pre-search when its guess
 *   about the next edit to be played is incorrect.  In essense, pre-search
 *   is like a cache.  vcrEdit used to turn off caching as soon as the first
 *   cache miss occurred.  Now, the cache remains on, assuming that the
 *   cache will succeed most of the time.  This assumption was made based
 *   on real-world authors using vcrEdit over the past five months.
 * 
 *   Also, pre-search checks for the boundary condition when the last edit
 *   has been played.  Pre-search is not turned off, but a flag is set
 *   to indicate that the last edit on the current edit list was played,
 *   so a real search must be done, no matter what edit is to be played
 *   next.
 * 
 * * Made a number of formatting changes to the code to conform to
 *   coding standards.
 * * Made some changes to the code for better ANSI-C conformance.
 * 
 * Revision 0.30  92/01/09  18:18:46  drapeau
 * Slight modifications to make code ANSI-compliant.
 * 
 * Revision 0.29  92/01/03  18:02:55  drapeau
 * Changed all calls to Browse() to use "0" instead of "NULL", to
 * take into account ANSI-C definition of NULL as (void*)0.
 * 
 * Revision 0.28  91/09/29  16:13:54  lim
 * *** empty log message ***
 * 
 * Revision 0.27  91/09/26  12:46:11  lim
 * *** empty log message ***
 * 
 * Revision 0.26  91/09/26  12:34:30  lim
 * 1. Returns from protocol functions immediately if no driver
 * is installed.
 * 2. Corrected duration for GetSelection for the case when no
 * driver is installed.
 * 3. Did casting.
 * 
 * Revision 0.25  91/08/30  15:15:33  lim
 * Don't send error message in InitNetwork.
 * 
 * Revision 0.24  91/08/28  12:58:29  lim
 * Presearching is implemented.
 * Set selection for the first edit is the same as before.
 * In perform selection, a timer is set for the length of
 * the edit. 
 * When the timer expires, SetSelection is called for the
 * next edit on the edit list. 
 * When the TimeLine next calls SetSelection (when the playhead
 * hits the next edit), if the presearched edit is the correct
 * edit, SetSelection will return, allowing PerformSelection to
 * be called almost immediately.
 * If however, it is the wrong edit, then it is likely that the
 * user is not using PreSearching as it was intended (for 
 * edits on the TimeLine that are in the same sequence as in
 * the edit list). In this case, PreSearching is automatically
 * turned off, to prevent performance degradation from searching
 * for the wrong edit.
 * If PauseSelection is hit, the timer is killed, and so presearching
 * will resume only after PerformSelection is called on the next
 * edit.
 *  The number of edits on the TimeLine is estimated by the number of
 * times GetSelection is called. The exact number depends on how many
 * edits are deleted. If the user only places 2 out of 5 edits in
 * his edit list, then the application will not call search on the
 * 3rd edit on the edit list. 
 * 
 * Revision 0.23  91/08/24  18:00:13  lim
 * Implemented PrintDiagnostics. With command line flag '-d', the 
 * app will launch such that it prints all diag messages.
 * 
 * Revision 0.22  91/08/23  17:24:32  lim
 *  Document format has changed.
 * 
 * Revision 0.21  91/08/21  14:34:01  lim
 * Implemented GetAppIcon().
 * 
 * Revision 0.20  91/08/20  16:32:29  lim
 * Set Selection now blanks out screen after search and 
 * Perform Selection will unblank it.
 * 
 * Revision 0.19  91/08/20  15:46:14  lim
 * Get selection now returns a duration of -1 if no edit has been selected.
 * 
 * Revision 0.18  91/08/20  14:46:35  lim
 * Corrected set selection offset.
 * 
 * Revision 0.17  91/08/20  10:01:56  lim
 * HaltSelection() and PauseSelection() now directly call DevStop() and
 * DevStill() instead of Stop() and Pause(). This is because Pause() is 
 * implemented to toggle between Still() and Play() and that is not
 * the behavior we want.
 * 
 * Revision 0.16  91/08/17  20:46:34  lim
 * No longer have to worry about interruptibility. All stops and pauses
 * have interrupt capability.
 * OpenPanel now called Browse.
 * 
 * Revision 0.15  91/08/16  13:03:12  lim
 * Always display 2-digit numbers for hr, sec, min, frame.
 * 
 * Revision 0.14  91/08/16  12:36:52  lim
 * 1. editFilename gone. Read from title bar.
 * 2. Call Browse() with OPENPANEL_CHECK for open doc.
 * 3. Functions will check for validity of command before proceeding.
 * 
 * Revision 0.13  91/08/15  12:59:29  lim
 * Changed FileLoad() call to OpenHandler() calls.
 * 
 * Revision 0.12  91/08/08  16:35:07  lim
 * 1. Added instance pointer to each public videoObject function call.
 * 2. speedStg is now an array of integers.
 * 3. For PauseSelection() and HaltSelection(), interruptible is set 
 *    appropriately so that either Pause() (Stop()) or DevClearMarker()
 *    is called.
 * 
 * Revision 0.11  91/07/20  11:44:46  lim
 * Replaced DevPlayFromTo calls with PlayEdit.
 * Cleaned up code.
 * 
 * Revision 0.10  91/07/16  16:17:35  lim
 * Initial revision.
 *  */

#include <xview/notify.h>
#include "vcrEdit.h"

void OpenDoc();
void SetSelection();
void PerformSelection();
char ** GetDoc();
MAESelection* GetSelection();
void HaltSelection();
void PauseSelection();
void ResumeSelection();
void HideApplication();
void ShowApplication();
IconData* GetAppIcon();

static char *canonFilename;
static char lprcsid[] = "$Header: /Source/Media/collab/vcrEdit/RCS/vcrEditProtocol.c,v 0.35 92/09/29 18:10:43 drapeau Exp $";

Sender*		sender;
Receiver*	receiver;
enum Boolean	preSearch = No;					    /* Flag : whether the device should presearch when */
enum Boolean	lastEditPlayed = No;				    /* Flag : Reflects boundary condition when last... */
								    /* ...edit on the edit list was just played, but... */
								    /* ...pre-search is still on.  If the author wanted... */
								    /* ...to re-play the last edit, whatever function that... */
								    /* ...checks if presearch is set (SetSelection here)... */
								    /* ...should also check this flag.  If this flag is... */
								    /* ..."Yes", it is not possible to pre-search (i.e., the... */
								    /* ...last edit on the list was played).  If this... */
								    /* ...flag is "No", then there remain edits on the list... */
								    /* ...to be played, and so pre-search is still valid. */

void 
  InitNetwork(char* h)
{
  Port senderPort;
  char diagMsg[30];
  
  static  DispatchTable  DT = 
  {
    OpenDoc,
    GetDoc,
    GetSelection,
    SetSelection,
    PerformSelection,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    HaltSelection,
    PauseSelection,
    ResumeSelection,
    HideApplication,
    ShowApplication,
    GetAppIcon
    };
  
  lines = 0;
  clearframe = 0;
  change = 0;
  search = 0;
  editnum = -1;
  senderPort.hostName = h;					    /* Networking stuff */
  senderPort.portNumber = PortMgrPortNumber;
  sender = NewSender(&senderPort);
  if (sender)
  {    
    receiver = NewReceiver(sender, "vcrEdit", receiverPort);
    sprintf(diagMsg, "Listening on port %d.\n", receiver->transport->xp_port);
    PrintDiagnostics(diagMsg);
    
    BuildDispatchTable (&DT);
  }
  canonFilename = (char *) malloc(MAX_PATHLENGTH);
  (void) notify_enable_rpc_svc (TRUE);
}								    /* end function InitNetwork */



/* Open document as directed by TimeLine.
 * Check that the current file is saved first.
 */
void OpenDoc(char** filename)
{
  int result;
  char fullFilename[MAX_PATHLENGTH];
  char* currentFile = (char*) xv_get(vcrEdit_editPopup->editPopup,
				     XV_LABEL);
  if (strcmp(*filename, "Untitled") != 0)			    /* Check that filename is not 'Untitled' */
  {
    if (strlen(currentFile) == 0) strcpy (currentFile, ".");	    /* If no filename specified in panel, set to '.' (directory) */
    
    if (realpath(currentFile, fullFilename) != NULL)
      strcpy(fullFilename, (char*) realpath(currentFile, fullFilename));
    else strcpy(fullFilename, ".");
    if ((strcmp(fullFilename, *filename) != 0) &&		    /* Check if filename given by the TimeLine is... */
	(strcmp(currentFile, *filename) != 0))			    /* ...the same as that on the panel textfield */
    {
      result = NOTICE_NO;					    /* Check if unsaved changes exist in edit list */
      if ((change) && (lines > 0)) 
	result = DisplayChoice("Unsaved changes exist in the video edit list.",
			       "Loading a file will erase all changes. Go ahead and load the file?",
			       "No", "Yes");	
      if (result == NOTICE_NO)
      {
	change = 0;
	Browse(*filename, BrowseCheckOpen, 0, "#VCR Edit Document#", "vcrEdit");
      }
    }
  }  
}								    /* end function OpenDoc */



void SetSelection(MAESelection* selection)			    /* Seek to selection as directed by TimeLine */
{
  char shr[2];
  char smin[2];
  char ssec[2];
  char sframe[2];
  char ehr[2];
  char emin[2];
  char esec[2];
  char eframe[2];
  char diagMsg[30];
  char msg[30];
  char audio[7];
  int starthr, startmin, startsec, startfr, startAddress;
  int endhr, endmin, endsec, endfr;
  
  if (!myVideo)							    /* If no video driver installed, return immediately */
    return;
  if ((preSearch == Yes) && (lastEditPlayed == No))		    /* Was the pre-search option set, and is it safe... */
  {								    /* ...to do pre-search? */
    if (editnum == selection->start - 1)			    /* Presearch is completed. Ignore this SetSelection */
      return;							    /* if it presearched to the correct edit */
  }
  lastEditPlayed = No;						    /* Reset flag saying that last edit hasn't been played */
  if (selection->start > 0 && selection->start <= lines)	   
  {
    sprintf(diagMsg, "In SetSelection, offset = %d ms\n",
	    selection->offset);
    PrintDiagnostics(diagMsg);
    if (editnum != -1)						    /* Deselect any current selection */
      xv_set(vcrEdit_editPopup->editScrollList,
	     PANEL_LIST_SELECT, editnum, FALSE,
	     NULL);
    editnum = selection->start - 1;
    sscanf(startframe[editnum], "%d:%d:%d:%d",			    /* Get original start */
	   &starthr, &startmin, &startsec, &startfr);
    
    startAddress = ConvertToAddress(starthr, startmin, startsec, startfr) + /* Add offset to start address */
      (selection->offset) / 1000 * FrameRate;
    ConvertToTime(&starthr, &startmin, &startsec, &startfr, startAddress);
    sprintf(shr, "%.2d", starthr);
    sprintf(smin, "%.2d", startmin);
    sprintf(ssec, "%.2d", startsec);
    sprintf(sframe, "%.2d", startfr);
    
    xv_set(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE, shr, NULL); /* Load the appropriate values into the textfields */
    xv_set(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE, smin, NULL);
    xv_set(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE, ssec, NULL);
    xv_set(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE, sframe, NULL);
    
    sscanf(endframe[editnum], "%d:%d:%d:%d", 
	   &endhr, &endmin, &endsec, &endfr);
    sprintf(ehr, "%.2d", endhr);
    sprintf(emin, "%.2d", endmin);
    sprintf(esec, "%.2d", endsec);
    sprintf(eframe, "%.2d", endfr);
    xv_set(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE, ehr, NULL);
    xv_set(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE, emin, NULL);
    xv_set(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE, esec, NULL);
    xv_set(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE, eframe, NULL);
    
    xv_set(vcrEdit_editPopup->editLabelTxt, PANEL_VALUE, 
	   label[editnum], NULL);				    /* label */
    xv_set(vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE,
	   (int) speedStg[editnum], NULL);			    /* speed */
    switch (audioStg[editnum])					    
    {
     case Stereo:
      strcpy(audio, "Stereo");
      break;
     case Right:
      strcpy(audio, "Right");
      break;
     case Left:
      strcpy(audio, "Left");
      break;
     case Mute:
      strcpy(audio, "Mute");
      break;
    }
    xv_set(vcrEdit_editPopup->editAudioMenuButton, 
	   PANEL_LABEL_STRING, audio, NULL);			    /* audio */
    SetDuration();						    /* duration */
    sprintf(msg, "Current Selection : Edit #%d", editnum+1);
    xv_set(vcrEdit_editPopup->editNewEditMsg, PANEL_VALUE, msg, NULL);    
    xv_set(vcrEdit_editPopup->editModButton,			    /* Make the modify, delete buttons active */
	   PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_editPopup->editDelButton,
	   PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum, TRUE,
	   NULL); 
    PlayEdit(startAddress, 0);					    /* Seek to selection first (block for completion of search) */
    DevSetVideo(myVideo, 0);					    /* Video muted */
  }
}								    /* end function SetSelection */


void DevResumeSelection(int end)
{    
  PlayEdit(NULL, end);
}



void PreSearch(Notify_client client, int which)
{
  static MAESelection select;
  
  notify_set_itimer_func(vcrEdit_window1->window1,		    /* Switch off timer */
			 NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
  select.start = editnum + 2;
  select.offset = 0;
  SetSelection(&select);					    /* Do presearch */
}  



/* Play selection */
void PerformSelection()
{
  int end; 
  int endhr; 
  int endmin; 
  int endsec; 
  int endfr;
  int start; 
  int starthr; 
  int startmin; 
  int startsec; 
  int startfr;
  static struct itimerval timer;
  
  if ((editnum == -1)						    /* Nothing selected or nothing in edit list or no device */
      || (lines == 0)
      || (myVideo == (VideoObject*)NULL))
    return;
  
  DevSetVideo(myVideo, 1);					    /* Video mute off */
  
  endfr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE));
  endhr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE));
  endmin = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE));
  endsec = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE));
  end = ConvertToAddress(endhr, endmin, endsec, endfr);
  
  starthr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE));
  startmin = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE));
  startsec = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE));
  startfr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE));
  start = ConvertToAddress(starthr, startmin, startsec, startfr);
  
  PlayEdit(0, end);
  if (preSearch)						    /* Is the pre-search option set? */
  {								    /* Yes, try to find the next edit */
    if (editnum + 1 <= lines)					    /* Does the edit list contain an edit after this one? */
    {								    /* Yes, set a timer to queue it up after this edit... */
      timer.it_value.tv_sec = (end - start) / FrameRate;	    /* ...is done playing */
      notify_set_itimer_func((Notify_client) vcrEdit_window1->window1, 
			     (Notify_func) PreSearch, ITIMER_REAL, &timer, 
			     (struct itimerval*) NULL);
    }
    else							    /* No, the last edit on the list was played, so... */
      lastEditPlayed = Yes;					    /* ...pre-search is temporarily invalid. */
  }
  return;
}								    /* end function PerformSelection */


/* Send name of edit list file to the TimeLine */
char** GetDoc(void* unusedArg)
{
  static char* doc;
  
  doc = (char*) xv_get(vcrEdit_editPopup->editPopup, XV_LABEL);
  if (strlen(doc) == 0 || realpath(doc, canonFilename) == NULL) 
    strcpy(canonFilename, "Untitled");
  else strcpy(canonFilename, realpath(doc, canonFilename));
  return(&canonFilename);
}								    /* end function GetDoc */



/* Send selection data to TimeLine */
MAESelection* GetSelection(void* unusedArg)
{
  static MAESelection select;
  int startmin;
  int starthr;
  int startsec;
  int startfr;
  int endmin;
  int endhr;
  int endsec;
  int endfr;
  int duration;
  int speedInDevice;
  
  if ((editnum == -1) || (!lines))
  {				    
    select.duration = -1;
    return &select;
  }
  
  sscanf(startframe[editnum], "%d:%d:%d:%d", &starthr, &startmin, &startsec, &startfr);
  sscanf(endframe[editnum], "%d:%d:%d:%d", &endhr, &endmin, &endsec, &endfr);
  
  if (myVideo)
    speedInDevice = (int) DevCalcSpeed(myVideo, speedStg[editnum], 1);    
  else 
    speedInDevice = speedStg[editnum];
  
  duration = (ConvertToAddress(endhr, endmin, endsec, endfr) 
	      - ConvertToAddress(starthr, startmin, startsec, startfr));
  
  if (speedInDevice)
    duration = (((duration%speedInDevice)/speedInDevice > 0.5) ? 
		(duration/speedInDevice) + 1 
		: duration/speedInDevice);
  else
    duration = -1;
  
  select.start =  editnum + 1;
  select.end = editnum + 1;
  select.offset = 0;
  select.duration = 1000 * duration;  
  if (strlen(label[editnum]) != 0)
    strcpy(select.label, label[editnum]);   
  else
    strcpy(select.label, "NO LABEL");
  
  return (&select);
}								    /* end function GetSelection */


/* Halts playback of Selection as directed by TimeLine */
void HaltSelection()
{
  if ((editnum != -1) && (lines != 0) && (myVideo != (VideoObject*)NULL))
  {
    notify_set_itimer_func((Notify_client) vcrEdit_window1->window1, /* Turn off timer */
			   NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
    DevStop(myVideo);
  }
  return;
}								    /* end function HaltSelection */


/* Pauses playback of Selection as directed by TimeLine */
void PauseSelection()
{
  if ((editnum != -1) && (lines != 0) &&
      (myVideo != (VideoObject*)NULL))
  {
    notify_set_itimer_func((Notify_client) vcrEdit_window1->window1, /* Turn off timer */
			   NOTIFY_FUNC_NULL, ITIMER_REAL,
			   NULL, NULL);
    DevStill(myVideo);
  }
  return;
}								    /* end function PauseSelection */


/* Resumes playback of Selection as directed by TimeLine */
void ResumeSelection()
{
  int end;
  int endhr;
  int endmin;
  int endsec;
  int endfr;
  
  if ((editnum != -1) && lines && myVideo)
  {
    sscanf(endframe[editnum], "%d:%d:%d:%d", &endhr, &endmin, &endsec, &endfr);
    end = ConvertToAddress(endhr, endmin, endsec, endfr);
    DevResumeSelection(end);
  }
}								    /* end function ResumeSelection */


/* Iconifies application as directed by TimeLine */
void HideApplication()
{
  xv_set(vcrEdit_window1->window1, FRAME_CLOSED, TRUE, NULL);
  XFlush((Display*)xv_get(vcrEdit_window1->window1, XV_DISPLAY));
}								    /* end function HideApplication */


/* Deiconifies application as directed by TimeLine */
void ShowApplication()
{
  xv_set(vcrEdit_window1->window1, FRAME_CLOSED, FALSE, NULL);
  xv_set(vcrEdit_window1->window1, XV_SHOW, TRUE, NULL);		   
  if (xv_get(vcrEdit_optionsPopup->optionsPopup, XV_SHOW) == TRUE)  /* Show all popups that were up originally */
  {
    xv_set(vcrEdit_optionsPopup->optionsPopup, XV_SHOW, TRUE, NULL);
  }
  if (xv_get(vcrEdit_previewPopup->previewPopup, XV_SHOW) == TRUE) 
  {
    xv_set(vcrEdit_previewPopup->previewPopup, XV_SHOW, TRUE, NULL);
  }
  if (xv_get(vcrEdit_editPopup->editPopup, XV_SHOW) == TRUE) 
  {
    xv_set(vcrEdit_editPopup->editPopup, XV_SHOW, TRUE, NULL);
  }  
  if (xv_get( vcrEdit_infoPopup->infoPopup, XV_SHOW) == TRUE) 
  {
    xv_set(vcrEdit_infoPopup->infoPopup, XV_SHOW, TRUE, NULL);
  }
  XFlush((Display*) xv_get(vcrEdit_window1->window1, XV_DISPLAY));    
  return;
}								    /* end function ShowApplication */
    

IconData* GetAppIcon()
{
  static IconData returnVal;
  static unsigned short baseWindow_bits[] = {
#include "icons/vcrEditIcon"
        };
  
  returnVal.iconData = (char *)malloc(sizeof(baseWindow_bits));
  bcopy(baseWindow_bits, returnVal.iconData, sizeof(baseWindow_bits));
  if (returnVal.iconData)
    returnVal.dataLength = sizeof(baseWindow_bits);
  return(&returnVal);
}								    /* end function GetAppIcon */
