/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/cdEdit/RCS/misc.c,v 2.10 92/07/17 16:33:48 drapeau Exp $ */
/* $Log:	misc.c,v $
 * Revision 2.10  92/07/17  16:33:48  drapeau
 * Minor change to ConvertRelativeToAbsolute(), made overflow code
 * more robust (previous code assumed overflow would be less than
 * one second; new code can handle overflow of any number of
 * seconds).
 * 
 * Revision 2.0  91/10/06  21:01:45  chua
 * Update to version 2.0
 * 
 * Revision 1.23  91/09/24  16:32:58  chua
 * The duration array stores time in milliseconds instead of seconds.  So, in line
 * 90, 97, do the appropriate conversions for the duration.
 * 
 * Revision 1.22  91/09/18  17:27:00  chua
 * Do proper casting of assignment statements to be ANSI-compliant.
 * 
 * Revision 1.21  91/09/03  15:19:10  chua
 * Added the copyright header.
 * 
 * In the SetDuration function, take a parameter, whichEdit, so that the duration for that
 * edit will be properly stored in the duration array.
 * 
 * All constants, which were previously all capitalized, are now only capitalized in the
 * first letter.
 * 
 * Deleted the DiscNotEjected function, as there is a variable, DiscInPlayer, which keeps
 * track of whether there is a disc currently in the player.
 * 
 * Replaced the calls to notice_prompt by a call to AlertMessage (to make things shorter
 * and tidier).
 * 
 * Revision 1.2  91/07/10  11:04:09  chua
 * Removed a debugging printf statement.
 * 
 * Revision 1.1  91/07/09  16:47:51  chua
 * 
 * 
 * Revision 1.0  91/07/08  13:46:16  chua
 * Initial revision
 *  */

static char miscrcsid[] = "$Header: /Source/Media/collab/cdEdit/RCS/misc.c,v 2.10 92/07/17 16:33:48 drapeau Exp $";

#include "main.h"

int fd;								    /* file descriptor to the port */
Toc toc;							    /* table of contents */
static int discData;

/*
 * This function displays the approximate duration of the current selection 
 */
int SetDuration(whichEdit)
     int whichEdit;
{
  int calculatedDuration;
  Msf absolutestart, absoluteend;
  Msf difference;
  char buf[80];

  absolutestart = (Msf) GetCurrentStart();			    /* Convert to absolute values first */
  if (absolutestart == NULL) 
  {
    return Error;
  }
  absoluteend = (Msf) GetCurrentEnd();
  if (absoluteend == NULL) 
  {
    return Error;
  }
  if (CheckSelection(absolutestart, absoluteend) == Error)	    /* Check if the end time is greater than the start time */
  {
    return Error;
  }
  else								    /* Calculate the approximate duration */
  {
    difference = diff_msf(absoluteend, absolutestart);
    calculatedDuration = (difference->min * 60 * 1000) + 
		(difference->sec * 1000) + (difference->frame * 1000 /75);
    if (whichEdit != -1) 
    {
      duration[whichEdit] = calculatedDuration;
    }
  }
  sprintf(buf, "%d", (calculatedDuration + 500) / 1000);
  xv_set(cdEdit_EditPopup->EditPopupDurationText, PANEL_VALUE, buf, NULL); /* Display the duration in the duration textfield */
  return OK;
}

/*
 * This function will read the table of contents off the CD if possible.  The table of contents is read into the appropriate data structure.
 * This code is obtained from the cdrom driver provided by Sun Microsystems.
 */
Toc ReadDiscToc()
{
  struct  cdrom_tochdr    hdr;
  struct  cdrom_tocentry  entry;
  int     firstTrack;
  int     lastTrack;
  int     i;
  Toc     toc;
  Msf     msf, tempMsf;
  int     control;
  
  if (cdrom_read_tochdr(fd, &hdr) == 0)
  {
    return (Toc)NULL;
  }
  firstTrack = hdr.cdth_trk0;
  lastTrack = hdr.cdth_trk1;
  
  entry.cdte_format = CDROM_MSF;
  entry.cdte_track = CDROM_LEADOUT;
  cdrom_read_tocentry(fd, &entry);
  msf = init_msf();
  msf->min = entry.cdte_addr.msf.minute;
  msf->sec = entry.cdte_addr.msf.second;
  msf->frame = entry.cdte_addr.msf.frame;
  
  toc = init_toc(firstTrack, lastTrack, msf);
  
  tempMsf = init_msf();
  entry.cdte_format = CDROM_MSF;
  discData = 0;		
  for (i=firstTrack; i<=lastTrack; i++) 
  {
    entry.cdte_track = i;
    cdrom_read_tocentry(fd, &entry);
    msf = init_msf();
    msf->min = entry.cdte_addr.msf.minute;
    msf->sec = entry.cdte_addr.msf.second;
    msf->frame = entry.cdte_addr.msf.frame;
    control = entry.cdte_ctrl;
    if (control & CDROM_DATA_TRACK) 
    {
      discData = 1;
    }
    entry.cdte_track = i + 1;
    if (entry.cdte_track > (unsigned char) lastTrack) 
    {
      entry.cdte_track = CDROM_LEADOUT;
    }
    entry.cdte_format = CDROM_MSF;
    cdrom_read_tocentry(fd, &entry);
    tempMsf->min = entry.cdte_addr.msf.minute;
    tempMsf->sec = entry.cdte_addr.msf.second;
    tempMsf->frame = entry.cdte_addr.msf.frame;
    add_track_entry(toc, i, control,
		    msf, tempMsf);
  }
  return (toc);
}

/*
 * Get the current track that is currently playing on the disc.
 */
int GetCurrentTrack()
{
  int currentTrack, dummy1, dummy2, dummy3;
  
  if (playerState == StopMode) 
  {
    currentTrack = 0;
  }
  else 
  {
    cdrom_get_relinfo(fd, &currentTrack, &dummy1, &dummy2, &dummy3);
  }
  return (currentTrack);
}

/*
 * Compare two msf values.  Return Equal if they are the same, Greater if the first is 
 * greater than the second, and Less if the second is greater than the first 
 */
int CompareMsf(first, second)
     Msf first, second;
{
  if (first->min == second->min && first->sec == second->sec &&
      first->frame == second->frame) 
  {
    return Equal;
  }
  if ((first->min > second->min) || 
      (first->min == second->min && first->sec > second->sec) ||
      (first->min == second->min && first->sec == second->sec 
       && first->frame > second->frame)) 
  {
    return Greater;
  }
  return Less;
}

/* 
 * This function checks if a selection (given by start and end times) is valid.  It first checks that each time is valid (falls within the total time
 * of the disc.  It then checks to see that the end time is greater than the start time.
 */
int CheckSelection(start, end)
     Msf start, end;
{
  Msf zeroMsf;

  if (start == NULL || end == NULL) 
  {
    return Error;
  }
  zeroMsf = (Msf) malloc(sizeof(struct msf));
  zeroMsf->min = 0;
  zeroMsf->sec = 0;
  zeroMsf->frame = 0;
  
  if (CompareMsf(start, zeroMsf) == Equal ||			    /* Check that the times are valid (falls within the disc times */
      CompareMsf(end, zeroMsf) == Equal || 
      CompareMsf(start, toc->total_msf) == Greater || 
      CompareMsf(end, toc->total_msf) == Greater) 
  { 
    AlertMessage("Invalid time specifed in current selection.", NULL);
    free(zeroMsf);
    return Error;
  }
  free(zeroMsf);
  if (CompareMsf(start, end) == Greater)			    /* Check that the end time is greater than the start time */
  {
    AlertMessage("Invalid current selection:",
		 "Start time is greater than end time.");
    return Error;
  }
  return OK;
}

/*
 * This function converts a relative msf to absolute msf. 
 */
Msf ConvertRelativeToAbsolute(track, msf)
     int track;
     Msf msf;
{
  Msf startMsf, resultMsf;
  
  if ((msf->min == 0 && msf->sec == 0 && msf->frame == 0) ||	    /* Check that the time is valid */
      (track < startTrack || track > endTrack)) 
  {
    AlertMessage("Illegal time specified in current selection.", NULL);
    return NULL;
  }
  resultMsf = (Msf) malloc(sizeof(struct msf));			    /* Initialize the absolute msf */
  resultMsf->min = 0;
  resultMsf->sec = 0;
  resultMsf->frame = 0;
  startMsf = get_track_msf(toc, track);
  
  resultMsf->frame = msf->frame + startMsf->frame;		    /* Perform the conversion */
  while (resultMsf->frame >= 75) 
  {
    resultMsf->frame -= 75;
    resultMsf->sec ++;
  }
  resultMsf->sec += msf->sec + startMsf->sec;
  while (resultMsf->sec >= 60) 
  {
    resultMsf->sec -= 60;
    resultMsf->min ++;
  }
  resultMsf->min += msf->min + startMsf->min;
  if (CompareMsf(resultMsf, toc->total_msf) == Greater)		    /* Check that the absolute msf does not exceed the total time on the disc */
  {
    AlertMessage("Illegal time specified in current selection.", NULL);
    return NULL;
  }
  return (resultMsf);
}

/*
 * Get the current start selection from the current selection textfields and return it as a msf structure.
 */
Msf GetCurrentStart()
{
  int track;
  Msf relative, result;
  
  track = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupStartTrackText, PANEL_VALUE));
  relative = (Msf) malloc(sizeof(struct msf));
  relative->min = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupStartMinText, PANEL_VALUE));
  relative->sec = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupStartSecText, PANEL_VALUE));
  relative->frame = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupStartFrameText, PANEL_VALUE));
  result = ConvertRelativeToAbsolute(track, relative);
  free(relative);
  return(result);
}

/*
 * Get the current end selection from the current selection textfields and return it as a msf structure.
 */
Msf GetCurrentEnd()
{
  int track;
  Msf relative, result;
  
  track = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupEndTrackText, PANEL_VALUE));
  relative = (Msf) malloc(sizeof(struct msf));
  relative->min = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupEndMinText, PANEL_VALUE));
  relative->sec = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupEndSecText, PANEL_VALUE));
  relative->frame = (int) atoi(xv_get(cdEdit_EditPopup->EditPopupEndFrameText, PANEL_VALUE));
  result = ConvertRelativeToAbsolute(track, relative);
  free(relative);
  return (result);
}

/*
 * Get the start and end times of the current selection and place them in the edit list entry given by the parameter num.
 */
void GetStartEnd(num)
     int num;
{
  starttrack[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupStartTrackText, PANEL_VALUE)); /* Get the start time */
  startmin[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupStartMinText, PANEL_VALUE));
  startsec[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupStartSecText, PANEL_VALUE));
  startframe[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupStartFrameText, PANEL_VALUE));

  endtrack[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupEndTrackText, PANEL_VALUE)); /* Get the end time */
  endmin[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupEndMinText, PANEL_VALUE));
  endsec[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupEndSecText, PANEL_VALUE));
  endframe[num] = (int) atoi(xv_get (cdEdit_EditPopup->EditPopupEndFrameText, PANEL_VALUE));

  strcpy (label[num], (char *) xv_get (cdEdit_EditPopup->EditPopupLabelText, /* Get the label */
				       PANEL_VALUE));

  volume[num] = xv_get(cdEdit_window1->VolumeSlider, PANEL_VALUE);
  balance[num] = xv_get(cdEdit_window1->BalanceSlider, PANEL_VALUE);
}

/*
 * Load the start and end times of the currently selected entry in the edit list onto the current selection textfields.
 */
void LoadStartEnd()
{
  char e[10], t[2], m[2], s[2], f[2];
  
  sprintf(e, "Current Selection : %d", editnum+1);		    /* Update the edit status information */
  xv_set(cdEdit_EditPopup->CurrentSelectionMsg, PANEL_LABEL_STRING, e, NULL);
  
  sprintf(t, "%d", starttrack[editnum]);			    /* Load the start and end times onto the current selection textfields */
  sprintf(m, "%d", startmin[editnum]);
  sprintf(s, "%d", startsec[editnum]);
  sprintf(f, "%d", startframe[editnum]);
  xv_set(cdEdit_EditPopup->EditPopupStartTrackText, PANEL_VALUE,
	 t, NULL);
  xv_set(cdEdit_EditPopup->EditPopupStartMinText, PANEL_VALUE,
	 m, NULL);
  xv_set(cdEdit_EditPopup->EditPopupStartSecText, PANEL_VALUE,
	 s, NULL);
  xv_set(cdEdit_EditPopup->EditPopupStartFrameText, PANEL_VALUE,
	 f, NULL);
  sprintf(t, "%d", endtrack[editnum]);
  sprintf(m, "%d", endmin[editnum]);
  sprintf(s, "%d", endsec[editnum]);
  sprintf(f, "%d", endframe[editnum]);
  xv_set(cdEdit_EditPopup->EditPopupEndTrackText, PANEL_VALUE,
	 t, NULL);
  xv_set(cdEdit_EditPopup->EditPopupEndMinText, PANEL_VALUE,
	 m, NULL);
  xv_set(cdEdit_EditPopup->EditPopupEndSecText, PANEL_VALUE,
	 s, NULL);
  xv_set(cdEdit_EditPopup->EditPopupEndFrameText, PANEL_VALUE,
	 f, NULL);
  xv_set(cdEdit_EditPopup->EditPopupLabelText, PANEL_VALUE, label[editnum], /* Load the label */
	 NULL);

  if (discInPlayer == 1) 
  {
    xv_set(cdEdit_EditPopup->ModifyButton, PANEL_INACTIVE, FALSE, NULL); /* Make the modify and delete buttons active */
    xv_set(cdEdit_EditPopup->DeleteButton, PANEL_INACTIVE, FALSE, NULL); /* only if there is a disc in the player */
  }
  xv_set(cdEdit_window1->VolumeSlider, PANEL_VALUE, volume[editnum], NULL);
  xv_set(cdEdit_window1->BalanceSlider, PANEL_VALUE, balance[editnum], NULL);
  SetVolume(volume[editnum], balance[editnum]);
  
}

/*
 * Notify callback function for `GetStartButton'.
 * Loads the current playing time on the disc to the start time textfield in the current selection area.
 */
void GetStart(item, event)
     Panel_item	item;
     Event		*event;
{
  int track, min, sec, frame;
  char t[2], m[2], s[2], f[2];
  
  cdrom_get_relinfo(fd, &track, &min, &sec, &frame);		    /* Get the current playing time */
  sprintf (t, "%d", track);					    /* Set the values in the start time textfield */
  sprintf (m, "%d", min);
  sprintf (s, "%d", sec);
  sprintf (f, "%d", frame);
  xv_set(cdEdit_EditPopup->EditPopupStartTrackText, PANEL_VALUE, t,  NULL);
  xv_set(cdEdit_EditPopup->EditPopupStartMinText, PANEL_VALUE, m,  NULL);
  xv_set(cdEdit_EditPopup->EditPopupStartSecText, PANEL_VALUE, s,  NULL);
  xv_set(cdEdit_EditPopup->EditPopupStartFrameText, PANEL_VALUE, f,  NULL);
}

/*
 * Notify callback function for `GetEndButton'.
 * Loads the current playing time on the disc to the end time textfield in the current selection area.
 */
void GetEnd(item, event)
     Panel_item	item;
     Event		*event;
{
  int track, min, sec, frame;
  char t[2], m[2], s[2], f[2];
  
  cdrom_get_relinfo(fd, &track, &min, &sec, &frame);	    /* Get the current playing time */
  sprintf (t, "%d", track);				    /* Set the values in the end time textfield */
  sprintf (m, "%d", min);
  sprintf (s, "%d", sec);
  sprintf (f, "%d", frame);
  xv_set(cdEdit_EditPopup->EditPopupEndTrackText, PANEL_VALUE, t,  NULL);
  xv_set(cdEdit_EditPopup->EditPopupEndMinText, PANEL_VALUE, m,  NULL);
  xv_set(cdEdit_EditPopup->EditPopupEndSecText, PANEL_VALUE, s,  NULL);
  xv_set(cdEdit_EditPopup->EditPopupEndFrameText, PANEL_VALUE, f,  NULL);
}
