/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/file.c,v 1.15 92/10/30 16:21:27 drapeau Exp $ */
/* $Log:	file.c,v $
 * Revision 1.15  92/10/30  16:21:27  drapeau
 * Fixed an error in the OpenHandler() function.  Simple, silly, common C
 * mistake: used "=" in the test of an if statement instead of the proper
 * "==", when testing for whether synchronization hints are available.
 * 
 * Revision 1.14  92/10/02  15:00:32  drapeau
 * Minor change made to OpenHandler().  It now corrects an error that failed to
 * reset the playback head to the beginning of a newly-opened document.
 * 
 * Revision 1.13  92/10/01  14:43:19  drapeau
 * Modified OpenHandler() and SaveHandler() to read and write additional
 * synchronization hints information.  If a TimeLine document does not
 * have any synchronization hints available, then when it is saved a new
 * field will be added to the TimeLine document indicating that there is no
 * synchronization information available; if the hints are available,
 * SaveHandler() will indicate the presence of these hints.
 * The OpenHandler() routine will look for this field in a TimeLine document
 * if it is a Version 2 document, and will then determine if the document
 * has sync hints available for it.
 * 
 * Revision 1.12  92/09/24  17:03:21  drapeau
 * The OpenHandler() and SaveHandler() functions were modified to support
 * documents with synchronization information, as well as older style
 * documents that do not contain "sync hints".  The new document style differs
 * from the old in two ways:
 * 1) The header is slightly different (new header is
 *    "#TimeLine Edit Document, Version 2#");
 * 2) There is an extra line of information for each note.  This extra line
 *    denotes an estimate of the time (in milliseconds) it will take to
 *    prepare the note for performance.
 * Both handlers had to be modified to read and write this additional
 * information.  In addition, when OpenHandler() reads a new-style document with
 * sync hints, it sets a flag for the current document indication that sync
 * hints are available for use.
 * Other changes are cosmetic, to improve readability and ANSI compliance.
 * Also, code that makes reference to the TimeLine document header (e.g., calls
 * to the Browse functions) have been changed to accommodate the new document
 * format.
 * 
 * Revision 1.11  92/09/12  19:17:11  drapeau
 * Slight modification to make code more portable: the order of include files
 * was changed to help insure that typedefs like "dev_t" are picked up before
 * the include file "<sys/stat.h>" is loaded.  To insure this, "main.h" is now
 * included before "<sys/stat.h>".
 * 
 * Revision 1.10  92/05/29  14:41:42  drapeau
 * Modified code to track new name of the MAEstro "Selection" structure;
 * it is now named "MAESelection".
 * 
 * Revision 1.0  91/09/30  16:56:15  chua
 * Update to version 1.0
 * 
 * Deleted an unused variable (found) in OpenHandler.
 * 
 * Both FreeInstrumentList and ClearAllNotes take tlFrame as the parameter, instead of
 * tlFrame->instHead.
 * 
 * Revision 0.85  91/09/25  13:48:01  chua
 * In CloseHandler, if there are multiple TimeLine frames open, destroy the current frame.
 * Else, just close the document and reset the filename to 'untitled'.
 * 
 * Revision 0.84  91/09/23  17:10:26  chua
 * Change the selectedInstrument element to noteInstrument, as the latter is
 * now used to indicate which instrument has the currently selected note.
 * 
 * Revision 0.83  91/09/19  17:28:46  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.82  91/09/17  18:10:12  chua
 * In SaveHandler, correct an error when counting the number of applications
 * that have notes.
 * 
 * Revision 0.81  91/09/17  17:15:30  chua
 * In SaveHandler, when writing the names of the applications, check to see that the
 * apps have notes first.
 * 
 * Revision 0.80  91/09/17  17:13:14  chua
 * Changed the name of the structure Region to tlRegion, to avoid clashes with 
 * definitions in some X library.
 * 
 * In OpenHandler, first read the applications required by the file and create an
 * instrument list using these applications.  Call OpenAppsInitialize to assign port
 *  numbers to the applications (if they are open).
 * 
 * In SaveHandler, only save the applications which have notes in them.
 * 
 * Revision 0.79  91/09/16  14:41:17  chua
 * Include the Browse.h file.
 * In calls to InstrumentNew, add a new parameter at the end.  This parameter indicates the
 * application name (used for applications which are not currently open).  
 * 
 * Revision 0.78  91/09/09  14:56:39  chua
 * Removed the struct stat stbuf declaration in line 321.
 * 
 * Revision 0.77  91/09/06  17:31:58  chua
 * In CheckforUntitled, cast rindex to char * (first line).
 * 
 * Revision 0.76  91/09/04  15:09:11  chua
 * Replaced the calls to notice_prompt with a call to AlertMessage.
 * 
 * Revision 0.75  91/08/26  14:21:16  chua
 * When saving a file, keep track of all the applications that are currently open,
 * instead of just saving those applications that have notes in them.
 * 
 * Revision 0.74  91/08/22  15:58:58  chua
 * Made some changes to both OpenHandler and SaveHandler when checking if the file being
 * opened is a TimeLine Edit file.
 * 
 * Revision 0.73  91/08/22  15:33:49  chua
 * Shortened the dividers ("****....**") in the document format.
 * 
 * Revision 0.72  91/08/22  15:01:26  chua
 * Deleted the FileError routine as error checking is now performed by the Browser.
 * 
 * Changed the TimeLine document format so that there are now more spaces and markings
 * to clearly denote different regions.
 * 
 * Revision 0.71  91/08/16  16:58:11  chua
 * Removed some extra variables which are not used.  Also, in the calls to Browse,
 * replace OPENPANEL_OPEN and OPENPANEL_SAVE by BrowseOpen and BrowseSave respectively.
 * 
 * In the file format, add spacing between information for region, pause and notes.
 * 
 * Revision 0.70  91/08/12  16:12:29  chua
 * Changed the comment heading.
 * 
 * Revision 0.69  91/08/12  16:08:20  chua
 * *** empty log message ***
 * 
 * Revision 0.68  91/08/12  16:02:48  chua
 * Added two new routines, SaveAsFileHandler and CloseHandler.
 * The SaveAsFileHandler will bring up the open/save panel for 
 * the user to select or type a filename.
 * The CloseHandler will close the current file and reset the
 * filename to 'untitled'. The canvas will also be cleared.
 *
 * In the SaveHandler, the open/save panel will not be brought up
 * unless the filename is 'untitled'.
 * 
 * Revision 0.67  91/08/09  15:12:32  chua
 * Changed the name of InfoHandler to AboutHandler.
 * 
 * Revision 0.66  91/08/08  14:21:35  chua
 * In the OpenFileHandler and SaveFileHandler, the call to Browse takes different parameters.
 * 
 * In OpenHandler, if an application in the file is not currently open, instead of just
 * printing an error message and not allowing the user to open the file, create an instrument
 * node for the application and display it as well.  However, make the instrument
 * permanently muted.
 * 
 * Revision 0.65  91/08/05  16:53:04  chua
 * Deleted the RepaintCanvas routine, as it is no longer necessary.  In places where it
 * is called, just call the ScrollToFirstQuarter routine, which will do the necessary
 * repaint as well.
 * 
 * Revision 0.64  91/07/30  11:21:25  chua
 * Incorporated the file browser.  The menu handlers for the open and save buttons
 * will call the Browse function in the Browse code.  This browse function will
 * then call either the OpenHandler or SaveHandler which will return a 0 if it
 * likes the pathname given by the browser, or 1 otherwise.
 * 
 * Revision 0.63  91/07/26  17:24:22  chua
 * In the OpenHandler and SaveHandler, where there is a 'return NULL', do a 
 * 'return item' instead.  Apparently, returning a NULL will cause Errors.
 * 
 * Also include code to write and read information about annotated regions in the
 * file.
 * 
 * Revision 0.62  91/07/24  10:33:11  chua
 * In the OpenHandler and SaveHandler, include code for saving the pause marker information.
 * 
 * 
 * Revision 0.61  91/07/17  16:31:26  chua
 * Added two new functions, InfoHandler and CloseInfoPopup to open and close the
 * Info popup window respectively.
 * 
 * Revision 0.60  91/07/09  16:57:48  chua
 * In the QuitHandler procedure, move most of the code into the QuitNotify procedure
 * in TimeLine.c
 * 
 * Revision 0.59  91/06/26  16:53:30  chua
 * In the Error message which is printed in OpenHandler when one of the applications are not
 * open, just add the line "The file cannot be opened".
 * 
 * Revision 0.58  91/06/25  17:15:05  chua
 * Reformatted a comment line.
 * 
 * Revision 0.57  91/06/04  17:37:14  chua
 * Added the copyright comments in the beginning of the file.
 * 
 * Revision 0.56  91/06/04  17:25:00  chua
 * In the QuitHandler, if it is the Clipboard, do not destroy the frame, but simply make
 * it not visible.
 * 
 * Destroy the clipboard if all Document windows have been destroyed.
 * 
 * Revision 0.55  91/06/04  10:42:22  chua
 * Added a call to UpdateHeader whenever there is a change in the status of
 * the current document (the change flag set to 1).
 * 
 * Revision 0.54  91/06/03  11:11:50  chua
 * Make changes to accomodate multiple documents.  This involves identifying
 * which is the current active window, that is, the one where the last mouse
 * click was done.
 * 
 * Revision 0.53  91/05/30  12:07:44  chua
 * Added an extra parameter in the call to InitNotesInfo.  The second parameter,
 * deselect, indicates if the currently selected note is to be deselected.
 * 
 * Revision 0.52  91/05/29  14:31:40  chua
 * In the LoadHandler, it is not necessary to call the ClearNoteInfoList function in lines
 * 236-241.
 * 
 * Revision 0.51  91/05/28  12:09:40  chua
 * *** empty log message ***
 * 
 * Revision 0.50  91/05/24  16:36:38  chua
 * *** empty log message ***
 * 
 * Revision 0.49  91/05/23  17:38:53  chua
 * *** empty log message ***
 * 
 * Revision 0.48  91/05/22  16:40:04  chua
 * 
 * 
 * Revision 0.47  91/05/22  13:55:56  chua
 * 
 * 
 * Revision 0.46  91/05/22  11:38:21  chua
 * In the procedure TextfieldLoad, the parameters passed in the call to OpenHandler are
 * tlFrame->changed to NULL, MENU_NOTIFY, so that pressing the return key after typing in the filename
 * will activate the loading procedure.  The parameters were NULL, NULL before, because the
 * Open procedure used to be a button notify procedure instead of a menu button handler.
 * 
 * Revision 0.45  91/05/17  16:57:47  chua
 * *** empty log message ***
 * 
 * Revision 0.44  91/05/17  16:56:51  chua
 * *** empty log message ***
 * 
 * Revision 0.43  91/05/16  15:19:34  chua
 * Changed the names of Load and Save to OpenHandler and SaveHandler (since they are now menu button
 * handlers).
 * 
 * Also added a new handler QuitHandler to this file (from TimeLine.c)
 * 
 * Revision 0.42  1991/04/24  00:57:45  chua
 * The only tlFrame->change made here is in the Load procedure. There is no longer any need to check if
 * (instrument->instInfo != NULL), since instInfo is now never NULL as the pop-up window will
 * always be created whenever an instrument is created.
 *
 * Revision 0.41  1991/04/08  20:55:35  chua
 * TlFrame->Changes are made in the Load procedure.  Before the loading of a file is done, the panel lists of all the
 * pop-up windows that have been created are cleared.  After loading is done, the panel lists are then updated
 * with the new notes information for each instrument.  This is done by calling the InitNotesInfo procedure
 * in notesInfo.c for each instrument where the pop-up window has already been created.
 *
 * Revision 0.40  1991/04/01  01:46:54  chua
 * This file contains the functions for loading and saving a TimeLine document (Load and Save).  There are also two
 * smaller functions to check for Errors when opening or saving a file (FileError and CheckforUntitled).
 * The text notify procedure for the filename textfield calls the Load procedure (notify procedure for the Load
 * button).  That is, if the user types in a filename and hits return, it is equivalent to pressing the Load button
 * after typing in the filename.
 * */

static char filercsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/file.c,v 1.15 92/10/30 16:21:27 drapeau Exp $";

#include "main.h"
#include <errno.h>
#include <sys/stat.h>
#include <Browse.h>

/*
 * Checks if the filename specified by the user is 'untitled'.  This is not allowed as it is the default used by the timeline if no filename
 * has been specified.  A return value of 1 indicates that the filename is 'untitled' and the user should choose another.  Otherwise,
 * a value of 0 is returned.
 * Called by procedures Load and Save, both in file.c
 */
int CheckforUntitled(filename)
     char *filename;
{
  char *tempfile;
    
  tempfile = (char *) rindex(filename, '/');			    /* check if filename is 'untitled'.  If so, ask for another filename */
  if (tempfile != NULL) 
  {
    if (strcmp(&tempfile[1], "untitled") == 0) 
      return 1;
  }  
  else 
  {
    if (strcmp(filename, "untitled") == 0) 
      return 1;
  }
  return 0;
}

/*
 * Menu handler for `DocumentMenu (Open)'.
 * Brings up the file browser which allows the user to select a file to be opened.
 */
Menu_item OpenFileHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    Browse(NULL, BrowseOpen, (int) xv_get(ip->controls, PANEL_CLIENT_DATA), 
	   "#TimeLine Edit Document", "TimeLine Editor");
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Menu handler for `DocumentMenu (Save)'.
 * Brings up the file browser which allows the user to select a file to save the current TimeLine document into.
 */
Menu_item SaveFileHandler(Menu_item	item,
			  Menu_generate	op)
{
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    if (strcmp(tlFrame->filename, "untitled") == 0)		    /* No filename specified yet */
      Browse(NULL, BrowseSave, (int) xv_get(ip->controls, PANEL_CLIENT_DATA), 
	     "#TimeLine Edit Document", "TimeLine Editor");
    else 
      Browse(tlFrame->filename, BrowseCheckSave, (int) xv_get(ip->controls, PANEL_CLIENT_DATA),
	     "#TimeLine Edit Document", "TimeLine Editor");
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * This function will read in a TimeLine document.  The following are
 * the steps taken by this function:
 * 1) Check if a filename has been specified.  If not, print error
 *    message and return.
 * 2) Check if the specified filename is called 'untitled', which is
 *    not acceptable.  If so, print error message and return.
 * 3) Check if the specified file exists.  If not, print error message
 *    and return.
 * 4) Check if there are unsaved changes in the current TimeLIne
 *    document on display.  Let the user decide what to do (go on or cancel
 *    the load file).
 * 5) Check if the specified file is a TimeLine document file.  This *
 *    is done by checking the header of the file, which should have the line
 *    "#TimeLine Edit Document#" for a document generated by older
 *    versions of TimeLine, or "#TimeLine Edit Document, Version 2#" for
 *    newer versions of TimeLine, which generate synchronization hints.  If
 *    the file is not a TimeLine document of some sort, print error message
 *    and return.
 * 6) Read in the names of the applications and create an instrument
 *    list.  Compare this instrument list with a list obtained from the port
 *    manager (in the OpenAppsInitialize function) to get the correct port
 *    numbers for each app (if they are open).
 * 7) Clear all the panel lists in the info pop-up windows.
 * 8) We are now ready to read in the document.  Read in the
 *    information for the pause markers, regions and notes respectively.
 * 9) Call the repaint canvas procedure to draw the newly loaded
 *    document on the display.
 *10) Call the InitNotesInfo procedure to update the panel lists on
 *    all the info pop-up windows that have been created.
 */

int OpenHandler(char* filename, int TimeLineFrameNum)
{
  FILE*			fp;					    /* File handler to the edit file */
  int			i, j;					    /* For loop variables */
  int			tempint;				    /* Dummy integer variable */
  int			readSyncHints = False;			    /* Indicates if document should be parsed for sync hints */
  char			buf[100];				    /* Used to store strings such as Error messages */
  char			tempappName[64];			    /* Stores the application name for an instrument */
  char			tempLabel[MaxLabelLength + 8];
  Instrument*		instrument;				    /* Pointer to an instrument node in the instrument list */
  Note*			newNote;				    /* New note to be inserted in an instrument note list */
  Pause*		newPause;
  tlRegion*		newRegion;
  MediaSegment*		ms;					    /* Media segment containing information about a note */
  TimeLineFramePtr	tlFrame;
  
  tlFrame = TimeLineWindow[TimeLineFrameNum];
  if (CheckforUntitled (filename))				    /* Check if filename is 'untitled' (unacceptable) */
  {
    AlertMessage(tlFrame, "'untitled' is not an acceptable filename.",
		 "Please type in a new one.", NULL);
    return Error;
  }
  if (CheckChanges(LoadFile, tlFrame) == NOTICE_YES)		    /* Check if unsaved changes exist in edit list */
    return Error;
  fp = fopen(filename, "r");					    /* Open the file for reading. */
  fgets(buf, 37, fp);						    /* Check if file is a TimeLine edit file by reading header */
  if ((strcmp(buf, "#TimeLine Edit Document#\n") != 0) &&
      (strcmp(buf, "#TimeLine Edit Document, Version 2#\n") != 0))
  {
    AlertMessage(tlFrame, "This is not a TimeLine document.",
		 "Please select another document.", NULL);
    fclose(fp);
    return Error;
  }
  if (strcmp(buf, "#TimeLine Edit Document, Version 2#\n") == 0)    /* Is this a TimeLine document with synchronization hints? */
  {								    /* Yes, so no matter what, parse the lines that describe... */
    readSyncHints = True;					    /* ...sync hints, even though the data might be invalid */
    fscanf(fp, "#Sync Hints Available : %s\n", buf);		    /* ...Does this document have valid sync info? */
    if (strcmp(buf, "Yes") == 0)				    /* Yes, indicate that this function should parse the... */
      tlFrame->syncHints = SyncHintsAvailable;			    /* ...document for synchronization hints. */
  }
  else
    tlFrame->syncHints = NoSyncHintsAvailable;
  strcpy (tlFrame->filename, filename);
  FreeInstrumentList(tlFrame);
  fscanf(fp, "#Number of applications : %d\n", &tlFrame->numberOfApps);	/* Read in the # of apps used by this edit document */
  for (i=0; i < tlFrame->numberOfApps; i++)			    /* Create a new instrument list consisting of apps in file */
  {
    fscanf(fp, "#Name of application : %s\n", tempappName);
    if (i == 0) 
    {
      tlFrame->instHead = (Instrument *) InstrumentNew(0, NULL, tlFrame, tempappName);
      instrument = tlFrame->instHead;
    }
    else 
    {
      instrument->next = (Instrument *) InstrumentNew(i, NULL, tlFrame, tempappName);
      instrument = instrument->next;
    }
  }
  OpenAppsInitialize(tlFrame, 0);				    /* Match the applications with those currently open */
  if (tlFrame->pauseEdit >= 0) 
    xv_set(tlFrame->PausePopup->PauseList,			    /* Deselect the selected entry on the panel list */
	   PANEL_LIST_SELECT, tlFrame->pauseEdit, 
	   FALSE, NULL);
  FreePause(tlFrame);
  fscanf(fp, "\n");
  fscanf(fp, "********************************************************************\n");
  fscanf(fp, "#Number of pauses : %d\n", &(tlFrame->numPause));
  fscanf(fp, "********************************************************************\n");
  for (i=0; i < tlFrame->numPause; i++) 
  {
    newPause = (Pause *) malloc (sizeof(Pause));
    newPause->next = NULL;
    fscanf(fp, "\n");
    fscanf(fp, "#Pause %d\n", &tempint);
    fscanf (fp, "Position : %d\n", &(newPause->position));
    fscanf (fp, "Min, Sec : %d %d\n", &(newPause->min), &(newPause->sec));
    InsertNewPause(tlFrame, newPause);
  }
  UpdatePauseList(tlFrame, 0);
  
  if (tlFrame->regionEdit >= 0) 
    xv_set(tlFrame->RegionPopup->RegionList,			    /* Deselect the selected entry on the panel list */
	   PANEL_LIST_SELECT, tlFrame->regionEdit, 
	   FALSE, NULL);
  FreeRegion(tlFrame);
  fscanf(fp, "\n");
  fscanf(fp, "********************************************************************\n");
  fscanf(fp, "#Number of regions : %d\n", &(tlFrame->numRegion));
  fscanf(fp, "********************************************************************\n");
  for (i=0; i < tlFrame->numRegion; i++) 
  {
    newRegion = (tlRegion *) malloc (sizeof(tlRegion));
    newRegion->next = NULL;
    fscanf(fp, "\n");
    fscanf (fp, "#Region %d\n", &tempint);
    fscanf (fp, "startX, endX : %d %d\n", &(newRegion->startX), &(newRegion->endX));
    fscanf (fp, "startMin, endMin : %d %d\n", &(newRegion->startMin), &(newRegion->endMin));
    fscanf (fp, "startSec, endSec : %d %d\n", &(newRegion->startSec), &(newRegion->endSec));
    fgets(tempLabel, MaxLabelLength + 8, fp);
    strcpy(newRegion->label, &tempLabel[8]);
    newRegion->label[strlen(newRegion->label) - 1] = '\0';	    /* Remove the newline character */
    InsertNewRegion(tlFrame, newRegion);
  }
  UpdateRegionList(tlFrame);
  if (tlFrame->noteInstrument != NULL) 
  {
    if (tlFrame->noteInstrument->infoNote != NULL)		    /* Deselect the note on the panel list */
    {
      xv_set(tlFrame->noteInstrument->editInfo->NoteInfoList,
	     PANEL_LIST_SELECT, tlFrame->noteInstrument->selectedInfoNote,
	     FALSE, NULL);
    }
  }
  ClearAllNotes(tlFrame);				    /* Clear the notes for all the instruments */
  instrument = tlFrame->instHead;
  for (i=0; i < tlFrame->numberOfApps; i++)			    /* Now load the notes info for each instrument */
  {
    fscanf(fp, "\n");
    fscanf(fp, "********************************************************************\n");
    fscanf(fp, "#Name of application : %s\n", tempappName);	    /* Read in the app name and check for it in the app list */
    fscanf(fp, "#Number of notes : %d\n", &(instrument->numnotes)); /* Read in the number of notes for this application */
    fscanf(fp, "********************************************************************\n");
    for (j=0; j < instrument->numnotes; j++)			    /* Add the notes in the notes list for this application */
    {
      ms = (MediaSegment *) malloc (sizeof(MediaSegment));	    /* Create a new media segment */
      ms->documentName = (char *) malloc(MAXPATHLEN);
      ms->selection = (MAESelection*) malloc(sizeof(MAESelection));
      newNote = (Note *) malloc (sizeof(Note));		    /* Create a new note */
      newNote->ms = ms;
      newNote->next = NULL;
      fscanf(fp, "\n");
      fscanf(fp, "#Note %d\n", &tempint);			    /* Read in the values for the note from the file */
      fscanf(fp, "Display parameters : %d %d\n", &(newNote->start), &(newNote->end));
      fscanf(fp, "Document name : %s\n", ms->documentName);
      fscanf(fp, "Selection parameters : %d %d %d\n", 
	     &(ms->selection->start),
	     &(ms->selection->end),
	     &(ms->selection->duration));			    /* Stored in milliseconds */
      ms->duration = ms->selection->duration / 500;		    /* Stored in half seconds */
      ms->selection->offset = 0;				    /* Initialize offset to zero */
      CalculateNoteTime(newNote);				    /* Calculate the min and sec representations of the... */
								    /* ...start and end of the note */
      if (readSyncHints == True)				    /* Does this document store synchronization hints? */
      {								    /* Yes, read the estimated setup time into this note's... */
	fscanf(fp, "Estimated Setup Time : %d milliseconds\n",	    /* ...data structure. */
	       &(ms->setupTime));
      }
      else
	ms->setupTime = 0;
      fgets(tempLabel, MaxLabelLength + 8, fp);
      strcpy(ms->selection->label, &tempLabel[8]);
      (ms->selection->label)[strlen(ms->selection->label) - 1] = '\0';
      InsertNewNote (instrument, newNote, tlFrame);		    /* Insert new note into the Notes List of that application */
    }
    instrument = instrument->next;
  }
  fclose(fp);
  instrument = tlFrame->instHead;				    /* Load panel list of info pop-up windows with new Note info */
  while (instrument != NULL) 
  {
    InitNotesInfo(instrument, 1, tlFrame);
    instrument = instrument->next;
  }
  tlFrame->change = 0;
  UpdateHeader(tlFrame, 0);
  DrawPlaybackHead(-1, tlFrame);
  tlFrame->canvasStart = 0;
  ScrollToFirstQuarter(tlFrame, 0, 1);				    /* Scroll to make the start of the document visible */
  AppCanvasRepaintHandler(tlFrame->TimeLine_window->AppCanvas, tlFrame->paintWinApp, 
			   tlFrame->dpyApp, tlFrame->xidApp, NULL);
  return OK;
}								    /* end function OpenHandler */


/*
 * The Browse function in Browse.c will call this function.  This
 * function will save the current TimeLine document in a file.  The
 * following are the steps taken by this function.
 * 1) Check if a filename has been specified.  If not, print error
 *    message and return.
 * 2) Check if the specified filename is called 'untitled', which is not
 *    acceptable.  If so, print Error message and return.
 * 3) Check if the specified file can be opened.  If the file already
 *    exists, check if it is a TimeLine document file by checking the header.
 *    Print a message warning the user that the file already exists and give
 *    the option to go ahead or abort the operation.
 * 4) Now, open the file and write the data about the current TimeLine
 *    document in it.  Information is written about the pause markers,
 *    annotated regions and the notes respectively.  Only applications with
 *    notes will be saved.
 */
int SaveHandler(char* filename, int TimeLineFrameNum)
{
  FILE*			fp;					    /* File handler to the edit file */
  struct stat		stbuf;
  int			count;					    /* Count of the number of notes in an instrument */
  char			buf[100];				    /* Used to store strings such as Error messages */
  Instrument*		instrument;				    /* Pointer to an instrument node in the instrument list */
  Note*			currentNote;				    /* Pointer to the current note being read */
  Pause*		currentPause;				    /* Pointer to the current pause marker being read */
  tlRegion*		currentRegion;				    /* Pointer to the current region being read */
  TimeLineFramePtr	tlFrame;
  
  tlFrame = TimeLineWindow[TimeLineFrameNum];
  if (CheckforUntitled (filename))				    /* Check if filename is 'untitled' (unacceptable) */
  {
    AlertMessage(tlFrame, "'untitled' is not an acceptable filename.",
		 "Please type in a new one.", NULL);
    return Error;
  }
  if (stat(filename, &stbuf) == 0)				    /* Check if the file can be opened properly */
  {
    fp = fopen(filename, "r");
    fgets(buf, 37, fp);						    /* Get the header of the file */
    fclose(fp);
  if ((strcmp(buf, "#TimeLine Edit Document#\n") != 0) &&	    /* Check if it is a TimeLine Edit file */
      (strcmp(buf, "#TimeLine Edit Document, Version 2#\n") != 0))
    sprintf(buf, "This is not a TimeLine edit document.");
    else 
      sprintf(buf, "This TimeLine edit document exists.");
    if (notice_prompt(tlFrame->TimeLine_window->window, NULL,	    /* Prompts the user if the file is to be overwritten */
		      NOTICE_MESSAGE_STRINGS,
		      buf, "Do you wish to overwrite it?", NULL,
		      NOTICE_BUTTON_NO,       "Yes",
		      NOTICE_BUTTON_YES,      "No",
		      NULL) == NOTICE_YES)
      return Error;
  }
  fp = fopen(filename, "w");					    /* Open the file for writing. */
  strcpy(tlFrame->filename, filename);
  fprintf(fp, "#TimeLine Edit Document, Version 2#\n");		    /* Write the header. */
  if (tlFrame->syncHints == SyncHintsAvailable)			    /* Indicate whether synchronization hints are available... */
    fprintf(fp, "#Sync Hints Available : Yes\n");		    /* ...for this document. */
  else
    fprintf(fp, "#Sync Hints Available : No\n");
  count = 0;
  instrument = tlFrame->instHead;
  while (instrument != NULL)					    /* Count how many applications there are that contain notes */
  {
    if (instrument->numnotes > 0) 
      count ++;
    instrument = instrument->next;
  }
  fprintf(fp, "#Number of applications : %d\n", count);		    /* Write the number of applications that this document needs */
  instrument = tlFrame->instHead;				    /* Write the names of the apps required first.  This is... */
								    /* ...to facilitate checking later when loading... */
								    /* ...a file.  We can straight away find out if all the... */
								    /* ...required apps are open by reading the first few... */
								    /* ...lines of the file. */
  while (instrument != NULL) 
  {
    if (instrument->numnotes > 0) 
      fprintf(fp, "#Name of application : %s\n", instrument->port->appName);
    instrument = instrument->next;
  }
  count = 1;
  currentPause = tlFrame->pauseHead;
  fprintf(fp, "\n");
  fprintf(fp, "********************************************************************\n");
  fprintf(fp, "#Number of pauses : %d\n", tlFrame->numPause);	    /* Write the number of pauses for this application */
  fprintf(fp, "********************************************************************\n");
  while (currentPause != NULL) 
  {
    fprintf(fp, "\n");
    fprintf (fp, "#Pause %d\n", count);
    fprintf (fp, "Position : %d\n", currentPause->position);
    fprintf (fp, "Min, Sec : %d %d\n", currentPause->min, currentPause->sec);
    count ++;
    currentPause = currentPause->next;
  }
  count = 1;
  currentRegion = tlFrame->regionHead;
  fprintf(fp, "\n");
  fprintf(fp, "********************************************************************\n");
  fprintf(fp, "#Number of regions : %d\n", tlFrame->numRegion);	    /* Write the number of regions for this application */
  fprintf(fp, "********************************************************************\n");
  while (currentRegion != NULL) 
  {
    fprintf(fp, "\n");
    fprintf (fp, "#Region %d\n", count);
    fprintf (fp, "startX, endX : %d %d\n", currentRegion->startX, currentRegion->endX);
    fprintf (fp, "startMin, endMin : %d %d\n", currentRegion->startMin, currentRegion->endMin);
    fprintf (fp, "startSec, endSec : %d %d\n", currentRegion->startSec, currentRegion->endSec);
    fprintf(fp, "Label : %s\n", currentRegion->label);
    count ++;
    currentRegion = currentRegion->next;
  }
  instrument = tlFrame->instHead;				    /* For each instrument, write the info required for... */
								    /* ...loading later. This information includes the name... */
								    /* ...of the application, the number of notes in each... */
								    /* ...application, media segment information for each... */
								    /* ...note, and where each note is displayed on the canvas. */
  while (instrument != NULL) 
  {
    if (instrument->numnotes > 0) 
    {
      fprintf(fp, "\n");
      fprintf(fp, "********************************************************************\n");
      fprintf(fp, "#Name of application : %s\n", instrument->port->appName); /* Write the name of the application */
      fprintf(fp, "#Number of notes : %d\n", instrument->numnotes); /* Write the number of notes for this application */
      fprintf(fp, "********************************************************************\n");
      currentNote = instrument->firstNote;			    /* Now write the information for each note */
      count = 1;
      while (currentNote != NULL) 
      {
	fprintf(fp, "\n");
	fprintf (fp, "#Note %d\n", count);
	fprintf (fp, "Display parameters : %d %d\n", currentNote->start, 
		 currentNote->end);
	fprintf (fp, "Document name : %s\n",  currentNote->ms->documentName);
	fprintf (fp, "Selection parameters : %d %d %d\n", 
		 currentNote->ms->selection->start,
		 currentNote->ms->selection->end,
		 currentNote->ms->selection->duration);
	fprintf(fp, "Estimated Setup Time : %d milliseconds\n",
		currentNote->ms->setupTime);
	fprintf(fp, "Label : %s\n", currentNote->ms->selection->label);
	currentNote = currentNote->next;
	count ++;
      }
    }
    instrument = instrument->next;
  }
  fclose (fp);
  tlFrame->change = 0;						    /* Reset the change flag */
  UpdateHeader(tlFrame, 0);
  return OK;
}								    /* end function SaveHandler */


/*
 * Menu handler for `DocumentMenu (Save as ...)'.
 */
Menu_item SaveAsFileHandler(Menu_item		item,
			    Menu_generate	op)
{
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    Browse(NULL, BrowseSave, (int) xv_get(ip->controls, PANEL_CLIENT_DATA), 
	   "#TimeLine Edit Document", "TimeLine Editor");
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Menu handler for `DocumentMenu (Quit)'.
 * This function will attempt to destroy a frame.  Because the notify interpose destroy function has been specified, the QuitNotify function will be
 * called to check if the frame is to be destroyed.
 */
Menu_item QuitHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  int whichWindow;
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  whichWindow = xv_get(ip->controls, PANEL_CLIENT_DATA);
  tlFrame = TimeLineWindow[whichWindow];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_destroy_safe(tlFrame->TimeLine_window->window);		    /* Destroy the application */
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Menu handler for `DocumentMenu (New)'.
 * This function will first check through the array indicating the status of a TimeLine Window to see which is available for creation.
 * It will only create a new window if there is one free.
 */
Menu_item NewHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  int whichWindow, count;
  
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    count = 1;
    whichWindow = MaxWindows + 1;
    while (count <= MaxWindows) 
    {
      if (WindowFree[count] == 0)				    /* Check to see which window is free */
      {
	whichWindow = count;
	count = MaxWindows;
      }
      count ++;
    }
    if (whichWindow <= MaxWindows)				    /* One window is free.  Create a new TimeLine */
      TimeLineInit(TimeLineWindow[whichWindow], whichWindow);
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Menu handler for `DocumentMenu (About TimeLine Editor ...)'.
 * This function pops up the Info window.
 */
Menu_item AboutHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_set(tlFrame->InfoPopup->InfoPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(tlFrame->InfoPopup->InfoPopup, XV_SHOW, TRUE, NULL);
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Notify callback function for `closeInfoButton'.
 * This function will close the Info popup window.
 */
void CloseInfoPopup(item, event)
     Panel_item	item;
     Event		*event;
{
  Info_InfoPopup_objects	*ip = (Info_InfoPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  xv_set(ip->InfoPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(ip->InfoPopup, XV_SHOW, FALSE, NULL);
}

/*
 * Menu handler for `DocumentMenu (Close)'.
 * This function will close the currently loaded file (if any), reset the filename to "untitled" and clear the whole canvas.
 * If multiple TimeLine windows are open, instead of closing the file, the whole TimeLine frame will be destroyed.
 */
Menu_item CloseHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    if (numberOfWindows == 1) 
    {
      if (CheckChanges(CloseFile, tlFrame) == NOTICE_YES)	    /* Check if unsaved changes exist in edit list */
	return item;
      tlFrame->change = 0;
      ClearAllHandler(item, MENU_NOTIFY);
      strcpy(tlFrame->filename, "untitled");
      tlFrame->change = 0;
      UpdateHeader(tlFrame, 0);
      ScrollToFirstQuarter(tlFrame, 0, 1);
    }
    else 
    {
      quitAll = 0;						    /* Destroy this window */
      xv_destroy_safe(tlFrame->TimeLine_window->window);
    }
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}
