/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
/* $Header: /Source/Media/collab/ShellEdit/RCS/ShellEdit.c,v 2.13 92/06/15 14:57:15 drapeau Exp $ */
/* $Log:	ShellEdit.c,v $
 * Revision 2.13  92/06/15  14:57:15  drapeau
 * Minor change: declared "UpdateHeader()" with function prototypes.
 * 
 * Revision 2.12  92/05/29  12:42:54  drapeau
 * Modified code to track new name of the MAEstro "Selection" structure;
 * it is now named "MAESelection".
 * 
 * Revision 2.11  92/04/30  16:24:00  drapeau
 * Several changes:
 * * Re-formatted and commented small portions of code for cosmetic reasons.
 * * Updated copyright notice.
 * * Added function UpdateHeader() that updates the title bar of 
 *   application to reflect the current status of the current
 *   document (whether modified or not).
 * * Made "Save" silent (i.e., does not always prompt for confirmation,
 *   only when the document is being saved for the first time).
 * 
 * Revision 2.1  92/01/09  18:20:41  drapeau
 * Slight modifications to make code ANSI-compliant.
 * 
 * Revision 2.0  92/01/06  12:57:02  drapeau
 * Changes too numerous to list here, due to lack of proper RCS logging
 * of previous changes.
 * 
 * Revision 1.14  91/07/18  14:45:15  bryant
 * Made it so that quitting from the frame is identical to quitting from the
 * Document window
 * 
 * Revision 1.13  91/07/17  10:08:11  bryant
 * Finished the added protocol items (partial selection using the offset field).
 * Implemented the command line support for -h hostName filename. Upgraded the
 * algorithm so that processes that must be run in the foreground (because of a
 * stop from TTY input or TTY output) can also be timed.
 * Cleaned up the interfacing with OpenPanel.
 * 
 * Revision 1.12  91/07/15  19:46:15  bryant
 * Upgraded the means by which ShellEdit gets the process ID's of its children -- instead
 * of reading in the job data from the tty window, it now uses a popen on ps -jx and
 * scans for any new processes owned by the shell of the tty window. I also improved
 * the quit function so that it would destroy all child processes before quitting.
 * 
 * Revision 1.11  91/07/02  18:21:30  bryant
 * Implemented the OpenPanel and SavePanel functionality. Changed the popen command to ps -jx
 * so that the first two fields would be the PPID and the PID of the process.
 * 
 * Revision 1.10  91/06/28  16:31:58  bryant
 * Forgot to put in the log comment
 *  */

/* Still to do:
   change notify response defaults
   comment code
*/

static char rcsid[] = "$Header: /Source/Media/collab/ShellEdit/RCS/ShellEdit.c,v 2.13 92/06/15 14:57:15 drapeau Exp $";
#include <xview/tty.h>  
#include <xview/xview.h>
#include <xview/font.h>
#include <xview/icon.h>
#include <xview/notice.h>
#include <xview/notify.h>
#include <xview/panel.h>
#include <xview/seln.h>
#include <xview/svrimage.h>
#include <xview/textsw.h>
#include <xview/termsw.h> 
#include <xview/xv_xrect.h>
#include <gdd.h>
#include <X11/Xos.h>
#include <errno.h>
#include <stdio.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <signal.h>
#include <ctype.h>
#include "ShellEdit_ui.h"
#include <Sender.h>
#include <Receiver.h>
#include <getopt.h>
#include <Browse.h>

extern char* realpath(char*, char*);
extern char* strdup(char*);

#define MaxLength 255
#define WriteDontExit 0
#define WriteThenExit 1

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute INSTANCE;

ShellEdit_baseWindow_objects *ShellEdit_baseWindow;
ShellEdit_ShellEditInfo_objects	*ShellEdit_ShellEditInfo;
int OpenHandler(), SaveHandler();                /* browser code */
char path[MaxLength];                           /* browser code */
struct itimerval timer[MaxLength], pauseTimer;
char command[MaxLength]; 
char *duration[MaxLength];
char dummy[MaxLength];
char *currentFilename = "untitled";
int editNum, bufferNum = 0, bufferSize = 0, timerIndex = 0, OpenEditFile(), AnElement();
int pauseCount = 0, pauseFreeze = 0, numJobs = 0, shellEditPid, foregroundMode = 0;
int jobPids[MaxLength], resetMode = 0;
int WriteFunction = WriteDontExit;
Notify_client pausePid;
FILE *filep;
unsigned int lastPid = 0;
Notify_value GetPid(), WaitForPs(), IncrementCounter(), CheckForStoppedProcess();
Notify_value QuitNotify();
Xv_font *listFont;
void itoa(), Reverse(), DeselectList(), DeselectAll(), WriteEditListToFile(), LoadEditList();
void DeleteAllLists(), FileAccessError(), Show(), Hide(), KillPids(), Show(), Hide();
void DeleteFromBuffer(), PrintBuffer(), KillProcesses(), CheckOptions();
void OpenDoc();
void SetSelection();
void PerformSelection();
char **GetDoc();
MAESelection*	GetSelection();
void HaltSelection();
void PauseSelection();
void ResumeSelection();
void HideApplication();
void ShowApplication();
IconData *GetAppIcon();
void PerformCommand(Panel_item item, Event* event);
void UpdateHeader (char* documentName, int modified);

static char *canonFilename;
Sender* sender;
Receiver* receiver;
Port senderPort;
int ReceiverPortNumber;

struct EditPacket {
  char filename[MaxLength];
  int numItems;
  int changes;
  char *command[MaxLength]; 
  char *label[MaxLength]; 
  char *duration[MaxLength];
} *editList[MaxLength];


void main(argc, argv)
     int		argc;
     char		**argv;
{
  static DispatchTable  DT = 
    {
      OpenDoc,
      GetDoc,
      GetSelection,
      SetSelection,
      PerformSelection,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      HaltSelection,
      PauseSelection,
      ResumeSelection,
      HideApplication,
      ShowApplication,
      GetAppIcon
    }; 

  /*
   * Initialize XView.
   */
  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);
  INSTANCE = xv_unique_key();

  /*
   * Initialize user interface components.
   */
  ShellEdit_baseWindow = ShellEdit_baseWindow_objects_initialize(NULL, NULL);
  ShellEdit_ShellEditInfo = ShellEdit_ShellEditInfo_objects_initialize(NULL, ShellEdit_baseWindow->baseWindow);
  CreateBrowse(OpenHandler, SaveHandler,
		ShellEdit_baseWindow->baseWindow);       /* browser code */  
  notify_interpose_destroy_func(ShellEdit_baseWindow->baseWindow, QuitNotify);

  textsw_insert(ShellEdit_baseWindow->terminal, "unset filec\n", strlen("unset filec\n"));
  textsw_insert(ShellEdit_baseWindow->terminal, "clear\n", strlen("clear\n"));

  listFont = (Xv_font *) xv_find(ShellEdit_baseWindow->baseWindow, FONT,
				 FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
				 FONT_STYLE, FONT_STYLE_NORMAL,
				 FONT_SIZE, 12,
				 NULL);
  
  xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, "Indefinite", NULL);
  shellEditPid = (unsigned int)getpid();
  
  /* Networking stuff */
  
  senderPort.hostName = "localhost";
  ReceiverPortNumber = AnyPort;
  CheckOptions(argc, argv);
  
  senderPort.portNumber = PortMgrPortNumber;
  sender = NewSender(&senderPort);
  if (!sender)
    {
      notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
		    NOTICE_MESSAGE_STRINGS,
		    "The PortManager is not running.",
		    "Please check that it is running and try again.",
		    NULL,
		    NOTICE_BUTTON_YES, "OK",
		    NULL);
      exit(0);
    }
  receiver = NewReceiver(sender, "ShellEdit", ReceiverPortNumber);
  BuildDispatchTable (&DT);
  canonFilename = (char *) malloc(MaxLength);
  
  (void) notify_enable_rpc_svc (TRUE);
  xv_main_loop(ShellEdit_baseWindow->baseWindow);		    /* Turn contol over to xview */
  exit(0);
}



void
OpenDoc(filename)
char **filename;
{
  char temp[MaxLength];

  if (strcmp(*filename, currentFilename) != 0 &&		    /* Check that filename is not 'untitled' or that... */
      strstr(*filename, "untitled") == NULL			    /* ...it hasn't been loaded already */
      && strlen(*filename) > 0)
    {
      currentFilename = (char *)strdup(*filename);
      sprintf(temp, "Shell Edit Document :  \"%s\"", *filename);
      xv_set(ShellEdit_baseWindow->baseWindow, XV_LABEL, temp, NULL);
      OpenEditFile();
    }
}


void
SetSelection(MAESelection* selection)
{
  char selectedNum[MaxLength];
  char adjustedDuration[MaxLength];
  int offset = (int)((selection->offset)/1000 + .5);
  editNum = (selection->start)+1;
  if (editNum > editList[bufferNum]->numItems)
    {
      notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
		    NOTICE_MESSAGE_STRINGS,
		    "Selected edit doesn't exist",
		    NULL,
		    NOTICE_BUTTON_YES, "OK",
		    NULL);
      return;
    }
  strcpy(adjustedDuration, editList[bufferNum]->duration[editNum-1]); 
  if (strncmp(adjustedDuration, "Indefinite", 10))
    itoa(atoi(editList[bufferNum]->duration[editNum-1])-offset, adjustedDuration);
  if (editNum != 0)
    {
      DeselectAll();
      itoa(editNum, selectedNum);  
      xv_set(ShellEdit_baseWindow->command, PANEL_VALUE, editList[bufferNum]->command[editNum-1], NULL);  
      if (strncmp(editList[bufferNum]->label[editNum-1], "No Label", 8) != 0)
	xv_set(ShellEdit_baseWindow->label, PANEL_VALUE, editList[bufferNum]->label[editNum-1], NULL);  
      xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, adjustedDuration, NULL);
      xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, selectedNum, NULL);  
      xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, FALSE, NULL);
      xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, FALSE, NULL);
      xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, FALSE, NULL);  
      xv_set(ShellEdit_baseWindow->scrollList, PANEL_LIST_SELECT, editNum-1, TRUE, NULL); 
    }
}

void
  PerformSelection()
{
  if (editNum != 0 && editNum <= editList[bufferNum]->numItems)
    PerformCommand(NULL, NULL);
}

char **
  GetDoc(unusedArg)
void *unusedArg;
{
  if (strlen(currentFilename) == 0 || realpath(currentFilename, canonFilename) == NULL) 
    strcpy(canonFilename, "untitled");
  return (&canonFilename);
}

MAESelection* GetSelection(void* unusedArg)
{
  static MAESelection select;
  if (editNum >= 1)
    {
      select.offset = 0;
      if (strncmp(editList[bufferNum]->label[editNum-1], "No Label", 8) != 0)
	sprintf(select.label, editList[bufferNum]->label[editNum-1]);
      else
	sprintf(select.label, editList[bufferNum]->command[editNum-1]);
      select.start = editNum - 1;
      select.end = editNum - 1;
      if (strncmp(editList[bufferNum]->duration[editNum-1], "Indefinite", 10))
	select.duration = 1000*atoi(editList[bufferNum]->duration[editNum-1]);   
      else 
	select.duration = 0;
    }
  else 
    {
      select.duration = -1;
      select.start = -1;
      select.end = -1;
      select.offset = 0;
      sprintf(select.label, "No Label");
    }
  return (&select);
}

void 
  HaltSelection()
{
  KillProcesses(NULL, NULL);
}

void 
  PauseSelection()
{
  pauseFreeze = 0;
/*  printf("tv_sec = %d\n", timer[timerIndex].it_value.tv_sec); */
  pauseTimer.it_value.tv_sec = 1;
  pauseTimer.it_value.tv_usec = 0;
  pauseTimer.it_interval.tv_sec = 1;
  pauseTimer.it_interval.tv_usec = 0;
  notify_set_itimer_func((Notify_client)ShellEdit_baseWindow->baseWindow, IncrementCounter, 
			 ITIMER_REAL, &pauseTimer, NULL);
  return;
}


Notify_value
  IncrementCounter(timerIndex, which)
Notify_client timerIndex;
int which;
{
  if (pauseFreeze == 0)
    pauseCount++;
}

void 
  ResumeSelection()
{
  if (++timerIndex > 255)
    timerIndex = 0;
  if (pauseFreeze)    /* occurs if note timer ends before resume is selected */
    {
      timer[timerIndex].it_value.tv_sec = pauseCount;
      pauseCount = 0;
      notify_set_itimer_func((Notify_client)pausePid, WaitForPs, 
			 ITIMER_REAL, &timer[timerIndex], NULL);
    }
  else
    pauseFreeze = 1;
  notify_set_itimer_func((Notify_client)ShellEdit_baseWindow->baseWindow, 
			 NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
}

void 
  HideApplication()
{
  xv_set(ShellEdit_baseWindow->baseWindow, FRAME_CLOSED, TRUE, NULL);
  XFlush((Display *)xv_get(ShellEdit_baseWindow->baseWindow, XV_DISPLAY));
}

void 
  ShowApplication()
{
  xv_set(ShellEdit_baseWindow->baseWindow, FRAME_CLOSED, FALSE, NULL); 
  XRaiseWindow((Display *)xv_get(ShellEdit_baseWindow->baseWindow, XV_DISPLAY),
	       (Window)xv_get(ShellEdit_baseWindow->baseWindow, XV_XID));
  XFlush((Display *)xv_get(ShellEdit_baseWindow->baseWindow, XV_DISPLAY));
}

IconData *
  GetAppIcon()
{
  static IconData returnVal;
  static unsigned short baseWindow_bits[] = {
#include "ShellEdit.icon"
        };
  returnVal.iconData = (char *)malloc(sizeof(baseWindow_bits));
  bcopy(baseWindow_bits, returnVal.iconData, sizeof(baseWindow_bits));
  if (returnVal.iconData)
    returnVal.dataLength = sizeof(baseWindow_bits);
  return(&returnVal);
}



Notify_value
  CheckForStoppedProcess(activePid, which)
Notify_client activePid;
int which;
{
  int i=0, pid, ppid, length;
  char line[MaxLength];
  filep = popen("ps jx", "r");
  if (filep != NULL)
    {
      while (fgets(line, MaxLength, filep)!=NULL)
	{
	  if (i > 0)               /* skip over the first line that has the labels PPID PID etc */
	    {
	      sscanf(line, "%u%u", &ppid, &pid);  
	      if ((pid==(int)(activePid-1)) && (line[34]=='T' || line[35]=='T' || line[36]=='T' || line[37]=='T'))
		{
		  printf("Stopped process %d\n", activePid-1);
		  WaitForPs(activePid-1, which);
		  resetMode = 1;
		  foregroundMode = 1;
		  PerformCommand(NULL, NULL);
		  textsw_insert(ShellEdit_baseWindow->terminal, "reset\n", strlen("reset\n"));
		  /* You might also set a reset flag to send the reset command to the terminal */
		}	    
	    }
	  i++;
	}
      pclose(filep);
    }
  else
    printf("Couldn't do a popen()\n");
  return NOTIFY_DONE;
}


/*
 * Notify callback function for `PerformCommand'.
 */
void PerformCommand(Panel_item item, Event* event)
{
  int pid, ppid, i;
  char line[MaxLength];
  if (++timerIndex > 255)
    timerIndex = 0;
  duration[timerIndex] = (char *)malloc(MaxLength);
  strcpy(duration[timerIndex], (char *)xv_get(ShellEdit_baseWindow->duration, PANEL_VALUE));
  if (strncmp(duration[timerIndex], "Indefinite", 10) != 0)  
    {
      if (atoi(duration[timerIndex]) < 1)
	{
	  notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
			NOTICE_MESSAGE_STRINGS,
			"Duration out of bounds: set to 1 sec.",
			NULL,
			NOTICE_BUTTON_YES, "OK",
			NULL);
	  strcpy(duration[timerIndex], "1");
	  xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, "1", NULL);
	}
      else if (atoi(duration[timerIndex]) > 10000)
	{
	  notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
			NOTICE_MESSAGE_STRINGS,
			"Duration out of bounds: set to 10000 sec.",
			NULL,
			NOTICE_BUTTON_YES, "OK",
			NULL);
	  strcpy(duration[timerIndex], "10000");
	  xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, "10000", NULL);
	}
    }
  filep = popen("ps jx", "r");
  if (filep != NULL)
    {
      numJobs = 0;
      i = 0;
      while (fgets(line, MaxLength, filep)!=NULL)
	{
	  if (i > 0)               /* skip over the first line that has the labels PPID PID etc */
	    {
	      sscanf(line, "%u%u", &ppid, &pid);   
	      if (ppid==(int)xv_get(ShellEdit_baseWindow->terminal, TTY_PID))
		jobPids[numJobs++] = pid;
	    }
	  i++;
	}
      pclose(filep);
    }
  else
    printf("Couldn't do a popen()\n");
  if (foregroundMode == 0)
    sprintf(command, "%.255s &\n", 
	    (char *)xv_get(ShellEdit_baseWindow->command, PANEL_VALUE)); 
  else
    {
      sprintf(command, "%.255s\n", 
	      (char *)xv_get(ShellEdit_baseWindow->command, PANEL_VALUE)); 
      foregroundMode = 0;
    }
  textsw_insert(ShellEdit_baseWindow->terminal, command, strlen(command));
  timer[timerIndex].it_value.tv_sec = 1;
  notify_set_itimer_func((Notify_client)timerIndex, GetPid,
                         ITIMER_REAL, &timer[timerIndex], NULL);
  return;
}


Notify_value
  GetPid(client, which)
Notify_client client;
int which;
{
  int killPid=-1, ppid, pid, i;
  unsigned int length;
  char line[MaxLength];
  filep = popen("ps jx", "r");
  if (filep != NULL)
    {
      i = 0;
      while (fgets(line, MaxLength, filep)!=NULL)
	{
	  if (i > 0)              /* skip over the first line that has the labels PPID PID etc */
	    {
	      sscanf(line, "%u%u", &ppid, &pid);   
	      if (ppid==(int)xv_get(ShellEdit_baseWindow->terminal, TTY_PID) &&
		  !AnElement(pid, jobPids))
		{
		  if (numJobs < 255)
		    jobPids[numJobs++] = pid;
		  else
		    {
		      notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
				    NOTICE_MESSAGE_STRINGS,
				    "Too many job processes",
				    NULL,
				    NOTICE_BUTTON_YES, "OK",
				    NULL);
		      return NOTIFY_DONE;
		    }
		  killPid = pid;
		}
	    }
	  i++;
	}
      pclose(filep);
    }
  else
    printf("Couldn't do a popen()\n");
  if (killPid > -1)
    {
      if (++timerIndex > 255)
	timerIndex = 0;
      timer[timerIndex].it_value.tv_sec = 2;
      notify_set_itimer_func((Notify_client)(killPid+1), CheckForStoppedProcess,
			 ITIMER_REAL, &timer[timerIndex], NULL);         
      if (strncmp(duration[(int)client], "Indefinite", 10) != 0)
	{
	  if ((length = atoi(duration[(int)client]))>1)
	    length -= 1;
	  if (++timerIndex > 255)
	    timerIndex = 0;
	  timer[timerIndex].it_value.tv_sec = length;
	  pausePid = killPid;
	  notify_set_itimer_func((Notify_client)killPid, WaitForPs, 
				 ITIMER_REAL, &timer[timerIndex], NULL);
	}
    }
  return NOTIFY_DONE;
}


int
  AnElement(pid, jobPids)
int pid;
int jobPids[];
{
  int i;
  for (i=0; i<numJobs; i++)
    {
/*    printf("jobPids[%d]=%d\n", i, jobPids[i]); */
      if (pid == jobPids[i])
	return 1;
    }
  return 0;
}


Notify_value
  WaitForPs(killPid, which)
Notify_client killPid;
int which;
{
  unsigned int i, pid, ppid, pgid, sid, pidIndex=0, pids[MaxLength][3];
  char line[MaxLength];
  if (pauseCount == 0)
    {
      if (resetMode > 0)
	if (resetMode++ ==2)
	  {
	    kill((int)killPid, SIGKILL);
	    textsw_insert(ShellEdit_baseWindow->terminal, "reset\n", strlen("reset\n"));
	    resetMode = 0;
	    return NOTIFY_DONE;
	  }
      filep = popen("ps jx", "r");
      if (filep != NULL)
	{
	  i = 0;
	  while (fgets(line, MaxLength, filep)!=NULL)
	    {
	      if (i > 0)     /* skip over the first line that has the labels PPID PID etc */
		{
		  sscanf(line, "%u%u%u%u", &ppid, &pid, &pgid, &sid);
		  pids[pidIndex][0] = ppid;
		  pids[pidIndex][1] = pid;
		  pids[pidIndex][2] = sid;
		  pidIndex++;
		}
	      i++;
	    }
	  pclose(filep);
	}
      else
	printf("Couldn't do a popen()\n");

      for(i=0; i<pidIndex; i++)
	if ((pids[i][1]==(int)killPid) || (pids[i][0]==1 && pids[i][2]==shellEditPid))        /* kill the killPid process as well as any process that */
	  KillPids(pids[i][1], pidIndex, pids, 0);                                            /* is of this session ID and is orphaned                */
    }
  else if (pauseFreeze == 0)
    pauseFreeze = 1;
  else
    {
      if (++timerIndex > 255)
	timerIndex = 0;
      timer[timerIndex].it_value.tv_sec = pauseCount;
      pauseCount = 0;
      notify_set_itimer_func((Notify_client)pausePid, WaitForPs, 
			     ITIMER_REAL, &timer[timerIndex], NULL);
    }
  return NOTIFY_DONE;
}

void 
  KillPids(killPid, pidIndex, pids, marker)
unsigned int killPid;
unsigned int pidIndex;
unsigned int pids[][3];
int marker;
{
  unsigned int i;
  char str[MaxLength];
/*   printf("killPid (recursive)=%d\n", killPid);   */
  kill(killPid, SIGKILL);
  for (i=marker; i<pidIndex; i++)
    if (pids[i][0]==killPid)
      KillPids(pids[i][1], pidIndex, pids, i+1);
}


/*
 * Notify callback function for `killProcesses'.
 */
void
  KillProcesses(item, event)
Panel_item	item;
Event		*event;
{
  char line[MaxLength];
  int i, pid, ppid, pgid, sid;
  filep = popen("ps jx", "r");
  if (filep != NULL)
    {
      i = 0;
      while (fgets(line, MaxLength, filep)!=NULL)
	{
	  if (i > 0)               /* skip over the first line that has the labels PPID PID etc */
	    {
	      sscanf(line, "%u%u%u%u", &ppid, &pid, &pgid, &sid);   
	      if (sid==shellEditPid && pid!=shellEditPid && pid!=
		  (int)xv_get(ShellEdit_baseWindow->terminal, TTY_PID))
		kill(pid, SIGKILL);
	    }
	  i++;
	}
      pclose(filep);
    }
  else
    printf("Couldn't do a popen()\n");
}


/*
 * Notify callback function for `indefinite'.
 */
void
  Indefinite(item, event)
Panel_item	item;
Event		*event;
{
  xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, "Indefinite", NULL);
}


/*
 * Notify callback function for `scrollList'.
 */
int 
  ScrollList(item, string, client_data, op, event)
Panel_item   	item;
char		*string;
Xv_opaque  	client_data;
Panel_list_op	op;
Event		*event;
{
  char selectedNum[MaxLength];
  int counter;
  switch(op) 
    {
    case PANEL_LIST_OP_DESELECT:
      DeselectAll();     
      xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, "New Edit", NULL);  
      xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, TRUE, NULL);
      xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, TRUE, NULL);
      xv_set(ShellEdit_baseWindow->label, PANEL_VALUE, "", NULL); 
      break;
      
    case PANEL_LIST_OP_SELECT:
      sscanf(string, "%d", &editNum);
      itoa(editNum, selectedNum);  
      xv_set(ShellEdit_baseWindow->command, PANEL_VALUE, editList[bufferNum]->command[editNum-1], NULL);  
      if (strncmp(editList[bufferNum]->label[editNum-1], "No Label", 8) != 0)
	xv_set(ShellEdit_baseWindow->label, PANEL_VALUE, editList[bufferNum]->label[editNum-1], NULL);  
      xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, editList[bufferNum]->duration[editNum-1], NULL);
      xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, selectedNum, NULL);  
      xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, FALSE, NULL);
      xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, FALSE, NULL);
      xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, FALSE, NULL);
      break;
      
    case PANEL_LIST_OP_VALIDATE:
    case PANEL_LIST_OP_DELETE:
      break;
      
    default:
      printf("unknown operation\n");
      break;
    }
  return XV_OK;
}


/*
 * Notify callback function for `filename'.
 */
Panel_setting 
  Filename(item, event)
Panel_item	item;
Event		*event;
{
  OpenEditFile();
  return panel_text_notify(item, event);
}

/*
 * Menu handler for `documentMenu (Info)'.
 */
Menu_item
  Info(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
    {
    case MENU_NOTIFY:
      Show(&ShellEdit_ShellEditInfo->ShellEditInfo);
      break;
    case MENU_DISPLAY:
    case MENU_DISPLAY_DONE:
    case MENU_NOTIFY_DONE:
      break;
    }
  return item;
}

/*
 * Menu handler for `documentMenu (Open)'.
 */
Menu_item
  OpenFile(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) {
  case MENU_NOTIFY:
    Browse(realpath(currentFilename, dummy), 
	   BrowseOpen, 0, "#Shell Edit Document#", "ShellEdit"); 
    break;
  case MENU_DISPLAY:
  case MENU_DISPLAY_DONE:
  case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


int 
  OpenEditFile()
{
  int i, size, response;
  struct stat stbuf;
  char asciiSize[MaxLength];
  char message[MaxLength]; 
  if (bufferSize > 0)
  {
    if (editList[bufferNum]->changes == 0)
      DeleteFromBuffer();
    for (i=0; i<bufferSize; i++)
    {
      if (realpath(currentFilename, dummy)==NULL)
      {
	Browse(realpath(currentFilename, dummy), 
	       BrowseOpen, 0, "#Shell Edit Document#", "ShellEdit"); 
	return -1;
      }
      if (strcmp(realpath(currentFilename, dummy),
		 editList[i]->filename)==0)
      {
	bufferNum = i;
	LoadEditList();
	return 1;
      }
    }
  }
  bufferSize++;
  bufferNum = bufferSize-1;
  editList[bufferNum] = (struct EditPacket *)malloc(sizeof(struct EditPacket));
  editList[bufferNum]->changes = 0;
  editList[bufferNum]->numItems = 0;
  if ((int)xv_get(ShellEdit_baseWindow->scrollList, PANEL_LIST_NROWS)!=0)
    DeleteAllLists();
  if (realpath(currentFilename, dummy) != NULL)
    strcpy(editList[bufferNum]->filename, 
	   realpath(currentFilename, dummy));
  else
    {
      Browse(realpath(currentFilename, dummy), 
	     BrowseOpen, 0, "#Shell Edit Document#", "ShellEdit"); 
      return -1;
    }            
  
  if (strlen(editList[bufferNum]->filename) == 0)
    {
      Browse(realpath(currentFilename, dummy), 
	     BrowseOpen, 0, "#Shell Edit Document#", "ShellEdit"); 
      return -1;
    }            

  if (CheckFile(&size) == 1)
    {
      itoa(size, asciiSize);
      xv_set(ShellEdit_baseWindow->numOfEdits, PANEL_LABEL_STRING, asciiSize, NULL);
      editList[bufferNum]->numItems = size;
      if (editList[bufferNum]->numItems > 0)
	xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, FALSE, NULL);
      UpdateHeader(editList[bufferNum]->filename,		    /* Display new document title and changed status... */
		   editList[bufferNum]->changes);		    /* ...on title bar */
    }
  else
    {
      xv_set(ShellEdit_baseWindow->numOfEdits, PANEL_LABEL_STRING, "0", NULL);
      /*   DeleteFromBuffer();   */
      return -1;
    }     
  return 1;
}

/*
 * Menu handler for `documentMenu (Save)'.
 */
Menu_item
  Save(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
    {
    case MENU_NOTIFY:
      if (strlen(currentFilename) == 0 ||
	  strstr(currentFilename, "untitled") != NULL) 
	Browse(realpath(currentFilename, dummy), 
	       BrowseSave, 0, "#Shell Edit Document#", "ShellEdit"); 
      else
	Browse(realpath(currentFilename, dummy), 
	       BrowseCheckSave, 0, "#Shell Edit Document#", "ShellEdit"); 
      break;
    case MENU_DISPLAY:
    case MENU_DISPLAY_DONE:
    case MENU_NOTIFY_DONE:
      break;
    }
  return item;
}								    /* end function Save */



/*
 * Notify callback function for `SaveAs'.
 */
Panel_setting
  SaveFile(item, event)
Panel_item	item;
Event		*event;
{
  FILE *fp;
  char fileType[MaxLength];
  char dummy[MaxLength];
  int response;
  struct stat stbuf;
  char message[MaxLength];
  if (editList[bufferNum] == NULL)
  {
    editList[bufferNum] = (struct EditPacket *)malloc(sizeof(struct EditPacket));
    editList[bufferNum]->numItems = 0;
    bufferSize++;
  }
  strcpy(editList[bufferNum]->filename, currentFilename);
  if (strlen(editList[bufferNum]->filename)<1)
  {
    response = notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
			     NOTICE_MESSAGE_STRINGS,
			     "Filename not specified.",
			     NULL,
			     NOTICE_BUTTON_YES, "OK",
			     NULL);
    return item;
  }
  if (stat(editList[bufferNum]->filename, &stbuf)!=-1) 
  {
    fp = fopen (editList[bufferNum]->filename, "r");				    
    fscanf (fp, "#%s Edit Document#\n", fileType);			
    if (strcmp(fileType, "Shell"))				
    {								
      sprintf (message, "'%s' exists and is not a Shell Edit list.\n", editList[bufferNum]->filename);
      response = notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       message,
			       "Go ahead and overwrite its contents?",
			       NULL,
			       NOTICE_BUTTON_NO, "Yes",
			       NOTICE_BUTTON_YES, "Cancel",
			       NULL);
      if (response == NOTICE_YES)
      {
	fclose (fp);							   
	return item;
      }
    }								    /* end if part of if (strcmp... */
    fclose (fp);							   
  }								    /* end if (stat... */
  WriteEditListToFile(bufferNum);
  return item;
}								    /* end function SaveFile */


/*
 * Menu handler for `documentMenu (Save As)'.
 */
Menu_item
SaveAs(item, op)
  Menu_item	item;
  Menu_generate	op;
{
  switch (op) 
    {
    case MENU_NOTIFY:
      Browse(realpath(currentFilename, dummy), 
	     BrowseSave, 0, "#Shell Edit Document#", "ShellEdit"); 
      break;
    case MENU_DISPLAY:
    case MENU_DISPLAY_DONE:
    case MENU_NOTIFY_DONE:
      break;
    }
  return item;
}

/*
 * Menu handler for `documentMenu (Quit)'.
 */
Menu_item
  QuitShellEdit(item, op)
Menu_item	item;
Menu_generate	op;
{
  switch (op) 
    {
    case MENU_NOTIFY:
      xv_destroy_safe(ShellEdit_baseWindow->baseWindow);
      break;
    case MENU_DISPLAY:
    case MENU_DISPLAY_DONE:
    case MENU_NOTIFY_DONE:
      break;
    }
  return item;
}


/*
   This is the interpose function for the frame destroy procedure.  It
   is called when the quit button is pressed, or when the frame's "Quit"
   menu selection is chosen.
   The frame destroying process happens in two phases (meaning this
   function will be called twice).  The first time it is called, status =
   DESTROY_CHECKING (this is done automatically by XView).  At this
   point, the interpose function can choose to veto the destruction, or
   let it proceed to phase two, where the actual destruction occurs.
*/
Notify_value 
  QuitNotify(client, status)
Notify_client client;
Destroy_status status;
{
  int response, i;
  char message[MaxLength];
  /* First phase:  Check to see if we really want to destroy the frame */
  if (status == DESTROY_CHECKING)
    {
      for (i=0; i<bufferSize; i++)
	if (editList[i]->changes)
	  {
	    sprintf(message, "Save changes to '%s' ?", editList[i]->filename);
	    response = notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
				     NOTICE_MESSAGE_STRINGS,
				     message,
				     NULL,
				     NOTICE_BUTTON_YES, "Yes",
				     NOTICE_BUTTON_NO,  "No",
				     NOTICE_BUTTON, "Cancel", 100,
				     NULL);
	    if (response == 100)
	      return notify_veto_destroy(client);
	    if (response == NOTICE_YES)
	      {
		WriteEditListToFile(i);
		notify_veto_destroy(client);
	      }
	  }
      WriteFunction = WriteThenExit;
    }
  else /* Second phase:  Proceed to destroy */
    {
      KillProcesses(NULL, NULL);
      SenderDisconnectFromPortMgr(sender, &(receiver->receivePort));
      return notify_next_destroy_func(client, status);
    }
  return NOTIFY_DONE;
}

int 
  CheckFile(size)
int             *size;
{
  int counter, temp;
  int response;
  char message[MaxLength];
  char line[MaxLength];
  char fileType[MaxLength];
  char workingDirectory[MaxLength];
  char *formatString[MaxLength];
  char *tempCommand, *tempLabel, *tempDuration, *desiredWorkingDirectory;
  FILE *fp;
  fp = (FILE *)fopen(editList[bufferNum]->filename, "r");
  fscanf (fp, "#%s Edit Document#\n", fileType);			
  if (strcmp(fileType, "Shell"))				
    {								
      sprintf (message, "'%s' is not a Shell Edit list\n", editList[bufferNum]->filename);
      response = notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       message,
			       NULL,
			       NOTICE_BUTTON_YES, "OK",
			       NULL);
      fclose(fp);
      return -1;
    }
  fgets(line,512,fp);	
  desiredWorkingDirectory = &line[20];				        
  desiredWorkingDirectory[strlen(desiredWorkingDirectory)-1] = '\0';	        
  if (getwd(workingDirectory))
    {
      if (strcmp(workingDirectory, desiredWorkingDirectory) && 
	  strncmp(desiredWorkingDirectory, "Unspecified", 11))
	{
	  sprintf(command, "cd %s\n", desiredWorkingDirectory);
	  textsw_insert(ShellEdit_baseWindow->terminal, command, strlen(command));
	}
    }
  else if (strncmp(desiredWorkingDirectory, "Unspecified", 11))
    { 
      sprintf(command, "cd %s\n", desiredWorkingDirectory);
      textsw_insert(ShellEdit_baseWindow->terminal, command, strlen(command));
    }
  fscanf (fp, "#Number of Edits:\t%d\n", &temp);		
  *size = temp;
  for (counter=0; counter<temp; counter++)
    {
      fgets(line,512,fp);	  /* skip the #Edit Number: */                              
      fgets(line,512,fp);	                                
      tempCommand = &line[10];				        
      tempCommand[strlen(tempCommand)-1] = '\0';	        
      editList[bufferNum]->command[counter] = (char *)strdup(tempCommand);
      fgets(line,512,fp);	                                
      tempLabel = &line[9];				        
      tempLabel[strlen(tempLabel)-1] = '\0';	        
      editList[bufferNum]->label[counter] = (char *)strdup(tempLabel);
      fgets(line,512,fp);	                                
      tempDuration = &line[11];			        
      tempDuration[strlen(tempDuration)-1] = '\0';	        
      editList[bufferNum]->duration[counter] = (char *)strdup(tempDuration);
      formatString[counter] = (char *)malloc(MaxLength);
      if (strncmp(editList[bufferNum]->label[counter], "No Label", 8) != 0)
	sprintf(formatString[counter], "%3d   %-25.25s  %-10s", 
		counter+1, editList[bufferNum]->label[counter], editList[bufferNum]->duration[counter]);
      else
	sprintf(formatString[counter], "%3d   %-25.25s  %-10s", 
		counter+1, editList[bufferNum]->command[counter], editList[bufferNum]->duration[counter]);
      fgets(line,512,fp);	                                
    }
  fclose(fp);

  xv_set(ShellEdit_baseWindow->scrollList, 
	 XV_SHOW, FALSE, NULL); 
  for (counter=0; counter<temp; counter++)
    xv_set(ShellEdit_baseWindow->scrollList, 
	   PANEL_LIST_INSERT, counter, 
	   XV_SHOW, FALSE, 
	   PANEL_LIST_STRING, counter, formatString[counter],
	   PANEL_LIST_FONT, counter, listFont,
	   NULL);
  xv_set(ShellEdit_baseWindow->scrollList, 
	 XV_SHOW, TRUE, NULL);   
  return 1;
}


void 
  WriteEditListToFile(i)
int i;
{
  FILE *fp;
  char workingDirectory[MaxLength];
  int counter;
  char errorMessage[MaxLength];
  DeselectAll();
  if (strstr(editList[i]->filename, "untitled") != NULL || strlen(editList[i]->filename) == 0) 
    {
      bufferNum = i;
      Browse(realpath(editList[i]->filename, dummy), 
	     BrowseSave, 0, "#Shell Edit Document#", "ShellEdit"); 
      return;
    }
  if ((fp = fopen(editList[i]->filename, "w")) == (FILE *)NULL)
    {
      sprintf(errorMessage, "Can't open '%s' for writing.", editList[i]->filename);
      notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
		    NOTICE_MESSAGE_STRINGS,
		    errorMessage,
		    NULL,
		    NOTICE_BUTTON_YES, "OK",
		    NULL);
      fclose (fp);							   
      return;
    }
  fprintf (fp, "#Shell Edit Document#\n");
  if (getwd(workingDirectory))	
    fprintf (fp, "#Working Directory:\t%s\n", workingDirectory);
  else
    fprintf (fp, "#Working Directory\t%s\n", "Unspecified");
  fprintf (fp, "#Number of Edits:\t%d\n\n", editList[i]->numItems);		   
  for (counter = 0; counter < editList[i]->numItems; counter ++)			   
    {  							   
      fprintf(fp, "#Edit Number:\t%d\n#Command:\t%s\n#Label:\t\t%s\n#Duration:\t%s\n\n",	   
	      counter+1, editList[i]->command[counter], editList[i]->label[counter], 
	      editList[i]->duration[counter]);
    }
  fclose (fp);							   
  editList[i]->changes = 0;
  UpdateHeader(editList[i]->filename,				    /* Display new document title and changed status... */
	       editList[i]->changes);				    /* ...on title bar */
  if (WriteFunction == WriteThenExit)
    xv_destroy_safe(ShellEdit_baseWindow->baseWindow);
  return;
}


/*
 * Notify callback function for `add'.
 */
void 
  Add(item, event)
Panel_item	item;
Event		*event;
{
  char formatString[MaxLength];
  char *tempCommand, *tempLabel, *tempDuration;
  char selectedNum[MaxLength];
  int lastEdit;
  tempCommand = (char *)strdup((char *)xv_get(ShellEdit_baseWindow->command, PANEL_VALUE));
  tempLabel = (char *)strdup((char *)xv_get(ShellEdit_baseWindow->label, PANEL_VALUE));
  if (strlen(tempLabel) < 1)
    tempLabel = (char *)strdup("No Label");
  tempDuration = (char *)strdup((char *)xv_get(ShellEdit_baseWindow->duration, PANEL_VALUE)); 
  if (editList[bufferNum] == NULL)
    {
      editList[bufferNum] = (struct EditPacket *)malloc(sizeof(struct EditPacket));
      editList[bufferNum]->numItems = 0;
      bufferSize++;
      sprintf(editList[bufferNum]->filename, "untitled.%d", bufferSize);
    }
  lastEdit = ++editList[bufferNum]->numItems;
  DeselectAll();
  if (strncmp(tempLabel, "No Label", 8) != 0)
    sprintf(formatString, "%3d   %-25.25s  %-10s", lastEdit, tempLabel, tempDuration);
  else
    sprintf(formatString, "%3d   %-25.25s  %-10s", lastEdit, tempCommand, tempDuration);
  xv_set(ShellEdit_baseWindow->scrollList, 
	 PANEL_LIST_INSERT, lastEdit-1, 
	 PANEL_LIST_STRING, lastEdit-1, formatString, 
	 PANEL_LIST_FONT, lastEdit-1, listFont, 
	 NULL);  
  editNum = lastEdit;
  editList[bufferNum]->command[editNum-1] = (char *)strdup(tempCommand);
  editList[bufferNum]->label[editNum-1] = (char *)strdup(tempLabel);
  editList[bufferNum]->duration[editNum-1] = (char *)strdup(tempDuration);
  itoa(editNum, selectedNum);
  xv_set(ShellEdit_baseWindow->numOfEdits, PANEL_LABEL_STRING, selectedNum, NULL);
  xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, selectedNum, NULL);
  xv_set(ShellEdit_baseWindow->scrollList, 
	 PANEL_LIST_SELECT, lastEdit-1, TRUE,
	 NULL);  
  xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, FALSE, NULL);
  editList[bufferNum]->changes = 1;
  UpdateHeader(editList[bufferNum]->filename,			    /* Display new document title and changed status... */
	       editList[bufferNum]->changes);			    /* ...on title bar */
}								    /* end function Add */


/*
 * Notify callback function for `modify'.
 */
void 
  Modify(item, event)
Panel_item	item;
Event		*event;
{
  char formatString[MaxLength];
  char *tempCommand, *tempDuration, *tempLabel;
  char selectedNum[MaxLength];
  tempCommand = (char *)strdup((char *)xv_get(ShellEdit_baseWindow->command, PANEL_VALUE));
  tempLabel = (char *)strdup((char *)xv_get(ShellEdit_baseWindow->label, PANEL_VALUE));
  if (strlen(tempLabel) < 1)
    tempLabel = (char *)strdup("No Label");
  tempDuration = (char *)strdup((char *)xv_get(ShellEdit_baseWindow->duration, PANEL_VALUE)); 
  if (strncmp(tempLabel, "No Label", 8) != 0)
    sprintf(formatString, "%3d   %-25.25s  %-10s", editNum, tempLabel, tempDuration);
  else
    sprintf(formatString, "%3d   %-25.25s  %-10s", editNum, tempCommand, tempDuration);
  xv_set(ShellEdit_baseWindow->scrollList,  
	 PANEL_LIST_STRING, editNum-1, formatString, 
	 PANEL_LIST_SELECT, editNum-1, TRUE,
	 PANEL_LIST_FONT, editNum-1, listFont, 
	 NULL);
  editList[bufferNum]->command[editNum-1] = (char *)strdup(tempCommand); 
  editList[bufferNum]->label[editNum-1] = (char *)strdup(tempLabel);
  editList[bufferNum]->duration[editNum-1] = (char *)strdup(tempDuration);
  xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, FALSE, NULL);
  editList[bufferNum]->changes = 1;
  UpdateHeader(editList[bufferNum]->filename,			    /* Display new document title and changed status... */
	       editList[bufferNum]->changes);			    /* ...on title bar */
}								    /* end function Modify */


/*
 * Notify callback function for `delete'.
 */
void 
  Delete(item, event)
Panel_item	item;
Event		*event;
{
  int lastEdit, counter;
  char formatString[MaxLength];
  char newTot[MaxLength];
  lastEdit = editList[bufferNum]->numItems--;
  xv_set(ShellEdit_baseWindow->scrollList, XV_SHOW, FALSE, NULL); 
  DeselectList();   
  if (editNum > 1)
    xv_set(ShellEdit_baseWindow->scrollList, PANEL_LIST_SELECT, editNum-2, TRUE, NULL);
  xv_set(ShellEdit_baseWindow->scrollList, PANEL_LIST_DELETE, lastEdit - 1, NULL);
  for (counter = editNum-1; counter < lastEdit-1; counter++)
    { 
      editList[bufferNum]->command[counter] = editList[bufferNum]->command[counter+1];
      editList[bufferNum]->label[counter] = editList[bufferNum]->label[counter+1];
      editList[bufferNum]->duration[counter] = editList[bufferNum]->duration[counter+1];
      if (strncmp(editList[bufferNum]->label[counter], "No Label", 8) != 0)
	sprintf(formatString, "%3d   %-25.25s  %-10s", 
		counter+1, editList[bufferNum]->label[counter], editList[bufferNum]->duration[counter]);
      else
	sprintf(formatString, "%3d   %-25.25s  %-10s", 
		counter+1, editList[bufferNum]->command[counter], editList[bufferNum]->duration[counter]);
      xv_set(ShellEdit_baseWindow->scrollList,  
	     PANEL_LIST_STRING, counter, formatString, 
	     PANEL_LIST_FONT, counter, listFont, 
	     NULL);    
    }
  DeselectAll();
  xv_set(ShellEdit_baseWindow->scrollList, XV_SHOW, TRUE, NULL);   
  itoa(lastEdit-1, newTot);
  xv_set(ShellEdit_baseWindow->numOfEdits, PANEL_LABEL_STRING, newTot, NULL);
  xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, "New Edit", NULL); 
  xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, TRUE, NULL);	
  xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, TRUE, NULL);	
  if (editList[bufferNum]->numItems == 0)	
    xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, TRUE, NULL);
  editList[bufferNum]->changes = 1;
  UpdateHeader(editList[bufferNum]->filename,			    /* Display new document title and changed status... */
	       editList[bufferNum]->changes);			    /* ...on title bar */
}								    /* end function Delete */


/*
 * Notify callback function for `deleteAll'.
 */ 
void 
  DeleteAll(item, event)
Panel_item	item;
Event		*event;
{
  DeleteAllLists();
  editList[bufferNum]->numItems = 0;
  editList[bufferNum]->changes = 1;
  UpdateHeader(editList[bufferNum]->filename,			    /* Display new document title and changed status... */
	       editList[bufferNum]->changes);			    /* ...on title bar */
}								    /* end function DeleteAll */


void 
  LoadEditList()
{
  int counter;
  char formatString[MaxLength];
  char newTot[MaxLength];
  xv_set(ShellEdit_baseWindow->scrollList, XV_SHOW, FALSE, NULL); 
  DeleteAllLists();
  for (counter = 0; counter < editList[bufferNum]->numItems; counter++)
    { 
      if (strncmp(editList[bufferNum]->label[counter], "No Label", 8) != 0)
	sprintf(formatString, "%3d   %-25.25s  %-10s", 
		counter+1, editList[bufferNum]->label[counter], editList[bufferNum]->duration[counter]);
      else
	sprintf(formatString, "%3d   %-25.25s  %-10s", 
		counter+1, editList[bufferNum]->command[counter], editList[bufferNum]->duration[counter]);
      xv_set(ShellEdit_baseWindow->scrollList,  
	     PANEL_LIST_STRING, counter, formatString, 
	     PANEL_LIST_FONT, counter, listFont, 
	     NULL);    
    }
  DeselectAll();
  xv_set(ShellEdit_baseWindow->scrollList, XV_SHOW, TRUE, NULL);   
  itoa(editList[bufferNum]->numItems, newTot);
  xv_set(ShellEdit_baseWindow->numOfEdits, PANEL_LABEL_STRING, newTot, NULL);
  xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, "New Edit", NULL); 
  xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, TRUE, NULL);	
  xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, TRUE, NULL);	
  if (editList[bufferNum]->numItems == 0)	
    xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, TRUE, NULL);
  else
    xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, FALSE, NULL);
}

void 
  DeleteFromBuffer()
{
  int i;
  free(editList[bufferNum]);
  for (i=bufferNum; i<bufferSize-1; i++)
    {
      editList[i] = editList[i+1];
    }
  bufferSize--;
  DeleteAllLists();
}

void 
  PrintBuffer()
{
  int i;
  for (i=0; i<bufferSize; i++)
    {
      printf("%d  %s\n", i, editList[i]->filename);
    }
}

void 
  DeleteAllLists()
{
  int i, nrows;
  xv_set(ShellEdit_baseWindow->scrollList, XV_SHOW, FALSE, NULL);
  nrows = (int)xv_get(ShellEdit_baseWindow->scrollList, PANEL_LIST_NROWS);
  for (i = 0; i < nrows; i++)
    xv_set(ShellEdit_baseWindow->scrollList, PANEL_LIST_DELETE, 0, NULL);
  xv_set(ShellEdit_baseWindow->scrollList, XV_SHOW, TRUE, NULL);   
  xv_set(ShellEdit_baseWindow->numOfEdits, PANEL_LABEL_STRING, "0", NULL);
  xv_set(ShellEdit_baseWindow->currentSelectionField, PANEL_LABEL_STRING, "New Edit", NULL); 
  xv_set(ShellEdit_baseWindow->command, PANEL_VALUE, "", NULL); 
  xv_set(ShellEdit_baseWindow->label, PANEL_VALUE, "", NULL); 
  xv_set(ShellEdit_baseWindow->duration, PANEL_VALUE, "", NULL); 
  xv_set(ShellEdit_baseWindow->delete, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ShellEdit_baseWindow->modify, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ShellEdit_baseWindow->deleteAll, PANEL_INACTIVE, TRUE, NULL);
}


void 
  DeselectList()
{
  xv_set (ShellEdit_baseWindow->scrollList, PANEL_LIST_SELECT, editNum-1, FALSE, NULL);
}


void 
  DeselectAll()
{
  int counter = 0;
  while (counter < editList[bufferNum]->numItems)
    {
      if ((int)xv_get(ShellEdit_baseWindow->scrollList, PANEL_LIST_SELECTED, counter))
	  xv_set(ShellEdit_baseWindow->scrollList, PANEL_LIST_SELECT, counter, FALSE, NULL);
      counter++;
    }
}


void 
  itoa(num, string)
int num;
char string[];
{
  int i=0, sign;
  if ((sign=num) < 0) num = -num;
  do 
    string[i++] = num%10 + '0';
  while ((num /= 10) > 0);
  if (sign < 0) string[i++] = '-';
  string[i] = '\0';
  Reverse(string);
}

void 
  Reverse(string)
char string[];
{
  int c, i, j;
  for (i=0, j=strlen(string)-1; i<j; i++, j--)
    {
      c = string[i];
      string[i] = string[j];
      string[j] = c;
    }
}

int                                            /* browser code */
  OpenHandler(char *proposedPath, int id)
{
  char temp[MaxLength];
  currentFilename = (char *)strdup(proposedPath);
  if (OpenEditFile() == 1)
    {
      sprintf(temp, "Shell Edit Document :  \"%s\"", currentFilename);
      xv_set(ShellEdit_baseWindow->baseWindow, XV_LABEL, temp, NULL);
      return 0;
    }
  else
    return 1;
}

int                                            /* browser code */
  SaveHandler(char *proposedPath, int id)
{
  char temp[MaxLength];
  currentFilename = (char *)strdup(proposedPath);
  sprintf(temp, "Shell Edit Document :  \"%s\"", currentFilename);
  xv_set(ShellEdit_baseWindow->baseWindow, XV_LABEL, temp, NULL);
  if (strstr(proposedPath, "untitled") == NULL && strlen(proposedPath) > 0) 
    {
      SaveFile(NULL, NULL);
      return 0;
    }
  else
    {
      sprintf(temp, "Can't save to %s.\n", proposedPath);
      notice_prompt(ShellEdit_baseWindow->baseWindow, NULL,
		    NOTICE_MESSAGE_STRINGS,
		    temp,
		    "Try saving to a different name.",
		    NULL,
		    NOTICE_BUTTON_YES, "OK",
		    NULL);
      return -1;
    }
}


/* 
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  In this case, it will be the filename to be
 * loaded.  At present, only one filename will be handled.  So if there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(argc, argv)
     int 	argc;
     char 	**argv;
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
  {
    {"hostname", 1, 0, 'h'},		

    {"portnumber", 1, 0, 'p'},
    {0, 0, 0, 0}
  };

  while (1)							    /* Start parsing all known options */
  {
    optionChar = getopt_long_only (argc, argv, "h:p:",
			      long_options, &option_index);
    if (optionChar == EOF)					    /* Done with all known options */
    {
      break;
    }
    switch (optionChar)
    {
     case 'h':
      if (optarg) 
      {
	senderPort.hostName = strdup(optarg);
      }
      break;
     case 'p':
      if (optarg) 
      {
	ReceiverPortNumber = atoi(optarg);
      }
      break;
     default:
      break;
    }
  }
  if (optind < argc)						    /* Check if a filename has been specified */
    Browse(argv[optind], BrowseCheckOpen, 0, "#Shell Edit Document#", "ShellEdit"); 
}

void 
  Show(popup)
Frame *popup;
{
  if ((int)xv_get(*popup,FRAME_CMD_PUSHPIN_IN) == FALSE)		   
    xv_set (*popup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);		   
  if ((int)xv_get(*popup, XV_SHOW) == FALSE)			   
    xv_set (*popup, XV_SHOW, TRUE, NULL);			   
}								   

void 
  Hide(popup)
Frame *popup;
{
  if ((int)xv_get(*popup,FRAME_CMD_PUSHPIN_IN) == TRUE)		   
    xv_set (*popup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);		   
  if ((int)xv_get(*popup, XV_SHOW) == TRUE)			   
    xv_set (*popup,XV_SHOW, FALSE, NULL);			   
}      


/*
 * Notify callback function for `info'.
 */
void
  HideInfo(item, event)
Panel_item	item;
Event		*event;
{
  Hide(&ShellEdit_ShellEditInfo->ShellEditInfo);
}



void UpdateHeader (char* documentName, int modified)
{
  char	label[MAXPATHLEN+32];
  
  if (modified == 1)
  {
    sprintf(label, "Shell Edit Document : \"%s\" (modified)",
	    documentName);
  }
  else
  {
    sprintf(label, "Shell Edit Document : \"%s\" ", documentName);
  }
  xv_set(ShellEdit_baseWindow->baseWindow,			    /* Print the new label string on the title bar */
	 XV_LABEL, label, NULL);
}								    /* end function UpdateHeader */
